/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    querySelect.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point select query
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-04/AW: Allow to read from multiple nodes at once (#2368)
 * \version 2017-11-24/AW: Constructors declared as "explicit" (#2825)
 * \version 2018-07-05/AW: Wrapper function bindBuffer() added
 * \version 2020-01-17/AW: Wrapper function bindMember() added (#3497)
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_QUERYSELECT_HPP
#define RIEGL_RDB_POINTCLOUD_QUERYSELECT_HPP

//---< INCLUDES >---------------------------------------------------------------

#include <memory>
#include <string>
#include <vector>
#include <cstdlib>
#include <cstdint>

#include "riegl/rdb/pointcloud/dataTypes.hpp"
#include "riegl/rdb/pointcloud/graphNode.hpp"
#include "riegl/rdb/pointcloud/pointcloudData.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< CLASS QuerySelect >------------------------------------------------------
/*!
 * \brief Point select query
 *
 * This class can be used to select (read) attributes of existing points.
 *
 * \see riegl::rdb::Pointcloud::select()
 *
 * \note You either must delete the query object or call close()
 *       __before__ the parent Pointcloud instance is closed/deleted!
 */
class QuerySelect
{
public:
    /*!
     * \brief Default constructor
     *
     * Creates a null query - i.e. the query cannot be used to read points.
     *
     * \see riegl::rdb::Pointcloud::select()
     */
    explicit QuerySelect();

    /*!
     * \brief Constructor
     *
     * Creates a query prepared for reading points.
     *
     * \note You cannot create new QuerySelect objects this way,
     *       use riegl::rdb::Pointcloud::select() instead.
     */
    explicit QuerySelect(
        riegl::rdb::PointcloudData       *pointcloud,
        const std::vector<GraphNode::ID> *nodes,
        const std::string                &filter
    );

    /*!
     * \brief Check if query is not null
     *
     * \see valid()
     */
    operator bool() const;

    /*!
     * \brief Check if query is not null
     *
     * A null query cannot be used to read points.
     */
    bool valid() const;

    /*!
     * \brief Finish query
     *
     * Call this function when done with reading points.
     */
    void close();

    /*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a target buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * The buffer is expected to be n*s*d bytes large, where
     * __n__ is the number of points defined in next(),
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * \note This function just stores the buffer pointer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     */
    void bind(
        const std::string &attribute, //!< [in] attribute name
        const DataType     dataType,  //!< [in] buffer data type
        void              *buffer,    //!< [in] buffer location
        const std::int32_t stride = 0 //!< [in] bytes between beginnings of successive elements (0: auto)
    );

    //! \copydoc bind()
    template <typename ValueType>
    void bindBuffer(
        const std::string &attribute, //!< [in] attribute name
        ValueType         &buffer,    //!< [in] buffer (data, pointer to data, std::array or std::vector)
        const std::int32_t stride = 0 //!< [in] bytes between beginnings of successive elements (0: auto)
    )
    {
        bind(attribute, dataTypeOf(buffer), dataPointerOf(buffer), stride);
    }

    //! \copydoc bind()
    template <typename ValueType>
    void bindBuffer(
        const std::string &attribute, //!< [in] attribute name
        ValueType         *buffer,    //!< [in] buffer (data, pointer to data, std::array or std::vector)
        const std::int32_t stride = 0 //!< [in] bytes between beginnings of successive elements (0: auto)
    )
    {
        bind(attribute, dataTypeOf(buffer), buffer, stride);
    }

    /*!
     * \brief Bind attribute buffer
     *
     * This is a variant of bindBuffer() that allows to bind a member variable
     * of an object as attribute buffer. The object can be part of a container
     * that stores the objects contiguously (e.g. std::vector, std::array) and
     * the stride is automatically derived from the object size.
     *
     * \see bindBuffer()
     */
    template <typename ObjectType, typename MemberPointer>
    void bindMember(
        const std::string  &attribute, //!< [in] attribute name
        ObjectType         &object,    //!< [in] e.g. first object of container
        const MemberPointer member     //!< [in] object member variable pointer
    )
    {
        bindBuffer(
            attribute, object.*member,
            static_cast<std::int32_t>(sizeof(ObjectType))
        );
    }

    //! \copydoc bindMember()
    template <typename ObjectType, typename MemberPointer>
    void bindMember(
        const std::string  &attribute, //!< [in] attribute name
        ObjectType         &object,    //!< [in] e.g. first object of container
        const MemberPointer member,    //!< [in] object member variable pointer
        const std::size_t   index      //!< [in] index for array-like object members
    )
    {
        bindBuffer(
            attribute, (object.*member)[index],
            static_cast<std::int32_t>(sizeof(ObjectType))
        );
    }

    /*!
     * \brief Select points
     *
     * Use this function to actually read the selected points from
     * database and copy the attributes to the defined buffers.
     *
     * Afterwards you may re-use the buffers or define new buffers
     * with bind() and call next() again until all points have been
     * read (i.e. next() returns zero).
     *
     * If no buffers were bound, calling next() is still valid. This can be
     * used for counting points - either all or just those that meet some
     * filter criteria. If no filter is specified the total point count is
     * equal to what you get with riegl::rdb::Pointcloud::stat().
     *
     * \returns the number of points read
     */
    std::uint32_t next(
        std::uint32_t count //!< [in] size of target buffers in terms of points
    );

    /*!
     * \brief Progress
     *
     * This function returns a coarse progress information in percent (0..100%).
     * Since the total number of returned points is not known in advance, this
     * value just reflects the progress of the (internal) index traversal.
     *
     * \note When the select query is used to count points (i.e. no buffers
     *       were bound), then this function always returns 0%.
     */
    std::uint32_t progress() const;

private:
    struct Private;
    std::shared_ptr<Private> data;
};

}}} // namespace riegl::rdb::pointcloud

#endif // RIEGL_RDB_POINTCLOUD_QUERYSELECT_HPP
