/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    openSettings.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Database open settings
 * \version 2015-10-14/AW: Initial version
 * \version 2016-09-22/AW: New functions to get/set 64 bit cache size parameter
 * \version 2016-12-19/AW: New functions to load/save settings from/to JSON
 * \version 2017-03-28/AW: Documentation of JSON load/save functions updated
 * \version 2020-02-21/AW: Add "context-free" function variants (#3544)
 *
 * This class defines settings for opening a point cloud database.
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_OPENSETTINGS_H
#define RIEGL_RDB_POINTCLOUD_OPENSETTINGS_H

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"

//---< TYPE DEFINITIONS >-------------------------------------------------------

typedef struct RDBPointcloudOpenSettings RDBPointcloudOpenSettings; // forward declaration of implementation details

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN

//______________________________________________________________________________
/*!
 * \brief Default constructor
 *
 * All values are set to default values.
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_new,
    RDBContext                 *context, //!< [in] library context
    RDBPointcloudOpenSettings **settings //!< [out] handle of created object
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_new_cf,
    RDBPointcloudOpenSettings **settings //!< [out] handle of created object
);

//______________________________________________________________________________
/*!
 * \brief Destroy settings object
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_delete,
    RDBContext                 *context, //!< [in] library context
    RDBPointcloudOpenSettings **settings //!< [in] handle of object to delete
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_delete_cf,
    RDBPointcloudOpenSettings **settings //!< [in] handle of object to delete
);

//______________________________________________________________________________
/*!
 * \brief Load settings from JSON string
 *
 * This function parses the given JSON string and applies all available
 * properties - missing properties are silently ignored (i.e. the value
 * remains unchanged). When parsing the JSON string fails, an exception
 * is thrown.
 *
 * Example JSON string:
 *
 *     {
 *         "cache_size": 524288000
 *     }
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_json_load,
    RDBContext                *context,  //!< [in] library context
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    RDBString                  json      //!< [in] JSON string
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_json_load_cf,
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    RDBString                  json      //!< [in] JSON string
);

//______________________________________________________________________________
/*!
 * \brief Save settings to JSON string
 * \see rdb_pointcloud_open_settings_json_load()
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_json_save,
    RDBContext                *context,  //!< [in] library context
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    RDBString                 *json      //!< [out] JSON string
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_json_save_cf,
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    RDBString                 *json      //!< [out] JSON string
);

//______________________________________________________________________________
/*!
 * \brief Get point cache size
 *
 * The database engine may buffer read and write operations in an
 * internal cache. This value defines the cache size in bytes (octets).
 *
 * Default: 500 MB
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_get_cache_size,
    RDBContext                *context,  //!< [in] library context
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint32_t                  *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_get_cache_size_cf,
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint32_t                  *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Get point cache size
 * \see rdb_pointcloud_open_settings_get_cache_size
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_get_cache_size_64,
    RDBContext                *context,  //!< [in] library context
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint64_t                  *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_get_cache_size_64_cf,
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint64_t                  *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set point cache size
 * \see rdb_pointcloud_open_settings_get_cache_size()
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_set_cache_size,
    RDBContext                *context,  //!< [in] library context
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint32_t                   value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_set_cache_size_cf,
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint32_t                   value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Set point cache size
 * \see rdb_pointcloud_open_settings_set_cache_size()
 */
RDB_FUNCTION(rdb_pointcloud_open_settings_set_cache_size_64,
    RDBContext                *context,  //!< [in] library context
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint64_t                   value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_open_settings_set_cache_size_64_cf,
    RDBPointcloudOpenSettings *settings, //!< [in] handle of object
    uint64_t                   value     //!< [in] new value
);

RDB_LIBRARY_API_END

#endif // RIEGL_RDB_POINTCLOUD_OPENSETTINGS_H
