#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
#*******************************************************************************
#
#  Copyright 2026 RIEGL Laser Measurement Systems
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
#  SPDX-License-Identifier: Apache-2.0
#
#*******************************************************************************
#
"""
rdb-example-02-insert-points.py

This example shows how to open an existing database and add some points.
This example is based on the database of rdb-example-1-create-database.
"""

import random

import riegl.rdb


def example_plain():
    """
    Basic example to add some points.
    """
    # Access existing database
    with riegl.rdb.rdb_open("pointcloud.rdbx") as rdb:

        # Query some attribute details
        details_coordinates = rdb.point_attributes["riegl.xyz"]
        details_reflectance = rdb.point_attributes["riegl.reflectance"]

        # Before we can modify the database, we must start a transaction
        with riegl.rdb.Transaction(
            rdb,  # point cloud object
            "Import",  # transaction title
            "Point Importer v1.0"  # software name
        ) as transaction:

            # Prepare point attribute buffers
            buffer_size = 100  # point block/chunk size
            point_count = 1500  # total number of points
            buffer_coordinates = riegl.rdb.AttributeBuffer(
                details_coordinates, buffer_size
            )
            buffer_reflectance = riegl.rdb.AttributeBuffer(
                details_reflectance, buffer_size
            )

            # Start new insert query and define buffers
            with rdb.insert() as query:
                query.bind(buffer_coordinates)
                query.bind(buffer_reflectance)

                # Insert points block-wise
                total = 0
                while total < point_count:
                    # Fill buffers with some random data
                    for i in range(buffer_size):
                        buffer_coordinates[i] = [random.uniform(
                            details_coordinates.minimum_value,
                            details_coordinates.maximum_value
                        ) for _ in range(3)]
                        buffer_reflectance[i] = random.uniform(
                            details_reflectance.minimum_value,
                            details_reflectance.maximum_value
                        )

                    # Actually insert points
                    total += query.next(buffer_size)

                # Finally commit transaction
                transaction.commit()


def example_numpy():
    """
    Similar to example_plain() but we use NumPy arrays here.
    """
    import numpy as np

    # Create some numpy arrays
    c = np.array([[1, 2, 3], [4, 5, 6]])
    r = np.array([7, 8])

    # Access existing database
    with riegl.rdb.rdb_open("pointcloud.rdbx") as rdb:

        # Before we can modify the database, we must start a transaction
        with riegl.rdb.Transaction(
            rdb,  # point cloud object
            "Import",  # transaction title
            "Point Importer v1.0"  # software name
        ) as transaction:

            # Start inserting points
            with rdb.insert() as insert:
                # Create buffers for all point attributes
                buffers = riegl.rdb.PointBuffer(rdb, count=len(c), attributes=[
                    "riegl.xyz",
                    "riegl.reflectance"
                ])

                # Copy data to point attribute buffers
                np.copyto(buffers["riegl.xyz"]        .data, c)
                np.copyto(buffers["riegl.reflectance"].data, r)

                # Actually insert points
                insert.bind(buffers)
                insert.next(len(c))

            # Finally commit transaction
            transaction.commit()


if __name__ == "__main__":
    example_plain()
    example_numpy()
