#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
#*******************************************************************************
#
#  Copyright 2026 RIEGL Laser Measurement Systems
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
#  SPDX-License-Identifier: Apache-2.0
#
#*******************************************************************************
#
"""
rdb-example-11-meta-data.py

This example shows how to open an existing database and access the meta data.
"""

import json

import riegl.rdb

# open existing database:
with riegl.rdb.rdb_open("pointcloud.rdbx") as rdb:

    # print list of all meta data item names:
    print(rdb.meta_data.list())

    # iterate over all meta data items:
    for name in rdb.meta_data:
        print(name)

    # check if a meta data item exists:
    # alternative: if rdb.meta_data.exists("riegl.time_base"):
    if "riegl.time_base" in rdb.meta_data:
        print("found riegl.time_base")
    else:
        print("missing riegl.time_base")

    # read meta data item by name:
    print(rdb.meta_data["riegl.time_base"])
    print(rdb.meta_data.get("riegl.time_base", "default value"))
    # if the item does not exist, the first variant prints an
    # empty line and the second variant prints "default value"

    # read meta data item using RIEGL default names (this
    # is the recommended way as it helps to avoid typos):
    print(rdb.meta_data[riegl.rdb.RDB_RIEGL_TIME_BASE])

    # meta data modification:
    with riegl.rdb.Transaction(rdb, "modify meta data", "example 11") as transaction:
        # set plain text meta data (all three lines are equal):
        rdb.meta_data["riegl.time_base"] = "plain text"
        rdb.meta_data[riegl.rdb.RDB_RIEGL_TIME_BASE] = "plain text"
        rdb.meta_data.set("riegl.time_base", "plain text")
        # as with regular Python dictionaries, meta data items
        # are created on demand and existing items are overwritten

        # set dictionary ("object") as JSON:
        rdb.meta_data[riegl.rdb.RDB_RIEGL_TIME_BASE] = json.dumps({
            "epoch":  "2020-04-15T15:09:10+02:00",
            "source": "GNSS"
        })

        # delete meta data (both lines are equal):
        rdb.meta_data.remove(riegl.rdb.RDB_RIEGL_TIME_BASE)
        del rdb.meta_data[riegl.rdb.RDB_RIEGL_TIME_BASE]

        # set example meta data:
        rdb.meta_data[riegl.rdb.RDB_RIEGL_TIME_BASE] = riegl.rdb.RDB_RIEGL_TIME_BASE_EXAMPLE

        # validate meta data:
        try:
            rdb.meta_data.validate(riegl.rdb.RDB_RIEGL_TIME_BASE)
            print("successfully validated")
        except riegl.rdb.Error as error:
            print("validation failed:", error)

        # finalize transaction
        transaction.commit()
