/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    openSettings.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Database open settings
 * \version 2015-10-14/AW: Initial version
 * \version 2016-09-22/AW: Parameter 'cacheSize' changed from 32 to 64 bit type
 * \version 2016-12-19/AW: New functions to load/save settings from/to JSON
 * \version 2017-03-28/AW: Documentation of JSON load/save functions updated
 * \version 2017-11-24/AW: Constructors declared as "explicit" (#2825)
 * \version 2019-02-15/AW: Fix C++ API wrapper of OpenSettings class
 * \version 2020-02-21/AW: Class 'OpenSettings' is now context-free (#3544)
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_OPENSETTINGS_HPP
#define RIEGL_RDB_POINTCLOUD_OPENSETTINGS_HPP

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>
#include <cstdint>

#include "riegl/rdb/context.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< CLASS OpenSettings >-----------------------------------------------------
/*!
 * \brief Database open settings
 *
 * This class defines settings for opening a point cloud database.
 */
class OpenSettings
{
public:
    /*!
     * \brief Point cache size
     *
     * The database engine may buffer read and write operations in an
     * internal cache. This value defines the cache size in bytes (octets).
     *
     * Default: 500 MB
     */
    std::uint64_t cacheSize;

public:
    ~OpenSettings();

    /*!
     * \brief Default constructor
     *
     * All properties are set to default values.
     */
    OpenSettings();

    /// \deprecated since 2.2.3 - use the context-free constructors instead
    explicit OpenSettings(riegl::rdb::Context &context);

    /*!
     * \brief Copy constructor
     *
     * All properties are copied from the given settings object.
     */
    OpenSettings(const OpenSettings &settings);

    /*!
     * \brief Assignment operator
     *
     * All properties are copied from the given settings object.
     */
    OpenSettings& operator=(const OpenSettings &settings);

    /*!
     * \brief Load settings from JSON string
     *
     * This function parses the given JSON string and applies all available
     * properties - missing properties are silently ignored (i.e. the value
     * remains unchanged). When parsing the JSON string fails, an exception
     * is thrown.
     *
     * Example JSON string:
     *
     *     {
     *         "cache_size": 524288000
     *     }
     */
    void load(const std::string &json);

    /*!
     * \brief Save settings to JSON string
     * \see load()
     */
    std::string save() const;

private:
    friend class OpenSettingsWrapper;
    void *data;
};

}}} // namespace riegl::rdb::pointcloud

#endif // RIEGL_RDB_POINTCLOUD_OPENSETTINGS_HPP
