/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    library.c
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB library management tools
 * \version 2020-04-08/AW: Initial version
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb/library.h"

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN
#ifdef RDB_RUNTIME_LINKING

//---< BASIC LIBRARY HANDLING >-------------------------------------------------

#if defined(_WIN32) || defined(__WIN32__) || defined(__WINDOWS__)

#include <windows.h>

typedef HMODULE LibraryHandle;
typedef FARPROC FeatureHandle;

static LibraryHandle library_load(RDBString filename)
{
    enum {BUFFER_SIZE = 1024};
    WCHAR buffer[BUFFER_SIZE];
    memset(buffer, 0, sizeof(WCHAR) * BUFFER_SIZE);
    if (MultiByteToWideChar(CP_UTF8, 0, filename, -1, buffer, BUFFER_SIZE))
         return LoadLibraryW(buffer);
    else return 0;
}

static FeatureHandle library_resolve(LibraryHandle handle, RDBString name)
{
    return GetProcAddress(handle, name);
}

static int library_free(LibraryHandle handle)
{
    return FreeLibrary(handle) != 0;
}

#elif defined(__linux__) || (defined(__APPLE__) && defined(__MACH__))

#include <dlfcn.h>

typedef void* LibraryHandle;
typedef void* FeatureHandle;

static LibraryHandle library_load(RDBString filename)
{
    return dlopen(filename, RTLD_LAZY);
}

static FeatureHandle library_resolve(LibraryHandle handle, RDBString name)
{
    return dlsym(handle, name);
}

static int library_free(LibraryHandle handle)
{
    return dlclose(handle) == 0;
}

#else
    //
    ERROR: operating system not detected or supported
    //
#endif

//---< LIBRARY FUNCTION POINTERS >----------------------------------------------

typedef void* rdb_factory_set_authentication_function_type;

rdb_context_delete_type                                        rdb_context_delete                                        = 0;
rdb_context_get_last_error_type                                rdb_context_get_last_error                                = 0;
rdb_context_new_type                                           rdb_context_new                                           = 0;
rdb_context_new_with_api_type                                  rdb_context_new_with_api                                  = 0;
rdb_database_file_type_check_type                              rdb_database_file_type_check                              = 0;
rdb_database_file_type_suffix_type                             rdb_database_file_type_suffix                             = 0;
rdb_database_file_type_title_type                              rdb_database_file_type_title                              = 0;
rdb_factory_set_authentication_function_type                   rdb_factory_set_authentication_function                   = 0;
rdb_library_filename_type                                      rdb_library_filename                                      = 0;
rdb_library_license_type                                       rdb_library_license                                       = 0;
rdb_library_name_type                                          rdb_library_name                                          = 0;
rdb_library_version_type                                       rdb_library_version                                       = 0;
rdb_object_lock_cf_type                                        rdb_object_lock_cf                                        = 0;
rdb_object_unlock_cf_type                                      rdb_object_unlock_cf                                      = 0;
rdb_pointcloud_changelog_append_log_message_type               rdb_pointcloud_changelog_append_log_message               = 0;
rdb_pointcloud_changelog_export_to_textfile_type               rdb_pointcloud_changelog_export_to_textfile               = 0;
rdb_pointcloud_changelog_import_from_database_type             rdb_pointcloud_changelog_import_from_database             = 0;
rdb_pointcloud_changelog_verify_log_entry_cf_type              rdb_pointcloud_changelog_verify_log_entry_cf              = 0;
rdb_pointcloud_clear_cache_type                                rdb_pointcloud_clear_cache                                = 0;
rdb_pointcloud_close_type                                      rdb_pointcloud_close                                      = 0;
rdb_pointcloud_create_type                                     rdb_pointcloud_create                                     = 0;
rdb_pointcloud_create_settings_delete_type                     rdb_pointcloud_create_settings_delete                     = 0;
rdb_pointcloud_create_settings_delete_cf_type                  rdb_pointcloud_create_settings_delete_cf                  = 0;
rdb_pointcloud_create_settings_get_cache_size_type             rdb_pointcloud_create_settings_get_cache_size             = 0;
rdb_pointcloud_create_settings_get_cache_size_64_type          rdb_pointcloud_create_settings_get_cache_size_64          = 0;
rdb_pointcloud_create_settings_get_cache_size_64_cf_type       rdb_pointcloud_create_settings_get_cache_size_64_cf       = 0;
rdb_pointcloud_create_settings_get_cache_size_cf_type          rdb_pointcloud_create_settings_get_cache_size_cf          = 0;
rdb_pointcloud_create_settings_get_chunk_mode_type             rdb_pointcloud_create_settings_get_chunk_mode             = 0;
rdb_pointcloud_create_settings_get_chunk_mode_cf_type          rdb_pointcloud_create_settings_get_chunk_mode_cf          = 0;
rdb_pointcloud_create_settings_get_chunk_size_type             rdb_pointcloud_create_settings_get_chunk_size             = 0;
rdb_pointcloud_create_settings_get_chunk_size_cf_type          rdb_pointcloud_create_settings_get_chunk_size_cf          = 0;
rdb_pointcloud_create_settings_get_chunk_size_lod_type         rdb_pointcloud_create_settings_get_chunk_size_lod         = 0;
rdb_pointcloud_create_settings_get_chunk_size_lod_cf_type      rdb_pointcloud_create_settings_get_chunk_size_lod_cf      = 0;
rdb_pointcloud_create_settings_get_compression_level_type      rdb_pointcloud_create_settings_get_compression_level      = 0;
rdb_pointcloud_create_settings_get_compression_level_cf_type   rdb_pointcloud_create_settings_get_compression_level_cf   = 0;
rdb_pointcloud_create_settings_get_lod_mode_type               rdb_pointcloud_create_settings_get_lod_mode               = 0;
rdb_pointcloud_create_settings_get_lod_mode_cf_type            rdb_pointcloud_create_settings_get_lod_mode_cf            = 0;
rdb_pointcloud_create_settings_get_optimize_point_id_type      rdb_pointcloud_create_settings_get_optimize_point_id      = 0;
rdb_pointcloud_create_settings_get_optimize_point_id_cf_type   rdb_pointcloud_create_settings_get_optimize_point_id_cf   = 0;
rdb_pointcloud_create_settings_get_point_insert_mode_type      rdb_pointcloud_create_settings_get_point_insert_mode      = 0;
rdb_pointcloud_create_settings_get_point_insert_mode_cf_type   rdb_pointcloud_create_settings_get_point_insert_mode_cf   = 0;
rdb_pointcloud_create_settings_get_primary_attribute_type      rdb_pointcloud_create_settings_get_primary_attribute      = 0;
rdb_pointcloud_create_settings_get_primary_attribute_cf_type   rdb_pointcloud_create_settings_get_primary_attribute_cf   = 0;
rdb_pointcloud_create_settings_json_load_type                  rdb_pointcloud_create_settings_json_load                  = 0;
rdb_pointcloud_create_settings_json_load_cf_type               rdb_pointcloud_create_settings_json_load_cf               = 0;
rdb_pointcloud_create_settings_json_save_type                  rdb_pointcloud_create_settings_json_save                  = 0;
rdb_pointcloud_create_settings_json_save_cf_type               rdb_pointcloud_create_settings_json_save_cf               = 0;
rdb_pointcloud_create_settings_new_type                        rdb_pointcloud_create_settings_new                        = 0;
rdb_pointcloud_create_settings_new_cf_type                     rdb_pointcloud_create_settings_new_cf                     = 0;
rdb_pointcloud_create_settings_set_cache_size_type             rdb_pointcloud_create_settings_set_cache_size             = 0;
rdb_pointcloud_create_settings_set_cache_size_64_type          rdb_pointcloud_create_settings_set_cache_size_64          = 0;
rdb_pointcloud_create_settings_set_cache_size_64_cf_type       rdb_pointcloud_create_settings_set_cache_size_64_cf       = 0;
rdb_pointcloud_create_settings_set_cache_size_cf_type          rdb_pointcloud_create_settings_set_cache_size_cf          = 0;
rdb_pointcloud_create_settings_set_chunk_mode_type             rdb_pointcloud_create_settings_set_chunk_mode             = 0;
rdb_pointcloud_create_settings_set_chunk_mode_cf_type          rdb_pointcloud_create_settings_set_chunk_mode_cf          = 0;
rdb_pointcloud_create_settings_set_chunk_size_type             rdb_pointcloud_create_settings_set_chunk_size             = 0;
rdb_pointcloud_create_settings_set_chunk_size_cf_type          rdb_pointcloud_create_settings_set_chunk_size_cf          = 0;
rdb_pointcloud_create_settings_set_chunk_size_lod_type         rdb_pointcloud_create_settings_set_chunk_size_lod         = 0;
rdb_pointcloud_create_settings_set_chunk_size_lod_cf_type      rdb_pointcloud_create_settings_set_chunk_size_lod_cf      = 0;
rdb_pointcloud_create_settings_set_compression_level_type      rdb_pointcloud_create_settings_set_compression_level      = 0;
rdb_pointcloud_create_settings_set_compression_level_cf_type   rdb_pointcloud_create_settings_set_compression_level_cf   = 0;
rdb_pointcloud_create_settings_set_lod_mode_type               rdb_pointcloud_create_settings_set_lod_mode               = 0;
rdb_pointcloud_create_settings_set_lod_mode_cf_type            rdb_pointcloud_create_settings_set_lod_mode_cf            = 0;
rdb_pointcloud_create_settings_set_optimize_point_id_type      rdb_pointcloud_create_settings_set_optimize_point_id      = 0;
rdb_pointcloud_create_settings_set_optimize_point_id_cf_type   rdb_pointcloud_create_settings_set_optimize_point_id_cf   = 0;
rdb_pointcloud_create_settings_set_point_insert_mode_type      rdb_pointcloud_create_settings_set_point_insert_mode      = 0;
rdb_pointcloud_create_settings_set_point_insert_mode_cf_type   rdb_pointcloud_create_settings_set_point_insert_mode_cf   = 0;
rdb_pointcloud_create_settings_set_primary_attribute_type      rdb_pointcloud_create_settings_set_primary_attribute      = 0;
rdb_pointcloud_create_settings_set_primary_attribute_cf_type   rdb_pointcloud_create_settings_set_primary_attribute_cf   = 0;
rdb_pointcloud_create_with_schema_type                         rdb_pointcloud_create_with_schema                         = 0;
rdb_pointcloud_delete_type                                     rdb_pointcloud_delete                                     = 0;
rdb_pointcloud_get_uuid_type                                   rdb_pointcloud_get_uuid                                   = 0;
rdb_pointcloud_graph_node_delete_type                          rdb_pointcloud_graph_node_delete                          = 0;
rdb_pointcloud_graph_node_new_type                             rdb_pointcloud_graph_node_new                             = 0;
rdb_pointcloud_inspect_type                                    rdb_pointcloud_inspect                                    = 0;
rdb_pointcloud_is_empty_type                                   rdb_pointcloud_is_empty                                   = 0;
rdb_pointcloud_is_open_type                                    rdb_pointcloud_is_open                                    = 0;
rdb_pointcloud_management_finalize_type                        rdb_pointcloud_management_finalize                        = 0;
rdb_pointcloud_management_get_chunk_size_lod_type              rdb_pointcloud_management_get_chunk_size_lod              = 0;
rdb_pointcloud_management_get_lod_mode_type                    rdb_pointcloud_management_get_lod_mode                    = 0;
rdb_pointcloud_management_set_chunk_size_lod_type              rdb_pointcloud_management_set_chunk_size_lod              = 0;
rdb_pointcloud_management_set_lod_mode_type                    rdb_pointcloud_management_set_lod_mode                    = 0;
rdb_pointcloud_management_vacuum_type                          rdb_pointcloud_management_vacuum                          = 0;
rdb_pointcloud_management_validate_type                        rdb_pointcloud_management_validate                        = 0;
rdb_pointcloud_meta_data_create_signature_type                 rdb_pointcloud_meta_data_create_signature                 = 0;
rdb_pointcloud_meta_data_exists_type                           rdb_pointcloud_meta_data_exists                           = 0;
rdb_pointcloud_meta_data_get_type                              rdb_pointcloud_meta_data_get                              = 0;
rdb_pointcloud_meta_data_list_type                             rdb_pointcloud_meta_data_list                             = 0;
rdb_pointcloud_meta_data_remove_type                           rdb_pointcloud_meta_data_remove                           = 0;
rdb_pointcloud_meta_data_set_type                              rdb_pointcloud_meta_data_set                              = 0;
rdb_pointcloud_meta_data_validate_type                         rdb_pointcloud_meta_data_validate                         = 0;
rdb_pointcloud_meta_data_validate_json_type                    rdb_pointcloud_meta_data_validate_json                    = 0;
rdb_pointcloud_meta_data_verify_signature_type                 rdb_pointcloud_meta_data_verify_signature                 = 0;
rdb_pointcloud_new_type                                        rdb_pointcloud_new                                        = 0;
rdb_pointcloud_open_type                                       rdb_pointcloud_open                                       = 0;
rdb_pointcloud_open_settings_delete_type                       rdb_pointcloud_open_settings_delete                       = 0;
rdb_pointcloud_open_settings_delete_cf_type                    rdb_pointcloud_open_settings_delete_cf                    = 0;
rdb_pointcloud_open_settings_get_cache_size_type               rdb_pointcloud_open_settings_get_cache_size               = 0;
rdb_pointcloud_open_settings_get_cache_size_64_type            rdb_pointcloud_open_settings_get_cache_size_64            = 0;
rdb_pointcloud_open_settings_get_cache_size_64_cf_type         rdb_pointcloud_open_settings_get_cache_size_64_cf         = 0;
rdb_pointcloud_open_settings_get_cache_size_cf_type            rdb_pointcloud_open_settings_get_cache_size_cf            = 0;
rdb_pointcloud_open_settings_json_load_type                    rdb_pointcloud_open_settings_json_load                    = 0;
rdb_pointcloud_open_settings_json_load_cf_type                 rdb_pointcloud_open_settings_json_load_cf                 = 0;
rdb_pointcloud_open_settings_json_save_type                    rdb_pointcloud_open_settings_json_save                    = 0;
rdb_pointcloud_open_settings_json_save_cf_type                 rdb_pointcloud_open_settings_json_save_cf                 = 0;
rdb_pointcloud_open_settings_new_type                          rdb_pointcloud_open_settings_new                          = 0;
rdb_pointcloud_open_settings_new_cf_type                       rdb_pointcloud_open_settings_new_cf                       = 0;
rdb_pointcloud_open_settings_set_cache_size_type               rdb_pointcloud_open_settings_set_cache_size               = 0;
rdb_pointcloud_open_settings_set_cache_size_64_type            rdb_pointcloud_open_settings_set_cache_size_64            = 0;
rdb_pointcloud_open_settings_set_cache_size_64_cf_type         rdb_pointcloud_open_settings_set_cache_size_64_cf         = 0;
rdb_pointcloud_open_settings_set_cache_size_cf_type            rdb_pointcloud_open_settings_set_cache_size_cf            = 0;
rdb_pointcloud_point_attribute_data_type_type                  rdb_pointcloud_point_attribute_data_type                  = 0;
rdb_pointcloud_point_attribute_data_type_cf_type               rdb_pointcloud_point_attribute_data_type_cf               = 0;
rdb_pointcloud_point_attribute_decode_descriptor_type          rdb_pointcloud_point_attribute_decode_descriptor          = 0;
rdb_pointcloud_point_attribute_delete_type                     rdb_pointcloud_point_attribute_delete                     = 0;
rdb_pointcloud_point_attribute_delete_cf_type                  rdb_pointcloud_point_attribute_delete_cf                  = 0;
rdb_pointcloud_point_attribute_get_compression_options_type    rdb_pointcloud_point_attribute_get_compression_options    = 0;
rdb_pointcloud_point_attribute_get_compression_options_cf_type rdb_pointcloud_point_attribute_get_compression_options_cf = 0;
rdb_pointcloud_point_attribute_get_default_value_type          rdb_pointcloud_point_attribute_get_default_value          = 0;
rdb_pointcloud_point_attribute_get_default_value_cf_type       rdb_pointcloud_point_attribute_get_default_value_cf       = 0;
rdb_pointcloud_point_attribute_get_description_type            rdb_pointcloud_point_attribute_get_description            = 0;
rdb_pointcloud_point_attribute_get_description_cf_type         rdb_pointcloud_point_attribute_get_description_cf         = 0;
rdb_pointcloud_point_attribute_get_invalid_value_type          rdb_pointcloud_point_attribute_get_invalid_value          = 0;
rdb_pointcloud_point_attribute_get_invalid_value_cf_type       rdb_pointcloud_point_attribute_get_invalid_value_cf       = 0;
rdb_pointcloud_point_attribute_get_length_type                 rdb_pointcloud_point_attribute_get_length                 = 0;
rdb_pointcloud_point_attribute_get_length_cf_type              rdb_pointcloud_point_attribute_get_length_cf              = 0;
rdb_pointcloud_point_attribute_get_length_u32_type             rdb_pointcloud_point_attribute_get_length_u32             = 0;
rdb_pointcloud_point_attribute_get_length_u32_cf_type          rdb_pointcloud_point_attribute_get_length_u32_cf          = 0;
rdb_pointcloud_point_attribute_get_lod_settings_type           rdb_pointcloud_point_attribute_get_lod_settings           = 0;
rdb_pointcloud_point_attribute_get_lod_settings_cf_type        rdb_pointcloud_point_attribute_get_lod_settings_cf        = 0;
rdb_pointcloud_point_attribute_get_maximum_value_type          rdb_pointcloud_point_attribute_get_maximum_value          = 0;
rdb_pointcloud_point_attribute_get_maximum_value_cf_type       rdb_pointcloud_point_attribute_get_maximum_value_cf       = 0;
rdb_pointcloud_point_attribute_get_minimum_value_type          rdb_pointcloud_point_attribute_get_minimum_value          = 0;
rdb_pointcloud_point_attribute_get_minimum_value_cf_type       rdb_pointcloud_point_attribute_get_minimum_value_cf       = 0;
rdb_pointcloud_point_attribute_get_name_type                   rdb_pointcloud_point_attribute_get_name                   = 0;
rdb_pointcloud_point_attribute_get_name_cf_type                rdb_pointcloud_point_attribute_get_name_cf                = 0;
rdb_pointcloud_point_attribute_get_named_values_type           rdb_pointcloud_point_attribute_get_named_values           = 0;
rdb_pointcloud_point_attribute_get_named_values_cf_type        rdb_pointcloud_point_attribute_get_named_values_cf        = 0;
rdb_pointcloud_point_attribute_get_resolution_type             rdb_pointcloud_point_attribute_get_resolution             = 0;
rdb_pointcloud_point_attribute_get_resolution_cf_type          rdb_pointcloud_point_attribute_get_resolution_cf          = 0;
rdb_pointcloud_point_attribute_get_scale_factor_type           rdb_pointcloud_point_attribute_get_scale_factor           = 0;
rdb_pointcloud_point_attribute_get_scale_factor_cf_type        rdb_pointcloud_point_attribute_get_scale_factor_cf        = 0;
rdb_pointcloud_point_attribute_get_storage_class_type          rdb_pointcloud_point_attribute_get_storage_class          = 0;
rdb_pointcloud_point_attribute_get_storage_class_cf_type       rdb_pointcloud_point_attribute_get_storage_class_cf       = 0;
rdb_pointcloud_point_attribute_get_tags_type                   rdb_pointcloud_point_attribute_get_tags                   = 0;
rdb_pointcloud_point_attribute_get_tags_cf_type                rdb_pointcloud_point_attribute_get_tags_cf                = 0;
rdb_pointcloud_point_attribute_get_title_type                  rdb_pointcloud_point_attribute_get_title                  = 0;
rdb_pointcloud_point_attribute_get_title_cf_type               rdb_pointcloud_point_attribute_get_title_cf               = 0;
rdb_pointcloud_point_attribute_get_unit_symbol_type            rdb_pointcloud_point_attribute_get_unit_symbol            = 0;
rdb_pointcloud_point_attribute_get_unit_symbol_cf_type         rdb_pointcloud_point_attribute_get_unit_symbol_cf         = 0;
rdb_pointcloud_point_attribute_json_load_type                  rdb_pointcloud_point_attribute_json_load                  = 0;
rdb_pointcloud_point_attribute_json_load_cf_type               rdb_pointcloud_point_attribute_json_load_cf               = 0;
rdb_pointcloud_point_attribute_json_save_type                  rdb_pointcloud_point_attribute_json_save                  = 0;
rdb_pointcloud_point_attribute_json_save_cf_type               rdb_pointcloud_point_attribute_json_save_cf               = 0;
rdb_pointcloud_point_attribute_new_type                        rdb_pointcloud_point_attribute_new                        = 0;
rdb_pointcloud_point_attribute_new_cf_type                     rdb_pointcloud_point_attribute_new_cf                     = 0;
rdb_pointcloud_point_attribute_set_compression_options_type    rdb_pointcloud_point_attribute_set_compression_options    = 0;
rdb_pointcloud_point_attribute_set_compression_options_cf_type rdb_pointcloud_point_attribute_set_compression_options_cf = 0;
rdb_pointcloud_point_attribute_set_default_value_type          rdb_pointcloud_point_attribute_set_default_value          = 0;
rdb_pointcloud_point_attribute_set_default_value_cf_type       rdb_pointcloud_point_attribute_set_default_value_cf       = 0;
rdb_pointcloud_point_attribute_set_description_type            rdb_pointcloud_point_attribute_set_description            = 0;
rdb_pointcloud_point_attribute_set_description_cf_type         rdb_pointcloud_point_attribute_set_description_cf         = 0;
rdb_pointcloud_point_attribute_set_invalid_value_type          rdb_pointcloud_point_attribute_set_invalid_value          = 0;
rdb_pointcloud_point_attribute_set_invalid_value_cf_type       rdb_pointcloud_point_attribute_set_invalid_value_cf       = 0;
rdb_pointcloud_point_attribute_set_length_type                 rdb_pointcloud_point_attribute_set_length                 = 0;
rdb_pointcloud_point_attribute_set_length_cf_type              rdb_pointcloud_point_attribute_set_length_cf              = 0;
rdb_pointcloud_point_attribute_set_length_u32_type             rdb_pointcloud_point_attribute_set_length_u32             = 0;
rdb_pointcloud_point_attribute_set_length_u32_cf_type          rdb_pointcloud_point_attribute_set_length_u32_cf          = 0;
rdb_pointcloud_point_attribute_set_lod_settings_type           rdb_pointcloud_point_attribute_set_lod_settings           = 0;
rdb_pointcloud_point_attribute_set_lod_settings_cf_type        rdb_pointcloud_point_attribute_set_lod_settings_cf        = 0;
rdb_pointcloud_point_attribute_set_maximum_value_type          rdb_pointcloud_point_attribute_set_maximum_value          = 0;
rdb_pointcloud_point_attribute_set_maximum_value_cf_type       rdb_pointcloud_point_attribute_set_maximum_value_cf       = 0;
rdb_pointcloud_point_attribute_set_minimum_value_type          rdb_pointcloud_point_attribute_set_minimum_value          = 0;
rdb_pointcloud_point_attribute_set_minimum_value_cf_type       rdb_pointcloud_point_attribute_set_minimum_value_cf       = 0;
rdb_pointcloud_point_attribute_set_name_type                   rdb_pointcloud_point_attribute_set_name                   = 0;
rdb_pointcloud_point_attribute_set_name_cf_type                rdb_pointcloud_point_attribute_set_name_cf                = 0;
rdb_pointcloud_point_attribute_set_named_values_type           rdb_pointcloud_point_attribute_set_named_values           = 0;
rdb_pointcloud_point_attribute_set_named_values_cf_type        rdb_pointcloud_point_attribute_set_named_values_cf        = 0;
rdb_pointcloud_point_attribute_set_resolution_type             rdb_pointcloud_point_attribute_set_resolution             = 0;
rdb_pointcloud_point_attribute_set_resolution_cf_type          rdb_pointcloud_point_attribute_set_resolution_cf          = 0;
rdb_pointcloud_point_attribute_set_scale_factor_type           rdb_pointcloud_point_attribute_set_scale_factor           = 0;
rdb_pointcloud_point_attribute_set_scale_factor_cf_type        rdb_pointcloud_point_attribute_set_scale_factor_cf        = 0;
rdb_pointcloud_point_attribute_set_storage_class_type          rdb_pointcloud_point_attribute_set_storage_class          = 0;
rdb_pointcloud_point_attribute_set_storage_class_cf_type       rdb_pointcloud_point_attribute_set_storage_class_cf       = 0;
rdb_pointcloud_point_attribute_set_tags_type                   rdb_pointcloud_point_attribute_set_tags                   = 0;
rdb_pointcloud_point_attribute_set_tags_cf_type                rdb_pointcloud_point_attribute_set_tags_cf                = 0;
rdb_pointcloud_point_attribute_set_title_type                  rdb_pointcloud_point_attribute_set_title                  = 0;
rdb_pointcloud_point_attribute_set_title_cf_type               rdb_pointcloud_point_attribute_set_title_cf               = 0;
rdb_pointcloud_point_attribute_set_unit_symbol_type            rdb_pointcloud_point_attribute_set_unit_symbol            = 0;
rdb_pointcloud_point_attribute_set_unit_symbol_cf_type         rdb_pointcloud_point_attribute_set_unit_symbol_cf         = 0;
rdb_pointcloud_point_attributes_add_type                       rdb_pointcloud_point_attributes_add                       = 0;
rdb_pointcloud_point_attributes_add_default_type               rdb_pointcloud_point_attributes_add_default               = 0;
rdb_pointcloud_point_attributes_discard_type                   rdb_pointcloud_point_attributes_discard                   = 0;
rdb_pointcloud_point_attributes_duplicate_type                 rdb_pointcloud_point_attributes_duplicate                 = 0;
rdb_pointcloud_point_attributes_exists_type                    rdb_pointcloud_point_attributes_exists                    = 0;
rdb_pointcloud_point_attributes_get_type                       rdb_pointcloud_point_attributes_get                       = 0;
rdb_pointcloud_point_attributes_get_default_type               rdb_pointcloud_point_attributes_get_default               = 0;
rdb_pointcloud_point_attributes_get_merged_type                rdb_pointcloud_point_attributes_get_merged                = 0;
rdb_pointcloud_point_attributes_get_merged_all_type            rdb_pointcloud_point_attributes_get_merged_all            = 0;
rdb_pointcloud_point_attributes_group_type                     rdb_pointcloud_point_attributes_group                     = 0;
rdb_pointcloud_point_attributes_group_default_type             rdb_pointcloud_point_attributes_group_default             = 0;
rdb_pointcloud_point_attributes_list_type                      rdb_pointcloud_point_attributes_list                      = 0;
rdb_pointcloud_point_attributes_list_default_type              rdb_pointcloud_point_attributes_list_default              = 0;
rdb_pointcloud_point_attributes_list_filtered_type             rdb_pointcloud_point_attributes_list_filtered             = 0;
rdb_pointcloud_point_attributes_point_id_name_type             rdb_pointcloud_point_attributes_point_id_name             = 0;
rdb_pointcloud_point_attributes_point_id_type_type             rdb_pointcloud_point_attributes_point_id_type             = 0;
rdb_pointcloud_point_attributes_point_id_unit_type             rdb_pointcloud_point_attributes_point_id_unit             = 0;
rdb_pointcloud_point_attributes_primary_attribute_name_type    rdb_pointcloud_point_attributes_primary_attribute_name    = 0;
rdb_pointcloud_point_attributes_put_type                       rdb_pointcloud_point_attributes_put                       = 0;
rdb_pointcloud_point_attributes_remove_type                    rdb_pointcloud_point_attributes_remove                    = 0;
rdb_pointcloud_query_fill_bind_type                            rdb_pointcloud_query_fill_bind                            = 0;
rdb_pointcloud_query_fill_delete_type                          rdb_pointcloud_query_fill_delete                          = 0;
rdb_pointcloud_query_fill_new_type                             rdb_pointcloud_query_fill_new                             = 0;
rdb_pointcloud_query_fill_next_type                            rdb_pointcloud_query_fill_next                            = 0;
rdb_pointcloud_query_fill_nodes_new_type                       rdb_pointcloud_query_fill_nodes_new                       = 0;
rdb_pointcloud_query_fill_progress_type                        rdb_pointcloud_query_fill_progress                        = 0;
rdb_pointcloud_query_insert_bind_type                          rdb_pointcloud_query_insert_bind                          = 0;
rdb_pointcloud_query_insert_bind_id_type                       rdb_pointcloud_query_insert_bind_id                       = 0;
rdb_pointcloud_query_insert_delete_type                        rdb_pointcloud_query_insert_delete                        = 0;
rdb_pointcloud_query_insert_load_multi_pass_settings_type      rdb_pointcloud_query_insert_load_multi_pass_settings      = 0;
rdb_pointcloud_query_insert_new_type                           rdb_pointcloud_query_insert_new                           = 0;
rdb_pointcloud_query_insert_next_type                          rdb_pointcloud_query_insert_next                          = 0;
rdb_pointcloud_query_invert_attribute_type                     rdb_pointcloud_query_invert_attribute                     = 0;
rdb_pointcloud_query_invert_delete_type                        rdb_pointcloud_query_invert_delete                        = 0;
rdb_pointcloud_query_invert_new_type                           rdb_pointcloud_query_invert_new                           = 0;
rdb_pointcloud_query_invert_next_type                          rdb_pointcloud_query_invert_next                          = 0;
rdb_pointcloud_query_invert_nodes_new_type                     rdb_pointcloud_query_invert_nodes_new                     = 0;
rdb_pointcloud_query_invert_progress_type                      rdb_pointcloud_query_invert_progress                      = 0;
rdb_pointcloud_query_remove_bind_type                          rdb_pointcloud_query_remove_bind                          = 0;
rdb_pointcloud_query_remove_delete_type                        rdb_pointcloud_query_remove_delete                        = 0;
rdb_pointcloud_query_remove_new_type                           rdb_pointcloud_query_remove_new                           = 0;
rdb_pointcloud_query_remove_next_type                          rdb_pointcloud_query_remove_next                          = 0;
rdb_pointcloud_query_select_bind_type                          rdb_pointcloud_query_select_bind                          = 0;
rdb_pointcloud_query_select_delete_type                        rdb_pointcloud_query_select_delete                        = 0;
rdb_pointcloud_query_select_new_type                           rdb_pointcloud_query_select_new                           = 0;
rdb_pointcloud_query_select_next_type                          rdb_pointcloud_query_select_next                          = 0;
rdb_pointcloud_query_select_nodes_new_type                     rdb_pointcloud_query_select_nodes_new                     = 0;
rdb_pointcloud_query_select_progress_type                      rdb_pointcloud_query_select_progress                      = 0;
rdb_pointcloud_query_stat_delete_type                          rdb_pointcloud_query_stat_delete                          = 0;
rdb_pointcloud_query_stat_index_type                           rdb_pointcloud_query_stat_index                           = 0;
rdb_pointcloud_query_stat_index_filter_type                    rdb_pointcloud_query_stat_index_filter                    = 0;
rdb_pointcloud_query_stat_maximum_type                         rdb_pointcloud_query_stat_maximum                         = 0;
rdb_pointcloud_query_stat_maximum_cleaned_type                 rdb_pointcloud_query_stat_maximum_cleaned                 = 0;
rdb_pointcloud_query_stat_minimum_type                         rdb_pointcloud_query_stat_minimum                         = 0;
rdb_pointcloud_query_stat_minimum_cleaned_type                 rdb_pointcloud_query_stat_minimum_cleaned                 = 0;
rdb_pointcloud_query_stat_new_type                             rdb_pointcloud_query_stat_new                             = 0;
rdb_pointcloud_query_stat_revision_type                        rdb_pointcloud_query_stat_revision                        = 0;
rdb_pointcloud_query_update_bind_type                          rdb_pointcloud_query_update_bind                          = 0;
rdb_pointcloud_query_update_delete_type                        rdb_pointcloud_query_update_delete                        = 0;
rdb_pointcloud_query_update_new_type                           rdb_pointcloud_query_update_new                           = 0;
rdb_pointcloud_query_update_next_type                          rdb_pointcloud_query_update_next                          = 0;
rdb_pointcloud_transaction_begin_type                          rdb_pointcloud_transaction_begin                          = 0;
rdb_pointcloud_transaction_commit_type                         rdb_pointcloud_transaction_commit                         = 0;
rdb_pointcloud_transaction_commit_with_progress_type           rdb_pointcloud_transaction_commit_with_progress           = 0;
rdb_pointcloud_transaction_commit_with_signature_type          rdb_pointcloud_transaction_commit_with_signature          = 0;
rdb_pointcloud_transaction_current_type                        rdb_pointcloud_transaction_current                        = 0;
rdb_pointcloud_transaction_delete_type                         rdb_pointcloud_transaction_delete                         = 0;
rdb_pointcloud_transaction_details_type                        rdb_pointcloud_transaction_details                        = 0;
rdb_pointcloud_transaction_discard_type                        rdb_pointcloud_transaction_discard                        = 0;
rdb_pointcloud_transaction_get_agent_type                      rdb_pointcloud_transaction_get_agent                      = 0;
rdb_pointcloud_transaction_get_comments_type                   rdb_pointcloud_transaction_get_comments                   = 0;
rdb_pointcloud_transaction_get_id_type                         rdb_pointcloud_transaction_get_id                         = 0;
rdb_pointcloud_transaction_get_rdb_type                        rdb_pointcloud_transaction_get_rdb                        = 0;
rdb_pointcloud_transaction_get_settings_type                   rdb_pointcloud_transaction_get_settings                   = 0;
rdb_pointcloud_transaction_get_start_type                      rdb_pointcloud_transaction_get_start                      = 0;
rdb_pointcloud_transaction_get_stop_type                       rdb_pointcloud_transaction_get_stop                       = 0;
rdb_pointcloud_transaction_get_title_type                      rdb_pointcloud_transaction_get_title                      = 0;
rdb_pointcloud_transaction_list_type                           rdb_pointcloud_transaction_list                           = 0;
rdb_pointcloud_transaction_new_type                            rdb_pointcloud_transaction_new                            = 0;
rdb_pointcloud_transaction_pending_type                        rdb_pointcloud_transaction_pending                        = 0;
rdb_pointcloud_transaction_restore_type                        rdb_pointcloud_transaction_restore                        = 0;
rdb_pointcloud_transaction_rollback_type                       rdb_pointcloud_transaction_rollback                       = 0;
rdb_pointcloud_transaction_size_type                           rdb_pointcloud_transaction_size                           = 0;
rdb_result_delete_cf_type                                      rdb_result_delete_cf                                      = 0;
rdb_result_get_error_code_cf_type                              rdb_result_get_error_code_cf                              = 0;
rdb_result_get_error_details_cf_type                           rdb_result_get_error_details_cf                           = 0;
rdb_result_get_error_text_cf_type                              rdb_result_get_error_text_cf                              = 0;
rdb_string_delete_cf_type                                      rdb_string_delete_cf                                      = 0;

//---< LIBRARY HANDLING FUNCTIONS >---------------------------------------------

static LibraryHandle rdb_library_handle = 0;

RDBString rdb_library_load(RDBString filename/*=0*/)
{
    if (rdb_library_handle != 0)
    {
        return 0; // library already loaded
    }

    rdb_library_handle = library_load(
        filename ? filename :
#if defined(_WIN32) || defined(__WIN32__) || defined(__WINDOWS__)
        "rdblib.dll"
#elif defined(__APPLE__) && defined(__MACH__)
        "librdb.2.dylib"
#elif defined(__linux__)
        "librdb.so.2"
#endif
    );

    if (rdb_library_handle == 0)
    {
        return "Failed to load RDB library.";
    }

    if ((rdb_context_delete                                        = (rdb_context_delete_type)                                       library_resolve(rdb_library_handle, "rdb_context_delete"))                                        == 0) return "Failed to load RDB function \"rdb_context_delete()\".";
    if ((rdb_context_get_last_error                                = (rdb_context_get_last_error_type)                               library_resolve(rdb_library_handle, "rdb_context_get_last_error"))                                == 0) return "Failed to load RDB function \"rdb_context_get_last_error()\".";
    if ((rdb_context_new                                           = (rdb_context_new_type)                                          library_resolve(rdb_library_handle, "rdb_context_new"))                                           == 0) return "Failed to load RDB function \"rdb_context_new()\".";
    if ((rdb_context_new_with_api                                  = (rdb_context_new_with_api_type)                                 library_resolve(rdb_library_handle, "rdb_context_new_with_api"))                                  == 0) return "Failed to load RDB function \"rdb_context_new_with_api()\".";
    if ((rdb_database_file_type_check                              = (rdb_database_file_type_check_type)                             library_resolve(rdb_library_handle, "rdb_database_file_type_check"))                              == 0) return "Failed to load RDB function \"rdb_database_file_type_check()\".";
    if ((rdb_database_file_type_suffix                             = (rdb_database_file_type_suffix_type)                            library_resolve(rdb_library_handle, "rdb_database_file_type_suffix"))                             == 0) return "Failed to load RDB function \"rdb_database_file_type_suffix()\".";
    if ((rdb_database_file_type_title                              = (rdb_database_file_type_title_type)                             library_resolve(rdb_library_handle, "rdb_database_file_type_title"))                              == 0) return "Failed to load RDB function \"rdb_database_file_type_title()\".";
    if ((rdb_factory_set_authentication_function                   = (rdb_factory_set_authentication_function_type)                  library_resolve(rdb_library_handle, "rdb_factory_set_authentication_function"))                   == 0) return "Failed to load RDB function \"rdb_factory_set_authentication_function()\".";
    if ((rdb_library_filename                                      = (rdb_library_filename_type)                                     library_resolve(rdb_library_handle, "rdb_library_filename"))                                      == 0) return "Failed to load RDB function \"rdb_library_filename()\".";
    if ((rdb_library_license                                       = (rdb_library_license_type)                                      library_resolve(rdb_library_handle, "rdb_library_license"))                                       == 0) return "Failed to load RDB function \"rdb_library_license()\".";
    if ((rdb_library_name                                          = (rdb_library_name_type)                                         library_resolve(rdb_library_handle, "rdb_library_name"))                                          == 0) return "Failed to load RDB function \"rdb_library_name()\".";
    if ((rdb_library_version                                       = (rdb_library_version_type)                                      library_resolve(rdb_library_handle, "rdb_library_version"))                                       == 0) return "Failed to load RDB function \"rdb_library_version()\".";
    if ((rdb_object_lock_cf                                        = (rdb_object_lock_cf_type)                                       library_resolve(rdb_library_handle, "rdb_object_lock_cf"))                                        == 0) return "Failed to load RDB function \"rdb_object_lock_cf()\".";
    if ((rdb_object_unlock_cf                                      = (rdb_object_unlock_cf_type)                                     library_resolve(rdb_library_handle, "rdb_object_unlock_cf"))                                      == 0) return "Failed to load RDB function \"rdb_object_unlock_cf()\".";
    if ((rdb_pointcloud_changelog_append_log_message               = (rdb_pointcloud_changelog_append_log_message_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_changelog_append_log_message"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_changelog_append_log_message()\".";
    if ((rdb_pointcloud_changelog_export_to_textfile               = (rdb_pointcloud_changelog_export_to_textfile_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_changelog_export_to_textfile"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_changelog_export_to_textfile()\".";
    if ((rdb_pointcloud_changelog_import_from_database             = (rdb_pointcloud_changelog_import_from_database_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_changelog_import_from_database"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_changelog_import_from_database()\".";
    if ((rdb_pointcloud_changelog_verify_log_entry_cf              = (rdb_pointcloud_changelog_verify_log_entry_cf_type)             library_resolve(rdb_library_handle, "rdb_pointcloud_changelog_verify_log_entry_cf"))              == 0) return "Failed to load RDB function \"rdb_pointcloud_changelog_verify_log_entry_cf()\".";
    if ((rdb_pointcloud_clear_cache                                = (rdb_pointcloud_clear_cache_type)                               library_resolve(rdb_library_handle, "rdb_pointcloud_clear_cache"))                                == 0) return "Failed to load RDB function \"rdb_pointcloud_clear_cache()\".";
    if ((rdb_pointcloud_close                                      = (rdb_pointcloud_close_type)                                     library_resolve(rdb_library_handle, "rdb_pointcloud_close"))                                      == 0) return "Failed to load RDB function \"rdb_pointcloud_close()\".";
    if ((rdb_pointcloud_create                                     = (rdb_pointcloud_create_type)                                    library_resolve(rdb_library_handle, "rdb_pointcloud_create"))                                     == 0) return "Failed to load RDB function \"rdb_pointcloud_create()\".";
    if ((rdb_pointcloud_create_settings_delete                     = (rdb_pointcloud_create_settings_delete_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_delete"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_delete()\".";
    if ((rdb_pointcloud_create_settings_delete_cf                  = (rdb_pointcloud_create_settings_delete_cf_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_delete_cf"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_delete_cf()\".";
    if ((rdb_pointcloud_create_settings_get_cache_size             = (rdb_pointcloud_create_settings_get_cache_size_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_cache_size"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_cache_size()\".";
    if ((rdb_pointcloud_create_settings_get_cache_size_64          = (rdb_pointcloud_create_settings_get_cache_size_64_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_cache_size_64"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_cache_size_64()\".";
    if ((rdb_pointcloud_create_settings_get_cache_size_64_cf       = (rdb_pointcloud_create_settings_get_cache_size_64_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_cache_size_64_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_cache_size_64_cf()\".";
    if ((rdb_pointcloud_create_settings_get_cache_size_cf          = (rdb_pointcloud_create_settings_get_cache_size_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_cache_size_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_cache_size_cf()\".";
    if ((rdb_pointcloud_create_settings_get_chunk_mode             = (rdb_pointcloud_create_settings_get_chunk_mode_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_chunk_mode"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_chunk_mode()\".";
    if ((rdb_pointcloud_create_settings_get_chunk_mode_cf          = (rdb_pointcloud_create_settings_get_chunk_mode_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_chunk_mode_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_chunk_mode_cf()\".";
    if ((rdb_pointcloud_create_settings_get_chunk_size             = (rdb_pointcloud_create_settings_get_chunk_size_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_chunk_size"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_chunk_size()\".";
    if ((rdb_pointcloud_create_settings_get_chunk_size_cf          = (rdb_pointcloud_create_settings_get_chunk_size_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_chunk_size_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_chunk_size_cf()\".";
    if ((rdb_pointcloud_create_settings_get_chunk_size_lod         = (rdb_pointcloud_create_settings_get_chunk_size_lod_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_chunk_size_lod"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_chunk_size_lod()\".";
    if ((rdb_pointcloud_create_settings_get_chunk_size_lod_cf      = (rdb_pointcloud_create_settings_get_chunk_size_lod_cf_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_chunk_size_lod_cf"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_chunk_size_lod_cf()\".";
    if ((rdb_pointcloud_create_settings_get_compression_level      = (rdb_pointcloud_create_settings_get_compression_level_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_compression_level"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_compression_level()\".";
    if ((rdb_pointcloud_create_settings_get_compression_level_cf   = (rdb_pointcloud_create_settings_get_compression_level_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_compression_level_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_compression_level_cf()\".";
    if ((rdb_pointcloud_create_settings_get_lod_mode               = (rdb_pointcloud_create_settings_get_lod_mode_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_lod_mode"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_lod_mode()\".";
    if ((rdb_pointcloud_create_settings_get_lod_mode_cf            = (rdb_pointcloud_create_settings_get_lod_mode_cf_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_lod_mode_cf"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_lod_mode_cf()\".";
    if ((rdb_pointcloud_create_settings_get_optimize_point_id      = (rdb_pointcloud_create_settings_get_optimize_point_id_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_optimize_point_id"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_optimize_point_id()\".";
    if ((rdb_pointcloud_create_settings_get_optimize_point_id_cf   = (rdb_pointcloud_create_settings_get_optimize_point_id_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_optimize_point_id_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_optimize_point_id_cf()\".";
    if ((rdb_pointcloud_create_settings_get_point_insert_mode      = (rdb_pointcloud_create_settings_get_point_insert_mode_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_point_insert_mode"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_point_insert_mode()\".";
    if ((rdb_pointcloud_create_settings_get_point_insert_mode_cf   = (rdb_pointcloud_create_settings_get_point_insert_mode_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_point_insert_mode_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_point_insert_mode_cf()\".";
    if ((rdb_pointcloud_create_settings_get_primary_attribute      = (rdb_pointcloud_create_settings_get_primary_attribute_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_primary_attribute"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_primary_attribute()\".";
    if ((rdb_pointcloud_create_settings_get_primary_attribute_cf   = (rdb_pointcloud_create_settings_get_primary_attribute_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_get_primary_attribute_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_get_primary_attribute_cf()\".";
    if ((rdb_pointcloud_create_settings_json_load                  = (rdb_pointcloud_create_settings_json_load_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_json_load"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_json_load()\".";
    if ((rdb_pointcloud_create_settings_json_load_cf               = (rdb_pointcloud_create_settings_json_load_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_json_load_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_json_load_cf()\".";
    if ((rdb_pointcloud_create_settings_json_save                  = (rdb_pointcloud_create_settings_json_save_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_json_save"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_json_save()\".";
    if ((rdb_pointcloud_create_settings_json_save_cf               = (rdb_pointcloud_create_settings_json_save_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_json_save_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_json_save_cf()\".";
    if ((rdb_pointcloud_create_settings_new                        = (rdb_pointcloud_create_settings_new_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_new"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_new()\".";
    if ((rdb_pointcloud_create_settings_new_cf                     = (rdb_pointcloud_create_settings_new_cf_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_new_cf"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_new_cf()\".";
    if ((rdb_pointcloud_create_settings_set_cache_size             = (rdb_pointcloud_create_settings_set_cache_size_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_cache_size"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_cache_size()\".";
    if ((rdb_pointcloud_create_settings_set_cache_size_64          = (rdb_pointcloud_create_settings_set_cache_size_64_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_cache_size_64"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_cache_size_64()\".";
    if ((rdb_pointcloud_create_settings_set_cache_size_64_cf       = (rdb_pointcloud_create_settings_set_cache_size_64_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_cache_size_64_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_cache_size_64_cf()\".";
    if ((rdb_pointcloud_create_settings_set_cache_size_cf          = (rdb_pointcloud_create_settings_set_cache_size_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_cache_size_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_cache_size_cf()\".";
    if ((rdb_pointcloud_create_settings_set_chunk_mode             = (rdb_pointcloud_create_settings_set_chunk_mode_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_chunk_mode"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_chunk_mode()\".";
    if ((rdb_pointcloud_create_settings_set_chunk_mode_cf          = (rdb_pointcloud_create_settings_set_chunk_mode_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_chunk_mode_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_chunk_mode_cf()\".";
    if ((rdb_pointcloud_create_settings_set_chunk_size             = (rdb_pointcloud_create_settings_set_chunk_size_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_chunk_size"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_chunk_size()\".";
    if ((rdb_pointcloud_create_settings_set_chunk_size_cf          = (rdb_pointcloud_create_settings_set_chunk_size_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_chunk_size_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_chunk_size_cf()\".";
    if ((rdb_pointcloud_create_settings_set_chunk_size_lod         = (rdb_pointcloud_create_settings_set_chunk_size_lod_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_chunk_size_lod"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_chunk_size_lod()\".";
    if ((rdb_pointcloud_create_settings_set_chunk_size_lod_cf      = (rdb_pointcloud_create_settings_set_chunk_size_lod_cf_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_chunk_size_lod_cf"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_chunk_size_lod_cf()\".";
    if ((rdb_pointcloud_create_settings_set_compression_level      = (rdb_pointcloud_create_settings_set_compression_level_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_compression_level"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_compression_level()\".";
    if ((rdb_pointcloud_create_settings_set_compression_level_cf   = (rdb_pointcloud_create_settings_set_compression_level_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_compression_level_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_compression_level_cf()\".";
    if ((rdb_pointcloud_create_settings_set_lod_mode               = (rdb_pointcloud_create_settings_set_lod_mode_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_lod_mode"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_lod_mode()\".";
    if ((rdb_pointcloud_create_settings_set_lod_mode_cf            = (rdb_pointcloud_create_settings_set_lod_mode_cf_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_lod_mode_cf"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_lod_mode_cf()\".";
    if ((rdb_pointcloud_create_settings_set_optimize_point_id      = (rdb_pointcloud_create_settings_set_optimize_point_id_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_optimize_point_id"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_optimize_point_id()\".";
    if ((rdb_pointcloud_create_settings_set_optimize_point_id_cf   = (rdb_pointcloud_create_settings_set_optimize_point_id_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_optimize_point_id_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_optimize_point_id_cf()\".";
    if ((rdb_pointcloud_create_settings_set_point_insert_mode      = (rdb_pointcloud_create_settings_set_point_insert_mode_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_point_insert_mode"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_point_insert_mode()\".";
    if ((rdb_pointcloud_create_settings_set_point_insert_mode_cf   = (rdb_pointcloud_create_settings_set_point_insert_mode_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_point_insert_mode_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_point_insert_mode_cf()\".";
    if ((rdb_pointcloud_create_settings_set_primary_attribute      = (rdb_pointcloud_create_settings_set_primary_attribute_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_primary_attribute"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_primary_attribute()\".";
    if ((rdb_pointcloud_create_settings_set_primary_attribute_cf   = (rdb_pointcloud_create_settings_set_primary_attribute_cf_type)  library_resolve(rdb_library_handle, "rdb_pointcloud_create_settings_set_primary_attribute_cf"))   == 0) return "Failed to load RDB function \"rdb_pointcloud_create_settings_set_primary_attribute_cf()\".";
    if ((rdb_pointcloud_create_with_schema                         = (rdb_pointcloud_create_with_schema_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_create_with_schema"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_create_with_schema()\".";
    if ((rdb_pointcloud_delete                                     = (rdb_pointcloud_delete_type)                                    library_resolve(rdb_library_handle, "rdb_pointcloud_delete"))                                     == 0) return "Failed to load RDB function \"rdb_pointcloud_delete()\".";
    if ((rdb_pointcloud_get_uuid                                   = (rdb_pointcloud_get_uuid_type)                                  library_resolve(rdb_library_handle, "rdb_pointcloud_get_uuid"))                                   == 0) return "Failed to load RDB function \"rdb_pointcloud_get_uuid()\".";
    if ((rdb_pointcloud_graph_node_delete                          = (rdb_pointcloud_graph_node_delete_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_graph_node_delete"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_graph_node_delete()\".";
    if ((rdb_pointcloud_graph_node_new                             = (rdb_pointcloud_graph_node_new_type)                            library_resolve(rdb_library_handle, "rdb_pointcloud_graph_node_new"))                             == 0) return "Failed to load RDB function \"rdb_pointcloud_graph_node_new()\".";
    if ((rdb_pointcloud_inspect                                    = (rdb_pointcloud_inspect_type)                                   library_resolve(rdb_library_handle, "rdb_pointcloud_inspect"))                                    == 0) return "Failed to load RDB function \"rdb_pointcloud_inspect()\".";
    if ((rdb_pointcloud_is_empty                                   = (rdb_pointcloud_is_empty_type)                                  library_resolve(rdb_library_handle, "rdb_pointcloud_is_empty"))                                   == 0) return "Failed to load RDB function \"rdb_pointcloud_is_empty()\".";
    if ((rdb_pointcloud_is_open                                    = (rdb_pointcloud_is_open_type)                                   library_resolve(rdb_library_handle, "rdb_pointcloud_is_open"))                                    == 0) return "Failed to load RDB function \"rdb_pointcloud_is_open()\".";
    if ((rdb_pointcloud_management_finalize                        = (rdb_pointcloud_management_finalize_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_management_finalize"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_management_finalize()\".";
    if ((rdb_pointcloud_management_get_chunk_size_lod              = (rdb_pointcloud_management_get_chunk_size_lod_type)             library_resolve(rdb_library_handle, "rdb_pointcloud_management_get_chunk_size_lod"))              == 0) return "Failed to load RDB function \"rdb_pointcloud_management_get_chunk_size_lod()\".";
    if ((rdb_pointcloud_management_get_lod_mode                    = (rdb_pointcloud_management_get_lod_mode_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_management_get_lod_mode"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_management_get_lod_mode()\".";
    if ((rdb_pointcloud_management_set_chunk_size_lod              = (rdb_pointcloud_management_set_chunk_size_lod_type)             library_resolve(rdb_library_handle, "rdb_pointcloud_management_set_chunk_size_lod"))              == 0) return "Failed to load RDB function \"rdb_pointcloud_management_set_chunk_size_lod()\".";
    if ((rdb_pointcloud_management_set_lod_mode                    = (rdb_pointcloud_management_set_lod_mode_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_management_set_lod_mode"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_management_set_lod_mode()\".";
    if ((rdb_pointcloud_management_vacuum                          = (rdb_pointcloud_management_vacuum_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_management_vacuum"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_management_vacuum()\".";
    if ((rdb_pointcloud_management_validate                        = (rdb_pointcloud_management_validate_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_management_validate"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_management_validate()\".";
    if ((rdb_pointcloud_meta_data_create_signature                 = (rdb_pointcloud_meta_data_create_signature_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_create_signature"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_create_signature()\".";
    if ((rdb_pointcloud_meta_data_exists                           = (rdb_pointcloud_meta_data_exists_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_exists"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_exists()\".";
    if ((rdb_pointcloud_meta_data_get                              = (rdb_pointcloud_meta_data_get_type)                             library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_get"))                              == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_get()\".";
    if ((rdb_pointcloud_meta_data_list                             = (rdb_pointcloud_meta_data_list_type)                            library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_list"))                             == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_list()\".";
    if ((rdb_pointcloud_meta_data_remove                           = (rdb_pointcloud_meta_data_remove_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_remove"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_remove()\".";
    if ((rdb_pointcloud_meta_data_set                              = (rdb_pointcloud_meta_data_set_type)                             library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_set"))                              == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_set()\".";
    if ((rdb_pointcloud_meta_data_validate                         = (rdb_pointcloud_meta_data_validate_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_validate"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_validate()\".";
    if ((rdb_pointcloud_meta_data_validate_json                    = (rdb_pointcloud_meta_data_validate_json_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_validate_json"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_validate_json()\".";
    if ((rdb_pointcloud_meta_data_verify_signature                 = (rdb_pointcloud_meta_data_verify_signature_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_meta_data_verify_signature"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_meta_data_verify_signature()\".";
    if ((rdb_pointcloud_new                                        = (rdb_pointcloud_new_type)                                       library_resolve(rdb_library_handle, "rdb_pointcloud_new"))                                        == 0) return "Failed to load RDB function \"rdb_pointcloud_new()\".";
    if ((rdb_pointcloud_open                                       = (rdb_pointcloud_open_type)                                      library_resolve(rdb_library_handle, "rdb_pointcloud_open"))                                       == 0) return "Failed to load RDB function \"rdb_pointcloud_open()\".";
    if ((rdb_pointcloud_open_settings_delete                       = (rdb_pointcloud_open_settings_delete_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_delete"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_delete()\".";
    if ((rdb_pointcloud_open_settings_delete_cf                    = (rdb_pointcloud_open_settings_delete_cf_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_delete_cf"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_delete_cf()\".";
    if ((rdb_pointcloud_open_settings_get_cache_size               = (rdb_pointcloud_open_settings_get_cache_size_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_get_cache_size"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_get_cache_size()\".";
    if ((rdb_pointcloud_open_settings_get_cache_size_64            = (rdb_pointcloud_open_settings_get_cache_size_64_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_get_cache_size_64"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_get_cache_size_64()\".";
    if ((rdb_pointcloud_open_settings_get_cache_size_64_cf         = (rdb_pointcloud_open_settings_get_cache_size_64_cf_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_get_cache_size_64_cf"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_get_cache_size_64_cf()\".";
    if ((rdb_pointcloud_open_settings_get_cache_size_cf            = (rdb_pointcloud_open_settings_get_cache_size_cf_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_get_cache_size_cf"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_get_cache_size_cf()\".";
    if ((rdb_pointcloud_open_settings_json_load                    = (rdb_pointcloud_open_settings_json_load_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_json_load"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_json_load()\".";
    if ((rdb_pointcloud_open_settings_json_load_cf                 = (rdb_pointcloud_open_settings_json_load_cf_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_json_load_cf"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_json_load_cf()\".";
    if ((rdb_pointcloud_open_settings_json_save                    = (rdb_pointcloud_open_settings_json_save_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_json_save"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_json_save()\".";
    if ((rdb_pointcloud_open_settings_json_save_cf                 = (rdb_pointcloud_open_settings_json_save_cf_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_json_save_cf"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_json_save_cf()\".";
    if ((rdb_pointcloud_open_settings_new                          = (rdb_pointcloud_open_settings_new_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_new"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_new()\".";
    if ((rdb_pointcloud_open_settings_new_cf                       = (rdb_pointcloud_open_settings_new_cf_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_new_cf"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_new_cf()\".";
    if ((rdb_pointcloud_open_settings_set_cache_size               = (rdb_pointcloud_open_settings_set_cache_size_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_set_cache_size"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_set_cache_size()\".";
    if ((rdb_pointcloud_open_settings_set_cache_size_64            = (rdb_pointcloud_open_settings_set_cache_size_64_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_set_cache_size_64"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_set_cache_size_64()\".";
    if ((rdb_pointcloud_open_settings_set_cache_size_64_cf         = (rdb_pointcloud_open_settings_set_cache_size_64_cf_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_set_cache_size_64_cf"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_set_cache_size_64_cf()\".";
    if ((rdb_pointcloud_open_settings_set_cache_size_cf            = (rdb_pointcloud_open_settings_set_cache_size_cf_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_open_settings_set_cache_size_cf"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_open_settings_set_cache_size_cf()\".";
    if ((rdb_pointcloud_point_attribute_data_type                  = (rdb_pointcloud_point_attribute_data_type_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_data_type"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_data_type()\".";
    if ((rdb_pointcloud_point_attribute_data_type_cf               = (rdb_pointcloud_point_attribute_data_type_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_data_type_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_data_type_cf()\".";
    if ((rdb_pointcloud_point_attribute_decode_descriptor          = (rdb_pointcloud_point_attribute_decode_descriptor_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_decode_descriptor"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_decode_descriptor()\".";
    if ((rdb_pointcloud_point_attribute_delete                     = (rdb_pointcloud_point_attribute_delete_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_delete"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_delete()\".";
    if ((rdb_pointcloud_point_attribute_delete_cf                  = (rdb_pointcloud_point_attribute_delete_cf_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_delete_cf"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_delete_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_compression_options    = (rdb_pointcloud_point_attribute_get_compression_options_type)   library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_compression_options"))    == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_compression_options()\".";
    if ((rdb_pointcloud_point_attribute_get_compression_options_cf = (rdb_pointcloud_point_attribute_get_compression_options_cf_type)library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_compression_options_cf")) == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_compression_options_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_default_value          = (rdb_pointcloud_point_attribute_get_default_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_default_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_default_value()\".";
    if ((rdb_pointcloud_point_attribute_get_default_value_cf       = (rdb_pointcloud_point_attribute_get_default_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_default_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_default_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_description            = (rdb_pointcloud_point_attribute_get_description_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_description"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_description()\".";
    if ((rdb_pointcloud_point_attribute_get_description_cf         = (rdb_pointcloud_point_attribute_get_description_cf_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_description_cf"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_description_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_invalid_value          = (rdb_pointcloud_point_attribute_get_invalid_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_invalid_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_invalid_value()\".";
    if ((rdb_pointcloud_point_attribute_get_invalid_value_cf       = (rdb_pointcloud_point_attribute_get_invalid_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_invalid_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_invalid_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_length                 = (rdb_pointcloud_point_attribute_get_length_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_length"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_length()\".";
    if ((rdb_pointcloud_point_attribute_get_length_cf              = (rdb_pointcloud_point_attribute_get_length_cf_type)             library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_length_cf"))              == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_length_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_length_u32             = (rdb_pointcloud_point_attribute_get_length_u32_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_length_u32"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_length_u32()\".";
    if ((rdb_pointcloud_point_attribute_get_length_u32_cf          = (rdb_pointcloud_point_attribute_get_length_u32_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_length_u32_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_length_u32_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_lod_settings           = (rdb_pointcloud_point_attribute_get_lod_settings_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_lod_settings"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_lod_settings()\".";
    if ((rdb_pointcloud_point_attribute_get_lod_settings_cf        = (rdb_pointcloud_point_attribute_get_lod_settings_cf_type)       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_lod_settings_cf"))        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_lod_settings_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_maximum_value          = (rdb_pointcloud_point_attribute_get_maximum_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_maximum_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_maximum_value()\".";
    if ((rdb_pointcloud_point_attribute_get_maximum_value_cf       = (rdb_pointcloud_point_attribute_get_maximum_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_maximum_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_maximum_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_minimum_value          = (rdb_pointcloud_point_attribute_get_minimum_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_minimum_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_minimum_value()\".";
    if ((rdb_pointcloud_point_attribute_get_minimum_value_cf       = (rdb_pointcloud_point_attribute_get_minimum_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_minimum_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_minimum_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_name                   = (rdb_pointcloud_point_attribute_get_name_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_name"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_name()\".";
    if ((rdb_pointcloud_point_attribute_get_name_cf                = (rdb_pointcloud_point_attribute_get_name_cf_type)               library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_name_cf"))                == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_name_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_named_values           = (rdb_pointcloud_point_attribute_get_named_values_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_named_values"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_named_values()\".";
    if ((rdb_pointcloud_point_attribute_get_named_values_cf        = (rdb_pointcloud_point_attribute_get_named_values_cf_type)       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_named_values_cf"))        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_named_values_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_resolution             = (rdb_pointcloud_point_attribute_get_resolution_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_resolution"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_resolution()\".";
    if ((rdb_pointcloud_point_attribute_get_resolution_cf          = (rdb_pointcloud_point_attribute_get_resolution_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_resolution_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_resolution_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_scale_factor           = (rdb_pointcloud_point_attribute_get_scale_factor_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_scale_factor"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_scale_factor()\".";
    if ((rdb_pointcloud_point_attribute_get_scale_factor_cf        = (rdb_pointcloud_point_attribute_get_scale_factor_cf_type)       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_scale_factor_cf"))        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_scale_factor_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_storage_class          = (rdb_pointcloud_point_attribute_get_storage_class_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_storage_class"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_storage_class()\".";
    if ((rdb_pointcloud_point_attribute_get_storage_class_cf       = (rdb_pointcloud_point_attribute_get_storage_class_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_storage_class_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_storage_class_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_tags                   = (rdb_pointcloud_point_attribute_get_tags_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_tags"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_tags()\".";
    if ((rdb_pointcloud_point_attribute_get_tags_cf                = (rdb_pointcloud_point_attribute_get_tags_cf_type)               library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_tags_cf"))                == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_tags_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_title                  = (rdb_pointcloud_point_attribute_get_title_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_title"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_title()\".";
    if ((rdb_pointcloud_point_attribute_get_title_cf               = (rdb_pointcloud_point_attribute_get_title_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_title_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_title_cf()\".";
    if ((rdb_pointcloud_point_attribute_get_unit_symbol            = (rdb_pointcloud_point_attribute_get_unit_symbol_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_unit_symbol"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_unit_symbol()\".";
    if ((rdb_pointcloud_point_attribute_get_unit_symbol_cf         = (rdb_pointcloud_point_attribute_get_unit_symbol_cf_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_get_unit_symbol_cf"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_get_unit_symbol_cf()\".";
    if ((rdb_pointcloud_point_attribute_json_load                  = (rdb_pointcloud_point_attribute_json_load_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_json_load"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_json_load()\".";
    if ((rdb_pointcloud_point_attribute_json_load_cf               = (rdb_pointcloud_point_attribute_json_load_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_json_load_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_json_load_cf()\".";
    if ((rdb_pointcloud_point_attribute_json_save                  = (rdb_pointcloud_point_attribute_json_save_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_json_save"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_json_save()\".";
    if ((rdb_pointcloud_point_attribute_json_save_cf               = (rdb_pointcloud_point_attribute_json_save_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_json_save_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_json_save_cf()\".";
    if ((rdb_pointcloud_point_attribute_new                        = (rdb_pointcloud_point_attribute_new_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_new"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_new()\".";
    if ((rdb_pointcloud_point_attribute_new_cf                     = (rdb_pointcloud_point_attribute_new_cf_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_new_cf"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_new_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_compression_options    = (rdb_pointcloud_point_attribute_set_compression_options_type)   library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_compression_options"))    == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_compression_options()\".";
    if ((rdb_pointcloud_point_attribute_set_compression_options_cf = (rdb_pointcloud_point_attribute_set_compression_options_cf_type)library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_compression_options_cf")) == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_compression_options_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_default_value          = (rdb_pointcloud_point_attribute_set_default_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_default_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_default_value()\".";
    if ((rdb_pointcloud_point_attribute_set_default_value_cf       = (rdb_pointcloud_point_attribute_set_default_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_default_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_default_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_description            = (rdb_pointcloud_point_attribute_set_description_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_description"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_description()\".";
    if ((rdb_pointcloud_point_attribute_set_description_cf         = (rdb_pointcloud_point_attribute_set_description_cf_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_description_cf"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_description_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_invalid_value          = (rdb_pointcloud_point_attribute_set_invalid_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_invalid_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_invalid_value()\".";
    if ((rdb_pointcloud_point_attribute_set_invalid_value_cf       = (rdb_pointcloud_point_attribute_set_invalid_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_invalid_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_invalid_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_length                 = (rdb_pointcloud_point_attribute_set_length_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_length"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_length()\".";
    if ((rdb_pointcloud_point_attribute_set_length_cf              = (rdb_pointcloud_point_attribute_set_length_cf_type)             library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_length_cf"))              == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_length_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_length_u32             = (rdb_pointcloud_point_attribute_set_length_u32_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_length_u32"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_length_u32()\".";
    if ((rdb_pointcloud_point_attribute_set_length_u32_cf          = (rdb_pointcloud_point_attribute_set_length_u32_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_length_u32_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_length_u32_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_lod_settings           = (rdb_pointcloud_point_attribute_set_lod_settings_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_lod_settings"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_lod_settings()\".";
    if ((rdb_pointcloud_point_attribute_set_lod_settings_cf        = (rdb_pointcloud_point_attribute_set_lod_settings_cf_type)       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_lod_settings_cf"))        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_lod_settings_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_maximum_value          = (rdb_pointcloud_point_attribute_set_maximum_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_maximum_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_maximum_value()\".";
    if ((rdb_pointcloud_point_attribute_set_maximum_value_cf       = (rdb_pointcloud_point_attribute_set_maximum_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_maximum_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_maximum_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_minimum_value          = (rdb_pointcloud_point_attribute_set_minimum_value_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_minimum_value"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_minimum_value()\".";
    if ((rdb_pointcloud_point_attribute_set_minimum_value_cf       = (rdb_pointcloud_point_attribute_set_minimum_value_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_minimum_value_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_minimum_value_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_name                   = (rdb_pointcloud_point_attribute_set_name_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_name"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_name()\".";
    if ((rdb_pointcloud_point_attribute_set_name_cf                = (rdb_pointcloud_point_attribute_set_name_cf_type)               library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_name_cf"))                == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_name_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_named_values           = (rdb_pointcloud_point_attribute_set_named_values_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_named_values"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_named_values()\".";
    if ((rdb_pointcloud_point_attribute_set_named_values_cf        = (rdb_pointcloud_point_attribute_set_named_values_cf_type)       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_named_values_cf"))        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_named_values_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_resolution             = (rdb_pointcloud_point_attribute_set_resolution_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_resolution"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_resolution()\".";
    if ((rdb_pointcloud_point_attribute_set_resolution_cf          = (rdb_pointcloud_point_attribute_set_resolution_cf_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_resolution_cf"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_resolution_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_scale_factor           = (rdb_pointcloud_point_attribute_set_scale_factor_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_scale_factor"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_scale_factor()\".";
    if ((rdb_pointcloud_point_attribute_set_scale_factor_cf        = (rdb_pointcloud_point_attribute_set_scale_factor_cf_type)       library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_scale_factor_cf"))        == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_scale_factor_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_storage_class          = (rdb_pointcloud_point_attribute_set_storage_class_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_storage_class"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_storage_class()\".";
    if ((rdb_pointcloud_point_attribute_set_storage_class_cf       = (rdb_pointcloud_point_attribute_set_storage_class_cf_type)      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_storage_class_cf"))       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_storage_class_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_tags                   = (rdb_pointcloud_point_attribute_set_tags_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_tags"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_tags()\".";
    if ((rdb_pointcloud_point_attribute_set_tags_cf                = (rdb_pointcloud_point_attribute_set_tags_cf_type)               library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_tags_cf"))                == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_tags_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_title                  = (rdb_pointcloud_point_attribute_set_title_type)                 library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_title"))                  == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_title()\".";
    if ((rdb_pointcloud_point_attribute_set_title_cf               = (rdb_pointcloud_point_attribute_set_title_cf_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_title_cf"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_title_cf()\".";
    if ((rdb_pointcloud_point_attribute_set_unit_symbol            = (rdb_pointcloud_point_attribute_set_unit_symbol_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_unit_symbol"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_unit_symbol()\".";
    if ((rdb_pointcloud_point_attribute_set_unit_symbol_cf         = (rdb_pointcloud_point_attribute_set_unit_symbol_cf_type)        library_resolve(rdb_library_handle, "rdb_pointcloud_point_attribute_set_unit_symbol_cf"))         == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attribute_set_unit_symbol_cf()\".";
    if ((rdb_pointcloud_point_attributes_add                       = (rdb_pointcloud_point_attributes_add_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_add"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_add()\".";
    if ((rdb_pointcloud_point_attributes_add_default               = (rdb_pointcloud_point_attributes_add_default_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_add_default"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_add_default()\".";
    if ((rdb_pointcloud_point_attributes_discard                   = (rdb_pointcloud_point_attributes_discard_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_discard"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_discard()\".";
    if ((rdb_pointcloud_point_attributes_duplicate                 = (rdb_pointcloud_point_attributes_duplicate_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_duplicate"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_duplicate()\".";
    if ((rdb_pointcloud_point_attributes_exists                    = (rdb_pointcloud_point_attributes_exists_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_exists"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_exists()\".";
    if ((rdb_pointcloud_point_attributes_get                       = (rdb_pointcloud_point_attributes_get_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_get"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_get()\".";
    if ((rdb_pointcloud_point_attributes_get_default               = (rdb_pointcloud_point_attributes_get_default_type)              library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_get_default"))               == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_get_default()\".";
    if ((rdb_pointcloud_point_attributes_get_merged                = (rdb_pointcloud_point_attributes_get_merged_type)               library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_get_merged"))                == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_get_merged()\".";
    if ((rdb_pointcloud_point_attributes_get_merged_all            = (rdb_pointcloud_point_attributes_get_merged_all_type)           library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_get_merged_all"))            == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_get_merged_all()\".";
    if ((rdb_pointcloud_point_attributes_group                     = (rdb_pointcloud_point_attributes_group_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_group"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_group()\".";
    if ((rdb_pointcloud_point_attributes_group_default             = (rdb_pointcloud_point_attributes_group_default_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_group_default"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_group_default()\".";
    if ((rdb_pointcloud_point_attributes_list                      = (rdb_pointcloud_point_attributes_list_type)                     library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_list"))                      == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_list()\".";
    if ((rdb_pointcloud_point_attributes_list_default              = (rdb_pointcloud_point_attributes_list_default_type)             library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_list_default"))              == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_list_default()\".";
    if ((rdb_pointcloud_point_attributes_list_filtered             = (rdb_pointcloud_point_attributes_list_filtered_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_list_filtered"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_list_filtered()\".";
    if ((rdb_pointcloud_point_attributes_point_id_name             = (rdb_pointcloud_point_attributes_point_id_name_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_point_id_name"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_point_id_name()\".";
    if ((rdb_pointcloud_point_attributes_point_id_type             = (rdb_pointcloud_point_attributes_point_id_type_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_point_id_type"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_point_id_type()\".";
    if ((rdb_pointcloud_point_attributes_point_id_unit             = (rdb_pointcloud_point_attributes_point_id_unit_type)            library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_point_id_unit"))             == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_point_id_unit()\".";
    if ((rdb_pointcloud_point_attributes_primary_attribute_name    = (rdb_pointcloud_point_attributes_primary_attribute_name_type)   library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_primary_attribute_name"))    == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_primary_attribute_name()\".";
    if ((rdb_pointcloud_point_attributes_put                       = (rdb_pointcloud_point_attributes_put_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_put"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_put()\".";
    if ((rdb_pointcloud_point_attributes_remove                    = (rdb_pointcloud_point_attributes_remove_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_point_attributes_remove"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_point_attributes_remove()\".";
    if ((rdb_pointcloud_query_fill_bind                            = (rdb_pointcloud_query_fill_bind_type)                           library_resolve(rdb_library_handle, "rdb_pointcloud_query_fill_bind"))                            == 0) return "Failed to load RDB function \"rdb_pointcloud_query_fill_bind()\".";
    if ((rdb_pointcloud_query_fill_delete                          = (rdb_pointcloud_query_fill_delete_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_fill_delete"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_fill_delete()\".";
    if ((rdb_pointcloud_query_fill_new                             = (rdb_pointcloud_query_fill_new_type)                            library_resolve(rdb_library_handle, "rdb_pointcloud_query_fill_new"))                             == 0) return "Failed to load RDB function \"rdb_pointcloud_query_fill_new()\".";
    if ((rdb_pointcloud_query_fill_next                            = (rdb_pointcloud_query_fill_next_type)                           library_resolve(rdb_library_handle, "rdb_pointcloud_query_fill_next"))                            == 0) return "Failed to load RDB function \"rdb_pointcloud_query_fill_next()\".";
    if ((rdb_pointcloud_query_fill_nodes_new                       = (rdb_pointcloud_query_fill_nodes_new_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_query_fill_nodes_new"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_query_fill_nodes_new()\".";
    if ((rdb_pointcloud_query_fill_progress                        = (rdb_pointcloud_query_fill_progress_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_fill_progress"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_fill_progress()\".";
    if ((rdb_pointcloud_query_insert_bind                          = (rdb_pointcloud_query_insert_bind_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_insert_bind"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_insert_bind()\".";
    if ((rdb_pointcloud_query_insert_bind_id                       = (rdb_pointcloud_query_insert_bind_id_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_query_insert_bind_id"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_query_insert_bind_id()\".";
    if ((rdb_pointcloud_query_insert_delete                        = (rdb_pointcloud_query_insert_delete_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_insert_delete"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_insert_delete()\".";
    if ((rdb_pointcloud_query_insert_load_multi_pass_settings      = (rdb_pointcloud_query_insert_load_multi_pass_settings_type)     library_resolve(rdb_library_handle, "rdb_pointcloud_query_insert_load_multi_pass_settings"))      == 0) return "Failed to load RDB function \"rdb_pointcloud_query_insert_load_multi_pass_settings()\".";
    if ((rdb_pointcloud_query_insert_new                           = (rdb_pointcloud_query_insert_new_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_query_insert_new"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_query_insert_new()\".";
    if ((rdb_pointcloud_query_insert_next                          = (rdb_pointcloud_query_insert_next_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_insert_next"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_insert_next()\".";
    if ((rdb_pointcloud_query_invert_attribute                     = (rdb_pointcloud_query_invert_attribute_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_query_invert_attribute"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_query_invert_attribute()\".";
    if ((rdb_pointcloud_query_invert_delete                        = (rdb_pointcloud_query_invert_delete_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_invert_delete"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_invert_delete()\".";
    if ((rdb_pointcloud_query_invert_new                           = (rdb_pointcloud_query_invert_new_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_query_invert_new"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_query_invert_new()\".";
    if ((rdb_pointcloud_query_invert_next                          = (rdb_pointcloud_query_invert_next_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_invert_next"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_invert_next()\".";
    if ((rdb_pointcloud_query_invert_nodes_new                     = (rdb_pointcloud_query_invert_nodes_new_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_query_invert_nodes_new"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_query_invert_nodes_new()\".";
    if ((rdb_pointcloud_query_invert_progress                      = (rdb_pointcloud_query_invert_progress_type)                     library_resolve(rdb_library_handle, "rdb_pointcloud_query_invert_progress"))                      == 0) return "Failed to load RDB function \"rdb_pointcloud_query_invert_progress()\".";
    if ((rdb_pointcloud_query_remove_bind                          = (rdb_pointcloud_query_remove_bind_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_remove_bind"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_remove_bind()\".";
    if ((rdb_pointcloud_query_remove_delete                        = (rdb_pointcloud_query_remove_delete_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_remove_delete"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_remove_delete()\".";
    if ((rdb_pointcloud_query_remove_new                           = (rdb_pointcloud_query_remove_new_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_query_remove_new"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_query_remove_new()\".";
    if ((rdb_pointcloud_query_remove_next                          = (rdb_pointcloud_query_remove_next_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_remove_next"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_remove_next()\".";
    if ((rdb_pointcloud_query_select_bind                          = (rdb_pointcloud_query_select_bind_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_select_bind"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_select_bind()\".";
    if ((rdb_pointcloud_query_select_delete                        = (rdb_pointcloud_query_select_delete_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_select_delete"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_select_delete()\".";
    if ((rdb_pointcloud_query_select_new                           = (rdb_pointcloud_query_select_new_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_query_select_new"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_query_select_new()\".";
    if ((rdb_pointcloud_query_select_next                          = (rdb_pointcloud_query_select_next_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_select_next"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_select_next()\".";
    if ((rdb_pointcloud_query_select_nodes_new                     = (rdb_pointcloud_query_select_nodes_new_type)                    library_resolve(rdb_library_handle, "rdb_pointcloud_query_select_nodes_new"))                     == 0) return "Failed to load RDB function \"rdb_pointcloud_query_select_nodes_new()\".";
    if ((rdb_pointcloud_query_select_progress                      = (rdb_pointcloud_query_select_progress_type)                     library_resolve(rdb_library_handle, "rdb_pointcloud_query_select_progress"))                      == 0) return "Failed to load RDB function \"rdb_pointcloud_query_select_progress()\".";
    if ((rdb_pointcloud_query_stat_delete                          = (rdb_pointcloud_query_stat_delete_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_delete"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_delete()\".";
    if ((rdb_pointcloud_query_stat_index                           = (rdb_pointcloud_query_stat_index_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_index"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_index()\".";
    if ((rdb_pointcloud_query_stat_index_filter                    = (rdb_pointcloud_query_stat_index_filter_type)                   library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_index_filter"))                    == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_index_filter()\".";
    if ((rdb_pointcloud_query_stat_maximum                         = (rdb_pointcloud_query_stat_maximum_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_maximum"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_maximum()\".";
    if ((rdb_pointcloud_query_stat_maximum_cleaned                 = (rdb_pointcloud_query_stat_maximum_cleaned_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_maximum_cleaned"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_maximum_cleaned()\".";
    if ((rdb_pointcloud_query_stat_minimum                         = (rdb_pointcloud_query_stat_minimum_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_minimum"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_minimum()\".";
    if ((rdb_pointcloud_query_stat_minimum_cleaned                 = (rdb_pointcloud_query_stat_minimum_cleaned_type)                library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_minimum_cleaned"))                 == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_minimum_cleaned()\".";
    if ((rdb_pointcloud_query_stat_new                             = (rdb_pointcloud_query_stat_new_type)                            library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_new"))                             == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_new()\".";
    if ((rdb_pointcloud_query_stat_revision                        = (rdb_pointcloud_query_stat_revision_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_stat_revision"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_stat_revision()\".";
    if ((rdb_pointcloud_query_update_bind                          = (rdb_pointcloud_query_update_bind_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_update_bind"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_update_bind()\".";
    if ((rdb_pointcloud_query_update_delete                        = (rdb_pointcloud_query_update_delete_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_query_update_delete"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_query_update_delete()\".";
    if ((rdb_pointcloud_query_update_new                           = (rdb_pointcloud_query_update_new_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_query_update_new"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_query_update_new()\".";
    if ((rdb_pointcloud_query_update_next                          = (rdb_pointcloud_query_update_next_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_query_update_next"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_query_update_next()\".";
    if ((rdb_pointcloud_transaction_begin                          = (rdb_pointcloud_transaction_begin_type)                         library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_begin"))                          == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_begin()\".";
    if ((rdb_pointcloud_transaction_commit                         = (rdb_pointcloud_transaction_commit_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_commit"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_commit()\".";
    if ((rdb_pointcloud_transaction_commit_with_progress           = (rdb_pointcloud_transaction_commit_with_progress_type)          library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_commit_with_progress"))           == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_commit_with_progress()\".";
    if ((rdb_pointcloud_transaction_commit_with_signature          = (rdb_pointcloud_transaction_commit_with_signature_type)         library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_commit_with_signature"))          == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_commit_with_signature()\".";
    if ((rdb_pointcloud_transaction_current                        = (rdb_pointcloud_transaction_current_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_current"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_current()\".";
    if ((rdb_pointcloud_transaction_delete                         = (rdb_pointcloud_transaction_delete_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_delete"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_delete()\".";
    if ((rdb_pointcloud_transaction_details                        = (rdb_pointcloud_transaction_details_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_details"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_details()\".";
    if ((rdb_pointcloud_transaction_discard                        = (rdb_pointcloud_transaction_discard_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_discard"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_discard()\".";
    if ((rdb_pointcloud_transaction_get_agent                      = (rdb_pointcloud_transaction_get_agent_type)                     library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_agent"))                      == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_agent()\".";
    if ((rdb_pointcloud_transaction_get_comments                   = (rdb_pointcloud_transaction_get_comments_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_comments"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_comments()\".";
    if ((rdb_pointcloud_transaction_get_id                         = (rdb_pointcloud_transaction_get_id_type)                        library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_id"))                         == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_id()\".";
    if ((rdb_pointcloud_transaction_get_rdb                        = (rdb_pointcloud_transaction_get_rdb_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_rdb"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_rdb()\".";
    if ((rdb_pointcloud_transaction_get_settings                   = (rdb_pointcloud_transaction_get_settings_type)                  library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_settings"))                   == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_settings()\".";
    if ((rdb_pointcloud_transaction_get_start                      = (rdb_pointcloud_transaction_get_start_type)                     library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_start"))                      == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_start()\".";
    if ((rdb_pointcloud_transaction_get_stop                       = (rdb_pointcloud_transaction_get_stop_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_stop"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_stop()\".";
    if ((rdb_pointcloud_transaction_get_title                      = (rdb_pointcloud_transaction_get_title_type)                     library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_get_title"))                      == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_get_title()\".";
    if ((rdb_pointcloud_transaction_list                           = (rdb_pointcloud_transaction_list_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_list"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_list()\".";
    if ((rdb_pointcloud_transaction_new                            = (rdb_pointcloud_transaction_new_type)                           library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_new"))                            == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_new()\".";
    if ((rdb_pointcloud_transaction_pending                        = (rdb_pointcloud_transaction_pending_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_pending"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_pending()\".";
    if ((rdb_pointcloud_transaction_restore                        = (rdb_pointcloud_transaction_restore_type)                       library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_restore"))                        == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_restore()\".";
    if ((rdb_pointcloud_transaction_rollback                       = (rdb_pointcloud_transaction_rollback_type)                      library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_rollback"))                       == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_rollback()\".";
    if ((rdb_pointcloud_transaction_size                           = (rdb_pointcloud_transaction_size_type)                          library_resolve(rdb_library_handle, "rdb_pointcloud_transaction_size"))                           == 0) return "Failed to load RDB function \"rdb_pointcloud_transaction_size()\".";
    if ((rdb_result_delete_cf                                      = (rdb_result_delete_cf_type)                                     library_resolve(rdb_library_handle, "rdb_result_delete_cf"))                                      == 0) return "Failed to load RDB function \"rdb_result_delete_cf()\".";
    if ((rdb_result_get_error_code_cf                              = (rdb_result_get_error_code_cf_type)                             library_resolve(rdb_library_handle, "rdb_result_get_error_code_cf"))                              == 0) return "Failed to load RDB function \"rdb_result_get_error_code_cf()\".";
    if ((rdb_result_get_error_details_cf                           = (rdb_result_get_error_details_cf_type)                          library_resolve(rdb_library_handle, "rdb_result_get_error_details_cf"))                           == 0) return "Failed to load RDB function \"rdb_result_get_error_details_cf()\".";
    if ((rdb_result_get_error_text_cf                              = (rdb_result_get_error_text_cf_type)                             library_resolve(rdb_library_handle, "rdb_result_get_error_text_cf"))                              == 0) return "Failed to load RDB function \"rdb_result_get_error_text_cf()\".";
    if ((rdb_string_delete_cf                                      = (rdb_string_delete_cf_type)                                     library_resolve(rdb_library_handle, "rdb_string_delete_cf"))                                      == 0) return "Failed to load RDB function \"rdb_string_delete_cf()\".";

    return 0; // success
}

RDBString rdb_library_free()
{
    if (rdb_library_handle == 0)
    {
        return 0; // library not loaded
    }

    if (!library_free(rdb_library_handle))
    {
        return "Failed to free RDB library.";
    }

    rdb_context_delete                                        = 0;
    rdb_context_get_last_error                                = 0;
    rdb_context_new                                           = 0;
    rdb_context_new_with_api                                  = 0;
    rdb_database_file_type_check                              = 0;
    rdb_database_file_type_suffix                             = 0;
    rdb_database_file_type_title                              = 0;
    rdb_factory_set_authentication_function                   = 0;
    rdb_library_filename                                      = 0;
    rdb_library_license                                       = 0;
    rdb_library_name                                          = 0;
    rdb_library_version                                       = 0;
    rdb_object_lock_cf                                        = 0;
    rdb_object_unlock_cf                                      = 0;
    rdb_pointcloud_changelog_append_log_message               = 0;
    rdb_pointcloud_changelog_export_to_textfile               = 0;
    rdb_pointcloud_changelog_import_from_database             = 0;
    rdb_pointcloud_changelog_verify_log_entry_cf              = 0;
    rdb_pointcloud_clear_cache                                = 0;
    rdb_pointcloud_close                                      = 0;
    rdb_pointcloud_create                                     = 0;
    rdb_pointcloud_create_settings_delete                     = 0;
    rdb_pointcloud_create_settings_delete_cf                  = 0;
    rdb_pointcloud_create_settings_get_cache_size             = 0;
    rdb_pointcloud_create_settings_get_cache_size_64          = 0;
    rdb_pointcloud_create_settings_get_cache_size_64_cf       = 0;
    rdb_pointcloud_create_settings_get_cache_size_cf          = 0;
    rdb_pointcloud_create_settings_get_chunk_mode             = 0;
    rdb_pointcloud_create_settings_get_chunk_mode_cf          = 0;
    rdb_pointcloud_create_settings_get_chunk_size             = 0;
    rdb_pointcloud_create_settings_get_chunk_size_cf          = 0;
    rdb_pointcloud_create_settings_get_chunk_size_lod         = 0;
    rdb_pointcloud_create_settings_get_chunk_size_lod_cf      = 0;
    rdb_pointcloud_create_settings_get_compression_level      = 0;
    rdb_pointcloud_create_settings_get_compression_level_cf   = 0;
    rdb_pointcloud_create_settings_get_lod_mode               = 0;
    rdb_pointcloud_create_settings_get_lod_mode_cf            = 0;
    rdb_pointcloud_create_settings_get_optimize_point_id      = 0;
    rdb_pointcloud_create_settings_get_optimize_point_id_cf   = 0;
    rdb_pointcloud_create_settings_get_point_insert_mode      = 0;
    rdb_pointcloud_create_settings_get_point_insert_mode_cf   = 0;
    rdb_pointcloud_create_settings_get_primary_attribute      = 0;
    rdb_pointcloud_create_settings_get_primary_attribute_cf   = 0;
    rdb_pointcloud_create_settings_json_load                  = 0;
    rdb_pointcloud_create_settings_json_load_cf               = 0;
    rdb_pointcloud_create_settings_json_save                  = 0;
    rdb_pointcloud_create_settings_json_save_cf               = 0;
    rdb_pointcloud_create_settings_new                        = 0;
    rdb_pointcloud_create_settings_new_cf                     = 0;
    rdb_pointcloud_create_settings_set_cache_size             = 0;
    rdb_pointcloud_create_settings_set_cache_size_64          = 0;
    rdb_pointcloud_create_settings_set_cache_size_64_cf       = 0;
    rdb_pointcloud_create_settings_set_cache_size_cf          = 0;
    rdb_pointcloud_create_settings_set_chunk_mode             = 0;
    rdb_pointcloud_create_settings_set_chunk_mode_cf          = 0;
    rdb_pointcloud_create_settings_set_chunk_size             = 0;
    rdb_pointcloud_create_settings_set_chunk_size_cf          = 0;
    rdb_pointcloud_create_settings_set_chunk_size_lod         = 0;
    rdb_pointcloud_create_settings_set_chunk_size_lod_cf      = 0;
    rdb_pointcloud_create_settings_set_compression_level      = 0;
    rdb_pointcloud_create_settings_set_compression_level_cf   = 0;
    rdb_pointcloud_create_settings_set_lod_mode               = 0;
    rdb_pointcloud_create_settings_set_lod_mode_cf            = 0;
    rdb_pointcloud_create_settings_set_optimize_point_id      = 0;
    rdb_pointcloud_create_settings_set_optimize_point_id_cf   = 0;
    rdb_pointcloud_create_settings_set_point_insert_mode      = 0;
    rdb_pointcloud_create_settings_set_point_insert_mode_cf   = 0;
    rdb_pointcloud_create_settings_set_primary_attribute      = 0;
    rdb_pointcloud_create_settings_set_primary_attribute_cf   = 0;
    rdb_pointcloud_create_with_schema                         = 0;
    rdb_pointcloud_delete                                     = 0;
    rdb_pointcloud_get_uuid                                   = 0;
    rdb_pointcloud_graph_node_delete                          = 0;
    rdb_pointcloud_graph_node_new                             = 0;
    rdb_pointcloud_inspect                                    = 0;
    rdb_pointcloud_is_empty                                   = 0;
    rdb_pointcloud_is_open                                    = 0;
    rdb_pointcloud_management_finalize                        = 0;
    rdb_pointcloud_management_get_chunk_size_lod              = 0;
    rdb_pointcloud_management_get_lod_mode                    = 0;
    rdb_pointcloud_management_set_chunk_size_lod              = 0;
    rdb_pointcloud_management_set_lod_mode                    = 0;
    rdb_pointcloud_management_vacuum                          = 0;
    rdb_pointcloud_management_validate                        = 0;
    rdb_pointcloud_meta_data_create_signature                 = 0;
    rdb_pointcloud_meta_data_exists                           = 0;
    rdb_pointcloud_meta_data_get                              = 0;
    rdb_pointcloud_meta_data_list                             = 0;
    rdb_pointcloud_meta_data_remove                           = 0;
    rdb_pointcloud_meta_data_set                              = 0;
    rdb_pointcloud_meta_data_validate                         = 0;
    rdb_pointcloud_meta_data_validate_json                    = 0;
    rdb_pointcloud_meta_data_verify_signature                 = 0;
    rdb_pointcloud_new                                        = 0;
    rdb_pointcloud_open                                       = 0;
    rdb_pointcloud_open_settings_delete                       = 0;
    rdb_pointcloud_open_settings_delete_cf                    = 0;
    rdb_pointcloud_open_settings_get_cache_size               = 0;
    rdb_pointcloud_open_settings_get_cache_size_64            = 0;
    rdb_pointcloud_open_settings_get_cache_size_64_cf         = 0;
    rdb_pointcloud_open_settings_get_cache_size_cf            = 0;
    rdb_pointcloud_open_settings_json_load                    = 0;
    rdb_pointcloud_open_settings_json_load_cf                 = 0;
    rdb_pointcloud_open_settings_json_save                    = 0;
    rdb_pointcloud_open_settings_json_save_cf                 = 0;
    rdb_pointcloud_open_settings_new                          = 0;
    rdb_pointcloud_open_settings_new_cf                       = 0;
    rdb_pointcloud_open_settings_set_cache_size               = 0;
    rdb_pointcloud_open_settings_set_cache_size_64            = 0;
    rdb_pointcloud_open_settings_set_cache_size_64_cf         = 0;
    rdb_pointcloud_open_settings_set_cache_size_cf            = 0;
    rdb_pointcloud_point_attribute_data_type                  = 0;
    rdb_pointcloud_point_attribute_data_type_cf               = 0;
    rdb_pointcloud_point_attribute_decode_descriptor          = 0;
    rdb_pointcloud_point_attribute_delete                     = 0;
    rdb_pointcloud_point_attribute_delete_cf                  = 0;
    rdb_pointcloud_point_attribute_get_compression_options    = 0;
    rdb_pointcloud_point_attribute_get_compression_options_cf = 0;
    rdb_pointcloud_point_attribute_get_default_value          = 0;
    rdb_pointcloud_point_attribute_get_default_value_cf       = 0;
    rdb_pointcloud_point_attribute_get_description            = 0;
    rdb_pointcloud_point_attribute_get_description_cf         = 0;
    rdb_pointcloud_point_attribute_get_invalid_value          = 0;
    rdb_pointcloud_point_attribute_get_invalid_value_cf       = 0;
    rdb_pointcloud_point_attribute_get_length                 = 0;
    rdb_pointcloud_point_attribute_get_length_cf              = 0;
    rdb_pointcloud_point_attribute_get_length_u32             = 0;
    rdb_pointcloud_point_attribute_get_length_u32_cf          = 0;
    rdb_pointcloud_point_attribute_get_lod_settings           = 0;
    rdb_pointcloud_point_attribute_get_lod_settings_cf        = 0;
    rdb_pointcloud_point_attribute_get_maximum_value          = 0;
    rdb_pointcloud_point_attribute_get_maximum_value_cf       = 0;
    rdb_pointcloud_point_attribute_get_minimum_value          = 0;
    rdb_pointcloud_point_attribute_get_minimum_value_cf       = 0;
    rdb_pointcloud_point_attribute_get_name                   = 0;
    rdb_pointcloud_point_attribute_get_name_cf                = 0;
    rdb_pointcloud_point_attribute_get_named_values           = 0;
    rdb_pointcloud_point_attribute_get_named_values_cf        = 0;
    rdb_pointcloud_point_attribute_get_resolution             = 0;
    rdb_pointcloud_point_attribute_get_resolution_cf          = 0;
    rdb_pointcloud_point_attribute_get_scale_factor           = 0;
    rdb_pointcloud_point_attribute_get_scale_factor_cf        = 0;
    rdb_pointcloud_point_attribute_get_storage_class          = 0;
    rdb_pointcloud_point_attribute_get_storage_class_cf       = 0;
    rdb_pointcloud_point_attribute_get_tags                   = 0;
    rdb_pointcloud_point_attribute_get_tags_cf                = 0;
    rdb_pointcloud_point_attribute_get_title                  = 0;
    rdb_pointcloud_point_attribute_get_title_cf               = 0;
    rdb_pointcloud_point_attribute_get_unit_symbol            = 0;
    rdb_pointcloud_point_attribute_get_unit_symbol_cf         = 0;
    rdb_pointcloud_point_attribute_json_load                  = 0;
    rdb_pointcloud_point_attribute_json_load_cf               = 0;
    rdb_pointcloud_point_attribute_json_save                  = 0;
    rdb_pointcloud_point_attribute_json_save_cf               = 0;
    rdb_pointcloud_point_attribute_new                        = 0;
    rdb_pointcloud_point_attribute_new_cf                     = 0;
    rdb_pointcloud_point_attribute_set_compression_options    = 0;
    rdb_pointcloud_point_attribute_set_compression_options_cf = 0;
    rdb_pointcloud_point_attribute_set_default_value          = 0;
    rdb_pointcloud_point_attribute_set_default_value_cf       = 0;
    rdb_pointcloud_point_attribute_set_description            = 0;
    rdb_pointcloud_point_attribute_set_description_cf         = 0;
    rdb_pointcloud_point_attribute_set_invalid_value          = 0;
    rdb_pointcloud_point_attribute_set_invalid_value_cf       = 0;
    rdb_pointcloud_point_attribute_set_length                 = 0;
    rdb_pointcloud_point_attribute_set_length_cf              = 0;
    rdb_pointcloud_point_attribute_set_length_u32             = 0;
    rdb_pointcloud_point_attribute_set_length_u32_cf          = 0;
    rdb_pointcloud_point_attribute_set_lod_settings           = 0;
    rdb_pointcloud_point_attribute_set_lod_settings_cf        = 0;
    rdb_pointcloud_point_attribute_set_maximum_value          = 0;
    rdb_pointcloud_point_attribute_set_maximum_value_cf       = 0;
    rdb_pointcloud_point_attribute_set_minimum_value          = 0;
    rdb_pointcloud_point_attribute_set_minimum_value_cf       = 0;
    rdb_pointcloud_point_attribute_set_name                   = 0;
    rdb_pointcloud_point_attribute_set_name_cf                = 0;
    rdb_pointcloud_point_attribute_set_named_values           = 0;
    rdb_pointcloud_point_attribute_set_named_values_cf        = 0;
    rdb_pointcloud_point_attribute_set_resolution             = 0;
    rdb_pointcloud_point_attribute_set_resolution_cf          = 0;
    rdb_pointcloud_point_attribute_set_scale_factor           = 0;
    rdb_pointcloud_point_attribute_set_scale_factor_cf        = 0;
    rdb_pointcloud_point_attribute_set_storage_class          = 0;
    rdb_pointcloud_point_attribute_set_storage_class_cf       = 0;
    rdb_pointcloud_point_attribute_set_tags                   = 0;
    rdb_pointcloud_point_attribute_set_tags_cf                = 0;
    rdb_pointcloud_point_attribute_set_title                  = 0;
    rdb_pointcloud_point_attribute_set_title_cf               = 0;
    rdb_pointcloud_point_attribute_set_unit_symbol            = 0;
    rdb_pointcloud_point_attribute_set_unit_symbol_cf         = 0;
    rdb_pointcloud_point_attributes_add                       = 0;
    rdb_pointcloud_point_attributes_add_default               = 0;
    rdb_pointcloud_point_attributes_discard                   = 0;
    rdb_pointcloud_point_attributes_duplicate                 = 0;
    rdb_pointcloud_point_attributes_exists                    = 0;
    rdb_pointcloud_point_attributes_get                       = 0;
    rdb_pointcloud_point_attributes_get_default               = 0;
    rdb_pointcloud_point_attributes_get_merged                = 0;
    rdb_pointcloud_point_attributes_get_merged_all            = 0;
    rdb_pointcloud_point_attributes_group                     = 0;
    rdb_pointcloud_point_attributes_group_default             = 0;
    rdb_pointcloud_point_attributes_list                      = 0;
    rdb_pointcloud_point_attributes_list_default              = 0;
    rdb_pointcloud_point_attributes_list_filtered             = 0;
    rdb_pointcloud_point_attributes_point_id_name             = 0;
    rdb_pointcloud_point_attributes_point_id_type             = 0;
    rdb_pointcloud_point_attributes_point_id_unit             = 0;
    rdb_pointcloud_point_attributes_primary_attribute_name    = 0;
    rdb_pointcloud_point_attributes_put                       = 0;
    rdb_pointcloud_point_attributes_remove                    = 0;
    rdb_pointcloud_query_fill_bind                            = 0;
    rdb_pointcloud_query_fill_delete                          = 0;
    rdb_pointcloud_query_fill_new                             = 0;
    rdb_pointcloud_query_fill_next                            = 0;
    rdb_pointcloud_query_fill_nodes_new                       = 0;
    rdb_pointcloud_query_fill_progress                        = 0;
    rdb_pointcloud_query_insert_bind                          = 0;
    rdb_pointcloud_query_insert_bind_id                       = 0;
    rdb_pointcloud_query_insert_delete                        = 0;
    rdb_pointcloud_query_insert_load_multi_pass_settings      = 0;
    rdb_pointcloud_query_insert_new                           = 0;
    rdb_pointcloud_query_insert_next                          = 0;
    rdb_pointcloud_query_invert_attribute                     = 0;
    rdb_pointcloud_query_invert_delete                        = 0;
    rdb_pointcloud_query_invert_new                           = 0;
    rdb_pointcloud_query_invert_next                          = 0;
    rdb_pointcloud_query_invert_nodes_new                     = 0;
    rdb_pointcloud_query_invert_progress                      = 0;
    rdb_pointcloud_query_remove_bind                          = 0;
    rdb_pointcloud_query_remove_delete                        = 0;
    rdb_pointcloud_query_remove_new                           = 0;
    rdb_pointcloud_query_remove_next                          = 0;
    rdb_pointcloud_query_select_bind                          = 0;
    rdb_pointcloud_query_select_delete                        = 0;
    rdb_pointcloud_query_select_new                           = 0;
    rdb_pointcloud_query_select_next                          = 0;
    rdb_pointcloud_query_select_nodes_new                     = 0;
    rdb_pointcloud_query_select_progress                      = 0;
    rdb_pointcloud_query_stat_delete                          = 0;
    rdb_pointcloud_query_stat_index                           = 0;
    rdb_pointcloud_query_stat_index_filter                    = 0;
    rdb_pointcloud_query_stat_maximum                         = 0;
    rdb_pointcloud_query_stat_maximum_cleaned                 = 0;
    rdb_pointcloud_query_stat_minimum                         = 0;
    rdb_pointcloud_query_stat_minimum_cleaned                 = 0;
    rdb_pointcloud_query_stat_new                             = 0;
    rdb_pointcloud_query_stat_revision                        = 0;
    rdb_pointcloud_query_update_bind                          = 0;
    rdb_pointcloud_query_update_delete                        = 0;
    rdb_pointcloud_query_update_new                           = 0;
    rdb_pointcloud_query_update_next                          = 0;
    rdb_pointcloud_transaction_begin                          = 0;
    rdb_pointcloud_transaction_commit                         = 0;
    rdb_pointcloud_transaction_commit_with_progress           = 0;
    rdb_pointcloud_transaction_commit_with_signature          = 0;
    rdb_pointcloud_transaction_current                        = 0;
    rdb_pointcloud_transaction_delete                         = 0;
    rdb_pointcloud_transaction_details                        = 0;
    rdb_pointcloud_transaction_discard                        = 0;
    rdb_pointcloud_transaction_get_agent                      = 0;
    rdb_pointcloud_transaction_get_comments                   = 0;
    rdb_pointcloud_transaction_get_id                         = 0;
    rdb_pointcloud_transaction_get_rdb                        = 0;
    rdb_pointcloud_transaction_get_settings                   = 0;
    rdb_pointcloud_transaction_get_start                      = 0;
    rdb_pointcloud_transaction_get_stop                       = 0;
    rdb_pointcloud_transaction_get_title                      = 0;
    rdb_pointcloud_transaction_list                           = 0;
    rdb_pointcloud_transaction_new                            = 0;
    rdb_pointcloud_transaction_pending                        = 0;
    rdb_pointcloud_transaction_restore                        = 0;
    rdb_pointcloud_transaction_rollback                       = 0;
    rdb_pointcloud_transaction_size                           = 0;
    rdb_result_delete_cf                                      = 0;
    rdb_result_get_error_code_cf                              = 0;
    rdb_result_get_error_details_cf                           = 0;
    rdb_result_get_error_text_cf                              = 0;
    rdb_string_delete_cf                                      = 0;

    rdb_library_handle = 0;
    return 0; // success
}

#endif // RDB_RUNTIME_LINKING
RDB_LIBRARY_API_END
