/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryStat.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point statistics query
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-07/AW: Optionally filter index graph nodes (#2390)
 * \version 2019-11-05/AW: Optionally return cleaned attribute extents (#3071)
 *
 * This query provides point attribute statistics like minimum and
 * maximum value.
 *
 * \see riegl::rdb::Pointcloud::stat()
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_QUERYSTAT_H
#define RIEGL_RDB_POINTCLOUD_QUERYSTAT_H

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"

//---< TYPE DEFINITIONS >-------------------------------------------------------

typedef struct RDBPointcloudQueryStat RDBPointcloudQueryStat; // forward declaration of implementation details

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN

//______________________________________________________________________________
/*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::stat()
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_new,
    RDBContext              *context,    //!< [in] library context
    RDBPointcloud           *pointcloud, //!< [in] point cloud
    RDBPointcloudQueryStat **query       //!< [out] query handle
);

//______________________________________________________________________________
/*!
 * \brief Destructor
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_delete,
    RDBContext              *context, //!< [in] library context
    RDBPointcloudQueryStat **query    //!< [in] query handle
);

//______________________________________________________________________________
/*!
 * \brief Get index graph
 *
 * This function returns a simple directed graph which represents
 * the index structure that is used to organize the point cloud.
 *
 * The optional filter expression can be used to select particular
 * graph nodes - if no filter is given, all nodes will be returned.
 * Filter expression syntax see riegl::rdb::Pointcloud::select().
 *
 * Note: The reported point counts and attribute extents are not
 * affected by the filter expressions given here or defined in the
 * meta data item riegl.stored_filters.
 *
 * Details see description of class GraphNode.
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_index,
    RDBContext             *context, //!< [in] library context
    RDBPointcloudQueryStat *query,   //!< [in] query handle
    RDBPointcloudGraphNode *root     //!< [in] graph root node
);
RDB_FUNCTION(rdb_pointcloud_query_stat_index_filter,
    RDBContext             *context, //!< [in] library context
    RDBPointcloudQueryStat *query,   //!< [in] query handle
    RDBPointcloudGraphNode *root,    //!< [in] graph root node
    RDBString               filter   //!< [in] filter string
);

//______________________________________________________________________________
/*!
 * \brief Lowest value of node
 *
 * Provides the attribute's minimum value of a branch (i.e. node and children).
 * If the node ID is zero, then the minimum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_minimum,
    RDBContext              *context,   //!< [in] library context
    RDBPointcloudQueryStat  *query,     //!< [in] query handle
    RDBPointcloudGraphNodeID nodeID,    //!< [in] node identifier
    RDBString                attribute, //!< [in] attribute name
    uint32_t                 dataType,  //!< [in] data type of target buffer
    void                    *buffer     //!< [out] buffer for minimum value
);

//______________________________________________________________________________
/*!
 * \brief Lowest value of node
 *
 * Provides the attribute's minimum value of a branch (i.e. node and children),
 * ignores invalid values (if the point attribute defines an invalid value).
 * If the node ID is zero, then the minimum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_minimum_cleaned,
    RDBContext              *context,   //!< [in] library context
    RDBPointcloudQueryStat  *query,     //!< [in] query handle
    RDBPointcloudGraphNodeID nodeID,    //!< [in] node identifier
    RDBString                attribute, //!< [in] attribute name
    uint32_t                 dataType,  //!< [in] data type of target buffer
    void                    *buffer     //!< [out] buffer for minimum value
);

//______________________________________________________________________________
/*!
 * \brief Highest value of node
 *
 * Provides the attribute's maximum value of a branch (i.e. node and children).
 * If the node ID is zero, then the maximum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_maximum,
    RDBContext              *context,   //!< [in] library context
    RDBPointcloudQueryStat  *query,     //!< [in] query handle
    RDBPointcloudGraphNodeID nodeID,    //!< [in] node identifier
    RDBString                attribute, //!< [in] attribute name
    uint32_t                 dataType,  //!< [in] data type of target buffer
    void                    *buffer     //!< [out] buffer for maximum value
);

//______________________________________________________________________________
/*!
 * \brief Highest value of node
 *
 * Provides the attribute's maximum value of a branch (i.e. node and children),
 * ignores invalid values (if the point attribute defines an invalid value).
 * If the node ID is zero, then the maximum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_maximum_cleaned,
    RDBContext              *context,   //!< [in] library context
    RDBPointcloudQueryStat  *query,     //!< [in] query handle
    RDBPointcloudGraphNodeID nodeID,    //!< [in] node identifier
    RDBString                attribute, //!< [in] attribute name
    uint32_t                 dataType,  //!< [in] data type of target buffer
    void                    *buffer     //!< [out] buffer for maximum value
);

//______________________________________________________________________________
/*!
 * \brief Attribute revision
 *
 * Provides the ID of the last transaction that has modified the attribute
 * in any node of the given branch.
 */
RDB_FUNCTION(rdb_pointcloud_query_stat_revision,
    RDBContext                 *context,   //!< [in] library context
    RDBPointcloudQueryStat     *query,     //!< [in] query handle
    RDBPointcloudGraphNodeID    nodeID,    //!< [in] node identifier
    RDBString                   attribute, //!< [in] attribute name
    RDBPointcloudTransactionID *revision   //!< [out] transaction ID, see above
);

RDB_LIBRARY_API_END

#endif // RIEGL_RDB_POINTCLOUD_QUERYSTAT_H
