/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    metaData.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Manage point cloud meta data (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2018-06-01/AW: New function to validate metadata entry (#3109)
 * \version 2020-03-30/AW: New functions for metadata signatures (#3570)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< MetaData::PUBLIC >-------------------------------------------------------

MetaData::MetaData(riegl::rdb::PointcloudData* pointcloud):
    data(pointcloud)
{
}

std::vector<std::string> MetaData::list() const
{
    std::uint32_t count(0); RDBString list(nullptr);
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_list(
        data->contextHandle(), data->pointcloud, &count, &list
    ));
    std::vector<std::string> result;
    for (std::uint32_t i = 0; i < count; i++)
    {
        std::string value(list);
        result.push_back(value);
        list += value.size()+1;
    }
    return result;
}

bool MetaData::exists(const std::string &name) const
{
    std::uint32_t result(0);
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_exists(
        data->contextHandle(), data->pointcloud, name.c_str(), &result
    ));
    return (result != 0);
}

void MetaData::set(const std::string &name, const std::string &value)
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_set(
        data->contextHandle(), data->pointcloud, name.c_str(), value.c_str()
    ));
}

std::string MetaData::get(const std::string &name, const std::string &defaultValue) const
{
    RDBString result(nullptr);
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_get(
        data->contextHandle(), data->pointcloud, name.c_str(), &result, defaultValue.c_str()
    ));
    return std::string(result);
}

void MetaData::remove(const std::string &name)
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_remove(
        data->contextHandle(), data->pointcloud, name.c_str()
    ));
}

void MetaData::validate(const std::string &name)
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_validate(
        data->contextHandle(), data->pointcloud, name.c_str()
    ));
}

void MetaData::validate(const std::string &value, const std::string &schema)
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_validate_json(
        data->contextHandle(), data->pointcloud, value.c_str(), schema.c_str()
    ));
}

void MetaData::createSignature(
    const std::string  &name,
    const std::uint32_t method,
    const std::uint32_t key_size,
    const void* const   key_data
)
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_create_signature(
        data->contextHandle(), data->pointcloud,
        name.c_str(), method, key_size, key_data
    ));
} // createSignature()

bool MetaData::verifySignature(
    const std::string  &name,
    const std::uint32_t method,
    const std::uint32_t key_size,
    const void* const   key_data
) const
{
    std::uint32_t valid(0);
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_meta_data_verify_signature(
        data->contextHandle(), data->pointcloud,
        name.c_str(), method, key_size, key_data,
        &valid
    ));
    return (valid != 0);
} // verifySignature()

}}} // namespace riegl::rdb::pointcloud
