/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-05-remove-points.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 4: Remove points
 * \version 2015-10-14/AW: Initial version
 * \version 2018-07-10/AW: Use new bindBuffer() instead of bind() function
 *
 *  This example shows how to open an existing database and delete some points.
 *  This example is based on the database of rdb-example-1-create-database.
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <vector>
#include <cstdint>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>
#include <riegl/rdb/default.hpp>

int main()
{
    try
    {
        // New RDB library context
        riegl::rdb::Context context;

        // Access existing database
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);

        // Before we can modify the database, we must start a transaction
        riegl::rdb::pointcloud::TransactionScope transaction(
            rdb,                // database object
            "Remove",           // transaction title
            "Point Filter v1.0" // software name
        );

        // Prepare point attribute buffers
        const uint32_t BUFFER_SIZE = 10000;
        std::vector< uint64_t > bufferIdentifier(BUFFER_SIZE);

        // Start new select query to get point ids of points with point class of 7
        riegl::rdb::pointcloud::QuerySelect select = rdb.select("riegl.class == 7");
        select.bindBuffer(riegl::rdb::pointcloud::RDB_RIEGL_ID, bufferIdentifier);

        // Start new remove query and define buffers
        riegl::rdb::pointcloud::QueryRemove remove = rdb.remove();
        remove.bindBuffer(riegl::rdb::pointcloud::RDB_RIEGL_ID, bufferIdentifier);

        // Process all points block-wise
        while (const uint32_t count = select.next(BUFFER_SIZE))
        {
            // Actually remove points
            remove.next(count);
        }

        // Finally commit transaction
        transaction.commit();

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
