/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    transaction.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point cloud transaction (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2017-02-16/AW: RDB library version string field added (#2515)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< TransactionWrapper >-----------------------------------------------------

struct TransactionWrapper
{
    RDBContext               *context;
    RDBPointcloudTransaction *handle;

    TransactionWrapper(RDBContext *context):
        context(context)
    {
        ErrorImpl::check(context, rdb_pointcloud_transaction_new(context, &handle));
    }

    ~TransactionWrapper()
    {
        ErrorImpl::check(context, rdb_pointcloud_transaction_delete(context, &handle));
    }
};

//---< TOOLS >------------------------------------------------------------------

//______________________________________________________________________________
/*!
 * \brief Read transaction from library
 * \note For wrapper internal use only.
 */
void transactionRead(
    RDBContext               *context,     //!< [in] library context
    RDBPointcloudTransaction *transaction, //!< [in] transaction handle
    Transaction              &target       //!< [out] target transaction
)
{
    RDBString localRDB;
    RDBString localTitle, localAgent, localComments;
    RDBString localSettings, localStart, localStop;
    //
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_id(
        context, transaction, &target.id
    ));
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_rdb(
        context, transaction, &localRDB
    ));                                                                         target.rdb = std::string(localRDB);
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_title(
        context, transaction, &localTitle
    ));                                                                         target.title = std::string(localTitle);
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_agent(
        context, transaction, &localAgent
    ));                                                                         target.agent = std::string(localAgent);
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_comments(
        context, transaction, &localComments
    ));                                                                         target.comments = std::string(localComments);
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_settings(
        context, transaction, &localSettings
    ));                                                                         target.settings = std::string(localSettings);
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_start(
        context, transaction, &localStart
    ));                                                                         target.start = std::string(localStart);
    ErrorImpl::check(context, rdb_pointcloud_transaction_get_stop(
        context, transaction, &localStop
    ));                                                                         target.stop = std::string(localStop);
} // pointAttributeRead()

//---< Transaction::PUBLIC >----------------------------------------------------

Transaction::Transaction():
    id(0),
    rdb(),
    title("none"),
    agent("unknown"),
    comments(),
    settings(),
    start(),
    stop()
{
}

Transaction::Transaction(riegl::rdb::Context &context)
{
    TransactionWrapper model(context.data->context);
    transactionRead(context.data->context, model.handle, *this);
}

}}} // namespace riegl::rdb::pointcloud
