/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    createSettings.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Database create settings (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-09-22/AW: Parameter 'cacheSize' changed from 32 to 64 bit type
 * \version 2016-12-20/AW: New functions to load/save settings from/to JSON
 * \version 2019-01-15/AW: Parameter 'chunkMode' added
 * \version 2019-01-18/AW: Parameter 'lodMode' added
 * \version 2019-02-15/AW: Fix C++ API wrapper of CreateSettings class
 * \version 2019-10-30/AW: Parameter 'optimizePointID' added (#3458)
 * \version 2020-02-21/AW: Class 'CreateSettings' is now context-free (#3544)
 * \version 2024-11-29/AW: Parameter 'pointInsertMode' added (#5339)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//______________________________________________________________________________
/*!
 * \brief C API wrapper class for CreateSettings
 * \note For C++ wrapper internal use only.
 */
class CreateSettingsWrapper
{
public:
    CreateSettingsWrapper(): hnd(nullptr)
    {
        ErrorImpl::check(rdb_pointcloud_create_settings_new_cf(&hnd));
    }

    ~CreateSettingsWrapper()
    {
        if (hnd) // then delete the create settings and ignore errors
        {
            ErrorImpl::check(rdb_pointcloud_create_settings_delete_cf(&hnd), true);
        }
    }

    static CreateSettingsWrapper& get(const CreateSettings &settings)
    {
        return *static_cast<CreateSettingsWrapper*>(settings.data);
    }

    static RDBPointcloudCreateSettings* handle(const CreateSettings &settings)
    {
        return get(settings).hnd;
    }

    static void assign(CreateSettings &target, const CreateSettings &source)
    {
        load(target, save(source));
    }

    static void read(CreateSettings &settings)
    {
        const CreateSettingsWrapper &self = get(settings);
        const PointAttributeWrapper::View primaryAttribute(settings.primaryAttribute, true);
        //
        ErrorImpl::check(rdb_pointcloud_create_settings_get_primary_attribute_cf(
            self.hnd, primaryAttribute.hnd
        ));
        PointAttributeWrapper::read(settings.primaryAttribute);
        //
        ErrorImpl::check(rdb_pointcloud_create_settings_get_chunk_mode_cf(
            self.hnd, &settings.chunkMode
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_get_chunk_size_cf(
            self.hnd, &settings.chunkSize
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_get_lod_mode_cf(
            self.hnd, &settings.lodMode
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_get_chunk_size_lod_cf(
            self.hnd, &settings.chunkSizeLOD
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_get_cache_size_64_cf(
            self.hnd, &settings.cacheSize
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_get_compression_level_cf(
            self.hnd, &settings.compressionLevel
        ));
        std::uint32_t settings_optimizePointID(0);
        ErrorImpl::check(rdb_pointcloud_create_settings_get_optimize_point_id_cf(
            self.hnd, &settings_optimizePointID
        ));
        settings.optimizePointID = (settings_optimizePointID != 0);
        RDBStringWrapper settings_pointInsertMode;
        ErrorImpl::check(rdb_pointcloud_create_settings_get_point_insert_mode_cf(
            self.hnd, &settings_pointInsertMode.str
        ));
        settings.pointInsertMode = settings_pointInsertMode;
    }

    static void post(const CreateSettings &settings)
    {
        const CreateSettingsWrapper &self = get(settings);
        const PointAttributeWrapper::View primaryAttribute(settings.primaryAttribute, false);
        //
        PointAttributeWrapper::post(settings.primaryAttribute);
        ErrorImpl::check(rdb_pointcloud_create_settings_set_primary_attribute_cf(
            self.hnd, primaryAttribute.hnd
        ));
        //
        ErrorImpl::check(rdb_pointcloud_create_settings_set_chunk_mode_cf(
            self.hnd, settings.chunkMode
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_chunk_size_cf(
            self.hnd, settings.chunkSize
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_lod_mode_cf(
            self.hnd, settings.lodMode
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_chunk_size_lod_cf(
            self.hnd, settings.chunkSizeLOD
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_cache_size_64_cf(
            self.hnd, settings.cacheSize
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_compression_level_cf(
            self.hnd, settings.compressionLevel
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_optimize_point_id_cf(
            self.hnd, settings.optimizePointID ? 1 : 0
        ));
        ErrorImpl::check(rdb_pointcloud_create_settings_set_point_insert_mode_cf(
            self.hnd, settings.pointInsertMode.c_str()
        ));
    }

    static void load(CreateSettings &settings, const std::string &json)
    {
        const CreateSettingsWrapper &self = get(settings);
        ErrorImpl::check(rdb_pointcloud_create_settings_json_load_cf(
            self.hnd, json.c_str()
        ));
        read(settings);
    }

    static std::string save(const CreateSettings &settings)
    {
        post(settings);
        RDBStringWrapper result;
        const CreateSettingsWrapper &self = get(settings);
        ErrorImpl::check(rdb_pointcloud_create_settings_json_save_cf(
            self.hnd, &result.str
        ));
        return result;
    }

private:
    friend class CreateSettings;
    RDBPointcloudCreateSettings *hnd;
};

//---< CreateSettings::PUBLIC >-------------------------------------------------

CreateSettings::~CreateSettings()
{
    if (data)
    {
        delete static_cast<CreateSettingsWrapper*>(data);
    }
}

CreateSettings::CreateSettings():
    primaryAttribute(),
    data(new CreateSettingsWrapper)
{
    CreateSettingsWrapper::read(*this);
}

CreateSettings::CreateSettings(riegl::rdb::Context &context):
    primaryAttribute(context),
    data(new CreateSettingsWrapper)
{
    CreateSettingsWrapper::read(*this);
}

CreateSettings::CreateSettings(const CreateSettings &settings):
    primaryAttribute(),
    data(new CreateSettingsWrapper)
{
    *this = settings;
}

CreateSettings& CreateSettings::operator=(const CreateSettings &settings)
{
    CreateSettingsWrapper::assign(*this, settings);
    return *this;
}

void CreateSettings::load(const std::string &json)
{
    CreateSettingsWrapper::load(*this, json);
}

std::string CreateSettings::save() const
{
    return CreateSettingsWrapper::save(*this);
}

}}} // namespace riegl::rdb::pointcloud
