/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryFill.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point fill query
 * \version 2016-11-08/AW: Initial version
 * \version 2017-11-24/AW: Constructors declared as "explicit" (#2825)
 * \version 2018-07-05/AW: Wrapper function bindBuffer() added
 * \version 2020-01-17/AW: Wrapper function bindMember() added (#3497)
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_QUERYFILL_HPP
#define RIEGL_RDB_POINTCLOUD_QUERYFILL_HPP

//---< INCLUDES >---------------------------------------------------------------

#include <memory>
#include <string>
#include <vector>
#include <cstdlib>
#include <cstdint>

#include "riegl/rdb/pointcloud/dataTypes.hpp"
#include "riegl/rdb/pointcloud/graphNode.hpp"
#include "riegl/rdb/pointcloud/pointcloudData.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< CLASS QueryFill >--------------------------------------------------------
/*!
 * \brief Point fill query
 *
 * This class can be used to set attributes of existing points to a (one)
 * specific value (e.g. set "riegl.selected" to "1" for all points). This
 * query is similar to using select() and update() except that it only accepts
 * one value per point attribute and this value is copied to all points.
 *
 * \see riegl::rdb::Pointcloud::fill()
 *
 * \note You either must delete the query object or call close()
 *       __before__ the parent Pointcloud instance is closed/deleted!
 */
class QueryFill
{
public:
    /*!
     * \brief Default constructor
     *
     * Creates a null query - i.e. the query cannot be used to modify points.
     *
     * \see riegl::rdb::Pointcloud::fill()
     */
    explicit QueryFill();

    /*!
     * \brief Constructor
     *
     * Creates a query prepared for modifying points.
     *
     * \note You cannot create new QueryFill objects this way,
     *       use riegl::rdb::Pointcloud::fill() instead.
     */
    explicit QueryFill(
        riegl::rdb::PointcloudData       *pointcloud,
        const std::vector<GraphNode::ID> *nodes,
        const std::string                &filter
    );

    /*!
     * \brief Check if query is not null
     *
     * \see valid()
     */
    operator bool() const;

    /*!
     * \brief Check if query is not null
     *
     * A null query cannot be used to read points.
     */
    bool valid() const;

    /*!
     * \brief Finish query
     *
     * Call this function when done with modifying points.
     */
    void close();

    /*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a source buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * The buffer is expected to be 1*s*d bytes large, where
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * \note This function just stores the buffer pointer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \note If the value contained in the buffer is equal to the value
     *       of a point in the database, then the point is not updated.
     *       So if the values of all points are equal to the given value,
     *       then no points are updated and the point attribute revision
     *       stored at the primary index node will not change.
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     */
    void bind(
        const std::string &attribute, //!< [in] attribute name
        const DataType     dataType,  //!< [in] buffer data type
        const void        *buffer     //!< [in] buffer location
    );

    //! \copydoc bind()
    template <typename ValueType>
    void bindBuffer(
        const std::string &attribute, //!< [in] attribute name
        const ValueType   &buffer     //!< [in] buffer (data, pointer to data, std::array or std::vector)
    )
    {
        bind(attribute, dataTypeOf(buffer), dataPointerOf(buffer));
    }

    /*!
     * \brief Bind attribute buffer
     *
     * This is a variant of bindBuffer() that allows to bind a member variable
     * of an object as attribute buffer.
     *
     * \see bindBuffer()
     */
    template <typename ObjectType, typename MemberPointer>
    void bindMember(
        const std::string  &attribute, //!< [in] attribute name
        const ObjectType   &object,    //!< [in] e.g. first object of container
        const MemberPointer member     //!< [in] object member variable pointer
    )
    {
        bindBuffer(attribute, object.*member);
    }

    //! \copydoc bindMember()
    template <typename ObjectType, typename MemberPointer>
    void bindMember(
        const std::string  &attribute, //!< [in] attribute name
        const ObjectType   &object,    //!< [in] e.g. first object of container
        const MemberPointer member,    //!< [in] object member variable pointer
        const std::size_t   index      //!< [in] index for array-like object members
    )
    {
        bindBuffer(attribute, (object.*member)[index]);
    }

    /*!
     * \brief Fill points
     *
     * Use this function to actually read the point attributes from
     * all defined buffers and update the points in the database.
     *
     * To process all points, you need to repeatedly call next() until it
     * returns 0 (zero, see example 7). The number of points to process in
     * one step is defined by 'count'. Please note that the actual number
     * of processed points may be slightly different. To cancel processing
     * just stop calling next() and cancel (rollback) the transaction.
     *
     * \note IEEE-754 "NaN" values contained in floating point source
     *       buffers are ignored and the attribute's default value is
     *       used instead. Furthermore IEEE-754 "Infinity" values will
     *       always cause next() to fail with error code 10414, i.e.
     *       riegl::rdb::Error::QueryAttributeValueOutOfRange.
     *
     * \note Since version 2.4.2, unit vector point attributes (e.g.
     *       "riegl.direction") are automatically normalized (divided
     *       by the vector length) on insertion. This is done for all
     *       point attributes that meet the following requirements:
     *       `length` is 3, `minimumValue` is -1, `maximumValue` is 1,
     *       `resolution` is not 1, `scaleFactor` is 1, `unitSymbol` is
     *       empty, `invalidValue` is undefined and `tags` contains the
     *       "direction" tag but not the "do-not-normalize" tag.
     *
     * \returns the number of points processed
     */
    std::uint32_t next(
        std::uint32_t count //!< [in] number of points to process
    );

    /*!
     * \brief Progress
     *
     * This function returns a coarse progress information in percent (0..100%).
     * Since the total number of modified points is not known in advance, this
     * value just reflects the progress of the (internal) index traversal.
     */
    std::uint32_t progress() const;

private:
    struct Private;
    std::shared_ptr<Private> data;
};

}}} // namespace riegl::rdb::pointcloud

#endif // RIEGL_RDB_POINTCLOUD_QUERYFILL_HPP
