/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    management.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Basic point cloud management interface
 * \version 2017-03-21/AW: Initial version
 * \version 2017-04-13/AW: Functions finalize() and vacuum() added
 * \version 2018-05-25/AW: Function validate() added (#3109)
 * \version 2019-01-18/AW: Parameter 'lodMode' added
 *
 * \see riegl::rdb::Pointcloud::management()
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_MANAGEMENT_H
#define RIEGL_RDB_POINTCLOUD_MANAGEMENT_H

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN

//______________________________________________________________________________
/*!
 * \brief Query level of detail mode
 * \see rdb::riegl::pointcloud::Management::LodMode
 */
RDB_FUNCTION(rdb_pointcloud_management_get_lod_mode,
    RDBContext    *context,    //!< [in] library context
    RDBPointcloud *pointcloud, //!< [in] point cloud handle
    uint32_t      *value       //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Modify level of detail mode
 * \see rdb::riegl::pointcloud::Management::LodMode
 */
RDB_FUNCTION(rdb_pointcloud_management_set_lod_mode,
    RDBContext    *context,    //!< [in] library context
    RDBPointcloud *pointcloud, //!< [in] point cloud handle
    uint32_t       value       //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Query level of detail size
 * \see rdb::riegl::pointcloud::Management::LodMode
 */
RDB_FUNCTION(rdb_pointcloud_management_get_chunk_size_lod,
    RDBContext    *context,    //!< [in] library context
    RDBPointcloud *pointcloud, //!< [in] point cloud handle
    uint32_t      *value       //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Modify level of detail size
 * \see rdb::riegl::pointcloud::Management::LodMode
 */
RDB_FUNCTION(rdb_pointcloud_management_set_chunk_size_lod,
    RDBContext    *context,    //!< [in] library context
    RDBPointcloud *pointcloud, //!< [in] point cloud handle
    uint32_t       value       //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Dismiss database history
 *
 * This function deletes all transactions except the first (database
 * creation) and the current transaction (last committed or restored).
 * Please note that this operation only removes the transactions from
 * the database history and releases the related data blocks in the
 * database file so that they can be re-used by subsequent transactions.
 * However the database file size will not decrease unless you call
 * vacuum().
 */
RDB_FUNCTION(rdb_pointcloud_management_finalize,
    RDBContext    *context,   //!< [in] library context
    RDBPointcloud *pointcloud //!< [in] point cloud handle
);

//______________________________________________________________________________
/*!
 * \brief Optimize database file
 *
 * This function reorganizes the data blocks in the database file so
 * that there are no (or as few as possible) unused blocks (gaps).
 * This is especially helpful after deleting point attributes or
 * calling finalize().
 *
 * \note This might be a lengthy operation and no other client can
 *       access the database in the meantime, not even to read.
 */
RDB_FUNCTION(rdb_pointcloud_management_vacuum,
    RDBContext    *context,    //!< [in] library context
    RDBPointcloud *pointcloud, //!< [in] point cloud handle
    RDBProgress    progress,   //!< [in] vacuum progress callback function
    void          *userdata    //!< [in] progress callback function user data
);

//______________________________________________________________________________
/*!
 * \brief Validate database file
 *
 * This function checks whether the database corresponds to the given schema.
 * The schema contains a list of required and optional point attributes and
 * metadata entries and is given in JSON format. Primary point attributes
 * are marked with a "*", optional attributes or metadata entries are
 * marked with a "?" appended to the name, all other items are required.
 *
 * The database must at least contain all primary and required point attributes
 * and all required metadata entries to correspond to the schema. If "strict"
 * is not "0", then the database additionally is not allowed to contain extra
 * point attributes or metadata entries that are not listed in the schema.
 *
 * If the database does not correspond to the schema, the function
 * returns 'RDB_FAILURE' (0) and 'RDB_SUCCESS' (1) otherwise.
 *
 * Example schema JSON string:
 *
 *     {
 *         "extension": "rdbx",
 *         "attributes": [
 *             "riegl.xyz*",
 *             "riegl.timestamp",
 *             "riegl.class?"
 *         ],
 *         "metadata": [
 *             "riegl.geo_tag",
 *             "riegl.device?"
 *         ]
 *     }
 */
RDB_FUNCTION(rdb_pointcloud_management_validate,
    RDBContext    *context,    //!< [in] library context
    RDBPointcloud *pointcloud, //!< [in] point cloud handle
    RDBString      schema,     //!< [in] database schema (JSON)
    uint32_t       strict      //!< [in] >0: strict mode
);

RDB_LIBRARY_API_END

#endif // RIEGL_RDB_POINTCLOUD_MANAGEMENT_H
