/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryStat.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point statistics query (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-07/AW: Optionally filter index graph nodes (#2390)
 * \version 2019-11-05/AW: Optionally return cleaned attribute extents (#3071)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< STRUCT QueryStat::Private >----------------------------------------------

struct QueryStat::Private
{
    RDBContext             *context;
    RDBPointcloud          *pointcloud;
    RDBPointcloudQueryStat *query;

    Private(riegl::rdb::PointcloudData *pcl):
        context(pcl->contextHandle()),
        pointcloud(pcl->pointcloud),
        query(nullptr)
    {
        ErrorImpl::check(context, rdb_pointcloud_query_stat_new(
            context, pointcloud, &query
        ));
    }

    ~Private()
    {
        ErrorImpl::check(context, rdb_pointcloud_query_stat_delete(
            context, &query
        ));
    }
};

//---< QueryStat::PUBLIC >------------------------------------------------------

QueryStat::QueryStat()
{
}

QueryStat::QueryStat(riegl::rdb::PointcloudData *pointcloud):
    data(new Private(pointcloud))
{
}

QueryStat::operator bool() const
{
    return valid();
}

bool QueryStat::valid() const
{
    return (data != nullptr);
}

void QueryStat::close()
{
    data.reset();
}

void QueryStatIndexCopyGraph(
    RDBPointcloudGraphNode *source,
    GraphNode              &target
)
{
    target.id              = source->id;
    target.revision        = source->revision;
    target.pointCountTotal = source->pointCountTotal;
    target.pointCountNode  = source->pointCountNode;
    target.children.resize(source->childCount);
    for (RDBPointcloudGraphNodeChildCount i = 0; i < source->childCount; i++)
    {
        QueryStatIndexCopyGraph(source->children[i], target.children[i]);
    }
}

void QueryStat::index(GraphNode &root, const std::string &filter)
{
    if (!data) return;
    GraphNodeWrapper node(data->context);
    ErrorImpl::check(data->context, rdb_pointcloud_query_stat_index_filter(
        data->context, data->query, node.handle, filter.c_str()
    ));
    QueryStatIndexCopyGraph(node.handle, root);
}

void QueryStat::minimum(
    const GraphNode::ID &nodeID,
    const std::string   &attribute,
    const DataType       dataType,
    void                *buffer,
    const bool           cleaned
)
{
    if (cleaned)
    {
        if (!data) return;
        ErrorImpl::check(data->context, rdb_pointcloud_query_stat_minimum_cleaned(
            data->context, data->query, nodeID,
            attribute.c_str(), dataType, buffer
        ));
    }
    else // full:
    {
        if (!data) return;
        ErrorImpl::check(data->context, rdb_pointcloud_query_stat_minimum(
            data->context, data->query, nodeID,
            attribute.c_str(), dataType, buffer
        ));
    }
}

void QueryStat::maximum(
    const GraphNode::ID &nodeID,
    const std::string   &attribute,
    const DataType       dataType,
    void                *buffer,
    const bool           cleaned
)
{
    if (cleaned)
    {
        if (!data) return;
        ErrorImpl::check(data->context, rdb_pointcloud_query_stat_maximum_cleaned(
            data->context, data->query, nodeID,
            attribute.c_str(), dataType, buffer
        ));
    }
    else // full:
    {
        if (!data) return;
        ErrorImpl::check(data->context, rdb_pointcloud_query_stat_maximum(
            data->context, data->query, nodeID,
            attribute.c_str(), dataType, buffer
        ));
    }
}

void QueryStat::revision(
    const GraphNode::ID &nodeID,
    const std::string   &attribute,
    Transaction::ID     &revision
)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_stat_revision(
        data->context, data->query, nodeID, attribute.c_str(), &revision
    ));
}

}}} // namespace riegl::rdb::pointcloud
