{*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \file    riegl.rdb.pointcloud.querySelect.pas
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point select query (Pascal wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-04/AW: Allow to read from multiple nodes at once (#2368)
 * \version 2018-07-06/AW: Wrapper function BindBuffer() added
 *
 * This class can be used to select (read) attributes of existing points.
 *
 * \see riegl::rdb::Pointcloud::select()
 *
 *******************************************************************************
 *}

unit riegl.rdb.pointcloud.querySelect;

{******************************************************************************}
{***} INTERFACE {**************************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb,
  riegl.rdb.context,
  riegl.rdb.pointcloud.graphNode,
  riegl.rdb.pointcloud.dataTypes,
  riegl.rdb.pointcloud.pointAttribute,
  riegl.rdb.default.attributes;

//---< RDB LIBRARY IMPORTS >----------------------------------------------------

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::select()
 *}
function rdb_pointcloud_query_select_new(
  Context    : TRDBContextHandle;              //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;           //!< [in] point cloud
  Node       : TRDBPointcloudGraphNodeID;      //!< [in] index node ID
  Filter     : TRDBString;                     //!< [in] filter string
  var Query  : TRDBPointcloudQuerySelectHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::select()
 *}
function rdb_pointcloud_query_select_nodes_new(
  Context    : TRDBContextHandle;              //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;           //!< [in] point cloud
  Nodes      : PRDBPointcloudGraphNodeID;      //!< [in] pointer to first element of node ID array
  Count      : TRDBUInt32;                     //!< [in] number of elements in node ID array
  Filter     : TRDBString;                     //!< [in] filter string
  var Query  : TRDBPointcloudQuerySelectHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Destroy query instance
 *}
function rdb_pointcloud_query_select_delete(
  Context   : TRDBContextHandle;              //!< [in] library context
  var Query : TRDBPointcloudQuerySelectHandle //!< [in] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Bind attribute buffer
 *
 * Use this function to define a target buffer for a point attribute.
 * Exactly one buffer can be defined for an attribute (i.e. only the
 * most recently defined buffer will be used).
 *
 * The buffer is expected to be n*s*d bytes large, where
 * __n__ is the number of points defined in next(),
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *
 * \note This function just stores the buffer pointer. So make
 *       sure that the buffer remains valid until you call next().
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 *}
function rdb_pointcloud_query_select_bind(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQuerySelectHandle; //!< [in] query handle
  Attribute : TRDBString;                      //!< [in] attribute name
  DataType  : TRDBUInt32;                      //!< [in] buffer data type \see dataTypes.h
  Buffer    : Pointer;                         //!< [in] buffer location
  Stride    : TRDBInt32 = 0                    //!< [in] bytes between beginnings of successive elements (0: auto)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Select points
 *
 * Use this function to actually read the selected points from
 * database and copy the attributes to the defined buffers.
 *
 * Afterwards you may re-use the buffers or define new buffers
 * with bind() and call next() again until all points have been
 * read (i.e. next() returns zero).
 *
 * \returns the number of points read
 *}
function rdb_pointcloud_query_select_next(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQuerySelectHandle; //!< [in] query handle
  Count     : TRDBUInt32;                      //!< [in] size of source buffers in terms of points
  Processed : PRDBUInt32 = nil                 //!< [out] number of processed points (optional)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Progress
 *
 * This function returns a coarse progress information in percent (0..100%).
 * Since the total number of returned points is not known in advance, this
 * value just reflects the progress of the (internal) index traversal.
 *
 * \note When the select query is used to count points (i.e. no buffers
 *       were bound), then this function always returns 0%.
 *}
function rdb_pointcloud_query_select_progress(
  Context  : TRDBContextHandle;               //!< [in] library context
  Query    : TRDBPointcloudQuerySelectHandle; //!< [in] query handle
  Progress : PRDBUInt32                       //!< [out] progress [0..100%]
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//---< CLASS TRDBPointcloudQuerySelect >----------------------------------------
{*!
 * \brief Point select query
 *
 * This class can be used to select (read) attributes of existing points.
 *
 * \see riegl::rdb::Pointcloud::select()
 *}
type
  TRDBPointcloudQuerySelect = class(System.TObject)
  public
    {*!
     * \brief Default constructor
     *
     * Creates a null query - i.e. the query cannot be used to read points.
     *
     * \see riegl::rdb::Pointcloud::select()
     *}
    constructor Create; reintroduce; overload;

    {*!
     * \brief Constructor
     * \note  You cannot create new TRDBPointcloudQuerySelect objects directly,
     *        use riegl::rdb::Pointcloud::select() instead.
     *}
    constructor Create(
      Parent : System.TObject;
      Nodes  : PRDBPointcloudGraphNodeIDArray;
      Filter : String
    ); overload;

    destructor Destroy; override;

    {*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a target buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * The buffer is expected to be n*s*d bytes large, where
     * __n__ is the number of points defined in next(),
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * \note This function just stores the buffer pointer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     *}
    procedure Bind(
      const Attribute : System.String;          //!< [in] attribute name
      const DataType  : TRDBPointcloudDataType; //!< [in] buffer data type
      const Buffer    : Pointer;                //!< [in] buffer location
      const Stride    : TRDBInt32 = 0           //!< [in] bytes between beginnings of successive elements (0: auto)
    );

    // Helper for fundamental types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;

    // Helper for pointer types
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;

    // Helper for array types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;

    {*!
     * \brief Select points
     *
     * Use this function to actually read the selected points from
     * database and copy the attributes to the defined buffers.
     *
     * Afterwards you may re-use the buffers or define new buffers
     * with bind() and call next() again until all points have been
     * read (i.e. next() returns zero).
     *
     * \returns the number of points read
     *}
    function Next(
      const Count : TRDBUInt32 //!< [in] size of source buffers in terms of points
    ) : TRDBUInt32;

    {*!
     * \brief Progress
     *
     * This function returns a coarse progress information in percent (0..100%).
     * Since the total number of returned points is not known in advance, this
     * value just reflects the progress of the (internal) index traversal.
     *
     * \note When the select query is used to count points (i.e. no buffers
     *       were bound), then this function always returns 0%.
     *}
    function Progress() : TRDBUInt32;

  private
    FContext    : TRDBContext;
    FPointcloud : TRDBPointcloudHandle;
    FQuery      : TRDBPointcloudQuerySelectHandle;
  end;

{******************************************************************************}
{***} IMPLEMENTATION {*********************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb.pointcloud;

//---< TRDBPointcloudQuerySelect::PUBLIC >--------------------------------------

constructor TRDBPointcloudQuerySelect.Create;
begin
  FContext    := nil;
  FPointcloud := nil;
  FQuery      := nil;
end; // Create()

constructor TRDBPointcloudQuerySelect.Create(
  Parent : System.TObject;
  Nodes  : PRDBPointcloudGraphNodeIDArray;
  Filter : String
);
begin
  inherited Create;
  FQuery      := nil;
  FContext    := (Parent as TRDBPointcloud).Context;
  FPointcloud := (Parent as TRDBPointcloud).Pointcloud;
  //
  if (Nodes <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_select_nodes_new(
      FContext.Handle, FPointcloud,
      @Nodes^[0], Length(Nodes^),
      TRDBString(AnsiToUtf8(Filter)),
      FQuery
    ));
  end
  else
  begin
    FContext.Check(rdb_pointcloud_query_select_new(
      FContext.Handle, FPointcloud, 0, // = all nodes
      TRDBString(AnsiToUtf8(Filter)),
      FQuery
    ));
  end;
end; // Create()

destructor TRDBPointcloudQuerySelect.Destroy;
begin
  FContext.Check(rdb_pointcloud_query_select_delete(FContext.Handle, FQuery));
  inherited;
end; // Destroy()

procedure TRDBPointcloudQuerySelect.Bind(
  const Attribute : System.String;
  const DataType  : TRDBPointcloudDataType;
  const Buffer    : Pointer;
  const Stride    : TRDBInt32
);
begin
  if (FQuery <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_select_bind(
      FContext.Handle, FQuery,
      TRDBString(AnsiToUtf8(Attribute)),
      Ord(DataType), Buffer, Stride
    ));
  end;
end; // Bind()

function TRDBPointcloudQuerySelect.Next(const Count : TRDBUInt32) : TRDBUInt32;
begin
  if (FQuery <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_select_next(
      FContext.Handle, FQuery, Count, @Result
    ));
  end
  else Result := 0;
end; // Next()

function TRDBPointcloudQuerySelect.Progress() : TRDBUInt32;
begin
  if (FQuery <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_select_progress(
      FContext.Handle, FQuery, @Result
    ));
  end
  else Result := 0;
end; // Progress()

//______________________________________________________________________________
//
// Helper for fundamental types
//______________________________________________________________________________
//

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

//______________________________________________________________________________
//
// Helper for pointer types
//______________________________________________________________________________
//

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

//______________________________________________________________________________
//
// Helper for array types
//______________________________________________________________________________
//

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQuerySelect.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

end.
