{*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \file    riegl.rdb.pointcloud.transaction.pas
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point cloud transaction (Pascal wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2017-02-16/AW: RDB library version string field added (#2515)
 *
 *******************************************************************************
 *}

unit riegl.rdb.pointcloud.transaction;

{******************************************************************************}
{***} INTERFACE {**************************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb,
  riegl.rdb.context;

//---< RDB LIBRARY IMPORTS >----------------------------------------------------

type
  TRDBPointcloudTransactionID = TRDBUInt32;
  PRDBPointcloudTransactionID = ^TRDBPointcloudTransactionID;

//______________________________________________________________________________
{*!
 * \brief Create transaction object
 *}
function rdb_pointcloud_transaction_new(
  Context         : TRDBContextHandle;              //!< [in] library context
  var Transaction : TRDBPointcloudTransactionHandle //!< [out] transaction handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Delete transaction object
 *}
function rdb_pointcloud_transaction_delete(
  Context         : TRDBContextHandle;              //!< [in] library context
  var Transaction : TRDBPointcloudTransactionHandle //!< [in] transaction handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief transaction number (TAN)
 *}
function rdb_pointcloud_transaction_get_id(
  Context     : TRDBContextHandle;               //!< [in] library context
  Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var ID      : TRDBPointcloudTransactionID      //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief RDB library version string
 *}
function rdb_pointcloud_transaction_get_rdb(
  Context     : TRDBContextHandle;               //!< [in] library context
  Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var RDB     : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief short description, e.g. "Import"
 *}
function rdb_pointcloud_transaction_get_title(
  Context     : TRDBContextHandle;               //!< [in] library context
  Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var Title   : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief software name, e.g. "rdbimport v1.0"
 *}
function rdb_pointcloud_transaction_get_agent(
  Context     : TRDBContextHandle;               //!< [in] library context
  Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var Agent   : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief e.g. process details for humans
 *}
function rdb_pointcloud_transaction_get_comments(
  Context      : TRDBContextHandle;               //!< [in] library context
  Transaction  : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var Comments : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief e.g. process settings for software
 *}
function rdb_pointcloud_transaction_get_settings(
  Context      : TRDBContextHandle;               //!< [in] library context
  Transaction  : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var Settings : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief start time as "yyyy-mm-dd hh:mm:ss.zzz", e.g. "2015-10-14 13:48:35.801" (local time)
 *}
function rdb_pointcloud_transaction_get_start(
  Context     : TRDBContextHandle;               //!< [in] library context
  Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var Start   : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief stop time as "yyyy-mm-dd hh:mm:ss.zzz", e.g. "2015-10-14 13:48:35.801" (local time)
 *}
function rdb_pointcloud_transaction_get_stop(
  Context     : TRDBContextHandle;               //!< [in] library context
  Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  var Stop    : TRDBString                       //!< [out] result value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//---< CLASS TRDBPointcloudTransaction >----------------------------------------
{*!
 * \brief Point cloud transaction
 * \see riegl::rdb::pointcloud::Transactions
 *}
type
  TRDBPointcloudTransaction = class(System.TObject)
  public
    ID       : TRDBPointcloudTransactionID; //!< transaction number (TAN)
    RDB      : System.String;               //!< RDB library version string
    Title    : System.String;               //!< short description, e.g. "Import"
    Agent    : System.String;               //!< software name, e.g. "rdbimport v1.0"
    Comments : System.String;               //!< e.g. process details for humans
    Settings : System.String;               //!< e.g. process settings for software
    Start    : System.String;               //!< start time as "yyyy-mm-dd hh:mm:ss.zzz", e.g. "2015-10-14 13:48:35.801" (local time)
    Stop     : System.String;               //!< stop  time as "yyyy-mm-dd hh:mm:ss.zzz", e.g. "2015-10-14 13:48:35.801" (local time)

    constructor Create(Context : TRDBContext); reintroduce;
    destructor  Destroy(); override;
  end;


//---< TOOLS >------------------------------------------------------------------

//______________________________________________________________________________
{*!
 * \brief Read transaction from library
 * \note For wrapper internal use only.
 *}
procedure PointcloudTransactionRead(
  const Context     : TRDBContext;                     //!< [in] library context
  const Transaction : TRDBPointcloudTransactionHandle; //!< [in] transaction handle
  const Target      : TRDBPointcloudTransaction        //!< [out] target transaction
);

{******************************************************************************}
{***} IMPLEMENTATION {*********************************************************}
{******************************************************************************}

//---< TRDBPointcloudTransaction::PUBLIC >--------------------------------------

constructor TRDBPointcloudTransaction.Create(Context : TRDBContext);
begin
  inherited Create;
end; // Create()

destructor TRDBPointcloudTransaction.Destroy;
begin
  inherited;
end; // Destroy()

//---< TOOLS >------------------------------------------------------------------

//______________________________________________________________________________
procedure PointcloudTransactionRead(
  const Context     : TRDBContext;
  const Transaction : TRDBPointcloudTransactionHandle;
  const Target      : TRDBPointcloudTransaction
);
var
  Local : record
    RDB      : TRDBString;
    Title    : TRDBString;
    Agent    : TRDBString;
    Comments : TRDBString;
    Settings : TRDBString;
    Start    : TRDBString;
    Stop     : TRDBString;
  end;
begin
  FillChar(Local, SizeOf(Local), 0);
  //
  Context.Check(rdb_pointcloud_transaction_get_id(
      Context.Handle, Transaction, Target.ID
  ));
  Context.Check(rdb_pointcloud_transaction_get_rdb(
      Context.Handle, Transaction, Local.RDB
  ));                                                                           Target.RDB := AsSTDString(Local.RDB);
  Context.Check(rdb_pointcloud_transaction_get_title(
      Context.Handle, Transaction, Local.Title
  ));                                                                           Target.Title := AsSTDString(Local.Title);
  Context.Check(rdb_pointcloud_transaction_get_agent(
      Context.Handle, Transaction, Local.Agent
  ));                                                                           Target.Agent := AsSTDString(Local.Agent);
  Context.Check(rdb_pointcloud_transaction_get_comments(
      Context.Handle, Transaction, Local.Comments
  ));                                                                           Target.Comments := AsSTDString(Local.Comments);
  Context.Check(rdb_pointcloud_transaction_get_settings(
      Context.Handle, Transaction, Local.Settings
  ));                                                                           Target.Settings := AsSTDString(Local.Settings);
  Context.Check(rdb_pointcloud_transaction_get_start(
      Context.Handle, Transaction, Local.Start
  ));                                                                           Target.Start := AsSTDString(Local.Start);
  Context.Check(rdb_pointcloud_transaction_get_stop(
      Context.Handle, Transaction, Local.Stop
  ));                                                                           Target.Stop := AsSTDString(Local.Stop);
end; // PointcloudTransactionRead()

end.
