{*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \file    riegl.rdb.pointcloud.queryInsert.pas
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point insert query (Pascal wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2018-07-06/AW: Wrapper function BindBuffer() added
 *
 * This class can be used to insert (new) points into the database.
 *
 * \see riegl::rdb::Pointcloud::insert()
 *
 *******************************************************************************
 *}

unit riegl.rdb.pointcloud.queryInsert;

{******************************************************************************}
{***} INTERFACE {**************************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb,
  riegl.rdb.context,
  riegl.rdb.pointcloud.dataTypes,
  riegl.rdb.pointcloud.pointAttribute,
  riegl.rdb.default.attributes;

//---< RDB LIBRARY IMPORTS >----------------------------------------------------

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::insert()
 *}
function rdb_pointcloud_query_insert_new(
  Context    : TRDBContextHandle;              //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;           //!< [in] point cloud
  var Query  : TRDBPointcloudQueryInsertHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Destroy query instance
 *}
function rdb_pointcloud_query_insert_delete(
  Context   : TRDBContextHandle;              //!< [in] library context
  var Query : TRDBPointcloudQueryInsertHandle //!< [in] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
(*!
 * \brief Load multi-pass insert settings from JSON string
 *
 * This function can be used to change settings for insert queries
 * in multi-pass insert mode (see 'pointInsertMode' in the database
 * creation settings). To be applied, the settings must be loaded
 * before the first call of next(). For settings that have not been
 * explicitly specified, the default value is used. The settings are
 * not saved in the database. When parsing the JSON string fails, an
 * exception is thrown.
 *
 * Supported settings:
 *
 *     - temporary_file_path (string):
 *       Path of the temporary file; if not specified, null or an
 *       empty string, then the database path with suffix ".temp"
 *       is used.
 *       Default: empty
 *
 *     - points_per_block (integer [1]):
 *       Inserted points are grouped into blocks with this number
 *       of points before they are processed further.
 *       Default: 1048576 (= 16 * 65536)
 *
 *     - initial_voxel_size (float [m]):
 *       Inserted points are grouped into voxels with this edge
 *       length in meters. The value must be a power of two of the
 *       resolution of the point coordinates (usually 0.00025 m).
 *       After all points have been inserted, voxels are combined
 *       or subdivided to achieve the desired maximum number of
 *       points per voxel set in the database creation settings
 *       (see 'chunkSize').
 *       Default: 2.048 m
 *
 * Example JSON string:
 *
 *     {
 *         "temporary_file_path": "",
 *         "points_per_block": 1048576,
 *         "initial_voxel_size": 2.048
 *     }
 *
 * \see riegl::rdb::pointcloud::CreateSettings::pointInsertMode
 * \see riegl::rdb::pointcloud::CreateSettings::chunkSize
 *)
function rdb_pointcloud_query_insert_load_multi_pass_settings(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQueryInsertHandle; //!< [in] query handle
  JSON      : TRDBString                       //!< [in] JSON string
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Bind attribute buffer
 *
 * Use this function to define a source buffer for a point attribute.
 * Exactly one buffer can be defined for an attribute (i.e. only the
 * most recently defined buffer will be used).
 *
 * You can but don't need to define a buffer for each attribute. If
 * no buffer is defined for an attribute, the attribute's default
 * value will be used instead.
 *
 * The buffer is expected to be n*s*d bytes large, where
 * __n__ is the number of points defined in next(),
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *
 * If you provide a buffer for the point identifier attribute ("riegl.id"),
 * then it will receive the identifiers of the inserted points (PID). Each
 * point is assigned a unique PID on insertion. The PID starts at 1 for the
 * first point and is incremented by 1 for each subsequent point (so that
 * the PID reflects the insertion order of the points, but only as long as
 * riegl::rdb::pointcloud::CreateSettings::optimizePointID is set to false).
 *
 * \note This function just stores the buffer pointer - it does
 *       __NOT__ copy the data contained in the buffer. So make
 *       sure that the buffer remains valid until you call next().
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 *}
function rdb_pointcloud_query_insert_bind(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQueryInsertHandle; //!< [in] query handle
  Attribute : TRDBString;                      //!< [in] attribute name
  DataType  : TRDBUInt32;                      //!< [in] buffer data type \see dataTypes.h
  Buffer    : Pointer;                         //!< [in] buffer location
  Stride    : TRDBInt32 = 0                    //!< [in] bytes between beginnings of successive elements (0: auto)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Insert points
 *
 * Use this function to actually read the point attributes from
 * all defined buffers and insert the points into the database.
 *
 * Afterwards you may re-fill the buffers or define new buffers
 * with bind() and call next() again until all points have been
 * inserted.
 *
 * \note IEEE-754 "NaN" values contained in floating point source
 *       buffers are ignored and the attribute's default value is
 *       used instead. Furthermore IEEE-754 "Infinity" values will
 *       always cause next() to fail with error code 10414, i.e.
 *       riegl::rdb::Error::QueryAttributeValueOutOfRange.
 *
 * \note Since version 2.4.2, unit vector point attributes (e.g.
 *       "riegl.direction") are automatically normalized (divided
 *       by the vector length) on insertion. This is done for all
 *       point attributes that meet the following requirements:
 *       `length` is 3, `minimumValue` is -1, `maximumValue` is 1,
 *       `resolution` is not 1, `scaleFactor` is 1, `unitSymbol` is
 *       empty, `invalidValue` is undefined and `tags` contains the
 *       "direction" tag but not the "do-not-normalize" tag.
 *
 * \returns the number of points inserted
 *}
function rdb_pointcloud_query_insert_next(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQueryInsertHandle; //!< [in] query handle
  Count     : TRDBUInt32;                      //!< [in] size of source buffers in terms of points
  Processed : PRDBUInt32 = nil                 //!< [out] number of processed points (optional)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//---< CLASS TRDBPointcloudQueryInsert >----------------------------------------
{*!
 * \brief Point insert query
 *
 * This class is used to insert new points into the database.
 *
 * \see riegl::rdb::Pointcloud::insert()
 *}
type
  TRDBPointcloudQueryInsert = class(System.TObject)
  public
    {*!
     * \brief Constructor
     * \note  You cannot create new TRDBPointcloudQueryInsert objects directly,
     *        use riegl::rdb::Pointcloud::insert() instead.
     *}
    constructor Create(Parent : System.TObject); reintroduce;
    destructor  Destroy; override;

    (*!
    * \brief Load multi-pass insert settings from JSON string
    *
    * This function can be used to change settings for insert queries
    * in multi-pass insert mode (see 'pointInsertMode' in the database
    * creation settings). To be applied, the settings must be loaded
    * before the first call of next(). For settings that have not been
    * explicitly specified, the default value is used. The settings are
    * not saved in the database. When parsing the JSON string fails, an
    * exception is thrown.
    *
    * Supported settings:
    *
    *     - temporary_file_path (string):
    *       Path of the temporary file; if not specified, null or an
    *       empty string, then the database path with suffix ".temp"
    *       is used.
    *       Default: empty
    *
    *     - points_per_block (integer [1]):
    *       Inserted points are grouped into blocks with this number
    *       of points before they are processed further.
    *       Default: 1048576 (= 16 * 65536)
    *
    *     - initial_voxel_size (float [m]):
    *       Inserted points are grouped into voxels with this edge
    *       length in meters. The value must be a power of two of the
    *       resolution of the point coordinates (usually 0.00025 m).
    *       After all points have been inserted, voxels are combined
    *       or subdivided to achieve the desired maximum number of
    *       points per voxel set in the database creation settings
    *       (see 'chunkSize').
    *       Default: 2.048 m
    *
    * Example JSON string:
    *
    *     {
    *         "temporary_file_path": "",
    *         "points_per_block": 1048576,
    *         "initial_voxel_size": 2.048
    *     }
    *
    * \see riegl::rdb::pointcloud::CreateSettings::pointInsertMode
    * \see riegl::rdb::pointcloud::CreateSettings::chunkSize
    *)
    procedure LoadMultiPassSettings(const JSON : String);

    {/*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a source buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * You can but don't need to define a buffer for each attribute. If
     * no buffer is defined for an attribute, the attribute's default
     * value will be used instead.
     *
     * The buffer is expected to be n*s*d bytes large, where
     * __n__ is the number of points defined in next(),
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * If you provide a buffer for the point identifier attribute ("riegl.id"),
     * then it will receive the identifiers of the inserted points (PID). Each
     * point is assigned a unique PID on insertion. The PID starts at 1 for the
     * first point and is incremented by 1 for each subsequent point (so that
     * the PID reflects the insertion order of the points, but only as long as
     * riegl::rdb::pointcloud::CreateSettings::optimizePointID is set to false).
     *
     * \note This function just stores the buffer pointer - it does
     *       __NOT__ copy the data contained in the buffer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     *}
    procedure Bind(
      const Attribute : System.String;          //!< [in] attribute name
      const DataType  : TRDBPointcloudDataType; //!< [in] buffer data type
      const Buffer    : Pointer;                //!< [in] buffer location
      const Stride    : TRDBInt32 = 0           //!< [in] bytes between beginnings of successive elements (0: auto)
    );

    // Helper for fundamental types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;

    // Helper for pointer types
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;

    // Helper for array types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;

    {*!
     * \brief Insert points
     *
     * Use this function to actually read the point attributes from
     * all defined buffers and insert the points into the database.
     *
     * Afterwards you may re-fill the buffers or define new buffers
     * with bind() and call next() again until all points have been
     * inserted.
     *
     * \note IEEE-754 "NaN" values contained in floating point source
     *       buffers are ignored and the attribute's default value is
     *       used instead. Furthermore IEEE-754 "Infinity" values will
     *       always cause next() to fail with error code 10414, i.e.
     *       riegl::rdb::Error::QueryAttributeValueOutOfRange.
     *
     * \note Since version 2.4.2, unit vector point attributes (e.g.
     *       "riegl.direction") are automatically normalized (divided
     *       by the vector length) on insertion. This is done for all
     *       point attributes that meet the following requirements:
     *       `length` is 3, `minimumValue` is -1, `maximumValue` is 1,
     *       `resolution` is not 1, `scaleFactor` is 1, `unitSymbol` is
     *       empty, `invalidValue` is undefined and `tags` contains the
     *       "direction" tag but not the "do-not-normalize" tag.
     *
     * \returns the number of points inserted
     *}
    function Next(
      const Count : TRDBUInt32 //!< [in] size of source buffers in terms of points
    ) : TRDBUInt32;

  private
    FContext    : TRDBContext;
    FPointcloud : TRDBPointcloudHandle;
    FQuery      : TRDBPointcloudQueryInsertHandle;
  end;

{******************************************************************************}
{***} IMPLEMENTATION {*********************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb.pointcloud;

//---< TRDBPointcloudQueryInsert::PUBLIC >--------------------------------------

constructor TRDBPointcloudQueryInsert.Create(Parent : System.TObject);
begin
  inherited Create;
  FQuery      := nil;
  FContext    := (Parent as TRDBPointcloud).Context;
  FPointcloud := (Parent as TRDBPointcloud).Pointcloud;
  FContext.Check(rdb_pointcloud_query_insert_new(FContext.Handle, FPointcloud, FQuery));
end; // Create()

destructor TRDBPointcloudQueryInsert.Destroy;
begin
  FContext.Check(rdb_pointcloud_query_insert_delete(FContext.Handle, FQuery));
  inherited;
end; // Destroy()

procedure TRDBPointcloudQueryInsert.LoadMultiPassSettings(const JSON : String);
begin
  FContext.Check(rdb_pointcloud_query_insert_load_multi_pass_settings(
    FContext.Handle, FQuery, TRDBString(AnsiToUtf8(JSON))
  ));
end; // LoadMultiPassSettings()

procedure TRDBPointcloudQueryInsert.Bind(
  const Attribute : System.String;
  const DataType  : TRDBPointcloudDataType;
  const Buffer    : Pointer;
  const Stride    : TRDBInt32
);
begin
  FContext.Check(rdb_pointcloud_query_insert_bind(
    FContext.Handle, FQuery,
    TRDBString(AnsiToUtf8(Attribute)),
    Ord(DataType), Buffer, Stride
  ));
end; // Bind()

function TRDBPointcloudQueryInsert.Next(const Count : TRDBUInt32) : TRDBUInt32;
begin
  FContext.Check(rdb_pointcloud_query_insert_next(
    FContext.Handle, FQuery, Count, @Result
  ));
end; // Next()

//______________________________________________________________________________
//
// Helper for fundamental types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

//______________________________________________________________________________
//
// Helper for pointer types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

//______________________________________________________________________________
//
// Helper for array types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryInsert.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

end.
