/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryUpdate.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point update query
 * \version 2015-10-14/AW: Initial version
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< STRUCT QueryUpdate::Private >--------------------------------------------

struct QueryUpdate::Private
{
    RDBContext               *context;
    RDBPointcloud            *pointcloud;
    RDBPointcloudQueryUpdate *query;

    Private(riegl::rdb::PointcloudData *pcl):
        context(pcl->contextHandle()),
        pointcloud(pcl->pointcloud),
        query(nullptr)
    {
        ErrorImpl::check(context, rdb_pointcloud_query_update_new(
            context, pointcloud, &query
        ));
    }

    ~Private()
    {
        ErrorImpl::check(context, rdb_pointcloud_query_update_delete(
            context, &query
        ));
    }
};

//---< QueryUpdate::PUBLIC >----------------------------------------------------

QueryUpdate::QueryUpdate()
{
}

QueryUpdate::QueryUpdate(riegl::rdb::PointcloudData *pointcloud):
    data(new Private(pointcloud))
{
}

QueryUpdate::operator bool() const
{
    return valid();
}

bool QueryUpdate::valid() const
{
    return (data != nullptr);
}

void QueryUpdate::close()
{
    data.reset();
}

void QueryUpdate::bind(
    const std::string &attribute,
    const DataType     dataType,
    const void        *buffer,
    const std::int32_t stride
)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_update_bind(
        data->context, data->query,
        attribute.c_str(),
        dataType, buffer, stride
    ));
}

std::uint32_t QueryUpdate::next(std::uint32_t count)
{
    if (!data) return 0;
    std::uint32_t result(0);
    ErrorImpl::check(data->context, rdb_pointcloud_query_update_next(
        data->context, data->query, count, &result
    ));
    return result;
}

}}} // namespace riegl::rdb::pointcloud
