/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Main RDB library (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-08/AW: New "fill query" (class QueryFill) added (#1926)
 * \version 2016-11-14/AW: New "invert query" (class QueryInvert) added (#2406)
 * \version 2017-03-21/AW: Database management interface added (#2550)
 * \version 2020-06-29/AW: Database changelog interface added (#3614)
 * \version 2021-05-07/AW: Class Pointcloud is default constructible (#3887)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <memory>
#include <utility>
#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< STRUCT PointcloudData >--------------------------------------------------

namespace riegl {
namespace rdb {

struct PointcloudData // shared by multiple classes
{
    std::shared_ptr<Context::Private> context;
    RDBPointcloud                    *pointcloud;

    std::shared_ptr<pointcloud::Management>      management;
    std::shared_ptr<pointcloud::Changelog>       changelog;
    std::shared_ptr<pointcloud::MetaData>        metaData;
    std::shared_ptr<pointcloud::PointAttributes> pointAttributes;
    std::shared_ptr<pointcloud::Transactions>    transactions;

    explicit PointcloudData(Context parent);
    ~PointcloudData();

    RDBContext *contextHandle();
};

}} // namespace riegl::rdb

//---< C++ WRAPPER CODE >-------------------------------------------------------

#include "riegl/rdb/error.cpp"
#include "riegl/rdb/library.cpp"
#include "riegl/rdb/version.cpp"
#include "riegl/rdb/context.cpp"
#include "riegl/rdb/pointcloud/changelog.cpp"
#include "riegl/rdb/pointcloud/pointAttribute.cpp"
#include "riegl/rdb/pointcloud/createSettings.cpp"
#include "riegl/rdb/pointcloud/openSettings.cpp"
#include "riegl/rdb/pointcloud/management.cpp"
#include "riegl/rdb/pointcloud/metaData.cpp"
#include "riegl/rdb/pointcloud/pointAttributes.cpp"
#include "riegl/rdb/pointcloud/transaction.cpp"
#include "riegl/rdb/pointcloud/transactions.cpp"
#include "riegl/rdb/pointcloud/graphNode.cpp"
#include "riegl/rdb/pointcloud/queryStat.cpp"
#include "riegl/rdb/pointcloud/queryFill.cpp"
#include "riegl/rdb/pointcloud/queryInvert.cpp"
#include "riegl/rdb/pointcloud/queryInsert.cpp"
#include "riegl/rdb/pointcloud/queryUpdate.cpp"
#include "riegl/rdb/pointcloud/queryRemove.cpp"
#include "riegl/rdb/pointcloud/querySelect.cpp"
#include "riegl/rdb/pointcloud.cpp"

//---< PointcloudData::PUBLIC >-------------------------------------------------

namespace riegl {
namespace rdb {

PointcloudData::PointcloudData(Context parent):
    context(std::move(parent.data)),
    pointcloud(nullptr)
{
    ErrorImpl::check(contextHandle(), rdb_pointcloud_new(
        contextHandle(), &pointcloud
    ));
    management      = std::make_shared<pointcloud::Management>     (this);
    changelog       = std::make_shared<pointcloud::Changelog>      (this);
    metaData        = std::make_shared<pointcloud::MetaData>       (this);
    pointAttributes = std::make_shared<pointcloud::PointAttributes>(this);
    transactions    = std::make_shared<pointcloud::Transactions>   (this);
} // constructor

PointcloudData::~PointcloudData()
{
    ErrorImpl::check(contextHandle(), rdb_pointcloud_delete(
        contextHandle(), &pointcloud
    ));
} // destructor

RDBContext *PointcloudData::contextHandle()
{
    return context->context;
} // contextHandle()

}} // namespace riegl::rdb
