/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 * \version 2025-11-11/AW: Item "riegl.lag_angle_correction" added (#6068)
 *
 *******************************************************************************
 */

#ifndef RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1
#define RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1

// Meta data item definition version
const char* const RDB_META_DATA_ITEMS_VERSION = "1.1.57";
const char* const RDB_META_DATA_ITEMS_DATE = "2025-11-11";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
typedef struct MetaDataItemInfo
{
    const char*   name;
    const char*   title;
    const char*   description;
    unsigned char status;
    const char*   schema;
    const char*   example;
} MetaDataItemInfo;

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Angular Notch Filter\",\"properties\":{\"amplitu"
    "de_mean\":{\"type\":\"array\",\"description\":\"Mean amplitude [dB]\",\"items\":{"
    "\"type\":\"number\"}},\"range_mean\":{\"type\":\"array\",\"description\":\"Mean "
    "range [m]\",\"items\":{\"type\":\"number\"}},\"angle\":{\"type\":\"array\",\"descrip"
    "tion\":\"Angle [deg]\",\"items\":{\"type\":\"number\"}}},\"description\":\"Angular"
    " notch filter parameters for window glass echoes\",\"required\":[\"angle\","
    "\"range_mean\",\"amplitude_mean\"],\"$schema\":\"http://json-schema.org/draft"
    "-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude_mean\":[3.780913,3.780913,3.480913,3.120913,2.850913,2.7209"
    "13,2.680913,2.610913,2.530913,2.570913,2.570913],\"range_mean\":[0.094,0"
    ".094,0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475],\""
    "angle\":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Atmospheric Parameters\",\"properties\":{\"tempe"
    "rature\":{\"type\":\"number\",\"description\":\"Temperature along measurement "
    "path "
    "[\\u00b0C]\"},\"rel_humidity\":{\"type\":\"number\",\"description\":\"Relative "
    "humidity along measurement path "
    "[%]\"},\"attenuation\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "attenuation "
    "[1/km]\"},\"pressure\":{\"type\":\"number\",\"description\":\"Pressure along "
    "measurment path [mbar]\"},\"amsl\":{\"type\":\"number\",\"description\":\"Height"
    " above mean sea level (AMSL) "
    "[m]\"},\"pressure_sl\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\"},\"wavelength\":{\"type\":\"number\",\"description\":\"Laser wavelength"
    " [nm]\"},\"group_velocity\":{\"type\":\"number\",\"description\":\"Group "
    "velocity of laser beam [m/s]\"}},\"description\":\"Atmospheric parameters\""
    ",\"required\":[\"temperature\",\"pressure\",\"rel_humidity\",\"pressure_sl\",\"am"
    "sl\",\"group_velocity\",\"attenuation\",\"wavelength\"],\"$schema\":\"http://jso"
    "n-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"temperature\":7,\"rel_humidity\":63,\"attenuation\":0.028125,\"pressure\":9"
    "70,\"amsl\":0,\"pressure_sl\":970,\"wavelength\":1550,\"group_velocity\":29971"
    "1000.0}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Beam Geometry\",\"properties\":{\"beam_exit_diam"
    "eter\":{\"type\":\"number\",\"exclusiveMinimum\":false,\"description\":\"Beam "
    "width at exit aperture [m]\",\"minimum\":0},\"beam_divergence\":{\"type\":\"nu"
    "mber\",\"exclusiveMinimum\":false,\"description\":\"Beam divergence in far "
    "field [rad]\",\"minimum\":0}},\"description\":\"Laser beam geometry details\""
    ",\"required\":[\"beam_exit_diameter\",\"beam_divergence\"],\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"beam_exit_diameter\":0.0072,\"beam_divergence\":0.0003}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Control Object Catalog\",\"properties\":{\"types"
    "\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"oneOf\":[{\"$ref\":\"#/definit"
    "ions/rectangle\"},{\"$ref\":\"#/definitions/checkerboard2x2\"},{\"$ref\":\"#/d"
    "efinitions/chevron\"},{\"$ref\":\"#/definitions/circular_disk\"},{\"$ref\":\"#"
    "/definitions/cylinder\"},{\"$ref\":\"#/definitions/sphere\"},{\"$ref\":\"#/def"
    "initions/round_corner_cube_prism\"}]},\"uniqueItems\":true}},\"description"
    "\":\"List of control object type definitions\",\"required\":[\"types\"],\"$sch"
    "ema\":\"http://json-schema.org/draft-04/schema#\",\"definitions\":{\"sphere\""
    ":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"type\":\"ob"
    "ject\",\"properties\":{\"diameter\":{\"type\":\"number\",\"exclusiveMinimum\":tru"
    "e,\"description\":\"diameter in meters\",\"minimum\":0},\"shape\":{\"type\":\"str"
    "ing\",\"enum\":[\"sphere\"],\"description\":\"shape "
    "identifier\"}},\"description\":\"sphere specific properties\",\"required\":[\""
    "shape\",\"diameter\"]}],\"description\":\"sphere\"},\"cylinder\":{\"type\":\"objec"
    "t\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"type\":\"object\",\"properti"
    "es\":{\"diameter\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"description\""
    ":\"diameter in meters\",\"minimum\":0},\"shape\":{\"type\":\"string\",\"enum\":[\"c"
    "ylinder\"],\"description\":\"shape identifier\"},\"height\":{\"type\":\"number\","
    "\"exclusiveMinimum\":true,\"description\":\"height in "
    "meters\",\"minimum\":0}},\"description\":\"cylinder specific properties\",\"re"
    "quired\":[\"shape\",\"diameter\",\"height\"]}],\"description\":\"cylinder\"},\"che"
    "vron\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"type"
    "\":\"object\",\"properties\":{\"outside_edge_length\":{\"type\":\"number\",\"exclu"
    "siveMinimum\":true,\"description\":\"length of the two outer edges in mete"
    "rs\",\"minimum\":0},\"shape\":{\"type\":\"string\",\"enum\":[\"chevron\"],\"descript"
    "ion\":\"shape identifier\"},\"thickness\":{\"type\":\"number\",\"exclusiveMinimu"
    "m\":true,\"description\":\"thickness in "
    "meters\",\"minimum\":0}},\"description\":\"chevron specific properties\",\"req"
    "uired\":[\"shape\",\"outside_edge_length\",\"thickness\"]}],\"description\":\"ch"
    "evron\"},\"checkerboard2x2\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definit"
    "ions/common\"},{\"type\":\"object\",\"properties\":{\"square_length\":{\"type\":\""
    "number\",\"exclusiveMinimum\":true,\"description\":\"length of a square of "
    "the checkerboard in meters\",\"minimum\":0},\"shape\":{\"type\":\"string\",\"enu"
    "m\":[\"checkerboard2x2\"],\"description\":\"shape "
    "identifier\"}},\"description\":\"checkerboard specific properties\",\"requir"
    "ed\":[\"shape\",\"square_length\"]}],\"description\":\"checkerboard 2 by 2\"},\""
    "circular_disk\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common"
    "\"},{\"type\":\"object\",\"properties\":{\"diameter\":{\"type\":\"number\",\"exclusi"
    "veMinimum\":true,\"description\":\"diameter in meters\",\"minimum\":0},\"shape"
    "\":{\"type\":\"string\",\"enum\":[\"circular_disk\"],\"description\":\"shape "
    "identifier\"},\"offset\":{\"type\":\"number\",\"description\":\"offset in "
    "meters, e.g. reflector constant (optional)\"}},\"description\":\"circular "
    "disk specific "
    "properties\",\"required\":[\"shape\",\"diameter\"]}],\"description\":\"circular "
    "disk\"},\"rectangle\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/co"
    "mmon\"},{\"type\":\"object\",\"properties\":{\"length\":{\"type\":\"number\",\"exclu"
    "siveMinimum\":true,\"description\":\"length in meters\",\"minimum\":0},\"shape"
    "\":{\"type\":\"string\",\"enum\":[\"rectangle\"]},\"width\":{\"type\":\"number\",\"exc"
    "lusiveMinimum\":true,\"description\":\"width in "
    "meters\",\"minimum\":0}},\"description\":\"rectangle specific properties\",\"r"
    "equired\":[\"shape\",\"length\",\"width\"]}],\"description\":\"rectangle\"},\"comm"
    "on\":{\"type\":\"object\",\"properties\":{\"surface_type\":{\"type\":\"string\",\"en"
    "um\":[\"retro_reflective_foil\",\"diffuse\"],\"description\":\"surface "
    "material type\"},\"description\":{\"type\":\"string\",\"description\":\"string "
    "describing the object\"},\"shape\":{\"type\":\"string\",\"enum\":[\"rectangle\",\""
    "checkerboard2x2\",\"chevron\",\"circular_disk\",\"cylinder\",\"sphere\",\"round_"
    "corner_cube_prism\"],\"description\":\"shape identifier\"},\"name\":{\"type\":\""
    "string\",\"minLength\":3,\"description\":\"unique type "
    "identifier\"}},\"description\":\"common object properties\",\"required\":[\"na"
    "me\",\"shape\"]},\"round_corner_cube_prism\":{\"type\":\"object\",\"allOf\":[{\"$r"
    "ef\":\"#/definitions/common\"},{\"type\":\"object\",\"properties\":{\"diameter\":"
    "{\"type\":\"number\",\"exclusiveMinimum\":true,\"description\":\"diameter in me"
    "ters\",\"minimum\":0},\"shape\":{\"type\":\"string\",\"enum\":[\"round_corner_cube"
    "_prism\"],\"description\":\"shape "
    "identifier\"},\"offset\":{\"type\":\"number\",\"description\":\"offset in "
    "meters, e.g. reflector constant (optional)\"}},\"description\":\"round "
    "corner cube prism specific "
    "properties\",\"required\":[\"shape\",\"diameter\"]}],\"description\":\"round "
    "corner cube prism\"}}}"
    ,
    /* example: */
    "{\"types\":[{\"length\":0.6,\"description\":\"Rectangle (60cm x "
    "30cm)\",\"shape\":\"rectangle\",\"name\":\"Rectangle "
    "60x30\",\"width\":0.3},{\"length\":0.8,\"description\":\"Rectangle (80cm x "
    "40cm)\",\"shape\":\"rectangle\",\"name\":\"Rectangle "
    "80x40\",\"width\":0.4},{\"square_length\":0.3,\"description\":\"Checkerboard "
    "(square length: "
    "30cm)\",\"shape\":\"checkerboard2x2\",\"name\":\"Checkerboard2x2 "
    "30\"},{\"square_length\":0.5,\"description\":\"Checkerboard (square length: "
    "50cm)\",\"shape\":\"checkerboard2x2\",\"name\":\"Checkerboard2x2 "
    "50\"},{\"outside_edge_length\":0.6096,\"description\":\"Chevron (a=24''; "
    "b=4'')\",\"shape\":\"chevron\",\"name\":\"Chevron 24''/4''\",\"thickness\":0.1016"
    "},{\"surface_type\":\"diffuse\",\"description\":\" Circular Disk (diameter: "
    "50cm)\",\"shape\":\"circular_disk\",\"name\":\"Circular Disk 50\",\"diameter\":0."
    "5},{\"diameter\":0.05,\"surface_type\":\"retro_reflective_foil\",\"descriptio"
    "n\":\"flat circular reflector from retro-reflective "
    "foil\",\"shape\":\"circular_disk\",\"name\":\"RIEGL flat reflector 50 mm\",\"off"
    "set\":0.0},{\"diameter\":0.1,\"surface_type\":\"retro_reflective_foil\",\"desc"
    "ription\":\"flat circular reflector from retro-reflective "
    "foil\",\"shape\":\"circular_disk\",\"name\":\"RIEGL flat reflector 100 mm\",\"of"
    "fset\":0.0},{\"diameter\":0.15,\"surface_type\":\"retro_reflective_foil\",\"de"
    "scription\":\"flat circular reflector from retro-reflective "
    "foil\",\"shape\":\"circular_disk\",\"name\":\"RIEGL flat reflector 150 mm\",\"of"
    "fset\":0.0},{\"diameter\":0.05,\"surface_type\":\"retro_reflective_foil\",\"de"
    "scription\":\"cylindrical reflector from retro-reflective "
    "foil\",\"shape\":\"cylinder\",\"name\":\"RIEGL cylindrical reflector 50 mm\",\"h"
    "eight\":0.05},{\"diameter\":0.1,\"surface_type\":\"retro_reflective_foil\",\"d"
    "escription\":\"cylindrical reflector from retro-reflective "
    "foil\",\"shape\":\"cylinder\",\"name\":\"RIEGL cylindrical reflector 100 "
    "mm\",\"height\":0.1},{\"description\":\"Sphere (diameter: 200 "
    "mm)\",\"shape\":\"sphere\",\"name\":\"Sphere 200 "
    "mm\",\"diameter\":0.2},{\"offset\":0.0,\"description\":\"round corner cube "
    "prism\",\"shape\":\"round_corner_cube_prism\",\"name\":\"Corner Cube Prism 50 "
    "mm\",\"diameter\":0.05}],\"comments\":[\"This file contains a list of "
    "control object types (aka. 'catalog').\",\"Each type is described by an "
    "object,\",\"which must contain at least the following parameters:\",\"  - "
    "name: unique identifier of the type\",\"  - shape: one of the following "
    "supported shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\""
    "      - chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - "
    "sphere\",\"      - round_corner_cube_prism\",\"Depending on 'shape', the "
    "following parameters must/may be specified:\",\"  - rectangle:\",\"      -"
    " length: length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or 'diffuse')\"]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"reference_file\":{\"type\":\"object\",\"prop"
    "erties\":{\"file_uuid\":{\"type\":\"string\",\"description\":\"Control object "
    "file's Universally Unique Identifier (RFC "
    "4122)\"},\"file_path\":{\"type\":\"string\",\"description\":\"Path of the "
    "control object file relative to referring "
    "file\"}},\"description\":\"Reference to a control object "
    "file\",\"required\":[\"file_uuid\",\"file_path\"]}},\"description\":\"Details "
    "about the control object reference file\",\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\",\"title\":\"Control Object Reference file\"}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\","
    "\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoints.cpx\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Detection Probability\",\"properties\":{\"amplit"
    "ude\":{\"type\":\"array\",\"description\":\"Amplitude [dB]\",\"items\":{\"type\":\"n"
    "umber\"}},\"detection_probability\":{\"type\":\"array\",\"description\":\"Detect"
    "ion probability "
    "[0..1]\",\"items\":{\"type\":\"number\"}}},\"description\":\"Detection "
    "probability as a function of amplitude\",\"required\":[\"amplitude\",\"detec"
    "tion_probability\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\""
    "}"
    ,
    /* example: */
    "{\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],\"detection"
    "_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Information\",\"properties\":{\"channel_t"
    "ext\":{\"type\":\"string\",\"description\":\"Optional channel description "
    "(e.g. 'Green Channel' for multi-channel "
    "devices)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. S2221234)\"},\"channel_number\":{\"type\":\"integer\",\"ex"
    "clusiveMinimum\":false,\"description\":\"Laser channel number (not defined"
    " or 0: single channel device)\",\"minimum\":0},\"device_build\":{\"type\":\"st"
    "ring\",\"description\":\"Device build "
    "variant\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\"}},\"description\":\"Details about the device used to acquire "
    "the point cloud\",\"required\":[\"device_type\",\"serial_number\"],\"$schema\":"
    "\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"serial_number\":\"S2221234\","
    "\"channel_number\":0,\"device_build\":\"\",\"device_name\":\"Scanner 1\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Geometry\",\"properties\":{\"primary\":{\"t"
    "ype\":\"object\",\"properties\":{\"ID\":{\"type\":\"array\",\"maxItems\":2,\"descrip"
    "tion\":\"Structure identifier\",\"items\":{\"type\":\"integer\"},\"minItems\":2},"
    "\"content\":{\"type\":\"array\",\"description\":\"Internal calibration "
    "values\",\"items\":{\"type\":\"number\"}}},\"description\":\"Primary device "
    "geometry structure (mandatory)\",\"required\":[\"ID\",\"content\"]},\"secondar"
    "y\":{\"type\":\"object\",\"properties\":{\"ID\":{\"type\":\"array\",\"maxItems\":2,\"d"
    "escription\":\"Structure identifier\",\"items\":{\"type\":\"integer\"},\"minItem"
    "s\":2},\"content\":{\"type\":\"array\",\"description\":\"Internal calibration "
    "values\",\"items\":{\"type\":\"number\"}}},\"description\":\"Additional device "
    "geometry structure (optional)\",\"required\":[\"ID\",\"content\"]},\"amu\":{\"ty"
    "pe\":\"object\",\"properties\":{\"frameCC\":{\"type\":\"number\",\"exclusiveMinimu"
    "m\":false,\"description\":\"Frame Circle Count (number of LSBs per full "
    "rotation about frame axis)\",\"minimum\":0},\"lineCC\":{\"type\":\"number\",\"ex"
    "clusiveMinimum\":false,\"description\":\"Line Circle Count (number of LSBs"
    " per full rotation about line "
    "axis)\",\"minimum\":0}},\"description\":\"Angle Measurement "
    "Unit\"}},\"description\":\"Scanner device geometry details\",\"required\":[\"p"
    "rimary\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[4,0],\"content\":[0]},\"secondary\":{\"ID\":[91,0],\"conten"
    "t\":[0]},\"amu\":{\"frameCC\":124000,\"lineCC\":124000}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Geometry Passive Channel\",\"properties"
    "\":{\"primary\":{\"type\":\"object\",\"properties\":{\"ID\":{\"type\":\"array\",\"maxI"
    "tems\":2,\"description\":\"Structure identifier\",\"items\":{\"type\":\"integer\""
    "},\"minItems\":2},\"content\":{\"type\":\"array\",\"description\":\"Internal "
    "calibration values\",\"items\":{\"type\":\"number\"}}},\"description\":\"Primary"
    " device geometry structure "
    "(mandatory)\",\"required\":[\"ID\",\"content\"]}},\"description\":\"Scanner "
    "passive channel device geometry details\",\"required\":[\"primary\"],\"$sche"
    "ma\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[143,0],\"content\":[0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"echo_count_maximum\":{\"type\":\"number\",\""
    "description\":\"Maximum number of echoes a laser shot can "
    "have.\"},\"reflectance_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible reflectance in dB.\"},\"background_radiation_maximum\":{\"type\":\""
    "number\",\"description\":\"Maximum possible background radiation.\"},\"backg"
    "round_radiation_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible background radiation.\"},\"deviation_minimum\":{\"type\":\"number\","
    "\"description\":\"Minimum possible pulse shape deviation.\"},\"reflectance_"
    "minimum\":{\"type\":\"number\",\"description\":\"Minimum possible reflectance "
    "in dB.\"},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible range in "
    "meters.\"},\"amplitude_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible amplitude in "
    "dB.\"},\"amplitude_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible amplitude in "
    "dB.\"},\"mta_zone_count_maximum\":{\"type\":\"number\",\"description\":\"Maximum"
    " number of MTA "
    "zones.\"},\"range_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible range in "
    "meters.\"},\"deviation_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible pulse shape deviation.\"}},\"description\":\"Limits of the "
    "measured values output by the device. The limits depend on the device "
    "type, measurement program and/or scan pattern.\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"title\":\"Device Output Limits\"}"
    ,
    /* example: */
    "{\"reflectance_minimum\":-100.0,\"mta_zone_count_maximum\":7,\"range_minimu"
    "m\":2.9,\"amplitude_minimum\":0.0,\"reflectance_maximum\":100.0,\"background"
    "_radiation_minimum\":0,\"background_radiation_maximum\":0,\"amplitude_maxi"
    "mum\":100.0,\"range_maximum\":10000.0,\"deviation_minimum\":-1,\"deviation_m"
    "aximum\":32767}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"array\",\"description\":\"Details about the devices used to "
    "acquire the point cloud (e.g. when merging point clouds of different d"
    "evices)\",\"items\":{\"type\":\"object\",\"properties\":{\"channel_text\":{\"type\""
    ":\"string\",\"description\":\"Optional channel description (e.g. 'Green "
    "Channel' for multi-channel "
    "devices)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. S2221234)\"},\"channel_number\":{\"type\":\"integer\",\"ex"
    "clusiveMinimum\":false,\"description\":\"Laser channel number (not defined"
    " or 0: single channel device)\",\"minimum\":0},\"device_build\":{\"type\":\"st"
    "ring\",\"description\":\"Device build "
    "variant\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\"},\"signed\":{\"type\":\"boolean\",\"description\":\"Flag that is set "
    "when the original 'riegl.device' entry in the source file was "
    "correctly signed.\"}},\"required\":[\"device_type\",\"serial_number\"]},\"$sch"
    "ema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Devices "
    "Information\"}"
    ,
    /* example: */
    "[{\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"serial_number\":\"S2221234\""
    ",\"channel_number\":0,\"device_build\":\"\",\"device_name\":\"Scanner 1\",\"signe"
    "d\":false},{\"channel_text\":\"\",\"device_type\":\"VQ-1560i-DW\",\"serial_numbe"
    "r\":\"S2222680\",\"channel_number\":1,\"device_build\":\"\",\"device_name\":\"Scan"
    "ner 2\",\"signed\":true},{\"channel_text\":\"\",\"device_type\":\"VQ-1560i-DW\",\""
    "serial_number\":\"S2222680\",\"channel_number\":2,\"device_build\":\"\",\"device"
    "_name\":\"Scanner 3\",\"signed\":true}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"DMI Configuration\",\"properties\":{\"dmi_incr\":"
    "{\"type\":\"integer\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":false,\"m"
    "aximum\":2147483647,\"description\":\"Increment in ticks, 0 means distance"
    " trigger is deactivated\",\"minimum\":0},\"dmi_max_time\":{\"type\":\"integer\""
    ",\"exclusiveMinimum\":false,\"exclusiveMaximum\":false,\"maximum\":65535,\"de"
    "scription\":\"Maximum time interval between trigger events, 0 means "
    "timeout is inactive [0.1 msec]\",\"minimum\":0},\"dmi_min_time\":{\"type\":\"i"
    "nteger\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":false,\"maximum\":65"
    "535,\"description\":\"Minimum time interval between trigger events, 0 "
    "means rate limitation is inactive [0.1 msec]\",\"minimum\":0},\"dmi_count_"
    "dir\":{\"type\":\"integer\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":fal"
    "se,\"maximum\":1,\"description\":\"DMI count direction is inverted\",\"minimu"
    "m\":0},\"dmi_dist_per_tick\":{\"type\":\"number\",\"description\":\"Distance per"
    " tick, just informative [m]\"}},\"description\":\"Distance measuring "
    "instrument configuration\",\"required\":[\"dmi_count_dir\",\"dmi_incr\",\"dmi_"
    "min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\"],\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"dmi_incr\":0,\"dmi_max_time\":100,\"dmi_min_time\":0,\"dmi_count_dir\":0,\"d"
    "mi_dist_per_tick\":0.00024399999529123306}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"tab\":{\"type\":\"array\",\"maxItems\":128,\"description\":\"..."
    "\",\"items\":{\"type\":\"object\",\"properties\":{\"dec2\":{\"type\":\"number\",\"desc"
    "ription\":\"Rise of pre-trigger threshold after del2 in units of "
    "dB/m\"},\"del1\":{\"type\":\"number\",\"description\":\"Length of constant "
    "post-trigger threshold elevation in units of "
    "m\"},\"thra2\":{\"type\":\"number\",\"description\":\"Pre-trigger threshold in "
    "units of dB\"},\"thra1\":{\"type\":\"number\",\"description\":\"Post-trigger "
    "threshold in units of "
    "dB\"},\"del2\":{\"type\":\"number\",\"description\":\"Length of constant "
    "pre-trigger threshold elevation in units of "
    "m\"},\"dec1\":{\"type\":\"number\",\"description\":\"Decay of post-trigger "
    "threshold after del1 in units of dB/m\"}},\"description\":\"...\",\"required"
    "\":[\"thra1\",\"del1\",\"dec1\",\"thra2\",\"del2\",\"dec2\"]},\"minItems\":1},\"post\":"
    "{\"type\":\"object\",\"properties\":{\"a\":{\"type\":\"array\",\"maxItems\":2080,\"de"
    "scription\":\"...\",\"items\":{\"type\":\"number\"},\"minItems\":1},\"nc\":{\"type\":"
    "\"integer\",\"description\":\"Number of "
    "entries\",\"minimum\":0,\"maximum\":255}},\"description\":\"Dyntrig post value"
    "s\",\"required\":[\"nc\",\"a\"]},\"scale\":{\"type\":\"number\",\"description\":\"Scal"
    "e in units of dB, for calculation of line: line number N=(amplitude-of"
    "fset)/scale\"},\"pre\":{\"type\":\"object\",\"properties\":{\"a\":{\"type\":\"array\""
    ",\"maxItems\":2080,\"description\":\"...\",\"items\":{\"type\":\"number\"},\"minIte"
    "ms\":1},\"nc\":{\"type\":\"integer\",\"description\":\"Number of "
    "entries\",\"minimum\":0,\"maximum\":255}},\"description\":\"Dyntrig pre values"
    "\",\"required\":[\"nc\",\"a\"]},\"offset\":{\"type\":\"number\",\"description\":\"Offs"
    "et in units of dB, for calculation of line\"}},\"description\":\"Dyntrig\","
    "\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Dyntrig\"}"
    ,
    /* example: */
    "{\"tab\":[{\"dec2\":0.0,\"del1\":0.0,\"thra2\":0.0,\"thra1\":0.0,\"del2\":0.0,\"dec"
    "1\":0.0}],\"post\":{\"a\":[78,86,126,134,31],\"nc\":128},\"scale\":1.0,\"pre\":{\""
    "a\":[78,86,126,134,31],\"nc\":128},\"offset\":0.0}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Echo Information\",\"properties\":{\"echo_file\":"
    "{\"type\":\"object\",\"properties\":{\"file_uuid\":{\"type\":\"string\",\"descripti"
    "on\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Echo file "
    "extension, without the leading "
    "dot\"}},\"required\":[\"file_extension\"]}},\"description\":\"Details about "
    "echo files\",\"required\":[\"echo_file\"],\"$schema\":\"http://json-schema.org"
    "/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_uuid\":\"26a03615-67c0-4bea-8fe8-c577378fe661\",\"file"
    "_extension\":\"owp\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/definition"
    "s/channel\",\"description\":\"one field per channel, field name is channel"
    " index\"}},\"title\":\"Exponential Decomposition\",\"description\":\"Details "
    "for exponential decomposition of full waveform data\",\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\",\"definitions\":{\"channel\":{\"type\":\"o"
    "bject\",\"properties\":{\"parameter\":{\"type\":\"object\",\"properties\":{\"gamma"
    "\":{\"type\":\"array\",\"description\":\"decay in 1/second\",\"items\":{\"type\":\"n"
    "umber\"}},\"A\":{\"type\":\"array\",\"description\":\"real part of amplitude "
    "factor in units of full-scale\",\"items\":{\"type\":\"number\"}},\"omega\":{\"ty"
    "pe\":\"array\",\"description\":\"angular frequency in Hz\",\"items\":{\"type\":\"n"
    "umber\"}},\"B\":{\"type\":\"array\",\"description\":\"imaginary part of "
    "amplitude factor in units of "
    "full-scale\",\"items\":{\"type\":\"number\"}}},\"description\":\"parameters of "
    "the syswave exponential sum\",\"required\":[\"A\",\"B\",\"gamma\",\"omega\"]},\"de"
    "lay\":{\"type\":\"number\",\"description\":\"delay calibration in "
    "seconds\"},\"scale\":{\"type\":\"number\",\"description\":\"amplitude calibratio"
    "n\"},\"a_lin\":{\"type\":\"number\",\"exclusiveMinimum\":false,\"exclusiveMaximu"
    "m\":false,\"maximum\":1,\"description\":\"relative linear amplitude range [0"
    "..1]\",\"minimum\":0}},\"required\":[\"delay\",\"scale\",\"parameter\"]}},\"additi"
    "onalProperties\":false}"
    ,
    /* example: */
    "{\"1\":{\"parameter\":{\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\""
    "A\":[0.9,0.3,-1.3],\"omega\":[352020896.0,3647927552.0,-1977987072.0],\"B\""
    ":[-3.9,0.0,-0.3]},\"delay\":3.5e-09,\"scale\":1.0,\"a_lin\":0.9},\"0\":{\"param"
    "eter\":{\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\"A\":[0.977245"
    "0923919678,0.3354335129261017,-1.312678575515747],\"omega\":[352020896.0"
    ",3647927552.0,-1977987072.0],\"B\":[-3.9813032150268555,0.08622030913829"
    "803,-0.3152860999107361]},\"delay\":3.783458418887631e-09,\"scale\":1.0,\"a"
    "_lin\":0.27}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Gaussian Decomposition\",\"properties\":{\"range"
    "_offset_sec_high_power\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"am"
    "plitude_lsb_high_power\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"ra"
    "nge_offset_sec_low_power\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}},\""
    "amplitude_db\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"amplitude_ls"
    "b_low_power\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"description\""
    ":\"riegl.gaussian_decomposition contains information relevant for "
    "extracting calibrated amplitudes and calibrated ranges from a Gaussian"
    " decomposition of full waveform data. This information is contained in"
    " a table with five columns. Two columns are to be used as input: "
    "amplitude_lsb_low_power and amplitude_lsb_high_power. The other three "
    "columns provide the outputs. Amplitude_db gives the calibrated "
    "amplitude in the optical regime in decibels. The range offset columns "
    "provide additive range offsets, given in units of seconds, for each ch"
    "annel.\",\"required\":[\"amplitude_lsb_low_power\",\"amplitude_lsb_high_powe"
    "r\",\"amplitude_db\",\"range_offset_sec_low_power\",\"range_offset_sec_high_"
    "power\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"range_offset_sec_high_power\":[],\"amplitude_lsb_high_power\":[],\"range"
    "_offset_sec_low_power\":[],\"amplitude_db\":[],\"amplitude_lsb_low_power\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"pose\":{\"type\":\"array\",\"maxItems\":4,\"de"
    "scription\":\"Coordinate Transformation Matrix to transform from File "
    "Coordinate System to Global Coordinate Reference System. 4x4 matrix "
    "stored as two dimensional array, row major order.\",\"items\":{\"type\":\"ar"
    "ray\",\"maxItems\":4,\"description\":\"rows\",\"items\":{\"type\":\"number\",\"descr"
    "iption\":\"columns\"},\"minItems\":4},\"minItems\":4},\"crs\":{\"type\":\"object\","
    "\"properties\":{\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known "
    "Text\\\" string, OGC WKT dialect (see http://www.opengeospatial.org/stan"
    "dards/wkt-crs)\"},\"epsg\":{\"type\":\"integer\",\"description\":\"EPSG "
    "code\",\"minimum\":0},\"name\":{\"type\":\"string\",\"description\":\"Coordinate "
    "reference system name\"}},\"description\":\"Global Coordinate Reference "
    "System. Please note that only 3D Cartesian Coordinate Systems are "
    "allowed.\"}},\"description\":\"Point cloud georeferencing information\",\"$s"
    "chema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Geo Tag\"}"
    ,
    /* example: */
    "{\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,4063"
    "882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847,113"
    "8787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196],[0"
    ".0,0.0,0.0,1.0]],\"crs\":{\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\",DATUM[\\\"WGS"
    "84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\"
    "\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000"
    "000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.0000000000000"
    "0000000,AUTHORITY[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AXIS[\\\"Y\\\",EAS"
    "T],AXIS[\\\"Z\\\",NORTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\",\"epsg\":4978,\"name\""
    ":\"WGS84 Geocentric\"}}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"number\",\"exclusiveMinimum\":true,\"description\":\"Geometric "
    "scale factor applied to point coordinates\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"minimum\":0}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"socs_to_rocs_matrix\":{\"type\":\"array\",\""
    "maxItems\":4,\"description\":\"Coordinate Transformation Matrix to "
    "transform from Scanner's Own Coordinate System to Record Coordinate "
    "System. 4x4 matrix stored as two dimensional array, row major order.\","
    "\"items\":{\"type\":\"array\",\"maxItems\":4,\"description\":\"rows\",\"items\":{\"ty"
    "pe\":\"number\",\"description\":\"columns\"},\"minItems\":4},\"minItems\":4},\"soc"
    "s_to_body_matrix\":{\"type\":\"array\",\"maxItems\":4,\"description\":\"Coordina"
    "te Transformation Matrix to transform from Scanner's Own Coordinate "
    "System to Body Coordinate System. 4x4 matrix stored as two dimensional"
    " array, row major order.\",\"items\":{\"type\":\"array\",\"maxItems\":4,\"descri"
    "ption\":\"rows\",\"items\":{\"type\":\"number\",\"description\":\"columns\"},\"minIt"
    "ems\":4},\"minItems\":4},\"trajectory_offsets\":{\"type\":\"object\",\"propertie"
    "s\":{\"offset_north\":{\"type\":\"number\",\"description\":\"[m]\"},\"offset_yaw\":"
    "{\"type\":\"number\",\"description\":\"[deg]\"},\"offset_time\":{\"type\":\"number\""
    ",\"description\":\"[s]\"},\"offset_roll\":{\"type\":\"number\",\"description\":\"[d"
    "eg]\"},\"version\":{\"type\":\"integer\",\"description\":\"Meaning of offset "
    "values and how to apply them; version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\"},\"offset_pitch\":{\"type\":\"number\",\"description\":\"[de"
    "g]\"},\"offset_height\":{\"type\":\"number\",\"description\":\"[m]\"},\"offset_eas"
    "t\":{\"type\":\"number\",\"description\":\"[m]\"}},\"description\":\"Correction "
    "offsets applied to the trajectory data\"},\"body_coordinate_system_type\""
    ":{\"type\":\"string\",\"enum\":[\"NED\",\"ENU\"],\"description\":\"BODY coordinate "
    "frame (NED: North-East-Down, ENU: East-North-Up), default: NED\"},\"traj"
    "ectory_file\":{\"type\":\"object\",\"properties\":{\"file_uuid\":{\"type\":\"strin"
    "g\",\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Trajectory "
    "file extension, without the leading dot\"}},\"description\":\"Trajectory "
    "data used for georeferencing of the point "
    "cloud\",\"required\":[\"file_extension\"]}},\"description\":\"Parameters used "
    "for georeferencing of the point cloud\",\"$schema\":\"http://json-schema.o"
    "rg/draft-04/schema#\",\"title\":\"Georeferencing Parameters\"}"
    ,
    /* example: */
    "{\"socs_to_body_matrix\":[[-0.269827776749716,-0.723017716139738,0.63595"
    "4678449952,0.0],[0.962908599449764,-0.20260517250352,0.178208229833847"
    ",0.0],[0.0,0.660451759194338,0.7508684796801,0.0],[0.0,0.0,0.0,1.0]],\""
    "trajectory_offsets\":{\"offset_north\":0.07,\"offset_yaw\":-0.45,\"offset_ti"
    "me\":18.007,\"offset_roll\":0.03,\"version\":0,\"offset_pitch\":0.01,\"offset_"
    "height\":-0.2,\"offset_east\":0.15},\"body_coordinate_system_type\":\"NED\",\""
    "trajectory_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\",\""
    "file_extension\":\"pofx\"}}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"base_stations\":{\"type\":\"array\",\"descri"
    "ption\":\"List of GNSS base stations\",\"items\":{\"type\":\"object\",\"properti"
    "es\":{\"longitude\":{\"type\":\"number\",\"description\":\"Longitude "
    "[deg]\"},\"altitude\":{\"type\":\"number\",\"description\":\"Altitude "
    "[m]\"},\"latitude\":{\"type\":\"number\",\"description\":\"Latitude "
    "[deg]\"}},\"description\":\"GNSS base station details\",\"required\":[\"latitu"
    "de\",\"longitude\",\"altitude\"]}}},\"description\":\"Details about GNSS proce"
    "ssing\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"GN"
    "SS Processing Information\"}"
    ,
    /* example: */
    "{\"base_stations\":[{\"longitude\":15.663389822,\"altitude\":379.80041503906"
    "25,\"latitude\":48.658391162}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"patternProperties\":{\"^-?[0-9]+$\":{\"type\":\"string\",\"d"
    "escription\":\"One field per item, field name is item id, field value is"
    " item name\"}},\"title\":\"Item Names\",\"description\":\"Map of item names\",\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\",\"additionalProperti"
    "es\":false}"
    ,
    /* example: */
    "{\"1\":\"Name of item with id 1\",\"2\":\"Name of item with id 2\",\"47\":\"Name "
    "of item with id 47\",\"-1\":\"Name of item with id -1\"}"
};

// Lag angle depended range and reflectance correction
const MetaDataItemInfo RDB_RIEGL_LAG_ANGLE_CORRECTION = {
    /* name        */ "riegl.lag_angle_correction",
    /* title       */ "Lag Angle Correction",
    /* description */ "Lag angle depended range and reflectance correction",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Lag Angle Correction\",\"properties\":{\"range_c"
    "orrection_m\":{\"type\":\"array\",\"description\":\"Range correction [m] as a "
    "function of the range and for applied scanrate, which defines the lag "
    "angle\",\"items\":{\"type\":\"number\"}},\"ranges_m\":{\"type\":\"array\",\"descript"
    "ion\":\"Range [m]\",\"items\":{\"type\":\"number\"}},\"reflectance_correction_db"
    "\":{\"type\":\"array\",\"description\":\"Reflectance correction [db] as a "
    "function of the range and for applied "
    "scanrate\",\"items\":{\"type\":\"number\"}}},\"description\":\"Lag angle "
    "depended range and reflectance correction\",\"required\":[\"ranges_m\",\"ran"
    "ge_correction_m\",\"reflectance_correction_db\"],\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"range_correction_m\":[0.0,0.0,0.0004,0.0023,0.0039,0.0053,0.0053],\"ra"
    "nges_m\":[0.0,4.795,31.964,63.928,95.892,127.856,10000.0],\"reflectance_"
    "correction_db\":[0.0,0.0,0.16,0.46,1.18,2.97,2.97]}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"patternProperties\":{\"^.*$\":{\"type\":\"array\",\"descript"
    "ion\":\"Each field of the object represents a feature and holds a list "
    "of license keys, where the field name is the feature "
    "name.\",\"items\":{\"type\":\"string\",\"description\":\"License key (example: "
    "'46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73')\"},\"minItems\":1}},\"title\":\"Software License "
    "Keys\",\"description\":\"License keys for software features\",\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"additionalProperties\":false}"
    ,
    /* example: */
    "{\"Full Waveform Analysis Topography with GPU support\":[\"8AB44126 - "
    "23B92250 - 16E2689F - 34EF7E7B\"],\"Georeferencing\":[\"46AE032A - "
    "39882AC4 - 9EC0A184 - 6F163D73\"],\"Full Waveform Analysis "
    "Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA - 8D2ED557\"],\"MTA "
    "resolution\":[\"468E020A - 39A922E4 - B681A184 - 673E3D72\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"zone_count\":{\"type\":\"integer\",\"descrip"
    "tion\":\"Maximum number of MTA zones.\",\"minimum\":0,\"maximum\":255},\"zone_"
    "width\":{\"type\":\"number\",\"description\":\"Width of a MTA zone in meter.\","
    "\"minimum\":0},\"modulation_depth\":{\"type\":\"number\",\"description\":\"Depth "
    "of pulse position modulation in "
    "meter.\",\"minimum\":0}},\"description\":\"Parameters for MTA processing\",\"$"
    "schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"MTA "
    "Settings\"}"
    ,
    /* example: */
    "{\"zone_count\":23,\"zone_width\":149.896225,\"modulation_depth\":9.368514}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Near Range Correction "
    "Table\",\"properties\":{\"delta\":{\"type\":\"number\",\"description\":\"Delta "
    "between table entries [m], first entry is at range = 0 m\"},\"content\":{"
    "\"type\":\"array\",\"maxItems\":2000,\"description\":\"Correction value [m] to "
    "be added to the raw "
    "range\",\"items\":{\"type\":\"number\"},\"minItems\":1}},\"description\":\"Lookup "
    "table for range correction based on raw range\",\"required\":[\"delta\",\"co"
    "ntent\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"delta\":0.512,\"content\":[0.0]}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Noise Estimates\",\"properties\":{\"amplitude_si"
    "gma\":{\"type\":\"array\",\"description\":\"Sigma amplitude [dB]\",\"items\":{\"ty"
    "pe\":\"number\"}},\"amplitude\":{\"type\":\"array\",\"description\":\"Amplitude [d"
    "B]\",\"items\":{\"type\":\"number\"}},\"range_sigma\":{\"type\":\"array\",\"descript"
    "ion\":\"Sigma range "
    "[m]\",\"items\":{\"type\":\"number\"}}},\"description\":\"Standard deviation for"
    " range and amplitude as a function of amplitude\",\"required\":[\"amplitud"
    "e\",\"range_sigma\",\"amplitude_sigma\"],\"$schema\":\"http://json-schema.org/"
    "draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0.4"
    "32,0.39,0.354],\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10"
    ".0],\"range_sigma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.0"
    "18,0.016,0.014]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Notch Filter\",\"properties\":{\"amplitude_maxim"
    "um\":{\"type\":\"number\",\"description\":\"Maximum amplitude [dB]\",\"minimum\":"
    "0},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum range "
    "[m]\"},\"range_maximum\":{\"type\":\"number\",\"description\":\"Maximum range "
    "[m]\"}},\"description\":\"Notch filter parameters for window glass echoes\""
    ",\"required\":[\"range_minimum\",\"range_maximum\",\"amplitude_maximum\"],\"$sc"
    "hema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"amplitude_maximum\":18.0,\"range_minimum\":-0.5,\"range_maximum\":0.2}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pixel Information\",\"properties\":{\"size_llcs\""
    ":{\"$ref\":\"#/definitions/pixel_size\",\"description\":\"Size of pixels in a"
    " locally levelled cartesian coordinate system (xy). This is only used "
    "for pixels based on a map projection.\"},\"size\":{\"$ref\":\"#/definitions/"
    "pixel_size\",\"description\":\"Size of pixels in file coordinate "
    "system.\"}},\"description\":\"Details about the pixels contained in the fi"
    "le\",\"required\":[\"size\"],\"$schema\":\"http://json-schema.org/draft-04/sch"
    "ema#\",\"definitions\":{\"pixel_size\":{\"type\":\"array\",\"maxItems\":2,\"descri"
    "ption\":\"Size of "
    "pixels.\",\"items\":{\"type\":\"number\",\"description\":\"Length of pixel edge "
    "[m].\",\"minimum\":0},\"minItems\":2}}}"
    ,
    /* example: */
    "{\"size_llcs\":[0.5156575252891171,0.5130835356683303],\"size\":[0.5971642"
    "834779395,0.5971642834779395]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Plane Patch Matching\",\"properties\":{\"plane_p"
    "atch_file_one\":{\"$ref\":\"#/definitions/file_reference\",\"description\":\"R"
    "eference to the plane patch file one\"},\"plane_patch_file_two\":{\"$ref\":"
    "\"#/definitions/file_reference\",\"description\":\"Reference to the plane "
    "patch file two\"}},\"description\":\"Details about the plane patch "
    "matching process\",\"required\":[\"plane_patch_file_one\",\"plane_patch_file"
    "_two\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definition"
    "s\":{\"file_reference\":{\"type\":\"object\",\"properties\":{\"file_uuid\":{\"type"
    "\":\"string\",\"description\":\"Plane patch file's Universally Unique "
    "Identifier (RFC "
    "4122)\"},\"file_path\":{\"type\":\"string\",\"description\":\"Path of the plane "
    "patch file relative to the match file\"}},\"description\":\"Reference to a"
    " plane patch file\",\"required\":[\"file_uuid\",\"file_path\"]}}}"
    ,
    /* example: */
    "{\"plane_patch_file_one\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b"
    "4213\",\"file_path\":\"Record009_Line001/191025_121410_Scanner_1.ptch\"},\"p"
    "lane_patch_file_two\":{\"file_uuid\":\"fa47d509-a64e-49ce-8b14-ff3130fbefa"
    "9\",\"file_path\":\"project.ptch\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"total_horizontal_area\":{\"type\":\"number"
    "\",\"description\":\"sum of all plane patch areas projected to horizontal "
    "plane [m\\u00b2]\"},\"total_area\":{\"type\":\"number\",\"description\":\"sum of "
    "all plane patch areas [m\\u00b2]\"}},\"description\":\"Statistics about "
    "plane patches found by plane patch extractor\",\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"title\":\"Plane Patch Statistics\"}"
    ,
    /* example: */
    "{\"total_horizontal_area\":13954.601,\"total_area\":14007.965}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Plane Slope Class Info\",\"properties\":{\"setti"
    "ngs\":{\"type\":\"object\",\"description\":\"Classification settings, details "
    "see documentation of rdbplanes\",\"oneOf\":[{\"$ref\":\"#/definitions/method"
    "-1\"},{\"$ref\":\"#/definitions/method-2\"}]},\"classes\":{\"type\":\"object\",\"d"
    "escription\":\"Class definition table\",\"patternProperties\":{\"^[0-9]+$\":{"
    "\"type\":\"string\",\"description\":\"one field per class, field name is "
    "class number, field value is class "
    "name\"}},\"additionalProperties\":false}},\"description\":\"Settings and "
    "classes for plane slope classification\",\"required\":[\"settings\",\"classe"
    "s\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definitions\":"
    "{\"method-2\":{\"type\":\"object\",\"properties\":{\"sloping_plane_classes_maxi"
    "mum_angle\":{\"type\":\"number\",\"description\":\"maximum inclination angle "
    "of sloping plane patches [deg]\",\"minimum\":-360.0,\"maximum\":360.0},\"slo"
    "ping_plane_classes_minimum_angle\":{\"type\":\"number\",\"description\":\"mini"
    "mum inclination angle of sloping plane patches [deg]\",\"minimum\":-360.0"
    ",\"maximum\":360.0},\"plane_classification_method\":{\"type\":\"integer\",\"des"
    "cription\":\"method ID "
    "(=2)\",\"minimum\":2,\"maximum\":2}},\"description\":\"Classification method 2"
    "\",\"required\":[\"plane_classification_method\",\"sloping_plane_classes_min"
    "imum_angle\",\"sloping_plane_classes_maximum_angle\"]},\"method-1\":{\"type\""
    ":\"object\",\"properties\":{\"maximum_inclination_angle_horizontal\":{\"type\""
    ":\"number\",\"description\":\"maximum inclination angle of horizontal plane"
    " patches [deg]\",\"minimum\":-360.0,\"maximum\":360.0},\"plane_classificatio"
    "n_method\":{\"type\":\"integer\",\"description\":\"method ID "
    "(=1)\",\"minimum\":1,\"maximum\":1}},\"description\":\"Classification method 1"
    "\",\"required\":[\"plane_classification_method\",\"maximum_inclination_angle"
    "_horizontal\"]}}}"
    ,
    /* example: */
    "{\"settings\":{\"sloping_plane_classes_maximum_angle\":70.0,\"sloping_plane"
    "_classes_minimum_angle\":10.0,\"plane_classification_method\":2},\"classes"
    "\":{\"3\":\"sloping, pointing up and south\",\"4\":\"sloping, pointing up and "
    "north\",\"1\":\"horizontal, pointing up\",\"10\":\"sloping, pointing down and "
    "east\",\"11\":\"sloping, pointing down and south\",\"14\":\"horizontal, "
    "pointing down\",\"12\":\"sloping, pointing down and north\",\"7\":\"vertical, "
    "pointing south\",\"13\":\"sloping, pointing down and west\",\"6\":\"vertical, "
    "pointing east\",\"2\":\"sloping, pointing up and east\",\"9\":\"vertical, "
    "pointing west\",\"8\":\"vertical, pointing north\",\"5\":\"sloping, pointing "
    "up and west\"}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"patternProperties\":{\"^.*$\":{\"type\":\"array\",\"descript"
    "ion\":\"Each field of the object represents a point attribute group and "
    "holds a list of point attributes, where the field name is the group "
    "name.\",\"items\":{\"type\":\"string\",\"description\":\"Point attribute full "
    "name or name pattern (perl regular expression "
    "syntax)\"},\"minItems\":1}},\"title\":\"Point Attribute "
    "Groups\",\"description\":\"Grouping and sorting of point attributes for "
    "visualization purposes\",\"$schema\":\"http://json-schema.org/draft-04/sch"
    "ema#\",\"additionalProperties\":false}"
    ,
    /* example: */
    "{\"Other Attributes\":[\"riegl.selected\",\"riegl.visible\"],\"Primary Attrib"
    "utes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.deviation\"],\"Coord"
    "inates/Vectors\":[\"riegl.xyz\",\"riegl.range\",\"riegl.theta\",\"riegl.phi\"],"
    "\"Secondary Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available"
    "\"],\"Time\":[\"riegl.timestamp\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"project\":{\"type\":\"string\",\"description"
    "\":\"Project name\"},\"comments\":{\"type\":\"string\",\"description\":\"Comments\""
    "},\"field_of_application\":{\"type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS"
    "\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"description\":\"Field of "
    "application\"}},\"description\":\"Details about point cloud files\",\"$schem"
    "a\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Point Cloud "
    "Information\"}"
    ,
    /* example: */
    "{\"project\":\"Campaign 4\",\"comments\":\"Line "
    "3\",\"field_of_application\":\"ALS\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pose Estimation\",\"properties\":{\"position\":{\""
    "type\":\"object\",\"properties\":{\"coordinate_1\":{\"type\":\"number\",\"descript"
    "ion\":\"Coordinate 1 as defined by axis 1 of the specified CRS (e.g., X,"
    " Latitude)\"},\"coordinate_3\":{\"type\":\"number\",\"description\":\"Coordinate"
    " 3 as defined by axis 3 of the specified CRS (e.g., Z, Altitude)\"},\"ho"
    "rizontal_accuracy\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"descripti"
    "on\":\"Horizontal accuracy [m]\",\"minimum\":0},\"vertical_accuracy\":{\"type\""
    ":\"number\",\"exclusiveMinimum\":true,\"description\":\"Vertical accuracy [m]"
    "\",\"minimum\":0},\"coordinate_2\":{\"type\":\"number\",\"description\":\"Coordina"
    "te 2 as defined by axis 2 of the specified CRS (e.g., Y, Longitude)\"},"
    "\"crs\":{\"type\":\"object\",\"properties\":{\"wkt\":{\"type\":\"string\",\"descripti"
    "on\":\"\\\"Well-Known Text\\\" string, OGC WKT dialect (see http://www.openg"
    "eospatial.org/standards/wkt-crs)\"},\"epsg\":{\"type\":\"integer\",\"descripti"
    "on\":\"EPSG code\",\"minimum\":0}},\"description\":\"Global Coordinate "
    "Reference System\",\"required\":[\"epsg\"]}},\"description\":\"Position "
    "coordinates and position accuracy values as measured by GNSS in the "
    "specified Coordinate Reference System (CRS)\",\"required\":[\"coordinate_1"
    "\",\"coordinate_2\",\"coordinate_3\",\"horizontal_accuracy\",\"vertical_accura"
    "cy\",\"crs\"]},\"orientation\":{\"type\":\"object\",\"properties\":{\"roll\":{\"type"
    "\":\"number\",\"description\":\"Roll angle about scanner X-axis [deg]\",\"mini"
    "mum\":-360,\"maximum\":360},\"roll_accuracy\":{\"type\":\"number\",\"exclusiveMi"
    "nimum\":true,\"description\":\"Roll angle accuracy [deg]\",\"minimum\":0},\"pi"
    "tch_accuracy\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"description\":\""
    "Pitch angle accuracy [deg]\",\"minimum\":0},\"yaw_accuracy\":{\"type\":\"numbe"
    "r\",\"exclusiveMinimum\":true,\"description\":\"Yaw angle accuracy "
    "[deg]\",\"minimum\":0},\"pitch\":{\"type\":\"number\",\"description\":\"Pitch "
    "angle about scanner Y-axis [deg]\",\"minimum\":-360,\"maximum\":360},\"yaw\":"
    "{\"type\":\"number\",\"description\":\"Yaw angle about scanner Z-axis "
    "[deg]\",\"minimum\":-360,\"maximum\":360}},\"description\":\"Orientation "
    "values and orientation accuracies, measured with IMU or inclination se"
    "nsors.\",\"required\":[\"roll\",\"pitch\",\"yaw\",\"roll_accuracy\",\"pitch_accura"
    "cy\",\"yaw_accuracy\"]},\"barometric_height_amsl\":{\"type\":\"number\",\"descri"
    "ption\":\"Altitude determined based on the atmospheric pressure "
    "according to the standard atmosphere laws "
    "[m].\"}},\"description\":\"Estimated position and orientation information "
    "as measured by GNSS, IMU or inclination sensors\",\"required\":[\"orientat"
    "ion\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"position\":{\"coordinate_1\":48.655799473,\"coordinate_3\":362.7124938964"
    "844,\"horizontal_accuracy\":0.810699999332428,\"vertical_accuracy\":1.3314"
    "999341964722,\"coordinate_2\":15.645033406,\"crs\":{\"wkt\":\"GEOGCS[\\\"WGS84 "
    "/ Geographic\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.2572"
    "23563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIM"
    "EM[\\\"Greenwich\\\",0.0000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT"
    "[\\\"Degree\\\",0.01745329251994329547,AUTHORITY[\\\"EPSG\\\",\\\"9102\\\"]],AXIS["
    "\\\"Latitude\\\",NORTH],AXIS[\\\"Longitude\\\",EAST],AUTHORITY[\\\"EPSG\\\",\\\"4979"
    "\\\"]]\",\"epsg\":4979}},\"orientation\":{\"roll\":3.14743073066123,\"roll_accur"
    "acy\":0.009433783936875745,\"pitch_accuracy\":0.009433783936875745,\"yaw_a"
    "ccuracy\":1.0094337839368757,\"pitch\":1.509153024827064,\"yaw\":101.872936"
    "30292045},\"barometric_height_amsl\":386.7457796227932}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pose Sensors\",\"properties\":{\"accelerometer\":"
    "{\"type\":\"object\",\"properties\":{\"origin\":{\"$ref\":\"#/definitions/vector\""
    ",\"description\":\"Sensor origin in SOCS [m] at frame angle = 0\"},\"unit\":"
    "{\"type\":\"number\",\"exclusiveMinimum\":true,\"description\":\"Unit of raw "
    "data and calibration values, 1 LSB in 9.81 m/s\\u00b2\",\"minimum\":0},\"re"
    "lative_nonlinearity\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Rel"
    "ative nonlinearity, dimensionless\"},\"x_axis\":{\"$ref\":\"#/definitions/ve"
    "ctor\",\"description\":\"Sensitive X axis of sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = "
    "0\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement "
    "values\"}},\"description\":\"Accelerometer details\",\"required\":[\"unit\",\"x_"
    "axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_nonlinearity\"]},\"m"
    "agnetic_field_sensor\":{\"type\":\"object\",\"properties\":{\"unit\":{\"type\":\"n"
    "umber\",\"exclusiveMinimum\":true,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in nT\",\"minimum\":0},\"relative_nonlinearity\":"
    "{\"$ref\":\"#/definitions/vector\",\"description\":\"Relative nonlinearity, d"
    "imensionless\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\""
    "Sensitive X axis of sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = "
    "0\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement values\"},\"fixed\":{\"$ref\":\"#/definitio"
    "ns/vector\",\"description\":\"Distortion of magnetic field caused by "
    "non-rotating scanner part\"}},\"description\":\"Magnetic Field Sensor deta"
    "ils\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"fixed\",\"r"
    "elative_nonlinearity\"]},\"gyroscope\":{\"type\":\"object\",\"properties\":{\"or"
    "igin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor origin in "
    "SOCS [m] at frame angle = 0\"},\"unit\":{\"type\":\"number\",\"exclusiveMinimu"
    "m\":true,\"description\":\"Unit of raw data and calibration values, 1 LSB "
    "in rad/s\",\"minimum\":0},\"relative_nonlinearity\":{\"$ref\":\"#/definitions/"
    "vector\",\"description\":\"Relative nonlinearity, dimensionless\"},\"x_axis\""
    ":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X axis of "
    "sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = "
    "0\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement values\"}},\"description\":\"Gyroscope de"
    "tails\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\""
    ",\"relative_nonlinearity\"]}},\"description\":\"Details on position and "
    "orientation sensors\",\"required\":[\"gyroscope\",\"accelerometer\",\"magnetic"
    "_field_sensor\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"d"
    "efinitions\":{\"vector\":{\"type\":\"array\",\"maxItems\":3,\"items\":{\"type\":\"nu"
    "mber\",\"description\":\"Index 0=X, 1=Y, 2=Z component\"},\"minItems\":3}}}"
    ,
    /* example: */
    "{\"accelerometer\":{\"origin\":[0.026900000870227814,-0.03999999910593033,"
    "-0.08950000256299973],\"unit\":6.666666740784422e-05,\"relative_nonlinear"
    "ity\":[0.0,0.0,0.0],\"x_axis\":[-15008.123046875,56.956390380859375,-60.5"
    "175666809082],\"y_axis\":[-7.027288913726807,-44.12333679199219,14952.37"
    "01171875],\"z_axis\":[1.639882206916809,15166.744140625,-116.99742889404"
    "297],\"offset\":[-733.3636474609375,58.969032287597656,1060.255004882812"
    "5]},\"magnetic_field_sensor\":{\"unit\":91.74311828613281,\"relative_nonlin"
    "earity\":[0.0,0.0,0.0],\"x_axis\":[-0.011162743903696537,-2.3159627744462"
    "34e-05,0.00016818844596855342],\"y_axis\":[0.00027888521435670555,-0.011"
    "427424848079681,-5.204829722060822e-05],\"z_axis\":[0.000419878662796691"
    "06,7.876977906562388e-05,0.011407104320824146],\"offset\":[-23812.052734"
    "375,5606.57666015625,2493.28125],\"fixed\":[-1576.010498046875,1596.0817"
    "87109375,0.0]},\"gyroscope\":{\"origin\":[0.026900000870227814,-0.03999999"
    "910593033,-0.08950000256299973],\"unit\":0.00014544410805683583,\"relativ"
    "e_nonlinearity\":[2.888176311444113e-07,1.06274164579645e-07,-1.7186295"
    "080634935e-39],\"x_axis\":[-121.195556640625,0.8219714164733887,0.231303"
    "1703233719],\"y_axis\":[-0.440765917301178,-0.7897399663925171,119.58947"
    "75390625],\"z_axis\":[0.555869996547699,119.22135162353516,0.46758502721"
    "7865],\"offset\":[-50.92609786987305,146.15643310546875,62.4327278137207"
    "]}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pulse Position Modulation\",\"properties\":{\"ph"
    "ase_step\":{\"type\":\"integer\",\"description\":\"Step width in phase of "
    "modulation code from line to line\",\"minimum\":0,\"maximum\":255},\"length\""
    ":{\"type\":\"integer\",\"description\":\"Length of code\",\"minimum\":0,\"maximum"
    "\":255},\"pulse_interval\":{\"type\":\"array\",\"description\":\"Explicit table "
    "of the pulse position modulation used for MTA resolution. Table gives "
    "times between successive laser pulses in seconds.\",\"items\":{\"type\":\"nu"
    "mber\",\"minimum\":0}},\"code_phase_mode\":{\"type\":\"integer\",\"description\":"
    "\"0: no synchronization, 1: toggle between 2 phases, 2: increment with "
    "phase_increment\",\"minimum\":0,\"maximum\":255},\"num_mod_ampl\":{\"type\":\"in"
    "teger\",\"description\":\"Number of different modulation amplitudes (2: "
    "binary modulation)\",\"minimum\":0,\"maximum\":255}},\"description\":\"Laser "
    "pulse position modulation used for MTA resolution\",\"required\":[\"length"
    "\",\"num_mod_ampl\",\"pulse_interval\"],\"$schema\":\"http://json-schema.org/d"
    "raft-04/schema#\"}"
    ,
    /* example: */
    "{\"phase_step\":5,\"length\":31,\"pulse_interval\":[2.759375e-06,2.759375e-0"
    "6,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.8"
    "21875e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375"
    "e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,"
    "2.759375e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759"
    "375e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-"
    "06,2.759375e-06,2.821875e-06],\"code_phase_mode\":2,\"num_mod_ampl\":2}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Range Statistics\",\"properties\":{\"std_dev\":{\""
    "type\":\"number\",\"description\":\"Standard "
    "deviation\"},\"average\":{\"type\":\"number\",\"description\":\"Average "
    "value\"},\"minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "value\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "value\"}},\"description\":\"Statistics about target distance wrt. SOCS ori"
    "gin\",\"required\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"std_dev\":24.349,\"average\":15.49738,\"minimum\":0.919,\"maximum\":574.35}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Receiver Internals\",\"properties\":{\"t\":{\"type"
    "\":\"object\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/definitions/fp\","
    "\"description\":\"one field per channel, field name is channel index\"}},\""
    "additionalProperties\":false},\"tbl\":{\"type\":\"array\",\"description\":\"vari"
    "ous internal data\",\"items\":{\"$ref\":\"#/definitions/fp_table\"},\"minItems"
    "\":1},\"ex\":{\"type\":\"object\",\"description\":\"DEPRECATED, use "
    "'riegl.exponential_decomposition' instead\"},\"mw\":{\"type\":\"number\",\"exc"
    "lusiveMinimum\":true,\"description\":\"Maximum weight\",\"minimum\":0,\"maximu"
    "m\":4095},\"nt\":{\"type\":\"integer\",\"description\":\"Number of traces\",\"mini"
    "mum\":0,\"maximum\":255},\"sr\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"d"
    "escription\":\"Sample rate "
    "[Hz]\",\"minimum\":0},\"ns\":{\"type\":\"integer\",\"description\":\"Number of sam"
    "ples\",\"minimum\":0,\"maximum\":4095},\"a\":{\"type\":\"array\",\"maxItems\":256,\""
    "description\":\"Amplitude [dB]\",\"items\":{\"type\":\"number\"},\"minItems\":1},"
    "\"si\":{\"type\":\"number\",\"description\":\"Start index "
    "(hw_start)\",\"minimum\":0,\"maximum\":255}},\"description\":\"Receiver Intern"
    "als\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definitions\""
    ":{\"fp_table_row\":{\"type\":\"array\",\"maxItems\":2048,\"items\":{\"type\":\"numb"
    "er\"},\"minItems\":1},\"fp\":{\"type\":\"object\",\"properties\":{\"w\":{\"type\":\"ar"
    "ray\",\"maxItems\":256,\"items\":{\"type\":\"array\",\"maxItems\":5,\"items\":{\"typ"
    "e\":\"number\"},\"minItems\":5},\"minItems\":1},\"s\":{\"type\":\"array\",\"maxItems"
    "\":256,\"items\":{\"type\":\"array\",\"maxItems\":4096,\"items\":{\"type\":\"number\""
    "},\"minItems\":1},\"minItems\":1}},\"description\":\"Fingerprint values\",\"req"
    "uired\":[\"s\",\"w\"]},\"fp_table\":{\"type\":\"object\",\"properties\":{\"tc\":{\"typ"
    "e\":\"integer\",\"description\":\"table data type "
    "code\",\"max\":255,\"min\":0},\"ch\":{\"type\":\"integer\",\"description\":\"channel"
    " number\",\"max\":255,\"min\":0},\"tv\":{\"type\":\"array\",\"maxItems\":2048,\"item"
    "s\":{\"oneOf\":[{\"$ref\":\"#/definitions/fp_table_row\"},{\"type\":\"number\"}]}"
    ",\"minItems\":1},\"ny\":{\"type\":\"integer\",\"description\":\"number of y entri"
    "es\",\"max\":2048,\"min\":1},\"nx\":{\"type\":\"integer\",\"description\":\"number "
    "of x entries\",\"max\":2048,\"min\":1}},\"required\":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv"
    "\"],\"desription\":\"scanner internal data\"}}}"
    ,
    /* example: */
    "{\"t\":{\"1\":{\"w\":[[78,86,126,134,31],[78,86,126,134,31]],\"s\":[[1.23,4.56"
    "],[7.89,0.12]]},\"0\":{\"w\":[[78,86,126,134,31],[78,86,126,134,31]],\"s\":["
    "[1.23,4.56],[7.89,0.12]]}},\"tbl\":[{\"tc\":1,\"ch\":0,\"tv\":[[1,2,3,4,5],[1."
    "1,2.2,3.3,4.4,5.5]],\"ny\":2,\"nx\":5}],\"mw\":31,\"nt\":128,\"sr\":7959997000.0"
    ",\"ns\":400,\"a\":[-1.55],\"si\":48}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Reflectance Calculation "
    "Table\",\"properties\":{\"delta\":{\"type\":\"number\",\"description\":\"Delta "
    "between table entries [m], first entry is at range = 0 m\"},\"content\":{"
    "\"type\":\"array\",\"maxItems\":2000,\"description\":\"Correction value [dB] to"
    " be added to the amplitude\",\"items\":{\"type\":\"number\"},\"minItems\":1}},\""
    "description\":\"Lookup table for reflectance calculation based on "
    "amplitude and range\",\"required\":[\"delta\",\"content\"],\"$schema\":\"http://"
    "json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"delta\":0.150918,\"content\":[-33.01]}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Near-range reflectance correction\",\"properti"
    "es\":{\"line_angles_deg\":{\"type\":\"array\",\"description\":\"Angle [deg]\",\"it"
    "ems\":{\"type\":\"number\"}},\"ranges_m\":{\"type\":\"array\",\"description\":\"Rang"
    "e [m]\",\"items\":{\"type\":\"number\"}},\"reflectance_correction_db\":{\"type\":"
    "\"array\",\"description\":\"Near range reflectance correction in dB as a "
    "function of range over "
    "angle\",\"items\":{\"type\":\"array\",\"description\":\"rows (each array "
    "corresponds to a "
    "range)\",\"items\":{\"type\":\"number\",\"description\":\"columns (each value "
    "corresponds to an angle)\"}}}},\"description\":\"Range-dependent and "
    "scan-angle-dependent correction of reflectance reading\",\"required\":[\"r"
    "anges_m\",\"line_angles_deg\",\"reflectance_correction_db\"],\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2.0,2.5,3.0,3.5,4.0],\"ranges_m"
    "\":[0.0,1.0,2.0,3.0],\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,"
    "0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0."
    "8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,"
    "0.1,0.05,0.01]]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"RiLOC Processing Information\",\"properties\":{"
    "\"trajectory\":{\"type\":\"object\",\"properties\":{},\"description\":\"Trajector"
    "y information and statistics.\"},\"optimizer\":{\"type\":\"object\",\"properti"
    "es\":{},\"description\":\"Optimization result information and statistics.\""
    "},\"planes\":{\"type\":\"object\",\"properties\":{},\"description\":\"Information"
    " on plane correspondences and used extraction/matching templates.\"},\"g"
    "eoreferencing\":{\"type\":\"object\",\"properties\":{},\"description\":\"Interna"
    "l georeferencing parameters.\"}},\"description\":\"Details about RiLOC pro"
    "cessing\",\"required\":[\"trajectory\",\"optimizer\",\"georeferencing\"],\"$sche"
    "ma\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"trajectory\":{},\"optimizer\":{},\"planes\":{},\"georeferencing\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"line\":{\"type\":\"object\",\"properties\":{\"start\":{\"type\":\""
    "number\",\"description\":\"Start angle in SOCS [deg]\",\"minimum\":0.0,\"maxim"
    "um\":360.0},\"increment\":{\"type\":\"number\",\"description\":\"Increment of "
    "angle in SOCS [deg]\",\"minimum\":0.0,\"maximum\":90.0},\"stop\":{\"type\":\"num"
    "ber\",\"description\":\"Stop angle in SOCS [deg]\",\"minimum\":0.0,\"maximum\":"
    "720.0},\"program\":{\"$ref\":\"#/definitions/program\"}},\"description\":\"Line"
    " Scan Pattern\",\"required\":[\"start\",\"stop\",\"increment\"]},\"rectangular\":"
    "{\"type\":\"object\",\"properties\":{\"phi_stop\":{\"type\":\"number\",\"descriptio"
    "n\":\"Stop phi angle in SOCS [deg]\",\"minimum\":0.0,\"maximum\":720.0},\"thet"
    "a_stop\":{\"type\":\"number\",\"description\":\"Stop theta angle in SOCS [deg]"
    "\",\"minimum\":0.0,\"maximum\":180.0},\"theta_increment\":{\"type\":\"number\",\"d"
    "escription\":\"Increment of theta angle in SOCS [deg]\",\"minimum\":0.0,\"ma"
    "ximum\":90.0},\"phi_start\":{\"type\":\"number\",\"description\":\"Start phi "
    "angle in SOCS [deg]\",\"minimum\":0.0,\"maximum\":360.0},\"program\":{\"$ref\":"
    "\"#/definitions/program\"},\"theta_start\":{\"type\":\"number\",\"description\":"
    "\"Start theta angle in SOCS [deg]\",\"minimum\":0.0,\"maximum\":180.0},\"phi_"
    "increment\":{\"type\":\"number\",\"description\":\"Increment of phi angle in "
    "SOCS [deg]\",\"minimum\":0.0,\"maximum\":90.0}},\"description\":\"Rectangular "
    "Field Of View Scan Pattern\",\"required\":[\"phi_start\",\"phi_stop\",\"phi_in"
    "crement\",\"theta_start\",\"theta_stop\",\"theta_increment\"]},\"segments\":{\"t"
    "ype\":\"object\",\"properties\":{\"program\":{\"$ref\":\"#/definitions/program\"}"
    ",\"list\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"properties\":{\"start\""
    ":{\"type\":\"number\",\"description\":\"Start angle in SOCS [deg]\",\"minimum\":"
    "0.0,\"maximum\":360.0},\"increment\":{\"type\":\"number\",\"description\":\"Incre"
    "ment of angle in SOCS [deg]\",\"minimum\":0.0,\"maximum\":90.0},\"stop\":{\"ty"
    "pe\":\"number\",\"description\":\"Stop angle in SOCS "
    "[deg]\",\"minimum\":0.0,\"maximum\":720.0}},\"description\":\"Line Scan Segmen"
    "t\",\"required\":[\"start\",\"stop\",\"increment\"]}}},\"description\":\"Segmented"
    " Line Scan Pattern\",\"required\":[\"list\"]}},\"description\":\"Scan pattern "
    "description\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"defi"
    "nitions\":{\"program\":{\"type\":\"object\",\"properties\":{\"laser_prr\":{\"type\""
    ":\"number\",\"exclusiveMinimum\":false,\"description\":\"Laser Pulse "
    "Repetition Rate "
    "[Hz]\",\"minimum\":0},\"name\":{\"type\":\"string\",\"description\":\"Name of "
    "measurement program\"}},\"description\":\"Measurement "
    "program\",\"required\":[\"name\"]}},\"title\":\"Scan Pattern\"}"
    ,
    /* example: */
    "{\"rectangular\":{\"phi_stop\":270.0,\"theta_stop\":130.0,\"theta_increment\":"
    "0.04,\"phi_start\":45.0,\"program\":{\"laser_prr\":100000.0,\"name\":\"High "
    "Speed\"},\"theta_start\":30.0,\"phi_increment\":0.04}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"shot_file\":{\"type\":\"object\",\"propertie"
    "s\":{\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Shot file "
    "extension, without the leading "
    "dot\"}},\"required\":[\"file_extension\"]}},\"description\":\"Details about "
    "laser shot files\",\"$schema\":\"http://json-schema.org/draft-04/schema#\","
    "\"title\":\"Shot Information\"}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_uuid\":\"26a00815-67c0-4bff-8fe8-c577378fe663\",\"file"
    "_extension\":\"sodx\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Stored filters\",\"properties\":{\"activated\":{\""
    "type\":\"boolean\",\"description\":\"Apply ('true') or ignore ('false') all "
    "filters\"},\"filters\":{\"type\":\"array\",\"description\":\"List of point filte"
    "rs\",\"items\":{\"type\":\"object\",\"properties\":{\"activated\":{\"type\":\"boolea"
    "n\",\"description\":\"Apply ('true') or ignore ('false') this "
    "filter\"},\"filter\":{\"type\":\"string\",\"description\":\"The RDB filter "
    "string to apply (e.g. when reading points or index), details see "
    "documentation of function "
    "select()\"},\"description\":{\"type\":\"string\",\"description\":\"A brief "
    "description of the filter (e.g. for display in a "
    "tooltip)\"},\"title\":{\"type\":\"string\",\"description\":\"A short filter "
    "title (e.g. for display in a selection list)\"}},\"description\":\"Point "
    "filter definition\",\"required\":[\"activated\",\"title\",\"description\",\"filt"
    "er\"]}}},\"description\":\"Point filters applied in addition to the "
    "application-defined filters\",\"required\":[\"activated\",\"filters\"],\"$sche"
    "ma\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"activated\":true,\"filters\":[{\"activated\":false,\"filter\":\"riegl.mta_un"
    "certain_point == 0\",\"description\":\"Ignore points with uncertain MTA "
    "zone assignment\",\"title\":\"Ignore uncertain points from MTA resolution\""
    "},{\"activated\":false,\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\",\"description\":\"Ignore points with an echo signal amplitude below "
    "the dynamic trigger\",\"title\":\"Ignore points below dynamic trigger "
    "level\"},{\"activated\":false,\"filter\":\"riegl.point_outside_aabb_bocs == "
    "0\",\"description\":\"Ignore points that are outside the outer AABB in "
    "BOCS\",\"title\":\"Ignore points outside outer "
    "AABB\"},{\"activated\":false,\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\",\"description\":\"Ignore points that are inside the inner AABB in "
    "BOCS\",\"title\":\"Ignore points inside inner AABB\"}]}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"System Description\",\"properties\":{\"system\":{"
    "\"type\":\"object\",\"description\":\"The actual system description, details "
    "see documentation of RIEGL System "
    "Description\"},\"timestamp\":{\"type\":\"string\",\"description\":\"Date and "
    "time of creation (e.g. 2019-06-19T13:32:10+02:00)\"},\"author\":{\"type\":\""
    "string\",\"description\":\"Name of the author that created the "
    "document\"},\"version\":{\"type\":\"string\",\"description\":\"Document format v"
    "ersion\"},\"$class\":{\"type\":\"string\",\"enum\":[\"Document\"],\"description\":\""
    "Object class name\"}},\"description\":\"Details of major system components"
    " like lidar sensors, cameras and navigation devices\",\"required\":[\"$cla"
    "ss\",\"version\",\"author\",\"timestamp\",\"system\"],\"$schema\":\"http://json-sc"
    "hema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"system\":{\"details\":\"see documentation of RIEGL System "
    "Description\"},\"timestamp\":\"2022-09-30T11:56:26+00:00\",\"author\":\"RIEGL "
    "LMS GmbH, Austria\",\"version\":\"1.2.1\",\"$class\":\"Document\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Temperature Calculation Table\",\"properties\":"
    "{\"InGaAs\":{\"$ref\":\"#/definitions/conversion_table\",\"description\":\"Conv"
    "ersion table for InGaAs channel\"},\"InGaAs_Si_Difference\":{\"$ref\":\"#/de"
    "finitions/conversion_table\",\"description\":\"Conversion table for InGaAs"
    " - Si difference\"},\"Si\":{\"$ref\":\"#/definitions/conversion_table\",\"desc"
    "ription\":\"Conversion table for Si channel\"}},\"description\":\"Conversion"
    " of background radiation raw values to temperatures in \\u00b0C\",\"$sche"
    "ma\":\"http://json-schema.org/draft-04/schema#\",\"definitions\":{\"conversi"
    "on_table\":{\"type\":\"object\",\"properties\":{\"value\":{\"type\":\"array\",\"desc"
    "ription\":\"LSB [1]\",\"items\":{\"type\":\"number\"}},\"temperature\":{\"type\":\"a"
    "rray\",\"description\":\"Temperature [\\u00b0C]\",\"items\":{\"type\":\"number\"}}"
    "},\"required\":[\"value\",\"temperature\"]}}}"
    ,
    /* example: */
    "{\"InGaAs\":{\"value\":[0.0,64.00097659230323,128.0019531846065,192.002929"
    "7769097,256.0039063692129,320.00488296151616,384.0058595538194,448.006"
    "8361461226,512.0078127384259],\"temperature\":[307.22196722535614,309.11"
    "53478247277,311.1188086915047,313.10025350480055,315.2137946389828,317"
    ".2172555057597,319.2207163725366,321.2021611858325,323.3157023200148]}"
    ",\"InGaAs_Si_Difference\":{\"value\":[1000.0,1100.090029602954,1200.044251"
    "83874,1300.1342814416948,1400.0885036774805,1500.0427259132668,1600.13"
    "27555162209,1700.0869777520065,1800.0411999877924],\"temperature\":[1749"
    ".977111117893,1749.977111117893,1749.977111117893,1749.977111117893,17"
    "49.977111117893,1749.977111117893,1744.7813348796044,1681.997131260109"
    "2,1622.3944822534868]},\"Si\":{\"value\":[0.0,64.00097659230323,128.001953"
    "1846065,192.0029297769097,256.0039063692129,320.00488296151616,384.005"
    "8595538194,448.0068361461226,512.0078127384259],\"temperature\":[546.300"
    "048828125,548.8164051212026,551.3143938500972,554.0144257850053,556.60"
    "4252334815,559.2124464488079,561.8022729986177,564.4104671126105,567.0"
    "002936624203]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Time "
    "Base\",\"properties\":{\"epoch\":{\"type\":\"string\",\"description\":\"Date and "
    "time of timestamp '0' as proposed by RFC 3339 (e.g. 2015-10-27T00:00:0"
    "0+01:00).\"},\"system\":{\"type\":\"string\",\"enum\":[\"unknown\",\"UTC\",\"GPS\"],\""
    "description\":\"Time system (time standard)\"},\"source\":{\"type\":\"string\","
    "\"enum\":[\"unknown\",\"RTC\",\"GNSS\"],\"description\":\"Timestamp "
    "source\"}},\"description\":\"Base of timestamps (epoch)\",\"required\":[\"epoc"
    "h\",\"source\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"epoch\":\"2015-10-27T00:00:00+00:00\",\"system\":\"UTC\",\"source\":\"GNSS\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Trajectory Information\",\"properties\":{\"proje"
    "ct\":{\"type\":\"string\",\"description\":\"Project "
    "name\"},\"location\":{\"type\":\"string\",\"description\":\"Project location, "
    "e.g. city/state/country\"},\"time_interval\":{\"type\":\"object\",\"properties"
    "\":{\"std_dev\":{\"type\":\"number\",\"description\":\"Standard deviation of "
    "intervals [s]\"},\"average\":{\"type\":\"number\",\"description\":\"Average time"
    " interval [s]\"},\"minimum\":{\"type\":\"number\",\"description\":\"Minimum time"
    " interval [s]\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum time"
    " interval [s]\"}},\"description\":\"Time interval statistics\",\"required\":["
    "\"minimum\",\"average\",\"maximum\",\"std_dev\"]},\"settings\":{\"type\":\"string\","
    "\"description\":\"Settings used to calculate the trajectory (descriptive "
    "text)\"},\"company\":{\"type\":\"string\",\"description\":\"Company name\"},\"fiel"
    "d_of_application\":{\"type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\""
    ",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"description\":\"Field of "
    "application\"},\"device\":{\"type\":\"string\",\"description\":\"Navigation "
    "device, e.g. name/type/serial\"},\"navigation_frame\":{\"type\":\"string\",\"e"
    "num\":[\"unknown\",\"NED\",\"ENU\"],\"description\":\"Navigation frame (NED: "
    "North-East-Down, ENU: "
    "East-North-Up)\"},\"software\":{\"type\":\"string\",\"description\":\"Software "
    "that calculated the trajectory (this may be the same or different "
    "software than the one that created the file)\"}},\"description\":\"Details"
    " about position+orientation files\",\"required\":[\"time_interval\",\"naviga"
    "tion_frame\"],\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"project\":\"Campaign 3\",\"location\":\"Horn\",\"time_interval\":{\"std_dev\":5"
    ".51e-07,\"average\":0.00500053,\"minimum\":0.00500032,\"maximum\":0.00500488"
    "3},\"settings\":\"default\",\"company\":\"RIEGL "
    "LMS\",\"field_of_application\":\"MLS\",\"device\":\"IMU Model 12/1, Serial# "
    "12345\",\"navigation_frame\":\"NED\",\"software\":\"Navigation Software XYZ\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"TTIP Configuration\",\"properties\":{\"ext_subdi"
    "vider\":{\"type\":\"integer\",\"description\":\"reduces the frequency, default"
    " 1\",\"minimum\":0},\"ttip_version\":{\"type\":\"integer\",\"description\":\"follo"
    "wing main.sub.ss.sss\",\"minimum\":0},\"dmi_dist_per_tick\":{\"type\":\"number"
    "\",\"description\":\"dmi, distance per tick, just informative "
    "[m]\"},\"output_usage\":{\"type\":\"integer\",\"description\":\"0 .. no output, "
    "1 .. output derived from internal clock, 2 .. output derived from dmi,"
    " 3 .. output derived from external signal, 4 .. output static low, 5 "
    ".. output static high\",\"minimum\":0},\"out_interval\":{\"type\":\"integer\",\""
    "description\":\"output pulse interval [0.1 msec]\",\"minimum\":0},\"in_min_d"
    "uration\":{\"type\":\"integer\",\"description\":\"input signals with smaller "
    "pulse durations are ignored [0.1 msec]\",\"minimum\":0},\"dmi_min_time\":{\""
    "type\":\"integer\",\"description\":\"dmi, minimum time interval between "
    "trigger events [0.1 msec]\",\"minimum\":0},\"ext_delay\":{\"type\":\"integer\","
    "\"description\":\"external trigger, time delay [0.1 msec]\",\"minimum\":0},\""
    "out_num_pulses\":{\"type\":\"integer\",\"description\":\"number of output "
    "pulses to be generated, 0 .. infinite\",\"minimum\":0},\"ext_channel\":{\"ty"
    "pe\":\"integer\",\"description\":\"ID of channel used as external trigger "
    "input, 32 indicates "
    "none\",\"minimum\":0},\"input_usage\":{\"type\":\"integer\",\"description\":\"0 .."
    " deactivated, 1 .. just detecting and timestamping\",\"minimum\":0},\"in_p"
    "olarity\":{\"type\":\"integer\",\"description\":\"0 .. positive edge, 1 .. "
    "negative edge\",\"minimum\":0},\"ext_bitmask\":{\"type\":\"integer\",\"descripti"
    "on\":\"defines which of the internal pulse generators are to be started\""
    ",\"minimum\":0},\"out_duration\":{\"type\":\"integer\",\"description\":\"output "
    "pulse duration [0.1 msec]\",\"minimum\":0},\"dmi_max_time\":{\"type\":\"intege"
    "r\",\"description\":\"dmi, maximum time interval between trigger events "
    "[0.1 msec]\",\"minimum\":0},\"in_max_duration\":{\"type\":\"integer\",\"descript"
    "ion\":\"stops measurement of pulse duration of input signal [0.1 msec]\","
    "\"minimum\":0},\"output_descr\":{\"type\":\"string\",\"description\":\"descriptiv"
    "e string\"},\"channel\":{\"type\":\"integer\",\"description\":\"ID of "
    "input/output channel\",\"minimum\":0},\"num_channel\":{\"type\":\"integer\",\"de"
    "scription\":\"number of input/output channels\",\"minimum\":0},\"input_descr"
    "\":{\"type\":\"string\",\"description\":\"descriptive "
    "string\"},\"ext_signal\":{\"type\":\"integer\",\"description\":\"0 .. use input "
    "signal, 1 .. use output signal of "
    "channel\",\"minimum\":0},\"dmi_incr\":{\"type\":\"integer\",\"description\":\"dmi,"
    " increment in "
    "ticks\"},\"in_max_delay\":{\"type\":\"integer\",\"description\":\"maximum delay "
    "to output pulse before fake event is generated [0.1 msec], zero "
    "indicates that no fake events are generated\",\"minimum\":0},\"out_polarit"
    "y\":{\"type\":\"integer\",\"description\":\"0 .. positive edge, 1 .. negative "
    "edge\",\"minimum\":0},\"out_delay\":{\"type\":\"integer\",\"description\":\"output"
    " pulse initial delay after start [0.1 "
    "msec]\",\"minimum\":0}},\"description\":\"Trigger-Timestamping-Unit/IP confi"
    "guration\",\"required\":[\"ttip_version\",\"num_channel\",\"ext_channel\",\"ext_"
    "signal\",\"ext_delay\",\"ext_subdivider\",\"ext_bitmask\",\"dmi_incr\",\"dmi_min"
    "_time\",\"dmi_max_time\",\"dmi_dist_per_tick\",\"channel\",\"output_descr\",\"in"
    "put_descr\",\"output_usage\",\"input_usage\",\"out_polarity\",\"out_duration\","
    "\"out_interval\",\"out_delay\",\"out_num_pulses\",\"in_polarity\",\"in_min_dura"
    "tion\",\"in_max_duration\",\"in_max_delay\"],\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"ext_subdivider\":1,\"ttip_version\":1,\"dmi_dist_per_tick\":0.0,\"output_u"
    "sage\":1,\"out_interval\":10000,\"in_min_duration\":0,\"dmi_min_time\":0,\"ext"
    "_delay\":0,\"out_num_pulses\":1,\"ext_channel\":32,\"input_usage\":1,\"in_pola"
    "rity\":0,\"ext_bitmask\":0,\"out_duration\":300,\"dmi_max_time\":0,\"in_max_du"
    "ration\":10000,\"output_descr\":\"Port 1 - "
    "Trigger\",\"channel\":0,\"num_channel\":9,\"input_descr\":\"Port 1 - Exposure\""
    ",\"ext_signal\":0,\"dmi_incr\":0,\"in_max_delay\":9990,\"out_polarity\":0,\"out"
    "_delay\":0}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"properties\":{\"vertex_file\":{\"type\":\"object\",\"propert"
    "ies\":{\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Vertex file "
    "extension, without the leading "
    "dot\"}},\"required\":[\"file_extension\"]}},\"description\":\"Details about "
    "vertex file\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"titl"
    "e\":\"Vertex Information\"}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_uuid\":\"51534d95-d71f-4f36-ae1a-3e63a21fd1c7\",\"fi"
    "le_extension\":\"vtx\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Voxel Information\",\"oneOf\":[{\"properties\":{\""
    "shape_thresholds\":{\"$ref\":\"#/definitions/shape_thresholds\"},\"voxel_typ"
    "e\":{\"$ref\":\"#/definitions/voxel_type\"},\"size\":{\"description\":\"Size of "
    "voxels in file coordinate system.\",\"oneOf\":[{\"$ref\":\"#/definitions/vox"
    "el_size\"},{\"$ref\":\"#/definitions/voxel_size_cubic\"}]},\"voxel_origin\":{"
    "\"$ref\":\"#/definitions/voxel_origin_enum\"}},\"required\":[\"size\",\"voxel_o"
    "rigin\",\"voxel_type\"],\"additionalProperties\":false},{\"properties\":{\"vox"
    "el_origin\":{\"oneOf\":[{\"$ref\":\"#/definitions/voxel_origin_enum\"},{\"$ref"
    "\":\"#/definitions/voxel_origin_point\",\"description\":\"The base point of "
    "the voxel grid. Used together with <tt>voxel_size</tt> and "
    "<tt>voxel_index</tt> to compute actual point coordinates.\"}]},\"size_ll"
    "cs\":{\"$ref\":\"#/definitions/voxel_size\",\"description\":\"Size of voxels "
    "in a locally levelled cartesian coordinate system (xyz). This is only "
    "used for voxels based on a map projection.\"},\"voxel_type\":{\"$ref\":\"#/d"
    "efinitions/voxel_type\"},\"reference_point\":{\"type\":\"array\",\"maxItems\":2"
    ",\"description\":\"Point in WGS84 geodetic decimal degree (EPSG:4326) "
    "that was used to compute the projection distortion parameters. The "
    "coefficient order is latitude, longitude. Only voxels with "
    "corresponding geo_tag, voxel_size and reference_point can be reliably "
    "processed together. This entry is available for voxel files in "
    "projected CRS only.\",\"items\":{\"type\":\"number\",\"minimum\":-180,\"maximum\""
    ":180},\"minItems\":2},\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_th"
    "resholds\"},\"size\":{\"$ref\":\"#/definitions/voxel_size\",\"description\":\"Si"
    "ze of voxels in file coordinate system.\"}},\"required\":[\"reference_poin"
    "t\",\"size_llcs\",\"size\",\"voxel_origin\",\"voxel_type\"],\"additionalProperti"
    "es\":false}],\"description\":\"Details about the voxels contained in the f"
    "ile\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definitions\""
    ":{\"voxel_origin_point\":{\"type\":\"array\",\"maxItems\":3,\"description\":\"Ori"
    "gin point for all voxel indices in voxel CRS.\",\"items\":{\"type\":\"number"
    "\"},\"minItems\":3},\"voxel_origin_enum\":{\"enum\":[\"center\",\"corner\"],\"desc"
    "ription\":\"Defines whether the voxel's center or a corner is placed on "
    "CRS origin <tt>(0/0/0)</tt>.\",\"default\":\"corner\"},\"voxel_type\":{\"enum\""
    ":[\"center\",\"centroid\",\"index\"],\"description\":\"Whether a point in a "
    "voxel represents its center or its centroid. If type is <tt>index</tt>"
    " there is no point but only an integer voxel index.\",\"default\":\"centro"
    "id\"},\"voxel_size_cubic\":{\"type\":\"number\",\"$ref\":\"#/definitions/edge_le"
    "ngth\"},\"shape_thresholds\":{\"type\":\"object\",\"properties\":{\"line\":{\"type"
    "\":\"number\",\"exclusiveMinimum\":true,\"description\":\"If the biggest "
    "eigenvalue is bigger than the median eigenvalue * line_threshold, the "
    "voxel is considered a line.\",\"minimum\":1},\"plane\":{\"type\":\"number\",\"ex"
    "clusiveMinimum\":true,\"exclusiveMaximum\":true,\"maximum\":1,\"description\""
    ":\"If the smallest eigenvalue is smaller than the median eigenvalue * "
    "plane_threshold, the voxel is considered a "
    "plane.\",\"minimum\":0}},\"description\":\"Thresholds used to compute the "
    "voxel's shape_id value.\",\"required\":[\"plane\",\"line\"]},\"edge_length\":{\""
    "type\":\"number\",\"exclusiveMinimum\":true,\"description\":\"Length of voxel "
    "edge [m].\",\"minimum\":0},\"voxel_size\":{\"type\":\"array\",\"maxItems\":3,\"des"
    "cription\":\"Size of "
    "voxels.\",\"items\":{\"$ref\":\"#/definitions/edge_length\"},\"minItems\":3}}}"
    ,
    /* example: */
    "{\"voxel_origin\":\"corner\",\"size_llcs\":[0.5156575252891171,0.51308353566"
    "83303,0.5143705304787237],\"voxel_type\":\"centroid\",\"reference_point\":[4"
    "8,16],\"shape_thresholds\":{\"line\":6,\"plane\":0.16},\"size\":[0.59716428347"
    "79395,0.5971642834779395,0.5143705304787237]}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Waveform Averaging Settings\",\"properties\":{\""
    "trim\":{\"type\":\"number\",\"description\":\"Percentage for robust averaging."
    "\",\"default\":0,\"minimum\":0,\"maximum\":0.5},\"num_shots\":{\"type\":\"integer\""
    ",\"description\":\"Number of consecutive shots to be used for averaging.\""
    ",\"minimum\":1},\"mta_zone\":{\"type\":\"integer\",\"description\":\"Fixed MTA "
    "zone for averaging.\",\"minimum\":1}},\"description\":\"Settings for "
    "waveform averaging\",\"required\":[\"num_shots\",\"mta_zone\"],\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"trim\":0.05,\"num_shots\":7,\"mta_zone\":1}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Waveform Information\",\"properties\":{\"sample_"
    "block_file\":{\"type\":\"object\",\"properties\":{\"file_uuid\":{\"type\":\"string"
    "\",\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Sample block "
    "file extension, without the leading dot\"}},\"required\":[\"file_extension"
    "\"]},\"sample_data_files\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"prop"
    "erties\":{\"channel\":{\"type\":\"integer\",\"exclusiveMinimum\":false,\"exclusi"
    "veMaximum\":false,\"maximum\":255,\"description\":\"Sample block channel "
    "number (255 = invalid)\",\"minimum\":0},\"file_extension\":{\"type\":\"string\""
    ",\"description\":\"Sample data file extension, without the leading dot\"},"
    "\"delta_st\":{\"type\":\"number\",\"description\":\"reserved\"},\"file_uuid\":{\"ty"
    "pe\":\"string\",\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"channel_name\":{\"type\":\"string\",\"description\":\"Sample block "
    "channel name\"},\"sample_interval\":{\"type\":\"number\",\"exclusiveMinimum\":f"
    "alse,\"description\":\"Sampling interval in seconds\",\"minimum\":0},\"sample"
    "_bits\":{\"type\":\"integer\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":f"
    "alse,\"maximum\":32,\"description\":\"Bitwidth of samples (e.g. 10 bit, 12 "
    "bit)\",\"minimum\":0},\"laser_wavelength\":{\"type\":\"number\",\"exclusiveMinim"
    "um\":false,\"description\":\"Laser wavelength in meters (0 = unknown)\",\"mi"
    "nimum\":0}},\"required\":[\"channel\",\"channel_name\",\"sample_interval\",\"sam"
    "ple_bits\",\"laser_wavelength\",\"delta_st\",\"file_extension\"]}},\"range_off"
    "set_waveform_samples_m\":{\"type\":\"number\",\"description\":\"Calibrated "
    "device specific range offset for waveform samples in "
    "meters.\"},\"range_offset_m\":{\"type\":\"number\",\"description\":\"Calibrated "
    "device specific range offset for waveform analysis by system response "
    "fitting in meters.\"}},\"description\":\"Details about waveform files\",\"re"
    "quired\":[\"sample_block_file\",\"sample_data_files\"],\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"sample_block_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe66"
    "0\",\"file_extension\":\"sbx\"},\"sample_data_files\":[{\"channel\":0,\"file_ext"
    "ension\":\"sp0\",\"channel_name\":\"high_power\",\"file_uuid\":\"da084413-e3e8-4"
    "655-a122-071de8490d8e\",\"delta_st\":0,\"sample_interval\":1.00503e-09,\"sam"
    "ple_bits\":12,\"laser_wavelength\":0},{\"channel\":1,\"file_extension\":\"sp1\""
    ",\"channel_name\":\"low_power\",\"file_uuid\":\"93585b5e-5ea9-43a1-947b-e7ba3"
    "be642d2\",\"delta_st\":0,\"sample_interval\":1.00503e-09,\"sample_bits\":12,\""
    "laser_wavelength\":0},{\"channel\":5,\"file_extension\":\"sp5\",\"channel_name"
    "\":\"wwf\",\"file_uuid\":\"9d2298c4-1036-464f-b5cb-1cf8e517f3a0\",\"delta_st\":"
    "0,\"sample_interval\":1.00503e-09,\"sample_bits\":12,\"laser_wavelength\":0}"
    "],\"range_offset_waveform_samples_m \":7.283,\"range_offset_m\":3.1415}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"array\",\"description\":\"Scanner settings for waveform output\",\""
    "items\":{\"type\":\"object\",\"properties\":{\"enabled\":{\"type\":\"boolean\",\"des"
    "cription\":\"Waveform output "
    "enabled\"},\"pass_dev_less\":{\"type\":\"integer\",\"description\":\"Threshold "
    "for deviation less than "
    "[1]\"},\"pass_ampl_less\":{\"type\":\"number\",\"description\":\"Threshold for "
    "amplitude less than "
    "[dB]\"},\"logic_expression\":{\"type\":\"string\",\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\"},\"pass_ampl_greater\":{\"type\":\"number\",\"description\":\"Threshold"
    " for amplitude greater than "
    "[dB]\"},\"smart_enabled\":{\"type\":\"boolean\",\"description\":\"Smart waveform"
    " output enabled\"},\"sbl_name\":{\"type\":\"string\",\"description\":\"Name of "
    "sample block, e.g.: wfm, "
    "wwf\"},\"pass_rng_greater\":{\"type\":\"number\",\"description\":\"Threshold for"
    " range greater than "
    "[m]\"},\"pass_rng_less\":{\"type\":\"number\",\"description\":\"Threshold for "
    "range less than "
    "[m]\"},\"pass_dev_greater\":{\"type\":\"integer\",\"description\":\"Threshold "
    "for deviation greater than "
    "[1]\"},\"channel_idx_mask\":{\"type\":\"integer\",\"description\":\"Bit mask for"
    " channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, ...\"}},\"required\":[\"sbl_name\",\"enabled\",\"channel_idx_mask\"]},\"$s"
    "chema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Waveform "
    "Settings\"}"
    ,
    /* example: */
    "[{\"enabled\":true,\"pass_ampl_less\":5.0,\"channel_idx_mask\":11,\"pass_ampl"
    "_greater\":1.0,\"smart_enabled\":true,\"sbl_name\":\"wfm\",\"pass_rng_greater\""
    ":9.27,\"pass_rng_less\":13.11},{\"enabled\":false,\"sbl_name\":\"wwf\",\"channe"
    "l_idx_mask\":32}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Window Analysis\",\"properties\":{\"settings\":{\""
    "type\":\"object\",\"properties\":{\"amplitude\":{\"type\":\"object\",\"properties\""
    ":{\"sigma_factor\":{\"type\":\"number\"},\"additive_value\":{\"type\":\"number\"}}"
    ",\"required\":[\"sigma_factor\",\"additive_value\"]},\"range\":{\"type\":\"object"
    "\",\"properties\":{\"sigma_factor\":{\"type\":\"number\"},\"additive_value\":{\"ty"
    "pe\":\"number\"}},\"required\":[\"sigma_factor\",\"additive_value\"]}},\"require"
    "d\":[\"range\",\"amplitude\"]},\"filter\":{\"type\":\"object\",\"properties\":{\"amp"
    "litude_max\":{\"type\":\"array\",\"description\":\"[dB]\",\"items\":{\"type\":\"numb"
    "er\"}},\"range_max\":{\"type\":\"array\",\"description\":\"[m]\",\"items\":{\"type\":"
    "\"number\"}},\"angle\":{\"type\":\"array\",\"description\":\"[deg]\",\"items\":{\"typ"
    "e\":\"number\"}},\"range_min\":{\"type\":\"array\",\"description\":\"[m]\",\"items\":"
    "{\"type\":\"number\"}}},\"required\":[\"angle\",\"range_min\",\"range_max\",\"ampli"
    "tude_max\"]},\"result\":{\"type\":\"object\",\"properties\":{\"amplitude_mean\":{"
    "\"type\":\"array\",\"description\":\"[dB]\",\"items\":{\"type\":\"number\"}},\"range_"
    "mean\":{\"type\":\"array\",\"description\":\"[m]\",\"items\":{\"type\":\"number\"}},\""
    "amplitude_sigma\":{\"type\":\"array\",\"description\":\"[dB]\",\"items\":{\"type\":"
    "\"number\"}},\"range_sigma\":{\"type\":\"array\",\"description\":\"[m]\",\"items\":{"
    "\"type\":\"number\"}},\"timestamp\":{\"type\":\"array\",\"description\":\"[s]\",\"ite"
    "ms\":{\"type\":\"number\"}},\"angle\":{\"type\":\"array\",\"description\":\"[deg]\",\""
    "items\":{\"type\":\"number\"}},\"amplitude_offset\":{\"type\":\"array\",\"descript"
    "ion\":\"[dB]\",\"items\":{\"type\":\"number\"}}},\"required\":[\"angle\",\"range_mea"
    "n\",\"range_sigma\",\"amplitude_mean\",\"amplitude_sigma\",\"amplitude_offset\""
    "]}},\"description\":\"Window analysis data estimated from scandata and "
    "resulting filter parameters\",\"required\":[\"result\",\"filter\",\"settings\"]"
    ",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "{\"settings\":{\"amplitude\":{\"sigma_factor\":4,\"additive_value\":1.0},\"rang"
    "e\":{\"sigma_factor\":8,\"additive_value\":0.1}},\"filter\":{\"amplitude_max\":"
    "[8.04,8.01,7.99,7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78],\"range_max\":[0."
    "424,0.425,0.426,0.427,0.428,0.428,0.429,0.43,0.431,0.431,0.432],\"angle"
    "\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],\"range_min\""
    ":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.218,-0.219,-0.221,-0.223,-0.225"
    ",-0.227]},\"result\":{\"amplitude_mean\":[5.347396,5.263155,5.224655,5.179"
    "926,5.097782,5.116479,5.051756,4.983473,5.007885,5.002441,4.982],\"rang"
    "e_mean\":[0.1105621,0.1079564,0.1087088,0.1067261,0.1054582,0.1090412,0"
    ".102871,0.1019044,0.1051523,0.1058445,0.1031261],\"amplitude_sigma\":[0."
    "4272844,0.4298479,0.4236816,0.4283583,0.4362353,0.4315141,0.4373984,0."
    "4472798,0.4346001,0.4345487,0.4540681],\"range_sigma\":[0.01869652,0.021"
    "51435,0.01747969,0.01918765,0.01945776,0.01934862,0.01955329,0.0222558"
    "9,0.02229977,0.01899122,0.02009433],\"timestamp\":[408.4441,411.4443],\"a"
    "ngle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],\"amplit"
    "ude_offset\":[1.9,1.9]}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Window Echo Correction\",\"properties\":{\"slice"
    "s\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"properties\":{\"amplitude\":"
    "{\"type\":\"number\",\"description\":\"Window echo amplitude of slice in "
    "dB\"},\"table\":{\"type\":\"array\",\"description\":\"Correction table "
    "(dimension defined by the 'amplitude' and 'range' "
    "objects)\",\"items\":{\"type\":\"array\",\"description\":\"Table row (= "
    "amplitude "
    "axis)\",\"items\":{\"type\":\"array\",\"maxItems\":3,\"description\":\"Table "
    "column (= range axis)\",\"items\":{\"type\":\"number\",\"description\":\"Table "
    "cell (item 0: amplitude in dB, 1: range in m, 2: flags)\"},\"minItems\":3"
    "},\"minItems\":1},\"minItems\":1}},\"description\":\"Window echo correction "
    "parameter slice\",\"required\":[\"amplitude\",\"table\"]}},\"range\":{\"type\":\"o"
    "bject\",\"properties\":{\"entries\":{\"type\":\"integer\",\"description\":\"Number"
    " of range entries\",\"minimum\":1},\"minimum\":{\"type\":\"number\",\"descriptio"
    "n\":\"Minimum range in m\",\"minimum\":-2.0,\"maximum\":2.0},\"maximum\":{\"type"
    "\":\"number\",\"description\":\"Maximum range in "
    "m\",\"minimum\":-2.0,\"maximum\":2.0}},\"description\":\"Range axis of "
    "correction table\",\"required\":[\"minimum\",\"maximum\",\"entries\"]},\"amplitu"
    "de\":{\"type\":\"object\",\"properties\":{\"entries\":{\"type\":\"integer\",\"descri"
    "ption\":\"Number of amplitude entries\",\"minimum\":1},\"minimum\":{\"type\":\"n"
    "umber\",\"description\":\"Minimum amplitude in "
    "dB\",\"minimum\":0.0},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "amplitude in dB\",\"minimum\":0.0}},\"description\":\"Amplitude axis of "
    "correction table\",\"required\":[\"minimum\",\"maximum\",\"entries\"]}},\"descri"
    "ption\":\"Correction parameters for window glass echoes\",\"required\":[\"am"
    "plitude\",\"range\",\"slices\"],\"$schema\":\"http://json-schema.org/draft-04/"
    "schema#\"}"
    ,
    /* example: */
    "{\"slices\":[{\"amplitude\":1.5,\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,"
    "1]]]},{\"amplitude\":2.0,\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]}"
    "],\"range\":{\"entries\":128,\"minimum\":-1.5060822940732335,\"maximum\":1.506"
    "0822940732335},\"amplitude\":{\"entries\":128,\"minimum\":2,\"maximum\":20}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LAG_ANGLE_CORRECTION,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);

#endif // RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1
