/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    openSettings.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Database open settings (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-09-22/AW: Parameter 'cacheSize' changed from 32 to 64 bit type
 * \version 2016-12-19/AW: New functions to load/save settings from/to JSON
 * \version 2019-02-15/AW: Fix C++ API wrapper of OpenSettings class
 * \version 2020-02-21/AW: Class 'OpenSettings' is now context-free (#3544)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//______________________________________________________________________________
/*!
 * \brief C API wrapper class for OpenSettings
 * \note For C++ wrapper internal use only.
 */
class OpenSettingsWrapper
{
public:
    OpenSettingsWrapper(): hnd(nullptr)
    {
        ErrorImpl::check(rdb_pointcloud_open_settings_new_cf(&hnd));
    }

    ~OpenSettingsWrapper()
    {
        if (hnd) // then delete the open settings and ignore errors
        {
            ErrorImpl::check(rdb_pointcloud_open_settings_delete_cf(&hnd), true);
        }
    }

    static OpenSettingsWrapper& get(const OpenSettings &settings)
    {
        return *static_cast<OpenSettingsWrapper*>(settings.data);
    }

    static RDBPointcloudOpenSettings* handle(const OpenSettings &settings)
    {
        return get(settings).hnd;
    }

    static void assign(OpenSettings &target, const OpenSettings &source)
    {
        load(target, save(source));
    }

    static void read(OpenSettings &settings)
    {
        const OpenSettingsWrapper &self = get(settings);
        ErrorImpl::check(rdb_pointcloud_open_settings_get_cache_size_64_cf(
            self.hnd, &settings.cacheSize
        ));
    }

    static void post(const OpenSettings &settings)
    {
        const OpenSettingsWrapper &self = get(settings);
        ErrorImpl::check(rdb_pointcloud_open_settings_set_cache_size_64_cf(
            self.hnd, settings.cacheSize
        ));
    }

    static void load(OpenSettings &settings, const std::string &json)
    {
        const OpenSettingsWrapper &self = get(settings);
        ErrorImpl::check(rdb_pointcloud_open_settings_json_load_cf(
            self.hnd, json.c_str()
        ));
        read(settings);
    }

    static std::string save(const OpenSettings &settings)
    {
        post(settings);
        RDBStringWrapper result;
        const OpenSettingsWrapper &self = get(settings);
        ErrorImpl::check(rdb_pointcloud_open_settings_json_save_cf(
            self.hnd, &result.str
        ));
        return result;
    }

private:
    friend class OpenSettings;
    RDBPointcloudOpenSettings *hnd;
};

//---< OpenSettings::PUBLIC >---------------------------------------------------

OpenSettings::~OpenSettings()
{
    if (data)
    {
        delete static_cast<OpenSettingsWrapper*>(data);
    }
}

OpenSettings::OpenSettings():
    data(new OpenSettingsWrapper)
{
    OpenSettingsWrapper::read(*this);
}

OpenSettings::OpenSettings(riegl::rdb::Context &context):
    data(new OpenSettingsWrapper)
{
    (void)context;
    OpenSettingsWrapper::read(*this);
}

OpenSettings::OpenSettings(const OpenSettings &settings):
    data(new OpenSettingsWrapper)
{
    *this = settings;
}

OpenSettings& OpenSettings::operator=(const OpenSettings &settings)
{
    OpenSettingsWrapper::assign(*this, settings);
    return *this;
}

void OpenSettings::load(const std::string &json)
{
    OpenSettingsWrapper::load(*this, json);
}

std::string OpenSettings::save() const
{
    return OpenSettingsWrapper::save(*this);
}

}}} // namespace riegl::rdb::pointcloud
