/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryRemove.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point remove query
 * \version 2015-10-14/AW: Initial version
 *
 * This class can be used to remove (delete) existing points.
 *
 * \see riegl::rdb::Pointcloud::remove()
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_QUERYREMOVE_H
#define RIEGL_RDB_POINTCLOUD_QUERYREMOVE_H

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"

//---< TYPE DEFINITIONS >-------------------------------------------------------

typedef struct RDBPointcloudQueryRemove RDBPointcloudQueryRemove; // forward declaration of implementation details

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN

//______________________________________________________________________________
/*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::remove()
 */
RDB_FUNCTION(rdb_pointcloud_query_remove_new,
    RDBContext                *context,    //!< [in] library context
    RDBPointcloud             *pointcloud, //!< [in] point cloud
    RDBPointcloudQueryRemove **query       //!< [out] query handle
);

//______________________________________________________________________________
/*!
 * \brief Destroy query instance
 */
RDB_FUNCTION(rdb_pointcloud_query_remove_delete,
    RDBContext                *context, //!< [in] library context
    RDBPointcloudQueryRemove **query    //!< [in] query handle
);

//______________________________________________________________________________
/*!
 * \brief Bind attribute buffer
 *
 * Use this function to define a source buffer for a point attribute.
 * Exactly one buffer can be defined for an attribute (i.e. only the
 * most recently defined buffer will be used).
 *
 * The buffer is expected to be n*s*d bytes large, where
 * __n__ is the number of points defined in next(),
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *
 * \note This function just stores the buffer pointer - it does
 *       __NOT__ copy the data contained in the buffer. So make
 *       sure that the buffer remains valid until you call next().
 *
 * \note This function expects a buffer for the point ID attribute.
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 */
RDB_FUNCTION(rdb_pointcloud_query_remove_bind,
    RDBContext               *context,                    //!< [in] library context
    RDBPointcloudQueryRemove *query,                      //!< [in] query handle
    RDBString                 attribute,                  //!< [in] attribute name
    uint32_t                  dataType,                   //!< [in] buffer data type \see dataTypes.h
    const void               *buffer,                     //!< [in] buffer location
    int32_t                   stride RDB_DEFAULT_VALUE(0) //!< [in] bytes between beginnings of successive elements (0: auto)
);

//______________________________________________________________________________
/*!
 * \brief Remove points
 *
 * Use this function to actually remove (delete) the points.
 * All points given by their point ID stored in the previously
 * bound attribute buffer are removed from the database.
 *
 * Afterwards you may re-fill the buffers or define a new buffer
 * with bind() and call next() again until all points have been
 * removed.
 *
 * \note If __no__ point ID buffer is given and 'count' is 0xDEADFEED,
 *       then __all__ points are removed and next() always returns 1.
 *
 * \returns the number of points removed
 */
RDB_FUNCTION(rdb_pointcloud_query_remove_next,
    RDBContext               *context,                       //!< [in] library context
    RDBPointcloudQueryRemove *query,                         //!< [in] query handle
    uint32_t                  count,                         //!< [in] size of source buffers in terms of points
    uint32_t                 *processed RDB_DEFAULT_VALUE(0) //!< [out] number of processed points (optional)
);

RDB_LIBRARY_API_END

#endif // RIEGL_RDB_POINTCLOUD_QUERYREMOVE_H
