/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-07-fill-points.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 7: Fill points
 * \version 2016-11-14/AW: Initial version
 * \version 2018-07-10/AW: Use new bindBuffer() instead of bind() function
 *
 *  This example shows how to open an existing database and to fill (set)
 *  a point attribute with a constant value for all points. Please note
 *  that this could also be accomplished by combining select and update
 *  queries as shown in "rdb-example-4-update-points.cpp", but the fill
 *  query might be faster (lower processing time) and easier to use.
 *  This example is based on the database of rdb-example-1-create-database.
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <vector>
#include <cstdint>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>
#include <riegl/rdb/default.hpp>

int main()
{
    try
    {
        // New RDB library context
        riegl::rdb::Context context;

        // Access existing database
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);

        // Before we can modify the database, we must start a transaction
        riegl::rdb::pointcloud::TransactionScope transaction(
            rdb,                    // point cloud object
            "Fill",                 // transaction title
            "Point Classifier v1.0" // software name
        );

        // Set point class and color to a fixed value for all points
        {
            using namespace riegl::rdb::pointcloud;

            const uint8_t point_class = 0;
            const uint8_t point_color[4] = {255, 255, 255, 255};

            QueryFill fill = rdb.fill();
            fill.bindBuffer(RDB_RIEGL_CLASS, point_class);
            fill.bindBuffer(RDB_RIEGL_RGBA,  point_color);

            while (const uint32_t count = fill.next(10000))
            {
                std::cout << fill.progress() << "% done" << std::endl;
            }
        }

        // Set different class and color for points that have a reflectance of
        // at least 10 dB. This is actually the same as above but with different
        // constants and a filter expression supplied to the fill() function.
        {
            using namespace riegl::rdb::pointcloud;

            const uint8_t point_class = 101;
            const uint8_t point_color[4] = {255, 102, 0, 255};

            QueryFill fill = rdb.fill("riegl.reflectance >= 10");
            fill.bindBuffer(RDB_RIEGL_CLASS, point_class);
            fill.bindBuffer(RDB_RIEGL_RGBA,  point_color);

            while (const uint32_t count = fill.next(10000))
            {
                std::cout << fill.progress() << "% done" << std::endl;
            }
        }

        // Finally commit transaction
        transaction.commit();

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
