/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-06-database-stat.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 6: Database statistics
 * \version 2015-10-14/AW: Initial version
 * \version 2016-01-20/AW: Typos fixed
 * \version 2017-01-31/AW: Check whether the file is a RDB 2 database file
 * \version 2018-06-22/AW: Attribute length type changed to uint32 (#3117)
 * \version 2018-07-10/AW: Use simplified versions of minimum() and maximum()
 *
 *  This example shows how to open an existing database and output some
 *  information about the database like the list of point attributes,
 *  number of points and point cloud extents.
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <array>
#include <vector>
#include <iomanip>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>
#include <riegl/rdb/default.hpp>

int main()
{
    try
    {
        // New RDB library context
        riegl::rdb::Context context;

        // Before opening the file, we can (optionally) check whether it is
        // a RDB 2 database file (otherwise open() will raise an exception):
        if (!context.databaseFileTypeCheck("pointcloud.rdbx"))
        {
            throw std::runtime_error("This is not a RDB 2 database file!");
        }

        // Access existing database
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);
        std::cout << std::endl << "POINT ATTRIBUTES:" << std::endl;
        {
            // Get list of point attributes
            std::vector<std::string> attributes = rdb.pointAttribute().list();
            for (auto it = attributes.begin(); it != attributes.end(); it++)
            {
                // Get details of current point attribute
                using riegl::rdb::pointcloud::PointAttribute;
                const PointAttribute attribute = rdb.pointAttribute().get(*it);

                // Output point attribute
                std::cout << "  Name:         " << attribute.name         << std::endl;
                std::cout << "  Title:        " << attribute.title        << std::endl;
                std::cout << "  Description:  " << attribute.description  << std::endl;
                std::cout << "  Unit:         " << attribute.unitSymbol   << std::endl;
                std::cout << "  Length:       " << attribute.length       << std::endl;
                std::cout << "  Resolution:   " << attribute.resolution   << std::endl;
                std::cout << "  MinimumValue: " << attribute.minimumValue << std::endl;
                std::cout << "  MaximumValue: " << attribute.maximumValue << std::endl;
                std::cout << "  DefaultValue: " << attribute.defaultValue << std::endl;
                std::cout << "------------------------------------------" << std::endl;
            }
        }

        std::cout << std::endl << "POINT CLOUD STATISTICS:" << std::endl;
        {
            // Get index graph root node
            riegl::rdb::pointcloud::QueryStat stat = rdb.stat();
            riegl::rdb::pointcloud::GraphNode root = stat.index();

            // Get point cloud extents
            using namespace riegl::rdb::pointcloud;
            std::array<double, 3> minimum, maximum;
            stat.minimum(root, RDB_RIEGL_XYZ, minimum);
            stat.maximum(root, RDB_RIEGL_XYZ, maximum);

            // Output point cloud statistics
            std::cout << std::fixed << std::setprecision(3);
            std::cout << "  Total point count: " << root.pointCountTotal << std::endl;
            std::cout << "  Minimum X, Y, Z:   " << minimum[0] << ", " << minimum[1] << ", " << minimum[2] << std::endl;
            std::cout << "  Maximum X, Y, Z:   " << maximum[0] << ", " << maximum[1] << ", " << maximum[2] << std::endl;
        }

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
