/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryInsert.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point insert query (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2020-02-27/AW: Add bindId() for "riegl.id" data buffers (#3576)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< STRUCT QueryInsert::Private >--------------------------------------------

struct QueryInsert::Private
{
    RDBContext               *context;
    RDBPointcloud            *pointcloud;
    RDBPointcloudQueryInsert *query;

    Private(riegl::rdb::PointcloudData *pcl):
        context(pcl->contextHandle()),
        pointcloud(pcl->pointcloud),
        query(nullptr)
    {
        ErrorImpl::check(context, rdb_pointcloud_query_insert_new(
            context, pointcloud, &query
        ));
    }

    ~Private()
    {
        ErrorImpl::check(context, rdb_pointcloud_query_insert_delete(
            context, &query
        ));
    }
};

//---< QueryInsert::PUBLIC >----------------------------------------------------

QueryInsert::QueryInsert()
{
}

QueryInsert::QueryInsert(riegl::rdb::PointcloudData *pointcloud):
    data(new Private(pointcloud))
{
}

QueryInsert::operator bool() const
{
    return valid();
}

bool QueryInsert::valid() const
{
    return (data != nullptr);
}

void QueryInsert::close()
{
    data.reset();
}

void QueryInsert::loadMultiPassSettings(const std::string &json)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_insert_load_multi_pass_settings(
        data->context, data->query, json.c_str()
    ));
}

void QueryInsert::bind(
    const std::string &attribute,
    const DataType     dataType,
    const void        *buffer,
    const std::int32_t stride
)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_insert_bind(
        data->context, data->query,
        attribute.c_str(),
        dataType, buffer, stride
    ));
}

void QueryInsert::bindId(
    const DataType     dataType,
    void              *buffer,
    const std::int32_t stride
)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_insert_bind_id(
        data->context, data->query,
        dataType, buffer, stride
    ));
}

std::uint32_t QueryInsert::next(std::uint32_t count)
{
    if (!data) return 0;
    std::uint32_t result(0);
    ErrorImpl::check(data->context, rdb_pointcloud_query_insert_next(
        data->context, data->query, count, &result
    ));
    return result;
}

}}} // namespace riegl::rdb::pointcloud
