/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 * \version 2025-11-11/AW: Item "riegl.lag_angle_correction" added (#6068)
 *
 *******************************************************************************
 */

#ifndef RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1
#define RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1

// Meta data item definition version
const char* const RDB_META_DATA_ITEMS_VERSION = "1.1.57";
const char* const RDB_META_DATA_ITEMS_DATE = "2025-11-11";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
typedef struct MetaDataItemInfo
{
    const char*   name;
    const char*   title;
    const char*   description;
    unsigned char status;
    const char*   schema;
    const char*   example;
} MetaDataItemInfo;

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"angl"
    "e\",\"range_mean\",\"amplitude_mean\"],\"type\":\"object\",\"properties\":{\"ampli"
    "tude_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"Mean amplitude [d"
    "B]\",\"type\":\"array\"},\"angle\":{\"items\":{\"type\":\"number\"},\"description\":\""
    "Angle [deg]\",\"type\":\"array\"},\"range_mean\":{\"items\":{\"type\":\"number\"},\""
    "description\":\"Mean range [m]\",\"type\":\"array\"}},\"title\":\"Angular Notch "
    "Filter\",\"description\":\"Angular notch filter parameters for window "
    "glass echoes\"}"
    ,
    /* example: */
    "{\"amplitude_mean\":[3.780913,3.780913,3.480913,3.120913,2.850913,2.7209"
    "13,2.680913,2.610913,2.530913,2.570913,2.570913],\"angle\":[14.0,15.0,16"
    ".0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],\"range_mean\":[0.094,0.094,"
    "0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"temp"
    "erature\",\"pressure\",\"rel_humidity\",\"pressure_sl\",\"amsl\",\"group_velocit"
    "y\",\"attenuation\",\"wavelength\"],\"type\":\"object\",\"properties\":{\"pressure"
    "_sl\":{\"description\":\"Atmospheric pressure at sea level "
    "[mbar]\",\"type\":\"number\"},\"amsl\":{\"description\":\"Height above mean sea "
    "level (AMSL) [m]\",\"type\":\"number\"},\"wavelength\":{\"description\":\"Laser "
    "wavelength [nm]\",\"type\":\"number\"},\"pressure\":{\"description\":\"Pressure "
    "along measurment path "
    "[mbar]\",\"type\":\"number\"},\"temperature\":{\"description\":\"Temperature "
    "along measurement path "
    "[\\u00b0C]\",\"type\":\"number\"},\"attenuation\":{\"description\":\"Atmospheric "
    "attenuation "
    "[1/km]\",\"type\":\"number\"},\"group_velocity\":{\"description\":\"Group "
    "velocity of laser beam "
    "[m/s]\",\"type\":\"number\"},\"rel_humidity\":{\"description\":\"Relative "
    "humidity along measurement path "
    "[%]\",\"type\":\"number\"}},\"title\":\"Atmospheric "
    "Parameters\",\"description\":\"Atmospheric parameters\"}"
    ,
    /* example: */
    "{\"pressure_sl\":970,\"amsl\":0,\"wavelength\":1550,\"pressure\":970,\"temperat"
    "ure\":7,\"attenuation\":0.028125,\"group_velocity\":299711000.0,\"rel_humidi"
    "ty\":63}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"beam"
    "_exit_diameter\",\"beam_divergence\"],\"type\":\"object\",\"properties\":{\"beam"
    "_exit_diameter\":{\"exclusiveMinimum\":false,\"minimum\":0,\"type\":\"number\","
    "\"description\":\"Beam width at exit aperture [m]\"},\"beam_divergence\":{\"e"
    "xclusiveMinimum\":false,\"minimum\":0,\"type\":\"number\",\"description\":\"Beam"
    " divergence in far field [rad]\"}},\"title\":\"Beam "
    "Geometry\",\"description\":\"Laser beam geometry details\"}"
    ,
    /* example: */
    "{\"beam_exit_diameter\":0.0072,\"beam_divergence\":0.0003}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"type"
    "s\"],\"type\":\"object\",\"definitions\":{\"rectangle\":{\"allOf\":[{\"$ref\":\"#/de"
    "finitions/common\"},{\"required\":[\"shape\",\"length\",\"width\"],\"type\":\"obje"
    "ct\",\"description\":\"rectangle specific properties\",\"properties\":{\"shape"
    "\":{\"enum\":[\"rectangle\"],\"type\":\"string\"},\"length\":{\"exclusiveMinimum\":"
    "true,\"minimum\":0,\"type\":\"number\",\"description\":\"length in meters\"},\"wi"
    "dth\":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description"
    "\":\"width in meters\"}}}],\"type\":\"object\",\"description\":\"rectangle\"},\"co"
    "mmon\":{\"required\":[\"name\",\"shape\"],\"type\":\"object\",\"description\":\"comm"
    "on object properties\",\"properties\":{\"name\":{\"minLength\":3,\"type\":\"stri"
    "ng\",\"description\":\"unique type identifier\"},\"shape\":{\"enum\":[\"rectangl"
    "e\",\"checkerboard2x2\",\"chevron\",\"circular_disk\",\"cylinder\",\"sphere\",\"ro"
    "und_corner_cube_prism\"],\"type\":\"string\",\"description\":\"shape identifie"
    "r\"},\"surface_type\":{\"enum\":[\"retro_reflective_foil\",\"diffuse\"],\"type\":"
    "\"string\",\"description\":\"surface material "
    "type\"},\"description\":{\"type\":\"string\",\"description\":\"string describing"
    " the object\"}}},\"circular_disk\":{\"allOf\":[{\"$ref\":\"#/definitions/commo"
    "n\"},{\"required\":[\"shape\",\"diameter\"],\"type\":\"object\",\"description\":\"ci"
    "rcular disk specific properties\",\"properties\":{\"diameter\":{\"exclusiveM"
    "inimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"diameter in me"
    "ters\"},\"shape\":{\"enum\":[\"circular_disk\"],\"type\":\"string\",\"description\""
    ":\"shape identifier\"},\"offset\":{\"type\":\"number\",\"description\":\"offset "
    "in meters, e.g. reflector constant "
    "(optional)\"}}}],\"type\":\"object\",\"description\":\"circular disk\"},\"cylind"
    "er\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"shape\",\"di"
    "ameter\",\"height\"],\"type\":\"object\",\"description\":\"cylinder specific pro"
    "perties\",\"properties\":{\"diameter\":{\"exclusiveMinimum\":true,\"minimum\":0"
    ",\"type\":\"number\",\"description\":\"diameter in meters\"},\"shape\":{\"enum\":["
    "\"cylinder\"],\"type\":\"string\",\"description\":\"shape identifier\"},\"height\""
    ":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"h"
    "eight in meters\"}}}],\"type\":\"object\",\"description\":\"cylinder\"},\"checke"
    "rboard2x2\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"sha"
    "pe\",\"square_length\"],\"type\":\"object\",\"description\":\"checkerboard "
    "specific properties\",\"properties\":{\"shape\":{\"enum\":[\"checkerboard2x2\"]"
    ",\"type\":\"string\",\"description\":\"shape identifier\"},\"square_length\":{\"e"
    "xclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"lengt"
    "h of a square of the checkerboard in "
    "meters\"}}}],\"type\":\"object\",\"description\":\"checkerboard 2 by 2\"},\"chev"
    "ron\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"shape\",\"o"
    "utside_edge_length\",\"thickness\"],\"type\":\"object\",\"description\":\"chevro"
    "n specific properties\",\"properties\":{\"outside_edge_length\":{\"exclusive"
    "Minimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"length of the"
    " two outer edges in meters\"},\"thickness\":{\"exclusiveMinimum\":true,\"min"
    "imum\":0,\"type\":\"number\",\"description\":\"thickness in meters\"},\"shape\":{"
    "\"enum\":[\"chevron\"],\"type\":\"string\",\"description\":\"shape identifier\"}}}"
    "],\"type\":\"object\",\"description\":\"chevron\"},\"sphere\":{\"allOf\":[{\"$ref\":"
    "\"#/definitions/common\"},{\"required\":[\"shape\",\"diameter\"],\"type\":\"objec"
    "t\",\"description\":\"sphere specific properties\",\"properties\":{\"diameter\""
    ":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"d"
    "iameter in meters\"},\"shape\":{\"enum\":[\"sphere\"],\"type\":\"string\",\"descri"
    "ption\":\"shape identifier\"}}}],\"type\":\"object\",\"description\":\"sphere\"},"
    "\"round_corner_cube_prism\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\""
    "required\":[\"shape\",\"diameter\"],\"type\":\"object\",\"description\":\"round "
    "corner cube prism specific properties\",\"properties\":{\"diameter\":{\"excl"
    "usiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"diameter"
    " in meters\"},\"shape\":{\"enum\":[\"round_corner_cube_prism\"],\"type\":\"strin"
    "g\",\"description\":\"shape "
    "identifier\"},\"offset\":{\"type\":\"number\",\"description\":\"offset in "
    "meters, e.g. reflector constant "
    "(optional)\"}}}],\"type\":\"object\",\"description\":\"round corner cube prism"
    "\"}},\"properties\":{\"types\":{\"items\":{\"oneOf\":[{\"$ref\":\"#/definitions/re"
    "ctangle\"},{\"$ref\":\"#/definitions/checkerboard2x2\"},{\"$ref\":\"#/definiti"
    "ons/chevron\"},{\"$ref\":\"#/definitions/circular_disk\"},{\"$ref\":\"#/defini"
    "tions/cylinder\"},{\"$ref\":\"#/definitions/sphere\"},{\"$ref\":\"#/definition"
    "s/round_corner_cube_prism\"}],\"type\":\"object\"},\"uniqueItems\":true,\"type"
    "\":\"array\"}},\"title\":\"Control Object Catalog\",\"description\":\"List of "
    "control object type definitions\"}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or "
    "'diffuse')\"],\"types\":[{\"name\":\"Rectangle "
    "60x30\",\"shape\":\"rectangle\",\"length\":0.6,\"description\":\"Rectangle (60cm"
    " x 30cm)\",\"width\":0.3},{\"name\":\"Rectangle "
    "80x40\",\"shape\":\"rectangle\",\"length\":0.8,\"description\":\"Rectangle (80cm"
    " x 40cm)\",\"width\":0.4},{\"name\":\"Checkerboard2x2 30\",\"shape\":\"checkerbo"
    "ard2x2\",\"square_length\":0.3,\"description\":\"Checkerboard (square "
    "length: 30cm)\"},{\"name\":\"Checkerboard2x2 50\",\"shape\":\"checkerboard2x2\""
    ",\"square_length\":0.5,\"description\":\"Checkerboard (square length: "
    "50cm)\"},{\"outside_edge_length\":0.6096,\"name\":\"Chevron "
    "24''/4''\",\"shape\":\"chevron\",\"thickness\":0.1016,\"description\":\"Chevron "
    "(a=24''; b=4'')\"},{\"name\":\"Circular Disk 50\",\"diameter\":0.5,\"shape\":\"c"
    "ircular_disk\",\"surface_type\":\"diffuse\",\"description\":\" Circular Disk "
    "(diameter: 50cm)\"},{\"name\":\"RIEGL flat reflector 50 mm\",\"diameter\":0.0"
    "5,\"shape\":\"circular_disk\",\"surface_type\":\"retro_reflective_foil\",\"offs"
    "et\":0.0,\"description\":\"flat circular reflector from retro-reflective "
    "foil\"},{\"name\":\"RIEGL flat reflector 100 mm\",\"diameter\":0.1,\"shape\":\"c"
    "ircular_disk\",\"surface_type\":\"retro_reflective_foil\",\"offset\":0.0,\"des"
    "cription\":\"flat circular reflector from retro-reflective "
    "foil\"},{\"name\":\"RIEGL flat reflector 150 mm\",\"diameter\":0.15,\"shape\":\""
    "circular_disk\",\"surface_type\":\"retro_reflective_foil\",\"offset\":0.0,\"de"
    "scription\":\"flat circular reflector from retro-reflective "
    "foil\"},{\"name\":\"RIEGL cylindrical reflector 50 mm\",\"diameter\":0.05,\"sh"
    "ape\":\"cylinder\",\"surface_type\":\"retro_reflective_foil\",\"height\":0.05,\""
    "description\":\"cylindrical reflector from retro-reflective "
    "foil\"},{\"name\":\"RIEGL cylindrical reflector 100 mm\",\"diameter\":0.1,\"sh"
    "ape\":\"cylinder\",\"surface_type\":\"retro_reflective_foil\",\"height\":0.1,\"d"
    "escription\":\"cylindrical reflector from retro-reflective "
    "foil\"},{\"name\":\"Sphere 200 "
    "mm\",\"diameter\":0.2,\"shape\":\"sphere\",\"description\":\"Sphere (diameter: "
    "200 mm)\"},{\"name\":\"Corner Cube Prism 50 mm\",\"diameter\":0.05,\"shape\":\"r"
    "ound_corner_cube_prism\",\"offset\":0.0,\"description\":\"round corner cube "
    "prism\"}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the control object reference "
    "file\",\"type\":\"object\",\"title\":\"Control Object Reference file\",\"$schema"
    "\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"reference_f"
    "ile\":{\"required\":[\"file_uuid\",\"file_path\"],\"description\":\"Reference to"
    " a control object file\",\"type\":\"object\",\"properties\":{\"file_path\":{\"ty"
    "pe\":\"string\",\"description\":\"Path of the control object file relative "
    "to referring "
    "file\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"Control object "
    "file's Universally Unique Identifier (RFC 4122)\"}}}}}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoi"
    "nts.cpx\",\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ampl"
    "itude\",\"detection_probability\"],\"type\":\"object\",\"properties\":{\"detecti"
    "on_probability\":{\"items\":{\"type\":\"number\"},\"description\":\"Detection "
    "probability [0..1]\",\"type\":\"array\"},\"amplitude\":{\"items\":{\"type\":\"numb"
    "er\"},\"description\":\"Amplitude "
    "[dB]\",\"type\":\"array\"}},\"title\":\"Detection "
    "Probability\",\"description\":\"Detection probability as a function of "
    "amplitude\"}"
    ,
    /* example: */
    "{\"detection_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]"
    ",\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"devi"
    "ce_type\",\"serial_number\"],\"type\":\"object\",\"properties\":{\"device_build\""
    ":{\"type\":\"string\",\"description\":\"Device build variant\"},\"channel_numbe"
    "r\":{\"exclusiveMinimum\":false,\"minimum\":0,\"type\":\"integer\",\"description"
    "\":\"Laser channel number (not defined or 0: single channel "
    "device)\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\"},\"channel_text\":{\"type\":\"string\",\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\"}},\"title\":\"Device Information\",\"description\":\"Details about "
    "the device used to acquire the point cloud\"}"
    ,
    /* example: */
    "{\"device_build\":\"\",\"channel_number\":0,\"serial_number\":\"S2221234\",\"chan"
    "nel_text\":\"\",\"device_type\":\"VZ-400i\",\"device_name\":\"Scanner 1\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"prim"
    "ary\"],\"type\":\"object\",\"properties\":{\"primary\":{\"required\":[\"ID\",\"conte"
    "nt\"],\"type\":\"object\",\"description\":\"Primary device geometry structure "
    "(mandatory)\",\"properties\":{\"content\":{\"items\":{\"type\":\"number\"},\"type\""
    ":\"array\",\"description\":\"Internal calibration values\"},\"ID\":{\"minItems\""
    ":2,\"items\":{\"type\":\"integer\"},\"type\":\"array\",\"description\":\"Structure "
    "identifier\",\"maxItems\":2}}},\"amu\":{\"type\":\"object\",\"description\":\"Angl"
    "e Measurement Unit\",\"properties\":{\"lineCC\":{\"exclusiveMinimum\":false,\""
    "minimum\":0,\"type\":\"number\",\"description\":\"Line Circle Count (number of"
    " LSBs per full rotation about line axis)\"},\"frameCC\":{\"exclusiveMinimu"
    "m\":false,\"minimum\":0,\"type\":\"number\",\"description\":\"Frame Circle Count"
    " (number of LSBs per full rotation about frame axis)\"}}},\"secondary\":{"
    "\"required\":[\"ID\",\"content\"],\"type\":\"object\",\"description\":\"Additional "
    "device geometry structure (optional)\",\"properties\":{\"content\":{\"items\""
    ":{\"type\":\"number\"},\"type\":\"array\",\"description\":\"Internal calibration "
    "values\"},\"ID\":{\"minItems\":2,\"items\":{\"type\":\"integer\"},\"type\":\"array\","
    "\"description\":\"Structure identifier\",\"maxItems\":2}}}},\"title\":\"Device "
    "Geometry\",\"description\":\"Scanner device geometry details\"}"
    ,
    /* example: */
    "{\"primary\":{\"content\":[0],\"ID\":[4,0]},\"amu\":{\"lineCC\":124000,\"frameCC\""
    ":124000},\"secondary\":{\"content\":[0],\"ID\":[91,0]}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"prim"
    "ary\"],\"type\":\"object\",\"properties\":{\"primary\":{\"required\":[\"ID\",\"conte"
    "nt\"],\"type\":\"object\",\"description\":\"Primary device geometry structure "
    "(mandatory)\",\"properties\":{\"content\":{\"items\":{\"type\":\"number\"},\"type\""
    ":\"array\",\"description\":\"Internal calibration values\"},\"ID\":{\"minItems\""
    ":2,\"items\":{\"type\":\"integer\"},\"type\":\"array\",\"description\":\"Structure "
    "identifier\",\"maxItems\":2}}}},\"title\":\"Device Geometry Passive "
    "Channel\",\"description\":\"Scanner passive channel device geometry "
    "details\"}"
    ,
    /* example: */
    "{\"primary\":{\"content\":[0],\"ID\":[143,0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Limits of the measured values output by the device. "
    "The limits depend on the device type, measurement program and/or scan "
    "pattern.\",\"type\":\"object\",\"title\":\"Device Output Limits\",\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"properties\":{\"range_minimum\":{"
    "\"description\":\"Minimum possible range in "
    "meters.\",\"type\":\"number\"},\"deviation_minimum\":{\"description\":\"Minimum "
    "possible pulse shape deviation.\",\"type\":\"number\"},\"deviation_maximum\":"
    "{\"description\":\"Maximum possible pulse shape deviation.\",\"type\":\"numbe"
    "r\"},\"background_radiation_maximum\":{\"description\":\"Maximum possible "
    "background radiation.\",\"type\":\"number\"},\"background_radiation_minimum\""
    ":{\"description\":\"Minimum possible background radiation.\",\"type\":\"numbe"
    "r\"},\"reflectance_maximum\":{\"description\":\"Maximum possible reflectance"
    " in dB.\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum "
    "possible range in meters.\",\"type\":\"number\"},\"reflectance_minimum\":{\"de"
    "scription\":\"Minimum possible reflectance in "
    "dB.\",\"type\":\"number\"},\"mta_zone_count_maximum\":{\"description\":\"Maximum"
    " number of MTA "
    "zones.\",\"type\":\"number\"},\"echo_count_maximum\":{\"description\":\"Maximum "
    "number of echoes a laser shot can "
    "have.\",\"type\":\"number\"},\"amplitude_minimum\":{\"description\":\"Minimum "
    "possible amplitude in "
    "dB.\",\"type\":\"number\"},\"amplitude_maximum\":{\"description\":\"Maximum "
    "possible amplitude in dB.\",\"type\":\"number\"}}}"
    ,
    /* example: */
    "{\"range_minimum\":2.9,\"range_maximum\":10000.0,\"reflectance_minimum\":-10"
    "0.0,\"mta_zone_count_maximum\":7,\"deviation_minimum\":-1,\"deviation_maxim"
    "um\":32767,\"background_radiation_maximum\":0,\"background_radiation_minim"
    "um\":0,\"reflectance_maximum\":100.0,\"amplitude_minimum\":0.0,\"amplitude_m"
    "aximum\":100.0}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the devices used to acquire the point "
    "cloud (e.g. when merging point clouds of different devices)\",\"type\":\"a"
    "rray\",\"items\":{\"required\":[\"device_type\",\"serial_number\"],\"type\":\"obje"
    "ct\",\"properties\":{\"device_build\":{\"type\":\"string\",\"description\":\"Devic"
    "e build variant\"},\"channel_number\":{\"exclusiveMinimum\":false,\"minimum\""
    ":0,\"type\":\"integer\",\"description\":\"Laser channel number (not defined "
    "or 0: single channel "
    "device)\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\"},\"signed\":{\"type\":\"boolean\",\"description\":\"Flag that is set"
    " when the original 'riegl.device' entry in the source file was "
    "correctly "
    "signed.\"},\"channel_text\":{\"type\":\"string\",\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\"}}},\"title\":\"Devices "
    "Information\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "[{\"device_build\":\"\",\"channel_number\":0,\"serial_number\":\"S2221234\",\"sig"
    "ned\":false,\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"device_name\":\"Sc"
    "anner 1\"},{\"device_build\":\"\",\"channel_number\":1,\"serial_number\":\"S2222"
    "680\",\"signed\":true,\"channel_text\":\"\",\"device_type\":\"VQ-1560i-DW\",\"devi"
    "ce_name\":\"Scanner 2\"},{\"device_build\":\"\",\"channel_number\":2,\"serial_nu"
    "mber\":\"S2222680\",\"signed\":true,\"channel_text\":\"\",\"device_type\":\"VQ-156"
    "0i-DW\",\"device_name\":\"Scanner 3\"}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"dmi_"
    "count_dir\",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick"
    "\"],\"type\":\"object\",\"properties\":{\"dmi_count_dir\":{\"maximum\":1,\"exclusi"
    "veMinimum\":false,\"minimum\":0,\"description\":\"DMI count direction is inv"
    "erted\",\"exclusiveMaximum\":false,\"type\":\"integer\"},\"dmi_incr\":{\"maximum"
    "\":2147483647,\"exclusiveMinimum\":false,\"minimum\":0,\"description\":\"Incre"
    "ment in ticks, 0 means distance trigger is deactivated\",\"exclusiveMaxi"
    "mum\":false,\"type\":\"integer\"},\"dmi_dist_per_tick\":{\"type\":\"number\",\"des"
    "cription\":\"Distance per tick, just informative [m]\"},\"dmi_max_time\":{\""
    "maximum\":65535,\"exclusiveMinimum\":false,\"minimum\":0,\"description\":\"Max"
    "imum time interval between trigger events, 0 means timeout is inactive"
    " [0.1 msec]\",\"exclusiveMaximum\":false,\"type\":\"integer\"},\"dmi_min_time\""
    ":{\"maximum\":65535,\"exclusiveMinimum\":false,\"minimum\":0,\"description\":\""
    "Minimum time interval between trigger events, 0 means rate limitation "
    "is inactive [0.1 "
    "msec]\",\"exclusiveMaximum\":false,\"type\":\"integer\"}},\"title\":\"DMI "
    "Configuration\",\"description\":\"Distance measuring instrument "
    "configuration\"}"
    ,
    /* example: */
    "{\"dmi_count_dir\":0,\"dmi_incr\":0,\"dmi_dist_per_tick\":0.0002439999952912"
    "3306,\"dmi_max_time\":100,\"dmi_min_time\":0}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Dyntrig\",\"title\":\"Dyntrig\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"properties\":{\"pre\":{\"required\":[\"nc\",\"a\"],\"d"
    "escription\":\"Dyntrig pre values\",\"type\":\"object\",\"properties\":{\"a\":{\"m"
    "inItems\":1,\"items\":{\"type\":\"number\"},\"description\":\"...\",\"type\":\"array"
    "\",\"maxItems\":2080},\"nc\":{\"maximum\":255,\"minimum\":0,\"description\":\"Numb"
    "er of entries\",\"type\":\"integer\"}}},\"scale\":{\"description\":\"Scale in "
    "units of dB, for calculation of line: line number N=(amplitude-offset)"
    "/scale\",\"type\":\"number\"},\"post\":{\"required\":[\"nc\",\"a\"],\"description\":\""
    "Dyntrig post values\",\"type\":\"object\",\"properties\":{\"a\":{\"minItems\":1,\""
    "items\":{\"type\":\"number\"},\"description\":\"...\",\"type\":\"array\",\"maxItems\""
    ":2080},\"nc\":{\"maximum\":255,\"minimum\":0,\"description\":\"Number of "
    "entries\",\"type\":\"integer\"}}},\"offset\":{\"description\":\"Offset in units "
    "of dB, for calculation of line\",\"type\":\"number\"},\"tab\":{\"minItems\":1,\""
    "items\":{\"required\":[\"thra1\",\"del1\",\"dec1\",\"thra2\",\"del2\",\"dec2\"],\"type"
    "\":\"object\",\"description\":\"...\",\"properties\":{\"dec1\":{\"type\":\"number\",\""
    "description\":\"Decay of post-trigger threshold after del1 in units of "
    "dB/m\"},\"thra1\":{\"type\":\"number\",\"description\":\"Post-trigger threshold "
    "in units of dB\"},\"del2\":{\"type\":\"number\",\"description\":\"Length of "
    "constant pre-trigger threshold elevation in units of "
    "m\"},\"del1\":{\"type\":\"number\",\"description\":\"Length of constant "
    "post-trigger threshold elevation in units of "
    "m\"},\"thra2\":{\"type\":\"number\",\"description\":\"Pre-trigger threshold in "
    "units of dB\"},\"dec2\":{\"type\":\"number\",\"description\":\"Rise of "
    "pre-trigger threshold after del2 in units of "
    "dB/m\"}}},\"description\":\"...\",\"type\":\"array\",\"maxItems\":128}}}"
    ,
    /* example: */
    "{\"pre\":{\"a\":[78,86,126,134,31],\"nc\":128},\"scale\":1.0,\"post\":{\"a\":[78,8"
    "6,126,134,31],\"nc\":128},\"offset\":0.0,\"tab\":[{\"dec1\":0.0,\"thra1\":0.0,\"d"
    "el2\":0.0,\"del1\":0.0,\"thra2\":0.0,\"dec2\":0.0}]}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"echo"
    "_file\"],\"type\":\"object\",\"properties\":{\"echo_file\":{\"required\":[\"file_e"
    "xtension\"],\"type\":\"object\",\"properties\":{\"file_extension\":{\"type\":\"str"
    "ing\",\"description\":\"Echo file extension, without the leading "
    "dot\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\"}}}},\"title\":\"Echo "
    "Information\",\"description\":\"Details about echo files\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_extension\":\"owp\",\"file_uuid\":\"26a03615-67c0-4bea-8"
    "fe8-c577378fe661\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"additionalProper"
    "ties\":false,\"type\":\"object\",\"definitions\":{\"channel\":{\"required\":[\"del"
    "ay\",\"scale\",\"parameter\"],\"type\":\"object\",\"properties\":{\"a_lin\":{\"maxim"
    "um\":1,\"exclusiveMinimum\":false,\"minimum\":0,\"description\":\"relative "
    "linear amplitude range [0..1]\",\"exclusiveMaximum\":false,\"type\":\"number"
    "\"},\"scale\":{\"type\":\"number\",\"description\":\"amplitude "
    "calibration\"},\"delay\":{\"type\":\"number\",\"description\":\"delay "
    "calibration in seconds\"},\"parameter\":{\"required\":[\"A\",\"B\",\"gamma\",\"ome"
    "ga\"],\"type\":\"object\",\"description\":\"parameters of the syswave "
    "exponential sum\",\"properties\":{\"B\":{\"items\":{\"type\":\"number\"},\"type\":\""
    "array\",\"description\":\"imaginary part of amplitude factor in units of f"
    "ull-scale\"},\"gamma\":{\"items\":{\"type\":\"number\"},\"type\":\"array\",\"descrip"
    "tion\":\"decay in 1/second\"},\"omega\":{\"items\":{\"type\":\"number\"},\"type\":\""
    "array\",\"description\":\"angular frequency in "
    "Hz\"},\"A\":{\"items\":{\"type\":\"number\"},\"type\":\"array\",\"description\":\"real"
    " part of amplitude factor in units of "
    "full-scale\"}}}}}},\"title\":\"Exponential "
    "Decomposition\",\"description\":\"Details for exponential decomposition of"
    " full waveform data\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/defini"
    "tions/channel\",\"description\":\"one field per channel, field name is "
    "channel index\"}}}"
    ,
    /* example: */
    "{\"0\":{\"a_lin\":0.27,\"scale\":1.0,\"delay\":3.783458418887631e-09,\"paramete"
    "r\":{\"B\":[-3.9813032150268555,0.08622030913829803,-0.3152860999107361],"
    "\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\"omega\":[352020896.0"
    ",3647927552.0,-1977987072.0],\"A\":[0.9772450923919678,0.335433512926101"
    "7,-1.312678575515747]}},\"1\":{\"a_lin\":0.9,\"scale\":1.0,\"delay\":3.5e-09,\""
    "parameter\":{\"B\":[-3.9,0.0,-0.3],\"gamma\":[-1094726528.0,-769562752.0,-8"
    "48000064.0],\"omega\":[352020896.0,3647927552.0,-1977987072.0],\"A\":[0.9,"
    "0.3,-1.3]}}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ampl"
    "itude_lsb_low_power\",\"amplitude_lsb_high_power\",\"amplitude_db\",\"range_"
    "offset_sec_low_power\",\"range_offset_sec_high_power\"],\"type\":\"object\",\""
    "properties\":{\"amplitude_lsb_high_power\":{\"items\":{\"type\":\"number\"},\"ty"
    "pe\":\"array\"},\"range_offset_sec_low_power\":{\"items\":{\"type\":\"number\"},\""
    "type\":\"array\"},\"amplitude_db\":{\"items\":{\"type\":\"number\"},\"type\":\"array"
    "\"},\"range_offset_sec_high_power\":{\"items\":{\"type\":\"number\"},\"type\":\"ar"
    "ray\"},\"amplitude_lsb_low_power\":{\"items\":{\"type\":\"number\"},\"type\":\"arr"
    "ay\"}},\"title\":\"Gaussian "
    "Decomposition\",\"description\":\"riegl.gaussian_decomposition contains "
    "information relevant for extracting calibrated amplitudes and "
    "calibrated ranges from a Gaussian decomposition of full waveform data."
    " This information is contained in a table with five columns. Two "
    "columns are to be used as input: amplitude_lsb_low_power and "
    "amplitude_lsb_high_power. The other three columns provide the outputs."
    " Amplitude_db gives the calibrated amplitude in the optical regime in "
    "decibels. The range offset columns provide additive range offsets, "
    "given in units of seconds, for each channel.\"}"
    ,
    /* example: */
    "{\"amplitude_lsb_high_power\":[],\"range_offset_sec_low_power\":[],\"amplit"
    "ude_db\":[],\"range_offset_sec_high_power\":[],\"amplitude_lsb_low_power\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Point cloud georeferencing "
    "information\",\"type\":\"object\",\"title\":\"Geo Tag\",\"$schema\":\"http://json-"
    "schema.org/draft-04/schema#\",\"properties\":{\"pose\":{\"minItems\":4,\"items"
    "\":{\"minItems\":4,\"items\":{\"type\":\"number\",\"description\":\"columns\"},\"typ"
    "e\":\"array\",\"description\":\"rows\",\"maxItems\":4},\"description\":\"Coordinat"
    "e Transformation Matrix to transform from File Coordinate System to "
    "Global Coordinate Reference System. 4x4 matrix stored as two "
    "dimensional array, row major "
    "order.\",\"type\":\"array\",\"maxItems\":4},\"crs\":{\"description\":\"Global "
    "Coordinate Reference System. Please note that only 3D Cartesian "
    "Coordinate Systems are allowed.\",\"type\":\"object\",\"properties\":{\"name\":"
    "{\"type\":\"string\",\"description\":\"Coordinate reference system "
    "name\"},\"epsg\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"EPSG "
    "code\"},\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see "
    "http://www.opengeospatial.org/standards/wkt-crs)\"}}}}}"
    ,
    /* example: */
    "{\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,4063"
    "882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847,113"
    "8787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196],[0"
    ".0,0.0,0.0,1.0]],\"crs\":{\"name\":\"WGS84 "
    "Geocentric\",\"epsg\":4978,\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\",DATUM[\\\"WGS"
    "84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\"
    "\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000"
    "000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.0000000000000"
    "0000000,AUTHORITY[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AXIS[\\\"Y\\\",EAS"
    "T],AXIS[\\\"Z\\\",NORTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\"}}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"exclusiveMinimum\":true,\"type\":\"number\",\"description\":\"Geometric "
    "scale factor applied to point coordinates\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"minimum\":0}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Parameters used for georeferencing of the point "
    "cloud\",\"type\":\"object\",\"title\":\"Georeferencing Parameters\",\"$schema\":\""
    "http://json-schema.org/draft-04/schema#\",\"properties\":{\"trajectory_off"
    "sets\":{\"description\":\"Correction offsets applied to the trajectory dat"
    "a\",\"type\":\"object\",\"properties\":{\"offset_pitch\":{\"description\":\"[deg]\""
    ",\"type\":\"number\"},\"offset_time\":{\"description\":\"[s]\",\"type\":\"number\"},"
    "\"offset_north\":{\"description\":\"[m]\",\"type\":\"number\"},\"offset_east\":{\"d"
    "escription\":\"[m]\",\"type\":\"number\"},\"offset_yaw\":{\"description\":\"[deg]\""
    ",\"type\":\"number\"},\"offset_roll\":{\"description\":\"[deg]\",\"type\":\"number\""
    "},\"version\":{\"description\":\"Meaning of offset values and how to apply "
    "them; version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\",\"type\":\"integer\"},\"offset_height\":{\"description\":\"["
    "m]\",\"type\":\"number\"}}},\"trajectory_file\":{\"required\":[\"file_extension\""
    "],\"description\":\"Trajectory data used for georeferencing of the point "
    "cloud\",\"type\":\"object\",\"properties\":{\"file_extension\":{\"type\":\"string\""
    ",\"description\":\"Trajectory file extension, without the leading "
    "dot\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\"}}},\"body_coordinate_system_type\":{\"enum\""
    ":[\"NED\",\"ENU\"],\"description\":\"BODY coordinate frame (NED: "
    "North-East-Down, ENU: East-North-Up), default: NED\",\"type\":\"string\"},\""
    "socs_to_body_matrix\":{\"minItems\":4,\"items\":{\"minItems\":4,\"items\":{\"typ"
    "e\":\"number\",\"description\":\"columns\"},\"type\":\"array\",\"description\":\"row"
    "s\",\"maxItems\":4},\"description\":\"Coordinate Transformation Matrix to "
    "transform from Scanner's Own Coordinate System to Body Coordinate "
    "System. 4x4 matrix stored as two dimensional array, row major order.\","
    "\"type\":\"array\",\"maxItems\":4},\"socs_to_rocs_matrix\":{\"minItems\":4,\"item"
    "s\":{\"minItems\":4,\"items\":{\"type\":\"number\",\"description\":\"columns\"},\"ty"
    "pe\":\"array\",\"description\":\"rows\",\"maxItems\":4},\"description\":\"Coordina"
    "te Transformation Matrix to transform from Scanner's Own Coordinate "
    "System to Record Coordinate System. 4x4 matrix stored as two "
    "dimensional array, row major order.\",\"type\":\"array\",\"maxItems\":4}}}"
    ,
    /* example: */
    "{\"trajectory_offsets\":{\"offset_pitch\":0.01,\"offset_time\":18.007,\"offse"
    "t_north\":0.07,\"offset_east\":0.15,\"offset_yaw\":-0.45,\"offset_roll\":0.03"
    ",\"version\":0,\"offset_height\":-0.2},\"trajectory_file\":{\"file_extension\""
    ":\"pofx\",\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\"},\"socs_to_b"
    "ody_matrix\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,"
    "0.0],[0.962908599449764,-0.20260517250352,0.178208229833847,0.0],[0.0,"
    "0.660451759194338,0.7508684796801,0.0],[0.0,0.0,0.0,1.0]],\"body_coordi"
    "nate_system_type\":\"NED\"}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about GNSS "
    "processing\",\"type\":\"object\",\"title\":\"GNSS Processing Information\",\"$sc"
    "hema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"base_st"
    "ations\":{\"items\":{\"required\":[\"latitude\",\"longitude\",\"altitude\"],\"type"
    "\":\"object\",\"description\":\"GNSS base station details\",\"properties\":{\"la"
    "titude\":{\"type\":\"number\",\"description\":\"Latitude "
    "[deg]\"},\"longitude\":{\"type\":\"number\",\"description\":\"Longitude "
    "[deg]\"},\"altitude\":{\"type\":\"number\",\"description\":\"Altitude "
    "[m]\"}}},\"type\":\"array\",\"description\":\"List of GNSS base stations\"}}}"
    ,
    /* example: */
    "{\"base_stations\":[{\"latitude\":48.658391162,\"longitude\":15.663389822,\"a"
    "ltitude\":379.8004150390625}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"additionalProper"
    "ties\":false,\"type\":\"object\",\"title\":\"Item Names\",\"description\":\"Map of"
    " item names\",\"patternProperties\":{\"^-?[0-9]+$\":{\"type\":\"string\",\"descr"
    "iption\":\"One field per item, field name is item id, field value is "
    "item name\"}}}"
    ,
    /* example: */
    "{\"1\":\"Name of item with id 1\",\"-1\":\"Name of item with id -1\",\"2\":\"Name"
    " of item with id 2\",\"47\":\"Name of item with id 47\"}"
};

// Lag angle depended range and reflectance correction
const MetaDataItemInfo RDB_RIEGL_LAG_ANGLE_CORRECTION = {
    /* name        */ "riegl.lag_angle_correction",
    /* title       */ "Lag Angle Correction",
    /* description */ "Lag angle depended range and reflectance correction",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"rang"
    "es_m\",\"range_correction_m\",\"reflectance_correction_db\"],\"type\":\"object"
    "\",\"properties\":{\"ranges_m\":{\"items\":{\"type\":\"number\"},\"description\":\"R"
    "ange [m]\",\"type\":\"array\"},\"reflectance_correction_db\":{\"items\":{\"type\""
    ":\"number\"},\"description\":\"Reflectance correction [db] as a function of"
    " the range and for applied scanrate\",\"type\":\"array\"},\"range_correction"
    "_m\":{\"items\":{\"type\":\"number\"},\"description\":\"Range correction [m] as "
    "a function of the range and for applied scanrate, which defines the "
    "lag angle\",\"type\":\"array\"}},\"title\":\"Lag Angle "
    "Correction\",\"description\":\"Lag angle depended range and reflectance "
    "correction\"}"
    ,
    /* example: */
    "{\"ranges_m\":[0.0,4.795,31.964,63.928,95.892,127.856,10000.0],\"reflecta"
    "nce_correction_db\":[0.0,0.0,0.16,0.46,1.18,2.97,2.97],\"range_correctio"
    "n_m\":[0.0,0.0,0.0004,0.0023,0.0039,0.0053,0.0053]}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"additionalProper"
    "ties\":false,\"type\":\"object\",\"title\":\"Software License "
    "Keys\",\"description\":\"License keys for software features\",\"patternPrope"
    "rties\":{\"^.*$\":{\"minItems\":1,\"items\":{\"type\":\"string\",\"description\":\"L"
    "icense key (example: '46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73')\"},\"type\":\"array\",\"description\":\"Each field of the object "
    "represents a feature and holds a list of license keys, where the field"
    " name is the feature name.\"}}}"
    ,
    /* example: */
    "{\"Georeferencing\":[\"46AE032A - 39882AC4 - 9EC0A184 - 6F163D73\"],\"Full "
    "Waveform Analysis Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA - "
    "8D2ED557\"],\"MTA resolution\":[\"468E020A - 39A922E4 - B681A184 - "
    "673E3D72\"],\"Full Waveform Analysis Topography with GPU "
    "support\":[\"8AB44126 - 23B92250 - 16E2689F - 34EF7E7B\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Parameters for MTA "
    "processing\",\"type\":\"object\",\"title\":\"MTA Settings\",\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\",\"properties\":{\"zone_width\":{\"minimum\""
    ":0,\"description\":\"Width of a MTA zone in meter.\",\"type\":\"number\"},\"mod"
    "ulation_depth\":{\"minimum\":0,\"description\":\"Depth of pulse position "
    "modulation in meter.\",\"type\":\"number\"},\"zone_count\":{\"maximum\":255,\"mi"
    "nimum\":0,\"description\":\"Maximum number of MTA "
    "zones.\",\"type\":\"integer\"}}}"
    ,
    /* example: */
    "{\"zone_width\":149.896225,\"modulation_depth\":9.368514,\"zone_count\":23}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"delt"
    "a\",\"content\"],\"type\":\"object\",\"properties\":{\"content\":{\"minItems\":1,\"i"
    "tems\":{\"type\":\"number\"},\"type\":\"array\",\"description\":\"Correction value"
    " [m] to be added to the raw "
    "range\",\"maxItems\":2000},\"delta\":{\"type\":\"number\",\"description\":\"Delta "
    "between table entries [m], first entry is at range = 0 "
    "m\"}},\"title\":\"Near Range Correction Table\",\"description\":\"Lookup table"
    " for range correction based on raw range\"}"
    ,
    /* example: */
    "{\"content\":[0.0],\"delta\":0.512}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ampl"
    "itude\",\"range_sigma\",\"amplitude_sigma\"],\"type\":\"object\",\"properties\":{"
    "\"range_sigma\":{\"items\":{\"type\":\"number\"},\"description\":\"Sigma range [m"
    "]\",\"type\":\"array\"},\"amplitude_sigma\":{\"items\":{\"type\":\"number\"},\"descr"
    "iption\":\"Sigma amplitude [dB]\",\"type\":\"array\"},\"amplitude\":{\"items\":{\""
    "type\":\"number\"},\"description\":\"Amplitude "
    "[dB]\",\"type\":\"array\"}},\"title\":\"Noise "
    "Estimates\",\"description\":\"Standard deviation for range and amplitude "
    "as a function of amplitude\"}"
    ,
    /* example: */
    "{\"range_sigma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,"
    "0.016,0.014],\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0."
    "54,0.482,0.432,0.39,0.354],\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7."
    "0,8.0,9.0,10.0]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"rang"
    "e_minimum\",\"range_maximum\",\"amplitude_maximum\"],\"type\":\"object\",\"prope"
    "rties\":{\"range_minimum\":{\"description\":\"Minimum range "
    "[m]\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum range [m"
    "]\",\"type\":\"number\"},\"amplitude_maximum\":{\"minimum\":0,\"description\":\"Ma"
    "ximum amplitude [dB]\",\"type\":\"number\"}},\"title\":\"Notch "
    "Filter\",\"description\":\"Notch filter parameters for window glass "
    "echoes\"}"
    ,
    /* example: */
    "{\"range_minimum\":-0.5,\"range_maximum\":0.2,\"amplitude_maximum\":18.0}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"size"
    "\"],\"type\":\"object\",\"definitions\":{\"pixel_size\":{\"minItems\":2,\"items\":{"
    "\"minimum\":0,\"description\":\"Length of pixel edge "
    "[m].\",\"type\":\"number\"},\"description\":\"Size of pixels.\",\"type\":\"array\","
    "\"maxItems\":2}},\"properties\":{\"size\":{\"$ref\":\"#/definitions/pixel_size\""
    ",\"description\":\"Size of pixels in file coordinate system.\"},\"size_llcs"
    "\":{\"$ref\":\"#/definitions/pixel_size\",\"description\":\"Size of pixels in "
    "a locally levelled cartesian coordinate system (xy). This is only used"
    " for pixels based on a map projection.\"}},\"title\":\"Pixel "
    "Information\",\"description\":\"Details about the pixels contained in the "
    "file\"}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395],\"size_llcs\":[0.5156575"
    "252891171,0.5130835356683303]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"plan"
    "e_patch_file_one\",\"plane_patch_file_two\"],\"type\":\"object\",\"definitions"
    "\":{\"file_reference\":{\"required\":[\"file_uuid\",\"file_path\"],\"description"
    "\":\"Reference to a plane patch file\",\"type\":\"object\",\"properties\":{\"fil"
    "e_path\":{\"type\":\"string\",\"description\":\"Path of the plane patch file "
    "relative to the match "
    "file\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"Plane patch file's "
    "Universally Unique Identifier (RFC 4122)\"}}}},\"properties\":{\"plane_pat"
    "ch_file_two\":{\"$ref\":\"#/definitions/file_reference\",\"description\":\"Ref"
    "erence to the plane patch file two\"},\"plane_patch_file_one\":{\"$ref\":\"#"
    "/definitions/file_reference\",\"description\":\"Reference to the plane "
    "patch file one\"}},\"title\":\"Plane Patch "
    "Matching\",\"description\":\"Details about the plane patch matching "
    "process\"}"
    ,
    /* example: */
    "{\"plane_patch_file_two\":{\"file_path\":\"project.ptch\",\"file_uuid\":\"fa47d"
    "509-a64e-49ce-8b14-ff3130fbefa9\"},\"plane_patch_file_one\":{\"file_path\":"
    "\"Record009_Line001/191025_121410_Scanner_1.ptch\",\"file_uuid\":\"810f5d57"
    "-eccf-49ed-b07a-0cdd109b4213\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Statistics about plane patches found by plane patch "
    "extractor\",\"type\":\"object\",\"title\":\"Plane Patch Statistics\",\"$schema\":"
    "\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"total_horizon"
    "tal_area\":{\"description\":\"sum of all plane patch areas projected to "
    "horizontal plane "
    "[m\\u00b2]\",\"type\":\"number\"},\"total_area\":{\"description\":\"sum of all "
    "plane patch areas [m\\u00b2]\",\"type\":\"number\"}}}"
    ,
    /* example: */
    "{\"total_horizontal_area\":13954.601,\"total_area\":14007.965}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"sett"
    "ings\",\"classes\"],\"type\":\"object\",\"definitions\":{\"method-1\":{\"required\""
    ":[\"plane_classification_method\",\"maximum_inclination_angle_horizontal\""
    "],\"description\":\"Classification method 1\",\"type\":\"object\",\"properties\""
    ":{\"maximum_inclination_angle_horizontal\":{\"maximum\":360.0,\"minimum\":-3"
    "60.0,\"description\":\"maximum inclination angle of horizontal plane "
    "patches [deg]\",\"type\":\"number\"},\"plane_classification_method\":{\"maximu"
    "m\":1,\"minimum\":1,\"description\":\"method ID (=1)\",\"type\":\"integer\"}}},\"m"
    "ethod-2\":{\"required\":[\"plane_classification_method\",\"sloping_plane_cla"
    "sses_minimum_angle\",\"sloping_plane_classes_maximum_angle\"],\"descriptio"
    "n\":\"Classification method 2\",\"type\":\"object\",\"properties\":{\"sloping_pl"
    "ane_classes_maximum_angle\":{\"maximum\":360.0,\"minimum\":-360.0,\"descript"
    "ion\":\"maximum inclination angle of sloping plane patches [deg]\",\"type\""
    ":\"number\"},\"sloping_plane_classes_minimum_angle\":{\"maximum\":360.0,\"min"
    "imum\":-360.0,\"description\":\"minimum inclination angle of sloping plane"
    " patches [deg]\",\"type\":\"number\"},\"plane_classification_method\":{\"maxim"
    "um\":2,\"minimum\":2,\"description\":\"method ID (=2)\",\"type\":\"integer\"}}}},"
    "\"properties\":{\"classes\":{\"additionalProperties\":false,\"description\":\"C"
    "lass definition table\",\"type\":\"object\",\"patternProperties\":{\"^[0-9]+$\""
    ":{\"type\":\"string\",\"description\":\"one field per class, field name is "
    "class number, field value is class name\"}}},\"settings\":{\"oneOf\":[{\"$re"
    "f\":\"#/definitions/method-1\"},{\"$ref\":\"#/definitions/method-2\"}],\"descr"
    "iption\":\"Classification settings, details see documentation of "
    "rdbplanes\",\"type\":\"object\"}},\"title\":\"Plane Slope Class "
    "Info\",\"description\":\"Settings and classes for plane slope "
    "classification\"}"
    ,
    /* example: */
    "{\"classes\":{\"1\":\"horizontal, pointing up\",\"8\":\"vertical, pointing "
    "north\",\"6\":\"vertical, pointing east\",\"11\":\"sloping, pointing down and "
    "south\",\"13\":\"sloping, pointing down and west\",\"14\":\"horizontal, "
    "pointing down\",\"7\":\"vertical, pointing south\",\"3\":\"sloping, pointing "
    "up and south\",\"9\":\"vertical, pointing west\",\"2\":\"sloping, pointing up "
    "and east\",\"12\":\"sloping, pointing down and north\",\"5\":\"sloping, "
    "pointing up and west\",\"4\":\"sloping, pointing up and "
    "north\",\"10\":\"sloping, pointing down and east\"},\"settings\":{\"sloping_pl"
    "ane_classes_maximum_angle\":70.0,\"sloping_plane_classes_minimum_angle\":"
    "10.0,\"plane_classification_method\":2}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"additionalProper"
    "ties\":false,\"type\":\"object\",\"title\":\"Point Attribute "
    "Groups\",\"description\":\"Grouping and sorting of point attributes for "
    "visualization purposes\",\"patternProperties\":{\"^.*$\":{\"minItems\":1,\"ite"
    "ms\":{\"type\":\"string\",\"description\":\"Point attribute full name or name "
    "pattern (perl regular expression "
    "syntax)\"},\"type\":\"array\",\"description\":\"Each field of the object "
    "represents a point attribute group and holds a list of point "
    "attributes, where the field name is the group name.\"}}}"
    ,
    /* example: */
    "{\"Coordinates/Vectors\":[\"riegl.xyz\",\"riegl.range\",\"riegl.theta\",\"riegl"
    ".phi\"],\"Time\":[\"riegl.timestamp\"],\"Secondary "
    "Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available\"],\"Other "
    "Attributes\":[\"riegl.selected\",\"riegl.visible\"],\"Primary "
    "Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.deviation\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about point cloud "
    "files\",\"type\":\"object\",\"title\":\"Point Cloud Information\",\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"properties\":{\"comments\":{\"type"
    "\":\"string\",\"description\":\"Comments\"},\"field_of_application\":{\"enum\":[\""
    "unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"type\":\"stri"
    "ng\",\"description\":\"Field of "
    "application\"},\"project\":{\"type\":\"string\",\"description\":\"Project "
    "name\"}}}"
    ,
    /* example: */
    "{\"comments\":\"Line 3\",\"field_of_application\":\"ALS\",\"project\":\"Campaign "
    "4\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"orie"
    "ntation\"],\"type\":\"object\",\"properties\":{\"position\":{\"required\":[\"coord"
    "inate_1\",\"coordinate_2\",\"coordinate_3\",\"horizontal_accuracy\",\"vertical"
    "_accuracy\",\"crs\"],\"type\":\"object\",\"description\":\"Position coordinates "
    "and position accuracy values as measured by GNSS in the specified "
    "Coordinate Reference System (CRS)\",\"properties\":{\"horizontal_accuracy\""
    ":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"H"
    "orizontal accuracy "
    "[m]\"},\"crs\":{\"required\":[\"epsg\"],\"description\":\"Global Coordinate "
    "Reference System\",\"type\":\"object\",\"properties\":{\"epsg\":{\"minimum\":0,\"t"
    "ype\":\"integer\",\"description\":\"EPSG "
    "code\"},\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\"}}},\"coordinate_2\":{\"type\":\"number\",\"description\":\"Coordinate "
    "2 as defined by axis 2 of the specified CRS (e.g., Y, "
    "Longitude)\"},\"coordinate_3\":{\"type\":\"number\",\"description\":\"Coordinate"
    " 3 as defined by axis 3 of the specified CRS (e.g., Z, "
    "Altitude)\"},\"coordinate_1\":{\"type\":\"number\",\"description\":\"Coordinate "
    "1 as defined by axis 1 of the specified CRS (e.g., X, Latitude)\"},\"ver"
    "tical_accuracy\":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\""
    "description\":\"Vertical accuracy [m]\"}}},\"orientation\":{\"required\":[\"ro"
    "ll\",\"pitch\",\"yaw\",\"roll_accuracy\",\"pitch_accuracy\",\"yaw_accuracy\"],\"ty"
    "pe\":\"object\",\"description\":\"Orientation values and orientation "
    "accuracies, measured with IMU or inclination sensors.\",\"properties\":{\""
    "yaw\":{\"maximum\":360,\"minimum\":-360,\"type\":\"number\",\"description\":\"Yaw "
    "angle about scanner Z-axis [deg]\"},\"roll_accuracy\":{\"exclusiveMinimum\""
    ":true,\"minimum\":0,\"type\":\"number\",\"description\":\"Roll angle accuracy ["
    "deg]\"},\"pitch_accuracy\":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"n"
    "umber\",\"description\":\"Pitch angle accuracy [deg]\"},\"roll\":{\"maximum\":3"
    "60,\"minimum\":-360,\"type\":\"number\",\"description\":\"Roll angle about "
    "scanner X-axis [deg]\"},\"pitch\":{\"maximum\":360,\"minimum\":-360,\"type\":\"n"
    "umber\",\"description\":\"Pitch angle about scanner Y-axis [deg]\"},\"yaw_ac"
    "curacy\":{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"descript"
    "ion\":\"Yaw angle accuracy [deg]\"}}},\"barometric_height_amsl\":{\"type\":\"n"
    "umber\",\"description\":\"Altitude determined based on the atmospheric "
    "pressure according to the standard atmosphere laws "
    "[m].\"}},\"title\":\"Pose Estimation\",\"description\":\"Estimated position "
    "and orientation information as measured by GNSS, IMU or inclination "
    "sensors\"}"
    ,
    /* example: */
    "{\"position\":{\"horizontal_accuracy\":0.810699999332428,\"crs\":{\"epsg\":497"
    "9,\"wkt\":\"GEOGCS[\\\"WGS84 / Geographic\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS8"
    "4\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY["
    "\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000000000000,AUTHORITY["
    "\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Degree\\\",0.01745329251994329547,AUTHORITY[\\"
    "\"EPSG\\\",\\\"9102\\\"]],AXIS[\\\"Latitude\\\",NORTH],AXIS[\\\"Longitude\\\",EAST],A"
    "UTHORITY[\\\"EPSG\\\",\\\"4979\\\"]]\"},\"coordinate_2\":15.645033406,\"coordinate"
    "_3\":362.7124938964844,\"coordinate_1\":48.655799473,\"vertical_accuracy\":"
    "1.3314999341964722},\"orientation\":{\"yaw\":101.87293630292045,\"roll_accu"
    "racy\":0.009433783936875745,\"pitch_accuracy\":0.009433783936875745,\"roll"
    "\":3.14743073066123,\"pitch\":1.509153024827064,\"yaw_accuracy\":1.00943378"
    "39368757},\"barometric_height_amsl\":386.7457796227932}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"gyro"
    "scope\",\"accelerometer\",\"magnetic_field_sensor\"],\"type\":\"object\",\"defin"
    "itions\":{\"vector\":{\"minItems\":3,\"items\":{\"description\":\"Index 0=X, "
    "1=Y, 2=Z component\",\"type\":\"number\"},\"type\":\"array\",\"maxItems\":3}},\"pr"
    "operties\":{\"magnetic_field_sensor\":{\"required\":[\"unit\",\"x_axis\",\"y_axi"
    "s\",\"z_axis\",\"offset\",\"fixed\",\"relative_nonlinearity\"],\"description\":\"M"
    "agnetic Field Sensor details\",\"type\":\"object\",\"properties\":{\"fixed\":{\""
    "$ref\":\"#/definitions/vector\",\"description\":\"Distortion of magnetic "
    "field caused by non-rotating scanner part\"},\"x_axis\":{\"$ref\":\"#/defini"
    "tions/vector\",\"description\":\"Sensitive X axis of sensor at frame angle"
    " = 0\"},\"unit\":{\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Unit"
    " of raw data and calibration values, 1 LSB in nT\",\"type\":\"number\"},\"re"
    "lative_nonlinearity\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Rel"
    "ative nonlinearity, dimensionless\"},\"offset\":{\"$ref\":\"#/definitions/ve"
    "ctor\",\"description\":\"Value to be subtracted from raw measurement value"
    "s\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\"}}},\"gyroscope\":{\"required\":[\"unit\""
    ",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_nonlinearity\"]"
    ",\"description\":\"Gyroscope details\",\"type\":\"object\",\"properties\":{\"x_ax"
    "is\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X axis of "
    "sensor at frame angle = "
    "0\"},\"unit\":{\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Unit of"
    " raw data and calibration values, 1 LSB in rad/s\",\"type\":\"number\"},\"re"
    "lative_nonlinearity\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Rel"
    "ative nonlinearity, dimensionless\"},\"origin\":{\"$ref\":\"#/definitions/ve"
    "ctor\",\"description\":\"Sensor origin in SOCS [m] at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement values\"},\"z_axis\":{\"$ref\":\"#/definiti"
    "ons/vector\",\"description\":\"Sensitive Z axis of sensor at frame angle ="
    " 0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive "
    "Y axis of sensor at frame angle = 0\"}}},\"accelerometer\":{\"required\":[\""
    "unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_nonlinear"
    "ity\"],\"description\":\"Accelerometer details\",\"type\":\"object\",\"propertie"
    "s\":{\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X"
    " axis of sensor at frame angle = "
    "0\"},\"unit\":{\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Unit of"
    " raw data and calibration values, 1 LSB in 9.81 m/s\\u00b2\",\"type\":\"num"
    "ber\"},\"relative_nonlinearity\":{\"$ref\":\"#/definitions/vector\",\"descript"
    "ion\":\"Relative nonlinearity, dimensionless\"},\"origin\":{\"$ref\":\"#/defin"
    "itions/vector\",\"description\":\"Sensor origin in SOCS [m] at frame angle"
    " = 0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to"
    " be subtracted from raw measurement values\"},\"z_axis\":{\"$ref\":\"#/defin"
    "itions/vector\",\"description\":\"Sensitive Z axis of sensor at frame "
    "angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\"}}}},\"title\":\"Pose "
    "Sensors\",\"description\":\"Details on position and orientation sensors\"}"
    ,
    /* example: */
    "{\"magnetic_field_sensor\":{\"fixed\":[-1576.010498046875,1596.08178710937"
    "5,0.0],\"x_axis\":[-0.011162743903696537,-2.315962774446234e-05,0.000168"
    "18844596855342],\"unit\":91.74311828613281,\"relative_nonlinearity\":[0.0,"
    "0.0,0.0],\"offset\":[-23812.052734375,5606.57666015625,2493.28125],\"z_ax"
    "is\":[0.00041987866279669106,7.876977906562388e-05,0.011407104320824146"
    "],\"y_axis\":[0.00027888521435670555,-0.011427424848079681,-5.2048297220"
    "60822e-05]},\"gyroscope\":{\"x_axis\":[-121.195556640625,0.821971416473388"
    "7,0.2313031703233719],\"unit\":0.00014544410805683583,\"relative_nonlinea"
    "rity\":[2.888176311444113e-07,1.06274164579645e-07,-1.7186295080634935e"
    "-39],\"origin\":[0.026900000870227814,-0.03999999910593033,-0.0895000025"
    "6299973],\"offset\":[-50.92609786987305,146.15643310546875,62.4327278137"
    "207],\"z_axis\":[0.555869996547699,119.22135162353516,0.467585027217865]"
    ",\"y_axis\":[-0.440765917301178,-0.7897399663925171,119.5894775390625]},"
    "\"accelerometer\":{\"x_axis\":[-15008.123046875,56.956390380859375,-60.517"
    "5666809082],\"unit\":6.666666740784422e-05,\"relative_nonlinearity\":[0.0,"
    "0.0,0.0],\"origin\":[0.026900000870227814,-0.03999999910593033,-0.089500"
    "00256299973],\"offset\":[-733.3636474609375,58.969032287597656,1060.2550"
    "048828125],\"z_axis\":[1.639882206916809,15166.744140625,-116.9974288940"
    "4297],\"y_axis\":[-7.027288913726807,-44.12333679199219,14952.3701171875"
    "]}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"leng"
    "th\",\"num_mod_ampl\",\"pulse_interval\"],\"type\":\"object\",\"properties\":{\"pu"
    "lse_interval\":{\"items\":{\"minimum\":0,\"type\":\"number\"},\"description\":\"Ex"
    "plicit table of the pulse position modulation used for MTA resolution."
    " Table gives times between successive laser pulses in seconds.\",\"type\""
    ":\"array\"},\"phase_step\":{\"maximum\":255,\"minimum\":0,\"description\":\"Step "
    "width in phase of modulation code from line to line\",\"type\":\"integer\"}"
    ",\"length\":{\"maximum\":255,\"minimum\":0,\"description\":\"Length of code\",\"t"
    "ype\":\"integer\"},\"code_phase_mode\":{\"maximum\":255,\"minimum\":0,\"descript"
    "ion\":\"0: no synchronization, 1: toggle between 2 phases, 2: increment "
    "with phase_increment\",\"type\":\"integer\"},\"num_mod_ampl\":{\"maximum\":255,"
    "\"minimum\":0,\"description\":\"Number of different modulation amplitudes "
    "(2: binary modulation)\",\"type\":\"integer\"}},\"title\":\"Pulse Position "
    "Modulation\",\"description\":\"Laser pulse position modulation used for "
    "MTA resolution\"}"
    ,
    /* example: */
    "{\"pulse_interval\":[2.759375e-06,2.759375e-06,2.759375e-06,2.759375e-06"
    ",2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.82"
    "1875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e"
    "-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2"
    ".759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.8218"
    "75e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.821875e-0"
    "6],\"phase_step\":5,\"length\":31,\"code_phase_mode\":2,\"num_mod_ampl\":2}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"mini"
    "mum\",\"average\",\"maximum\",\"std_dev\"],\"type\":\"object\",\"properties\":{\"std"
    "_dev\":{\"description\":\"Standard "
    "deviation\",\"type\":\"number\"},\"maximum\":{\"description\":\"Maximum "
    "value\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum "
    "value\",\"type\":\"number\"},\"average\":{\"description\":\"Average "
    "value\",\"type\":\"number\"}},\"title\":\"Range "
    "Statistics\",\"description\":\"Statistics about target distance wrt. SOCS "
    "origin\"}"
    ,
    /* example: */
    "{\"std_dev\":24.349,\"maximum\":574.35,\"minimum\":0.919,\"average\":15.49738}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"type\":\"object\",\""
    "definitions\":{\"fp\":{\"required\":[\"s\",\"w\"],\"description\":\"Fingerprint va"
    "lues\",\"type\":\"object\",\"properties\":{\"s\":{\"minItems\":1,\"items\":{\"minIte"
    "ms\":1,\"items\":{\"type\":\"number\"},\"type\":\"array\",\"maxItems\":4096},\"type\""
    ":\"array\",\"maxItems\":256},\"w\":{\"minItems\":1,\"items\":{\"minItems\":5,\"item"
    "s\":{\"type\":\"number\"},\"type\":\"array\",\"maxItems\":5},\"type\":\"array\",\"maxI"
    "tems\":256}}},\"fp_table_row\":{\"minItems\":1,\"items\":{\"type\":\"number\"},\"t"
    "ype\":\"array\",\"maxItems\":2048},\"fp_table\":{\"desription\":\"scanner "
    "internal data\",\"required\":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv\"],\"type\":\"object\",\""
    "properties\":{\"tc\":{\"max\":255,\"min\":0,\"description\":\"table data type co"
    "de\",\"type\":\"integer\"},\"tv\":{\"minItems\":1,\"items\":{\"oneOf\":[{\"$ref\":\"#/"
    "definitions/fp_table_row\"},{\"type\":\"number\"}]},\"type\":\"array\",\"maxItem"
    "s\":2048},\"ny\":{\"max\":2048,\"min\":1,\"description\":\"number of y entries\","
    "\"type\":\"integer\"},\"ch\":{\"max\":255,\"min\":0,\"description\":\"channel numbe"
    "r\",\"type\":\"integer\"},\"nx\":{\"max\":2048,\"min\":1,\"description\":\"number of"
    " x entries\",\"type\":\"integer\"}}}},\"properties\":{\"sr\":{\"exclusiveMinimum"
    "\":true,\"minimum\":0,\"description\":\"Sample rate [Hz]\",\"type\":\"number\"},\""
    "t\":{\"additionalProperties\":false,\"patternProperties\":{\"^[0-9]+$\":{\"$re"
    "f\":\"#/definitions/fp\",\"description\":\"one field per channel, field name"
    " is channel index\"}},\"type\":\"object\"},\"ex\":{\"description\":\"DEPRECATED,"
    " use 'riegl.exponential_decomposition' instead\",\"type\":\"object\"},\"si\":"
    "{\"maximum\":255,\"minimum\":0,\"description\":\"Start index (hw_start)\",\"typ"
    "e\":\"number\"},\"mw\":{\"maximum\":4095,\"exclusiveMinimum\":true,\"minimum\":0,"
    "\"description\":\"Maximum weight\",\"type\":\"number\"},\"nt\":{\"maximum\":255,\"m"
    "inimum\":0,\"description\":\"Number of traces\",\"type\":\"integer\"},\"tbl\":{\"m"
    "inItems\":1,\"items\":{\"$ref\":\"#/definitions/fp_table\"},\"description\":\"va"
    "rious internal data\",\"type\":\"array\"},\"a\":{\"minItems\":1,\"items\":{\"type\""
    ":\"number\"},\"description\":\"Amplitude [dB]\",\"type\":\"array\",\"maxItems\":25"
    "6},\"ns\":{\"maximum\":4095,\"minimum\":0,\"description\":\"Number of "
    "samples\",\"type\":\"integer\"}},\"title\":\"Receiver "
    "Internals\",\"description\":\"Receiver Internals\"}"
    ,
    /* example: */
    "{\"sr\":7959997000.0,\"t\":{\"0\":{\"s\":[[1.23,4.56],[7.89,0.12]],\"w\":[[78,86"
    ",126,134,31],[78,86,126,134,31]]},\"1\":{\"s\":[[1.23,4.56],[7.89,0.12]],\""
    "w\":[[78,86,126,134,31],[78,86,126,134,31]]}},\"si\":48,\"mw\":31,\"nt\":128,"
    "\"tbl\":[{\"tc\":1,\"tv\":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.5]],\"ny\":2,\"ch\":0,"
    "\"nx\":5}],\"a\":[-1.55],\"ns\":400}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"delt"
    "a\",\"content\"],\"type\":\"object\",\"properties\":{\"content\":{\"minItems\":1,\"i"
    "tems\":{\"type\":\"number\"},\"type\":\"array\",\"description\":\"Correction value"
    " [dB] to be added to the amplitude\",\"maxItems\":2000},\"delta\":{\"type\":\""
    "number\",\"description\":\"Delta between table entries [m], first entry is"
    " at range = 0 m\"}},\"title\":\"Reflectance Calculation "
    "Table\",\"description\":\"Lookup table for reflectance calculation based "
    "on amplitude and range\"}"
    ,
    /* example: */
    "{\"content\":[-33.01],\"delta\":0.150918}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"rang"
    "es_m\",\"line_angles_deg\",\"reflectance_correction_db\"],\"type\":\"object\",\""
    "properties\":{\"ranges_m\":{\"items\":{\"type\":\"number\"},\"description\":\"Rang"
    "e [m]\",\"type\":\"array\"},\"line_angles_deg\":{\"items\":{\"type\":\"number\"},\"d"
    "escription\":\"Angle [deg]\",\"type\":\"array\"},\"reflectance_correction_db\":"
    "{\"items\":{\"items\":{\"type\":\"number\",\"description\":\"columns (each value "
    "corresponds to an angle)\"},\"type\":\"array\",\"description\":\"rows (each "
    "array corresponds to a range)\"},\"description\":\"Near range reflectance "
    "correction in dB as a function of range over "
    "angle\",\"type\":\"array\"}},\"title\":\"Near-range reflectance "
    "correction\",\"description\":\"Range-dependent and scan-angle-dependent "
    "correction of reflectance reading\"}"
    ,
    /* example: */
    "{\"ranges_m\":[0.0,1.0,2.0,3.0],\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2"
    ".0,2.5,3.0,3.5,4.0],\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,"
    "0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0."
    "8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,"
    "0.1,0.05,0.01]]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"traj"
    "ectory\",\"optimizer\",\"georeferencing\"],\"type\":\"object\",\"properties\":{\"p"
    "lanes\":{\"type\":\"object\",\"description\":\"Information on plane "
    "correspondences and used extraction/matching templates.\",\"properties\":"
    "{}},\"trajectory\":{\"type\":\"object\",\"description\":\"Trajectory "
    "information and statistics.\",\"properties\":{}},\"optimizer\":{\"type\":\"obj"
    "ect\",\"description\":\"Optimization result information and statistics.\",\""
    "properties\":{}},\"georeferencing\":{\"type\":\"object\",\"description\":\"Inter"
    "nal georeferencing parameters.\",\"properties\":{}}},\"title\":\"RiLOC "
    "Processing Information\",\"description\":\"Details about RiLOC "
    "processing\"}"
    ,
    /* example: */
    "{\"planes\":{},\"trajectory\":{},\"optimizer\":{},\"georeferencing\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scan pattern description\",\"properties\":{\"rectangular\":"
    "{\"required\":[\"phi_start\",\"phi_stop\",\"phi_increment\",\"theta_start\",\"the"
    "ta_stop\",\"theta_increment\"],\"type\":\"object\",\"description\":\"Rectangular"
    " Field Of View Scan Pattern\",\"properties\":{\"phi_start\":{\"maximum\":360."
    "0,\"minimum\":0.0,\"type\":\"number\",\"description\":\"Start phi angle in SOCS"
    " [deg]\"},\"theta_start\":{\"maximum\":180.0,\"minimum\":0.0,\"type\":\"number\","
    "\"description\":\"Start theta angle in SOCS [deg]\"},\"phi_stop\":{\"maximum\""
    ":720.0,\"minimum\":0.0,\"type\":\"number\",\"description\":\"Stop phi angle in "
    "SOCS [deg]\"},\"phi_increment\":{\"maximum\":90.0,\"minimum\":0.0,\"type\":\"num"
    "ber\",\"description\":\"Increment of phi angle in SOCS [deg]\"},\"theta_stop"
    "\":{\"maximum\":180.0,\"minimum\":0.0,\"type\":\"number\",\"description\":\"Stop "
    "theta angle in SOCS [deg]\"},\"theta_increment\":{\"maximum\":90.0,\"minimum"
    "\":0.0,\"type\":\"number\",\"description\":\"Increment of theta angle in SOCS "
    "[deg]\"},\"program\":{\"$ref\":\"#/definitions/program\"}}},\"segments\":{\"requ"
    "ired\":[\"list\"],\"type\":\"object\",\"description\":\"Segmented Line Scan Patt"
    "ern\",\"properties\":{\"program\":{\"$ref\":\"#/definitions/program\"},\"list\":{"
    "\"items\":{\"required\":[\"start\",\"stop\",\"increment\"],\"type\":\"object\",\"desc"
    "ription\":\"Line Scan Segment\",\"properties\":{\"stop\":{\"maximum\":720.0,\"mi"
    "nimum\":0.0,\"type\":\"number\",\"description\":\"Stop angle in SOCS [deg]\"},\""
    "increment\":{\"maximum\":90.0,\"minimum\":0.0,\"type\":\"number\",\"description\""
    ":\"Increment of angle in SOCS [deg]\"},\"start\":{\"maximum\":360.0,\"minimum"
    "\":0.0,\"type\":\"number\",\"description\":\"Start angle in SOCS [deg]\"}}},\"ty"
    "pe\":\"array\"}}},\"line\":{\"required\":[\"start\",\"stop\",\"increment\"],\"type\":"
    "\"object\",\"description\":\"Line Scan Pattern\",\"properties\":{\"stop\":{\"maxi"
    "mum\":720.0,\"minimum\":0.0,\"type\":\"number\",\"description\":\"Stop angle in "
    "SOCS [deg]\"},\"increment\":{\"maximum\":90.0,\"minimum\":0.0,\"type\":\"number\""
    ",\"description\":\"Increment of angle in SOCS [deg]\"},\"program\":{\"$ref\":\""
    "#/definitions/program\"},\"start\":{\"maximum\":360.0,\"minimum\":0.0,\"type\":"
    "\"number\",\"description\":\"Start angle in SOCS [deg]\"}}}},\"title\":\"Scan P"
    "attern\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definitio"
    "ns\":{\"program\":{\"required\":[\"name\"],\"type\":\"object\",\"description\":\"Mea"
    "surement "
    "program\",\"properties\":{\"name\":{\"type\":\"string\",\"description\":\"Name of "
    "measurement program\"},\"laser_prr\":{\"exclusiveMinimum\":false,\"minimum\":"
    "0,\"type\":\"number\",\"description\":\"Laser Pulse Repetition Rate "
    "[Hz]\"}}}}}"
    ,
    /* example: */
    "{\"rectangular\":{\"phi_start\":45.0,\"theta_start\":30.0,\"phi_stop\":270.0,\""
    "phi_increment\":0.04,\"theta_stop\":130.0,\"theta_increment\":0.04,\"program"
    "\":{\"name\":\"High Speed\",\"laser_prr\":100000.0}}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about laser shot "
    "files\",\"type\":\"object\",\"title\":\"Shot Information\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"properties\":{\"shot_file\":{\"required\":"
    "[\"file_extension\"],\"type\":\"object\",\"properties\":{\"file_extension\":{\"ty"
    "pe\":\"string\",\"description\":\"Shot file extension, without the leading "
    "dot\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\"}}}}}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_extension\":\"sodx\",\"file_uuid\":\"26a00815-67c0-4bff-"
    "8fe8-c577378fe663\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"acti"
    "vated\",\"filters\"],\"type\":\"object\",\"properties\":{\"activated\":{\"type\":\"b"
    "oolean\",\"description\":\"Apply ('true') or ignore ('false') all filters\""
    "},\"filters\":{\"items\":{\"required\":[\"activated\",\"title\",\"description\",\"f"
    "ilter\"],\"type\":\"object\",\"description\":\"Point filter "
    "definition\",\"properties\":{\"filter\":{\"type\":\"string\",\"description\":\"The"
    " RDB filter string to apply (e.g. when reading points or index), "
    "details see documentation of function "
    "select()\"},\"activated\":{\"type\":\"boolean\",\"description\":\"Apply ('true')"
    " or ignore ('false') this "
    "filter\"},\"title\":{\"type\":\"string\",\"description\":\"A short filter title "
    "(e.g. for display in a selection "
    "list)\"},\"description\":{\"type\":\"string\",\"description\":\"A brief "
    "description of the filter (e.g. for display in a "
    "tooltip)\"}}},\"type\":\"array\",\"description\":\"List of point "
    "filters\"}},\"title\":\"Stored filters\",\"description\":\"Point filters "
    "applied in addition to the application-defined filters\"}"
    ,
    /* example: */
    "{\"activated\":true,\"filters\":[{\"filter\":\"riegl.mta_uncertain_point == "
    "0\",\"activated\":false,\"title\":\"Ignore uncertain points from MTA "
    "resolution\",\"description\":\"Ignore points with uncertain MTA zone "
    "assignment\"},{\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\",\"activated\":false,\"title\":\"Ignore points below dynamic trigger "
    "level\",\"description\":\"Ignore points with an echo signal amplitude "
    "below the dynamic trigger\"},{\"filter\":\"riegl.point_outside_aabb_bocs "
    "== 0\",\"activated\":false,\"title\":\"Ignore points outside outer "
    "AABB\",\"description\":\"Ignore points that are outside the outer AABB in "
    "BOCS\"},{\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\",\"activated\":false,\"title\":\"Ignore points inside inner "
    "AABB\",\"description\":\"Ignore points that are inside the inner AABB in "
    "BOCS\"}]}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"$cla"
    "ss\",\"version\",\"author\",\"timestamp\",\"system\"],\"type\":\"object\",\"properti"
    "es\":{\"timestamp\":{\"type\":\"string\",\"description\":\"Date and time of "
    "creation (e.g. 2019-06-19T13:32:10+02:00)\"},\"$class\":{\"enum\":[\"Documen"
    "t\"],\"type\":\"string\",\"description\":\"Object class "
    "name\"},\"system\":{\"type\":\"object\",\"description\":\"The actual system "
    "description, details see documentation of RIEGL System "
    "Description\"},\"author\":{\"type\":\"string\",\"description\":\"Name of the "
    "author that created the "
    "document\"},\"version\":{\"type\":\"string\",\"description\":\"Document format "
    "version\"}},\"title\":\"System Description\",\"description\":\"Details of "
    "major system components like lidar sensors, cameras and navigation "
    "devices\"}"
    ,
    /* example: */
    "{\"timestamp\":\"2022-09-30T11:56:26+00:00\",\"$class\":\"Document\",\"system\":"
    "{\"details\":\"see documentation of RIEGL System "
    "Description\"},\"author\":\"RIEGL LMS GmbH, Austria\",\"version\":\"1.2.1\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"type\":\"object\",\""
    "definitions\":{\"conversion_table\":{\"required\":[\"value\",\"temperature\"],\""
    "type\":\"object\",\"properties\":{\"temperature\":{\"items\":{\"type\":\"number\"},"
    "\"description\":\"Temperature [\\u00b0C]\",\"type\":\"array\"},\"value\":{\"items\""
    ":{\"type\":\"number\"},\"description\":\"LSB [1]\",\"type\":\"array\"}}}},\"propert"
    "ies\":{\"Si\":{\"$ref\":\"#/definitions/conversion_table\",\"description\":\"Con"
    "version table for Si channel\"},\"InGaAs_Si_Difference\":{\"$ref\":\"#/defin"
    "itions/conversion_table\",\"description\":\"Conversion table for InGaAs - "
    "Si difference\"},\"InGaAs\":{\"$ref\":\"#/definitions/conversion_table\",\"des"
    "cription\":\"Conversion table for InGaAs channel\"}},\"title\":\"Temperature"
    " Calculation Table\",\"description\":\"Conversion of background radiation "
    "raw values to temperatures in \\u00b0C\"}"
    ,
    /* example: */
    "{\"Si\":{\"temperature\":[546.300048828125,548.8164051212026,551.314393850"
    "0972,554.0144257850053,556.604252334815,559.2124464488079,561.80227299"
    "86177,564.4104671126105,567.0002936624203],\"value\":[0.0,64.00097659230"
    "323,128.0019531846065,192.0029297769097,256.0039063692129,320.00488296"
    "151616,384.0058595538194,448.0068361461226,512.0078127384259]},\"InGaAs"
    "_Si_Difference\":{\"temperature\":[1749.977111117893,1749.977111117893,17"
    "49.977111117893,1749.977111117893,1749.977111117893,1749.977111117893,"
    "1744.7813348796044,1681.9971312601092,1622.3944822534868],\"value\":[100"
    "0.0,1100.090029602954,1200.04425183874,1300.1342814416948,1400.0885036"
    "774805,1500.0427259132668,1600.1327555162209,1700.0869777520065,1800.0"
    "411999877924]},\"InGaAs\":{\"temperature\":[307.22196722535614,309.1153478"
    "247277,311.1188086915047,313.10025350480055,315.2137946389828,317.2172"
    "555057597,319.2207163725366,321.2021611858325,323.3157023200148],\"valu"
    "e\":[0.0,64.00097659230323,128.0019531846065,192.0029297769097,256.0039"
    "063692129,320.00488296151616,384.0058595538194,448.0068361461226,512.0"
    "078127384259]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"epoc"
    "h\",\"source\"],\"type\":\"object\",\"properties\":{\"system\":{\"enum\":[\"unknown\""
    ",\"UTC\",\"GPS\"],\"type\":\"string\",\"description\":\"Time system (time standar"
    "d)\"},\"source\":{\"enum\":[\"unknown\",\"RTC\",\"GNSS\"],\"type\":\"string\",\"descri"
    "ption\":\"Timestamp "
    "source\"},\"epoch\":{\"type\":\"string\",\"description\":\"Date and time of "
    "timestamp '0' as proposed by RFC 3339 (e.g. "
    "2015-10-27T00:00:00+01:00).\"}},\"title\":\"Time Base\",\"description\":\"Base"
    " of timestamps (epoch)\"}"
    ,
    /* example: */
    "{\"system\":\"UTC\",\"source\":\"GNSS\",\"epoch\":\"2015-10-27T00:00:00+00:00\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"time"
    "_interval\",\"navigation_frame\"],\"type\":\"object\",\"properties\":{\"company\""
    ":{\"type\":\"string\",\"description\":\"Company name\"},\"time_interval\":{\"requ"
    "ired\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"type\":\"object\",\"descr"
    "iption\":\"Time interval statistics\",\"properties\":{\"std_dev\":{\"type\":\"nu"
    "mber\",\"description\":\"Standard deviation of intervals "
    "[s]\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum time interval "
    "[s]\"},\"minimum\":{\"type\":\"number\",\"description\":\"Minimum time interval "
    "[s]\"},\"average\":{\"type\":\"number\",\"description\":\"Average time interval "
    "[s]\"}}},\"location\":{\"type\":\"string\",\"description\":\"Project location, "
    "e.g. "
    "city/state/country\"},\"project\":{\"type\":\"string\",\"description\":\"Project"
    " name\"},\"settings\":{\"type\":\"string\",\"description\":\"Settings used to "
    "calculate the trajectory (descriptive text)\"},\"field_of_application\":{"
    "\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"ty"
    "pe\":\"string\",\"description\":\"Field of application\"},\"navigation_frame\":"
    "{\"enum\":[\"unknown\",\"NED\",\"ENU\"],\"type\":\"string\",\"description\":\"Navigat"
    "ion frame (NED: North-East-Down, ENU: "
    "East-North-Up)\"},\"device\":{\"type\":\"string\",\"description\":\"Navigation "
    "device, e.g. "
    "name/type/serial\"},\"software\":{\"type\":\"string\",\"description\":\"Software"
    " that calculated the trajectory (this may be the same or different "
    "software than the one that created the file)\"}},\"title\":\"Trajectory "
    "Information\",\"description\":\"Details about position+orientation files\"}"
    ,
    /* example: */
    "{\"company\":\"RIEGL LMS\",\"time_interval\":{\"std_dev\":5.51e-07,\"maximum\":0"
    ".005004883,\"minimum\":0.00500032,\"average\":0.00500053},\"location\":\"Horn"
    "\",\"project\":\"Campaign 3\",\"settings\":\"default\",\"field_of_application\":\""
    "MLS\",\"navigation_frame\":\"NED\",\"device\":\"IMU Model 12/1, Serial# "
    "12345\",\"software\":\"Navigation Software XYZ\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ttip"
    "_version\",\"num_channel\",\"ext_channel\",\"ext_signal\",\"ext_delay\",\"ext_su"
    "bdivider\",\"ext_bitmask\",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_"
    "dist_per_tick\",\"channel\",\"output_descr\",\"input_descr\",\"output_usage\",\""
    "input_usage\",\"out_polarity\",\"out_duration\",\"out_interval\",\"out_delay\","
    "\"out_num_pulses\",\"in_polarity\",\"in_min_duration\",\"in_max_duration\",\"in"
    "_max_delay\"],\"type\":\"object\",\"properties\":{\"in_max_duration\":{\"minimum"
    "\":0,\"type\":\"integer\",\"description\":\"stops measurement of pulse "
    "duration of input signal [0.1 msec]\"},\"out_interval\":{\"minimum\":0,\"typ"
    "e\":\"integer\",\"description\":\"output pulse interval [0.1 msec]\"},\"dmi_mi"
    "n_time\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"dmi, minimum time"
    " interval between trigger events [0.1 "
    "msec]\"},\"dmi_incr\":{\"type\":\"integer\",\"description\":\"dmi, increment in "
    "ticks\"},\"channel\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"ID of "
    "input/output "
    "channel\"},\"input_usage\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"0"
    " .. deactivated, 1 .. just detecting and timestamping\"},\"ext_signal\":{"
    "\"minimum\":0,\"type\":\"integer\",\"description\":\"0 .. use input signal, 1 "
    ".. use output signal of channel\"},\"in_min_duration\":{\"minimum\":0,\"type"
    "\":\"integer\",\"description\":\"input signals with smaller pulse durations "
    "are ignored [0.1 "
    "msec]\"},\"out_polarity\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"0 "
    ".. positive edge, 1 .. negative "
    "edge\"},\"output_usage\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"0 "
    ".. no output, 1 .. output derived from internal clock, 2 .. output "
    "derived from dmi, 3 .. output derived from external signal, 4 .. "
    "output static low, 5 .. output static "
    "high\"},\"dmi_dist_per_tick\":{\"type\":\"number\",\"description\":\"dmi, "
    "distance per tick, just informative "
    "[m]\"},\"output_descr\":{\"type\":\"string\",\"description\":\"descriptive strin"
    "g\"},\"dmi_max_time\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"dmi, "
    "maximum time interval between trigger events [0.1 msec]\"},\"out_num_pul"
    "ses\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"number of output "
    "pulses to be generated, 0 .. infinite\"},\"ttip_version\":{\"minimum\":0,\"t"
    "ype\":\"integer\",\"description\":\"following main.sub.ss.sss\"},\"input_descr"
    "\":{\"type\":\"string\",\"description\":\"descriptive string\"},\"ext_subdivider"
    "\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"reduces the frequency, "
    "default "
    "1\"},\"out_delay\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"output "
    "pulse initial delay after start [0.1 msec]\"},\"ext_bitmask\":{\"minimum\":"
    "0,\"type\":\"integer\",\"description\":\"defines which of the internal pulse "
    "generators are to be started\"},\"ext_channel\":{\"minimum\":0,\"type\":\"inte"
    "ger\",\"description\":\"ID of channel used as external trigger input, 32 "
    "indicates none\"},\"in_max_delay\":{\"minimum\":0,\"type\":\"integer\",\"descrip"
    "tion\":\"maximum delay to output pulse before fake event is generated "
    "[0.1 msec], zero indicates that no fake events are generated\"},\"num_ch"
    "annel\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"number of "
    "input/output channels\"},\"out_duration\":{\"minimum\":0,\"type\":\"integer\",\""
    "description\":\"output pulse duration [0.1 "
    "msec]\"},\"in_polarity\":{\"minimum\":0,\"type\":\"integer\",\"description\":\"0 "
    ".. positive edge, 1 .. negative edge\"},\"ext_delay\":{\"minimum\":0,\"type\""
    ":\"integer\",\"description\":\"external trigger, time delay [0.1 "
    "msec]\"}},\"title\":\"TTIP "
    "Configuration\",\"description\":\"Trigger-Timestamping-Unit/IP "
    "configuration\"}"
    ,
    /* example: */
    "{\"in_max_duration\":10000,\"out_interval\":10000,\"dmi_min_time\":0,\"dmi_in"
    "cr\":0,\"channel\":0,\"input_usage\":1,\"ext_signal\":0,\"in_min_duration\":0,\""
    "out_polarity\":0,\"output_usage\":1,\"dmi_dist_per_tick\":0.0,\"output_descr"
    "\":\"Port 1 - Trigger\",\"dmi_max_time\":0,\"out_num_pulses\":1,\"ttip_version"
    "\":1,\"input_descr\":\"Port 1 - Exposure\",\"ext_subdivider\":1,\"out_delay\":0"
    ",\"ext_bitmask\":0,\"ext_channel\":32,\"in_max_delay\":9990,\"num_channel\":9,"
    "\"out_duration\":300,\"in_polarity\":0,\"ext_delay\":0}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about vertex "
    "file\",\"type\":\"object\",\"title\":\"Vertex Information\",\"$schema\":\"http://j"
    "son-schema.org/draft-04/schema#\",\"properties\":{\"vertex_file\":{\"require"
    "d\":[\"file_extension\"],\"type\":\"object\",\"properties\":{\"file_extension\":{"
    "\"type\":\"string\",\"description\":\"Vertex file extension, without the "
    "leading dot\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's "
    "Universally Unique Identifier (RFC 4122)\"}}}}}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_extension\":\"vtx\",\"file_uuid\":\"51534d95-d71f-4f36"
    "-ae1a-3e63a21fd1c7\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"oneOf\":[{\"requir"
    "ed\":[\"size\",\"voxel_origin\",\"voxel_type\"],\"additionalProperties\":false,"
    "\"properties\":{\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_threshol"
    "ds\"},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_type\"},\"voxel_origin\":{"
    "\"$ref\":\"#/definitions/voxel_origin_enum\"},\"size\":{\"oneOf\":[{\"$ref\":\"#/"
    "definitions/voxel_size\"},{\"$ref\":\"#/definitions/voxel_size_cubic\"}],\"d"
    "escription\":\"Size of voxels in file coordinate system.\"}}},{\"required\""
    ":[\"reference_point\",\"size_llcs\",\"size\",\"voxel_origin\",\"voxel_type\"],\"a"
    "dditionalProperties\":false,\"properties\":{\"size\":{\"$ref\":\"#/definitions"
    "/voxel_size\",\"description\":\"Size of voxels in file coordinate system.\""
    "},\"size_llcs\":{\"$ref\":\"#/definitions/voxel_size\",\"description\":\"Size "
    "of voxels in a locally levelled cartesian coordinate system (xyz). "
    "This is only used for voxels based on a map projection.\"},\"voxel_origi"
    "n\":{\"oneOf\":[{\"$ref\":\"#/definitions/voxel_origin_enum\"},{\"$ref\":\"#/def"
    "initions/voxel_origin_point\",\"description\":\"The base point of the "
    "voxel grid. Used together with <tt>voxel_size</tt> and "
    "<tt>voxel_index</tt> to compute actual point coordinates.\"}]},\"referen"
    "ce_point\":{\"minItems\":2,\"items\":{\"maximum\":180,\"minimum\":-180,\"type\":\""
    "number\"},\"type\":\"array\",\"description\":\"Point in WGS84 geodetic decimal"
    " degree (EPSG:4326) that was used to compute the projection distortion"
    " parameters. The coefficient order is latitude, longitude. Only voxels"
    " with corresponding geo_tag, voxel_size and reference_point can be "
    "reliably processed together. This entry is available for voxel files "
    "in projected CRS only.\",\"maxItems\":2},\"shape_thresholds\":{\"$ref\":\"#/de"
    "finitions/shape_thresholds\"},\"voxel_type\":{\"$ref\":\"#/definitions/voxel"
    "_type\"}}}],\"type\":\"object\",\"definitions\":{\"voxel_origin_point\":{\"minIt"
    "ems\":3,\"items\":{\"type\":\"number\"},\"type\":\"array\",\"description\":\"Origin "
    "point for all voxel indices in voxel CRS.\",\"maxItems\":3},\"voxel_origin"
    "_enum\":{\"enum\":[\"center\",\"corner\"],\"default\":\"corner\",\"description\":\"D"
    "efines whether the voxel's center or a corner is placed on CRS origin "
    "<tt>(0/0/0)</tt>.\"},\"voxel_size\":{\"minItems\":3,\"items\":{\"$ref\":\"#/defi"
    "nitions/edge_length\"},\"type\":\"array\",\"description\":\"Size of voxels.\",\""
    "maxItems\":3},\"shape_thresholds\":{\"required\":[\"plane\",\"line\"],\"type\":\"o"
    "bject\",\"description\":\"Thresholds used to compute the voxel's shape_id "
    "value.\",\"properties\":{\"plane\":{\"maximum\":1,\"exclusiveMinimum\":true,\"mi"
    "nimum\":0,\"description\":\"If the smallest eigenvalue is smaller than the"
    " median eigenvalue * plane_threshold, the voxel is considered a plane."
    "\",\"exclusiveMaximum\":true,\"type\":\"number\"},\"line\":{\"exclusiveMinimum\":"
    "true,\"minimum\":1,\"type\":\"number\",\"description\":\"If the biggest "
    "eigenvalue is bigger than the median eigenvalue * line_threshold, the "
    "voxel is considered a line.\"}}},\"voxel_type\":{\"enum\":[\"center\",\"centro"
    "id\",\"index\"],\"default\":\"centroid\",\"description\":\"Whether a point in a "
    "voxel represents its center or its centroid. If type is <tt>index</tt>"
    " there is no point but only an integer voxel index.\"},\"voxel_size_cubi"
    "c\":{\"$ref\":\"#/definitions/edge_length\",\"type\":\"number\"},\"edge_length\":"
    "{\"exclusiveMinimum\":true,\"minimum\":0,\"type\":\"number\",\"description\":\"Le"
    "ngth of voxel edge [m].\"}},\"title\":\"Voxel "
    "Information\",\"description\":\"Details about the voxels contained in the "
    "file\"}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395,0.5143705304787237],\"si"
    "ze_llcs\":[0.5156575252891171,0.5130835356683303,0.5143705304787237],\"s"
    "hape_thresholds\":{\"plane\":0.16,\"line\":6},\"reference_point\":[48,16],\"vo"
    "xel_origin\":\"corner\",\"voxel_type\":\"centroid\"}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"num_"
    "shots\",\"mta_zone\"],\"type\":\"object\",\"properties\":{\"trim\":{\"maximum\":0.5"
    ",\"minimum\":0,\"description\":\"Percentage for robust averaging.\",\"type\":\""
    "number\",\"default\":0},\"num_shots\":{\"minimum\":1,\"description\":\"Number of"
    " consecutive shots to be used for averaging.\",\"type\":\"integer\"},\"mta_z"
    "one\":{\"minimum\":1,\"description\":\"Fixed MTA zone for "
    "averaging.\",\"type\":\"integer\"}},\"title\":\"Waveform Averaging "
    "Settings\",\"description\":\"Settings for waveform averaging\"}"
    ,
    /* example: */
    "{\"trim\":0.05,\"num_shots\":7,\"mta_zone\":1}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"samp"
    "le_block_file\",\"sample_data_files\"],\"type\":\"object\",\"properties\":{\"ran"
    "ge_offset_waveform_samples_m\":{\"type\":\"number\",\"description\":\"Calibrat"
    "ed device specific range offset for waveform samples in "
    "meters.\"},\"range_offset_m\":{\"type\":\"number\",\"description\":\"Calibrated "
    "device specific range offset for waveform analysis by system response "
    "fitting in meters.\"},\"sample_data_files\":{\"items\":{\"required\":[\"channe"
    "l\",\"channel_name\",\"sample_interval\",\"sample_bits\",\"laser_wavelength\",\""
    "delta_st\",\"file_extension\"],\"type\":\"object\",\"properties\":{\"file_extens"
    "ion\":{\"type\":\"string\",\"description\":\"Sample data file extension, "
    "without the leading dot\"},\"channel\":{\"maximum\":255,\"exclusiveMinimum\":"
    "false,\"minimum\":0,\"type\":\"integer\",\"exclusiveMaximum\":false,\"descripti"
    "on\":\"Sample block channel number (255 = "
    "invalid)\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's "
    "Universally Unique Identifier (RFC 4122)\"},\"laser_wavelength\":{\"exclus"
    "iveMinimum\":false,\"minimum\":0,\"type\":\"number\",\"description\":\"Laser "
    "wavelength in meters (0 = unknown)\"},\"sample_interval\":{\"exclusiveMini"
    "mum\":false,\"minimum\":0,\"type\":\"number\",\"description\":\"Sampling "
    "interval in seconds\"},\"sample_bits\":{\"maximum\":32,\"exclusiveMinimum\":f"
    "alse,\"minimum\":0,\"type\":\"integer\",\"exclusiveMaximum\":false,\"descriptio"
    "n\":\"Bitwidth of samples (e.g. 10 bit, 12 "
    "bit)\"},\"channel_name\":{\"type\":\"string\",\"description\":\"Sample block "
    "channel name\"},\"delta_st\":{\"type\":\"number\",\"description\":\"reserved\"}}}"
    ",\"type\":\"array\"},\"sample_block_file\":{\"required\":[\"file_extension\"],\"t"
    "ype\":\"object\",\"properties\":{\"file_extension\":{\"type\":\"string\",\"descrip"
    "tion\":\"Sample block file extension, without the leading "
    "dot\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\"}}}},\"title\":\"Waveform "
    "Information\",\"description\":\"Details about waveform files\"}"
    ,
    /* example: */
    "{\"range_offset_m\":3.1415,\"sample_data_files\":[{\"file_extension\":\"sp0\","
    "\"channel\":0,\"file_uuid\":\"da084413-e3e8-4655-a122-071de8490d8e\",\"laser_"
    "wavelength\":0,\"sample_interval\":1.00503e-09,\"sample_bits\":12,\"channel_"
    "name\":\"high_power\",\"delta_st\":0},{\"file_extension\":\"sp1\",\"channel\":1,\""
    "file_uuid\":\"93585b5e-5ea9-43a1-947b-e7ba3be642d2\",\"laser_wavelength\":0"
    ",\"sample_interval\":1.00503e-09,\"sample_bits\":12,\"channel_name\":\"low_po"
    "wer\",\"delta_st\":0},{\"file_extension\":\"sp5\",\"channel\":5,\"file_uuid\":\"9d"
    "2298c4-1036-464f-b5cb-1cf8e517f3a0\",\"laser_wavelength\":0,\"sample_inter"
    "val\":1.00503e-09,\"sample_bits\":12,\"channel_name\":\"wwf\",\"delta_st\":0}],"
    "\"range_offset_waveform_samples_m \":7.283,\"sample_block_file\":{\"file_ex"
    "tension\":\"sbx\",\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\"}}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scanner settings for waveform output\",\"type\":\"array\",\""
    "items\":{\"required\":[\"sbl_name\",\"enabled\",\"channel_idx_mask\"],\"type\":\"o"
    "bject\",\"properties\":{\"pass_rng_less\":{\"type\":\"number\",\"description\":\"T"
    "hreshold for range less than "
    "[m]\"},\"pass_rng_greater\":{\"type\":\"number\",\"description\":\"Threshold for"
    " range greater than "
    "[m]\"},\"pass_dev_less\":{\"type\":\"integer\",\"description\":\"Threshold for "
    "deviation less than "
    "[1]\"},\"logic_expression\":{\"type\":\"string\",\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\"},\"channel_idx_mask\":{\"type\":\"integer\",\"description\":\"Bit mask "
    "for channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, "
    "...\"},\"pass_ampl_greater\":{\"type\":\"number\",\"description\":\"Threshold "
    "for amplitude greater than "
    "[dB]\"},\"pass_dev_greater\":{\"type\":\"integer\",\"description\":\"Threshold "
    "for deviation greater than "
    "[1]\"},\"smart_enabled\":{\"type\":\"boolean\",\"description\":\"Smart waveform "
    "output enabled\"},\"enabled\":{\"type\":\"boolean\",\"description\":\"Waveform "
    "output "
    "enabled\"},\"pass_ampl_less\":{\"type\":\"number\",\"description\":\"Threshold "
    "for amplitude less than "
    "[dB]\"},\"sbl_name\":{\"type\":\"string\",\"description\":\"Name of sample "
    "block, e.g.: wfm, wwf\"}}},\"title\":\"Waveform "
    "Settings\",\"$schema\":\"http://json-schema.org/draft-04/schema#\"}"
    ,
    /* example: */
    "[{\"pass_rng_less\":13.11,\"pass_rng_greater\":9.27,\"channel_idx_mask\":11,"
    "\"pass_ampl_greater\":1.0,\"smart_enabled\":true,\"enabled\":true,\"pass_ampl"
    "_less\":5.0,\"sbl_name\":\"wfm\"},{\"enabled\":false,\"channel_idx_mask\":32,\"s"
    "bl_name\":\"wwf\"}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"resu"
    "lt\",\"filter\",\"settings\"],\"type\":\"object\",\"properties\":{\"filter\":{\"requ"
    "ired\":[\"angle\",\"range_min\",\"range_max\",\"amplitude_max\"],\"type\":\"object"
    "\",\"properties\":{\"range_max\":{\"items\":{\"type\":\"number\"},\"description\":\""
    "[m]\",\"type\":\"array\"},\"angle\":{\"items\":{\"type\":\"number\"},\"description\":"
    "\"[deg]\",\"type\":\"array\"},\"range_min\":{\"items\":{\"type\":\"number\"},\"descri"
    "ption\":\"[m]\",\"type\":\"array\"},\"amplitude_max\":{\"items\":{\"type\":\"number\""
    "},\"description\":\"[dB]\",\"type\":\"array\"}}},\"result\":{\"required\":[\"angle\""
    ",\"range_mean\",\"range_sigma\",\"amplitude_mean\",\"amplitude_sigma\",\"amplit"
    "ude_offset\"],\"type\":\"object\",\"properties\":{\"amplitude_offset\":{\"items\""
    ":{\"type\":\"number\"},\"description\":\"[dB]\",\"type\":\"array\"},\"angle\":{\"item"
    "s\":{\"type\":\"number\"},\"description\":\"[deg]\",\"type\":\"array\"},\"timestamp\""
    ":{\"items\":{\"type\":\"number\"},\"description\":\"[s]\",\"type\":\"array\"},\"range"
    "_sigma\":{\"items\":{\"type\":\"number\"},\"description\":\"[m]\",\"type\":\"array\"}"
    ",\"amplitude_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"[dB]\",\"typ"
    "e\":\"array\"},\"range_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"[m]"
    "\",\"type\":\"array\"},\"amplitude_sigma\":{\"items\":{\"type\":\"number\"},\"descri"
    "ption\":\"[dB]\",\"type\":\"array\"}}},\"settings\":{\"required\":[\"range\",\"ampli"
    "tude\"],\"type\":\"object\",\"properties\":{\"range\":{\"required\":[\"sigma_facto"
    "r\",\"additive_value\"],\"type\":\"object\",\"properties\":{\"sigma_factor\":{\"ty"
    "pe\":\"number\"},\"additive_value\":{\"type\":\"number\"}}},\"amplitude\":{\"requi"
    "red\":[\"sigma_factor\",\"additive_value\"],\"type\":\"object\",\"properties\":{\""
    "sigma_factor\":{\"type\":\"number\"},\"additive_value\":{\"type\":\"number\"}}}}}"
    "},\"title\":\"Window Analysis\",\"description\":\"Window analysis data "
    "estimated from scandata and resulting filter parameters\"}"
    ,
    /* example: */
    "{\"filter\":{\"range_max\":[0.424,0.425,0.426,0.427,0.428,0.428,0.429,0.43"
    ",0.431,0.431,0.432],\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,1"
    "5.7,15.8,15.9],\"range_min\":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.218,-"
    "0.219,-0.221,-0.223,-0.225,-0.227],\"amplitude_max\":[8.04,8.01,7.99,7.9"
    "6,7.93,7.9,7.88,7.85,7.83,7.8,7.78]},\"result\":{\"amplitude_offset\":[1.9"
    ",1.9],\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9]"
    ",\"timestamp\":[408.4441,411.4443],\"range_sigma\":[0.01869652,0.02151435,"
    "0.01747969,0.01918765,0.01945776,0.01934862,0.01955329,0.02225589,0.02"
    "229977,0.01899122,0.02009433],\"amplitude_mean\":[5.347396,5.263155,5.22"
    "4655,5.179926,5.097782,5.116479,5.051756,4.983473,5.007885,5.002441,4."
    "982],\"range_mean\":[0.1105621,0.1079564,0.1087088,0.1067261,0.1054582,0"
    ".1090412,0.102871,0.1019044,0.1051523,0.1058445,0.1031261],\"amplitude_"
    "sigma\":[0.4272844,0.4298479,0.4236816,0.4283583,0.4362353,0.4315141,0."
    "4373984,0.4472798,0.4346001,0.4345487,0.4540681]},\"settings\":{\"range\":"
    "{\"sigma_factor\":8,\"additive_value\":0.1},\"amplitude\":{\"sigma_factor\":4,"
    "\"additive_value\":1.0}}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ampl"
    "itude\",\"range\",\"slices\"],\"type\":\"object\",\"properties\":{\"range\":{\"requi"
    "red\":[\"minimum\",\"maximum\",\"entries\"],\"description\":\"Range axis of "
    "correction table\",\"type\":\"object\",\"properties\":{\"maximum\":{\"maximum\":2"
    ".0,\"minimum\":-2.0,\"description\":\"Maximum range in m\",\"type\":\"number\"},"
    "\"minimum\":{\"maximum\":2.0,\"minimum\":-2.0,\"description\":\"Minimum range "
    "in m\",\"type\":\"number\"},\"entries\":{\"minimum\":1,\"description\":\"Number of"
    " range entries\",\"type\":\"integer\"}}},\"slices\":{\"items\":{\"required\":[\"am"
    "plitude\",\"table\"],\"description\":\"Window echo correction parameter slic"
    "e\",\"type\":\"object\",\"properties\":{\"table\":{\"minItems\":1,\"items\":{\"minIt"
    "ems\":1,\"items\":{\"minItems\":3,\"items\":{\"type\":\"number\",\"description\":\"T"
    "able cell (item 0: amplitude in dB, 1: range in m, 2: "
    "flags)\"},\"description\":\"Table column (= range "
    "axis)\",\"type\":\"array\",\"maxItems\":3},\"description\":\"Table row (= "
    "amplitude axis)\",\"type\":\"array\"},\"description\":\"Correction table "
    "(dimension defined by the 'amplitude' and 'range' "
    "objects)\",\"type\":\"array\"},\"amplitude\":{\"description\":\"Window echo "
    "amplitude of slice in dB\",\"type\":\"number\"}}},\"type\":\"array\"},\"amplitud"
    "e\":{\"required\":[\"minimum\",\"maximum\",\"entries\"],\"description\":\"Amplitud"
    "e axis of correction table\",\"type\":\"object\",\"properties\":{\"maximum\":{\""
    "minimum\":0.0,\"description\":\"Maximum amplitude in "
    "dB\",\"type\":\"number\"},\"minimum\":{\"minimum\":0.0,\"description\":\"Minimum "
    "amplitude in "
    "dB\",\"type\":\"number\"},\"entries\":{\"minimum\":1,\"description\":\"Number of "
    "amplitude entries\",\"type\":\"integer\"}}}},\"title\":\"Window Echo "
    "Correction\",\"description\":\"Correction parameters for window glass "
    "echoes\"}"
    ,
    /* example: */
    "{\"range\":{\"maximum\":1.5060822940732335,\"minimum\":-1.5060822940732335,\""
    "entries\":128},\"slices\":[{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]"
    "],\"amplitude\":1.5},{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],\"am"
    "plitude\":2.0}],\"amplitude\":{\"maximum\":20,\"minimum\":2,\"entries\":128}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LAG_ANGLE_CORRECTION,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);

#endif // RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1
