{*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 *
 *******************************************************************************
 *}

unit riegl.rdb.default.metadata;

{$WRITEABLECONST ON} // see section "initialization"

interface

// Meta data items definition version
const
  RDB_META_DATA_ITEMS_VERSION = '1.1.56';
  RDB_META_DATA_ITEMS_DATE = '2025-07-01';

// Meta data item states
const
  RDB_STATUS_REQUIRED = 1; // mandatory item
  RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
type
  TRDBPointcloudMetaDataItemInfo = record
    Name        : System.String;
    Title       : System.String;
    Description : System.String;
    Status      : System.Byte;
    Schema      : System.String;
    Example     : System.String;
  end;
  PRDBPointcloudMetaDataItemInfo = ^TRDBPointcloudMetaDataItemInfo;

// Angular notch filter parameters for window glass echoes
const
  RDB_RIEGL_ANGULAR_NOTCH_FILTER : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.angular_notch_filter';
    Title       : 'Angular Notch Filter';
    Description : 'Angular notch filter parameters for window glass echoes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["angle","range_mean","amplitude_mean"],"description":"Ang'
      + 'ular notch filter parameters for window glass echoes","title":"Angular'
      + ' Notch Filter","$schema":"http://json-schema.org/draft-04/schema#","pr'
      + 'operties":{"angle":{"items":{"type":"number"},"description":"Angle [de'
      + 'g]","type":"array"},"range_mean":{"items":{"type":"number"},"descripti'
      + 'on":"Mean range [m]","type":"array"},"amplitude_mean":{"items":{"type"'
      + ':"number"},"description":"Mean amplitude '
      + '[dB]","type":"array"}},"type":"object"}'
    );

    Example : (''
      + '{"angle":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],"ran'
      + 'ge_mean":[0.094,0.094,0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,'
      + '0.0675,0.06475],"amplitude_mean":[3.780913,3.780913,3.480913,3.120913,'
      + '2.850913,2.720913,2.680913,2.610913,2.530913,2.570913,2.570913]}'
    );
  );

// Atmospheric parameters
const
  RDB_RIEGL_ATMOSPHERE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.atmosphere';
    Title       : 'Atmosphere';
    Description : 'Atmospheric parameters';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["temperature","pressure","rel_humidity","pressure_sl","am'
      + 'sl","group_velocity","attenuation","wavelength"],"description":"Atmosp'
      + 'heric parameters","title":"Atmospheric Parameters","$schema":"http://j'
      + 'son-schema.org/draft-04/schema#","properties":{"amsl":{"description":"'
      + 'Height above mean sea level (AMSL) '
      + '[m]","type":"number"},"pressure_sl":{"description":"Atmospheric '
      + 'pressure at sea level '
      + '[mbar]","type":"number"},"temperature":{"description":"Temperature '
      + 'along measurement path '
      + '[\u00b0C]","type":"number"},"wavelength":{"description":"Laser '
      + 'wavelength [nm]","type":"number"},"pressure":{"description":"Pressure '
      + 'along measurment path '
      + '[mbar]","type":"number"},"rel_humidity":{"description":"Relative '
      + 'humidity along measurement path '
      + '[%]","type":"number"},"attenuation":{"description":"Atmospheric '
      + 'attenuation '
      + '[1/km]","type":"number"},"group_velocity":{"description":"Group '
      + 'velocity of laser beam [m/s]","type":"number"}},"type":"object"}'
    );

    Example : (''
      + '{"amsl":0,"pressure_sl":970,"temperature":7,"wavelength":1550,"pressur'
      + 'e":970,"rel_humidity":63,"attenuation":0.028125,"group_velocity":29971'
      + '1000.0}'
    );
  );

// Laser beam geometry details
const
  RDB_RIEGL_BEAM_GEOMETRY : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.beam_geometry';
    Title       : 'Beam Geometry';
    Description : 'Laser beam geometry details';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["beam_exit_diameter","beam_divergence"],"description":"La'
      + 'ser beam geometry details","title":"Beam Geometry","$schema":"http://j'
      + 'son-schema.org/draft-04/schema#","properties":{"beam_exit_diameter":{"'
      + 'description":"Beam width at exit aperture [m]","minimum":0,"type":"num'
      + 'ber","exclusiveMinimum":false},"beam_divergence":{"description":"Beam '
      + 'divergence in far field [rad]","minimum":0,"type":"number","exclusiveM'
      + 'inimum":false}},"type":"object"}'
    );

    Example : (''
      + '{"beam_exit_diameter":0.0072,"beam_divergence":0.0003}'
    );
  );

// List of control object type definitions
const
  RDB_RIEGL_CONTROL_OBJECT_CATALOG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.control_object_catalog';
    Title       : 'Control Object Catalog';
    Description : 'List of control object type definitions';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"types":{"items":{"oneOf":[{"$ref":"#/definitions/recta'
      + 'ngle"},{"$ref":"#/definitions/checkerboard2x2"},{"$ref":"#/definitions'
      + '/chevron"},{"$ref":"#/definitions/circular_disk"},{"$ref":"#/definitio'
      + 'ns/cylinder"},{"$ref":"#/definitions/sphere"},{"$ref":"#/definitions/r'
      + 'ound_corner_cube_prism"}],"type":"object"},"uniqueItems":true,"type":"'
      + 'array"}},"required":["types"],"description":"List of control object '
      + 'type definitions","title":"Control Object Catalog","$schema":"http://j'
      + 'son-schema.org/draft-04/schema#","type":"object","definitions":{"circu'
      + 'lar_disk":{"description":"circular disk","type":"object","allOf":[{"$r'
      + 'ef":"#/definitions/common"},{"description":"circular disk specific pro'
      + 'perties","type":"object","required":["shape","diameter"],"properties":'
      + '{"offset":{"description":"offset in meters, e.g. reflector constant '
      + '(optional)","type":"number"},"diameter":{"description":"diameter in me'
      + 'ters","minimum":0,"type":"number","exclusiveMinimum":true},"shape":{"d'
      + 'escription":"shape identifier","enum":["circular_disk"],"type":"string'
      + '"}}}]},"checkerboard2x2":{"description":"checkerboard 2 by 2","type":"'
      + 'object","allOf":[{"$ref":"#/definitions/common"},{"description":"check'
      + 'erboard specific properties","type":"object","required":["shape","squa'
      + 're_length"],"properties":{"square_length":{"description":"length of a '
      + 'square of the checkerboard in meters","minimum":0,"type":"number","exc'
      + 'lusiveMinimum":true},"shape":{"description":"shape identifier","enum":'
      + '["checkerboard2x2"],"type":"string"}}}]},"cylinder":{"description":"cy'
      + 'linder","type":"object","allOf":[{"$ref":"#/definitions/common"},{"des'
      + 'cription":"cylinder specific properties","type":"object","required":["'
      + 'shape","diameter","height"],"properties":{"height":{"description":"hei'
      + 'ght in meters","minimum":0,"type":"number","exclusiveMinimum":true},"d'
      + 'iameter":{"description":"diameter in meters","minimum":0,"type":"numbe'
      + 'r","exclusiveMinimum":true},"shape":{"description":"shape identifier",'
      + '"enum":["cylinder"],"type":"string"}}}]},"common":{"description":"comm'
      + 'on object properties","type":"object","required":["name","shape"],"pro'
      + 'perties":{"description":{"description":"string describing the '
      + 'object","type":"string"},"surface_type":{"description":"surface '
      + 'material type","enum":["retro_reflective_foil","diffuse"],"type":"stri'
      + 'ng"},"name":{"description":"unique type identifier","type":"string","m'
      + 'inLength":3},"shape":{"description":"shape identifier","enum":["rectan'
      + 'gle","checkerboard2x2","chevron","circular_disk","cylinder","sphere","'
      + 'round_corner_cube_prism"],"type":"string"}}},"sphere":{"description":"'
      + 'sphere","type":"object","allOf":[{"$ref":"#/definitions/common"},{"des'
      + 'cription":"sphere specific properties","type":"object","required":["sh'
      + 'ape","diameter"],"properties":{"diameter":{"description":"diameter in '
      + 'meters","minimum":0,"type":"number","exclusiveMinimum":true},"shape":{'
      + '"description":"shape identifier","enum":["sphere"],"type":"string"}}}]'
      + '},"round_corner_cube_prism":{"description":"round corner cube prism","'
      + 'type":"object","allOf":[{"$ref":"#/definitions/common"},{"description"'
      + ':"round corner cube prism specific properties","type":"object","requir'
      + 'ed":["shape","diameter"],"properties":{"offset":{"description":"offset'
      + ' in meters, e.g. reflector constant '
      + '(optional)","type":"number"},"diameter":{"description":"diameter in me'
      + 'ters","minimum":0,"type":"number","exclusiveMinimum":true},"shape":{"d'
      + 'escription":"shape identifier","enum":["round_corner_cube_prism"],"typ'
      + 'e":"string"}}}]},"rectangle":{"description":"rectangle","type":"object'
      + '","allOf":[{"$ref":"#/definitions/common"},{"description":"rectangle '
      + 'specific properties","type":"object","required":["shape","length","wid'
      + 'th"],"properties":{"length":{"description":"length in meters","minimum'
      + '":0,"type":"number","exclusiveMinimum":true},"width":{"description":"w'
      + 'idth in meters","minimum":0,"type":"number","exclusiveMinimum":true},"'
      + 'shape":{"enum":["rectangle"],"type":"string"}}}]},"chevron":{"descript'
      + 'ion":"chevron","type":"object","allOf":[{"$ref":"#/definitions/common"'
      + '},{"description":"chevron specific properties","type":"object","requir'
      + 'ed":["shape","outside_edge_length","thickness"],"properties":{"outside'
      + '_edge_length":{"description":"length of the two outer edges in meters"'
      + ',"minimum":0,"type":"number","exclusiveMinimum":true},"thickness":{"de'
      + 'scription":"thickness in meters","minimum":0,"type":"number","exclusiv'
      + 'eMinimum":true},"shape":{"description":"shape '
      + 'identifier","enum":["chevron"],"type":"string"}}}]}}}'
    );

    Example : (''
      + '{"comments":["This file contains a list of control object types (aka. '
      + '''catalog'').","Each type is described by an object,","which must '
      + 'contain at least the following parameters:","  - name: unique '
      + 'identifier of the type","  - shape: one of the following supported '
      + 'shapes:","      - rectangle","      - checkerboard2x2","      - '
      + 'chevron","      - circular_disk","      - cylinder","      - sphere","'
      + '      - round_corner_cube_prism","Depending on ''shape'', the following '
      + 'parameters must/may be specified:","  - rectangle:","      - length: '
      + 'length in meters","      - width: width in meters","  - '
      + 'checkerboard2x2:","      - square_length: length of a square of the '
      + 'checkerboard in meters","  - circular_disk:","      - diameter: '
      + 'diameter in meters","      - offset: offset in meters, e.g. reflector '
      + 'constant (optional)","  - chevron:","      - outside_edge_length: '
      + 'length of the two outer edges in meters","      - thickness: thickness'
      + ' in meters","  - cylinder:","      - diameter: diameter in meters","'
      + '      - height: height in meters","  - sphere:","      - diameter: '
      + 'diameter in meters","  - round_corner_cube_prism:","      - diameter: '
      + 'diameter in meters","      - offset: offset in meters, e.g. reflector '
      + 'constant (optional)","Optional parameters:","    - description: string'
      + ' describing the object","    - surface_type: surface material type '
      + '(either ''retro_reflective_foil'' or '
      + '''diffuse'')"],"types":[{"description":"Rectangle (60cm x '
      + '30cm)","length":0.6,"width":0.3,"name":"Rectangle '
      + '60x30","shape":"rectangle"},{"description":"Rectangle (80cm x '
      + '40cm)","length":0.8,"width":0.4,"name":"Rectangle '
      + '80x40","shape":"rectangle"},{"description":"Checkerboard (square '
      + 'length: 30cm)","square_length":0.3,"name":"Checkerboard2x2 '
      + '30","shape":"checkerboard2x2"},{"description":"Checkerboard (square '
      + 'length: 50cm)","square_length":0.5,"name":"Checkerboard2x2 50","shape"'
      + ':"checkerboard2x2"},{"thickness":0.1016,"description":"Chevron '
      + '(a=24''''; b=4'''')","outside_edge_length":0.6096,"name":"Chevron '
      + '24''''/4''''","shape":"chevron"},{"description":" Circular Disk (diameter:'
      + ' 50cm)","surface_type":"diffuse","diameter":0.5,"name":"Circular Disk '
      + '50","shape":"circular_disk"},{"diameter":0.05,"name":"RIEGL flat '
      + 'reflector 50 mm","offset":0.0,"description":"flat circular reflector '
      + 'from retro-reflective foil","surface_type":"retro_reflective_foil","sh'
      + 'ape":"circular_disk"},{"diameter":0.1,"name":"RIEGL flat reflector 100'
      + ' mm","offset":0.0,"description":"flat circular reflector from '
      + 'retro-reflective foil","surface_type":"retro_reflective_foil","shape":'
      + '"circular_disk"},{"diameter":0.15,"name":"RIEGL flat reflector 150 '
      + 'mm","offset":0.0,"description":"flat circular reflector from '
      + 'retro-reflective foil","surface_type":"retro_reflective_foil","shape":'
      + '"circular_disk"},{"diameter":0.05,"name":"RIEGL cylindrical reflector '
      + '50 mm","height":0.05,"description":"cylindrical reflector from '
      + 'retro-reflective foil","surface_type":"retro_reflective_foil","shape":'
      + '"cylinder"},{"diameter":0.1,"name":"RIEGL cylindrical reflector 100 '
      + 'mm","height":0.1,"description":"cylindrical reflector from '
      + 'retro-reflective foil","surface_type":"retro_reflective_foil","shape":'
      + '"cylinder"},{"description":"Sphere (diameter: 200 '
      + 'mm)","diameter":0.2,"name":"Sphere 200 '
      + 'mm","shape":"sphere"},{"offset":0.0,"description":"round corner cube '
      + 'prism","diameter":0.05,"name":"Corner Cube Prism 50 '
      + 'mm","shape":"round_corner_cube_prism"}]}'
    );
  );

// Details about the control object reference file
const
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.control_object_reference_file';
    Title       : 'Control Object Reference file';
    Description : 'Details about the control object reference file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Details about the control object reference '
      + 'file","properties":{"reference_file":{"description":"Reference to a '
      + 'control object file","type":"object","required":["file_uuid","file_pat'
      + 'h"],"properties":{"file_path":{"description":"Path of the control '
      + 'object file relative to referring '
      + 'file","type":"string"},"file_uuid":{"description":"Control object '
      + 'file''s Universally Unique Identifier (RFC 4122)","type":"string"}}}},"'
      + '$schema":"http://json-schema.org/draft-04/schema#","type":"object","ti'
      + 'tle":"Control Object Reference file"}'
    );

    Example : (''
      + '{"reference_file":{"file_path":"../../../10_CONTROL_OBJECTS/ControlPoi'
      + 'nts.cpx","file_uuid":"810f5d57-eccf-49ed-b07a-0cdd109b4213"}}'
    );
  );

// Detection probability as a function of amplitude
const
  RDB_RIEGL_DETECTION_PROBABILITY : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.detection_probability';
    Title       : 'Detection Probability';
    Description : 'Detection probability as a function of amplitude';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["amplitude","detection_probability"],"description":"Detec'
      + 'tion probability as a function of amplitude","title":"Detection Probab'
      + 'ility","$schema":"http://json-schema.org/draft-04/schema#","properties'
      + '":{"amplitude":{"items":{"type":"number"},"description":"Amplitude [dB'
      + ']","type":"array"},"detection_probability":{"items":{"type":"number"},'
      + '"description":"Detection probability '
      + '[0..1]","type":"array"}},"type":"object"}'
    );

    Example : (''
      + '{"amplitude":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],"detection'
      + '_probability":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]}'
    );
  );

// Details about the device used to acquire the point cloud
const
  RDB_RIEGL_DEVICE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device';
    Title       : 'Device Information';
    Description : 'Details about the device used to acquire the point cloud';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["device_type","serial_number"],"description":"Details '
      + 'about the device used to acquire the point cloud","title":"Device Info'
      + 'rmation","$schema":"http://json-schema.org/draft-04/schema#","properti'
      + 'es":{"device_type":{"description":"Device type identifier (e.g. '
      + 'VZ-400i)","type":"string"},"channel_text":{"description":"Optional '
      + 'channel description (e.g. ''Green Channel'' for multi-channel '
      + 'devices)","type":"string"},"serial_number":{"description":"Device '
      + 'serial number (e.g. '
      + 'S2221234)","type":"string"},"device_name":{"description":"Optional '
      + 'device name (e.g. ''Scanner 1'' for multi-scanner '
      + 'systems)","type":"string"},"device_build":{"description":"Device build'
      + ' variant","type":"string"},"channel_number":{"description":"Laser '
      + 'channel number (not defined or 0: single channel device)","minimum":0,'
      + '"type":"integer","exclusiveMinimum":false}},"type":"object"}'
    );

    Example : (''
      + '{"device_type":"VZ-400i","channel_text":"","serial_number":"S2221234",'
      + '"device_name":"Scanner 1","device_build":"","channel_number":0}'
    );
  );

// Scanner device geometry details
const
  RDB_RIEGL_DEVICE_GEOMETRY : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device_geometry';
    Title       : 'Device Geometry';
    Description : 'Scanner device geometry details';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["primary"],"description":"Scanner device geometry '
      + 'details","title":"Device Geometry","$schema":"http://json-schema.org/d'
      + 'raft-04/schema#","properties":{"amu":{"description":"Angle Measurement'
      + ' Unit","type":"object","properties":{"frameCC":{"description":"Frame '
      + 'Circle Count (number of LSBs per full rotation about frame axis)","min'
      + 'imum":0,"type":"number","exclusiveMinimum":false},"lineCC":{"descripti'
      + 'on":"Line Circle Count (number of LSBs per full rotation about line ax'
      + 'is)","minimum":0,"type":"number","exclusiveMinimum":false}}},"secondar'
      + 'y":{"description":"Additional device geometry structure (optional)","t'
      + 'ype":"object","required":["ID","content"],"properties":{"ID":{"items":'
      + '{"type":"integer"},"description":"Structure identifier","type":"array"'
      + ',"minItems":2,"maxItems":2},"content":{"items":{"type":"number"},"desc'
      + 'ription":"Internal calibration '
      + 'values","type":"array"}}},"primary":{"description":"Primary device '
      + 'geometry structure (mandatory)","type":"object","required":["ID","cont'
      + 'ent"],"properties":{"ID":{"items":{"type":"integer"},"description":"St'
      + 'ructure identifier","type":"array","minItems":2,"maxItems":2},"content'
      + '":{"items":{"type":"number"},"description":"Internal calibration '
      + 'values","type":"array"}}}},"type":"object"}'
    );

    Example : (''
      + '{"amu":{"frameCC":124000,"lineCC":124000},"secondary":{"ID":[91,0],"co'
      + 'ntent":[0]},"primary":{"ID":[4,0],"content":[0]}}'
    );
  );

// Scanner passive channel device geometry details
const
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device_geometry_passive_channel';
    Title       : 'Device Geometry Passive Channel';
    Description : 'Scanner passive channel device geometry details';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["primary"],"description":"Scanner passive channel device '
      + 'geometry details","title":"Device Geometry Passive Channel","$schema":'
      + '"http://json-schema.org/draft-04/schema#","properties":{"primary":{"de'
      + 'scription":"Primary device geometry structure (mandatory)","type":"obj'
      + 'ect","required":["ID","content"],"properties":{"ID":{"items":{"type":"'
      + 'integer"},"description":"Structure identifier","type":"array","minItem'
      + 's":2,"maxItems":2},"content":{"items":{"type":"number"},"description":'
      + '"Internal calibration values","type":"array"}}}},"type":"object"}'
    );

    Example : (''
      + '{"primary":{"ID":[143,0],"content":[0]}}'
    );
  );

// Limits of the measured values output by the device
const
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device_output_limits';
    Title       : 'Device Output Limits';
    Description : 'Limits of the measured values output by the device';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Limits of the measured values output by the device. '
      + 'The limits depend on the device type, measurement program and/or scan '
      + 'pattern.","properties":{"range_maximum":{"description":"Maximum '
      + 'possible range in '
      + 'meters.","type":"number"},"amplitude_maximum":{"description":"Maximum '
      + 'possible amplitude in dB.","type":"number"},"background_radiation_maxi'
      + 'mum":{"description":"Maximum possible background radiation.","type":"n'
      + 'umber"},"background_radiation_minimum":{"description":"Minimum '
      + 'possible background radiation.","type":"number"},"deviation_minimum":{'
      + '"description":"Minimum possible pulse shape deviation.","type":"number'
      + '"},"mta_zone_count_maximum":{"description":"Maximum number of MTA '
      + 'zones.","type":"number"},"echo_count_maximum":{"description":"Maximum '
      + 'number of echoes a laser shot can '
      + 'have.","type":"number"},"reflectance_maximum":{"description":"Maximum '
      + 'possible reflectance in '
      + 'dB.","type":"number"},"deviation_maximum":{"description":"Maximum '
      + 'possible pulse shape deviation.","type":"number"},"amplitude_minimum":'
      + '{"description":"Minimum possible amplitude in '
      + 'dB.","type":"number"},"reflectance_minimum":{"description":"Minimum '
      + 'possible reflectance in '
      + 'dB.","type":"number"},"range_minimum":{"description":"Minimum possible'
      + ' range in meters.","type":"number"}},"$schema":"http://json-schema.org'
      + '/draft-04/schema#","type":"object","title":"Device Output Limits"}'
    );

    Example : (''
      + '{"range_maximum":10000.0,"amplitude_maximum":100.0,"mta_zone_count_max'
      + 'imum":7,"reflectance_maximum":100.0,"deviation_maximum":32767,"deviati'
      + 'on_minimum":-1,"reflectance_minimum":-100.0,"background_radiation_mini'
      + 'mum":0,"background_radiation_maximum":0,"amplitude_minimum":0.0,"range'
      + '_minimum":2.9}'
    );
  );

// Details about the devices used to acquire the point cloud
const
  RDB_RIEGL_DEVICES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.devices';
    Title       : 'Devices Information';
    Description : 'Details about the devices used to acquire the point cloud';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"items":{"type":"object","required":["device_type","serial_number"],"'
      + 'properties":{"device_type":{"description":"Device type identifier '
      + '(e.g. VZ-400i)","type":"string"},"signed":{"description":"Flag that is'
      + ' set when the original ''riegl.device'' entry in the source file was '
      + 'correctly '
      + 'signed.","type":"boolean"},"channel_text":{"description":"Optional '
      + 'channel description (e.g. ''Green Channel'' for multi-channel '
      + 'devices)","type":"string"},"serial_number":{"description":"Device '
      + 'serial number (e.g. '
      + 'S2221234)","type":"string"},"device_name":{"description":"Optional '
      + 'device name (e.g. ''Scanner 1'' for multi-scanner '
      + 'systems)","type":"string"},"device_build":{"description":"Device build'
      + ' variant","type":"string"},"channel_number":{"description":"Laser '
      + 'channel number (not defined or 0: single channel device)","minimum":0,'
      + '"type":"integer","exclusiveMinimum":false}}},"description":"Details '
      + 'about the devices used to acquire the point cloud (e.g. when merging '
      + 'point clouds of different devices)","$schema":"http://json-schema.org/'
      + 'draft-04/schema#","type":"array","title":"Devices Information"}'
    );

    Example : (''
      + '[{"device_type":"VZ-400i","signed":false,"channel_text":"","serial_num'
      + 'ber":"S2221234","device_name":"Scanner 1","device_build":"","channel_n'
      + 'umber":0},{"device_type":"VQ-1560i-DW","signed":true,"channel_text":""'
      + ',"serial_number":"S2222680","device_name":"Scanner 2","device_build":"'
      + '","channel_number":1},{"device_type":"VQ-1560i-DW","signed":true,"chan'
      + 'nel_text":"","serial_number":"S2222680","device_name":"Scanner '
      + '3","device_build":"","channel_number":2}]'
    );
  );

// Distance measuring instrument configuration
const
  RDB_RIEGL_DMI_CONFIG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.dmi_config';
    Title       : 'DMI Configuration';
    Description : 'Distance measuring instrument configuration';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["dmi_count_dir","dmi_incr","dmi_min_time","dmi_max_time",'
      + '"dmi_dist_per_tick"],"description":"Distance measuring instrument '
      + 'configuration","title":"DMI Configuration","$schema":"http://json-sche'
      + 'ma.org/draft-04/schema#","properties":{"dmi_max_time":{"exclusiveMaxim'
      + 'um":false,"description":"Maximum time interval between trigger events,'
      + ' 0 means timeout is inactive [0.1 msec]","maximum":65535,"minimum":0,"'
      + 'type":"integer","exclusiveMinimum":false},"dmi_min_time":{"exclusiveMa'
      + 'ximum":false,"description":"Minimum time interval between trigger '
      + 'events, 0 means rate limitation is inactive [0.1 msec]","maximum":6553'
      + '5,"minimum":0,"type":"integer","exclusiveMinimum":false},"dmi_incr":{"'
      + 'exclusiveMaximum":false,"description":"Increment in ticks, 0 means '
      + 'distance trigger is deactivated","maximum":2147483647,"minimum":0,"typ'
      + 'e":"integer","exclusiveMinimum":false},"dmi_dist_per_tick":{"descripti'
      + 'on":"Distance per tick, just informative [m]","type":"number"},"dmi_co'
      + 'unt_dir":{"exclusiveMaximum":false,"description":"DMI count direction '
      + 'is inverted","maximum":1,"minimum":0,"type":"integer","exclusiveMinimu'
      + 'm":false}},"type":"object"}'
    );

    Example : (''
      + '{"dmi_max_time":100,"dmi_min_time":0,"dmi_incr":0,"dmi_dist_per_tick":'
      + '0.00024399999529123306,"dmi_count_dir":0}'
    );
  );

// Dyntrig
const
  RDB_RIEGL_DYNTRIG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.dyntrig';
    Title       : 'Dyntrig';
    Description : 'Dyntrig';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Dyntrig","$schema":"http://json-schema.org/draft-04/sc'
      + 'hema#","properties":{"offset":{"description":"Offset in units of dB, '
      + 'for calculation of line","type":"number"},"tab":{"items":{"description'
      + '":"...","type":"object","required":["thra1","del1","dec1","thra2","del'
      + '2","dec2"],"properties":{"thra1":{"description":"Post-trigger '
      + 'threshold in units of '
      + 'dB","type":"number"},"del1":{"description":"Length of constant '
      + 'post-trigger threshold elevation in units of '
      + 'm","type":"number"},"thra2":{"description":"Pre-trigger threshold in '
      + 'units of dB","type":"number"},"del2":{"description":"Length of '
      + 'constant pre-trigger threshold elevation in units of '
      + 'm","type":"number"},"dec1":{"description":"Decay of post-trigger '
      + 'threshold after del1 in units of '
      + 'dB/m","type":"number"},"dec2":{"description":"Rise of pre-trigger '
      + 'threshold after del2 in units of dB/m","type":"number"}}},"description'
      + '":"...","type":"array","minItems":1,"maxItems":128},"scale":{"descript'
      + 'ion":"Scale in units of dB, for calculation of line: line number N=(am'
      + 'plitude-offset)/scale","type":"number"},"post":{"description":"Dyntrig'
      + ' post values","type":"object","required":["nc","a"],"properties":{"a":'
      + '{"items":{"type":"number"},"description":"...","type":"array","minItem'
      + 's":1,"maxItems":2080},"nc":{"description":"Number of entries","maximum'
      + '":255,"minimum":0,"type":"integer"}}},"pre":{"description":"Dyntrig '
      + 'pre values","type":"object","required":["nc","a"],"properties":{"a":{"'
      + 'items":{"type":"number"},"description":"...","type":"array","minItems"'
      + ':1,"maxItems":2080},"nc":{"description":"Number of entries","maximum":'
      + '255,"minimum":0,"type":"integer"}}}},"title":"Dyntrig"}'
    );

    Example : (''
      + '{"offset":0.0,"tab":[{"thra1":0.0,"del1":0.0,"thra2":0.0,"del2":0.0,"d'
      + 'ec1":0.0,"dec2":0.0}],"scale":1.0,"post":{"a":[78,86,126,134,31],"nc":'
      + '128},"pre":{"a":[78,86,126,134,31],"nc":128}}'
    );
  );

// Details about echo files
const
  RDB_RIEGL_ECHO_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.echo_info';
    Title       : 'Echo Information';
    Description : 'Details about echo files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["echo_file"],"description":"Details about echo '
      + 'files","title":"Echo Information","$schema":"http://json-schema.org/dr'
      + 'aft-04/schema#","properties":{"echo_file":{"type":"object","required":'
      + '["file_extension"],"properties":{"file_extension":{"description":"Echo'
      + ' file extension, without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC 4122)","type":"string"}}}},"type":"object"}'
    );

    Example : (''
      + '{"echo_file":{"file_extension":"owp","file_uuid":"26a03615-67c0-4bea-8'
      + 'fe8-c577378fe661"}}'
    );
  );

// Details for exponential decomposition of full waveform data
const
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.exponential_decomposition';
    Title       : 'Exponential Decomposition';
    Description : 'Details for exponential decomposition of full waveform data';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"additionalProperties":false,"patternProperties":{"^[0-9]+$":{"$ref":'
      + '"#/definitions/channel","description":"one field per channel, field '
      + 'name is channel index"}},"description":"Details for exponential '
      + 'decomposition of full waveform data","title":"Exponential Decompositio'
      + 'n","$schema":"http://json-schema.org/draft-04/schema#","type":"object"'
      + ',"definitions":{"channel":{"type":"object","required":["delay","scale"'
      + ',"parameter"],"properties":{"parameter":{"description":"parameters of '
      + 'the syswave exponential sum","type":"object","required":["A","B","gamm'
      + 'a","omega"],"properties":{"omega":{"items":{"type":"number"},"descript'
      + 'ion":"angular frequency in Hz","type":"array"},"gamma":{"items":{"type'
      + '":"number"},"description":"decay in 1/second","type":"array"},"B":{"it'
      + 'ems":{"type":"number"},"description":"imaginary part of amplitude '
      + 'factor in units of full-scale","type":"array"},"A":{"items":{"type":"n'
      + 'umber"},"description":"real part of amplitude factor in units of '
      + 'full-scale","type":"array"}}},"scale":{"description":"amplitude '
      + 'calibration","type":"number"},"delay":{"description":"delay '
      + 'calibration in seconds","type":"number"},"a_lin":{"exclusiveMaximum":f'
      + 'alse,"description":"relative linear amplitude range [0..1]","maximum":'
      + '1,"minimum":0,"type":"number","exclusiveMinimum":false}}}}}'
    );

    Example : (''
      + '{"0":{"parameter":{"omega":[352020896.0,3647927552.0,-1977987072.0],"g'
      + 'amma":[-1094726528.0,-769562752.0,-848000064.0],"B":[-3.98130321502685'
      + '55,0.08622030913829803,-0.3152860999107361],"A":[0.9772450923919678,0.'
      + '3354335129261017,-1.312678575515747]},"scale":1.0,"delay":3.7834584188'
      + '87631e-09,"a_lin":0.27},"1":{"parameter":{"omega":[352020896.0,3647927'
      + '552.0,-1977987072.0],"gamma":[-1094726528.0,-769562752.0,-848000064.0]'
      + ',"B":[-3.9,0.0,-0.3],"A":[0.9,0.3,-1.3]},"scale":1.0,"delay":3.5e-09,"'
      + 'a_lin":0.9}}'
    );
  );

// Details for Gaussian decomposition of full waveform data
const
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.gaussian_decomposition';
    Title       : 'Gaussian Decomposition';
    Description : 'Details for Gaussian decomposition of full waveform data';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["amplitude_lsb_low_power","amplitude_lsb_high_power","amp'
      + 'litude_db","range_offset_sec_low_power","range_offset_sec_high_power"]'
      + ',"description":"riegl.gaussian_decomposition contains information '
      + 'relevant for extracting calibrated amplitudes and calibrated ranges '
      + 'from a Gaussian decomposition of full waveform data. This information '
      + 'is contained in a table with five columns. Two columns are to be used '
      + 'as input: amplitude_lsb_low_power and amplitude_lsb_high_power. The '
      + 'other three columns provide the outputs. Amplitude_db gives the '
      + 'calibrated amplitude in the optical regime in decibels. The range '
      + 'offset columns provide additive range offsets, given in units of '
      + 'seconds, for each channel.","title":"Gaussian Decomposition","$schema"'
      + ':"http://json-schema.org/draft-04/schema#","properties":{"range_offset'
      + '_sec_low_power":{"items":{"type":"number"},"type":"array"},"amplitude_'
      + 'lsb_high_power":{"items":{"type":"number"},"type":"array"},"range_offs'
      + 'et_sec_high_power":{"items":{"type":"number"},"type":"array"},"amplitu'
      + 'de_db":{"items":{"type":"number"},"type":"array"},"amplitude_lsb_low_p'
      + 'ower":{"items":{"type":"number"},"type":"array"}},"type":"object"}'
    );

    Example : (''
      + '{"range_offset_sec_low_power":[],"amplitude_lsb_high_power":[],"range_'
      + 'offset_sec_high_power":[],"amplitude_db":[],"amplitude_lsb_low_power":'
      + '[]}'
    );
  );

// Point cloud georeferencing information
const
  RDB_RIEGL_GEO_TAG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.geo_tag';
    Title       : 'Geo Tag';
    Description : 'Point cloud georeferencing information';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Point cloud georeferencing information","properties":{'
      + '"pose":{"items":{"items":{"description":"columns","type":"number"},"de'
      + 'scription":"rows","minItems":4,"type":"array","maxItems":4},"descripti'
      + 'on":"Coordinate Transformation Matrix to transform from File '
      + 'Coordinate System to Global Coordinate Reference System. 4x4 matrix '
      + 'stored as two dimensional array, row major order.","minItems":4,"type"'
      + ':"array","maxItems":4},"crs":{"description":"Global Coordinate '
      + 'Reference System. Please note that only 3D Cartesian Coordinate '
      + 'Systems are '
      + 'allowed.","type":"object","properties":{"epsg":{"description":"EPSG '
      + 'code","minimum":0,"type":"integer"},"wkt":{"description":"\"Well-Known'
      + ' Text\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta'
      + 'ndards/wkt-crs)","type":"string"},"name":{"description":"Coordinate '
      + 'reference system name","type":"string"}}}},"$schema":"http://json-sche'
      + 'ma.org/draft-04/schema#","type":"object","title":"Geo Tag"}'
    );

    Example : (''
      + '{"pose":[[-0.269827776749716,-0.723017716139738,0.635954678449952,4063'
      + '882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847,113'
      + '8787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196],[0'
      + '.0,0.0,0.0,1.0]],"crs":{"epsg":4978,"wkt":"GEOCCS[\"WGS84 Geocentric\"'
      + ',DATUM[\"WGS84\",SPHEROID[\"WGS84\",6378137.000,298.257223563,AUTHORIT'
      + 'Y[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]],PRIMEM[\"Greenwich'
      + '\",0.0000000000000000,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"Meter\",1.0'
      + '0000000000000000000,AUTHORITY[\"EPSG\",\"9001\"]],AXIS[\"X\",OTHER],AX'
      + 'IS[\"Y\",EAST],AXIS[\"Z\",NORTH],AUTHORITY[\"EPSG\",\"4978\"]]","name"'
      + ':"WGS84 Geocentric"}}'
    );
  );

// Geometric scale factor applied to point coordinates
const
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.geometric_scale_factor';
    Title       : 'Geometric Scale Factor';
    Description : 'Geometric scale factor applied to point coordinates';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"minimum":0,"description":"Geometric scale factor applied to point co'
      + 'ordinates","$schema":"http://json-schema.org/draft-04/schema#","type":'
      + '"number","exclusiveMinimum":true}'
    );

    Example : (''
      + '1.0'
    );
  );

// Parameters used for georeferencing of the point cloud
const
  RDB_RIEGL_GEOREFERENCING_PARAMETERS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.georeferencing_parameters';
    Title       : 'Georeferencing Parameters';
    Description : 'Parameters used for georeferencing of the point cloud';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Parameters used for georeferencing of the point cloud"'
      + ',"properties":{"socs_to_body_matrix":{"items":{"items":{"description":'
      + '"columns","type":"number"},"description":"rows","minItems":4,"type":"a'
      + 'rray","maxItems":4},"description":"Coordinate Transformation Matrix to'
      + ' transform from Scanner''s Own Coordinate System to Body Coordinate '
      + 'System. 4x4 matrix stored as two dimensional array, row major order.",'
      + '"minItems":4,"type":"array","maxItems":4},"body_coordinate_system_type'
      + '":{"description":"BODY coordinate frame (NED: North-East-Down, ENU: '
      + 'East-North-Up), default: NED","enum":["NED","ENU"],"type":"string"},"t'
      + 'rajectory_offsets":{"description":"Correction offsets applied to the '
      + 'trajectory data","type":"object","properties":{"offset_height":{"descr'
      + 'iption":"[m]","type":"number"},"offset_time":{"description":"[s]","typ'
      + 'e":"number"},"offset_east":{"description":"[m]","type":"number"},"offs'
      + 'et_pitch":{"description":"[deg]","type":"number"},"version":{"descript'
      + 'ion":"Meaning of offset values and how to apply them; version 0: '
      + 'Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), '
      + 'version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch'
      + ')*Rx(offset_roll)","type":"integer"},"offset_roll":{"description":"[de'
      + 'g]","type":"number"},"offset_yaw":{"description":"[deg]","type":"numbe'
      + 'r"},"offset_north":{"description":"[m]","type":"number"}}},"trajectory'
      + '_file":{"description":"Trajectory data used for georeferencing of the '
      + 'point cloud","type":"object","required":["file_extension"],"properties'
      + '":{"file_extension":{"description":"Trajectory file extension, without'
      + ' the leading dot","type":"string"},"file_uuid":{"description":"File''s '
      + 'Universally Unique Identifier (RFC 4122)","type":"string"}}},"socs_to_'
      + 'rocs_matrix":{"items":{"items":{"description":"columns","type":"number'
      + '"},"description":"rows","minItems":4,"type":"array","maxItems":4},"des'
      + 'cription":"Coordinate Transformation Matrix to transform from '
      + 'Scanner''s Own Coordinate System to Record Coordinate System. 4x4 '
      + 'matrix stored as two dimensional array, row major order.","minItems":4'
      + ',"type":"array","maxItems":4}},"$schema":"http://json-schema.org/draft'
      + '-04/schema#","type":"object","title":"Georeferencing Parameters"}'
    );

    Example : (''
      + '{"socs_to_body_matrix":[[-0.269827776749716,-0.723017716139738,0.63595'
      + '4678449952,0.0],[0.962908599449764,-0.20260517250352,0.178208229833847'
      + ',0.0],[0.0,0.660451759194338,0.7508684796801,0.0],[0.0,0.0,0.0,1.0]],"'
      + 'body_coordinate_system_type":"NED","trajectory_offsets":{"offset_heigh'
      + 't":-0.2,"offset_time":18.007,"offset_east":0.15,"offset_pitch":0.01,"v'
      + 'ersion":0,"offset_roll":0.03,"offset_yaw":-0.45,"offset_north":0.07},"'
      + 'trajectory_file":{"file_extension":"pofx","file_uuid":"93a03615-66c0-4'
      + 'bea-8ff8-c577378fe660"}}'
    );
  );

// Details about GNSS processing
const
  RDB_RIEGL_GNSS_PROCESSING_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.gnss_processing_info';
    Title       : 'GNSS Processing Information';
    Description : 'Details about GNSS processing';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Details about GNSS processing","properties":{"base_sta'
      + 'tions":{"items":{"description":"GNSS base station details","type":"obj'
      + 'ect","required":["latitude","longitude","altitude"],"properties":{"alt'
      + 'itude":{"description":"Altitude '
      + '[m]","type":"number"},"longitude":{"description":"Longitude '
      + '[deg]","type":"number"},"latitude":{"description":"Latitude '
      + '[deg]","type":"number"}}},"description":"List of GNSS base stations","'
      + 'type":"array"}},"$schema":"http://json-schema.org/draft-04/schema#","t'
      + 'ype":"object","title":"GNSS Processing Information"}'
    );

    Example : (''
      + '{"base_stations":[{"altitude":379.8004150390625,"longitude":15.6633898'
      + '22,"latitude":48.658391162}]}'
    );
  );

// Map of item names
const
  RDB_RIEGL_ITEM_NAMES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.item_names';
    Title       : 'Item Names';
    Description : 'Map of item names';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"additionalProperties":false,"patternProperties":{"^-?[0-9]+$":{"desc'
      + 'ription":"One field per item, field name is item id, field value is '
      + 'item name","type":"string"}},"description":"Map of item names","$schem'
      + 'a":"http://json-schema.org/draft-04/schema#","type":"object","title":"'
      + 'Item Names"}'
    );

    Example : (''
      + '{"2":"Name of item with id 2","-1":"Name of item with id '
      + '-1","47":"Name of item with id 47","1":"Name of item with id 1"}'
    );
  );

// License keys for software features
const
  RDB_RIEGL_LICENSES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.licenses';
    Title       : 'Software License Keys';
    Description : 'License keys for software features';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"additionalProperties":false,"patternProperties":{"^.*$":{"items":{"d'
      + 'escription":"License key (example: ''46AE032A - 39882AC4 - 9EC0A184 - '
      + '6F163D73'')","type":"string"},"description":"Each field of the object '
      + 'represents a feature and holds a list of license keys, where the field'
      + ' name is the feature '
      + 'name.","type":"array","minItems":1}},"description":"License keys for '
      + 'software features","$schema":"http://json-schema.org/draft-04/schema#"'
      + ',"type":"object","title":"Software License Keys"}'
    );

    Example : (''
      + '{"Georeferencing":["46AE032A - 39882AC4 - 9EC0A184 - 6F163D73"],"MTA '
      + 'resolution":["468E020A - 39A922E4 - B681A184 - 673E3D72"],"Full '
      + 'Waveform Analysis Topography with GPU support":["8AB44126 - 23B92250 -'
      + ' 16E2689F - 34EF7E7B"],"Full Waveform Analysis Topography":["0FD5FF07 '
      + '- 011A1255 - 9F76CACA - 8D2ED557"]}'
    );
  );

// Parameters for MTA processing
const
  RDB_RIEGL_MTA_SETTINGS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.mta_settings';
    Title       : 'MTA Settings';
    Description : 'Parameters for MTA processing';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Parameters for MTA '
      + 'processing","properties":{"zone_width":{"description":"Width of a MTA '
      + 'zone in meter.","minimum":0,"type":"number"},"zone_count":{"descriptio'
      + 'n":"Maximum number of MTA zones.","maximum":255,"minimum":0,"type":"in'
      + 'teger"},"modulation_depth":{"description":"Depth of pulse position '
      + 'modulation in meter.","minimum":0,"type":"number"}},"$schema":"http://'
      + 'json-schema.org/draft-04/schema#","type":"object","title":"MTA '
      + 'Settings"}'
    );

    Example : (''
      + '{"zone_width":149.896225,"zone_count":23,"modulation_depth":9.368514}'
    );
  );

// Lookup table for range correction based on raw range
const
  RDB_RIEGL_NEAR_RANGE_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.near_range_correction';
    Title       : 'Near Range Correction Table';
    Description : 'Lookup table for range correction based on raw range';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["delta","content"],"description":"Lookup table for range '
      + 'correction based on raw range","title":"Near Range Correction Table","'
      + '$schema":"http://json-schema.org/draft-04/schema#","properties":{"delt'
      + 'a":{"description":"Delta between table entries [m], first entry is at '
      + 'range = 0 m","type":"number"},"content":{"items":{"type":"number"},"de'
      + 'scription":"Correction value [m] to be added to the raw '
      + 'range","type":"array","minItems":1,"maxItems":2000}},"type":"object"}'
    );

    Example : (''
      + '{"delta":0.512,"content":[0.0]}'
    );
  );

// Standard deviation for range and amplitude as a function of amplitude
const
  RDB_RIEGL_NOISE_ESTIMATES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.noise_estimates';
    Title       : 'Noise Estimates';
    Description : 'Standard deviation for range and amplitude as a function of amplitude';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["amplitude","range_sigma","amplitude_sigma"],"description'
      + '":"Standard deviation for range and amplitude as a function of '
      + 'amplitude","title":"Noise Estimates","$schema":"http://json-schema.org'
      + '/draft-04/schema#","properties":{"range_sigma":{"items":{"type":"numbe'
      + 'r"},"description":"Sigma range [m]","type":"array"},"amplitude":{"item'
      + 's":{"type":"number"},"description":"Amplitude [dB]","type":"array"},"a'
      + 'mplitude_sigma":{"items":{"type":"number"},"description":"Sigma '
      + 'amplitude [dB]","type":"array"}},"type":"object"}'
    );

    Example : (''
      + '{"range_sigma":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,'
      + '0.016,0.014],"amplitude":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0'
      + '],"amplitude_sigma":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0.'
      + '432,0.39,0.354]}'
    );
  );

// Notch filter parameters for window glass echoes
const
  RDB_RIEGL_NOTCH_FILTER : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.notch_filter';
    Title       : 'Notch Filter';
    Description : 'Notch filter parameters for window glass echoes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["range_minimum","range_maximum","amplitude_maximum"],"des'
      + 'cription":"Notch filter parameters for window glass '
      + 'echoes","title":"Notch Filter","$schema":"http://json-schema.org/draft'
      + '-04/schema#","properties":{"range_maximum":{"description":"Maximum '
      + 'range '
      + '[m]","type":"number"},"amplitude_maximum":{"description":"Maximum '
      + 'amplitude [dB]","minimum":0,"type":"number"},"range_minimum":{"descrip'
      + 'tion":"Minimum range [m]","type":"number"}},"type":"object"}'
    );

    Example : (''
      + '{"range_maximum":0.2,"amplitude_maximum":18.0,"range_minimum":-0.5}'
    );
  );

// Details about the pixels contained in the file
const
  RDB_RIEGL_PIXEL_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pixel_info';
    Title       : 'Pixel Information';
    Description : 'Details about the pixels contained in the file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"size_llcs":{"$ref":"#/definitions/pixel_size","descrip'
      + 'tion":"Size of pixels in a locally levelled cartesian coordinate '
      + 'system (xy). This is only used for pixels based on a map projection."}'
      + ',"size":{"$ref":"#/definitions/pixel_size","description":"Size of '
      + 'pixels in file coordinate '
      + 'system."}},"required":["size"],"description":"Details about the pixels'
      + ' contained in the file","title":"Pixel Information","$schema":"http://'
      + 'json-schema.org/draft-04/schema#","type":"object","definitions":{"pixe'
      + 'l_size":{"items":{"description":"Length of pixel edge '
      + '[m].","minimum":0,"type":"number"},"description":"Size of '
      + 'pixels.","minItems":2,"type":"array","maxItems":2}}}'
    );

    Example : (''
      + '{"size_llcs":[0.5156575252891171,0.5130835356683303],"size":[0.5971642'
      + '834779395,0.5971642834779395]}'
    );
  );

// Details about the plane patch matching process
const
  RDB_RIEGL_PLANE_PATCH_MATCHING : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.plane_patch_matching';
    Title       : 'Plane Patch Matching';
    Description : 'Details about the plane patch matching process';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"plane_patch_file_one":{"$ref":"#/definitions/file_refe'
      + 'rence","description":"Reference to the plane patch file one"},"plane_p'
      + 'atch_file_two":{"$ref":"#/definitions/file_reference","description":"R'
      + 'eference to the plane patch file two"}},"required":["plane_patch_file_'
      + 'one","plane_patch_file_two"],"description":"Details about the plane '
      + 'patch matching process","title":"Plane Patch Matching","$schema":"http'
      + '://json-schema.org/draft-04/schema#","type":"object","definitions":{"f'
      + 'ile_reference":{"description":"Reference to a plane patch file","type"'
      + ':"object","required":["file_uuid","file_path"],"properties":{"file_pat'
      + 'h":{"description":"Path of the plane patch file relative to the match '
      + 'file","type":"string"},"file_uuid":{"description":"Plane patch file''s '
      + 'Universally Unique Identifier (RFC 4122)","type":"string"}}}}}'
    );

    Example : (''
      + '{"plane_patch_file_one":{"file_path":"Record009_Line001/191025_121410_'
      + 'Scanner_1.ptch","file_uuid":"810f5d57-eccf-49ed-b07a-0cdd109b4213"},"p'
      + 'lane_patch_file_two":{"file_path":"project.ptch","file_uuid":"fa47d509'
      + '-a64e-49ce-8b14-ff3130fbefa9"}}'
    );
  );

// Statistics about plane patches found by plane patch extractor
const
  RDB_RIEGL_PLANE_PATCH_STATISTICS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.plane_patch_statistics';
    Title       : 'Plane Patch Statistics';
    Description : 'Statistics about plane patches found by plane patch extractor';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Statistics about plane patches found by plane patch '
      + 'extractor","properties":{"total_area":{"description":"sum of all plane'
      + ' patch areas [m\u00b2]","type":"number"},"total_horizontal_area":{"des'
      + 'cription":"sum of all plane patch areas projected to horizontal plane '
      + '[m\u00b2]","type":"number"}},"$schema":"http://json-schema.org/draft-0'
      + '4/schema#","type":"object","title":"Plane Patch Statistics"}'
    );

    Example : (''
      + '{"total_area":14007.965,"total_horizontal_area":13954.601}'
    );
  );

// Settings and classes for plane slope classification
const
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.plane_slope_class_info';
    Title       : 'Plane Slope Class Info';
    Description : 'Settings and classes for plane slope classification';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"classes":{"description":"Class definition table","type'
      + '":"object","additionalProperties":false,"patternProperties":{"^[0-9]+$'
      + '":{"description":"one field per class, field name is class number, '
      + 'field value is class name","type":"string"}}},"settings":{"oneOf":[{"$'
      + 'ref":"#/definitions/method-1"},{"$ref":"#/definitions/method-2"}],"des'
      + 'cription":"Classification settings, details see documentation of rdbpl'
      + 'anes","type":"object"}},"required":["settings","classes"],"description'
      + '":"Settings and classes for plane slope classification","title":"Plane'
      + ' Slope Class Info","$schema":"http://json-schema.org/draft-04/schema#"'
      + ',"definitions":{"method-1":{"description":"Classification method 1","t'
      + 'ype":"object","required":["plane_classification_method","maximum_incli'
      + 'nation_angle_horizontal"],"properties":{"plane_classification_method":'
      + '{"description":"method ID (=1)","maximum":1,"minimum":1,"type":"intege'
      + 'r"},"maximum_inclination_angle_horizontal":{"description":"maximum '
      + 'inclination angle of horizontal plane patches [deg]","maximum":360.0,"'
      + 'minimum":-360.0,"type":"number"}}},"method-2":{"description":"Classifi'
      + 'cation method 2","type":"object","required":["plane_classification_met'
      + 'hod","sloping_plane_classes_minimum_angle","sloping_plane_classes_maxi'
      + 'mum_angle"],"properties":{"plane_classification_method":{"description"'
      + ':"method ID (=2)","maximum":2,"minimum":2,"type":"integer"},"sloping_p'
      + 'lane_classes_minimum_angle":{"description":"minimum inclination angle '
      + 'of sloping plane patches [deg]","maximum":360.0,"minimum":-360.0,"type'
      + '":"number"},"sloping_plane_classes_maximum_angle":{"description":"maxi'
      + 'mum inclination angle of sloping plane patches [deg]","maximum":360.0,'
      + '"minimum":-360.0,"type":"number"}}}},"type":"object"}'
    );

    Example : (''
      + '{"classes":{"7":"vertical, pointing south","8":"vertical, pointing '
      + 'north","12":"sloping, pointing down and north","9":"vertical, pointing'
      + ' west","6":"vertical, pointing east","2":"sloping, pointing up and '
      + 'east","13":"sloping, pointing down and west","5":"sloping, pointing up'
      + ' and west","11":"sloping, pointing down and south","14":"horizontal, '
      + 'pointing down","1":"horizontal, pointing up","10":"sloping, pointing '
      + 'down and east","3":"sloping, pointing up and south","4":"sloping, '
      + 'pointing up and north"},"settings":{"plane_classification_method":2,"s'
      + 'loping_plane_classes_minimum_angle":10.0,"sloping_plane_classes_maximu'
      + 'm_angle":70.0}}'
    );
  );

// Grouping and sorting of point attributes for visualization purposes
const
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.point_attribute_groups';
    Title       : 'Point Attribute Groups';
    Description : 'Grouping and sorting of point attributes for visualization purposes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"additionalProperties":false,"patternProperties":{"^.*$":{"items":{"d'
      + 'escription":"Point attribute full name or name pattern (perl regular '
      + 'expression syntax)","type":"string"},"description":"Each field of the '
      + 'object represents a point attribute group and holds a list of point '
      + 'attributes, where the field name is the group '
      + 'name.","type":"array","minItems":1}},"description":"Grouping and '
      + 'sorting of point attributes for visualization purposes","$schema":"htt'
      + 'p://json-schema.org/draft-04/schema#","type":"object","title":"Point '
      + 'Attribute Groups"}'
    );

    Example : (''
      + '{"Time":["riegl.timestamp"],"Primary Attributes":["riegl.reflectance",'
      + '"riegl.amplitude","riegl.deviation"],"Coordinates/Vectors":["riegl.xyz'
      + '","riegl.range","riegl.theta","riegl.phi"],"Other '
      + 'Attributes":["riegl.selected","riegl.visible"],"Secondary '
      + 'Attributes":["riegl.mirror_facet","riegl.waveform_available"]}'
    );
  );

// Details about point cloud files
const
  RDB_RIEGL_POINTCLOUD_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pointcloud_info';
    Title       : 'Point Cloud Information';
    Description : 'Details about point cloud files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Details about point cloud files","properties":{"commen'
      + 'ts":{"description":"Comments","type":"string"},"project":{"description'
      + '":"Project '
      + 'name","type":"string"},"field_of_application":{"description":"Field of'
      + ' application","enum":["unknown","SLS","TLS","KLS","MLS","ULS","ALS","B'
      + 'LS","ILS"],"type":"string"}},"$schema":"http://json-schema.org/draft-0'
      + '4/schema#","type":"object","title":"Point Cloud Information"}'
    );

    Example : (''
      + '{"comments":"Line 3","project":"Campaign '
      + '4","field_of_application":"ALS"}'
    );
  );

// Estimated position and orientation information
const
  RDB_RIEGL_POSE_ESTIMATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pose_estimation';
    Title       : 'Pose Estimation';
    Description : 'Estimated position and orientation information';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["orientation"],"description":"Estimated position and '
      + 'orientation information as measured by GNSS, IMU or inclination '
      + 'sensors","title":"Pose Estimation","$schema":"http://json-schema.org/d'
      + 'raft-04/schema#","properties":{"barometric_height_amsl":{"description"'
      + ':"Altitude determined based on the atmospheric pressure according to '
      + 'the standard atmosphere laws '
      + '[m].","type":"number"},"position":{"description":"Position coordinates'
      + ' and position accuracy values as measured by GNSS in the specified '
      + 'Coordinate Reference System (CRS)","type":"object","required":["coordi'
      + 'nate_1","coordinate_2","coordinate_3","horizontal_accuracy","vertical_'
      + 'accuracy","crs"],"properties":{"coordinate_1":{"description":"Coordina'
      + 'te 1 as defined by axis 1 of the specified CRS (e.g., X, '
      + 'Latitude)","type":"number"},"crs":{"description":"Global Coordinate '
      + 'Reference System","type":"object","required":["epsg"],"properties":{"e'
      + 'psg":{"description":"EPSG '
      + 'code","minimum":0,"type":"integer"},"wkt":{"description":"\"Well-Known'
      + ' Text\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta'
      + 'ndards/wkt-crs)","type":"string"}}},"horizontal_accuracy":{"descriptio'
      + 'n":"Horizontal accuracy [m]","minimum":0,"type":"number","exclusiveMin'
      + 'imum":true},"vertical_accuracy":{"description":"Vertical accuracy [m]"'
      + ',"minimum":0,"type":"number","exclusiveMinimum":true},"coordinate_2":{'
      + '"description":"Coordinate 2 as defined by axis 2 of the specified CRS '
      + '(e.g., Y, '
      + 'Longitude)","type":"number"},"coordinate_3":{"description":"Coordinate'
      + ' 3 as defined by axis 3 of the specified CRS (e.g., Z, Altitude)","typ'
      + 'e":"number"}}},"orientation":{"description":"Orientation values and '
      + 'orientation accuracies, measured with IMU or inclination sensors.","ty'
      + 'pe":"object","required":["roll","pitch","yaw","roll_accuracy","pitch_a'
      + 'ccuracy","yaw_accuracy"],"properties":{"roll_accuracy":{"description":'
      + '"Roll angle accuracy [deg]","minimum":0,"type":"number","exclusiveMini'
      + 'mum":true},"yaw":{"description":"Yaw angle about scanner Z-axis [deg]"'
      + ',"maximum":360,"minimum":-360,"type":"number"},"roll":{"description":"'
      + 'Roll angle about scanner X-axis [deg]","maximum":360,"minimum":-360,"t'
      + 'ype":"number"},"yaw_accuracy":{"description":"Yaw angle accuracy [deg]'
      + '","minimum":0,"type":"number","exclusiveMinimum":true},"pitch_accuracy'
      + '":{"description":"Pitch angle accuracy [deg]","minimum":0,"type":"numb'
      + 'er","exclusiveMinimum":true},"pitch":{"description":"Pitch angle about'
      + ' scanner Y-axis [deg]","maximum":360,"minimum":-360,"type":"number"}}}'
      + '},"type":"object"}'
    );

    Example : (''
      + '{"barometric_height_amsl":386.7457796227932,"position":{"coordinate_1"'
      + ':48.655799473,"crs":{"epsg":4979,"wkt":"GEOGCS[\"WGS84 / Geographic\",'
      + 'DATUM[\"WGS84\",SPHEROID[\"WGS84\",6378137.000,298.257223563,AUTHORITY'
      + '[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]],PRIMEM[\"Greenwich\'
      + '",0.0000000000000000,AUTHORITY[\"EPSG\",\"8901\"]],UNIT[\"Degree\",0.0'
      + '1745329251994329547,AUTHORITY[\"EPSG\",\"9102\"]],AXIS[\"Latitude\",NO'
      + 'RTH],AXIS[\"Longitude\",EAST],AUTHORITY[\"EPSG\",\"4979\"]]"},"horizon'
      + 'tal_accuracy":0.810699999332428,"vertical_accuracy":1.3314999341964722'
      + ',"coordinate_2":15.645033406,"coordinate_3":362.7124938964844},"orient'
      + 'ation":{"roll_accuracy":0.009433783936875745,"yaw":101.87293630292045,'
      + '"roll":3.14743073066123,"yaw_accuracy":1.0094337839368757,"pitch_accur'
      + 'acy":0.009433783936875745,"pitch":1.509153024827064}}'
    );
  );

// Details on position and orientation sensors
const
  RDB_RIEGL_POSE_SENSORS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pose_sensors';
    Title       : 'Pose Sensors';
    Description : 'Details on position and orientation sensors';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"accelerometer":{"description":"Accelerometer details",'
      + '"type":"object","required":["unit","x_axis","y_axis","z_axis","offset"'
      + ',"origin","relative_nonlinearity"],"properties":{"z_axis":{"$ref":"#/d'
      + 'efinitions/vector","description":"Sensitive Z axis of sensor at frame '
      + 'angle = '
      + '0"},"origin":{"$ref":"#/definitions/vector","description":"Sensor '
      + 'origin in SOCS [m] at frame angle = '
      + '0"},"offset":{"$ref":"#/definitions/vector","description":"Value to be'
      + ' subtracted from raw measurement values"},"x_axis":{"$ref":"#/definiti'
      + 'ons/vector","description":"Sensitive X axis of sensor at frame angle ='
      + ' 0"},"relative_nonlinearity":{"$ref":"#/definitions/vector","descripti'
      + 'on":"Relative nonlinearity, '
      + 'dimensionless"},"unit":{"description":"Unit of raw data and '
      + 'calibration values, 1 LSB in 9.81 m/s\u00b2","minimum":0,"type":"numbe'
      + 'r","exclusiveMinimum":true},"y_axis":{"$ref":"#/definitions/vector","d'
      + 'escription":"Sensitive Y axis of sensor at frame angle = '
      + '0"}}},"gyroscope":{"description":"Gyroscope details","type":"object","'
      + 'required":["unit","x_axis","y_axis","z_axis","offset","origin","relati'
      + 've_nonlinearity"],"properties":{"z_axis":{"$ref":"#/definitions/vector'
      + '","description":"Sensitive Z axis of sensor at frame angle = '
      + '0"},"origin":{"$ref":"#/definitions/vector","description":"Sensor '
      + 'origin in SOCS [m] at frame angle = '
      + '0"},"offset":{"$ref":"#/definitions/vector","description":"Value to be'
      + ' subtracted from raw measurement values"},"x_axis":{"$ref":"#/definiti'
      + 'ons/vector","description":"Sensitive X axis of sensor at frame angle ='
      + ' 0"},"relative_nonlinearity":{"$ref":"#/definitions/vector","descripti'
      + 'on":"Relative nonlinearity, '
      + 'dimensionless"},"unit":{"description":"Unit of raw data and '
      + 'calibration values, 1 LSB in rad/s","minimum":0,"type":"number","exclu'
      + 'siveMinimum":true},"y_axis":{"$ref":"#/definitions/vector","descriptio'
      + 'n":"Sensitive Y axis of sensor at frame angle = '
      + '0"}}},"magnetic_field_sensor":{"description":"Magnetic Field Sensor de'
      + 'tails","type":"object","required":["unit","x_axis","y_axis","z_axis","'
      + 'offset","fixed","relative_nonlinearity"],"properties":{"z_axis":{"$ref'
      + '":"#/definitions/vector","description":"Sensitive Z axis of sensor at '
      + 'frame angle = '
      + '0"},"fixed":{"$ref":"#/definitions/vector","description":"Distortion '
      + 'of magnetic field caused by non-rotating scanner '
      + 'part"},"offset":{"$ref":"#/definitions/vector","description":"Value to'
      + ' be subtracted from raw measurement values"},"x_axis":{"$ref":"#/defin'
      + 'itions/vector","description":"Sensitive X axis of sensor at frame '
      + 'angle = 0"},"relative_nonlinearity":{"$ref":"#/definitions/vector","de'
      + 'scription":"Relative nonlinearity, '
      + 'dimensionless"},"unit":{"description":"Unit of raw data and '
      + 'calibration values, 1 LSB in nT","minimum":0,"type":"number","exclusiv'
      + 'eMinimum":true},"y_axis":{"$ref":"#/definitions/vector","description":'
      + '"Sensitive Y axis of sensor at frame angle = 0"}}}},"required":["gyros'
      + 'cope","accelerometer","magnetic_field_sensor"],"description":"Details '
      + 'on position and orientation sensors","title":"Pose Sensors","$schema":'
      + '"http://json-schema.org/draft-04/schema#","type":"object","definitions'
      + '":{"vector":{"items":{"description":"Index 0=X, 1=Y, 2=Z component","t'
      + 'ype":"number"},"type":"array","minItems":3,"maxItems":3}}}'
    );

    Example : (''
      + '{"accelerometer":{"z_axis":[1.639882206916809,15166.744140625,-116.997'
      + '42889404297],"origin":[0.026900000870227814,-0.03999999910593033,-0.08'
      + '950000256299973],"offset":[-733.3636474609375,58.969032287597656,1060.'
      + '2550048828125],"x_axis":[-15008.123046875,56.956390380859375,-60.51756'
      + '66809082],"relative_nonlinearity":[0.0,0.0,0.0],"unit":6.6666667407844'
      + '22e-05,"y_axis":[-7.027288913726807,-44.12333679199219,14952.370117187'
      + '5]},"gyroscope":{"z_axis":[0.555869996547699,119.22135162353516,0.4675'
      + '85027217865],"origin":[0.026900000870227814,-0.03999999910593033,-0.08'
      + '950000256299973],"offset":[-50.92609786987305,146.15643310546875,62.43'
      + '27278137207],"x_axis":[-121.195556640625,0.8219714164733887,0.23130317'
      + '03233719],"relative_nonlinearity":[2.888176311444113e-07,1.06274164579'
      + '645e-07,-1.7186295080634935e-39],"unit":0.00014544410805683583,"y_axis'
      + '":[-0.440765917301178,-0.7897399663925171,119.5894775390625]},"magneti'
      + 'c_field_sensor":{"z_axis":[0.00041987866279669106,7.876977906562388e-0'
      + '5,0.011407104320824146],"fixed":[-1576.010498046875,1596.081787109375,'
      + '0.0],"offset":[-23812.052734375,5606.57666015625,2493.28125],"x_axis":'
      + '[-0.011162743903696537,-2.315962774446234e-05,0.00016818844596855342],'
      + '"relative_nonlinearity":[0.0,0.0,0.0],"unit":91.74311828613281,"y_axis'
      + '":[0.00027888521435670555,-0.011427424848079681,-5.204829722060822e-05'
      + ']}}'
    );
  );

// Laser pulse position modulation used for MTA resolution
const
  RDB_RIEGL_PULSE_POSITION_MODULATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pulse_position_modulation';
    Title       : 'Pulse Position Modulation';
    Description : 'Laser pulse position modulation used for MTA resolution';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["length","num_mod_ampl","pulse_interval"],"description":"'
      + 'Laser pulse position modulation used for MTA '
      + 'resolution","title":"Pulse Position Modulation","$schema":"http://json'
      + '-schema.org/draft-04/schema#","properties":{"phase_step":{"description'
      + '":"Step width in phase of modulation code from line to line","maximum"'
      + ':255,"minimum":0,"type":"integer"},"pulse_interval":{"items":{"minimum'
      + '":0,"type":"number"},"description":"Explicit table of the pulse '
      + 'position modulation used for MTA resolution. Table gives times between'
      + ' successive laser pulses in '
      + 'seconds.","type":"array"},"num_mod_ampl":{"description":"Number of '
      + 'different modulation amplitudes (2: binary modulation)","maximum":255,'
      + '"minimum":0,"type":"integer"},"code_phase_mode":{"description":"0: no '
      + 'synchronization, 1: toggle between 2 phases, 2: increment with phase_i'
      + 'ncrement","maximum":255,"minimum":0,"type":"integer"},"length":{"descr'
      + 'iption":"Length of '
      + 'code","maximum":255,"minimum":0,"type":"integer"}},"type":"object"}'
    );

    Example : (''
      + '{"phase_step":5,"pulse_interval":[2.759375e-06,2.759375e-06,2.759375e-'
      + '06,2.759375e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.'
      + '759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.82187'
      + '5e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06'
      + ',2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.82'
      + '1875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.759375e'
      + '-06,2.821875e-06],"num_mod_ampl":2,"code_phase_mode":2,"length":31}'
    );
  );

// Statistics about target distance wrt. SOCS origin
const
  RDB_RIEGL_RANGE_STATISTICS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.range_statistics';
    Title       : 'Range Statistics';
    Description : 'Statistics about target distance wrt. SOCS origin';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["minimum","average","maximum","std_dev"],"description":"S'
      + 'tatistics about target distance wrt. SOCS origin","title":"Range Stati'
      + 'stics","$schema":"http://json-schema.org/draft-04/schema#","properties'
      + '":{"std_dev":{"description":"Standard '
      + 'deviation","type":"number"},"average":{"description":"Average '
      + 'value","type":"number"},"maximum":{"description":"Maximum '
      + 'value","type":"number"},"minimum":{"description":"Minimum '
      + 'value","type":"number"}},"type":"object"}'
    );

    Example : (''
      + '{"std_dev":24.349,"average":15.49738,"maximum":574.35,"minimum":0.919}'
    );
  );

// Receiver Internals
const
  RDB_RIEGL_RECEIVER_INTERNALS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.receiver_internals';
    Title       : 'Receiver Internals';
    Description : 'Receiver Internals';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"tbl":{"items":{"$ref":"#/definitions/fp_table"},"descr'
      + 'iption":"various internal '
      + 'data","type":"array","minItems":1},"sr":{"description":"Sample rate [H'
      + 'z]","minimum":0,"type":"number","exclusiveMinimum":true},"a":{"items":'
      + '{"type":"number"},"description":"Amplitude [dB]","type":"array","minIt'
      + 'ems":1,"maxItems":256},"t":{"type":"object","additionalProperties":fal'
      + 'se,"patternProperties":{"^[0-9]+$":{"$ref":"#/definitions/fp","descrip'
      + 'tion":"one field per channel, field name is channel '
      + 'index"}}},"ex":{"description":"DEPRECATED, use '
      + '''riegl.exponential_decomposition'' '
      + 'instead","type":"object"},"nt":{"description":"Number of traces","maxi'
      + 'mum":255,"minimum":0,"type":"integer"},"mw":{"description":"Maximum we'
      + 'ight","maximum":4095,"minimum":0,"type":"number","exclusiveMinimum":tr'
      + 'ue},"si":{"description":"Start index (hw_start)","maximum":255,"minimu'
      + 'm":0,"type":"number"},"ns":{"description":"Number of samples","maximum'
      + '":4095,"minimum":0,"type":"integer"}},"description":"Receiver '
      + 'Internals","title":"Receiver Internals","$schema":"http://json-schema.'
      + 'org/draft-04/schema#","type":"object","definitions":{"fp_table_row":{"'
      + 'items":{"type":"number"},"type":"array","minItems":1,"maxItems":2048},'
      + '"fp_table":{"properties":{"tc":{"min":0,"description":"table data type'
      + ' code","max":255,"type":"integer"},"ny":{"min":1,"description":"number'
      + ' of y entries","max":2048,"type":"integer"},"nx":{"min":1,"description'
      + '":"number of x entries","max":2048,"type":"integer"},"tv":{"items":{"o'
      + 'neOf":[{"$ref":"#/definitions/fp_table_row"},{"type":"number"}]},"type'
      + '":"array","minItems":1,"maxItems":2048},"ch":{"min":0,"description":"c'
      + 'hannel number","max":255,"type":"integer"}},"type":"object","required"'
      + ':["ch","tc","nx","ny","tv"],"desription":"scanner internal '
      + 'data"},"fp":{"description":"Fingerprint values","type":"object","requi'
      + 'red":["s","w"],"properties":{"s":{"items":{"items":{"type":"number"},"'
      + 'type":"array","minItems":1,"maxItems":4096},"type":"array","minItems":'
      + '1,"maxItems":256},"w":{"items":{"items":{"type":"number"},"type":"arra'
      + 'y","minItems":5,"maxItems":5},"type":"array","minItems":1,"maxItems":2'
      + '56}}}}}'
    );

    Example : (''
      + '{"tbl":[{"tc":1,"ny":2,"nx":5,"tv":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.5]]'
      + ',"ch":0}],"sr":7959997000.0,"a":[-1.55],"t":{"0":{"s":[[1.23,4.56],[7.'
      + '89,0.12]],"w":[[78,86,126,134,31],[78,86,126,134,31]]},"1":{"s":[[1.23'
      + ',4.56],[7.89,0.12]],"w":[[78,86,126,134,31],[78,86,126,134,31]]}},"nt"'
      + ':128,"mw":31,"si":48,"ns":400}'
    );
  );

// Lookup table for reflectance calculation based on amplitude and range
const
  RDB_RIEGL_REFLECTANCE_CALCULATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.reflectance_calculation';
    Title       : 'Reflectance Calculation Table';
    Description : 'Lookup table for reflectance calculation based on amplitude and range';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["delta","content"],"description":"Lookup table for '
      + 'reflectance calculation based on amplitude and '
      + 'range","title":"Reflectance Calculation Table","$schema":"http://json-'
      + 'schema.org/draft-04/schema#","properties":{"delta":{"description":"Del'
      + 'ta between table entries [m], first entry is at range = 0 m","type":"n'
      + 'umber"},"content":{"items":{"type":"number"},"description":"Correction'
      + ' value [dB] to be added to the amplitude","type":"array","minItems":1,'
      + '"maxItems":2000}},"type":"object"}'
    );

    Example : (''
      + '{"delta":0.150918,"content":[-33.01]}'
    );
  );

// Range-dependent and scan-angle-dependent correction of reflectance reading
const
  RDB_RIEGL_REFLECTANCE_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.reflectance_correction';
    Title       : 'Near-Range Reflectance Correction';
    Description : 'Range-dependent and scan-angle-dependent correction of reflectance reading';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["ranges_m","line_angles_deg","reflectance_correction_db"]'
      + ',"description":"Range-dependent and scan-angle-dependent correction of'
      + ' reflectance reading","title":"Near-range reflectance correction","$sc'
      + 'hema":"http://json-schema.org/draft-04/schema#","properties":{"ranges_'
      + 'm":{"items":{"type":"number"},"description":"Range [m]","type":"array"'
      + '},"reflectance_correction_db":{"items":{"items":{"description":"column'
      + 's (each value corresponds to an '
      + 'angle)","type":"number"},"description":"rows (each array corresponds '
      + 'to a range)","type":"array"},"description":"Near range reflectance '
      + 'correction in dB as a function of range over angle","type":"array"},"l'
      + 'ine_angles_deg":{"items":{"type":"number"},"description":"Angle '
      + '[deg]","type":"array"}},"type":"object"}'
    );

    Example : (''
      + '{"ranges_m":[0.0,1.0,2.0,3.0],"reflectance_correction_db":[[0.8,0.7,0.'
      + '6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05'
      + ',0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.'
      + '4,0.3,0.2,0.1,0.05,0.01]],"line_angles_deg":[0.0,0.5,1.0,1.5,1.0,2.0,2'
      + '.5,3.0,3.5,4.0]}'
    );
  );

// Details about RiLOC processing
const
  RDB_RIEGL_RILOC_PROCESSING_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.riloc_processing_info';
    Title       : 'RiLOC Processing Information';
    Description : 'Details about RiLOC processing';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["trajectory","optimizer","georeferencing"],"description":'
      + '"Details about RiLOC processing","title":"RiLOC Processing Information'
      + '","$schema":"http://json-schema.org/draft-04/schema#","properties":{"p'
      + 'lanes":{"description":"Information on plane correspondences and used '
      + 'extraction/matching templates.","type":"object","properties":{}},"traj'
      + 'ectory":{"description":"Trajectory information and statistics.","type"'
      + ':"object","properties":{}},"georeferencing":{"description":"Internal '
      + 'georeferencing parameters.","type":"object","properties":{}},"optimize'
      + 'r":{"description":"Optimization result information and '
      + 'statistics.","type":"object","properties":{}}},"type":"object"}'
    );

    Example : (''
      + '{"planes":{},"trajectory":{},"georeferencing":{},"optimizer":{}}'
    );
  );

// Scan pattern description
const
  RDB_RIEGL_SCAN_PATTERN : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.scan_pattern';
    Title       : 'Scan Pattern';
    Description : 'Scan pattern description';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Scan pattern '
      + 'description","properties":{"rectangular":{"description":"Rectangular '
      + 'Field Of View Scan Pattern","type":"object","required":["phi_start","p'
      + 'hi_stop","phi_increment","theta_start","theta_stop","theta_increment"]'
      + ',"properties":{"phi_start":{"description":"Start phi angle in SOCS [de'
      + 'g]","maximum":360.0,"minimum":0.0,"type":"number"},"phi_increment":{"d'
      + 'escription":"Increment of phi angle in SOCS [deg]","maximum":90.0,"min'
      + 'imum":0.0,"type":"number"},"theta_increment":{"description":"Increment'
      + ' of theta angle in SOCS [deg]","maximum":90.0,"minimum":0.0,"type":"nu'
      + 'mber"},"phi_stop":{"description":"Stop phi angle in SOCS [deg]","maxim'
      + 'um":720.0,"minimum":0.0,"type":"number"},"theta_start":{"description":'
      + '"Start theta angle in SOCS [deg]","maximum":180.0,"minimum":0.0,"type"'
      + ':"number"},"theta_stop":{"description":"Stop theta angle in SOCS [deg]'
      + '","maximum":180.0,"minimum":0.0,"type":"number"},"program":{"$ref":"#/'
      + 'definitions/program"}}},"line":{"description":"Line Scan Pattern","typ'
      + 'e":"object","required":["start","stop","increment"],"properties":{"inc'
      + 'rement":{"description":"Increment of angle in SOCS [deg]","maximum":90'
      + '.0,"minimum":0.0,"type":"number"},"program":{"$ref":"#/definitions/pro'
      + 'gram"},"start":{"description":"Start angle in SOCS [deg]","maximum":36'
      + '0.0,"minimum":0.0,"type":"number"},"stop":{"description":"Stop angle '
      + 'in SOCS [deg]","maximum":720.0,"minimum":0.0,"type":"number"}}},"segme'
      + 'nts":{"description":"Segmented Line Scan Pattern","type":"object","req'
      + 'uired":["list"],"properties":{"list":{"items":{"description":"Line '
      + 'Scan Segment","type":"object","required":["start","stop","increment"],'
      + '"properties":{"increment":{"description":"Increment of angle in SOCS ['
      + 'deg]","maximum":90.0,"minimum":0.0,"type":"number"},"start":{"descript'
      + 'ion":"Start angle in SOCS [deg]","maximum":360.0,"minimum":0.0,"type":'
      + '"number"},"stop":{"description":"Stop angle in SOCS [deg]","maximum":7'
      + '20.0,"minimum":0.0,"type":"number"}}},"type":"array"},"program":{"$ref'
      + '":"#/definitions/program"}}}},"$schema":"http://json-schema.org/draft-'
      + '04/schema#","definitions":{"program":{"description":"Measurement progr'
      + 'am","type":"object","required":["name"],"properties":{"laser_prr":{"de'
      + 'scription":"Laser Pulse Repetition Rate [Hz]","minimum":0,"type":"numb'
      + 'er","exclusiveMinimum":false},"name":{"description":"Name of '
      + 'measurement program","type":"string"}}}},"title":"Scan Pattern"}'
    );

    Example : (''
      + '{"rectangular":{"phi_start":45.0,"phi_increment":0.04,"theta_increment'
      + '":0.04,"phi_stop":270.0,"theta_start":30.0,"theta_stop":130.0,"program'
      + '":{"laser_prr":100000.0,"name":"High Speed"}}}'
    );
  );

// Details about laser shot files
const
  RDB_RIEGL_SHOT_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.shot_info';
    Title       : 'Shot Information';
    Description : 'Details about laser shot files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Details about laser shot files","properties":{"shot_fi'
      + 'le":{"type":"object","required":["file_extension"],"properties":{"file'
      + '_extension":{"description":"Shot file extension, without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC 4122)","type":"string"}}}},"$schema":"http://js'
      + 'on-schema.org/draft-04/schema#","type":"object","title":"Shot '
      + 'Information"}'
    );

    Example : (''
      + '{"shot_file":{"file_extension":"sodx","file_uuid":"26a00815-67c0-4bff-'
      + '8fe8-c577378fe663"}}'
    );
  );

// Point filters applied in addition to the application-defined filters
const
  RDB_RIEGL_STORED_FILTERS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.stored_filters';
    Title       : 'Stored Filters';
    Description : 'Point filters applied in addition to the application-defined filters';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["activated","filters"],"description":"Point filters '
      + 'applied in addition to the application-defined '
      + 'filters","title":"Stored filters","$schema":"http://json-schema.org/dr'
      + 'aft-04/schema#","properties":{"filters":{"items":{"description":"Point'
      + ' filter definition","type":"object","required":["activated","title","d'
      + 'escription","filter"],"properties":{"description":{"description":"A '
      + 'brief description of the filter (e.g. for display in a '
      + 'tooltip)","type":"string"},"activated":{"description":"Apply (''true'') '
      + 'or ignore (''false'') this '
      + 'filter","type":"boolean"},"filter":{"description":"The RDB filter '
      + 'string to apply (e.g. when reading points or index), details see '
      + 'documentation of function '
      + 'select()","type":"string"},"title":{"description":"A short filter '
      + 'title (e.g. for display in a selection '
      + 'list)","type":"string"}}},"description":"List of point '
      + 'filters","type":"array"},"activated":{"description":"Apply (''true'') or'
      + ' ignore (''false'') all filters","type":"boolean"}},"type":"object"}'
    );

    Example : (''
      + '{"filters":[{"description":"Ignore points with uncertain MTA zone '
      + 'assignment","activated":false,"filter":"riegl.mta_uncertain_point == '
      + '0","title":"Ignore uncertain points from MTA '
      + 'resolution"},{"description":"Ignore points with an echo signal '
      + 'amplitude below the dynamic '
      + 'trigger","activated":false,"filter":"riegl.dyntrig_uncertain_point == '
      + '0","title":"Ignore points below dynamic trigger '
      + 'level"},{"description":"Ignore points that are outside the outer AABB '
      + 'in BOCS","activated":false,"filter":"riegl.point_outside_aabb_bocs == '
      + '0","title":"Ignore points outside outer AABB"},{"description":"Ignore '
      + 'points that are inside the inner AABB in '
      + 'BOCS","activated":false,"filter":"riegl.point_inside_aabb_bocs == '
      + '0","title":"Ignore points inside inner AABB"}],"activated":true}'
    );
  );

// Details of major system components like lidar sensors, cameras and navigation devices
const
  RDB_RIEGL_SYSTEM_DESCRIPTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.system_description';
    Title       : 'System Description';
    Description : 'Details of major system components like lidar sensors, cameras and navigation devices';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["$class","version","author","timestamp","system"],"descri'
      + 'ption":"Details of major system components like lidar sensors, cameras'
      + ' and navigation devices","title":"System Description","$schema":"http:'
      + '//json-schema.org/draft-04/schema#","properties":{"$class":{"descripti'
      + 'on":"Object class name","enum":["Document"],"type":"string"},"version"'
      + ':{"description":"Document format '
      + 'version","type":"string"},"author":{"description":"Name of the author '
      + 'that created the '
      + 'document","type":"string"},"timestamp":{"description":"Date and time '
      + 'of creation (e.g. 2019-06-19T13:32:10+02:00)","type":"string"},"system'
      + '":{"description":"The actual system description, details see '
      + 'documentation of RIEGL System '
      + 'Description","type":"object"}},"type":"object"}'
    );

    Example : (''
      + '{"$class":"Document","version":"1.2.1","author":"RIEGL LMS GmbH, Austr'
      + 'ia","timestamp":"2022-09-30T11:56:26+00:00","system":{"details":"see '
      + 'documentation of RIEGL System Description"}}'
    );
  );

// Conversion of background radiation raw values to temperatures in °C
const
  RDB_RIEGL_TEMPERATURE_CALCULATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.temperature_calculation';
    Title       : 'Temperature Calculation Table';
    Description : 'Conversion of background radiation raw values to temperatures in °C';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"InGaAs":{"$ref":"#/definitions/conversion_table","desc'
      + 'ription":"Conversion table for InGaAs channel"},"Si":{"$ref":"#/defini'
      + 'tions/conversion_table","description":"Conversion table for Si channel'
      + '"},"InGaAs_Si_Difference":{"$ref":"#/definitions/conversion_table","de'
      + 'scription":"Conversion table for InGaAs - Si '
      + 'difference"}},"description":"Conversion of background radiation raw '
      + 'values to temperatures in \u00b0C","title":"Temperature Calculation Ta'
      + 'ble","$schema":"http://json-schema.org/draft-04/schema#","type":"objec'
      + 't","definitions":{"conversion_table":{"type":"object","required":["val'
      + 'ue","temperature"],"properties":{"value":{"items":{"type":"number"},"d'
      + 'escription":"LSB [1]","type":"array"},"temperature":{"items":{"type":"'
      + 'number"},"description":"Temperature [\u00b0C]","type":"array"}}}}}'
    );

    Example : (''
      + '{"InGaAs":{"value":[0.0,64.00097659230323,128.0019531846065,192.002929'
      + '7769097,256.0039063692129,320.00488296151616,384.0058595538194,448.006'
      + '8361461226,512.0078127384259],"temperature":[307.22196722535614,309.11'
      + '53478247277,311.1188086915047,313.10025350480055,315.2137946389828,317'
      + '.2172555057597,319.2207163725366,321.2021611858325,323.3157023200148]}'
      + ',"Si":{"value":[0.0,64.00097659230323,128.0019531846065,192.0029297769'
      + '097,256.0039063692129,320.00488296151616,384.0058595538194,448.0068361'
      + '461226,512.0078127384259],"temperature":[546.300048828125,548.81640512'
      + '12026,551.3143938500972,554.0144257850053,556.604252334815,559.2124464'
      + '488079,561.8022729986177,564.4104671126105,567.0002936624203]},"InGaAs'
      + '_Si_Difference":{"value":[1000.0,1100.090029602954,1200.04425183874,13'
      + '00.1342814416948,1400.0885036774805,1500.0427259132668,1600.1327555162'
      + '209,1700.0869777520065,1800.0411999877924],"temperature":[1749.9771111'
      + '17893,1749.977111117893,1749.977111117893,1749.977111117893,1749.97711'
      + '1117893,1749.977111117893,1744.7813348796044,1681.9971312601092,1622.3'
      + '944822534868]}}'
    );
  );

// Base of timestamps (epoch)
const
  RDB_RIEGL_TIME_BASE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.time_base';
    Title       : 'Time Base';
    Description : 'Base of timestamps (epoch)';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["epoch","source"],"description":"Base of timestamps '
      + '(epoch)","title":"Time Base","$schema":"http://json-schema.org/draft-0'
      + '4/schema#","properties":{"source":{"description":"Timestamp source","e'
      + 'num":["unknown","RTC","GNSS"],"type":"string"},"system":{"description"'
      + ':"Time system (time standard)","enum":["unknown","UTC","GPS"],"type":"'
      + 'string"},"epoch":{"description":"Date and time of timestamp ''0'' as '
      + 'proposed by RFC 3339 (e.g. '
      + '2015-10-27T00:00:00+01:00).","type":"string"}},"type":"object"}'
    );

    Example : (''
      + '{"source":"GNSS","system":"UTC","epoch":"2015-10-27T00:00:00+00:00"}'
    );
  );

// Details about position+orientation files
const
  RDB_RIEGL_TRAJECTORY_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.trajectory_info';
    Title       : 'Trajectory Information';
    Description : 'Details about position+orientation files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["time_interval","navigation_frame"],"description":"Detail'
      + 's about position+orientation files","title":"Trajectory Information","'
      + '$schema":"http://json-schema.org/draft-04/schema#","properties":{"devi'
      + 'ce":{"description":"Navigation device, e.g. name/type/serial","type":"'
      + 'string"},"time_interval":{"description":"Time interval statistics","ty'
      + 'pe":"object","required":["minimum","average","maximum","std_dev"],"pro'
      + 'perties":{"std_dev":{"description":"Standard deviation of intervals '
      + '[s]","type":"number"},"average":{"description":"Average time interval '
      + '[s]","type":"number"},"maximum":{"description":"Maximum time interval '
      + '[s]","type":"number"},"minimum":{"description":"Minimum time interval '
      + '[s]","type":"number"}}},"settings":{"description":"Settings used to '
      + 'calculate the trajectory (descriptive '
      + 'text)","type":"string"},"navigation_frame":{"description":"Navigation '
      + 'frame (NED: North-East-Down, ENU: East-North-Up)","enum":["unknown","N'
      + 'ED","ENU"],"type":"string"},"location":{"description":"Project '
      + 'location, e.g. city/state/country","type":"string"},"software":{"descr'
      + 'iption":"Software that calculated the trajectory (this may be the same'
      + ' or different software than the one that created the '
      + 'file)","type":"string"},"project":{"description":"Project '
      + 'name","type":"string"},"company":{"description":"Company '
      + 'name","type":"string"},"field_of_application":{"description":"Field of'
      + ' application","enum":["unknown","SLS","TLS","KLS","MLS","ULS","ALS","B'
      + 'LS","ILS"],"type":"string"}},"type":"object"}'
    );

    Example : (''
      + '{"device":"IMU Model 12/1, Serial# 12345","time_interval":{"std_dev":5'
      + '.51e-07,"average":0.00500053,"maximum":0.005004883,"minimum":0.0050003'
      + '2},"settings":"default","navigation_frame":"NED","location":"Horn","so'
      + 'ftware":"Navigation Software XYZ","project":"Campaign '
      + '3","company":"RIEGL LMS","field_of_application":"MLS"}'
    );
  );

// Trigger-Timestamping-Unit/IP configuration
const
  RDB_RIEGL_TTIP_CONFIGURATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.ttip_configuration';
    Title       : 'TTIP Configuration';
    Description : 'Trigger-Timestamping-Unit/IP configuration';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["ttip_version","num_channel","ext_channel","ext_signal","'
      + 'ext_delay","ext_subdivider","ext_bitmask","dmi_incr","dmi_min_time","d'
      + 'mi_max_time","dmi_dist_per_tick","channel","output_descr","input_descr'
      + '","output_usage","input_usage","out_polarity","out_duration","out_inte'
      + 'rval","out_delay","out_num_pulses","in_polarity","in_min_duration","in'
      + '_max_duration","in_max_delay"],"description":"Trigger-Timestamping-Uni'
      + 't/IP configuration","title":"TTIP Configuration","$schema":"http://jso'
      + 'n-schema.org/draft-04/schema#","properties":{"dmi_min_time":{"descript'
      + 'ion":"dmi, minimum time interval between trigger events [0.1 '
      + 'msec]","minimum":0,"type":"integer"},"dmi_incr":{"description":"dmi, '
      + 'increment in '
      + 'ticks","type":"integer"},"out_duration":{"description":"output pulse '
      + 'duration [0.1 msec]","minimum":0,"type":"integer"},"in_min_duration":{'
      + '"description":"input signals with smaller pulse durations are ignored '
      + '[0.1 msec]","minimum":0,"type":"integer"},"in_max_delay":{"description'
      + '":"maximum delay to output pulse before fake event is generated [0.1 '
      + 'msec], zero indicates that no fake events are generated","minimum":0,"'
      + 'type":"integer"},"in_max_duration":{"description":"stops measurement '
      + 'of pulse duration of input signal [0.1 '
      + 'msec]","minimum":0,"type":"integer"},"in_polarity":{"description":"0 '
      + '.. positive edge, 1 .. negative '
      + 'edge","minimum":0,"type":"integer"},"input_usage":{"description":"0 ..'
      + ' deactivated, 1 .. just detecting and timestamping","minimum":0,"type"'
      + ':"integer"},"channel":{"description":"ID of input/output channel","min'
      + 'imum":0,"type":"integer"},"out_delay":{"description":"output pulse '
      + 'initial delay after start [0.1 msec]","minimum":0,"type":"integer"},"o'
      + 'ut_interval":{"description":"output pulse interval [0.1 '
      + 'msec]","minimum":0,"type":"integer"},"out_polarity":{"description":"0 '
      + '.. positive edge, 1 .. negative edge","minimum":0,"type":"integer"},"o'
      + 'ut_num_pulses":{"description":"number of output pulses to be '
      + 'generated, 0 .. infinite","minimum":0,"type":"integer"},"dmi_max_time"'
      + ':{"description":"dmi, maximum time interval between trigger events '
      + '[0.1 '
      + 'msec]","minimum":0,"type":"integer"},"output_usage":{"description":"0 '
      + '.. no output, 1 .. output derived from internal clock, 2 .. output '
      + 'derived from dmi, 3 .. output derived from external signal, 4 .. '
      + 'output static low, 5 .. output static high","minimum":0,"type":"intege'
      + 'r"},"ttip_version":{"description":"following main.sub.ss.sss","minimum'
      + '":0,"type":"integer"},"dmi_dist_per_tick":{"description":"dmi, '
      + 'distance per tick, just informative '
      + '[m]","type":"number"},"ext_subdivider":{"description":"reduces the '
      + 'frequency, default '
      + '1","minimum":0,"type":"integer"},"ext_channel":{"description":"ID of '
      + 'channel used as external trigger input, 32 indicates none","minimum":0'
      + ',"type":"integer"},"ext_bitmask":{"description":"defines which of the '
      + 'internal pulse generators are to be started","minimum":0,"type":"integ'
      + 'er"},"output_descr":{"description":"descriptive '
      + 'string","type":"string"},"ext_delay":{"description":"external trigger,'
      + ' time delay [0.1 msec]","minimum":0,"type":"integer"},"num_channel":{"'
      + 'description":"number of input/output channels","minimum":0,"type":"int'
      + 'eger"},"input_descr":{"description":"descriptive '
      + 'string","type":"string"},"ext_signal":{"description":"0 .. use input '
      + 'signal, 1 .. use output signal of '
      + 'channel","minimum":0,"type":"integer"}},"type":"object"}'
    );

    Example : (''
      + '{"dmi_min_time":0,"dmi_incr":0,"out_duration":300,"in_min_duration":0,'
      + '"in_max_delay":9990,"in_max_duration":10000,"in_polarity":0,"input_usa'
      + 'ge":1,"channel":0,"out_delay":0,"out_interval":10000,"out_polarity":0,'
      + '"out_num_pulses":1,"dmi_max_time":0,"output_usage":1,"ttip_version":1,'
      + '"dmi_dist_per_tick":0.0,"ext_subdivider":1,"ext_channel":32,"ext_bitma'
      + 'sk":0,"output_descr":"Port 1 - '
      + 'Trigger","ext_delay":0,"num_channel":9,"input_descr":"Port 1 - '
      + 'Exposure","ext_signal":0}'
    );
  );

// Details about vertex file
const
  RDB_RIEGL_VERTEX_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.vertex_info';
    Title       : 'Vertex Information';
    Description : 'Details about vertex file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"description":"Details about vertex file","properties":{"vertex_file"'
      + ':{"type":"object","required":["file_extension"],"properties":{"file_ex'
      + 'tension":{"description":"Vertex file extension, without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC 4122)","type":"string"}}}},"$schema":"http://js'
      + 'on-schema.org/draft-04/schema#","type":"object","title":"Vertex '
      + 'Information"}'
    );

    Example : (''
      + '{"vertex_file":{"file_extension":"vtx","file_uuid":"51534d95-d71f-4f36'
      + '-ae1a-3e63a21fd1c7"}}'
    );
  );

// Details about the voxels contained in the file
const
  RDB_RIEGL_VOXEL_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.voxel_info';
    Title       : 'Voxel Information';
    Description : 'Details about the voxels contained in the file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"definitions":{"voxel_size_cubic":{"$ref":"#/definitions/edge_length"'
      + ',"type":"number"},"shape_thresholds":{"description":"Thresholds used '
      + 'to compute the voxel''s shape_id value.","type":"object","required":["p'
      + 'lane","line"],"properties":{"line":{"description":"If the biggest '
      + 'eigenvalue is bigger than the median eigenvalue * line_threshold, the '
      + 'voxel is considered a line.","minimum":1,"type":"number","exclusiveMin'
      + 'imum":true},"plane":{"exclusiveMaximum":true,"description":"If the '
      + 'smallest eigenvalue is smaller than the median eigenvalue * '
      + 'plane_threshold, the voxel is considered a plane.","maximum":1,"minimu'
      + 'm":0,"type":"number","exclusiveMinimum":true}}},"voxel_type":{"descrip'
      + 'tion":"Whether a point in a voxel represents its center or its '
      + 'centroid. If type is <tt>index</tt> there is no point but only an '
      + 'integer voxel index.","enum":["center","centroid","index"],"default":"'
      + 'centroid"},"edge_length":{"description":"Length of voxel edge [m].","m'
      + 'inimum":0,"type":"number","exclusiveMinimum":true},"voxel_size":{"item'
      + 's":{"$ref":"#/definitions/edge_length"},"description":"Size of voxels.'
      + '","type":"array","minItems":3,"maxItems":3},"voxel_origin_enum":{"desc'
      + 'ription":"Defines whether the voxel''s center or a corner is placed on '
      + 'CRS origin <tt>(0/0/0)</tt>.","enum":["center","corner"],"default":"co'
      + 'rner"},"voxel_origin_point":{"items":{"type":"number"},"description":"'
      + 'Origin point for all voxel indices in voxel CRS.","type":"array","minI'
      + 'tems":3,"maxItems":3}},"oneOf":[{"properties":{"voxel_origin":{"$ref":'
      + '"#/definitions/voxel_origin_enum"},"shape_thresholds":{"$ref":"#/defin'
      + 'itions/shape_thresholds"},"voxel_type":{"$ref":"#/definitions/voxel_ty'
      + 'pe"},"size":{"oneOf":[{"$ref":"#/definitions/voxel_size"},{"$ref":"#/d'
      + 'efinitions/voxel_size_cubic"}],"description":"Size of voxels in file '
      + 'coordinate system."}},"additionalProperties":false,"required":["size",'
      + '"voxel_origin","voxel_type"]},{"properties":{"reference_point":{"items'
      + '":{"maximum":180,"minimum":-180,"type":"number"},"description":"Point '
      + 'in WGS84 geodetic decimal degree (EPSG:4326) that was used to compute '
      + 'the projection distortion parameters. The coefficient order is '
      + 'latitude, longitude. Only voxels with corresponding geo_tag, '
      + 'voxel_size and reference_point can be reliably processed together. '
      + 'This entry is available for voxel files in projected CRS only.","type"'
      + ':"array","minItems":2,"maxItems":2},"shape_thresholds":{"$ref":"#/defi'
      + 'nitions/shape_thresholds"},"voxel_type":{"$ref":"#/definitions/voxel_t'
      + 'ype"},"voxel_origin":{"oneOf":[{"$ref":"#/definitions/voxel_origin_enu'
      + 'm"},{"$ref":"#/definitions/voxel_origin_point","description":"The base'
      + ' point of the voxel grid. Used together with <tt>voxel_size</tt> and '
      + '<tt>voxel_index</tt> to compute actual point coordinates."}]},"size_ll'
      + 'cs":{"$ref":"#/definitions/voxel_size","description":"Size of voxels '
      + 'in a locally levelled cartesian coordinate system (xyz). This is only '
      + 'used for voxels based on a map projection."},"size":{"$ref":"#/definit'
      + 'ions/voxel_size","description":"Size of voxels in file coordinate syst'
      + 'em."}},"additionalProperties":false,"required":["reference_point","siz'
      + 'e_llcs","size","voxel_origin","voxel_type"]}],"description":"Details '
      + 'about the voxels contained in the file","$schema":"http://json-schema.'
      + 'org/draft-04/schema#","type":"object","title":"Voxel Information"}'
    );

    Example : (''
      + '{"reference_point":[48,16],"shape_thresholds":{"line":6,"plane":0.16},'
      + '"voxel_type":"centroid","voxel_origin":"corner","size_llcs":[0.5156575'
      + '252891171,0.5130835356683303,0.5143705304787237],"size":[0.59716428347'
      + '79395,0.5971642834779395,0.5143705304787237]}'
    );
  );

// Settings for waveform averaging
const
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.waveform_averaging_settings';
    Title       : 'Waveform Averaging Settings';
    Description : 'Settings for waveform averaging';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["num_shots","mta_zone"],"description":"Settings for '
      + 'waveform averaging","title":"Waveform Averaging Settings","$schema":"h'
      + 'ttp://json-schema.org/draft-04/schema#","properties":{"num_shots":{"de'
      + 'scription":"Number of consecutive shots to be used for averaging.","mi'
      + 'nimum":1,"type":"integer"},"mta_zone":{"description":"Fixed MTA zone '
      + 'for averaging.","minimum":1,"type":"integer"},"trim":{"description":"P'
      + 'ercentage for robust averaging.","maximum":0.5,"minimum":0,"type":"num'
      + 'ber","default":0}},"type":"object"}'
    );

    Example : (''
      + '{"num_shots":7,"mta_zone":1,"trim":0.05}'
    );
  );

// Details about waveform files
const
  RDB_RIEGL_WAVEFORM_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.waveform_info';
    Title       : 'Waveform Information';
    Description : 'Details about waveform files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"range_offset_waveform_samples_m":{"description":"Calib'
      + 'rated device specific range offset for waveform samples in '
      + 'meters.","type":"number"},"range_offset_m":{"description":"Calibrated '
      + 'device specific range offset for waveform analysis by system response '
      + 'fitting in meters.","type":"number"},"sample_data_files":{"items":{"ty'
      + 'pe":"object","required":["channel","channel_name","sample_interval","s'
      + 'ample_bits","laser_wavelength","delta_st","file_extension"],"propertie'
      + 's":{"file_extension":{"description":"Sample data file extension, '
      + 'without the leading dot","type":"string"},"delta_st":{"description":"r'
      + 'eserved","type":"number"},"channel":{"exclusiveMaximum":false,"descrip'
      + 'tion":"Sample block channel number (255 = invalid)","maximum":255,"min'
      + 'imum":0,"type":"integer","exclusiveMinimum":false},"laser_wavelength":'
      + '{"description":"Laser wavelength in meters (0 = unknown)","minimum":0,'
      + '"type":"number","exclusiveMinimum":false},"channel_name":{"description'
      + '":"Sample block channel '
      + 'name","type":"string"},"sample_interval":{"description":"Sampling '
      + 'interval in seconds","minimum":0,"type":"number","exclusiveMinimum":fa'
      + 'lse},"file_uuid":{"description":"File''s Universally Unique Identifier '
      + '(RFC 4122)","type":"string"},"sample_bits":{"exclusiveMaximum":false,"'
      + 'description":"Bitwidth of samples (e.g. 10 bit, 12 bit)","maximum":32,'
      + '"minimum":0,"type":"integer","exclusiveMinimum":false}}},"type":"array'
      + '"},"sample_block_file":{"type":"object","required":["file_extension"],'
      + '"properties":{"file_extension":{"description":"Sample block file '
      + 'extension, without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC 4122)","type":"string"}}}},"required":["sample_'
      + 'block_file","sample_data_files"],"description":"Details about waveform'
      + ' files","$schema":"http://json-schema.org/draft-04/schema#","type":"ob'
      + 'ject","title":"Waveform Information"}'
    );

    Example : (''
      + '{"range_offset_waveform_samples_m ":7.283,"range_offset_m":3.1415,"sam'
      + 'ple_data_files":[{"file_extension":"sp0","delta_st":0,"channel":0,"las'
      + 'er_wavelength":0,"channel_name":"high_power","sample_interval":1.00503'
      + 'e-09,"file_uuid":"da084413-e3e8-4655-a122-071de8490d8e","sample_bits":'
      + '12},{"file_extension":"sp1","delta_st":0,"channel":1,"laser_wavelength'
      + '":0,"channel_name":"low_power","sample_interval":1.00503e-09,"file_uui'
      + 'd":"93585b5e-5ea9-43a1-947b-e7ba3be642d2","sample_bits":12},{"file_ext'
      + 'ension":"sp5","delta_st":0,"channel":5,"laser_wavelength":0,"channel_n'
      + 'ame":"wwf","sample_interval":1.00503e-09,"file_uuid":"9d2298c4-1036-46'
      + '4f-b5cb-1cf8e517f3a0","sample_bits":12}],"sample_block_file":{"file_ex'
      + 'tension":"sbx","file_uuid":"93a03615-66c0-4bea-8ff8-c577378fe660"}}'
    );
  );

// Scanner settings for waveform output
const
  RDB_RIEGL_WAVEFORM_SETTINGS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.waveform_settings';
    Title       : 'Waveform Settings';
    Description : 'Scanner settings for waveform output';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"items":{"type":"object","required":["sbl_name","enabled","channel_id'
      + 'x_mask"],"properties":{"pass_dev_less":{"description":"Threshold for '
      + 'deviation less than '
      + '[1]","type":"integer"},"logic_expression":{"description":"Logic '
      + 'expression of smart waveforms '
      + 'filter","type":"string"},"smart_enabled":{"description":"Smart '
      + 'waveform output '
      + 'enabled","type":"boolean"},"enabled":{"description":"Waveform output e'
      + 'nabled","type":"boolean"},"pass_dev_greater":{"description":"Threshold'
      + ' for deviation greater than '
      + '[1]","type":"integer"},"pass_ampl_less":{"description":"Threshold for '
      + 'amplitude less than '
      + '[dB]","type":"number"},"channel_idx_mask":{"description":"Bit mask for'
      + ' channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = '
      + 'Bit1, ...","type":"integer"},"sbl_name":{"description":"Name of sample'
      + ' block, e.g.: wfm, '
      + 'wwf","type":"string"},"pass_rng_less":{"description":"Threshold for '
      + 'range less than '
      + '[m]","type":"number"},"pass_ampl_greater":{"description":"Threshold '
      + 'for amplitude greater than '
      + '[dB]","type":"number"},"pass_rng_greater":{"description":"Threshold '
      + 'for range greater than [m]","type":"number"}}},"description":"Scanner '
      + 'settings for waveform output","$schema":"http://json-schema.org/draft-'
      + '04/schema#","type":"array","title":"Waveform Settings"}'
    );

    Example : (''
      + '[{"smart_enabled":true,"enabled":true,"pass_ampl_less":5.0,"channel_id'
      + 'x_mask":11,"sbl_name":"wfm","pass_rng_less":13.11,"pass_ampl_greater":'
      + '1.0,"pass_rng_greater":9.27},{"enabled":false,"channel_idx_mask":32,"s'
      + 'bl_name":"wwf"}]'
    );
  );

// Window analysis data estimated from scandata and resulting filter parameters
const
  RDB_RIEGL_WINDOW_ANALYSIS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.window_analysis';
    Title       : 'Window Analysis';
    Description : 'Window analysis data estimated from scandata and resulting filter parameters';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["result","filter","settings"],"description":"Window '
      + 'analysis data estimated from scandata and resulting filter '
      + 'parameters","title":"Window Analysis","$schema":"http://json-schema.or'
      + 'g/draft-04/schema#","properties":{"result":{"type":"object","required"'
      + ':["angle","range_mean","range_sigma","amplitude_mean","amplitude_sigma'
      + '","amplitude_offset"],"properties":{"amplitude_sigma":{"items":{"type"'
      + ':"number"},"description":"[dB]","type":"array"},"range_sigma":{"items"'
      + ':{"type":"number"},"description":"[m]","type":"array"},"angle":{"items'
      + '":{"type":"number"},"description":"[deg]","type":"array"},"timestamp":'
      + '{"items":{"type":"number"},"description":"[s]","type":"array"},"range_'
      + 'mean":{"items":{"type":"number"},"description":"[m]","type":"array"},"'
      + 'amplitude_offset":{"items":{"type":"number"},"description":"[dB]","typ'
      + 'e":"array"},"amplitude_mean":{"items":{"type":"number"},"description":'
      + '"[dB]","type":"array"}}},"filter":{"type":"object","required":["angle"'
      + ',"range_min","range_max","amplitude_max"],"properties":{"amplitude_max'
      + '":{"items":{"type":"number"},"description":"[dB]","type":"array"},"ang'
      + 'le":{"items":{"type":"number"},"description":"[deg]","type":"array"},"'
      + 'range_max":{"items":{"type":"number"},"description":"[m]","type":"arra'
      + 'y"},"range_min":{"items":{"type":"number"},"description":"[m]","type":'
      + '"array"}}},"settings":{"type":"object","required":["range","amplitude"'
      + '],"properties":{"range":{"type":"object","required":["sigma_factor","a'
      + 'dditive_value"],"properties":{"sigma_factor":{"type":"number"},"additi'
      + 've_value":{"type":"number"}}},"amplitude":{"type":"object","required":'
      + '["sigma_factor","additive_value"],"properties":{"sigma_factor":{"type"'
      + ':"number"},"additive_value":{"type":"number"}}}}}},"type":"object"}'
    );

    Example : (''
      + '{"result":{"amplitude_sigma":[0.4272844,0.4298479,0.4236816,0.4283583,'
      + '0.4362353,0.4315141,0.4373984,0.4472798,0.4346001,0.4345487,0.4540681]'
      + ',"range_sigma":[0.01869652,0.02151435,0.01747969,0.01918765,0.01945776'
      + ',0.01934862,0.01955329,0.02225589,0.02229977,0.01899122,0.02009433],"a'
      + 'ngle":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],"timest'
      + 'amp":[408.4441,411.4443],"range_mean":[0.1105621,0.1079564,0.1087088,0'
      + '.1067261,0.1054582,0.1090412,0.102871,0.1019044,0.1051523,0.1058445,0.'
      + '1031261],"amplitude_offset":[1.9,1.9],"amplitude_mean":[5.347396,5.263'
      + '155,5.224655,5.179926,5.097782,5.116479,5.051756,4.983473,5.007885,5.0'
      + '02441,4.982]},"filter":{"amplitude_max":[8.04,8.01,7.99,7.96,7.93,7.9,'
      + '7.88,7.85,7.83,7.8,7.78],"angle":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,1'
      + '5.6,15.7,15.8,15.9],"range_max":[0.424,0.425,0.426,0.427,0.428,0.428,0'
      + '.429,0.43,0.431,0.431,0.432],"range_min":[-0.208,-0.21,-0.212,-0.214,-'
      + '0.216,-0.218,-0.219,-0.221,-0.223,-0.225,-0.227]},"settings":{"range":'
      + '{"sigma_factor":8,"additive_value":0.1},"amplitude":{"sigma_factor":4,'
      + '"additive_value":1.0}}}'
    );
  );

// Correction parameters for window glass echoes
const
  RDB_RIEGL_WINDOW_ECHO_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.window_echo_correction';
    Title       : 'Window Echo Correction';
    Description : 'Correction parameters for window glass echoes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"required":["amplitude","range","slices"],"description":"Correction '
      + 'parameters for window glass echoes","title":"Window Echo Correction","'
      + '$schema":"http://json-schema.org/draft-04/schema#","properties":{"rang'
      + 'e":{"description":"Range axis of correction table","type":"object","re'
      + 'quired":["minimum","maximum","entries"],"properties":{"entries":{"desc'
      + 'ription":"Number of range entries","minimum":1,"type":"integer"},"maxi'
      + 'mum":{"description":"Maximum range in m","maximum":2.0,"minimum":-2.0,'
      + '"type":"number"},"minimum":{"description":"Minimum range in m","maximu'
      + 'm":2.0,"minimum":-2.0,"type":"number"}}},"amplitude":{"description":"A'
      + 'mplitude axis of correction table","type":"object","required":["minimu'
      + 'm","maximum","entries"],"properties":{"entries":{"description":"Number'
      + ' of amplitude entries","minimum":1,"type":"integer"},"maximum":{"descr'
      + 'iption":"Maximum amplitude in '
      + 'dB","minimum":0.0,"type":"number"},"minimum":{"description":"Minimum '
      + 'amplitude in dB","minimum":0.0,"type":"number"}}},"slices":{"items":{"'
      + 'description":"Window echo correction parameter slice","type":"object",'
      + '"required":["amplitude","table"],"properties":{"amplitude":{"descripti'
      + 'on":"Window echo amplitude of slice in dB","type":"number"},"table":{"'
      + 'items":{"items":{"items":{"description":"Table cell (item 0: amplitude'
      + ' in dB, 1: range in m, 2: '
      + 'flags)","type":"number"},"description":"Table column (= range '
      + 'axis)","type":"array","minItems":3,"maxItems":3},"description":"Table '
      + 'row (= amplitude '
      + 'axis)","type":"array","minItems":1},"description":"Correction table '
      + '(dimension defined by the ''amplitude'' and ''range'' objects)","type":"ar'
      + 'ray","minItems":1}}},"type":"array"}},"type":"object"}'
    );

    Example : (''
      + '{"range":{"entries":128,"maximum":1.5060822940732335,"minimum":-1.5060'
      + '822940732335},"amplitude":{"entries":128,"maximum":20,"minimum":2},"sl'
      + 'ices":[{"amplitude":1.5,"table":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]'
      + '},{"amplitude":2.0,"table":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]]}]}'
    );
  );

// Table of all meta data item details records
const
  RDB_META_DATA_ITEMS : array of PRDBPointcloudMetaDataItemInfo = nil; // filled during initialization

implementation

procedure FixStringEncoding;
begin
  // As the unit (.pas file) doesn't contain a BOM, we need to convert all strings
  // from UTF-8 to "native" encoding at runtime (tested on Delphi 7 and Delphi XE5)

  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Name)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Title)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Description)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Schema)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Example)));

  RDB_RIEGL_ATMOSPHERE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Name)));
  RDB_RIEGL_ATMOSPHERE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Title)));
  RDB_RIEGL_ATMOSPHERE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Description)));
  RDB_RIEGL_ATMOSPHERE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Schema)));
  RDB_RIEGL_ATMOSPHERE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Example)));

  RDB_RIEGL_BEAM_GEOMETRY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Name)));
  RDB_RIEGL_BEAM_GEOMETRY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Title)));
  RDB_RIEGL_BEAM_GEOMETRY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Description)));
  RDB_RIEGL_BEAM_GEOMETRY.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Schema)));
  RDB_RIEGL_BEAM_GEOMETRY.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Example)));

  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Name)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Title)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Description)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Schema)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Example)));

  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Name)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Title)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Description)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Schema)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Example)));

  RDB_RIEGL_DETECTION_PROBABILITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Name)));
  RDB_RIEGL_DETECTION_PROBABILITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Title)));
  RDB_RIEGL_DETECTION_PROBABILITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Description)));
  RDB_RIEGL_DETECTION_PROBABILITY.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Schema)));
  RDB_RIEGL_DETECTION_PROBABILITY.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Example)));

  RDB_RIEGL_DEVICE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Name)));
  RDB_RIEGL_DEVICE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Title)));
  RDB_RIEGL_DEVICE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Description)));
  RDB_RIEGL_DEVICE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Schema)));
  RDB_RIEGL_DEVICE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Example)));

  RDB_RIEGL_DEVICE_GEOMETRY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Name)));
  RDB_RIEGL_DEVICE_GEOMETRY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Title)));
  RDB_RIEGL_DEVICE_GEOMETRY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Description)));
  RDB_RIEGL_DEVICE_GEOMETRY.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Schema)));
  RDB_RIEGL_DEVICE_GEOMETRY.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Example)));

  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Name)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Title)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Description)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Schema)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Example)));

  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Name)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Title)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Description)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Schema)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Example)));

  RDB_RIEGL_DEVICES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Name)));
  RDB_RIEGL_DEVICES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Title)));
  RDB_RIEGL_DEVICES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Description)));
  RDB_RIEGL_DEVICES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Schema)));
  RDB_RIEGL_DEVICES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Example)));

  RDB_RIEGL_DMI_CONFIG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Name)));
  RDB_RIEGL_DMI_CONFIG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Title)));
  RDB_RIEGL_DMI_CONFIG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Description)));
  RDB_RIEGL_DMI_CONFIG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Schema)));
  RDB_RIEGL_DMI_CONFIG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Example)));

  RDB_RIEGL_DYNTRIG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Name)));
  RDB_RIEGL_DYNTRIG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Title)));
  RDB_RIEGL_DYNTRIG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Description)));
  RDB_RIEGL_DYNTRIG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Schema)));
  RDB_RIEGL_DYNTRIG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Example)));

  RDB_RIEGL_ECHO_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Name)));
  RDB_RIEGL_ECHO_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Title)));
  RDB_RIEGL_ECHO_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Description)));
  RDB_RIEGL_ECHO_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Schema)));
  RDB_RIEGL_ECHO_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Example)));

  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Name)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Title)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Description)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Schema)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Example)));

  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Name)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Title)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Description)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Schema)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Example)));

  RDB_RIEGL_GEO_TAG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Name)));
  RDB_RIEGL_GEO_TAG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Title)));
  RDB_RIEGL_GEO_TAG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Description)));
  RDB_RIEGL_GEO_TAG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Schema)));
  RDB_RIEGL_GEO_TAG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Example)));

  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Name)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Title)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Description)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Schema)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Example)));

  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Name)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Title)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Description)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Schema)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Example)));

  RDB_RIEGL_GNSS_PROCESSING_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Name)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Title)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Description)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Schema)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Example)));

  RDB_RIEGL_ITEM_NAMES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Name)));
  RDB_RIEGL_ITEM_NAMES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Title)));
  RDB_RIEGL_ITEM_NAMES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Description)));
  RDB_RIEGL_ITEM_NAMES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Schema)));
  RDB_RIEGL_ITEM_NAMES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Example)));

  RDB_RIEGL_LICENSES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Name)));
  RDB_RIEGL_LICENSES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Title)));
  RDB_RIEGL_LICENSES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Description)));
  RDB_RIEGL_LICENSES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Schema)));
  RDB_RIEGL_LICENSES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Example)));

  RDB_RIEGL_MTA_SETTINGS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Name)));
  RDB_RIEGL_MTA_SETTINGS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Title)));
  RDB_RIEGL_MTA_SETTINGS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Description)));
  RDB_RIEGL_MTA_SETTINGS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Schema)));
  RDB_RIEGL_MTA_SETTINGS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Example)));

  RDB_RIEGL_NEAR_RANGE_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Name)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Title)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Description)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Schema)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Example)));

  RDB_RIEGL_NOISE_ESTIMATES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Name)));
  RDB_RIEGL_NOISE_ESTIMATES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Title)));
  RDB_RIEGL_NOISE_ESTIMATES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Description)));
  RDB_RIEGL_NOISE_ESTIMATES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Schema)));
  RDB_RIEGL_NOISE_ESTIMATES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Example)));

  RDB_RIEGL_NOTCH_FILTER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Name)));
  RDB_RIEGL_NOTCH_FILTER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Title)));
  RDB_RIEGL_NOTCH_FILTER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Description)));
  RDB_RIEGL_NOTCH_FILTER.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Schema)));
  RDB_RIEGL_NOTCH_FILTER.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Example)));

  RDB_RIEGL_PIXEL_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Name)));
  RDB_RIEGL_PIXEL_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Title)));
  RDB_RIEGL_PIXEL_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Description)));
  RDB_RIEGL_PIXEL_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Schema)));
  RDB_RIEGL_PIXEL_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Example)));

  RDB_RIEGL_PLANE_PATCH_MATCHING.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Name)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Title)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Description)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Schema)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Example)));

  RDB_RIEGL_PLANE_PATCH_STATISTICS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Name)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Title)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Description)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Schema)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Example)));

  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Name)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Title)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Description)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Schema)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Example)));

  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Name)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Title)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Description)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Schema)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Example)));

  RDB_RIEGL_POINTCLOUD_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Name)));
  RDB_RIEGL_POINTCLOUD_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Title)));
  RDB_RIEGL_POINTCLOUD_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Description)));
  RDB_RIEGL_POINTCLOUD_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Schema)));
  RDB_RIEGL_POINTCLOUD_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Example)));

  RDB_RIEGL_POSE_ESTIMATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Name)));
  RDB_RIEGL_POSE_ESTIMATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Title)));
  RDB_RIEGL_POSE_ESTIMATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Description)));
  RDB_RIEGL_POSE_ESTIMATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Schema)));
  RDB_RIEGL_POSE_ESTIMATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Example)));

  RDB_RIEGL_POSE_SENSORS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Name)));
  RDB_RIEGL_POSE_SENSORS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Title)));
  RDB_RIEGL_POSE_SENSORS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Description)));
  RDB_RIEGL_POSE_SENSORS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Schema)));
  RDB_RIEGL_POSE_SENSORS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Example)));

  RDB_RIEGL_PULSE_POSITION_MODULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Name)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Title)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Description)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Schema)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Example)));

  RDB_RIEGL_RANGE_STATISTICS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Name)));
  RDB_RIEGL_RANGE_STATISTICS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Title)));
  RDB_RIEGL_RANGE_STATISTICS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Description)));
  RDB_RIEGL_RANGE_STATISTICS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Schema)));
  RDB_RIEGL_RANGE_STATISTICS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Example)));

  RDB_RIEGL_RECEIVER_INTERNALS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Name)));
  RDB_RIEGL_RECEIVER_INTERNALS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Title)));
  RDB_RIEGL_RECEIVER_INTERNALS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Description)));
  RDB_RIEGL_RECEIVER_INTERNALS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Schema)));
  RDB_RIEGL_RECEIVER_INTERNALS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Example)));

  RDB_RIEGL_REFLECTANCE_CALCULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Name)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Title)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Description)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Schema)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Example)));

  RDB_RIEGL_REFLECTANCE_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Name)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Title)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Description)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Schema)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Example)));

  RDB_RIEGL_RILOC_PROCESSING_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Name)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Title)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Description)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Schema)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Example)));

  RDB_RIEGL_SCAN_PATTERN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Name)));
  RDB_RIEGL_SCAN_PATTERN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Title)));
  RDB_RIEGL_SCAN_PATTERN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Description)));
  RDB_RIEGL_SCAN_PATTERN.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Schema)));
  RDB_RIEGL_SCAN_PATTERN.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Example)));

  RDB_RIEGL_SHOT_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Name)));
  RDB_RIEGL_SHOT_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Title)));
  RDB_RIEGL_SHOT_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Description)));
  RDB_RIEGL_SHOT_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Schema)));
  RDB_RIEGL_SHOT_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Example)));

  RDB_RIEGL_STORED_FILTERS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Name)));
  RDB_RIEGL_STORED_FILTERS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Title)));
  RDB_RIEGL_STORED_FILTERS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Description)));
  RDB_RIEGL_STORED_FILTERS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Schema)));
  RDB_RIEGL_STORED_FILTERS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Example)));

  RDB_RIEGL_SYSTEM_DESCRIPTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Name)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Title)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Description)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Schema)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Example)));

  RDB_RIEGL_TEMPERATURE_CALCULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Name)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Title)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Description)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Schema)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Example)));

  RDB_RIEGL_TIME_BASE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Name)));
  RDB_RIEGL_TIME_BASE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Title)));
  RDB_RIEGL_TIME_BASE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Description)));
  RDB_RIEGL_TIME_BASE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Schema)));
  RDB_RIEGL_TIME_BASE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Example)));

  RDB_RIEGL_TRAJECTORY_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Name)));
  RDB_RIEGL_TRAJECTORY_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Title)));
  RDB_RIEGL_TRAJECTORY_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Description)));
  RDB_RIEGL_TRAJECTORY_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Schema)));
  RDB_RIEGL_TRAJECTORY_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Example)));

  RDB_RIEGL_TTIP_CONFIGURATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Name)));
  RDB_RIEGL_TTIP_CONFIGURATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Title)));
  RDB_RIEGL_TTIP_CONFIGURATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Description)));
  RDB_RIEGL_TTIP_CONFIGURATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Schema)));
  RDB_RIEGL_TTIP_CONFIGURATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Example)));

  RDB_RIEGL_VERTEX_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Name)));
  RDB_RIEGL_VERTEX_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Title)));
  RDB_RIEGL_VERTEX_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Description)));
  RDB_RIEGL_VERTEX_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Schema)));
  RDB_RIEGL_VERTEX_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Example)));

  RDB_RIEGL_VOXEL_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Name)));
  RDB_RIEGL_VOXEL_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Title)));
  RDB_RIEGL_VOXEL_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Description)));
  RDB_RIEGL_VOXEL_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Schema)));
  RDB_RIEGL_VOXEL_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Example)));

  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Name)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Title)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Description)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Schema)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Example)));

  RDB_RIEGL_WAVEFORM_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Name)));
  RDB_RIEGL_WAVEFORM_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Title)));
  RDB_RIEGL_WAVEFORM_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Description)));
  RDB_RIEGL_WAVEFORM_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Schema)));
  RDB_RIEGL_WAVEFORM_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Example)));

  RDB_RIEGL_WAVEFORM_SETTINGS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Name)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Title)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Description)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Schema)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Example)));

  RDB_RIEGL_WINDOW_ANALYSIS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Name)));
  RDB_RIEGL_WINDOW_ANALYSIS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Title)));
  RDB_RIEGL_WINDOW_ANALYSIS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Description)));
  RDB_RIEGL_WINDOW_ANALYSIS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Schema)));
  RDB_RIEGL_WINDOW_ANALYSIS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Example)));

  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Name)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Title)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Description)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Schema)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Example)));
end;

procedure FillMetadataArray;
var
  i : Integer;
begin
  SetLength(RDB_META_DATA_ITEMS, 55); i := -1;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ANGULAR_NOTCH_FILTER;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ATMOSPHERE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_BEAM_GEOMETRY;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_CONTROL_OBJECT_CATALOG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DETECTION_PROBABILITY;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE_GEOMETRY;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE_OUTPUT_LIMITS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DMI_CONFIG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DYNTRIG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ECHO_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_EXPONENTIAL_DECOMPOSITION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GAUSSIAN_DECOMPOSITION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GEO_TAG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GEOMETRIC_SCALE_FACTOR;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GEOREFERENCING_PARAMETERS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GNSS_PROCESSING_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ITEM_NAMES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_LICENSES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_MTA_SETTINGS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_NEAR_RANGE_CORRECTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_NOISE_ESTIMATES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_NOTCH_FILTER;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PIXEL_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PLANE_PATCH_MATCHING;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PLANE_PATCH_STATISTICS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PLANE_SLOPE_CLASS_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POINT_ATTRIBUTE_GROUPS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POINTCLOUD_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POSE_ESTIMATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POSE_SENSORS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PULSE_POSITION_MODULATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_RANGE_STATISTICS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_RECEIVER_INTERNALS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_REFLECTANCE_CALCULATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_REFLECTANCE_CORRECTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_RILOC_PROCESSING_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_SCAN_PATTERN;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_SHOT_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_STORED_FILTERS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_SYSTEM_DESCRIPTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TEMPERATURE_CALCULATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TIME_BASE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TRAJECTORY_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TTIP_CONFIGURATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_VERTEX_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_VOXEL_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WAVEFORM_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WAVEFORM_SETTINGS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WINDOW_ANALYSIS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WINDOW_ECHO_CORRECTION;
end;

initialization

FixStringEncoding;
FillMetadataArray;

end.
