/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    querySelect.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point select query (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-04/AW: Allow to read from multiple nodes at once (#2368)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< STRUCT QuerySelect::Private >--------------------------------------------

struct QuerySelect::Private
{
    RDBContext               *context;
    RDBPointcloud            *pointcloud;
    RDBPointcloudQuerySelect *query;

    Private(
        riegl::rdb::PointcloudData       *pcl,
        const std::vector<GraphNode::ID> *nodes,
        const std::string                &filter
    ):
        context(pcl->contextHandle()),
        pointcloud(pcl->pointcloud),
        query(nullptr)
    {
        if (nodes)
        {
            ErrorImpl::check(context, rdb_pointcloud_query_select_nodes_new(
                context, pointcloud,
                nodes->data(), static_cast<std::uint32_t>(nodes->size()),
                filter.c_str(), &query
            ));
        }
        else
        {
            ErrorImpl::check(context, rdb_pointcloud_query_select_new(
                context, pointcloud, 0, // = all nodes
                filter.c_str(), &query
            ));
        }
    }

    ~Private()
    {
        ErrorImpl::check(context, rdb_pointcloud_query_select_delete(
            context, &query
        ));
    }
};

//---< QuerySelect::PUBLIC >----------------------------------------------------

QuerySelect::QuerySelect()
{
}

QuerySelect::QuerySelect(
    riegl::rdb::PointcloudData       *pointcloud,
    const std::vector<GraphNode::ID> *nodes,
    const std::string                &filter
):
    data(new Private(pointcloud, nodes, filter))
{
}

QuerySelect::operator bool() const
{
    return valid();
}

bool QuerySelect::valid() const
{
    return (data != nullptr);
}

void QuerySelect::close()
{
    data.reset();
}

void QuerySelect::bind(
    const std::string &attribute,
    const DataType     dataType,
    void              *buffer,
    const std::int32_t stride
)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_select_bind(
        data->context, data->query,
        attribute.c_str(),
        dataType, buffer, stride
    ));
}

std::uint32_t QuerySelect::next(std::uint32_t count)
{
    if (!data) return 0;
    std::uint32_t result(0);
    ErrorImpl::check(data->context, rdb_pointcloud_query_select_next(
        data->context, data->query, count, &result
    ));
    return result;
}

std::uint32_t QuerySelect::progress() const
{
    if (!data) return 0;
    std::uint32_t result(0);
    ErrorImpl::check(data->context, rdb_pointcloud_query_select_progress(
        data->context, data->query, &result
    ));
    return result;
}

}}} // namespace riegl::rdb::pointcloud
