/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 *
 *******************************************************************************
 */

#ifndef RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
#define RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87

#include <vector>
#include <string>
#include <cstdlib>
#include <cstdint>

namespace riegl {
namespace rdb {
namespace pointcloud {

// Meta data item definition version
const std::string RDB_META_DATA_ITEMS_VERSION = "1.1.56";
const std::string RDB_META_DATA_ITEMS_DATE = "2025-07-01";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
struct MetaDataItemInfo
{
    std::string   name;
    std::string   title;
    std::string   description;
    unsigned char status;
    std::string   schema;
    std::string   example;

    operator const std::string&() const { return name; }
};

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"amplitude_mean\":{\"description\":\"Mean amplitude [dB]\",\""
    "type\":\"array\",\"items\":{\"type\":\"number\"}},\"range_mean\":{\"description\":\""
    "Mean range [m]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"angle\":{\"de"
    "scription\":\"Angle [deg]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"$"
    "schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"angle\","
    "\"range_mean\",\"amplitude_mean\"],\"title\":\"Angular Notch "
    "Filter\",\"type\":\"object\",\"description\":\"Angular notch filter parameters"
    " for window glass echoes\"}"
    ,
    /* example: */
    "{\"amplitude_mean\":[3.780913,3.780913,3.480913,3.120913,2.850913,2.7209"
    "13,2.680913,2.610913,2.530913,2.570913,2.570913],\"range_mean\":[0.094,0"
    ".094,0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475],\""
    "angle\":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"attenuation\":{\"description\":\"Atmospheric attenuation "
    "[1/km]\",\"type\":\"number\"},\"rel_humidity\":{\"description\":\"Relative "
    "humidity along measurement path "
    "[%]\",\"type\":\"number\"},\"amsl\":{\"description\":\"Height above mean sea "
    "level (AMSL) "
    "[m]\",\"type\":\"number\"},\"pressure_sl\":{\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\",\"type\":\"number\"},\"temperature\":{\"description\":\"Temperature "
    "along measurement path "
    "[\\u00b0C]\",\"type\":\"number\"},\"group_velocity\":{\"description\":\"Group "
    "velocity of laser beam "
    "[m/s]\",\"type\":\"number\"},\"pressure\":{\"description\":\"Pressure along "
    "measurment path "
    "[mbar]\",\"type\":\"number\"},\"wavelength\":{\"description\":\"Laser wavelength"
    " [nm]\",\"type\":\"number\"}},\"$schema\":\"http://json-schema.org/draft-04/sc"
    "hema#\",\"required\":[\"temperature\",\"pressure\",\"rel_humidity\",\"pressure_s"
    "l\",\"amsl\",\"group_velocity\",\"attenuation\",\"wavelength\"],\"title\":\"Atmosp"
    "heric Parameters\",\"type\":\"object\",\"description\":\"Atmospheric "
    "parameters\"}"
    ,
    /* example: */
    "{\"attenuation\":0.028125,\"rel_humidity\":63,\"amsl\":0,\"pressure_sl\":970,\""
    "temperature\":7,\"group_velocity\":299711000.0,\"pressure\":970,\"wavelength"
    "\":1550}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"beam_divergence\":{\"minimum\":0,\"description\":\"Beam "
    "divergence in far field [rad]\",\"type\":\"number\",\"exclusiveMinimum\":fals"
    "e},\"beam_exit_diameter\":{\"minimum\":0,\"description\":\"Beam width at exit"
    " aperture [m]\",\"type\":\"number\",\"exclusiveMinimum\":false}},\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\",\"required\":[\"beam_exit_diamete"
    "r\",\"beam_divergence\"],\"title\":\"Beam "
    "Geometry\",\"type\":\"object\",\"description\":\"Laser beam geometry details\"}"
    ,
    /* example: */
    "{\"beam_divergence\":0.0003,\"beam_exit_diameter\":0.0072}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"types\":{\"uniqueItems\":true,\"type\":\"array\",\"items\":{\"on"
    "eOf\":[{\"$ref\":\"#/definitions/rectangle\"},{\"$ref\":\"#/definitions/checke"
    "rboard2x2\"},{\"$ref\":\"#/definitions/chevron\"},{\"$ref\":\"#/definitions/ci"
    "rcular_disk\"},{\"$ref\":\"#/definitions/cylinder\"},{\"$ref\":\"#/definitions"
    "/sphere\"},{\"$ref\":\"#/definitions/round_corner_cube_prism\"}],\"type\":\"ob"
    "ject\"}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definiti"
    "ons\":{\"common\":{\"properties\":{\"shape\":{\"enum\":[\"rectangle\",\"checkerboa"
    "rd2x2\",\"chevron\",\"circular_disk\",\"cylinder\",\"sphere\",\"round_corner_cub"
    "e_prism\"],\"description\":\"shape identifier\",\"type\":\"string\"},\"surface_t"
    "ype\":{\"enum\":[\"retro_reflective_foil\",\"diffuse\"],\"description\":\"surfac"
    "e material "
    "type\",\"type\":\"string\"},\"name\":{\"minLength\":3,\"description\":\"unique "
    "type identifier\",\"type\":\"string\"},\"description\":{\"description\":\"string"
    " describing the object\",\"type\":\"string\"}},\"description\":\"common object"
    " properties\",\"type\":\"object\",\"required\":[\"name\",\"shape\"]},\"checkerboar"
    "d2x2\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"properties\":{\"shape\""
    ":{\"enum\":[\"checkerboard2x2\"],\"description\":\"shape identifier\",\"type\":\""
    "string\"},\"square_length\":{\"minimum\":0,\"description\":\"length of a "
    "square of the checkerboard in meters\",\"type\":\"number\",\"exclusiveMinimu"
    "m\":true}},\"description\":\"checkerboard specific properties\",\"type\":\"obj"
    "ect\",\"required\":[\"shape\",\"square_length\"]}],\"description\":\"checkerboar"
    "d 2 by 2\",\"type\":\"object\"},\"circular_disk\":{\"allOf\":[{\"$ref\":\"#/defini"
    "tions/common\"},{\"properties\":{\"shape\":{\"enum\":[\"circular_disk\"],\"descr"
    "iption\":\"shape "
    "identifier\",\"type\":\"string\"},\"offset\":{\"description\":\"offset in "
    "meters, e.g. reflector constant (optional)\",\"type\":\"number\"},\"diameter"
    "\":{\"minimum\":0,\"description\":\"diameter in meters\",\"type\":\"number\",\"exc"
    "lusiveMinimum\":true}},\"description\":\"circular disk specific properties"
    "\",\"type\":\"object\",\"required\":[\"shape\",\"diameter\"]}],\"description\":\"cir"
    "cular disk\",\"type\":\"object\"},\"sphere\":{\"allOf\":[{\"$ref\":\"#/definitions"
    "/common\"},{\"properties\":{\"shape\":{\"enum\":[\"sphere\"],\"description\":\"sha"
    "pe identifier\",\"type\":\"string\"},\"diameter\":{\"minimum\":0,\"description\":"
    "\"diameter in meters\",\"type\":\"number\",\"exclusiveMinimum\":true}},\"descri"
    "ption\":\"sphere specific properties\",\"type\":\"object\",\"required\":[\"shape"
    "\",\"diameter\"]}],\"description\":\"sphere\",\"type\":\"object\"},\"cylinder\":{\"a"
    "llOf\":[{\"$ref\":\"#/definitions/common\"},{\"properties\":{\"shape\":{\"enum\":"
    "[\"cylinder\"],\"description\":\"shape identifier\",\"type\":\"string\"},\"diamet"
    "er\":{\"minimum\":0,\"description\":\"diameter in meters\",\"type\":\"number\",\"e"
    "xclusiveMinimum\":true},\"height\":{\"minimum\":0,\"description\":\"height in "
    "meters\",\"type\":\"number\",\"exclusiveMinimum\":true}},\"description\":\"cylin"
    "der specific properties\",\"type\":\"object\",\"required\":[\"shape\",\"diameter"
    "\",\"height\"]}],\"description\":\"cylinder\",\"type\":\"object\"},\"round_corner_"
    "cube_prism\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"properties\":{\""
    "shape\":{\"enum\":[\"round_corner_cube_prism\"],\"description\":\"shape "
    "identifier\",\"type\":\"string\"},\"offset\":{\"description\":\"offset in "
    "meters, e.g. reflector constant (optional)\",\"type\":\"number\"},\"diameter"
    "\":{\"minimum\":0,\"description\":\"diameter in "
    "meters\",\"type\":\"number\",\"exclusiveMinimum\":true}},\"description\":\"round"
    " corner cube prism specific properties\",\"type\":\"object\",\"required\":[\"s"
    "hape\",\"diameter\"]}],\"description\":\"round corner cube prism\",\"type\":\"ob"
    "ject\"},\"rectangle\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"propert"
    "ies\":{\"shape\":{\"enum\":[\"rectangle\"],\"type\":\"string\"},\"length\":{\"minimu"
    "m\":0,\"description\":\"length in meters\",\"type\":\"number\",\"exclusiveMinimu"
    "m\":true},\"width\":{\"minimum\":0,\"description\":\"width in meters\",\"type\":\""
    "number\",\"exclusiveMinimum\":true}},\"description\":\"rectangle specific pr"
    "operties\",\"type\":\"object\",\"required\":[\"shape\",\"length\",\"width\"]}],\"des"
    "cription\":\"rectangle\",\"type\":\"object\"},\"chevron\":{\"allOf\":[{\"$ref\":\"#/"
    "definitions/common\"},{\"properties\":{\"shape\":{\"enum\":[\"chevron\"],\"descr"
    "iption\":\"shape identifier\",\"type\":\"string\"},\"outside_edge_length\":{\"mi"
    "nimum\":0,\"description\":\"length of the two outer edges in meters\",\"type"
    "\":\"number\",\"exclusiveMinimum\":true},\"thickness\":{\"minimum\":0,\"descript"
    "ion\":\"thickness in meters\",\"type\":\"number\",\"exclusiveMinimum\":true}},\""
    "description\":\"chevron specific properties\",\"type\":\"object\",\"required\":"
    "[\"shape\",\"outside_edge_length\",\"thickness\"]}],\"description\":\"chevron\","
    "\"type\":\"object\"}},\"required\":[\"types\"],\"title\":\"Control Object "
    "Catalog\",\"type\":\"object\",\"description\":\"List of control object type "
    "definitions\"}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or 'diffuse')\"],\"types\":[{\"shape\":\"rec"
    "tangle\",\"length\":0.6,\"width\":0.3,\"name\":\"Rectangle "
    "60x30\",\"description\":\"Rectangle (60cm x 30cm)\"},{\"shape\":\"rectangle\",\""
    "length\":0.8,\"width\":0.4,\"name\":\"Rectangle "
    "80x40\",\"description\":\"Rectangle (80cm x 40cm)\"},{\"shape\":\"checkerboard"
    "2x2\",\"square_length\":0.3,\"name\":\"Checkerboard2x2 "
    "30\",\"description\":\"Checkerboard (square length: 30cm)\"},{\"shape\":\"chec"
    "kerboard2x2\",\"square_length\":0.5,\"name\":\"Checkerboard2x2 "
    "50\",\"description\":\"Checkerboard (square length: 50cm)\"},{\"shape\":\"chev"
    "ron\",\"outside_edge_length\":0.6096,\"thickness\":0.1016,\"name\":\"Chevron "
    "24''/4''\",\"description\":\"Chevron (a=24''; b=4'')\"},{\"shape\":\"circular_"
    "disk\",\"surface_type\":\"diffuse\",\"name\":\"Circular Disk "
    "50\",\"diameter\":0.5,\"description\":\" Circular Disk (diameter: "
    "50cm)\"},{\"surface_type\":\"retro_reflective_foil\",\"name\":\"RIEGL flat "
    "reflector 50 "
    "mm\",\"diameter\":0.05,\"shape\":\"circular_disk\",\"description\":\"flat "
    "circular reflector from retro-reflective foil\",\"offset\":0.0},{\"surface"
    "_type\":\"retro_reflective_foil\",\"name\":\"RIEGL flat reflector 100 "
    "mm\",\"diameter\":0.1,\"shape\":\"circular_disk\",\"description\":\"flat "
    "circular reflector from retro-reflective foil\",\"offset\":0.0},{\"surface"
    "_type\":\"retro_reflective_foil\",\"name\":\"RIEGL flat reflector 150 "
    "mm\",\"diameter\":0.15,\"shape\":\"circular_disk\",\"description\":\"flat "
    "circular reflector from retro-reflective foil\",\"offset\":0.0},{\"surface"
    "_type\":\"retro_reflective_foil\",\"name\":\"RIEGL cylindrical reflector 50 "
    "mm\",\"diameter\":0.05,\"shape\":\"cylinder\",\"description\":\"cylindrical "
    "reflector from retro-reflective foil\",\"height\":0.05},{\"surface_type\":\""
    "retro_reflective_foil\",\"name\":\"RIEGL cylindrical reflector 100 "
    "mm\",\"diameter\":0.1,\"shape\":\"cylinder\",\"description\":\"cylindrical "
    "reflector from retro-reflective "
    "foil\",\"height\":0.1},{\"shape\":\"sphere\",\"name\":\"Sphere 200 "
    "mm\",\"diameter\":0.2,\"description\":\"Sphere (diameter: 200 "
    "mm)\"},{\"shape\":\"round_corner_cube_prism\",\"offset\":0.0,\"name\":\"Corner "
    "Cube Prism 50 mm\",\"diameter\":0.05,\"description\":\"round corner cube "
    "prism\"}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"re"
    "ference_file\":{\"properties\":{\"file_path\":{\"description\":\"Path of the "
    "control object file relative to referring "
    "file\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"Control object "
    "file's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"}},\"description\":\"Reference to a control object "
    "file\",\"type\":\"object\",\"required\":[\"file_uuid\",\"file_path\"]}},\"title\":\""
    "Control Object Reference file\",\"type\":\"object\",\"description\":\"Details "
    "about the control object reference file\"}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoi"
    "nts.cpx\",\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"detection_probability\":{\"description\":\"Detection "
    "probability [0..1]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"amplitu"
    "de\":{\"description\":\"Amplitude [dB]\",\"type\":\"array\",\"items\":{\"type\":\"nu"
    "mber\"}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required"
    "\":[\"amplitude\",\"detection_probability\"],\"title\":\"Detection "
    "Probability\",\"type\":\"object\",\"description\":\"Detection probability as a"
    " function of amplitude\"}"
    ,
    /* example: */
    "{\"detection_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]"
    ",\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"device_name\":{\"description\":\"Optional device name "
    "(e.g. 'Scanner 1' for multi-scanner systems)\",\"type\":\"string\"},\"channe"
    "l_number\":{\"minimum\":0,\"description\":\"Laser channel number (not "
    "defined or 0: single channel device)\",\"type\":\"integer\",\"exclusiveMinim"
    "um\":false},\"serial_number\":{\"description\":\"Device serial number (e.g. "
    "S2221234)\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"channel_text\":{\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\",\"type\":\"string\"},\"device_build\":{\"description\":\"Device build"
    " variant\",\"type\":\"string\"}},\"$schema\":\"http://json-schema.org/draft-04"
    "/schema#\",\"required\":[\"device_type\",\"serial_number\"],\"title\":\"Device "
    "Information\",\"type\":\"object\",\"description\":\"Details about the device "
    "used to acquire the point cloud\"}"
    ,
    /* example: */
    "{\"device_name\":\"Scanner 1\",\"channel_number\":0,\"serial_number\":\"S222123"
    "4\",\"device_type\":\"VZ-400i\",\"channel_text\":\"\",\"device_build\":\"\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"primary\":{\"properties\":{\"content\":{\"description\":\"Inte"
    "rnal calibration values\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"ID"
    "\":{\"maxItems\":2,\"minItems\":2,\"description\":\"Structure identifier\",\"typ"
    "e\":\"array\",\"items\":{\"type\":\"integer\"}}},\"description\":\"Primary device "
    "geometry structure (mandatory)\",\"type\":\"object\",\"required\":[\"ID\",\"cont"
    "ent\"]},\"amu\":{\"properties\":{\"lineCC\":{\"minimum\":0,\"description\":\"Line "
    "Circle Count (number of LSBs per full rotation about line axis)\",\"type"
    "\":\"number\",\"exclusiveMinimum\":false},\"frameCC\":{\"minimum\":0,\"descripti"
    "on\":\"Frame Circle Count (number of LSBs per full rotation about frame "
    "axis)\",\"type\":\"number\",\"exclusiveMinimum\":false}},\"description\":\"Angle"
    " Measurement Unit\",\"type\":\"object\"},\"secondary\":{\"properties\":{\"conten"
    "t\":{\"description\":\"Internal calibration values\",\"type\":\"array\",\"items\""
    ":{\"type\":\"number\"}},\"ID\":{\"maxItems\":2,\"minItems\":2,\"description\":\"Str"
    "ucture identifier\",\"type\":\"array\",\"items\":{\"type\":\"integer\"}}},\"descri"
    "ption\":\"Additional device geometry structure (optional)\",\"type\":\"objec"
    "t\",\"required\":[\"ID\",\"content\"]}},\"$schema\":\"http://json-schema.org/dra"
    "ft-04/schema#\",\"required\":[\"primary\"],\"title\":\"Device "
    "Geometry\",\"type\":\"object\",\"description\":\"Scanner device geometry "
    "details\"}"
    ,
    /* example: */
    "{\"primary\":{\"content\":[0],\"ID\":[4,0]},\"amu\":{\"lineCC\":124000,\"frameCC\""
    ":124000},\"secondary\":{\"content\":[0],\"ID\":[91,0]}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"primary\":{\"properties\":{\"content\":{\"description\":\"Inte"
    "rnal calibration values\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"ID"
    "\":{\"maxItems\":2,\"minItems\":2,\"description\":\"Structure identifier\",\"typ"
    "e\":\"array\",\"items\":{\"type\":\"integer\"}}},\"description\":\"Primary device "
    "geometry structure (mandatory)\",\"type\":\"object\",\"required\":[\"ID\",\"cont"
    "ent\"]}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\""
    ":[\"primary\"],\"title\":\"Device Geometry Passive "
    "Channel\",\"type\":\"object\",\"description\":\"Scanner passive channel device"
    " geometry details\"}"
    ,
    /* example: */
    "{\"primary\":{\"content\":[0],\"ID\":[143,0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"re"
    "flectance_maximum\":{\"description\":\"Maximum possible reflectance in dB."
    "\",\"type\":\"number\"},\"background_radiation_maximum\":{\"description\":\"Maxi"
    "mum possible background radiation.\",\"type\":\"number\"},\"deviation_maximu"
    "m\":{\"description\":\"Maximum possible pulse shape deviation.\",\"type\":\"nu"
    "mber\"},\"background_radiation_minimum\":{\"description\":\"Minimum possible"
    " background radiation.\",\"type\":\"number\"},\"deviation_minimum\":{\"descrip"
    "tion\":\"Minimum possible pulse shape "
    "deviation.\",\"type\":\"number\"},\"range_minimum\":{\"description\":\"Minimum "
    "possible range in meters.\",\"type\":\"number\"},\"mta_zone_count_maximum\":{"
    "\"description\":\"Maximum number of MTA "
    "zones.\",\"type\":\"number\"},\"reflectance_minimum\":{\"description\":\"Minimum"
    " possible reflectance in "
    "dB.\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum possible"
    " range in "
    "meters.\",\"type\":\"number\"},\"echo_count_maximum\":{\"description\":\"Maximum"
    " number of echoes a laser shot can "
    "have.\",\"type\":\"number\"},\"amplitude_minimum\":{\"description\":\"Minimum "
    "possible amplitude in "
    "dB.\",\"type\":\"number\"},\"amplitude_maximum\":{\"description\":\"Maximum "
    "possible amplitude in dB.\",\"type\":\"number\"}},\"title\":\"Device Output "
    "Limits\",\"type\":\"object\",\"description\":\"Limits of the measured values "
    "output by the device. The limits depend on the device type, "
    "measurement program and/or scan pattern.\"}"
    ,
    /* example: */
    "{\"range_minimum\":2.9,\"mta_zone_count_maximum\":7,\"background_radiation_"
    "minimum\":0,\"range_maximum\":10000.0,\"reflectance_maximum\":100.0,\"backgr"
    "ound_radiation_maximum\":0,\"deviation_minimum\":-1,\"amplitude_minimum\":0"
    ".0,\"deviation_maximum\":32767,\"reflectance_minimum\":-100.0,\"amplitude_m"
    "aximum\":100.0}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"items\":{\"propert"
    "ies\":{\"device_name\":{\"description\":\"Optional device name (e.g. "
    "'Scanner 1' for multi-scanner systems)\",\"type\":\"string\"},\"channel_numb"
    "er\":{\"minimum\":0,\"description\":\"Laser channel number (not defined or "
    "0: single channel device)\",\"type\":\"integer\",\"exclusiveMinimum\":false},"
    "\"serial_number\":{\"description\":\"Device serial number (e.g. "
    "S2221234)\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"channel_text\":{\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\",\"type\":\"string\"},\"device_build\":{\"description\":\"Device build"
    " variant\",\"type\":\"string\"},\"signed\":{\"description\":\"Flag that is set "
    "when the original 'riegl.device' entry in the source file was "
    "correctly signed.\",\"type\":\"boolean\"}},\"type\":\"object\",\"required\":[\"dev"
    "ice_type\",\"serial_number\"]},\"title\":\"Devices "
    "Information\",\"type\":\"array\",\"description\":\"Details about the devices "
    "used to acquire the point cloud (e.g. when merging point clouds of "
    "different devices)\"}"
    ,
    /* example: */
    "[{\"device_name\":\"Scanner 1\",\"channel_number\":0,\"serial_number\":\"S22212"
    "34\",\"device_type\":\"VZ-400i\",\"channel_text\":\"\",\"device_build\":\"\",\"signe"
    "d\":false},{\"device_name\":\"Scanner 2\",\"channel_number\":1,\"serial_number"
    "\":\"S2222680\",\"device_type\":\"VQ-1560i-DW\",\"channel_text\":\"\",\"device_bui"
    "ld\":\"\",\"signed\":true},{\"device_name\":\"Scanner 3\",\"channel_number\":2,\"s"
    "erial_number\":\"S2222680\",\"device_type\":\"VQ-1560i-DW\",\"channel_text\":\"\""
    ",\"device_build\":\"\",\"signed\":true}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"dmi_min_time\":{\"maximum\":65535,\"minimum\":0,\"exclusiveM"
    "aximum\":false,\"description\":\"Minimum time interval between trigger "
    "events, 0 means rate limitation is inactive [0.1 msec]\",\"type\":\"intege"
    "r\",\"exclusiveMinimum\":false},\"dmi_dist_per_tick\":{\"description\":\"Dista"
    "nce per tick, just informative [m]\",\"type\":\"number\"},\"dmi_max_time\":{\""
    "maximum\":65535,\"minimum\":0,\"exclusiveMaximum\":false,\"description\":\"Max"
    "imum time interval between trigger events, 0 means timeout is inactive"
    " [0.1 msec]\",\"type\":\"integer\",\"exclusiveMinimum\":false},\"dmi_count_dir"
    "\":{\"maximum\":1,\"minimum\":0,\"exclusiveMaximum\":false,\"description\":\"DMI"
    " count direction is inverted\",\"type\":\"integer\",\"exclusiveMinimum\":fals"
    "e},\"dmi_incr\":{\"maximum\":2147483647,\"minimum\":0,\"exclusiveMaximum\":fal"
    "se,\"description\":\"Increment in ticks, 0 means distance trigger is deac"
    "tivated\",\"type\":\"integer\",\"exclusiveMinimum\":false}},\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\",\"required\":[\"dmi_count_dir\",\"dmi_in"
    "cr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\"],\"title\":\"DMI "
    "Configuration\",\"type\":\"object\",\"description\":\"Distance measuring "
    "instrument configuration\"}"
    ,
    /* example: */
    "{\"dmi_min_time\":0,\"dmi_dist_per_tick\":0.00024399999529123306,\"dmi_max_"
    "time\":100,\"dmi_count_dir\":0,\"dmi_incr\":0}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Dyntrig\""
    ",\"properties\":{\"post\":{\"properties\":{\"nc\":{\"maximum\":255,\"minimum\":0,\""
    "description\":\"Number of entries\",\"type\":\"integer\"},\"a\":{\"maxItems\":208"
    "0,\"minItems\":1,\"description\":\"...\",\"type\":\"array\",\"items\":{\"type\":\"num"
    "ber\"}}},\"description\":\"Dyntrig post values\",\"type\":\"object\",\"required\""
    ":[\"nc\",\"a\"]},\"pre\":{\"properties\":{\"nc\":{\"maximum\":255,\"minimum\":0,\"des"
    "cription\":\"Number of entries\",\"type\":\"integer\"},\"a\":{\"maxItems\":2080,\""
    "minItems\":1,\"description\":\"...\",\"type\":\"array\",\"items\":{\"type\":\"number"
    "\"}}},\"description\":\"Dyntrig pre values\",\"type\":\"object\",\"required\":[\"n"
    "c\",\"a\"]},\"scale\":{\"description\":\"Scale in units of dB, for calculation"
    " of line: line number N=(amplitude-offset)/scale\",\"type\":\"number\"},\"ta"
    "b\":{\"maxItems\":128,\"minItems\":1,\"description\":\"...\",\"type\":\"array\",\"it"
    "ems\":{\"properties\":{\"del1\":{\"description\":\"Length of constant "
    "post-trigger threshold elevation in units of "
    "m\",\"type\":\"number\"},\"dec2\":{\"description\":\"Rise of pre-trigger "
    "threshold after del2 in units of "
    "dB/m\",\"type\":\"number\"},\"thra1\":{\"description\":\"Post-trigger threshold "
    "in units of dB\",\"type\":\"number\"},\"thra2\":{\"description\":\"Pre-trigger "
    "threshold in units of "
    "dB\",\"type\":\"number\"},\"dec1\":{\"description\":\"Decay of post-trigger "
    "threshold after del1 in units of "
    "dB/m\",\"type\":\"number\"},\"del2\":{\"description\":\"Length of constant "
    "pre-trigger threshold elevation in units of m\",\"type\":\"number\"}},\"desc"
    "ription\":\"...\",\"type\":\"object\",\"required\":[\"thra1\",\"del1\",\"dec1\",\"thra"
    "2\",\"del2\",\"dec2\"]}},\"offset\":{\"description\":\"Offset in units of dB, "
    "for calculation of line\",\"type\":\"number\"}},\"description\":\"Dyntrig\"}"
    ,
    /* example: */
    "{\"post\":{\"nc\":128,\"a\":[78,86,126,134,31]},\"pre\":{\"nc\":128,\"a\":[78,86,1"
    "26,134,31]},\"scale\":1.0,\"tab\":[{\"del1\":0.0,\"dec2\":0.0,\"thra1\":0.0,\"thr"
    "a2\":0.0,\"dec1\":0.0,\"del2\":0.0}],\"offset\":0.0}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"echo_file\":{\"properties\":{\"file_uuid\":{\"description\":\""
    "File's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Echo file "
    "extension, without the leading dot\",\"type\":\"string\"}},\"type\":\"object\","
    "\"required\":[\"file_extension\"]}},\"$schema\":\"http://json-schema.org/draf"
    "t-04/schema#\",\"required\":[\"echo_file\"],\"title\":\"Echo "
    "Information\",\"type\":\"object\",\"description\":\"Details about echo files\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_uuid\":\"26a03615-67c0-4bea-8fe8-c577378fe661\",\"file"
    "_extension\":\"owp\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"additionalProperties\":false,\"patternProperties\":{\"^[0-9]+$\":{\"descri"
    "ption\":\"one field per channel, field name is channel index\",\"$ref\":\"#/"
    "definitions/channel\"}},\"$schema\":\"http://json-schema.org/draft-04/sche"
    "ma#\",\"definitions\":{\"channel\":{\"properties\":{\"delay\":{\"description\":\"d"
    "elay calibration in seconds\",\"type\":\"number\"},\"parameter\":{\"properties"
    "\":{\"omega\":{\"description\":\"angular frequency in Hz\",\"type\":\"array\",\"it"
    "ems\":{\"type\":\"number\"}},\"B\":{\"description\":\"imaginary part of "
    "amplitude factor in units of full-scale\",\"type\":\"array\",\"items\":{\"type"
    "\":\"number\"}},\"A\":{\"description\":\"real part of amplitude factor in "
    "units of full-scale\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"gamma\""
    ":{\"description\":\"decay in 1/second\",\"type\":\"array\",\"items\":{\"type\":\"nu"
    "mber\"}}},\"description\":\"parameters of the syswave exponential sum\",\"ty"
    "pe\":\"object\",\"required\":[\"A\",\"B\",\"gamma\",\"omega\"]},\"scale\":{\"descripti"
    "on\":\"amplitude calibration\",\"type\":\"number\"},\"a_lin\":{\"maximum\":1,\"min"
    "imum\":0,\"exclusiveMaximum\":false,\"description\":\"relative linear "
    "amplitude range [0..1]\",\"type\":\"number\",\"exclusiveMinimum\":false}},\"ty"
    "pe\":\"object\",\"required\":[\"delay\",\"scale\",\"parameter\"]}},\"title\":\"Expon"
    "ential Decomposition\",\"type\":\"object\",\"description\":\"Details for "
    "exponential decomposition of full waveform data\"}"
    ,
    /* example: */
    "{\"1\":{\"delay\":3.5e-09,\"parameter\":{\"omega\":[352020896.0,3647927552.0,-"
    "1977987072.0],\"B\":[-3.9,0.0,-0.3],\"A\":[0.9,0.3,-1.3],\"gamma\":[-1094726"
    "528.0,-769562752.0,-848000064.0]},\"scale\":1.0,\"a_lin\":0.9},\"0\":{\"delay"
    "\":3.783458418887631e-09,\"parameter\":{\"omega\":[352020896.0,3647927552.0"
    ",-1977987072.0],\"B\":[-3.9813032150268555,0.08622030913829803,-0.315286"
    "0999107361],\"A\":[0.9772450923919678,0.3354335129261017,-1.312678575515"
    "747],\"gamma\":[-1094726528.0,-769562752.0,-848000064.0]},\"scale\":1.0,\"a"
    "_lin\":0.27}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_offset_sec_high_power\":{\"type\":\"array\",\"items\":{"
    "\"type\":\"number\"}},\"range_offset_sec_low_power\":{\"type\":\"array\",\"items\""
    ":{\"type\":\"number\"}},\"amplitude_lsb_low_power\":{\"type\":\"array\",\"items\":"
    "{\"type\":\"number\"}},\"amplitude_lsb_high_power\":{\"type\":\"array\",\"items\":"
    "{\"type\":\"number\"}},\"amplitude_db\":{\"type\":\"array\",\"items\":{\"type\":\"num"
    "ber\"}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\""
    ":[\"amplitude_lsb_low_power\",\"amplitude_lsb_high_power\",\"amplitude_db\","
    "\"range_offset_sec_low_power\",\"range_offset_sec_high_power\"],\"title\":\"G"
    "aussian Decomposition\",\"type\":\"object\",\"description\":\"riegl.gaussian_d"
    "ecomposition contains information relevant for extracting calibrated "
    "amplitudes and calibrated ranges from a Gaussian decomposition of full"
    " waveform data. This information is contained in a table with five "
    "columns. Two columns are to be used as input: amplitude_lsb_low_power "
    "and amplitude_lsb_high_power. The other three columns provide the "
    "outputs. Amplitude_db gives the calibrated amplitude in the optical "
    "regime in decibels. The range offset columns provide additive range "
    "offsets, given in units of seconds, for each channel.\"}"
    ,
    /* example: */
    "{\"range_offset_sec_high_power\":[],\"range_offset_sec_low_power\":[],\"amp"
    "litude_lsb_low_power\":[],\"amplitude_lsb_high_power\":[],\"amplitude_db\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"cr"
    "s\":{\"properties\":{\"epsg\":{\"minimum\":0,\"description\":\"EPSG "
    "code\",\"type\":\"integer\"},\"wkt\":{\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\",\"type\":\"string\"},\"name\":{\"description\":\"Coordinate reference "
    "system name\",\"type\":\"string\"}},\"description\":\"Global Coordinate "
    "Reference System. Please note that only 3D Cartesian Coordinate "
    "Systems are allowed.\",\"type\":\"object\"},\"pose\":{\"maxItems\":4,\"minItems\""
    ":4,\"description\":\"Coordinate Transformation Matrix to transform from "
    "File Coordinate System to Global Coordinate Reference System. 4x4 "
    "matrix stored as two dimensional array, row major order.\",\"type\":\"arra"
    "y\",\"items\":{\"maxItems\":4,\"minItems\":4,\"description\":\"rows\",\"type\":\"arr"
    "ay\",\"items\":{\"description\":\"columns\",\"type\":\"number\"}}}},\"title\":\"Geo "
    "Tag\",\"type\":\"object\",\"description\":\"Point cloud georeferencing "
    "information\"}"
    ,
    /* example: */
    "{\"crs\":{\"epsg\":4978,\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\",DATUM[\\\"WGS84\\\""
    ",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"703"
    "0\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.00000000000"
    "00000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.00000000000000000"
    "000,AUTHORITY[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AXIS[\\\"Y\\\",EAST],A"
    "XIS[\\\"Z\\\",NORTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\",\"name\":\"WGS84 Geocentr"
    "ic\"},\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,"
    "4063882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847"
    ",1138787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196"
    "],[0.0,0.0,0.0,1.0]]}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"exclusiveMinimum"
    "\":true,\"description\":\"Geometric scale factor applied to point "
    "coordinates\",\"type\":\"number\",\"minimum\":0}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"tr"
    "ajectory_file\":{\"properties\":{\"file_uuid\":{\"description\":\"File's "
    "Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Trajectory "
    "file extension, without the leading "
    "dot\",\"type\":\"string\"}},\"description\":\"Trajectory data used for "
    "georeferencing of the point cloud\",\"type\":\"object\",\"required\":[\"file_e"
    "xtension\"]},\"socs_to_body_matrix\":{\"maxItems\":4,\"minItems\":4,\"descript"
    "ion\":\"Coordinate Transformation Matrix to transform from Scanner's Own"
    " Coordinate System to Body Coordinate System. 4x4 matrix stored as two"
    " dimensional array, row major order.\",\"type\":\"array\",\"items\":{\"maxItem"
    "s\":4,\"minItems\":4,\"description\":\"rows\",\"type\":\"array\",\"items\":{\"descri"
    "ption\":\"columns\",\"type\":\"number\"}}},\"socs_to_rocs_matrix\":{\"maxItems\":"
    "4,\"minItems\":4,\"description\":\"Coordinate Transformation Matrix to "
    "transform from Scanner's Own Coordinate System to Record Coordinate "
    "System. 4x4 matrix stored as two dimensional array, row major order.\","
    "\"type\":\"array\",\"items\":{\"maxItems\":4,\"minItems\":4,\"description\":\"rows\""
    ",\"type\":\"array\",\"items\":{\"description\":\"columns\",\"type\":\"number\"}}},\"t"
    "rajectory_offsets\":{\"properties\":{\"offset_east\":{\"description\":\"[m]\",\""
    "type\":\"number\"},\"offset_time\":{\"description\":\"[s]\",\"type\":\"number\"},\"o"
    "ffset_north\":{\"description\":\"[m]\",\"type\":\"number\"},\"offset_yaw\":{\"desc"
    "ription\":\"[deg]\",\"type\":\"number\"},\"offset_pitch\":{\"description\":\"[deg]"
    "\",\"type\":\"number\"},\"offset_height\":{\"description\":\"[m]\",\"type\":\"number"
    "\"},\"offset_roll\":{\"description\":\"[deg]\",\"type\":\"number\"},\"version\":{\"d"
    "escription\":\"Meaning of offset values and how to apply them; version "
    "0: Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\",\"type\":\"integer\"}},\"description\":\"Correction "
    "offsets applied to the trajectory data\",\"type\":\"object\"},\"body_coordin"
    "ate_system_type\":{\"enum\":[\"NED\",\"ENU\"],\"description\":\"BODY coordinate "
    "frame (NED: North-East-Down, ENU: East-North-Up), default: "
    "NED\",\"type\":\"string\"}},\"title\":\"Georeferencing "
    "Parameters\",\"type\":\"object\",\"description\":\"Parameters used for "
    "georeferencing of the point cloud\"}"
    ,
    /* example: */
    "{\"trajectory_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\""
    ",\"file_extension\":\"pofx\"},\"socs_to_body_matrix\":[[-0.269827776749716,-"
    "0.723017716139738,0.635954678449952,0.0],[0.962908599449764,-0.2026051"
    "7250352,0.178208229833847,0.0],[0.0,0.660451759194338,0.7508684796801,"
    "0.0],[0.0,0.0,0.0,1.0]],\"body_coordinate_system_type\":\"NED\",\"trajector"
    "y_offsets\":{\"offset_east\":0.15,\"offset_time\":18.007,\"offset_north\":0.0"
    "7,\"offset_yaw\":-0.45,\"offset_pitch\":0.01,\"offset_height\":-0.2,\"offset_"
    "roll\":0.03,\"version\":0}}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"ba"
    "se_stations\":{\"description\":\"List of GNSS base stations\",\"type\":\"array"
    "\",\"items\":{\"properties\":{\"altitude\":{\"description\":\"Altitude "
    "[m]\",\"type\":\"number\"},\"longitude\":{\"description\":\"Longitude "
    "[deg]\",\"type\":\"number\"},\"latitude\":{\"description\":\"Latitude "
    "[deg]\",\"type\":\"number\"}},\"description\":\"GNSS base station details\",\"ty"
    "pe\":\"object\",\"required\":[\"latitude\",\"longitude\",\"altitude\"]}}},\"title\""
    ":\"GNSS Processing Information\",\"type\":\"object\",\"description\":\"Details "
    "about GNSS processing\"}"
    ,
    /* example: */
    "{\"base_stations\":[{\"altitude\":379.8004150390625,\"longitude\":15.6633898"
    "22,\"latitude\":48.658391162}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"additionalProperties\":false,\"patternProperties\":{\"^-?[0-9]+$\":{\"desc"
    "ription\":\"One field per item, field name is item id, field value is "
    "item name\",\"type\":\"string\"}},\"$schema\":\"http://json-schema.org/draft-0"
    "4/schema#\",\"title\":\"Item Names\",\"type\":\"object\",\"description\":\"Map of "
    "item names\"}"
    ,
    /* example: */
    "{\"1\":\"Name of item with id 1\",\"-1\":\"Name of item with id -1\",\"2\":\"Name"
    " of item with id 2\",\"47\":\"Name of item with id 47\"}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"additionalProperties\":false,\"patternProperties\":{\"^.*$\":{\"minItems\":"
    "1,\"description\":\"Each field of the object represents a feature and "
    "holds a list of license keys, where the field name is the feature "
    "name.\",\"type\":\"array\",\"items\":{\"description\":\"License key (example: "
    "'46AE032A - 39882AC4 - 9EC0A184 - 6F163D73')\",\"type\":\"string\"}}},\"$sch"
    "ema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Software "
    "License Keys\",\"type\":\"object\",\"description\":\"License keys for software"
    " features\"}"
    ,
    /* example: */
    "{\"Georeferencing\":[\"46AE032A - 39882AC4 - 9EC0A184 - 6F163D73\"],\"Full "
    "Waveform Analysis Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA - "
    "8D2ED557\"],\"MTA resolution\":[\"468E020A - 39A922E4 - B681A184 - "
    "673E3D72\"],\"Full Waveform Analysis Topography with GPU "
    "support\":[\"8AB44126 - 23B92250 - 16E2689F - 34EF7E7B\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"zo"
    "ne_width\":{\"minimum\":0,\"description\":\"Width of a MTA zone in meter.\",\""
    "type\":\"number\"},\"modulation_depth\":{\"minimum\":0,\"description\":\"Depth "
    "of pulse position modulation in meter.\",\"type\":\"number\"},\"zone_count\":"
    "{\"maximum\":255,\"minimum\":0,\"description\":\"Maximum number of MTA "
    "zones.\",\"type\":\"integer\"}},\"title\":\"MTA "
    "Settings\",\"type\":\"object\",\"description\":\"Parameters for MTA "
    "processing\"}"
    ,
    /* example: */
    "{\"zone_width\":149.896225,\"modulation_depth\":9.368514,\"zone_count\":23}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"delta\":{\"description\":\"Delta between table entries "
    "[m], first entry is at range = 0 m\",\"type\":\"number\"},\"content\":{\"maxIt"
    "ems\":2000,\"minItems\":1,\"description\":\"Correction value [m] to be added"
    " to the raw range\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"$schema"
    "\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"delta\",\"conte"
    "nt\"],\"title\":\"Near Range Correction "
    "Table\",\"type\":\"object\",\"description\":\"Lookup table for range "
    "correction based on raw range\"}"
    ,
    /* example: */
    "{\"delta\":0.512,\"content\":[0.0]}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_sigma\":{\"description\":\"Sigma range [m]\",\"type\":\""
    "array\",\"items\":{\"type\":\"number\"}},\"amplitude\":{\"description\":\"Amplitud"
    "e [dB]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"amplitude_sigma\":{\""
    "description\":\"Sigma amplitude [dB]\",\"type\":\"array\",\"items\":{\"type\":\"nu"
    "mber\"}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required"
    "\":[\"amplitude\",\"range_sigma\",\"amplitude_sigma\"],\"title\":\"Noise "
    "Estimates\",\"type\":\"object\",\"description\":\"Standard deviation for range"
    " and amplitude as a function of amplitude\"}"
    ,
    /* example: */
    "{\"range_sigma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,"
    "0.016,0.014],\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0"
    "],\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0."
    "432,0.39,0.354]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_minimum\":{\"description\":\"Minimum range "
    "[m]\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum range [m"
    "]\",\"type\":\"number\"},\"amplitude_maximum\":{\"minimum\":0,\"description\":\"Ma"
    "ximum amplitude [dB]\",\"type\":\"number\"}},\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\",\"required\":[\"range_minimum\",\"range_maximum\",\"amp"
    "litude_maximum\"],\"title\":\"Notch "
    "Filter\",\"type\":\"object\",\"description\":\"Notch filter parameters for "
    "window glass echoes\"}"
    ,
    /* example: */
    "{\"range_minimum\":-0.5,\"range_maximum\":0.2,\"amplitude_maximum\":18.0}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"size\":{\"description\":\"Size of pixels in file "
    "coordinate system.\",\"$ref\":\"#/definitions/pixel_size\"},\"size_llcs\":{\"d"
    "escription\":\"Size of pixels in a locally levelled cartesian coordinate"
    " system (xy). This is only used for pixels based on a map projection.\""
    ",\"$ref\":\"#/definitions/pixel_size\"}},\"$schema\":\"http://json-schema.org"
    "/draft-04/schema#\",\"definitions\":{\"pixel_size\":{\"maxItems\":2,\"minItems"
    "\":2,\"description\":\"Size of "
    "pixels.\",\"type\":\"array\",\"items\":{\"minimum\":0,\"description\":\"Length of "
    "pixel edge [m].\",\"type\":\"number\"}}},\"required\":[\"size\"],\"title\":\"Pixel"
    " Information\",\"type\":\"object\",\"description\":\"Details about the pixels "
    "contained in the file\"}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395],\"size_llcs\":[0.5156575"
    "252891171,0.5130835356683303]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"plane_patch_file_two\":{\"description\":\"Reference to the"
    " plane patch file two\",\"$ref\":\"#/definitions/file_reference\"},\"plane_p"
    "atch_file_one\":{\"description\":\"Reference to the plane patch file one\","
    "\"$ref\":\"#/definitions/file_reference\"}},\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\",\"definitions\":{\"file_reference\":{\"properties\":{\""
    "file_path\":{\"description\":\"Path of the plane patch file relative to "
    "the match file\",\"type\":\"string\"},\"file_uuid\":{\"description\":\"Plane "
    "patch file's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"}},\"description\":\"Reference to a plane patch fil"
    "e\",\"type\":\"object\",\"required\":[\"file_uuid\",\"file_path\"]}},\"required\":["
    "\"plane_patch_file_one\",\"plane_patch_file_two\"],\"title\":\"Plane Patch "
    "Matching\",\"type\":\"object\",\"description\":\"Details about the plane patch"
    " matching process\"}"
    ,
    /* example: */
    "{\"plane_patch_file_two\":{\"file_path\":\"project.ptch\",\"file_uuid\":\"fa47d"
    "509-a64e-49ce-8b14-ff3130fbefa9\"},\"plane_patch_file_one\":{\"file_path\":"
    "\"Record009_Line001/191025_121410_Scanner_1.ptch\",\"file_uuid\":\"810f5d57"
    "-eccf-49ed-b07a-0cdd109b4213\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"to"
    "tal_horizontal_area\":{\"description\":\"sum of all plane patch areas "
    "projected to horizontal plane "
    "[m\\u00b2]\",\"type\":\"number\"},\"total_area\":{\"description\":\"sum of all "
    "plane patch areas [m\\u00b2]\",\"type\":\"number\"}},\"title\":\"Plane Patch "
    "Statistics\",\"type\":\"object\",\"description\":\"Statistics about plane "
    "patches found by plane patch extractor\"}"
    ,
    /* example: */
    "{\"total_horizontal_area\":13954.601,\"total_area\":14007.965}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"classes\":{\"patternProperties\":{\"^[0-9]+$\":{\"descriptio"
    "n\":\"one field per class, field name is class number, field value is "
    "class name\",\"type\":\"string\"}},\"additionalProperties\":false,\"descriptio"
    "n\":\"Class definition table\",\"type\":\"object\"},\"settings\":{\"oneOf\":[{\"$r"
    "ef\":\"#/definitions/method-1\"},{\"$ref\":\"#/definitions/method-2\"}],\"desc"
    "ription\":\"Classification settings, details see documentation of rdbpla"
    "nes\",\"type\":\"object\"}},\"$schema\":\"http://json-schema.org/draft-04/sche"
    "ma#\",\"definitions\":{\"method-1\":{\"properties\":{\"plane_classification_me"
    "thod\":{\"maximum\":1,\"minimum\":1,\"description\":\"method ID (=1)\",\"type\":\""
    "integer\"},\"maximum_inclination_angle_horizontal\":{\"maximum\":360.0,\"min"
    "imum\":-360.0,\"description\":\"maximum inclination angle of horizontal "
    "plane patches [deg]\",\"type\":\"number\"}},\"description\":\"Classification "
    "method 1\",\"type\":\"object\",\"required\":[\"plane_classification_method\",\"m"
    "aximum_inclination_angle_horizontal\"]},\"method-2\":{\"properties\":{\"slop"
    "ing_plane_classes_minimum_angle\":{\"maximum\":360.0,\"minimum\":-360.0,\"de"
    "scription\":\"minimum inclination angle of sloping plane patches [deg]\","
    "\"type\":\"number\"},\"sloping_plane_classes_maximum_angle\":{\"maximum\":360."
    "0,\"minimum\":-360.0,\"description\":\"maximum inclination angle of sloping"
    " plane patches [deg]\",\"type\":\"number\"},\"plane_classification_method\":{"
    "\"maximum\":2,\"minimum\":2,\"description\":\"method ID "
    "(=2)\",\"type\":\"integer\"}},\"description\":\"Classification method 2\",\"type"
    "\":\"object\",\"required\":[\"plane_classification_method\",\"sloping_plane_cl"
    "asses_minimum_angle\",\"sloping_plane_classes_maximum_angle\"]}},\"require"
    "d\":[\"settings\",\"classes\"],\"title\":\"Plane Slope Class "
    "Info\",\"type\":\"object\",\"description\":\"Settings and classes for plane "
    "slope classification\"}"
    ,
    /* example: */
    "{\"classes\":{\"1\":\"horizontal, pointing up\",\"11\":\"sloping, pointing down"
    " and south\",\"9\":\"vertical, pointing west\",\"12\":\"sloping, pointing down"
    " and north\",\"10\":\"sloping, pointing down and east\",\"7\":\"vertical, "
    "pointing south\",\"3\":\"sloping, pointing up and south\",\"6\":\"vertical, "
    "pointing east\",\"13\":\"sloping, pointing down and west\",\"5\":\"sloping, "
    "pointing up and west\",\"4\":\"sloping, pointing up and "
    "north\",\"8\":\"vertical, pointing north\",\"14\":\"horizontal, pointing "
    "down\",\"2\":\"sloping, pointing up and east\"},\"settings\":{\"sloping_plane_"
    "classes_minimum_angle\":10.0,\"sloping_plane_classes_maximum_angle\":70.0"
    ",\"plane_classification_method\":2}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"additionalProperties\":false,\"patternProperties\":{\"^.*$\":{\"minItems\":"
    "1,\"description\":\"Each field of the object represents a point attribute"
    " group and holds a list of point attributes, where the field name is "
    "the group name.\",\"type\":\"array\",\"items\":{\"description\":\"Point "
    "attribute full name or name pattern (perl regular expression syntax)\","
    "\"type\":\"string\"}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\""
    ",\"title\":\"Point Attribute "
    "Groups\",\"type\":\"object\",\"description\":\"Grouping and sorting of point "
    "attributes for visualization purposes\"}"
    ,
    /* example: */
    "{\"Other Attributes\":[\"riegl.selected\",\"riegl.visible\"],\"Coordinates/Ve"
    "ctors\":[\"riegl.xyz\",\"riegl.range\",\"riegl.theta\",\"riegl.phi\"],\"Secondar"
    "y "
    "Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available\"],\"Primary"
    " Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.deviation\"]"
    ",\"Time\":[\"riegl.timestamp\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"fi"
    "eld_of_application\":{\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\""
    "ALS\",\"BLS\",\"ILS\"],\"description\":\"Field of application\",\"type\":\"string\""
    "},\"comments\":{\"description\":\"Comments\",\"type\":\"string\"},\"project\":{\"de"
    "scription\":\"Project name\",\"type\":\"string\"}},\"title\":\"Point Cloud "
    "Information\",\"type\":\"object\",\"description\":\"Details about point cloud "
    "files\"}"
    ,
    /* example: */
    "{\"field_of_application\":\"ALS\",\"comments\":\"Line 3\",\"project\":\"Campaign "
    "4\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"position\":{\"properties\":{\"crs\":{\"properties\":{\"epsg\":{"
    "\"minimum\":0,\"description\":\"EPSG "
    "code\",\"type\":\"integer\"},\"wkt\":{\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\",\"type\":\"string\"}},\"description\":\"Global Coordinate Reference "
    "System\",\"type\":\"object\",\"required\":[\"epsg\"]},\"horizontal_accuracy\":{\"m"
    "inimum\":0,\"description\":\"Horizontal accuracy [m]\",\"type\":\"number\",\"exc"
    "lusiveMinimum\":true},\"vertical_accuracy\":{\"minimum\":0,\"description\":\"V"
    "ertical accuracy [m]\",\"type\":\"number\",\"exclusiveMinimum\":true},\"coordi"
    "nate_2\":{\"description\":\"Coordinate 2 as defined by axis 2 of the "
    "specified CRS (e.g., Y, "
    "Longitude)\",\"type\":\"number\"},\"coordinate_3\":{\"description\":\"Coordinate"
    " 3 as defined by axis 3 of the specified CRS (e.g., Z, "
    "Altitude)\",\"type\":\"number\"},\"coordinate_1\":{\"description\":\"Coordinate "
    "1 as defined by axis 1 of the specified CRS (e.g., X, "
    "Latitude)\",\"type\":\"number\"}},\"description\":\"Position coordinates and "
    "position accuracy values as measured by GNSS in the specified "
    "Coordinate Reference System (CRS)\",\"type\":\"object\",\"required\":[\"coordi"
    "nate_1\",\"coordinate_2\",\"coordinate_3\",\"horizontal_accuracy\",\"vertical_"
    "accuracy\",\"crs\"]},\"barometric_height_amsl\":{\"description\":\"Altitude "
    "determined based on the atmospheric pressure according to the standard"
    " atmosphere laws [m].\",\"type\":\"number\"},\"orientation\":{\"properties\":{\""
    "roll\":{\"maximum\":360,\"minimum\":-360,\"description\":\"Roll angle about "
    "scanner X-axis [deg]\",\"type\":\"number\"},\"pitch\":{\"maximum\":360,\"minimum"
    "\":-360,\"description\":\"Pitch angle about scanner Y-axis [deg]\",\"type\":\""
    "number\"},\"pitch_accuracy\":{\"minimum\":0,\"description\":\"Pitch angle "
    "accuracy [deg]\",\"type\":\"number\",\"exclusiveMinimum\":true},\"yaw_accuracy"
    "\":{\"minimum\":0,\"description\":\"Yaw angle accuracy [deg]\",\"type\":\"number"
    "\",\"exclusiveMinimum\":true},\"yaw\":{\"maximum\":360,\"minimum\":-360,\"descri"
    "ption\":\"Yaw angle about scanner Z-axis [deg]\",\"type\":\"number\"},\"roll_a"
    "ccuracy\":{\"minimum\":0,\"description\":\"Roll angle accuracy [deg]\",\"type\""
    ":\"number\",\"exclusiveMinimum\":true}},\"description\":\"Orientation values "
    "and orientation accuracies, measured with IMU or inclination sensors.\""
    ",\"type\":\"object\",\"required\":[\"roll\",\"pitch\",\"yaw\",\"roll_accuracy\",\"pit"
    "ch_accuracy\",\"yaw_accuracy\"]}},\"$schema\":\"http://json-schema.org/draft"
    "-04/schema#\",\"required\":[\"orientation\"],\"title\":\"Pose "
    "Estimation\",\"type\":\"object\",\"description\":\"Estimated position and "
    "orientation information as measured by GNSS, IMU or inclination "
    "sensors\"}"
    ,
    /* example: */
    "{\"position\":{\"crs\":{\"epsg\":4979,\"wkt\":\"GEOGCS[\\\"WGS84 / Geographic\\\",D"
    "ATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY["
    "\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\""
    ",0.0000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Degree\\\",0.01"
    "745329251994329547,AUTHORITY[\\\"EPSG\\\",\\\"9102\\\"]],AXIS[\\\"Latitude\\\",NOR"
    "TH],AXIS[\\\"Longitude\\\",EAST],AUTHORITY[\\\"EPSG\\\",\\\"4979\\\"]]\"},\"horizont"
    "al_accuracy\":0.810699999332428,\"vertical_accuracy\":1.3314999341964722,"
    "\"coordinate_2\":15.645033406,\"coordinate_3\":362.7124938964844,\"coordina"
    "te_1\":48.655799473},\"barometric_height_amsl\":386.7457796227932,\"orient"
    "ation\":{\"roll\":3.14743073066123,\"pitch\":1.509153024827064,\"pitch_accur"
    "acy\":0.009433783936875745,\"yaw_accuracy\":1.0094337839368757,\"yaw\":101."
    "87293630292045,\"roll_accuracy\":0.009433783936875745}}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"gyroscope\":{\"properties\":{\"origin\":{\"description\":\"Sen"
    "sor origin in SOCS [m] at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"y_axis\":{\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\",\"$ref\":\"#/definitions/vector\"},\"un"
    "it\":{\"minimum\":0,\"description\":\"Unit of raw data and calibration "
    "values, 1 LSB in rad/s\",\"type\":\"number\",\"exclusiveMinimum\":true},\"rela"
    "tive_nonlinearity\":{\"description\":\"Relative nonlinearity, dimensionles"
    "s\",\"$ref\":\"#/definitions/vector\"},\"x_axis\":{\"description\":\"Sensitive X"
    " axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"offset\":{\"description\":\"Value to be"
    " subtracted from raw measurement values\",\"$ref\":\"#/definitions/vector\""
    "},\"z_axis\":{\"description\":\"Sensitive Z axis of sensor at frame angle ="
    " 0\",\"$ref\":\"#/definitions/vector\"}},\"description\":\"Gyroscope details\","
    "\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\""
    ",\"origin\",\"relative_nonlinearity\"]},\"magnetic_field_sensor\":{\"properti"
    "es\":{\"relative_nonlinearity\":{\"description\":\"Relative nonlinearity, di"
    "mensionless\",\"$ref\":\"#/definitions/vector\"},\"y_axis\":{\"description\":\"S"
    "ensitive Y axis of sensor at frame angle = 0\",\"$ref\":\"#/definitions/ve"
    "ctor\"},\"unit\":{\"minimum\":0,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in nT\",\"type\":\"number\",\"exclusiveMinimum\":tr"
    "ue},\"fixed\":{\"description\":\"Distortion of magnetic field caused by "
    "non-rotating scanner part\",\"$ref\":\"#/definitions/vector\"},\"x_axis\":{\"d"
    "escription\":\"Sensitive X axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"offset\":{\"description\":\"Value to be"
    " subtracted from raw measurement values\",\"$ref\":\"#/definitions/vector\""
    "},\"z_axis\":{\"description\":\"Sensitive Z axis of sensor at frame angle ="
    " 0\",\"$ref\":\"#/definitions/vector\"}},\"description\":\"Magnetic Field "
    "Sensor details\",\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\""
    "z_axis\",\"offset\",\"fixed\",\"relative_nonlinearity\"]},\"accelerometer\":{\"p"
    "roperties\":{\"origin\":{\"description\":\"Sensor origin in SOCS [m] at "
    "frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"y_axis\":{\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\",\"$ref\":\"#/definitions/vector\"},\"un"
    "it\":{\"minimum\":0,\"description\":\"Unit of raw data and calibration "
    "values, 1 LSB in 9.81 m/s\\u00b2\",\"type\":\"number\",\"exclusiveMinimum\":tr"
    "ue},\"relative_nonlinearity\":{\"description\":\"Relative nonlinearity, dim"
    "ensionless\",\"$ref\":\"#/definitions/vector\"},\"x_axis\":{\"description\":\"Se"
    "nsitive X axis of sensor at frame angle = "
    "0\",\"$ref\":\"#/definitions/vector\"},\"offset\":{\"description\":\"Value to be"
    " subtracted from raw measurement values\",\"$ref\":\"#/definitions/vector\""
    "},\"z_axis\":{\"description\":\"Sensitive Z axis of sensor at frame angle ="
    " 0\",\"$ref\":\"#/definitions/vector\"}},\"description\":\"Accelerometer detai"
    "ls\",\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"off"
    "set\",\"origin\",\"relative_nonlinearity\"]}},\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\",\"definitions\":{\"vector\":{\"maxItems\":3,\"minItems"
    "\":3,\"type\":\"array\",\"items\":{\"description\":\"Index 0=X, 1=Y, 2=Z compone"
    "nt\",\"type\":\"number\"}}},\"required\":[\"gyroscope\",\"accelerometer\",\"magnet"
    "ic_field_sensor\"],\"title\":\"Pose "
    "Sensors\",\"type\":\"object\",\"description\":\"Details on position and "
    "orientation sensors\"}"
    ,
    /* example: */
    "{\"gyroscope\":{\"origin\":[0.026900000870227814,-0.03999999910593033,-0.0"
    "8950000256299973],\"y_axis\":[-0.440765917301178,-0.7897399663925171,119"
    ".5894775390625],\"unit\":0.00014544410805683583,\"relative_nonlinearity\":"
    "[2.888176311444113e-07,1.06274164579645e-07,-1.7186295080634935e-39],\""
    "x_axis\":[-121.195556640625,0.8219714164733887,0.2313031703233719],\"off"
    "set\":[-50.92609786987305,146.15643310546875,62.4327278137207],\"z_axis\""
    ":[0.555869996547699,119.22135162353516,0.467585027217865]},\"magnetic_f"
    "ield_sensor\":{\"relative_nonlinearity\":[0.0,0.0,0.0],\"y_axis\":[0.000278"
    "88521435670555,-0.011427424848079681,-5.204829722060822e-05],\"unit\":91"
    ".74311828613281,\"fixed\":[-1576.010498046875,1596.081787109375,0.0],\"x_"
    "axis\":[-0.011162743903696537,-2.315962774446234e-05,0.0001681884459685"
    "5342],\"offset\":[-23812.052734375,5606.57666015625,2493.28125],\"z_axis\""
    ":[0.00041987866279669106,7.876977906562388e-05,0.011407104320824146]},"
    "\"accelerometer\":{\"origin\":[0.026900000870227814,-0.03999999910593033,-"
    "0.08950000256299973],\"y_axis\":[-7.027288913726807,-44.12333679199219,1"
    "4952.3701171875],\"unit\":6.666666740784422e-05,\"relative_nonlinearity\":"
    "[0.0,0.0,0.0],\"x_axis\":[-15008.123046875,56.956390380859375,-60.517566"
    "6809082],\"offset\":[-733.3636474609375,58.969032287597656,1060.25500488"
    "28125],\"z_axis\":[1.639882206916809,15166.744140625,-116.99742889404297"
    "]}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"phase_step\":{\"maximum\":255,\"minimum\":0,\"description\":\""
    "Step width in phase of modulation code from line to line\",\"type\":\"inte"
    "ger\"},\"num_mod_ampl\":{\"maximum\":255,\"minimum\":0,\"description\":\"Number "
    "of different modulation amplitudes (2: binary modulation)\",\"type\":\"int"
    "eger\"},\"length\":{\"maximum\":255,\"minimum\":0,\"description\":\"Length of co"
    "de\",\"type\":\"integer\"},\"code_phase_mode\":{\"maximum\":255,\"minimum\":0,\"de"
    "scription\":\"0: no synchronization, 1: toggle between 2 phases, 2: "
    "increment with phase_increment\",\"type\":\"integer\"},\"pulse_interval\":{\"d"
    "escription\":\"Explicit table of the pulse position modulation used for "
    "MTA resolution. Table gives times between successive laser pulses in s"
    "econds.\",\"type\":\"array\",\"items\":{\"minimum\":0,\"type\":\"number\"}}},\"$sche"
    "ma\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"length\",\"nu"
    "m_mod_ampl\",\"pulse_interval\"],\"title\":\"Pulse Position "
    "Modulation\",\"type\":\"object\",\"description\":\"Laser pulse position "
    "modulation used for MTA resolution\"}"
    ,
    /* example: */
    "{\"phase_step\":5,\"num_mod_ampl\":2,\"length\":31,\"code_phase_mode\":2,\"puls"
    "e_interval\":[2.759375e-06,2.759375e-06,2.759375e-06,2.759375e-06,2.821"
    "875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.821875e-"
    "06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-06,2."
    "821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.75937"
    "5e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.821875e-06"
    ",2.821875e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.821875e-06]}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"maximum\":{\"description\":\"Maximum "
    "value\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum "
    "value\",\"type\":\"number\"},\"std_dev\":{\"description\":\"Standard "
    "deviation\",\"type\":\"number\"},\"average\":{\"description\":\"Average value\",\""
    "type\":\"number\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "required\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"title\":\"Range "
    "Statistics\",\"type\":\"object\",\"description\":\"Statistics about target "
    "distance wrt. SOCS origin\"}"
    ,
    /* example: */
    "{\"maximum\":574.35,\"minimum\":0.919,\"std_dev\":24.349,\"average\":15.49738}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"si\":{\"maximum\":255,\"minimum\":0,\"description\":\"Start "
    "index (hw_start)\",\"type\":\"number\"},\"sr\":{\"minimum\":0,\"exclusiveMinimum"
    "\":true,\"type\":\"number\",\"description\":\"Sample rate [Hz]\"},\"t\":{\"additio"
    "nalProperties\":false,\"patternProperties\":{\"^[0-9]+$\":{\"description\":\"o"
    "ne field per channel, field name is channel index\",\"$ref\":\"#/definitio"
    "ns/fp\"}},\"type\":\"object\"},\"nt\":{\"maximum\":255,\"minimum\":0,\"description"
    "\":\"Number of "
    "traces\",\"type\":\"integer\"},\"ex\":{\"description\":\"DEPRECATED, use "
    "'riegl.exponential_decomposition' "
    "instead\",\"type\":\"object\"},\"tbl\":{\"minItems\":1,\"description\":\"various "
    "internal data\",\"type\":\"array\",\"items\":{\"$ref\":\"#/definitions/fp_table\""
    "}},\"ns\":{\"maximum\":4095,\"minimum\":0,\"description\":\"Number of samples\","
    "\"type\":\"integer\"},\"mw\":{\"maximum\":4095,\"minimum\":0,\"exclusiveMinimum\":"
    "true,\"type\":\"number\",\"description\":\"Maximum "
    "weight\"},\"a\":{\"maxItems\":256,\"minItems\":1,\"description\":\"Amplitude [dB"
    "]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"definitions\":{\"fp\":{\"properties\":{\"w\":{\"m"
    "axItems\":256,\"minItems\":1,\"type\":\"array\",\"items\":{\"maxItems\":5,\"minIte"
    "ms\":5,\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"s\":{\"maxItems\":256,\""
    "minItems\":1,\"type\":\"array\",\"items\":{\"maxItems\":4096,\"minItems\":1,\"type"
    "\":\"array\",\"items\":{\"type\":\"number\"}}}},\"description\":\"Fingerprint valu"
    "es\",\"type\":\"object\",\"required\":[\"s\",\"w\"]},\"fp_table_row\":{\"maxItems\":2"
    "048,\"minItems\":1,\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"fp_table\":"
    "{\"desription\":\"scanner internal "
    "data\",\"properties\":{\"tc\":{\"min\":0,\"description\":\"table data type "
    "code\",\"type\":\"integer\",\"max\":255},\"ch\":{\"min\":0,\"description\":\"channel"
    " number\",\"type\":\"integer\",\"max\":255},\"tv\":{\"maxItems\":2048,\"minItems\":"
    "1,\"type\":\"array\",\"items\":{\"oneOf\":[{\"$ref\":\"#/definitions/fp_table_row"
    "\"},{\"type\":\"number\"}]}},\"nx\":{\"min\":1,\"description\":\"number of x entri"
    "es\",\"type\":\"integer\",\"max\":2048},\"ny\":{\"min\":1,\"description\":\"number "
    "of y entries\",\"type\":\"integer\",\"max\":2048}},\"type\":\"object\",\"required\""
    ":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv\"]}},\"title\":\"Receiver "
    "Internals\",\"type\":\"object\",\"description\":\"Receiver Internals\"}"
    ,
    /* example: */
    "{\"si\":48,\"sr\":7959997000.0,\"t\":{\"1\":{\"w\":[[78,86,126,134,31],[78,86,12"
    "6,134,31]],\"s\":[[1.23,4.56],[7.89,0.12]]},\"0\":{\"w\":[[78,86,126,134,31]"
    ",[78,86,126,134,31]],\"s\":[[1.23,4.56],[7.89,0.12]]}},\"nt\":128,\"tbl\":[{"
    "\"tc\":1,\"ch\":0,\"tv\":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.5]],\"nx\":5,\"ny\":2}]"
    ",\"ns\":400,\"mw\":31,\"a\":[-1.55]}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"delta\":{\"description\":\"Delta between table entries "
    "[m], first entry is at range = 0 m\",\"type\":\"number\"},\"content\":{\"maxIt"
    "ems\":2000,\"minItems\":1,\"description\":\"Correction value [dB] to be "
    "added to the amplitude\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"$s"
    "chema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"delta\",\""
    "content\"],\"title\":\"Reflectance Calculation "
    "Table\",\"type\":\"object\",\"description\":\"Lookup table for reflectance "
    "calculation based on amplitude and range\"}"
    ,
    /* example: */
    "{\"delta\":0.150918,\"content\":[-33.01]}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"reflectance_correction_db\":{\"description\":\"Near range "
    "reflectance correction in dB as a function of range over "
    "angle\",\"type\":\"array\",\"items\":{\"description\":\"rows (each array "
    "corresponds to a "
    "range)\",\"type\":\"array\",\"items\":{\"description\":\"columns (each value "
    "corresponds to an "
    "angle)\",\"type\":\"number\"}}},\"ranges_m\":{\"description\":\"Range [m]\",\"type"
    "\":\"array\",\"items\":{\"type\":\"number\"}},\"line_angles_deg\":{\"description\":"
    "\"Angle [deg]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"required\":[\"ranges_m\",\"line_an"
    "gles_deg\",\"reflectance_correction_db\"],\"title\":\"Near-range reflectance"
    " correction\",\"type\":\"object\",\"description\":\"Range-dependent and "
    "scan-angle-dependent correction of reflectance reading\"}"
    ,
    /* example: */
    "{\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0."
    "01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0"
    ".3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01]],\"ra"
    "nges_m\":[0.0,1.0,2.0,3.0],\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2.0,2"
    ".5,3.0,3.5,4.0]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"planes\":{\"properties\":{},\"description\":\"Information on"
    " plane correspondences and used extraction/matching templates.\",\"type\""
    ":\"object\"},\"trajectory\":{\"properties\":{},\"description\":\"Trajectory "
    "information and statistics.\",\"type\":\"object\"},\"georeferencing\":{\"prope"
    "rties\":{},\"description\":\"Internal georeferencing parameters.\",\"type\":\""
    "object\"},\"optimizer\":{\"properties\":{},\"description\":\"Optimization "
    "result information and statistics.\",\"type\":\"object\"}},\"$schema\":\"http:"
    "//json-schema.org/draft-04/schema#\",\"required\":[\"trajectory\",\"optimize"
    "r\",\"georeferencing\"],\"title\":\"RiLOC Processing "
    "Information\",\"type\":\"object\",\"description\":\"Details about RiLOC "
    "processing\"}"
    ,
    /* example: */
    "{\"planes\":{},\"trajectory\":{},\"georeferencing\":{},\"optimizer\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"definitions\":{\"p"
    "rogram\":{\"properties\":{\"laser_prr\":{\"minimum\":0,\"description\":\"Laser "
    "Pulse Repetition Rate [Hz]\",\"type\":\"number\",\"exclusiveMinimum\":false},"
    "\"name\":{\"description\":\"Name of measurement "
    "program\",\"type\":\"string\"}},\"description\":\"Measurement "
    "program\",\"type\":\"object\",\"required\":[\"name\"]}},\"title\":\"Scan Pattern\","
    "\"properties\":{\"line\":{\"properties\":{\"stop\":{\"maximum\":720.0,\"minimum\":"
    "0.0,\"description\":\"Stop angle in SOCS [deg]\",\"type\":\"number\"},\"start\":"
    "{\"maximum\":360.0,\"minimum\":0.0,\"description\":\"Start angle in SOCS [deg"
    "]\",\"type\":\"number\"},\"program\":{\"$ref\":\"#/definitions/program\"},\"increm"
    "ent\":{\"maximum\":90.0,\"minimum\":0.0,\"description\":\"Increment of angle "
    "in SOCS [deg]\",\"type\":\"number\"}},\"description\":\"Line Scan Pattern\",\"ty"
    "pe\":\"object\",\"required\":[\"start\",\"stop\",\"increment\"]},\"rectangular\":{\""
    "properties\":{\"phi_stop\":{\"maximum\":720.0,\"minimum\":0.0,\"description\":\""
    "Stop phi angle in SOCS [deg]\",\"type\":\"number\"},\"phi_start\":{\"maximum\":"
    "360.0,\"minimum\":0.0,\"description\":\"Start phi angle in SOCS [deg]\",\"typ"
    "e\":\"number\"},\"program\":{\"$ref\":\"#/definitions/program\"},\"theta_start\":"
    "{\"maximum\":180.0,\"minimum\":0.0,\"description\":\"Start theta angle in "
    "SOCS [deg]\",\"type\":\"number\"},\"phi_increment\":{\"maximum\":90.0,\"minimum\""
    ":0.0,\"description\":\"Increment of phi angle in SOCS [deg]\",\"type\":\"numb"
    "er\"},\"theta_stop\":{\"maximum\":180.0,\"minimum\":0.0,\"description\":\"Stop "
    "theta angle in SOCS [deg]\",\"type\":\"number\"},\"theta_increment\":{\"maximu"
    "m\":90.0,\"minimum\":0.0,\"description\":\"Increment of theta angle in SOCS "
    "[deg]\",\"type\":\"number\"}},\"description\":\"Rectangular Field Of View Scan"
    " Pattern\",\"type\":\"object\",\"required\":[\"phi_start\",\"phi_stop\",\"phi_incr"
    "ement\",\"theta_start\",\"theta_stop\",\"theta_increment\"]},\"segments\":{\"pro"
    "perties\":{\"program\":{\"$ref\":\"#/definitions/program\"},\"list\":{\"type\":\"a"
    "rray\",\"items\":{\"properties\":{\"stop\":{\"maximum\":720.0,\"minimum\":0.0,\"de"
    "scription\":\"Stop angle in SOCS [deg]\",\"type\":\"number\"},\"start\":{\"maxim"
    "um\":360.0,\"minimum\":0.0,\"description\":\"Start angle in SOCS [deg]\",\"typ"
    "e\":\"number\"},\"increment\":{\"maximum\":90.0,\"minimum\":0.0,\"description\":\""
    "Increment of angle in SOCS "
    "[deg]\",\"type\":\"number\"}},\"description\":\"Line Scan Segment\",\"type\":\"obj"
    "ect\",\"required\":[\"start\",\"stop\",\"increment\"]}}},\"description\":\"Segment"
    "ed Line Scan "
    "Pattern\",\"type\":\"object\",\"required\":[\"list\"]}},\"description\":\"Scan "
    "pattern description\"}"
    ,
    /* example: */
    "{\"rectangular\":{\"phi_stop\":270.0,\"phi_start\":45.0,\"program\":{\"laser_pr"
    "r\":100000.0,\"name\":\"High Speed\"},\"theta_start\":30.0,\"phi_increment\":0."
    "04,\"theta_stop\":130.0,\"theta_increment\":0.04}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"sh"
    "ot_file\":{\"properties\":{\"file_uuid\":{\"description\":\"File's Universally"
    " Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Shot file "
    "extension, without the leading dot\",\"type\":\"string\"}},\"type\":\"object\","
    "\"required\":[\"file_extension\"]}},\"title\":\"Shot "
    "Information\",\"type\":\"object\",\"description\":\"Details about laser shot "
    "files\"}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_uuid\":\"26a00815-67c0-4bff-8fe8-c577378fe663\",\"file"
    "_extension\":\"sodx\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"activated\":{\"description\":\"Apply ('true') or ignore "
    "('false') all "
    "filters\",\"type\":\"boolean\"},\"filters\":{\"description\":\"List of point fil"
    "ters\",\"type\":\"array\",\"items\":{\"properties\":{\"filter\":{\"description\":\"T"
    "he RDB filter string to apply (e.g. when reading points or index), "
    "details see documentation of function "
    "select()\",\"type\":\"string\"},\"activated\":{\"description\":\"Apply ('true') "
    "or ignore ('false') this "
    "filter\",\"type\":\"boolean\"},\"title\":{\"description\":\"A short filter title"
    " (e.g. for display in a selection "
    "list)\",\"type\":\"string\"},\"description\":{\"description\":\"A brief "
    "description of the filter (e.g. for display in a "
    "tooltip)\",\"type\":\"string\"}},\"description\":\"Point filter definition\",\"t"
    "ype\":\"object\",\"required\":[\"activated\",\"title\",\"description\",\"filter\"]}"
    "}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ac"
    "tivated\",\"filters\"],\"title\":\"Stored "
    "filters\",\"type\":\"object\",\"description\":\"Point filters applied in "
    "addition to the application-defined filters\"}"
    ,
    /* example: */
    "{\"activated\":true,\"filters\":[{\"filter\":\"riegl.mta_uncertain_point == "
    "0\",\"activated\":false,\"title\":\"Ignore uncertain points from MTA "
    "resolution\",\"description\":\"Ignore points with uncertain MTA zone "
    "assignment\"},{\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\",\"activated\":false,\"title\":\"Ignore points below dynamic trigger "
    "level\",\"description\":\"Ignore points with an echo signal amplitude "
    "below the dynamic trigger\"},{\"filter\":\"riegl.point_outside_aabb_bocs "
    "== 0\",\"activated\":false,\"title\":\"Ignore points outside outer "
    "AABB\",\"description\":\"Ignore points that are outside the outer AABB in "
    "BOCS\"},{\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\",\"activated\":false,\"title\":\"Ignore points inside inner "
    "AABB\",\"description\":\"Ignore points that are inside the inner AABB in "
    "BOCS\"}]}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"author\":{\"description\":\"Name of the author that "
    "created the document\",\"type\":\"string\"},\"system\":{\"description\":\"The "
    "actual system description, details see documentation of RIEGL System "
    "Description\",\"type\":\"object\"},\"version\":{\"description\":\"Document "
    "format version\",\"type\":\"string\"},\"$class\":{\"enum\":[\"Document\"],\"descri"
    "ption\":\"Object class "
    "name\",\"type\":\"string\"},\"timestamp\":{\"description\":\"Date and time of "
    "creation (e.g. 2019-06-19T13:32:10+02:00)\",\"type\":\"string\"}},\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"$class\",\"versi"
    "on\",\"author\",\"timestamp\",\"system\"],\"title\":\"System "
    "Description\",\"type\":\"object\",\"description\":\"Details of major system "
    "components like lidar sensors, cameras and navigation devices\"}"
    ,
    /* example: */
    "{\"author\":\"RIEGL LMS GmbH, Austria\",\"system\":{\"details\":\"see "
    "documentation of RIEGL System Description\"},\"version\":\"1.2.1\",\"$class\""
    ":\"Document\",\"timestamp\":\"2022-09-30T11:56:26+00:00\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"Si\":{\"description\":\"Conversion table for Si channel\",\""
    "$ref\":\"#/definitions/conversion_table\"},\"InGaAs\":{\"description\":\"Conve"
    "rsion table for InGaAs channel\",\"$ref\":\"#/definitions/conversion_table"
    "\"},\"InGaAs_Si_Difference\":{\"description\":\"Conversion table for InGaAs "
    "- Si difference\",\"$ref\":\"#/definitions/conversion_table\"}},\"$schema\":\""
    "http://json-schema.org/draft-04/schema#\",\"definitions\":{\"conversion_ta"
    "ble\":{\"properties\":{\"value\":{\"description\":\"LSB [1]\",\"type\":\"array\",\"i"
    "tems\":{\"type\":\"number\"}},\"temperature\":{\"description\":\"Temperature [\\u"
    "00b0C]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"type\":\"object\",\"re"
    "quired\":[\"value\",\"temperature\"]}},\"title\":\"Temperature Calculation "
    "Table\",\"type\":\"object\",\"description\":\"Conversion of background "
    "radiation raw values to temperatures in \\u00b0C\"}"
    ,
    /* example: */
    "{\"Si\":{\"value\":[0.0,64.00097659230323,128.0019531846065,192.0029297769"
    "097,256.0039063692129,320.00488296151616,384.0058595538194,448.0068361"
    "461226,512.0078127384259],\"temperature\":[546.300048828125,548.81640512"
    "12026,551.3143938500972,554.0144257850053,556.604252334815,559.2124464"
    "488079,561.8022729986177,564.4104671126105,567.0002936624203]},\"InGaAs"
    "\":{\"value\":[0.0,64.00097659230323,128.0019531846065,192.0029297769097,"
    "256.0039063692129,320.00488296151616,384.0058595538194,448.00683614612"
    "26,512.0078127384259],\"temperature\":[307.22196722535614,309.1153478247"
    "277,311.1188086915047,313.10025350480055,315.2137946389828,317.2172555"
    "057597,319.2207163725366,321.2021611858325,323.3157023200148]},\"InGaAs"
    "_Si_Difference\":{\"value\":[1000.0,1100.090029602954,1200.04425183874,13"
    "00.1342814416948,1400.0885036774805,1500.0427259132668,1600.1327555162"
    "209,1700.0869777520065,1800.0411999877924],\"temperature\":[1749.9771111"
    "17893,1749.977111117893,1749.977111117893,1749.977111117893,1749.97711"
    "1117893,1749.977111117893,1744.7813348796044,1681.9971312601092,1622.3"
    "944822534868]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"epoch\":{\"description\":\"Date and time of timestamp '0' "
    "as proposed by RFC 3339 (e.g. 2015-10-27T00:00:00+01:00).\",\"type\":\"str"
    "ing\"},\"system\":{\"enum\":[\"unknown\",\"UTC\",\"GPS\"],\"description\":\"Time "
    "system (time standard)\",\"type\":\"string\"},\"source\":{\"enum\":[\"unknown\",\""
    "RTC\",\"GNSS\"],\"description\":\"Timestamp source\",\"type\":\"string\"}},\"$sche"
    "ma\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"epoch\",\"sou"
    "rce\"],\"title\":\"Time Base\",\"type\":\"object\",\"description\":\"Base of "
    "timestamps (epoch)\"}"
    ,
    /* example: */
    "{\"epoch\":\"2015-10-27T00:00:00+00:00\",\"system\":\"UTC\",\"source\":\"GNSS\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"navigation_frame\":{\"enum\":[\"unknown\",\"NED\",\"ENU\"],\"des"
    "cription\":\"Navigation frame (NED: North-East-Down, ENU: East-North-Up)"
    "\",\"type\":\"string\"},\"field_of_application\":{\"enum\":[\"unknown\",\"SLS\",\"TL"
    "S\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"description\":\"Field of "
    "application\",\"type\":\"string\"},\"software\":{\"description\":\"Software that"
    " calculated the trajectory (this may be the same or different software"
    " than the one that created the file)\",\"type\":\"string\"},\"time_interval\""
    ":{\"properties\":{\"maximum\":{\"description\":\"Maximum time interval "
    "[s]\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum time interval "
    "[s]\",\"type\":\"number\"},\"std_dev\":{\"description\":\"Standard deviation of "
    "intervals [s]\",\"type\":\"number\"},\"average\":{\"description\":\"Average time"
    " interval [s]\",\"type\":\"number\"}},\"description\":\"Time interval statisti"
    "cs\",\"type\":\"object\",\"required\":[\"minimum\",\"average\",\"maximum\",\"std_dev"
    "\"]},\"settings\":{\"description\":\"Settings used to calculate the "
    "trajectory (descriptive "
    "text)\",\"type\":\"string\"},\"company\":{\"description\":\"Company "
    "name\",\"type\":\"string\"},\"device\":{\"description\":\"Navigation device, "
    "e.g. "
    "name/type/serial\",\"type\":\"string\"},\"project\":{\"description\":\"Project "
    "name\",\"type\":\"string\"},\"location\":{\"description\":\"Project location, "
    "e.g. city/state/country\",\"type\":\"string\"}},\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"required\":[\"time_interval\",\"navigation_frame"
    "\"],\"title\":\"Trajectory "
    "Information\",\"type\":\"object\",\"description\":\"Details about "
    "position+orientation files\"}"
    ,
    /* example: */
    "{\"navigation_frame\":\"NED\",\"field_of_application\":\"MLS\",\"software\":\"Nav"
    "igation Software XYZ\",\"time_interval\":{\"maximum\":0.005004883,\"minimum\""
    ":0.00500032,\"std_dev\":5.51e-07,\"average\":0.00500053},\"settings\":\"defau"
    "lt\",\"company\":\"RIEGL LMS\",\"device\":\"IMU Model 12/1, Serial# "
    "12345\",\"project\":\"Campaign 3\",\"location\":\"Horn\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"ext_delay\":{\"minimum\":0,\"description\":\"external "
    "trigger, time delay [0.1 msec]\",\"type\":\"integer\"},\"in_max_delay\":{\"min"
    "imum\":0,\"description\":\"maximum delay to output pulse before fake event"
    " is generated [0.1 msec], zero indicates that no fake events are gener"
    "ated\",\"type\":\"integer\"},\"out_polarity\":{\"minimum\":0,\"description\":\"0 "
    ".. positive edge, 1 .. negative "
    "edge\",\"type\":\"integer\"},\"input_descr\":{\"description\":\"descriptive "
    "string\",\"type\":\"string\"},\"input_usage\":{\"minimum\":0,\"description\":\"0 "
    ".. deactivated, 1 .. just detecting and timestamping\",\"type\":\"integer\""
    "},\"out_duration\":{\"minimum\":0,\"description\":\"output pulse duration "
    "[0.1 "
    "msec]\",\"type\":\"integer\"},\"output_descr\":{\"description\":\"descriptive st"
    "ring\",\"type\":\"string\"},\"out_num_pulses\":{\"minimum\":0,\"description\":\"nu"
    "mber of output pulses to be generated, 0 .. infinite\",\"type\":\"integer\""
    "},\"ext_bitmask\":{\"minimum\":0,\"description\":\"defines which of the "
    "internal pulse generators are to be started\",\"type\":\"integer\"},\"output"
    "_usage\":{\"minimum\":0,\"description\":\"0 .. no output, 1 .. output "
    "derived from internal clock, 2 .. output derived from dmi, 3 .. output"
    " derived from external signal, 4 .. output static low, 5 .. output "
    "static "
    "high\",\"type\":\"integer\"},\"ext_channel\":{\"minimum\":0,\"description\":\"ID "
    "of channel used as external trigger input, 32 indicates "
    "none\",\"type\":\"integer\"},\"channel\":{\"minimum\":0,\"description\":\"ID of "
    "input/output "
    "channel\",\"type\":\"integer\"},\"dmi_dist_per_tick\":{\"description\":\"dmi, "
    "distance per tick, just informative [m]\",\"type\":\"number\"},\"in_max_dura"
    "tion\":{\"minimum\":0,\"description\":\"stops measurement of pulse duration "
    "of input signal [0.1 "
    "msec]\",\"type\":\"integer\"},\"dmi_incr\":{\"description\":\"dmi, increment in "
    "ticks\",\"type\":\"integer\"},\"in_min_duration\":{\"minimum\":0,\"description\":"
    "\"input signals with smaller pulse durations are ignored [0.1 msec]\",\"t"
    "ype\":\"integer\"},\"out_delay\":{\"minimum\":0,\"description\":\"output pulse "
    "initial delay after start [0.1 msec]\",\"type\":\"integer\"},\"dmi_max_time\""
    ":{\"minimum\":0,\"description\":\"dmi, maximum time interval between "
    "trigger events [0.1 msec]\",\"type\":\"integer\"},\"ext_subdivider\":{\"minimu"
    "m\":0,\"description\":\"reduces the frequency, default "
    "1\",\"type\":\"integer\"},\"dmi_min_time\":{\"minimum\":0,\"description\":\"dmi, "
    "minimum time interval between trigger events [0.1 msec]\",\"type\":\"integ"
    "er\"},\"num_channel\":{\"minimum\":0,\"description\":\"number of input/output "
    "channels\",\"type\":\"integer\"},\"ttip_version\":{\"minimum\":0,\"description\":"
    "\"following main.sub.ss.sss\",\"type\":\"integer\"},\"out_interval\":{\"minimum"
    "\":0,\"description\":\"output pulse interval [0.1 "
    "msec]\",\"type\":\"integer\"},\"ext_signal\":{\"minimum\":0,\"description\":\"0 .."
    " use input signal, 1 .. use output signal of "
    "channel\",\"type\":\"integer\"},\"in_polarity\":{\"minimum\":0,\"description\":\"0"
    " .. positive edge, 1 .. negative edge\",\"type\":\"integer\"}},\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\",\"required\":[\"ttip_version\",\"nu"
    "m_channel\",\"ext_channel\",\"ext_signal\",\"ext_delay\",\"ext_subdivider\",\"ex"
    "t_bitmask\",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick"
    "\",\"channel\",\"output_descr\",\"input_descr\",\"output_usage\",\"input_usage\","
    "\"out_polarity\",\"out_duration\",\"out_interval\",\"out_delay\",\"out_num_puls"
    "es\",\"in_polarity\",\"in_min_duration\",\"in_max_duration\",\"in_max_delay\"],"
    "\"title\":\"TTIP Configuration\",\"type\":\"object\",\"description\":\"Trigger-Ti"
    "mestamping-Unit/IP configuration\"}"
    ,
    /* example: */
    "{\"ext_delay\":0,\"in_max_delay\":9990,\"out_polarity\":0,\"input_descr\":\"Por"
    "t 1 - "
    "Exposure\",\"input_usage\":1,\"out_duration\":300,\"output_descr\":\"Port 1 - "
    "Trigger\",\"out_num_pulses\":1,\"ext_bitmask\":0,\"output_usage\":1,\"ext_chan"
    "nel\":32,\"channel\":0,\"dmi_dist_per_tick\":0.0,\"in_max_duration\":10000,\"d"
    "mi_incr\":0,\"in_min_duration\":0,\"out_delay\":0,\"dmi_max_time\":0,\"ext_sub"
    "divider\":1,\"dmi_min_time\":0,\"num_channel\":9,\"ttip_version\":1,\"out_inte"
    "rval\":10000,\"ext_signal\":0,\"in_polarity\":0}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"properties\":{\"ve"
    "rtex_file\":{\"properties\":{\"file_uuid\":{\"description\":\"File's "
    "Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Vertex file "
    "extension, without the leading dot\",\"type\":\"string\"}},\"type\":\"object\","
    "\"required\":[\"file_extension\"]}},\"title\":\"Vertex "
    "Information\",\"type\":\"object\",\"description\":\"Details about vertex "
    "file\"}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_uuid\":\"51534d95-d71f-4f36-ae1a-3e63a21fd1c7\",\"fi"
    "le_extension\":\"vtx\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"oneOf\":[{\"properties\":{\"size\":{\"oneOf\":[{\"$ref\":\"#/definitions/voxel"
    "_size\"},{\"$ref\":\"#/definitions/voxel_size_cubic\"}],\"description\":\"Size"
    " of voxels in file coordinate system.\"},\"voxel_origin\":{\"$ref\":\"#/defi"
    "nitions/voxel_origin_enum\"},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_"
    "type\"},\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_thresholds\"}},\""
    "additionalProperties\":false,\"required\":[\"size\",\"voxel_origin\",\"voxel_t"
    "ype\"]},{\"properties\":{\"size_llcs\":{\"description\":\"Size of voxels in a "
    "locally levelled cartesian coordinate system (xyz). This is only used "
    "for voxels based on a map projection.\",\"$ref\":\"#/definitions/voxel_siz"
    "e\"},\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_thresholds\"},\"size"
    "\":{\"description\":\"Size of voxels in file coordinate system.\",\"$ref\":\"#"
    "/definitions/voxel_size\"},\"voxel_origin\":{\"oneOf\":[{\"$ref\":\"#/definiti"
    "ons/voxel_origin_enum\"},{\"description\":\"The base point of the voxel "
    "grid. Used together with <tt>voxel_size</tt> and <tt>voxel_index</tt> "
    "to compute actual point coordinates.\",\"$ref\":\"#/definitions/voxel_orig"
    "in_point\"}]},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_type\"},\"referen"
    "ce_point\":{\"maxItems\":2,\"minItems\":2,\"description\":\"Point in WGS84 "
    "geodetic decimal degree (EPSG:4326) that was used to compute the "
    "projection distortion parameters. The coefficient order is latitude, "
    "longitude. Only voxels with corresponding geo_tag, voxel_size and "
    "reference_point can be reliably processed together. This entry is "
    "available for voxel files in projected CRS only.\",\"type\":\"array\",\"item"
    "s\":{\"maximum\":180,\"minimum\":-180,\"type\":\"number\"}}},\"additionalPropert"
    "ies\":false,\"required\":[\"reference_point\",\"size_llcs\",\"size\",\"voxel_ori"
    "gin\",\"voxel_type\"]}],\"$schema\":\"http://json-schema.org/draft-04/schema"
    "#\",\"definitions\":{\"edge_length\":{\"minimum\":0,\"description\":\"Length of "
    "voxel edge [m].\",\"type\":\"number\",\"exclusiveMinimum\":true},\"shape_thres"
    "holds\":{\"properties\":{\"line\":{\"minimum\":1,\"description\":\"If the "
    "biggest eigenvalue is bigger than the median eigenvalue * "
    "line_threshold, the voxel is considered a line.\",\"type\":\"number\",\"excl"
    "usiveMinimum\":true},\"plane\":{\"maximum\":1,\"minimum\":0,\"exclusiveMaximum"
    "\":true,\"description\":\"If the smallest eigenvalue is smaller than the "
    "median eigenvalue * plane_threshold, the voxel is considered a plane.\""
    ",\"type\":\"number\",\"exclusiveMinimum\":true}},\"description\":\"Thresholds "
    "used to compute the voxel's shape_id value.\",\"type\":\"object\",\"required"
    "\":[\"plane\",\"line\"]},\"voxel_size_cubic\":{\"type\":\"number\",\"$ref\":\"#/defi"
    "nitions/edge_length\"},\"voxel_size\":{\"maxItems\":3,\"minItems\":3,\"descrip"
    "tion\":\"Size of voxels.\",\"type\":\"array\",\"items\":{\"$ref\":\"#/definitions/"
    "edge_length\"}},\"voxel_origin_enum\":{\"description\":\"Defines whether the"
    " voxel's center or a corner is placed on CRS origin <tt>(0/0/0)</tt>.\""
    ",\"enum\":[\"center\",\"corner\"],\"default\":\"corner\"},\"voxel_type\":{\"descrip"
    "tion\":\"Whether a point in a voxel represents its center or its "
    "centroid. If type is <tt>index</tt> there is no point but only an "
    "integer voxel index.\",\"enum\":[\"center\",\"centroid\",\"index\"],\"default\":\""
    "centroid\"},\"voxel_origin_point\":{\"maxItems\":3,\"minItems\":3,\"descriptio"
    "n\":\"Origin point for all voxel indices in voxel "
    "CRS.\",\"type\":\"array\",\"items\":{\"type\":\"number\"}}},\"title\":\"Voxel "
    "Information\",\"type\":\"object\",\"description\":\"Details about the voxels "
    "contained in the file\"}"
    ,
    /* example: */
    "{\"size_llcs\":[0.5156575252891171,0.5130835356683303,0.5143705304787237"
    "],\"shape_thresholds\":{\"line\":6,\"plane\":0.16},\"size\":[0.597164283477939"
    "5,0.5971642834779395,0.5143705304787237],\"voxel_origin\":\"corner\",\"voxe"
    "l_type\":\"centroid\",\"reference_point\":[48,16]}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"mta_zone\":{\"minimum\":1,\"description\":\"Fixed MTA zone "
    "for averaging.\",\"type\":\"integer\"},\"num_shots\":{\"minimum\":1,\"descriptio"
    "n\":\"Number of consecutive shots to be used for averaging.\",\"type\":\"int"
    "eger\"},\"trim\":{\"maximum\":0.5,\"minimum\":0,\"description\":\"Percentage for"
    " robust averaging.\",\"type\":\"number\",\"default\":0}},\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"required\":[\"num_shots\",\"mta_zone\"],\"t"
    "itle\":\"Waveform Averaging "
    "Settings\",\"type\":\"object\",\"description\":\"Settings for waveform "
    "averaging\"}"
    ,
    /* example: */
    "{\"mta_zone\":1,\"num_shots\":7,\"trim\":0.05}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"sample_data_files\":{\"type\":\"array\",\"items\":{\"propertie"
    "s\":{\"channel\":{\"maximum\":255,\"minimum\":0,\"exclusiveMaximum\":false,\"exc"
    "lusiveMinimum\":false,\"type\":\"integer\",\"description\":\"Sample block "
    "channel number (255 = invalid)\"},\"channel_name\":{\"description\":\"Sample"
    " block channel name\",\"type\":\"string\"},\"sample_interval\":{\"minimum\":0,\""
    "description\":\"Sampling interval in seconds\",\"type\":\"number\",\"exclusive"
    "Minimum\":false},\"sample_bits\":{\"maximum\":32,\"minimum\":0,\"exclusiveMaxi"
    "mum\":false,\"exclusiveMinimum\":false,\"type\":\"integer\",\"description\":\"Bi"
    "twidth of samples (e.g. 10 bit, 12 "
    "bit)\"},\"file_extension\":{\"description\":\"Sample data file extension, "
    "without the leading dot\",\"type\":\"string\"},\"delta_st\":{\"description\":\"r"
    "eserved\",\"type\":\"number\"},\"laser_wavelength\":{\"minimum\":0,\"description"
    "\":\"Laser wavelength in meters (0 = unknown)\",\"type\":\"number\",\"exclusiv"
    "eMinimum\":false},\"file_uuid\":{\"description\":\"File's Universally Unique"
    " Identifier (RFC 4122)\",\"type\":\"string\"}},\"type\":\"object\",\"required\":["
    "\"channel\",\"channel_name\",\"sample_interval\",\"sample_bits\",\"laser_wavele"
    "ngth\",\"delta_st\",\"file_extension\"]}},\"range_offset_m\":{\"description\":\""
    "Calibrated device specific range offset for waveform analysis by "
    "system response fitting in meters.\",\"type\":\"number\"},\"sample_block_fil"
    "e\":{\"properties\":{\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Sample block "
    "file extension, without the leading dot\",\"type\":\"string\"}},\"type\":\"obj"
    "ect\",\"required\":[\"file_extension\"]},\"range_offset_waveform_samples_m\":"
    "{\"description\":\"Calibrated device specific range offset for waveform "
    "samples in meters.\",\"type\":\"number\"}},\"description\":\"Details about "
    "waveform files\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"t"
    "itle\":\"Waveform Information\",\"type\":\"object\",\"required\":[\"sample_block"
    "_file\",\"sample_data_files\"]}"
    ,
    /* example: */
    "{\"sample_data_files\":[{\"channel\":0,\"channel_name\":\"high_power\",\"sample"
    "_interval\":1.00503e-09,\"sample_bits\":12,\"file_extension\":\"sp0\",\"delta_"
    "st\":0,\"laser_wavelength\":0,\"file_uuid\":\"da084413-e3e8-4655-a122-071de8"
    "490d8e\"},{\"channel\":1,\"channel_name\":\"low_power\",\"sample_interval\":1.0"
    "0503e-09,\"sample_bits\":12,\"file_extension\":\"sp1\",\"delta_st\":0,\"laser_w"
    "avelength\":0,\"file_uuid\":\"93585b5e-5ea9-43a1-947b-e7ba3be642d2\"},{\"cha"
    "nnel\":5,\"channel_name\":\"wwf\",\"sample_interval\":1.00503e-09,\"sample_bit"
    "s\":12,\"file_extension\":\"sp5\",\"delta_st\":0,\"laser_wavelength\":0,\"file_u"
    "uid\":\"9d2298c4-1036-464f-b5cb-1cf8e517f3a0\"}],\"range_offset_m\":3.1415,"
    "\"sample_block_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660"
    "\",\"file_extension\":\"sbx\"},\"range_offset_waveform_samples_m \":7.283}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"items\":{\"propert"
    "ies\":{\"enabled\":{\"description\":\"Waveform output "
    "enabled\",\"type\":\"boolean\"},\"logic_expression\":{\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\",\"type\":\"string\"},\"pass_ampl_greater\":{\"description\":\"Threshold"
    " for amplitude greater than "
    "[dB]\",\"type\":\"number\"},\"pass_rng_less\":{\"description\":\"Threshold for "
    "range less than "
    "[m]\",\"type\":\"number\"},\"pass_dev_greater\":{\"description\":\"Threshold for"
    " deviation greater than "
    "[1]\",\"type\":\"integer\"},\"channel_idx_mask\":{\"description\":\"Bit mask for"
    " channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, ...\",\"type\":\"integer\"},\"smart_enabled\":{\"description\":\"Smart "
    "waveform output enabled\",\"type\":\"boolean\"},\"pass_rng_greater\":{\"descri"
    "ption\":\"Threshold for range greater than "
    "[m]\",\"type\":\"number\"},\"pass_ampl_less\":{\"description\":\"Threshold for "
    "amplitude less than "
    "[dB]\",\"type\":\"number\"},\"sbl_name\":{\"description\":\"Name of sample "
    "block, e.g.: wfm, "
    "wwf\",\"type\":\"string\"},\"pass_dev_less\":{\"description\":\"Threshold for "
    "deviation less than [1]\",\"type\":\"integer\"}},\"type\":\"object\",\"required\""
    ":[\"sbl_name\",\"enabled\",\"channel_idx_mask\"]},\"title\":\"Waveform "
    "Settings\",\"type\":\"array\",\"description\":\"Scanner settings for waveform "
    "output\"}"
    ,
    /* example: */
    "[{\"enabled\":true,\"pass_ampl_greater\":1.0,\"pass_rng_less\":13.11,\"channe"
    "l_idx_mask\":11,\"smart_enabled\":true,\"pass_rng_greater\":9.27,\"pass_ampl"
    "_less\":5.0,\"sbl_name\":\"wfm\"},{\"enabled\":false,\"channel_idx_mask\":32,\"s"
    "bl_name\":\"wwf\"}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"result\":{\"properties\":{\"range_mean\":{\"description\":\"[m"
    "]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"angle\":{\"description\":\"["
    "deg]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"timestamp\":{\"descript"
    "ion\":\"[s]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"amplitude_mean\":"
    "{\"description\":\"[dB]\",\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"ampli"
    "tude_offset\":{\"description\":\"[dB]\",\"type\":\"array\",\"items\":{\"type\":\"num"
    "ber\"}},\"range_sigma\":{\"description\":\"[m]\",\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\"}},\"amplitude_sigma\":{\"description\":\"[dB]\",\"type\":\"array\",\""
    "items\":{\"type\":\"number\"}}},\"type\":\"object\",\"required\":[\"angle\",\"range_"
    "mean\",\"range_sigma\",\"amplitude_mean\",\"amplitude_sigma\",\"amplitude_offs"
    "et\"]},\"settings\":{\"properties\":{\"range\":{\"properties\":{\"additive_value"
    "\":{\"type\":\"number\"},\"sigma_factor\":{\"type\":\"number\"}},\"type\":\"object\","
    "\"required\":[\"sigma_factor\",\"additive_value\"]},\"amplitude\":{\"properties"
    "\":{\"additive_value\":{\"type\":\"number\"},\"sigma_factor\":{\"type\":\"number\"}"
    "},\"type\":\"object\",\"required\":[\"sigma_factor\",\"additive_value\"]}},\"type"
    "\":\"object\",\"required\":[\"range\",\"amplitude\"]},\"filter\":{\"properties\":{\""
    "amplitude_max\":{\"description\":\"[dB]\",\"type\":\"array\",\"items\":{\"type\":\"n"
    "umber\"}},\"range_min\":{\"description\":\"[m]\",\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\"}},\"range_max\":{\"description\":\"[m]\",\"type\":\"array\",\"items\":"
    "{\"type\":\"number\"}},\"angle\":{\"description\":\"[deg]\",\"type\":\"array\",\"item"
    "s\":{\"type\":\"number\"}}},\"type\":\"object\",\"required\":[\"angle\",\"range_min\""
    ",\"range_max\",\"amplitude_max\"]}},\"$schema\":\"http://json-schema.org/draf"
    "t-04/schema#\",\"required\":[\"result\",\"filter\",\"settings\"],\"title\":\"Windo"
    "w Analysis\",\"type\":\"object\",\"description\":\"Window analysis data "
    "estimated from scandata and resulting filter parameters\"}"
    ,
    /* example: */
    "{\"result\":{\"range_mean\":[0.1105621,0.1079564,0.1087088,0.1067261,0.105"
    "4582,0.1090412,0.102871,0.1019044,0.1051523,0.1058445,0.1031261],\"angl"
    "e\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],\"timestamp"
    "\":[408.4441,411.4443],\"amplitude_mean\":[5.347396,5.263155,5.224655,5.1"
    "79926,5.097782,5.116479,5.051756,4.983473,5.007885,5.002441,4.982],\"am"
    "plitude_offset\":[1.9,1.9],\"range_sigma\":[0.01869652,0.02151435,0.01747"
    "969,0.01918765,0.01945776,0.01934862,0.01955329,0.02225589,0.02229977,"
    "0.01899122,0.02009433],\"amplitude_sigma\":[0.4272844,0.4298479,0.423681"
    "6,0.4283583,0.4362353,0.4315141,0.4373984,0.4472798,0.4346001,0.434548"
    "7,0.4540681]},\"settings\":{\"range\":{\"additive_value\":0.1,\"sigma_factor\""
    ":8},\"amplitude\":{\"additive_value\":1.0,\"sigma_factor\":4}},\"filter\":{\"am"
    "plitude_max\":[8.04,8.01,7.99,7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78],\"r"
    "ange_min\":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.218,-0.219,-0.221,-0.2"
    "23,-0.225,-0.227],\"range_max\":[0.424,0.425,0.426,0.427,0.428,0.428,0.4"
    "29,0.43,0.431,0.431,0.432],\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5"
    ",15.6,15.7,15.8,15.9]}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range\":{\"properties\":{\"maximum\":{\"maximum\":2.0,\"minimu"
    "m\":-2.0,\"description\":\"Maximum range in m\",\"type\":\"number\"},\"minimum\":"
    "{\"maximum\":2.0,\"minimum\":-2.0,\"description\":\"Minimum range in "
    "m\",\"type\":\"number\"},\"entries\":{\"minimum\":1,\"description\":\"Number of "
    "range entries\",\"type\":\"integer\"}},\"description\":\"Range axis of "
    "correction table\",\"type\":\"object\",\"required\":[\"minimum\",\"maximum\",\"ent"
    "ries\"]},\"slices\":{\"type\":\"array\",\"items\":{\"properties\":{\"table\":{\"minI"
    "tems\":1,\"description\":\"Correction table (dimension defined by the "
    "'amplitude' and 'range' "
    "objects)\",\"type\":\"array\",\"items\":{\"minItems\":1,\"description\":\"Table "
    "row (= amplitude axis)\",\"type\":\"array\",\"items\":{\"maxItems\":3,\"minItems"
    "\":3,\"description\":\"Table column (= range "
    "axis)\",\"type\":\"array\",\"items\":{\"description\":\"Table cell (item 0: "
    "amplitude in dB, 1: range in m, 2: "
    "flags)\",\"type\":\"number\"}}}},\"amplitude\":{\"description\":\"Window echo "
    "amplitude of slice in dB\",\"type\":\"number\"}},\"description\":\"Window echo"
    " correction parameter slice\",\"type\":\"object\",\"required\":[\"amplitude\",\""
    "table\"]}},\"amplitude\":{\"properties\":{\"maximum\":{\"minimum\":0.0,\"descrip"
    "tion\":\"Maximum amplitude in "
    "dB\",\"type\":\"number\"},\"minimum\":{\"minimum\":0.0,\"description\":\"Minimum "
    "amplitude in "
    "dB\",\"type\":\"number\"},\"entries\":{\"minimum\":1,\"description\":\"Number of "
    "amplitude entries\",\"type\":\"integer\"}},\"description\":\"Amplitude axis of"
    " correction table\",\"type\":\"object\",\"required\":[\"minimum\",\"maximum\",\"en"
    "tries\"]}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"require"
    "d\":[\"amplitude\",\"range\",\"slices\"],\"title\":\"Window Echo "
    "Correction\",\"type\":\"object\",\"description\":\"Correction parameters for "
    "window glass echoes\"}"
    ,
    /* example: */
    "{\"range\":{\"maximum\":1.5060822940732335,\"minimum\":-1.5060822940732335,\""
    "entries\":128},\"slices\":[{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]"
    "],\"amplitude\":1.5},{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],\"am"
    "plitude\":2.0}],\"amplitude\":{\"maximum\":20,\"minimum\":2,\"entries\":128}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const std::size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);
const std::vector<const MetaDataItemInfo*> RDB_META_DATA_ITEMS_VECTOR(
    RDB_META_DATA_ITEMS, RDB_META_DATA_ITEMS + RDB_META_DATA_ITEM_COUNT
);

}}} // namespace riegl::rdb::pointcloud

#endif // RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
