/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    createSettings.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Database create settings
 * \version 2015-10-14/AW: Initial version
 * \version 2016-07-13/AW: Documentation of 'chunkSizeLOD' updated
 * \version 2016-09-22/AW: New functions to get/set 64 bit cache size parameter
 * \version 2016-12-20/AW: New functions to load/save settings from/to JSON
 * \version 2017-03-28/AW: Documentation of JSON load/save functions updated
 * \version 2019-01-15/AW: Parameter 'chunkMode' added
 * \version 2019-01-18/AW: Parameter 'lodMode' added
 * \version 2019-10-30/AW: Parameter 'optimizePointID' added (#3458)
 * \version 2020-02-21/AW: Add "context-free" function variants (#3544)
 * \version 2024-11-29/AW: Parameter 'pointInsertMode' added (#5339)
 *
 * This class defines settings for creating a new point cloud database.
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_CREATESETTINGS_H
#define RIEGL_RDB_POINTCLOUD_CREATESETTINGS_H

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"
#include "riegl/rdb/pointcloud/pointAttribute.h"

//---< TYPE DEFINITIONS >-------------------------------------------------------

typedef struct RDBPointcloudCreateSettings RDBPointcloudCreateSettings; // forward declaration of implementation details

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN

//______________________________________________________________________________
/*!
 * \brief Default constructor
 *
 * All values are set to default values.
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_new,
    RDBContext                   *context, //!< [in] library context
    RDBPointcloudCreateSettings **settings //!< [out] handle of created object
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_new_cf,
    RDBPointcloudCreateSettings **settings //!< [out] handle of created object
);

//______________________________________________________________________________
/*!
 * \brief Destroy settings object
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_delete,
    RDBContext                   *context, //!< [in] library context
    RDBPointcloudCreateSettings **settings //!< [in] handle of object to delete
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_delete_cf,
    RDBPointcloudCreateSettings **settings //!< [in] handle of object to delete
);

//______________________________________________________________________________
/*!
 * \brief Load settings from JSON string
 *
 * This function parses the given JSON string and applies all available
 * properties - missing properties are silently ignored (i.e. the value
 * remains unchanged). When parsing the JSON string fails, an exception
 * is thrown.
 *
 * Example JSON string:
 *
 *     {
 *         "primary_attribute": {
 *             "name": "riegl.xyz",
 *             "title": "XYZ",
 *             "description": "Cartesian point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)",
 *             "unit_symbol": "m",
 *             "length": 3,
 *             "resolution": 0.00025,
 *             "minimum_value": -535000.0,
 *             "maximum_value": 535000.0,
 *             "default_value": 0.0,
 *             "storage_class": "variable",
 *             "compression_options": "shuffle",
 *             "scale_factor": 1.0
 *         },
 *         "chunk_mode": "point_count",
 *         "chunk_size": 65536,
 *         "lod_mode": "thinout",
 *         "chunk_size_lod": 20,
 *         "cache_size": 524288000,
 *         "compression_level": 10,
 *         "optimize_point_id": false,
 *         "point_insert_mode": "single-pass"
 *     }
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_json_load,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                    json      //!< [in] JSON string
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_json_load_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                    json      //!< [in] JSON string
);

//______________________________________________________________________________
/*!
 * \brief Save settings to JSON string
 * \see rdb_pointcloud_create_settings_json_load()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_json_save,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                   *json      //!< [out] JSON string
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_json_save_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                   *json      //!< [out] JSON string
);

//______________________________________________________________________________
/*!
 * \brief Get primary point attribute
 *
 * The primary point attribute defines the attribute that is used
 * to sort and index the points. Usually the 3D point coordinates are
 * used for that. The primary attribute is automatically added to the
 * point cloud (using Pointcloud::attributeAdd()) and cannot be deleted.
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_primary_attribute,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBPointcloudPointAttribute *value     //!< [out] current attribute
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_primary_attribute_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBPointcloudPointAttribute *value     //!< [out] current attribute
);

//______________________________________________________________________________
/*!
 * \brief Set primary point attribute
 * \see rdb_pointcloud_create_settings_get_primary_attribute()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_primary_attribute,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBPointcloudPointAttribute *value     //!< [in] new attribute
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_primary_attribute_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBPointcloudPointAttribute *value     //!< [in] new attribute
);

//______________________________________________________________________________
/*!
 * \brief Get point chunk mode
 *
 * Points are internally organized in chunks (primary point attribute index
 * tree leaves). The size of a chunk (in the dimension of the primary point
 * attribute) may either be fixed (predefined) or adapted automatically so
 * that the number of points in a chunk does not exceed a certain limit. In
 * both cases, the "size" is defined by parameter CreateSettings::chunkSize
 * and parameter CreateSettings::chunkMode defines the meaning of the value.
 *
 * POINT_COUNT = 1
 *   the chunk size defines the maximum number of points per chunk
 *   (the default mode)
 *
 * EDGE_LENGTH = 2
 *   the chunk size defines the edge length of a chunk as 2^N times
 *   resolution of the primary point attribute
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_chunk_mode,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_chunk_mode_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set point chunk mode
 * \see rdb_pointcloud_create_settings_get_chunk_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_chunk_mode,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_chunk_mode_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Get point chunk size
 * \see rdb_pointcloud_create_settings_get_chunk_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_chunk_size,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_chunk_size_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set point chunk size
 * \see rdb_pointcloud_create_settings_get_chunk_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_chunk_size,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_chunk_size_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Get level of detail mode
 *
 * A tree structure is used to sort and organize the point cloud. To create
 * a coarse representation of the point cloud (level of detail = "LOD"), a
 * number of equally distributed points is extracted from the tree leaf
 * nodes and copied to the parent nodes.
 *
 * The parameter CreateSettings::chunkSizeLOD defines how many points to
 * extract for LOD whereas the meaning of the value and the LOD creation
 * algorithm are defined by the parameter CreateSettings::lodMode.
 *
 * THINOUT = 1
 *   the LOD size defines the number of points to copy as a fraction of
 *   the total (original) number of points. So if the original point count
 *   is for example 19820526 and the size is set to 20%, then the number
 *   of LOD points to add is 3964106 (rounded) and the final total number
 *   of points is 23784632 (actual value may differ a little bit).
 *
 * COMBINE = 2
 *   the LOD size defines the number of binary subdivisions of the LOD
 *   node's volume in each dimension. So if the primary point attribute
 *   for example has a length of 2 (2D data) and the LOD size is set to 8,
 *   then each LOD node is divided into 2^8 * 2^8 = 2^(8*2) = 2^16 = 65536
 *   sub-volumes. All points of the node's immediate sub-nodes that fall
 *   into one of the sub-volumes are merged to a single point and stored
 *   in the LOD node. The method to merge the attribute values of a group
 *   of points can be defined for each point attribute separately (details
 *   see class PointAttribute).
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_lod_mode,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_lod_mode_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set level of detail mode
 * \see rdb_pointcloud_create_settings_get_lod_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_lod_mode,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_lod_mode_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Get level of detail size
 * \see rdb_pointcloud_create_settings_get_lod_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_chunk_size_lod,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_chunk_size_lod_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set level of detail size
 * \see rdb_pointcloud_create_settings_get_lod_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_chunk_size_lod,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_chunk_size_lod_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Get point cache size
 *
 * The database engine may buffer read and write operations in an
 * internal cache. This value defines the cache size in bytes (octets).
 *
 * Default: 500 MB
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_cache_size,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_cache_size_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Get point cache size
 * \see rdb_pointcloud_create_settings_get_cache_size()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_cache_size_64,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint64_t                    *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_cache_size_64_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint64_t                    *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set point cache size
 * \see rdb_pointcloud_create_settings_get_cache_size()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_cache_size,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_cache_size_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Set point cache size
 * \see rdb_pointcloud_create_settings_set_cache_size()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_cache_size_64,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint64_t                     value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_cache_size_64_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint64_t                     value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Get data compression level
 *
 * The database automatically compresses point data before it is
 * stored. The compression level defines the compression quality,
 * i.e. higher values produce smaller files.
 *
 * Range: 0..100 (i.e. percent)
 * Default: 10%
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_compression_level,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint8_t                     *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_compression_level_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint8_t                     *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set data compression level
 * \see rdb_pointcloud_create_settings_get_compression_level()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_compression_level,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint8_t                      value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_compression_level_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint8_t                      value     //!< [in] new value
);

//______________________________________________________________________________
/*!
 * \brief Get point ID optimization
 *
 * Enable this option to apply optimizations to the point ID
 * attribute (riegl.id) that can result in smaller files.
 *
 * Default: false
 *
 * \note No optimizations are applied when:
 *       - points were inserted in a previous transaction
 *       - buffers for the point ID (riegl.id) or dynamic point attributes
 *         (e.g. "riegl.selected", "riegl.visible") are passed to the insert
 *         query (QueryInsert class)
 *
 * \warning When optimizations are enabled, the point ID no longer
 *          reflects the order in which the points were inserted.
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_optimize_point_id,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value (0: false)
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_optimize_point_id_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                    *value     //!< [out] current value (0: false)
);

//______________________________________________________________________________
/*!
 * \brief Set point ID optimization
 * \see rdb_pointcloud_create_settings_get_optimize_point_id()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_optimize_point_id,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value (0: false)
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_optimize_point_id_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    uint32_t                     value     //!< [in] new value (0: false)
);

//______________________________________________________________________________
/*!
 * \brief Get point insert mode
 *
 * This setting specifies how points are to be inserted into the database.
 *
 * Supported values:
 *   - "single-pass" (default)
 *          Points are sorted and inserted into the internal index structure
 *          each time QueryInsert::next() is called. This is suitable for
 *          scenarios where the same objects are only scanned once or just
 *          a few times (e.g in TLS or MLS).
 *
 *   - "multi-pass"
 *          Points are sorted and inserted into the internal index structure
 *          after the last call to QueryInsert::next() when the transaction is
 *          committed. This is suitable for scenarios where the same objects are
 *          scanned multiple times (e.g. in ALS or ULS when the system passes
 *          over the same area several times without toggling data acquisition).
 *          This mode is only supported for 3D point clouds (e.g. when the
 *          primary point attribute is "riegl.xyz") and when no points have
 *          been inserted before. Otherwise "single-pass" mode is used instead.
 *          This mode implies "optimizePointID" and point IDs are not returned
 *          by the insert query.
 *
 * \note It is recommended to only check whether the string starts with one of
 *       the above values and to ignore all subsequent characters (mode-specific
 *       parameters may be added after the mode name in the future).
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_get_point_insert_mode,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                   *value     //!< [out] current value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_get_point_insert_mode_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                   *value     //!< [out] current value
);

//______________________________________________________________________________
/*!
 * \brief Set point insert mode
 * \see rdb_pointcloud_create_settings_get_point_insert_mode()
 */
RDB_FUNCTION(rdb_pointcloud_create_settings_set_point_insert_mode,
    RDBContext                  *context,  //!< [in] library context
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                    value     //!< [in] new value
);
RDB_FUNCTION_CF(rdb_pointcloud_create_settings_set_point_insert_mode_cf,
    RDBPointcloudCreateSettings *settings, //!< [in] handle of object
    RDBString                    value     //!< [in] new value
);

RDB_LIBRARY_API_END

#endif // RIEGL_RDB_POINTCLOUD_CREATESETTINGS_H
