#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
#*******************************************************************************
#
#  Copyright 2026 RIEGL Laser Measurement Systems
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
#  SPDX-License-Identifier: Apache-2.0
#
#*******************************************************************************
#
"""
rdb-example-08-invert-points.py

This example shows how to open an existing database and to invert
a point attribute by switching the "riegl.selected" and "riegl.visible"
bits from 0 to 1 and vice-versa ("bit flip").
Please note that this could also be accomplished by combining select and
update queries as shown in "rdb-example-4-update-points.cpp", but the
invert query might be faster (lower processing time) and easier to use.
This example is based on the database of rdb-example-1-create-database.
"""

import riegl.rdb

# Access existing database
with riegl.rdb.rdb_open("pointcloud.rdbx") as rdb:
    # Before we can modify the database, we must start a transaction
    with riegl.rdb.Transaction(
        rdb,  # point cloud object
        "Invert",  # transaction title
        "Point Switcher v1.0"  # software name
    ) as transaction:

        def define_attribute(name, title, description, default_value):
            """Helper function to define a dynamic point attribute"""
            if not rdb.point_attributes.exists(name):
                attribute = riegl.rdb.PointAttribute(rdb)
                attribute.name = name
                attribute.title = title
                attribute.description = description
                attribute.unit_symbol = ""
                attribute.length = 1
                attribute.resolution = 1
                attribute.minimum_value = 0
                attribute.maximum_value = 1
                attribute.default_value = default_value
                attribute.storage_class = riegl.rdb.PointAttribute.StorageClass.DYNAMIC
                rdb.point_attributes.add(attribute)

        # Make sure that "selected" and "visible" attributes are defined
        define_attribute("riegl.selected", "Selected", "Point selected", 0.0)
        define_attribute("riegl.visible", "Visible", "Point visible", 1.0)

        # Flip the "riegl.selected" bits of all points
        with rdb.invert() as invert:
            invert.attribute("riegl.selected")
            while invert.next(10000) > 0:
                print(f"{invert.progress()}% done")

        # Flip the "riegl.visible" bits of all selected points (i.e.
        # where the point attribute "riegl.selected" is set to 1).
        with rdb.invert("riegl.selected == 1") as invert:
            invert.attribute("riegl.visible")
            while invert.next(10000) > 0:
                print(f"{invert.progress()}% done")

        # Finally commit transaction
        transaction.commit()
