/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-02-insert-points.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 2: Insert points
 * \version 2015-10-14/AW: Initial version
 * \version 2018-07-10/AW: Use new bindBuffer() instead of bind() function
 *
 *  This example shows how to open an existing database and add some points.
 *  This example is based on the database of rdb-example-1-create-database.
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <array>
#include <vector>
#include <random>
#include <cstdint>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>
#include <riegl/rdb/default.hpp>

int main()
{
    try
    {
        // New RDB library context
        riegl::rdb::Context context;

        // Access existing database
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);

        // Query some attribute details
        using riegl::rdb::pointcloud::PointAttribute;
        const PointAttribute detailsCoordinates = rdb.pointAttribute().get("riegl.xyz");
        const PointAttribute detailsReflectance = rdb.pointAttribute().get("riegl.reflectance");

        // Prepare random number generator (for dummy points)
        std::uniform_real_distribution<double> randomCoordinates(
            detailsCoordinates.minimumValue,
            detailsCoordinates.maximumValue
        );
        std::uniform_real_distribution<float> randomReflectance(
            float(detailsReflectance.minimumValue),
            float(detailsReflectance.maximumValue)
        );
        std::default_random_engine rng;

        // Before we can modify the database, we must start a transaction
        riegl::rdb::pointcloud::TransactionScope transaction(
            rdb,                  // point cloud object
            "Import",             // transaction title
            "Point Importer v1.0" // software name
        );

        // Prepare point attribute buffers
        const uint32_t BUFFER_SIZE =  10000; // point block/chunk size
        const uint32_t POINT_COUNT = 150000; // total number of points
        std::vector< std::array<double, 3> > bufferCoordinates(BUFFER_SIZE);
        std::vector< float >                 bufferReflectance(BUFFER_SIZE);

        // Start new insert query
        riegl::rdb::pointcloud::QueryInsert query = rdb.insert();

        // Tell insert query where to read the "reflectance" values from
        if (0) // you can either specify the name of the point attribute...
        {
            query.bindBuffer(
                "riegl.reflectance",
                bufferReflectance
            );
        }
        else // ...or use one of the constants from "riegl/rdb/default.hpp":
        {
            query.bindBuffer(
                riegl::rdb::pointcloud::RDB_RIEGL_REFLECTANCE,
                bufferReflectance
            );
        }

        // Tell insert query where to read the point coordinates from
        if (0) // buffers for vectors can be bound for all vector elements at once...
        {
            query.bindBuffer("riegl.xyz", bufferCoordinates);
        }
        else // ...or for each vector element separately:
        {
            // The 'stride' defines the number of bytes between the buffer
            // locations of the attribute values of two consecutive points.
            const int32_t stride = sizeof(bufferCoordinates[0]);
            query.bindBuffer("riegl.xyz[0]", bufferCoordinates[0][0], stride);
            query.bindBuffer("riegl.xyz[1]", bufferCoordinates[0][1], stride);
            query.bindBuffer("riegl.xyz[2]", bufferCoordinates[0][2], stride);
        }

        // Insert points block-wise
        for (uint32_t total = 0; total < POINT_COUNT;)
        {
            // Fill buffers with some random data
            for (uint32_t i = 0; i < BUFFER_SIZE; i++)
            {
                bufferCoordinates[i][0] = randomCoordinates(rng);
                bufferCoordinates[i][1] = randomCoordinates(rng);
                bufferCoordinates[i][2] = randomCoordinates(rng);
                bufferReflectance[i]    = randomReflectance(rng);
            }

            // Actually insert points
            total += query.next(BUFFER_SIZE);
            static uint32_t block = 1;
            std::cout << "block: " << block++ << ", "
                      << "total: " << total << std::endl;
        }

        // Finally commit transaction
        transaction.commit();

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
