{*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \file    riegl.rdb.pointcloud.queryStat.pas
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point statistics query (Pascal wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-07/AW: Optionally filter index graph nodes (#2390)
 * \version 2018-07-05/AW: Wrappers for Minimum() and Maximum() added
 * \version 2019-11-05/AW: Optionally return cleaned attribute extents (#3071)
 *
 * This query provides point attribute statistics like minimum and
 * maximum value.
 *
 * \see riegl::rdb::Pointcloud::stat()
 *
 *******************************************************************************
 *}

unit riegl.rdb.pointcloud.queryStat;

{******************************************************************************}
{***} INTERFACE {**************************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb,
  riegl.rdb.context,
  riegl.rdb.pointcloud.dataTypes,
  riegl.rdb.pointcloud.graphNode,
  riegl.rdb.pointcloud.transaction,
  riegl.rdb.pointcloud.pointAttribute,
  riegl.rdb.default.attributes;

//---< RDB LIBRARY IMPORTS >----------------------------------------------------

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::stat()
 *}
function rdb_pointcloud_query_stat_new(
  Context    : TRDBContextHandle;            //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;         //!< [in] point cloud
  var Query  : TRDBPointcloudQueryStatHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Destroy query instance
 *}
function rdb_pointcloud_query_stat_delete(
  Context   : TRDBContextHandle;            //!< [in] library context
  var Query : TRDBPointcloudQueryStatHandle //!< [in] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Get index graph
 *
 * This function returns a simple directed graph which represents
 * the index structure that is used to organize the point cloud.
 *
 * The optional filter expression can be used to select particular
 * graph nodes - if no filter is given, all nodes will be returned.
 * Filter expression syntax see riegl::rdb::Pointcloud::select().
 *
 * Note: The reported point counts and attribute extents are not
 * affected by the filter expressions given here or defined in the
 * meta data item riegl.stored_filters.
 *
 * Details see description of class GraphNode.
 *}
function rdb_pointcloud_query_stat_index(
  Context : TRDBContextHandle;             //!< [in] library context
  Query   : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  Root    : TRDBPointcloudGraphNodeHandle  //!< [in] graph root node
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;
function rdb_pointcloud_query_stat_index_filter(
  Context : TRDBContextHandle;             //!< [in] library context
  Query   : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  Root    : TRDBPointcloudGraphNodeHandle; //!< [in] graph root node
  Filter  : TRDBString                     //!< [in] filter string
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Lowest value of node
 *
 * Provides the attribute's minimum value of a branch (i.e. node and children).
 * If the node ID is zero, then the minimum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *}
function rdb_pointcloud_query_stat_minimum(
  Context   : TRDBContextHandle;             //!< [in] library context
  Query     : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  NodeID    : TRDBPointcloudGraphNodeID;     //!< [in] node identifier
  Attribute : TRDBString;                    //!< [in] attribute name
  DataType  : TRDBUInt32;                    //!< [in] data type of target buffer
  Buffer    : Pointer                        //!< [out] buffer for minimum value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Lowest value of node
 *
 * Provides the attribute's minimum value of a branch (i.e. node and children),
 * ignores invalid values (if the point attribute defines an invalid value).
 * If the node ID is zero, then the minimum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *}
function rdb_pointcloud_query_stat_minimum_cleaned(
  Context   : TRDBContextHandle;             //!< [in] library context
  Query     : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  NodeID    : TRDBPointcloudGraphNodeID;     //!< [in] node identifier
  Attribute : TRDBString;                    //!< [in] attribute name
  DataType  : TRDBUInt32;                    //!< [in] data type of target buffer
  Buffer    : Pointer                        //!< [out] buffer for minimum value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Highest value of node
 *
 * Provides the attribute's maximum value of a branch (i.e. node and children).
 * If the node ID is zero, then the maximum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *}
function rdb_pointcloud_query_stat_maximum(
  Context   : TRDBContextHandle;             //!< [in] library context
  Query     : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  NodeID    : TRDBPointcloudGraphNodeID;     //!< [in] node identifier
  Attribute : TRDBString;                    //!< [in] attribute name
  DataType  : TRDBUInt32;                    //!< [in] data type of target buffer
  Buffer    : Pointer                        //!< [out] buffer for maximum value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Highest value of node
 *
 * Provides the attribute's maximum value of a branch (i.e. node and children),
 * ignores invalid values (if the point attribute defines an invalid value).
 * If the node ID is zero, then the maximum value of all points is returned.
 *
 * The target buffer is expected to be s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *}
function rdb_pointcloud_query_stat_maximum_cleaned(
  Context   : TRDBContextHandle;             //!< [in] library context
  Query     : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  NodeID    : TRDBPointcloudGraphNodeID;     //!< [in] node identifier
  Attribute : TRDBString;                    //!< [in] attribute name
  DataType  : TRDBUInt32;                    //!< [in] data type of target buffer
  Buffer    : Pointer                        //!< [out] buffer for maximum value
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Attribute revision
 *
 * Provides the ID of the last transaction that has modified the attribute
 * in any node of the given branch.
 *}
function rdb_pointcloud_query_stat_revision(
  Context      : TRDBContextHandle;             //!< [in] library context
  Query        : TRDBPointcloudQueryStatHandle; //!< [in] query handle
  NodeID       : TRDBPointcloudGraphNodeID;     //!< [in] node identifier
  Attribute    : TRDBString;                    //!< [in] attribute name
  var Revision : TRDBPointcloudTransactionID    //!< [out] transaction ID, see above
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//---< CLASS TRDBPointcloudQueryStat >------------------------------------------
{*!
 * \brief Point statistics query
 *
 * This query provides point attribute statistics like minimum and
 * maximum value.
 *
 * \see riegl::rdb::Pointcloud::stat()
 *}
type
  TRDBPointcloudQueryStat = class(System.TObject)
  public
    {*!
     * \brief Constructor
     * \note  You cannot create new TRDBPointcloudQueryStat objects directly,
     *        use riegl::rdb::Pointcloud::stat() instead.
     *}
    constructor Create(Parent : System.TObject); reintroduce;
    destructor  Destroy; override;

    {*!
     * \brief Get index graph
     *
     * This function returns a simple directed graph which represents
     * the index structure that is used to organize the point cloud.
     *
     * Each graph node covers a certain range of the index space and
     * has a number of sub-nodes (aka. "child nodes"). All child nodes
     * lie within the range of the parent node whereas they are usually
     * smaller. A node without child nodes is called a leaf node. Note
     * that any nodes may overlap in index space as well as all other
     * point dimensions (attributes).
     *
     * The optional filter expression can be used to select particular
     * graph nodes - if no filter is given, all nodes will be returned.
     * Filter expression syntax see riegl::rdb::Pointcloud::select().
     *
     * Note: The reported point counts and attribute extents are not
     * affected by the filter expressions given here or defined in the
     * meta data item riegl.stored_filters.
     *
     * \returns root node of graph
     *
     * \note Bear in mind to delete the returned object when done!
     *}
    function Index(
      const Filter : System.String = '' //!< [in] optional filter expression
    ) : TRDBPointcloudGraphNode;

    {*!
     * \brief Lowest value of node
     *
     * Provides the attribute's minimum value of a branch (i.e. node and children).
     * If the node ID is zero, then the minimum value of all points is returned.
     *
     * The target buffer is expected to be s*d bytes large, where
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *}
    procedure Minimum(
        const NodeID    : TRDBPointcloudGraphNodeID; //!< [in] node identifier
        const Attribute : System.String;             //!< [in] attribute name
        const DataType  : TRDBPointcloudDataType;    //!< [in] data type of target buffer
        Buffer          : Pointer;                   //!< [out] buffer for minimum value
        const Cleaned   : Boolean = FALSE            //!< [in] TRUE: ignore invalid values
    ); overload;

    // Helper for fundamental types
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBFloat64); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64); overload;

    // Helper for pointer types
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBFloat64); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64); overload;

    // Helper for array types
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8);    overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);    overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64);   overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);   overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64); overload;
    procedure Minimum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64); overload;
    procedure Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64); overload;

    {*!
     * \brief Highest value of node
     *
     * Provides the attribute's maximum value of a branch (i.e. node and children).
     * If the node ID is zero, then the maximum value of all points is returned.
     *
     * The target buffer is expected to be s*d bytes large, where
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *}
    procedure Maximum(
        const NodeID    : TRDBPointcloudGraphNodeID; //!< [in] node identifier
        const Attribute : System.String;             //!< [in] attribute name
        const DataType  : TRDBPointcloudDataType;    //!< [in] data type of target buffer
        Buffer          : Pointer;                   //!< [out] buffer for maximum value
        const Cleaned   : Boolean = FALSE            //!< [in] TRUE: ignore invalid values
    ); overload;

    // Helper for fundamental types
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBFloat64); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64); overload;

    // Helper for pointer types
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    const Buffer : PRDBFloat64); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    const Buffer : PRDBFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64); overload;

    // Helper for array types
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8);    overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);    overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);  overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64);   overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);   overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64); overload;
    procedure Maximum(const Node   : TRDBPointcloudGraphNode;   const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64); overload;
    procedure Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64); overload;

    {*!
     * \brief Attribute revision
     *
     * Provides the ID of the last transaction that has modified the attribute.
     *}
    procedure Revision(
        const NodeID    : TRDBPointcloudGraphNodeID;  //!< [in] node identifier
        const Attribute : System.String;              //!< [in] attribute name
        var   Revision  : TRDBPointcloudTransactionID //!< [out] transaction ID, see above
    );

  private
    FContext    : TRDBContext;
    FPointcloud : TRDBPointcloudHandle;
    FQuery      : TRDBPointcloudQueryStatHandle;
  end;

{******************************************************************************}
{***} IMPLEMENTATION {*********************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  SysUtils,
  riegl.rdb.pointcloud;

//---< TRDBPointcloudQueryStat::PUBLIC >----------------------------------------

constructor TRDBPointcloudQueryStat.Create(Parent : System.TObject);
begin
  inherited Create;
  FQuery      := nil;
  FContext    := (Parent as TRDBPointcloud).Context;
  FPointcloud := (Parent as TRDBPointcloud).Pointcloud;
  FContext.Check(rdb_pointcloud_query_stat_new(FContext.Handle, FPointcloud, FQuery));
end; // Create()

destructor TRDBPointcloudQueryStat.Destroy;
begin
  FContext.Check(rdb_pointcloud_query_stat_delete(FContext.Handle, FQuery));
  inherited;
end; // Destroy()

function TRDBPointcloudQueryStat.Index(const Filter : System.String) : TRDBPointcloudGraphNode;
type
  PRDBPointcloudGraphNodeHandle = ^TRDBPointcloudGraphNodeHandle;

  TGraphNode = packed record
    Id              : TRDBPointcloudGraphNodeID;         //!< unique node identifier
    Revision        : TRDBPointcloudTransactionID;       //!< ID of last transaction that has modified any attribute of this branch
    Children        : PRDBPointcloudGraphNodeHandle;     //!< list of pointers to child nodes
    ChildCount      : TRDBPointcloudGraphNodeChildCount; //!< number of child nodes
    PointCountTotal : TRDBPointcloudGraphNodePointCount; //!< total number of points in all leaf nodes of the branch
    PointCountNode  : TRDBPointcloudGraphNodePointCount; //!< number of points in this node (see notes about LOD)
  end;
  PGraphNode = ^TGraphNode;

  procedure CopyGraph(Source : PGraphNode; Target : TRDBPointcloudGraphNode);
  var
    i : TRDBPointcloudGraphNodeChildCount;
    Child : PRDBPointcloudGraphNodeHandle;
  begin
    Target.Id              := Source.Id;
    Target.Revision        := Source.Revision;
    Target.PointCountTotal := Source.PointCountTotal;
    Target.PointCountNode  := Source.PointCountNode;
    Target.ChildCount      := Source.ChildCount;
    if (Source.ChildCount > 0) then
    begin
      Child := Source.Children; // first
      for i := 0 to Source.ChildCount-1 do
      begin
        CopyGraph(PGraphNode(Child^), Target.Children[i]);
        Inc(Child); // go to next child
      end;
    end;
  end; // CopyGraph()

var
  RootIn  : TRDBPointcloudGraphNodeHandle;
  RootOut : TRDBPointcloudGraphNode;
begin
  FContext.Check(rdb_pointcloud_graph_node_new(FContext.Handle, RootIn));
  try
    FContext.Check(rdb_pointcloud_query_stat_index_filter(
      FContext.Handle, FQuery, RootIn,
      TRDBString(AnsiToUtf8(Filter))
    ));
    RootOut := TRDBPointcloudGraphNode.Create;
    CopyGraph(PGraphNode(RootIn), RootOut);
    Result := RootOut; RootOut := nil;
  finally
    FContext.Check(rdb_pointcloud_graph_node_delete(FContext.Handle, RootIn));
    SysUtils.FreeAndNil(RootOut);
  end; // try..finally
end; // Index()

procedure TRDBPointcloudQueryStat.Minimum(
  const NodeID    : TRDBPointcloudGraphNodeID;
  const Attribute : System.String;
  const DataType  : TRDBPointcloudDataType;
  Buffer          : Pointer;
  const Cleaned   : Boolean
);
begin
  if (Cleaned) then
  begin
    FContext.Check(rdb_pointcloud_query_stat_minimum_cleaned(
      FContext.Handle, FQuery, NodeID,
      TRDBString(AnsiToUtf8(Attribute)),
      Ord(DataType), Buffer
    ));
  end
  else // full:
  begin
    FContext.Check(rdb_pointcloud_query_stat_minimum(
      FContext.Handle, FQuery, NodeID,
      TRDBString(AnsiToUtf8(Attribute)),
      Ord(DataType), Buffer
    ));
  end;
end; // Minimum()

procedure TRDBPointcloudQueryStat.Maximum(
  const NodeID    : TRDBPointcloudGraphNodeID;
  const Attribute : System.String;
  const DataType  : TRDBPointcloudDataType;
  Buffer          : Pointer;
  const Cleaned   : Boolean
);
begin
  if (Cleaned) then
  begin
    FContext.Check(rdb_pointcloud_query_stat_maximum_cleaned(
      FContext.Handle, FQuery, NodeID,
      TRDBString(AnsiToUtf8(Attribute)),
      Ord(DataType), Buffer
    ));
  end
  else // full:
  begin
    FContext.Check(rdb_pointcloud_query_stat_maximum(
      FContext.Handle, FQuery, NodeID,
      TRDBString(AnsiToUtf8(Attribute)),
      Ord(DataType), Buffer
    ));
  end;
end; // Maximum()

procedure TRDBPointcloudQueryStat.Revision(
  const NodeID    : TRDBPointcloudGraphNodeID;
  const Attribute : System.String;
  var   Revision  : TRDBPointcloudTransactionID
);
begin
  FContext.Check(rdb_pointcloud_query_stat_revision(
    FContext.Handle, FQuery, NodeID, TRDBString(AnsiToUtf8(Attribute)), Revision
  ));
end; // Revision()

//______________________________________________________________________________
//
// Helper for fundamental types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt8);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt8);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt8);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt8);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt8);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt8);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt8);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt8);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt16);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt16);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt16);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt16);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt16);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt16);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt16);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt16);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBUInt64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBUInt64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBInt64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBInt64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBFloat32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBFloat32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBFloat32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBFloat32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBFloat64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBFloat64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBFloat64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBFloat64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

//______________________________________________________________________________
//
// Helper for pointer types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt8);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt8);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt8);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt8);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt8);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt8);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt8);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt8);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt16);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt16);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt16);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt16);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt16);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt16);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt16);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt16);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBUInt64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBUInt64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBInt64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBInt64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBFloat32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBFloat32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBFloat32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBFloat32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBFloat64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBFloat64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; const Buffer : PRDBFloat64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; const Buffer : PRDBFloat64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

//______________________________________________________________________________
//
// Helper for array types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt8);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt8);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt8);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt8);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt8);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt8);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt8);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt8);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt16);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt16);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt16);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt16);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt16);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt16);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt16);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt16);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfUInt64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfUInt64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfInt64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfInt64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfFloat32);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfFloat32);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfFloat32);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfFloat32);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfFloat64);
begin
  Minimum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64);
begin
  Minimum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfFloat64);
begin
  Minimum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Minimum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64);
begin
  Minimum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : System.String; var Buffer : TRDBArrayOfFloat64);
begin
  Maximum(Node.Id, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const Node : TRDBPointcloudGraphNode; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64);
begin
  Maximum(Node.Id, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : System.String; var Buffer : TRDBArrayOfFloat64);
begin
  Maximum(NodeID, Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryStat.Maximum(const NodeID : TRDBPointcloudGraphNodeID; const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64);
begin
  Maximum(NodeID, Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

end.
