{*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \file    riegl.rdb.pointcloud.queryRemove.pas
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point remove query (Pascal wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2018-07-06/AW: Wrapper function BindBuffer() added
 *
 * This class can be used to remove (delete) existing points.
 *
 * \see riegl::rdb::Pointcloud::remove()
 *
 *******************************************************************************
 *}

unit riegl.rdb.pointcloud.queryRemove;

{******************************************************************************}
{***} INTERFACE {**************************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb,
  riegl.rdb.context,
  riegl.rdb.pointcloud.dataTypes,
  riegl.rdb.pointcloud.pointAttribute,
  riegl.rdb.default.attributes;

//---< RDB LIBRARY IMPORTS >----------------------------------------------------

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::remove()
 *}
function rdb_pointcloud_query_remove_new(
  Context    : TRDBContextHandle;              //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;           //!< [in] point cloud
  var Query  : TRDBPointcloudQueryRemoveHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Destroy query instance
 *}
function rdb_pointcloud_query_remove_delete(
  Context   : TRDBContextHandle;              //!< [in] library context
  var Query : TRDBPointcloudQueryRemoveHandle //!< [in] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Bind attribute buffer
 *
 * Use this function to define a source buffer for a point attribute.
 * Exactly one buffer can be defined for an attribute (i.e. only the
 * most recently defined buffer will be used).
 *
 * The buffer is expected to be n*s*d bytes large, where
 * __n__ is the number of points defined in next(),
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *
 * \note This function just stores the buffer pointer - it does
 *       __NOT__ copy the data contained in the buffer. So make
 *       sure that the buffer remains valid until you call next().
 *
 * \note This function expects a buffer for the point ID attribute.
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 *}
function rdb_pointcloud_query_remove_bind(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQueryRemoveHandle; //!< [in] query handle
  Attribute : TRDBString;                      //!< [in] attribute name
  DataType  : TRDBUInt32;                      //!< [in] buffer data type \see dataTypes.h
  Buffer    : Pointer;                         //!< [in] buffer location
  Stride    : TRDBInt32 = 0                    //!< [in] bytes between beginnings of successive elements (0: auto)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Remove points
 *
 * Use this function to actually remove (delete) the points.
 *
 * Afterwards you may re-fill the buffers or define a new buffer
 * with id() and call next() again until all points have been
 * removed.
 *
 * \returns the number of points removed
 *}
function rdb_pointcloud_query_remove_next(
  Context   : TRDBContextHandle;               //!< [in] library context
  Query     : TRDBPointcloudQueryRemoveHandle; //!< [in] query handle
  Count     : TRDBUInt32;                      //!< [in] size of source buffers in terms of points
  Processed : PRDBUInt32 = nil                 //!< [out] number of processed points (optional)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//---< CLASS TRDBPointcloudQueryRemove >----------------------------------------
{*!
 * \brief Point remove query
 *
 * This class is used to remove (delete) existing points
 *
 * \see riegl::rdb::Pointcloud::remove()
 *}
type
  TRDBPointcloudQueryRemove = class(System.TObject)
  public
    {*!
     * \brief Constructor
     * \note  You cannot create new TRDBPointcloudQueryRemove objects directly,
     *        use riegl::rdb::Pointcloud::remove() instead.
     *}
    constructor Create(Parent : System.TObject); reintroduce;
    destructor  Destroy; override;

    {*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a source buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * The buffer is expected to be n*s*d bytes large, where
     * __n__ is the number of points defined in next(),
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * \note This function just stores the buffer pointer - it does
     *       __NOT__ copy the data contained in the buffer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \note This function expects a buffer for the point ID attribute.
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     *}
    procedure Bind(
      const Attribute : System.String;          //!< [in] attribute name
      const DataType  : TRDBPointcloudDataType; //!< [in] buffer data type
      const Buffer    : Pointer;                //!< [in] buffer location
      const Stride    : TRDBInt32 = 0           //!< [in] bytes between beginnings of successive elements (0: auto)
    );

    // Helper for fundamental types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64; const Stride : TRDBInt32 = 0); overload;

    // Helper for pointer types
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64; const Stride : TRDBInt32 = 0); overload;

    // Helper for array types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8;    const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64;  const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64;   const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32 = 0); overload;

    {*!
     * \brief Remove points
     *
     * Use this function to actually remove (delete) the points.
     * All points given by their point ID stored in the previously
     * bound attribute buffer are removed from the database.
     *
     * Afterwards you may re-fill the buffers or define a new buffer
     * with bind() and call next() again until all points have been
     * removed.
     *
     * \note If __no__ point ID buffer is given and 'count' is $DEADFEED,
     *       then __all__ points are removed and next() always returns 1.
     *
     * \returns the number of points removed
     *}
    function Next(
      const Count : TRDBUInt32 //!< [in] size of source buffers in terms of points
    ) : TRDBUInt32;

  private
    FContext    : TRDBContext;
    FPointcloud : TRDBPointcloudHandle;
    FQuery      : TRDBPointcloudQueryRemoveHandle;
  end;

{******************************************************************************}
{***} IMPLEMENTATION {*********************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb.pointcloud;

//---< TRDBPointcloudQueryRemove::PUBLIC >--------------------------------------

constructor TRDBPointcloudQueryRemove.Create(Parent : System.TObject);
begin
  inherited Create;
  FQuery      := nil;
  FContext    := (Parent as TRDBPointcloud).Context;
  FPointcloud := (Parent as TRDBPointcloud).Pointcloud;
  FContext.Check(rdb_pointcloud_query_remove_new(FContext.Handle, FPointcloud, FQuery));
end; // Create()

destructor TRDBPointcloudQueryRemove.Destroy;
begin
  FContext.Check(rdb_pointcloud_query_remove_delete(FContext.Handle, FQuery));
  inherited;
end; // Destroy()

procedure TRDBPointcloudQueryRemove.Bind(
  const Attribute : System.String;
  const DataType  : TRDBPointcloudDataType;
  const Buffer    : Pointer;
  const Stride    : TRDBInt32
);
begin
  FContext.Check(rdb_pointcloud_query_remove_bind(
    FContext.Handle, FQuery, TRDBString(AnsiToUtf8(Attribute)), Ord(DataType), Buffer, Stride
  ));
end; // Bind()

function TRDBPointcloudQueryRemove.Next(const Count : TRDBUInt32) : TRDBUInt32;
begin
  FContext.Check(rdb_pointcloud_query_remove_next(
    FContext.Handle, FQuery, Count, @Result
  ));
end; // Next()

//______________________________________________________________________________
//
// Helper for fundamental types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer, Stride);
end;

//______________________________________________________________________________
//
// Helper for pointer types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer, Stride);
end;

//______________________________________________________________________________
//
// Helper for array types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

procedure TRDBPointcloudQueryRemove.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64; const Stride : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0], Stride);
end;

end.
