/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 * \version 2025-11-11/AW: Item "riegl.lag_angle_correction" added (#6068)
 *
 *******************************************************************************
 */

#ifndef RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
#define RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87

#include <vector>
#include <string>
#include <cstdlib>
#include <cstdint>

namespace riegl {
namespace rdb {
namespace pointcloud {

// Meta data item definition version
const std::string RDB_META_DATA_ITEMS_VERSION = "1.1.57";
const std::string RDB_META_DATA_ITEMS_DATE = "2025-11-11";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
struct MetaDataItemInfo
{
    std::string   name;
    std::string   title;
    std::string   description;
    unsigned char status;
    std::string   schema;
    std::string   example;

    operator const std::string&() const { return name; }
};

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"amplitude_mean\":{\"type\":\"array\",\"items\":{\"type\":\"numbe"
    "r\"},\"description\":\"Mean amplitude [dB]\"},\"angle\":{\"type\":\"array\",\"item"
    "s\":{\"type\":\"number\"},\"description\":\"Angle [deg]\"},\"range_mean\":{\"type\""
    ":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Mean range [m]\"}},\"t"
    "ype\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"req"
    "uired\":[\"angle\",\"range_mean\",\"amplitude_mean\"],\"title\":\"Angular Notch "
    "Filter\",\"description\":\"Angular notch filter parameters for window "
    "glass echoes\"}"
    ,
    /* example: */
    "{\"amplitude_mean\":[3.780913,3.780913,3.480913,3.120913,2.850913,2.7209"
    "13,2.680913,2.610913,2.530913,2.570913,2.570913],\"angle\":[14.0,15.0,16"
    ".0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],\"range_mean\":[0.094,0.094,"
    "0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"amsl\":{\"type\":\"number\",\"description\":\"Height above "
    "mean sea level (AMSL) "
    "[m]\"},\"group_velocity\":{\"type\":\"number\",\"description\":\"Group velocity "
    "of laser beam "
    "[m/s]\"},\"pressure\":{\"type\":\"number\",\"description\":\"Pressure along "
    "measurment path "
    "[mbar]\"},\"rel_humidity\":{\"type\":\"number\",\"description\":\"Relative "
    "humidity along measurement path "
    "[%]\"},\"temperature\":{\"type\":\"number\",\"description\":\"Temperature along "
    "measurement path "
    "[\\u00b0C]\"},\"pressure_sl\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\"},\"wavelength\":{\"type\":\"number\",\"description\":\"Laser wavelength"
    " [nm]\"},\"attenuation\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "attenuation [1/km]\"}},\"type\":\"object\",\"$schema\":\"http://json-schema.or"
    "g/draft-04/schema#\",\"required\":[\"temperature\",\"pressure\",\"rel_humidity"
    "\",\"pressure_sl\",\"amsl\",\"group_velocity\",\"attenuation\",\"wavelength\"],\"t"
    "itle\":\"Atmospheric Parameters\",\"description\":\"Atmospheric parameters\"}"
    ,
    /* example: */
    "{\"amsl\":0,\"group_velocity\":299711000.0,\"pressure\":970,\"rel_humidity\":6"
    "3,\"temperature\":7,\"pressure_sl\":970,\"wavelength\":1550,\"attenuation\":0."
    "028125}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"beam_divergence\":{\"type\":\"number\",\"exclusiveMinimum\":f"
    "alse,\"minimum\":0,\"description\":\"Beam divergence in far field [rad]\"},\""
    "beam_exit_diameter\":{\"type\":\"number\",\"exclusiveMinimum\":false,\"minimum"
    "\":0,\"description\":\"Beam width at exit aperture [m]\"}},\"type\":\"object\","
    "\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"beam_"
    "exit_diameter\",\"beam_divergence\"],\"title\":\"Beam "
    "Geometry\",\"description\":\"Laser beam geometry details\"}"
    ,
    /* example: */
    "{\"beam_divergence\":0.0003,\"beam_exit_diameter\":0.0072}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"types\":{\"type\":\"array\",\"uniqueItems\":true,\"items\":{\"ty"
    "pe\":\"object\",\"oneOf\":[{\"$ref\":\"#/definitions/rectangle\"},{\"$ref\":\"#/de"
    "finitions/checkerboard2x2\"},{\"$ref\":\"#/definitions/chevron\"},{\"$ref\":\""
    "#/definitions/circular_disk\"},{\"$ref\":\"#/definitions/cylinder\"},{\"$ref"
    "\":\"#/definitions/sphere\"},{\"$ref\":\"#/definitions/round_corner_cube_pri"
    "sm\"}]}}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/sc"
    "hema#\",\"required\":[\"types\"],\"title\":\"Control Object Catalog\",\"definiti"
    "ons\":{\"rectangle\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/com"
    "mon\"},{\"properties\":{\"width\":{\"type\":\"number\",\"exclusiveMinimum\":true,"
    "\"minimum\":0,\"description\":\"width in meters\"},\"length\":{\"type\":\"number\""
    ",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"length in meters\"}"
    ",\"shape\":{\"type\":\"string\",\"enum\":[\"rectangle\"]}},\"type\":\"object\",\"requ"
    "ired\":[\"shape\",\"length\",\"width\"],\"description\":\"rectangle specific pro"
    "perties\"}],\"description\":\"rectangle\"},\"cylinder\":{\"type\":\"object\",\"all"
    "Of\":[{\"$ref\":\"#/definitions/common\"},{\"properties\":{\"height\":{\"type\":\""
    "number\",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"height in m"
    "eters\"},\"diameter\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":"
    "0,\"description\":\"diameter in meters\"},\"shape\":{\"type\":\"string\",\"enum\":"
    "[\"cylinder\"],\"description\":\"shape identifier\"}},\"type\":\"object\",\"requi"
    "red\":[\"shape\",\"diameter\",\"height\"],\"description\":\"cylinder specific pr"
    "operties\"}],\"description\":\"cylinder\"},\"circular_disk\":{\"type\":\"object\""
    ",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"properties\":{\"offset\":{\"ty"
    "pe\":\"number\",\"description\":\"offset in meters, e.g. reflector constant "
    "(optional)\"},\"diameter\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"mini"
    "mum\":0,\"description\":\"diameter in meters\"},\"shape\":{\"type\":\"string\",\"e"
    "num\":[\"circular_disk\"],\"description\":\"shape identifier\"}},\"type\":\"obje"
    "ct\",\"required\":[\"shape\",\"diameter\"],\"description\":\"circular disk "
    "specific properties\"}],\"description\":\"circular disk\"},\"round_corner_cu"
    "be_prism\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\""
    "properties\":{\"offset\":{\"type\":\"number\",\"description\":\"offset in "
    "meters, e.g. reflector constant (optional)\"},\"diameter\":{\"type\":\"numbe"
    "r\",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"diameter in mete"
    "rs\"},\"shape\":{\"type\":\"string\",\"enum\":[\"round_corner_cube_prism\"],\"desc"
    "ription\":\"shape identifier\"}},\"type\":\"object\",\"required\":[\"shape\",\"dia"
    "meter\"],\"description\":\"round corner cube prism specific "
    "properties\"}],\"description\":\"round corner cube prism\"},\"checkerboard2x"
    "2\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"propert"
    "ies\":{\"square_length\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimu"
    "m\":0,\"description\":\"length of a square of the checkerboard in meters\"}"
    ",\"shape\":{\"type\":\"string\",\"enum\":[\"checkerboard2x2\"],\"description\":\"sh"
    "ape identifier\"}},\"type\":\"object\",\"required\":[\"shape\",\"square_length\"]"
    ",\"description\":\"checkerboard specific "
    "properties\"}],\"description\":\"checkerboard 2 by 2\"},\"common\":{\"properti"
    "es\":{\"surface_type\":{\"type\":\"string\",\"enum\":[\"retro_reflective_foil\",\""
    "diffuse\"],\"description\":\"surface material "
    "type\"},\"name\":{\"type\":\"string\",\"minLength\":3,\"description\":\"unique "
    "type identifier\"},\"shape\":{\"type\":\"string\",\"enum\":[\"rectangle\",\"checke"
    "rboard2x2\",\"chevron\",\"circular_disk\",\"cylinder\",\"sphere\",\"round_corner"
    "_cube_prism\"],\"description\":\"shape "
    "identifier\"},\"description\":{\"type\":\"string\",\"description\":\"string "
    "describing the object\"}},\"type\":\"object\",\"required\":[\"name\",\"shape\"],\""
    "description\":\"common object properties\"},\"chevron\":{\"type\":\"object\",\"a"
    "llOf\":[{\"$ref\":\"#/definitions/common\"},{\"properties\":{\"outside_edge_le"
    "ngth\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,\"descriptio"
    "n\":\"length of the two outer edges in meters\"},\"thickness\":{\"type\":\"num"
    "ber\",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"thickness in m"
    "eters\"},\"shape\":{\"type\":\"string\",\"enum\":[\"chevron\"],\"description\":\"sha"
    "pe identifier\"}},\"type\":\"object\",\"required\":[\"shape\",\"outside_edge_len"
    "gth\",\"thickness\"],\"description\":\"chevron specific properties\"}],\"descr"
    "iption\":\"chevron\"},\"sphere\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/defin"
    "itions/common\"},{\"properties\":{\"diameter\":{\"type\":\"number\",\"exclusiveM"
    "inimum\":true,\"minimum\":0,\"description\":\"diameter in meters\"},\"shape\":{"
    "\"type\":\"string\",\"enum\":[\"sphere\"],\"description\":\"shape identifier\"}},\""
    "type\":\"object\",\"required\":[\"shape\",\"diameter\"],\"description\":\"sphere "
    "specific properties\"}],\"description\":\"sphere\"}},\"description\":\"List of"
    " control object type definitions\"}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or "
    "'diffuse')\"],\"types\":[{\"width\":0.3,\"name\":\"Rectangle "
    "60x30\",\"length\":0.6,\"shape\":\"rectangle\",\"description\":\"Rectangle (60cm"
    " x 30cm)\"},{\"width\":0.4,\"name\":\"Rectangle "
    "80x40\",\"length\":0.8,\"shape\":\"rectangle\",\"description\":\"Rectangle (80cm"
    " x 40cm)\"},{\"square_length\":0.3,\"name\":\"Checkerboard2x2 "
    "30\",\"shape\":\"checkerboard2x2\",\"description\":\"Checkerboard (square "
    "length: 30cm)\"},{\"square_length\":0.5,\"name\":\"Checkerboard2x2 "
    "50\",\"shape\":\"checkerboard2x2\",\"description\":\"Checkerboard (square "
    "length: 50cm)\"},{\"outside_edge_length\":0.6096,\"thickness\":0.1016,\"name"
    "\":\"Chevron 24''/4''\",\"shape\":\"chevron\",\"description\":\"Chevron (a=24'';"
    " b=4'')\"},{\"surface_type\":\"diffuse\",\"name\":\"Circular Disk "
    "50\",\"diameter\":0.5,\"shape\":\"circular_disk\",\"description\":\" Circular "
    "Disk (diameter: 50cm)\"},{\"name\":\"RIEGL flat reflector 50 mm\",\"diameter"
    "\":0.05,\"offset\":0.0,\"surface_type\":\"retro_reflective_foil\",\"shape\":\"ci"
    "rcular_disk\",\"description\":\"flat circular reflector from "
    "retro-reflective foil\"},{\"name\":\"RIEGL flat reflector 100 mm\",\"diamete"
    "r\":0.1,\"offset\":0.0,\"surface_type\":\"retro_reflective_foil\",\"shape\":\"ci"
    "rcular_disk\",\"description\":\"flat circular reflector from "
    "retro-reflective foil\"},{\"name\":\"RIEGL flat reflector 150 mm\",\"diamete"
    "r\":0.15,\"offset\":0.0,\"surface_type\":\"retro_reflective_foil\",\"shape\":\"c"
    "ircular_disk\",\"description\":\"flat circular reflector from "
    "retro-reflective foil\"},{\"name\":\"RIEGL cylindrical reflector 50 mm\",\"d"
    "iameter\":0.05,\"surface_type\":\"retro_reflective_foil\",\"height\":0.05,\"sh"
    "ape\":\"cylinder\",\"description\":\"cylindrical reflector from "
    "retro-reflective foil\"},{\"name\":\"RIEGL cylindrical reflector 100 mm\",\""
    "diameter\":0.1,\"surface_type\":\"retro_reflective_foil\",\"height\":0.1,\"sha"
    "pe\":\"cylinder\",\"description\":\"cylindrical reflector from "
    "retro-reflective foil\"},{\"name\":\"Sphere 200 "
    "mm\",\"diameter\":0.2,\"shape\":\"sphere\",\"description\":\"Sphere (diameter: "
    "200 mm)\"},{\"offset\":0.0,\"name\":\"Corner Cube Prism 50 mm\",\"diameter\":0."
    "05,\"shape\":\"round_corner_cube_prism\",\"description\":\"round corner cube "
    "prism\"}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"reference_file\":{\"properties\":{\"file_uuid\":{\"type\":\"st"
    "ring\",\"description\":\"Control object file's Universally Unique "
    "Identifier (RFC "
    "4122)\"},\"file_path\":{\"type\":\"string\",\"description\":\"Path of the "
    "control object file relative to referring file\"}},\"type\":\"object\",\"req"
    "uired\":[\"file_uuid\",\"file_path\"],\"description\":\"Reference to a control"
    " object file\"}},\"description\":\"Details about the control object "
    "reference file\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"t"
    "ype\":\"object\",\"title\":\"Control Object Reference file\"}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\","
    "\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoints.cpx\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"detection_probability\":{\"type\":\"array\",\"items\":{\"type\""
    ":\"number\"},\"description\":\"Detection probability [0..1]\"},\"amplitude\":{"
    "\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Amplitude [dB]"
    "\"}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#"
    "\",\"required\":[\"amplitude\",\"detection_probability\"],\"title\":\"Detection "
    "Probability\",\"description\":\"Detection probability as a function of "
    "amplitude\"}"
    ,
    /* example: */
    "{\"detection_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]"
    ",\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"channel_number\":{\"type\":\"integer\",\"exclusiveMinimum\":f"
    "alse,\"minimum\":0,\"description\":\"Laser channel number (not defined or "
    "0: single channel "
    "device)\"},\"channel_text\":{\"type\":\"string\",\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\"},\"device_build\":{\"type\":\"string\",\"description\":\"Device "
    "build variant\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional"
    " device name (e.g. 'Scanner 1' for multi-scanner systems)\"}},\"type\":\"o"
    "bject\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":"
    "[\"device_type\",\"serial_number\"],\"title\":\"Device "
    "Information\",\"description\":\"Details about the device used to acquire "
    "the point cloud\"}"
    ,
    /* example: */
    "{\"channel_number\":0,\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"serial_"
    "number\":\"S2221234\",\"device_build\":\"\",\"device_name\":\"Scanner 1\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"secondary\":{\"properties\":{\"ID\":{\"type\":\"array\",\"items\""
    ":{\"type\":\"integer\"},\"minItems\":2,\"maxItems\":2,\"description\":\"Structure"
    " identifier\"},\"content\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"des"
    "cription\":\"Internal calibration values\"}},\"type\":\"object\",\"required\":["
    "\"ID\",\"content\"],\"description\":\"Additional device geometry structure (o"
    "ptional)\"},\"primary\":{\"properties\":{\"ID\":{\"type\":\"array\",\"items\":{\"typ"
    "e\":\"integer\"},\"minItems\":2,\"maxItems\":2,\"description\":\"Structure ident"
    "ifier\"},\"content\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"descripti"
    "on\":\"Internal calibration values\"}},\"type\":\"object\",\"required\":[\"ID\",\""
    "content\"],\"description\":\"Primary device geometry structure (mandatory)"
    "\"},\"amu\":{\"properties\":{\"frameCC\":{\"type\":\"number\",\"exclusiveMinimum\":"
    "false,\"minimum\":0,\"description\":\"Frame Circle Count (number of LSBs "
    "per full rotation about frame axis)\"},\"lineCC\":{\"type\":\"number\",\"exclu"
    "siveMinimum\":false,\"minimum\":0,\"description\":\"Line Circle Count "
    "(number of LSBs per full rotation about line "
    "axis)\"}},\"type\":\"object\",\"description\":\"Angle Measurement Unit\"}},\"typ"
    "e\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"requi"
    "red\":[\"primary\"],\"title\":\"Device Geometry\",\"description\":\"Scanner "
    "device geometry details\"}"
    ,
    /* example: */
    "{\"secondary\":{\"ID\":[91,0],\"content\":[0]},\"primary\":{\"ID\":[4,0],\"conten"
    "t\":[0]},\"amu\":{\"frameCC\":124000,\"lineCC\":124000}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"primary\":{\"properties\":{\"ID\":{\"type\":\"array\",\"items\":{"
    "\"type\":\"integer\"},\"minItems\":2,\"maxItems\":2,\"description\":\"Structure i"
    "dentifier\"},\"content\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"descr"
    "iption\":\"Internal calibration values\"}},\"type\":\"object\",\"required\":[\"I"
    "D\",\"content\"],\"description\":\"Primary device geometry structure (mandat"
    "ory)\"}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/sch"
    "ema#\",\"required\":[\"primary\"],\"title\":\"Device Geometry Passive "
    "Channel\",\"description\":\"Scanner passive channel device geometry "
    "details\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[143,0],\"content\":[0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"deviation_minimum\":{\"type\":\"number\",\"description\":\"Min"
    "imum possible pulse shape "
    "deviation.\"},\"range_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible range in "
    "meters.\"},\"echo_count_maximum\":{\"type\":\"number\",\"description\":\"Maximum"
    " number of echoes a laser shot can "
    "have.\"},\"amplitude_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible amplitude in "
    "dB.\"},\"reflectance_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible reflectance in "
    "dB.\"},\"reflectance_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible reflectance in "
    "dB.\"},\"mta_zone_count_maximum\":{\"type\":\"number\",\"description\":\"Maximum"
    " number of MTA zones.\"},\"background_radiation_maximum\":{\"type\":\"number"
    "\",\"description\":\"Maximum possible background radiation.\"},\"background_"
    "radiation_minimum\":{\"type\":\"number\",\"description\":\"Minimum possible "
    "background "
    "radiation.\"},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible range in "
    "meters.\"},\"amplitude_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible amplitude in "
    "dB.\"},\"deviation_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible pulse shape deviation.\"}},\"description\":\"Limits of the "
    "measured values output by the device. The limits depend on the device "
    "type, measurement program and/or scan pattern.\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"type\":\"object\",\"title\":\"Device Output "
    "Limits\"}"
    ,
    /* example: */
    "{\"deviation_minimum\":-1,\"background_radiation_maximum\":0,\"range_maximu"
    "m\":10000.0,\"amplitude_maximum\":100.0,\"reflectance_maximum\":100.0,\"back"
    "ground_radiation_minimum\":0,\"amplitude_minimum\":0.0,\"range_minimum\":2."
    "9,\"reflectance_minimum\":-100.0,\"mta_zone_count_maximum\":7,\"deviation_m"
    "aximum\":32767}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Details about the devices used to acquire the point "
    "cloud (e.g. when merging point clouds of different devices)\",\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\",\"items\":{\"properties\":{\"cha"
    "nnel_number\":{\"type\":\"integer\",\"exclusiveMinimum\":false,\"minimum\":0,\"d"
    "escription\":\"Laser channel number (not defined or 0: single channel "
    "device)\"},\"channel_text\":{\"type\":\"string\",\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\"},\"device_build\":{\"type\":\"string\",\"description\":\"Device "
    "build variant\"},\"signed\":{\"type\":\"boolean\",\"description\":\"Flag that is"
    " set when the original 'riegl.device' entry in the source file was "
    "correctly "
    "signed.\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner systems)\"}},\"type\":\"ob"
    "ject\",\"required\":[\"device_type\",\"serial_number\"]},\"type\":\"array\",\"titl"
    "e\":\"Devices Information\"}"
    ,
    /* example: */
    "[{\"channel_number\":0,\"channel_text\":\"\",\"device_type\":\"VZ-400i\",\"serial"
    "_number\":\"S2221234\",\"device_build\":\"\",\"signed\":false,\"device_name\":\"Sc"
    "anner 1\"},{\"channel_number\":1,\"channel_text\":\"\",\"device_type\":\"VQ-1560"
    "i-DW\",\"serial_number\":\"S2222680\",\"device_build\":\"\",\"signed\":true,\"devi"
    "ce_name\":\"Scanner 2\"},{\"channel_number\":2,\"channel_text\":\"\",\"device_ty"
    "pe\":\"VQ-1560i-DW\",\"serial_number\":\"S2222680\",\"device_build\":\"\",\"signed"
    "\":true,\"device_name\":\"Scanner 3\"}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"dmi_dist_per_tick\":{\"type\":\"number\",\"description\":\"Dis"
    "tance per tick, just informative [m]\"},\"dmi_count_dir\":{\"exclusiveMaxi"
    "mum\":false,\"type\":\"integer\",\"exclusiveMinimum\":false,\"maximum\":1,\"mini"
    "mum\":0,\"description\":\"DMI count direction is inverted\"},\"dmi_max_time\""
    ":{\"exclusiveMaximum\":false,\"type\":\"integer\",\"exclusiveMinimum\":false,\""
    "maximum\":65535,\"minimum\":0,\"description\":\"Maximum time interval "
    "between trigger events, 0 means timeout is inactive [0.1 msec]\"},\"dmi_"
    "incr\":{\"exclusiveMaximum\":false,\"type\":\"integer\",\"exclusiveMinimum\":fa"
    "lse,\"maximum\":2147483647,\"minimum\":0,\"description\":\"Increment in "
    "ticks, 0 means distance trigger is deactivated\"},\"dmi_min_time\":{\"excl"
    "usiveMaximum\":false,\"type\":\"integer\",\"exclusiveMinimum\":false,\"maximum"
    "\":65535,\"minimum\":0,\"description\":\"Minimum time interval between "
    "trigger events, 0 means rate limitation is inactive [0.1 msec]\"}},\"typ"
    "e\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"requi"
    "red\":[\"dmi_count_dir\",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_di"
    "st_per_tick\"],\"title\":\"DMI Configuration\",\"description\":\"Distance "
    "measuring instrument configuration\"}"
    ,
    /* example: */
    "{\"dmi_dist_per_tick\":0.00024399999529123306,\"dmi_count_dir\":0,\"dmi_max"
    "_time\":100,\"dmi_incr\":0,\"dmi_min_time\":0}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"pre\":{\"properties\":{\"a\":{\"type\":\"array\",\"minItems\":1,\""
    "description\":\"...\",\"maxItems\":2080,\"items\":{\"type\":\"number\"}},\"nc\":{\"t"
    "ype\":\"integer\",\"minimum\":0,\"maximum\":255,\"description\":\"Number of entr"
    "ies\"}},\"type\":\"object\",\"required\":[\"nc\",\"a\"],\"description\":\"Dyntrig "
    "pre values\"},\"scale\":{\"type\":\"number\",\"description\":\"Scale in units of"
    " dB, for calculation of line: line number N=(amplitude-offset)/scale\"}"
    ",\"tab\":{\"type\":\"array\",\"minItems\":1,\"description\":\"...\",\"maxItems\":128"
    ",\"items\":{\"properties\":{\"del1\":{\"type\":\"number\",\"description\":\"Length "
    "of constant post-trigger threshold elevation in units of "
    "m\"},\"thra2\":{\"type\":\"number\",\"description\":\"Pre-trigger threshold in "
    "units of dB\"},\"dec1\":{\"type\":\"number\",\"description\":\"Decay of "
    "post-trigger threshold after del1 in units of "
    "dB/m\"},\"del2\":{\"type\":\"number\",\"description\":\"Length of constant "
    "pre-trigger threshold elevation in units of "
    "m\"},\"dec2\":{\"type\":\"number\",\"description\":\"Rise of pre-trigger "
    "threshold after del2 in units of "
    "dB/m\"},\"thra1\":{\"type\":\"number\",\"description\":\"Post-trigger threshold "
    "in units of dB\"}},\"type\":\"object\",\"required\":[\"thra1\",\"del1\",\"dec1\",\"t"
    "hra2\",\"del2\",\"dec2\"],\"description\":\"...\"}},\"post\":{\"properties\":{\"a\":{"
    "\"type\":\"array\",\"minItems\":1,\"description\":\"...\",\"maxItems\":2080,\"items"
    "\":{\"type\":\"number\"}},\"nc\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":255,"
    "\"description\":\"Number of entries\"}},\"type\":\"object\",\"required\":[\"nc\",\""
    "a\"],\"description\":\"Dyntrig post "
    "values\"},\"offset\":{\"type\":\"number\",\"description\":\"Offset in units of "
    "dB, for calculation of line\"}},\"description\":\"Dyntrig\",\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"title\":\"Dyntrig\"}"
    ,
    /* example: */
    "{\"pre\":{\"a\":[78,86,126,134,31],\"nc\":128},\"scale\":1.0,\"tab\":[{\"del1\":0."
    "0,\"thra2\":0.0,\"dec1\":0.0,\"del2\":0.0,\"dec2\":0.0,\"thra1\":0.0}],\"post\":{\""
    "a\":[78,86,126,134,31],\"nc\":128},\"offset\":0.0}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"echo_file\":{\"properties\":{\"file_uuid\":{\"type\":\"string\""
    ",\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Echo file "
    "extension, without the leading dot\"}},\"type\":\"object\",\"required\":[\"fil"
    "e_extension\"]}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/draf"
    "t-04/schema#\",\"required\":[\"echo_file\"],\"title\":\"Echo "
    "Information\",\"description\":\"Details about echo files\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_uuid\":\"26a03615-67c0-4bea-8fe8-c577378fe661\",\"file"
    "_extension\":\"owp\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/definitions/channel\",\"descri"
    "ption\":\"one field per channel, field name is channel "
    "index\"}},\"additionalProperties\":false,\"title\":\"Exponential Decompositi"
    "on\",\"definitions\":{\"channel\":{\"properties\":{\"a_lin\":{\"exclusiveMaximum"
    "\":false,\"type\":\"number\",\"exclusiveMinimum\":false,\"maximum\":1,\"minimum\""
    ":0,\"description\":\"relative linear amplitude range "
    "[0..1]\"},\"scale\":{\"type\":\"number\",\"description\":\"amplitude calibration"
    "\"},\"parameter\":{\"properties\":{\"B\":{\"type\":\"array\",\"items\":{\"type\":\"num"
    "ber\"},\"description\":\"imaginary part of amplitude factor in units of fu"
    "ll-scale\"},\"omega\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"descript"
    "ion\":\"angular frequency in Hz\"},\"gamma\":{\"type\":\"array\",\"items\":{\"type"
    "\":\"number\"},\"description\":\"decay in 1/second\"},\"A\":{\"type\":\"array\",\"it"
    "ems\":{\"type\":\"number\"},\"description\":\"real part of amplitude factor in"
    " units of full-scale\"}},\"type\":\"object\",\"required\":[\"A\",\"B\",\"gamma\",\"o"
    "mega\"],\"description\":\"parameters of the syswave exponential "
    "sum\"},\"delay\":{\"type\":\"number\",\"description\":\"delay calibration in sec"
    "onds\"}},\"type\":\"object\",\"required\":[\"delay\",\"scale\",\"parameter\"]}},\"de"
    "scription\":\"Details for exponential decomposition of full waveform "
    "data\"}"
    ,
    /* example: */
    "{\"1\":{\"a_lin\":0.9,\"scale\":1.0,\"parameter\":{\"B\":[-3.9,0.0,-0.3],\"omega\""
    ":[352020896.0,3647927552.0,-1977987072.0],\"gamma\":[-1094726528.0,-7695"
    "62752.0,-848000064.0],\"A\":[0.9,0.3,-1.3]},\"delay\":3.5e-09},\"0\":{\"a_lin"
    "\":0.27,\"scale\":1.0,\"parameter\":{\"B\":[-3.9813032150268555,0.08622030913"
    "829803,-0.3152860999107361],\"omega\":[352020896.0,3647927552.0,-1977987"
    "072.0],\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\"A\":[0.977245"
    "0923919678,0.3354335129261017,-1.312678575515747]},\"delay\":3.783458418"
    "887631e-09}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_offset_sec_low_power\":{\"type\":\"array\",\"items\":{\""
    "type\":\"number\"}},\"amplitude_lsb_high_power\":{\"type\":\"array\",\"items\":{\""
    "type\":\"number\"}},\"range_offset_sec_high_power\":{\"type\":\"array\",\"items\""
    ":{\"type\":\"number\"}},\"amplitude_db\":{\"type\":\"array\",\"items\":{\"type\":\"nu"
    "mber\"}},\"amplitude_lsb_low_power\":{\"type\":\"array\",\"items\":{\"type\":\"num"
    "ber\"}}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/sch"
    "ema#\",\"required\":[\"amplitude_lsb_low_power\",\"amplitude_lsb_high_power\""
    ",\"amplitude_db\",\"range_offset_sec_low_power\",\"range_offset_sec_high_po"
    "wer\"],\"title\":\"Gaussian "
    "Decomposition\",\"description\":\"riegl.gaussian_decomposition contains "
    "information relevant for extracting calibrated amplitudes and "
    "calibrated ranges from a Gaussian decomposition of full waveform data."
    " This information is contained in a table with five columns. Two "
    "columns are to be used as input: amplitude_lsb_low_power and "
    "amplitude_lsb_high_power. The other three columns provide the outputs."
    " Amplitude_db gives the calibrated amplitude in the optical regime in "
    "decibels. The range offset columns provide additive range offsets, "
    "given in units of seconds, for each channel.\"}"
    ,
    /* example: */
    "{\"range_offset_sec_low_power\":[],\"amplitude_lsb_high_power\":[],\"range_"
    "offset_sec_high_power\":[],\"amplitude_db\":[],\"amplitude_lsb_low_power\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"pose\":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":"
    "{\"type\":\"number\",\"description\":\"columns\"},\"minItems\":4,\"maxItems\":4,\"d"
    "escription\":\"rows\"},\"minItems\":4,\"maxItems\":4,\"description\":\"Coordinat"
    "e Transformation Matrix to transform from File Coordinate System to "
    "Global Coordinate Reference System. 4x4 matrix stored as two "
    "dimensional array, row major order.\"},\"crs\":{\"properties\":{\"epsg\":{\"ty"
    "pe\":\"integer\",\"minimum\":0,\"description\":\"EPSG "
    "code\"},\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\"},\"name\":{\"type\":\"string\",\"description\":\"Coordinate reference "
    "system name\"}},\"type\":\"object\",\"description\":\"Global Coordinate "
    "Reference System. Please note that only 3D Cartesian Coordinate "
    "Systems are allowed.\"}},\"description\":\"Point cloud georeferencing info"
    "rmation\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"type\":\"o"
    "bject\",\"title\":\"Geo Tag\"}"
    ,
    /* example: */
    "{\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,4063"
    "882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847,113"
    "8787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196],[0"
    ".0,0.0,0.0,1.0]],\"crs\":{\"epsg\":4978,\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\""
    ",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORIT"
    "Y[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich"
    "\\\",0.0000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.0"
    "0000000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AX"
    "IS[\\\"Y\\\",EAST],AXIS[\\\"Z\\\",NORTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\",\"name\""
    ":\"WGS84 Geocentric\"}}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"number\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Geometric scale "
    "factor applied to point coordinates\"}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"socs_to_rocs_matrix\":{\"type\":\"array\",\"items\":{\"type\":\""
    "array\",\"items\":{\"type\":\"number\",\"description\":\"columns\"},\"minItems\":4,"
    "\"maxItems\":4,\"description\":\"rows\"},\"minItems\":4,\"maxItems\":4,\"descript"
    "ion\":\"Coordinate Transformation Matrix to transform from Scanner's Own"
    " Coordinate System to Record Coordinate System. 4x4 matrix stored as "
    "two dimensional array, row major order.\"},\"socs_to_body_matrix\":{\"type"
    "\":\"array\",\"items\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"descriptio"
    "n\":\"columns\"},\"minItems\":4,\"maxItems\":4,\"description\":\"rows\"},\"minItem"
    "s\":4,\"maxItems\":4,\"description\":\"Coordinate Transformation Matrix to "
    "transform from Scanner's Own Coordinate System to Body Coordinate "
    "System. 4x4 matrix stored as two dimensional array, row major order.\"}"
    ",\"body_coordinate_system_type\":{\"type\":\"string\",\"enum\":[\"NED\",\"ENU\"],\""
    "description\":\"BODY coordinate frame (NED: North-East-Down, ENU: "
    "East-North-Up), default: NED\"},\"trajectory_file\":{\"properties\":{\"file_"
    "uuid\":{\"type\":\"string\",\"description\":\"File's Universally Unique "
    "Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Trajectory "
    "file extension, without the leading dot\"}},\"type\":\"object\",\"required\":"
    "[\"file_extension\"],\"description\":\"Trajectory data used for "
    "georeferencing of the point cloud\"},\"trajectory_offsets\":{\"properties\""
    ":{\"offset_north\":{\"type\":\"number\",\"description\":\"[m]\"},\"version\":{\"typ"
    "e\":\"integer\",\"description\":\"Meaning of offset values and how to apply "
    "them; version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\"},\"offset_height\":{\"type\":\"number\",\"description\":\"[m"
    "]\"},\"offset_east\":{\"type\":\"number\",\"description\":\"[m]\"},\"offset_roll\":"
    "{\"type\":\"number\",\"description\":\"[deg]\"},\"offset_pitch\":{\"type\":\"number"
    "\",\"description\":\"[deg]\"},\"offset_time\":{\"type\":\"number\",\"description\":"
    "\"[s]\"},\"offset_yaw\":{\"type\":\"number\",\"description\":\"[deg]\"}},\"type\":\"o"
    "bject\",\"description\":\"Correction offsets applied to the trajectory "
    "data\"}},\"description\":\"Parameters used for georeferencing of the point"
    " cloud\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"type\":\"ob"
    "ject\",\"title\":\"Georeferencing Parameters\"}"
    ,
    /* example: */
    "{\"body_coordinate_system_type\":\"NED\",\"socs_to_body_matrix\":[[-0.269827"
    "776749716,-0.723017716139738,0.635954678449952,0.0],[0.962908599449764"
    ",-0.20260517250352,0.178208229833847,0.0],[0.0,0.660451759194338,0.750"
    "8684796801,0.0],[0.0,0.0,0.0,1.0]],\"trajectory_file\":{\"file_uuid\":\"93a"
    "03615-66c0-4bea-8ff8-c577378fe660\",\"file_extension\":\"pofx\"},\"trajector"
    "y_offsets\":{\"offset_north\":0.07,\"version\":0,\"offset_height\":-0.2,\"offs"
    "et_east\":0.15,\"offset_roll\":0.03,\"offset_pitch\":0.01,\"offset_time\":18."
    "007,\"offset_yaw\":-0.45}}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"base_stations\":{\"type\":\"array\",\"items\":{\"properties\":{"
    "\"latitude\":{\"type\":\"number\",\"description\":\"Latitude "
    "[deg]\"},\"longitude\":{\"type\":\"number\",\"description\":\"Longitude "
    "[deg]\"},\"altitude\":{\"type\":\"number\",\"description\":\"Altitude [m]\"}},\"ty"
    "pe\":\"object\",\"required\":[\"latitude\",\"longitude\",\"altitude\"],\"descripti"
    "on\":\"GNSS base station details\"},\"description\":\"List of GNSS base "
    "stations\"}},\"description\":\"Details about GNSS processing\",\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\",\"type\":\"object\",\"title\":\"GNSS "
    "Processing Information\"}"
    ,
    /* example: */
    "{\"base_stations\":[{\"latitude\":48.658391162,\"longitude\":15.663389822,\"a"
    "ltitude\":379.8004150390625}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "patternProperties\":{\"^-?[0-9]+$\":{\"type\":\"string\",\"description\":\"One "
    "field per item, field name is item id, field value is item "
    "name\"}},\"additionalProperties\":false,\"title\":\"Item "
    "Names\",\"description\":\"Map of item names\"}"
    ,
    /* example: */
    "{\"1\":\"Name of item with id 1\",\"-1\":\"Name of item with id -1\",\"2\":\"Name"
    " of item with id 2\",\"47\":\"Name of item with id 47\"}"
};

// Lag angle depended range and reflectance correction
const MetaDataItemInfo RDB_RIEGL_LAG_ANGLE_CORRECTION = {
    /* name        */ "riegl.lag_angle_correction",
    /* title       */ "Lag Angle Correction",
    /* description */ "Lag angle depended range and reflectance correction",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"ranges_m\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"d"
    "escription\":\"Range [m]\"},\"range_correction_m\":{\"type\":\"array\",\"items\":"
    "{\"type\":\"number\"},\"description\":\"Range correction [m] as a function of"
    " the range and for applied scanrate, which defines the lag angle\"},\"re"
    "flectance_correction_db\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"de"
    "scription\":\"Reflectance correction [db] as a function of the range and"
    " for applied scanrate\"}},\"type\":\"object\",\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\",\"required\":[\"ranges_m\",\"range_correction_m\",\"re"
    "flectance_correction_db\"],\"title\":\"Lag Angle "
    "Correction\",\"description\":\"Lag angle depended range and reflectance "
    "correction\"}"
    ,
    /* example: */
    "{\"ranges_m\":[0.0,4.795,31.964,63.928,95.892,127.856,10000.0],\"range_co"
    "rrection_m\":[0.0,0.0,0.0004,0.0023,0.0039,0.0053,0.0053],\"reflectance_"
    "correction_db\":[0.0,0.0,0.16,0.46,1.18,2.97,2.97]}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "patternProperties\":{\"^.*$\":{\"type\":\"array\",\"items\":{\"type\":\"string\",\"d"
    "escription\":\"License key (example: '46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73')\"},\"minItems\":1,\"description\":\"Each field of the object "
    "represents a feature and holds a list of license keys, where the field"
    " name is the feature "
    "name.\"}},\"additionalProperties\":false,\"title\":\"Software License "
    "Keys\",\"description\":\"License keys for software features\"}"
    ,
    /* example: */
    "{\"Full Waveform Analysis Topography with GPU support\":[\"8AB44126 - "
    "23B92250 - 16E2689F - 34EF7E7B\"],\"MTA resolution\":[\"468E020A - "
    "39A922E4 - B681A184 - 673E3D72\"],\"Georeferencing\":[\"46AE032A - "
    "39882AC4 - 9EC0A184 - 6F163D73\"],\"Full Waveform Analysis "
    "Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA - 8D2ED557\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"zone_width\":{\"type\":\"number\",\"minimum\":0,\"description\""
    ":\"Width of a MTA zone in meter.\"},\"modulation_depth\":{\"type\":\"number\","
    "\"minimum\":0,\"description\":\"Depth of pulse position modulation in meter"
    ".\"},\"zone_count\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":255,\"descript"
    "ion\":\"Maximum number of MTA zones.\"}},\"description\":\"Parameters for "
    "MTA processing\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"t"
    "ype\":\"object\",\"title\":\"MTA Settings\"}"
    ,
    /* example: */
    "{\"zone_width\":149.896225,\"modulation_depth\":9.368514,\"zone_count\":23}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"content\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"mi"
    "nItems\":1,\"maxItems\":2000,\"description\":\"Correction value [m] to be "
    "added to the raw range\"},\"delta\":{\"type\":\"number\",\"description\":\"Delta"
    " between table entries [m], first entry is at range = 0 m\"}},\"type\":\"o"
    "bject\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":"
    "[\"delta\",\"content\"],\"title\":\"Near Range Correction "
    "Table\",\"description\":\"Lookup table for range correction based on raw "
    "range\"}"
    ,
    /* example: */
    "{\"content\":[0.0],\"delta\":0.512}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_sigma\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}"
    ",\"description\":\"Sigma range [m]\"},\"amplitude_sigma\":{\"type\":\"array\",\"i"
    "tems\":{\"type\":\"number\"},\"description\":\"Sigma amplitude [dB]\"},\"amplitu"
    "de\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Amplitude"
    " [dB]\"}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/sc"
    "hema#\",\"required\":[\"amplitude\",\"range_sigma\",\"amplitude_sigma\"],\"title"
    "\":\"Noise Estimates\",\"description\":\"Standard deviation for range and "
    "amplitude as a function of amplitude\"}"
    ,
    /* example: */
    "{\"range_sigma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,"
    "0.016,0.014],\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0."
    "54,0.482,0.432,0.39,0.354],\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7."
    "0,8.0,9.0,10.0]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum"
    " range [m]\"},\"range_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "range [m]\"},\"amplitude_maximum\":{\"type\":\"number\",\"minimum\":0,\"descript"
    "ion\":\"Maximum amplitude [dB]\"}},\"type\":\"object\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"required\":[\"range_minimum\",\"range_maxim"
    "um\",\"amplitude_maximum\"],\"title\":\"Notch Filter\",\"description\":\"Notch "
    "filter parameters for window glass echoes\"}"
    ,
    /* example: */
    "{\"range_minimum\":-0.5,\"range_maximum\":0.2,\"amplitude_maximum\":18.0}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"size\":{\"$ref\":\"#/definitions/pixel_size\",\"description\""
    ":\"Size of pixels in file coordinate system.\"},\"size_llcs\":{\"$ref\":\"#/d"
    "efinitions/pixel_size\",\"description\":\"Size of pixels in a locally "
    "levelled cartesian coordinate system (xy). This is only used for "
    "pixels based on a map projection.\"}},\"type\":\"object\",\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\",\"required\":[\"size\"],\"title\":\"Pixel "
    "Information\",\"definitions\":{\"pixel_size\":{\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\",\"minimum\":0,\"description\":\"Length of pixel edge "
    "[m].\"},\"minItems\":2,\"maxItems\":2,\"description\":\"Size of "
    "pixels.\"}},\"description\":\"Details about the pixels contained in the "
    "file\"}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395],\"size_llcs\":[0.5156575"
    "252891171,0.5130835356683303]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"plane_patch_file_one\":{\"$ref\":\"#/definitions/file_refe"
    "rence\",\"description\":\"Reference to the plane patch file one\"},\"plane_p"
    "atch_file_two\":{\"$ref\":\"#/definitions/file_reference\",\"description\":\"R"
    "eference to the plane patch file two\"}},\"type\":\"object\",\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\",\"required\":[\"plane_patch_file_on"
    "e\",\"plane_patch_file_two\"],\"title\":\"Plane Patch Matching\",\"definitions"
    "\":{\"file_reference\":{\"properties\":{\"file_uuid\":{\"type\":\"string\",\"descr"
    "iption\":\"Plane patch file's Universally Unique Identifier (RFC "
    "4122)\"},\"file_path\":{\"type\":\"string\",\"description\":\"Path of the plane "
    "patch file relative to the match file\"}},\"type\":\"object\",\"required\":[\""
    "file_uuid\",\"file_path\"],\"description\":\"Reference to a plane patch "
    "file\"}},\"description\":\"Details about the plane patch matching "
    "process\"}"
    ,
    /* example: */
    "{\"plane_patch_file_one\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b"
    "4213\",\"file_path\":\"Record009_Line001/191025_121410_Scanner_1.ptch\"},\"p"
    "lane_patch_file_two\":{\"file_uuid\":\"fa47d509-a64e-49ce-8b14-ff3130fbefa"
    "9\",\"file_path\":\"project.ptch\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"total_horizontal_area\":{\"type\":\"number\",\"description\":"
    "\"sum of all plane patch areas projected to horizontal plane "
    "[m\\u00b2]\"},\"total_area\":{\"type\":\"number\",\"description\":\"sum of all "
    "plane patch areas [m\\u00b2]\"}},\"description\":\"Statistics about plane "
    "patches found by plane patch extractor\",\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\",\"type\":\"object\",\"title\":\"Plane Patch "
    "Statistics\"}"
    ,
    /* example: */
    "{\"total_horizontal_area\":13954.601,\"total_area\":14007.965}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"classes\":{\"additionalProperties\":false,\"type\":\"object\""
    ",\"patternProperties\":{\"^[0-9]+$\":{\"type\":\"string\",\"description\":\"one "
    "field per class, field name is class number, field value is class "
    "name\"}},\"description\":\"Class definition table\"},\"settings\":{\"type\":\"ob"
    "ject\",\"oneOf\":[{\"$ref\":\"#/definitions/method-1\"},{\"$ref\":\"#/definition"
    "s/method-2\"}],\"description\":\"Classification settings, details see "
    "documentation of rdbplanes\"}},\"definitions\":{\"method-1\":{\"properties\":"
    "{\"maximum_inclination_angle_horizontal\":{\"type\":\"number\",\"minimum\":-36"
    "0.0,\"maximum\":360.0,\"description\":\"maximum inclination angle of "
    "horizontal plane patches [deg]\"},\"plane_classification_method\":{\"type\""
    ":\"integer\",\"minimum\":1,\"maximum\":1,\"description\":\"method ID (=1)\"}},\"t"
    "ype\":\"object\",\"required\":[\"plane_classification_method\",\"maximum_incli"
    "nation_angle_horizontal\"],\"description\":\"Classification method 1\"},\"me"
    "thod-2\":{\"properties\":{\"sloping_plane_classes_maximum_angle\":{\"type\":\""
    "number\",\"minimum\":-360.0,\"maximum\":360.0,\"description\":\"maximum "
    "inclination angle of sloping plane patches [deg]\"},\"sloping_plane_clas"
    "ses_minimum_angle\":{\"type\":\"number\",\"minimum\":-360.0,\"maximum\":360.0,\""
    "description\":\"minimum inclination angle of sloping plane patches [deg]"
    "\"},\"plane_classification_method\":{\"type\":\"integer\",\"minimum\":2,\"maximu"
    "m\":2,\"description\":\"method ID (=2)\"}},\"type\":\"object\",\"required\":[\"pla"
    "ne_classification_method\",\"sloping_plane_classes_minimum_angle\",\"slopi"
    "ng_plane_classes_maximum_angle\"],\"description\":\"Classification method "
    "2\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\""
    "settings\",\"classes\"],\"title\":\"Plane Slope Class "
    "Info\",\"type\":\"object\",\"description\":\"Settings and classes for plane "
    "slope classification\"}"
    ,
    /* example: */
    "{\"classes\":{\"1\":\"horizontal, pointing up\",\"2\":\"sloping, pointing up "
    "and east\",\"7\":\"vertical, pointing south\",\"5\":\"sloping, pointing up and"
    " west\",\"13\":\"sloping, pointing down and west\",\"9\":\"vertical, pointing "
    "west\",\"11\":\"sloping, pointing down and south\",\"8\":\"vertical, pointing "
    "north\",\"10\":\"sloping, pointing down and east\",\"4\":\"sloping, pointing "
    "up and north\",\"6\":\"vertical, pointing east\",\"14\":\"horizontal, pointing"
    " down\",\"12\":\"sloping, pointing down and north\",\"3\":\"sloping, pointing "
    "up and south\"},\"settings\":{\"sloping_plane_classes_maximum_angle\":70.0,"
    "\"sloping_plane_classes_minimum_angle\":10.0,\"plane_classification_metho"
    "d\":2}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "patternProperties\":{\"^.*$\":{\"type\":\"array\",\"items\":{\"type\":\"string\",\"d"
    "escription\":\"Point attribute full name or name pattern (perl regular "
    "expression syntax)\"},\"minItems\":1,\"description\":\"Each field of the "
    "object represents a point attribute group and holds a list of point "
    "attributes, where the field name is the group "
    "name.\"}},\"additionalProperties\":false,\"title\":\"Point Attribute "
    "Groups\",\"description\":\"Grouping and sorting of point attributes for "
    "visualization purposes\"}"
    ,
    /* example: */
    "{\"Primary Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.de"
    "viation\"],\"Other Attributes\":[\"riegl.selected\",\"riegl.visible\"],\"Coord"
    "inates/Vectors\":[\"riegl.xyz\",\"riegl.range\",\"riegl.theta\",\"riegl.phi\"],"
    "\"Time\":[\"riegl.timestamp\"],\"Secondary "
    "Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"comments\":{\"type\":\"string\",\"description\":\"Comments\"},\""
    "field_of_application\":{\"type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\""
    "KLS\",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"description\":\"Field of "
    "application\"},\"project\":{\"type\":\"string\",\"description\":\"Project "
    "name\"}},\"description\":\"Details about point cloud files\",\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\",\"type\":\"object\",\"title\":\"Point "
    "Cloud Information\"}"
    ,
    /* example: */
    "{\"comments\":\"Line 3\",\"field_of_application\":\"ALS\",\"project\":\"Campaign "
    "4\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"position\":{\"properties\":{\"vertical_accuracy\":{\"type\":\""
    "number\",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Vertical "
    "accuracy "
    "[m]\"},\"coordinate_3\":{\"type\":\"number\",\"description\":\"Coordinate 3 as "
    "defined by axis 3 of the specified CRS (e.g., Z, Altitude)\"},\"horizont"
    "al_accuracy\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,\"des"
    "cription\":\"Horizontal accuracy "
    "[m]\"},\"coordinate_2\":{\"type\":\"number\",\"description\":\"Coordinate 2 as "
    "defined by axis 2 of the specified CRS (e.g., Y, "
    "Longitude)\"},\"coordinate_1\":{\"type\":\"number\",\"description\":\"Coordinate"
    " 1 as defined by axis 1 of the specified CRS (e.g., X, Latitude)\"},\"cr"
    "s\":{\"properties\":{\"epsg\":{\"type\":\"integer\",\"minimum\":0,\"description\":\""
    "EPSG code\"},\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\"}},\"type\":\"object\",\"required\":[\"epsg\"],\"description\":\"Global "
    "Coordinate Reference System\"}},\"type\":\"object\",\"required\":[\"coordinate"
    "_1\",\"coordinate_2\",\"coordinate_3\",\"horizontal_accuracy\",\"vertical_accu"
    "racy\",\"crs\"],\"description\":\"Position coordinates and position accuracy"
    " values as measured by GNSS in the specified Coordinate Reference "
    "System (CRS)\"},\"barometric_height_amsl\":{\"type\":\"number\",\"description\""
    ":\"Altitude determined based on the atmospheric pressure according to "
    "the standard atmosphere laws [m].\"},\"orientation\":{\"properties\":{\"yaw_"
    "accuracy\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,\"descri"
    "ption\":\"Yaw angle accuracy [deg]\"},\"yaw\":{\"type\":\"number\",\"minimum\":-3"
    "60,\"maximum\":360,\"description\":\"Yaw angle about scanner Z-axis [deg]\"}"
    ",\"roll_accuracy\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,"
    "\"description\":\"Roll angle accuracy [deg]\"},\"pitch\":{\"type\":\"number\",\"m"
    "inimum\":-360,\"maximum\":360,\"description\":\"Pitch angle about scanner "
    "Y-axis [deg]\"},\"pitch_accuracy\":{\"type\":\"number\",\"exclusiveMinimum\":tr"
    "ue,\"minimum\":0,\"description\":\"Pitch angle accuracy [deg]\"},\"roll\":{\"ty"
    "pe\":\"number\",\"minimum\":-360,\"maximum\":360,\"description\":\"Roll angle "
    "about scanner X-axis [deg]\"}},\"type\":\"object\",\"required\":[\"roll\",\"pitc"
    "h\",\"yaw\",\"roll_accuracy\",\"pitch_accuracy\",\"yaw_accuracy\"],\"description"
    "\":\"Orientation values and orientation accuracies, measured with IMU or"
    " inclination sensors.\"}},\"type\":\"object\",\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\",\"required\":[\"orientation\"],\"title\":\"Pose "
    "Estimation\",\"description\":\"Estimated position and orientation "
    "information as measured by GNSS, IMU or inclination sensors\"}"
    ,
    /* example: */
    "{\"position\":{\"vertical_accuracy\":1.3314999341964722,\"coordinate_3\":362"
    ".7124938964844,\"horizontal_accuracy\":0.810699999332428,\"coordinate_2\":"
    "15.645033406,\"coordinate_1\":48.655799473,\"crs\":{\"epsg\":4979,\"wkt\":\"GEO"
    "GCS[\\\"WGS84 / Geographic\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84\\\",6378137."
    "000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6"
    "326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000000000000,AUTHORITY[\\\"EPSG\\\",\\\"8"
    "901\\\"]],UNIT[\\\"Degree\\\",0.01745329251994329547,AUTHORITY[\\\"EPSG\\\",\\\"91"
    "02\\\"]],AXIS[\\\"Latitude\\\",NORTH],AXIS[\\\"Longitude\\\",EAST],AUTHORITY[\\\"E"
    "PSG\\\",\\\"4979\\\"]]\"}},\"barometric_height_amsl\":386.7457796227932,\"orient"
    "ation\":{\"yaw_accuracy\":1.0094337839368757,\"yaw\":101.87293630292045,\"ro"
    "ll_accuracy\":0.009433783936875745,\"pitch\":1.509153024827064,\"pitch_acc"
    "uracy\":0.009433783936875745,\"roll\":3.14743073066123}}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"accelerometer\":{\"properties\":{\"y_axis\":{\"$ref\":\"#/defi"
    "nitions/vector\",\"description\":\"Sensitive Y axis of sensor at frame "
    "angle = "
    "0\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\""
    "#/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionl"
    "ess\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to "
    "be subtracted from raw measurement values\"},\"x_axis\":{\"$ref\":\"#/defini"
    "tions/vector\",\"description\":\"Sensitive X axis of sensor at frame angle"
    " = 0\"},\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor "
    "origin in SOCS [m] at frame angle = 0\"},\"unit\":{\"type\":\"number\",\"exclu"
    "siveMinimum\":true,\"minimum\":0,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in 9.81 m/s\\u00b2\"}},\"type\":\"object\",\"requir"
    "ed\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_non"
    "linearity\"],\"description\":\"Accelerometer details\"},\"magnetic_field_sen"
    "sor\":{\"properties\":{\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"descripti"
    "on\":\"Sensitive Y axis of sensor at frame angle = "
    "0\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\""
    "#/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionl"
    "ess\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to "
    "be subtracted from raw measurement values\"},\"x_axis\":{\"$ref\":\"#/defini"
    "tions/vector\",\"description\":\"Sensitive X axis of sensor at frame angle"
    " = "
    "0\"},\"fixed\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Distortion "
    "of magnetic field caused by non-rotating scanner part\"},\"unit\":{\"type\""
    ":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Unit of "
    "raw data and calibration values, 1 LSB in nT\"}},\"type\":\"object\",\"requi"
    "red\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"fixed\",\"relative_non"
    "linearity\"],\"description\":\"Magnetic Field Sensor details\"},\"gyroscope\""
    ":{\"properties\":{\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":"
    "\"Sensitive Y axis of sensor at frame angle = "
    "0\"},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\""
    "#/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionl"
    "ess\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to "
    "be subtracted from raw measurement values\"},\"x_axis\":{\"$ref\":\"#/defini"
    "tions/vector\",\"description\":\"Sensitive X axis of sensor at frame angle"
    " = 0\"},\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor "
    "origin in SOCS [m] at frame angle = 0\"},\"unit\":{\"type\":\"number\",\"exclu"
    "siveMinimum\":true,\"minimum\":0,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in rad/s\"}},\"type\":\"object\",\"required\":[\"uni"
    "t\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_nonlinearity"
    "\"],\"description\":\"Gyroscope details\"}},\"type\":\"object\",\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"required\":[\"gyroscope\",\"accelero"
    "meter\",\"magnetic_field_sensor\"],\"title\":\"Pose Sensors\",\"definitions\":{"
    "\"vector\":{\"type\":\"array\",\"minItems\":3,\"maxItems\":3,\"items\":{\"type\":\"nu"
    "mber\",\"description\":\"Index 0=X, 1=Y, 2=Z "
    "component\"}}},\"description\":\"Details on position and orientation "
    "sensors\"}"
    ,
    /* example: */
    "{\"accelerometer\":{\"y_axis\":[-7.027288913726807,-44.12333679199219,1495"
    "2.3701171875],\"z_axis\":[1.639882206916809,15166.744140625,-116.9974288"
    "9404297],\"relative_nonlinearity\":[0.0,0.0,0.0],\"offset\":[-733.36364746"
    "09375,58.969032287597656,1060.2550048828125],\"x_axis\":[-15008.12304687"
    "5,56.956390380859375,-60.5175666809082],\"origin\":[0.026900000870227814"
    ",-0.03999999910593033,-0.08950000256299973],\"unit\":6.666666740784422e-"
    "05},\"magnetic_field_sensor\":{\"y_axis\":[0.00027888521435670555,-0.01142"
    "7424848079681,-5.204829722060822e-05],\"z_axis\":[0.00041987866279669106"
    ",7.876977906562388e-05,0.011407104320824146],\"relative_nonlinearity\":["
    "0.0,0.0,0.0],\"offset\":[-23812.052734375,5606.57666015625,2493.28125],\""
    "x_axis\":[-0.011162743903696537,-2.315962774446234e-05,0.00016818844596"
    "855342],\"fixed\":[-1576.010498046875,1596.081787109375,0.0],\"unit\":91.7"
    "4311828613281},\"gyroscope\":{\"y_axis\":[-0.440765917301178,-0.7897399663"
    "925171,119.5894775390625],\"z_axis\":[0.555869996547699,119.221351623535"
    "16,0.467585027217865],\"relative_nonlinearity\":[2.888176311444113e-07,1"
    ".06274164579645e-07,-1.7186295080634935e-39],\"offset\":[-50.92609786987"
    "305,146.15643310546875,62.4327278137207],\"x_axis\":[-121.195556640625,0"
    ".8219714164733887,0.2313031703233719],\"origin\":[0.026900000870227814,-"
    "0.03999999910593033,-0.08950000256299973],\"unit\":0.0001454441080568358"
    "3}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"phase_step\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":25"
    "5,\"description\":\"Step width in phase of modulation code from line to l"
    "ine\"},\"code_phase_mode\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":255,\"d"
    "escription\":\"0: no synchronization, 1: toggle between 2 phases, 2: "
    "increment with phase_increment\"},\"pulse_interval\":{\"type\":\"array\",\"ite"
    "ms\":{\"type\":\"number\",\"minimum\":0},\"description\":\"Explicit table of the"
    " pulse position modulation used for MTA resolution. Table gives times "
    "between successive laser pulses in seconds.\"},\"num_mod_ampl\":{\"type\":\""
    "integer\",\"minimum\":0,\"maximum\":255,\"description\":\"Number of different "
    "modulation amplitudes (2: binary modulation)\"},\"length\":{\"type\":\"integ"
    "er\",\"minimum\":0,\"maximum\":255,\"description\":\"Length of code\"}},\"type\":"
    "\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required"
    "\":[\"length\",\"num_mod_ampl\",\"pulse_interval\"],\"title\":\"Pulse Position "
    "Modulation\",\"description\":\"Laser pulse position modulation used for "
    "MTA resolution\"}"
    ,
    /* example: */
    "{\"phase_step\":5,\"code_phase_mode\":2,\"pulse_interval\":[2.759375e-06,2.7"
    "59375e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.759375"
    "e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,"
    "2.759375e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821"
    "875e-06,2.759375e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-"
    "06,2.759375e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2."
    "821875e-06,2.759375e-06,2.821875e-06],\"num_mod_ampl\":2,\"length\":31}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "value\"},\"std_dev\":{\"type\":\"number\",\"description\":\"Standard "
    "deviation\"},\"average\":{\"type\":\"number\",\"description\":\"Average "
    "value\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum value\"}},\"ty"
    "pe\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"requ"
    "ired\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"title\":\"Range "
    "Statistics\",\"description\":\"Statistics about target distance wrt. SOCS "
    "origin\"}"
    ,
    /* example: */
    "{\"minimum\":0.919,\"std_dev\":24.349,\"average\":15.49738,\"maximum\":574.35}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"ns\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":4095,\"desc"
    "ription\":\"Number of "
    "samples\"},\"tbl\":{\"type\":\"array\",\"minItems\":1,\"description\":\"various "
    "internal data\",\"items\":{\"$ref\":\"#/definitions/fp_table\"}},\"nt\":{\"type\""
    ":\"integer\",\"minimum\":0,\"maximum\":255,\"description\":\"Number of traces\"}"
    ",\"si\":{\"type\":\"number\",\"minimum\":0,\"maximum\":255,\"description\":\"Start "
    "index "
    "(hw_start)\"},\"a\":{\"type\":\"array\",\"minItems\":1,\"description\":\"Amplitude"
    " [dB]\",\"maxItems\":256,\"items\":{\"type\":\"number\"}},\"sr\":{\"type\":\"number\""
    ",\"exclusiveMinimum\":true,\"minimum\":0,\"description\":\"Sample rate [Hz]\"}"
    ",\"mw\":{\"type\":\"number\",\"exclusiveMinimum\":true,\"minimum\":0,\"maximum\":4"
    "095,\"description\":\"Maximum "
    "weight\"},\"ex\":{\"type\":\"object\",\"description\":\"DEPRECATED, use "
    "'riegl.exponential_decomposition' instead\"},\"t\":{\"additionalProperties"
    "\":false,\"type\":\"object\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/def"
    "initions/fp\",\"description\":\"one field per channel, field name is "
    "channel index\"}}}},\"type\":\"object\",\"$schema\":\"http://json-schema.org/d"
    "raft-04/schema#\",\"title\":\"Receiver Internals\",\"definitions\":{\"fp_table"
    "_row\":{\"type\":\"array\",\"minItems\":1,\"maxItems\":2048,\"items\":{\"type\":\"nu"
    "mber\"}},\"fp_table\":{\"desription\":\"scanner internal data\",\"type\":\"objec"
    "t\",\"required\":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv\"],\"properties\":{\"tv\":{\"type\":\"a"
    "rray\",\"minItems\":1,\"maxItems\":2048,\"items\":{\"oneOf\":[{\"$ref\":\"#/defini"
    "tions/fp_table_row\"},{\"type\":\"number\"}]}},\"tc\":{\"type\":\"integer\",\"desc"
    "ription\":\"table data type "
    "code\",\"max\":255,\"min\":0},\"nx\":{\"type\":\"integer\",\"description\":\"number "
    "of x entries\",\"max\":2048,\"min\":1},\"ch\":{\"type\":\"integer\",\"description\""
    ":\"channel number\",\"max\":255,\"min\":0},\"ny\":{\"type\":\"integer\",\"descripti"
    "on\":\"number of y entries\",\"max\":2048,\"min\":1}}},\"fp\":{\"properties\":{\"s"
    "\":{\"type\":\"array\",\"minItems\":1,\"maxItems\":256,\"items\":{\"type\":\"array\","
    "\"minItems\":1,\"maxItems\":4096,\"items\":{\"type\":\"number\"}}},\"w\":{\"type\":\""
    "array\",\"minItems\":1,\"maxItems\":256,\"items\":{\"type\":\"array\",\"minItems\":"
    "5,\"maxItems\":5,\"items\":{\"type\":\"number\"}}}},\"type\":\"object\",\"required\""
    ":[\"s\",\"w\"],\"description\":\"Fingerprint "
    "values\"}},\"description\":\"Receiver Internals\"}"
    ,
    /* example: */
    "{\"ns\":400,\"tbl\":[{\"tv\":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.5]],\"tc\":1,\"nx\""
    ":5,\"ch\":0,\"ny\":2}],\"nt\":128,\"si\":48,\"a\":[-1.55],\"sr\":7959997000.0,\"mw\""
    ":31,\"t\":{\"1\":{\"s\":[[1.23,4.56],[7.89,0.12]],\"w\":[[78,86,126,134,31],[7"
    "8,86,126,134,31]]},\"0\":{\"s\":[[1.23,4.56],[7.89,0.12]],\"w\":[[78,86,126,"
    "134,31],[78,86,126,134,31]]}}}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"content\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"mi"
    "nItems\":1,\"maxItems\":2000,\"description\":\"Correction value [dB] to be "
    "added to the amplitude\"},\"delta\":{\"type\":\"number\",\"description\":\"Delta"
    " between table entries [m], first entry is at range = 0 m\"}},\"type\":\"o"
    "bject\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":"
    "[\"delta\",\"content\"],\"title\":\"Reflectance Calculation "
    "Table\",\"description\":\"Lookup table for reflectance calculation based "
    "on amplitude and range\"}"
    ,
    /* example: */
    "{\"content\":[-33.01],\"delta\":0.150918}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"ranges_m\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"d"
    "escription\":\"Range [m]\"},\"line_angles_deg\":{\"type\":\"array\",\"items\":{\"t"
    "ype\":\"number\"},\"description\":\"Angle [deg]\"},\"reflectance_correction_db"
    "\":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"de"
    "scription\":\"columns (each value corresponds to an "
    "angle)\"},\"description\":\"rows (each array corresponds to a "
    "range)\"},\"description\":\"Near range reflectance correction in dB as a "
    "function of range over angle\"}},\"type\":\"object\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"required\":[\"ranges_m\",\"line_angles_deg\""
    ",\"reflectance_correction_db\"],\"title\":\"Near-range reflectance "
    "correction\",\"description\":\"Range-dependent and scan-angle-dependent "
    "correction of reflectance reading\"}"
    ,
    /* example: */
    "{\"ranges_m\":[0.0,1.0,2.0,3.0],\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2"
    ".0,2.5,3.0,3.5,4.0],\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,"
    "0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0."
    "8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,"
    "0.1,0.05,0.01]]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"georeferencing\":{\"properties\":{},\"type\":\"object\",\"desc"
    "ription\":\"Internal georeferencing parameters.\"},\"planes\":{\"properties\""
    ":{},\"type\":\"object\",\"description\":\"Information on plane "
    "correspondences and used extraction/matching templates.\"},\"trajectory\""
    ":{\"properties\":{},\"type\":\"object\",\"description\":\"Trajectory "
    "information and statistics.\"},\"optimizer\":{\"properties\":{},\"type\":\"obj"
    "ect\",\"description\":\"Optimization result information and statistics.\"}}"
    ",\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "required\":[\"trajectory\",\"optimizer\",\"georeferencing\"],\"title\":\"RiLOC "
    "Processing Information\",\"description\":\"Details about RiLOC "
    "processing\"}"
    ,
    /* example: */
    "{\"georeferencing\":{},\"planes\":{},\"trajectory\":{},\"optimizer\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"segments\":{\"properties\":{\"program\":{\"$ref\":\"#/definiti"
    "ons/program\"},\"list\":{\"type\":\"array\",\"items\":{\"properties\":{\"start\":{\""
    "type\":\"number\",\"minimum\":0.0,\"maximum\":360.0,\"description\":\"Start "
    "angle in SOCS [deg]\"},\"stop\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":"
    "720.0,\"description\":\"Stop angle in SOCS [deg]\"},\"increment\":{\"type\":\"n"
    "umber\",\"minimum\":0.0,\"maximum\":90.0,\"description\":\"Increment of angle "
    "in SOCS [deg]\"}},\"type\":\"object\",\"required\":[\"start\",\"stop\",\"increment"
    "\"],\"description\":\"Line Scan Segment\"}}},\"type\":\"object\",\"required\":[\"l"
    "ist\"],\"description\":\"Segmented Line Scan Pattern\"},\"rectangular\":{\"pro"
    "perties\":{\"phi_start\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":360.0,\""
    "description\":\"Start phi angle in SOCS [deg]\"},\"phi_stop\":{\"type\":\"numb"
    "er\",\"minimum\":0.0,\"maximum\":720.0,\"description\":\"Stop phi angle in "
    "SOCS [deg]\"},\"theta_increment\":{\"type\":\"number\",\"minimum\":0.0,\"maximum"
    "\":90.0,\"description\":\"Increment of theta angle in SOCS [deg]\"},\"phi_in"
    "crement\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":90.0,\"description\":\""
    "Increment of phi angle in SOCS [deg]\"},\"theta_stop\":{\"type\":\"number\",\""
    "minimum\":0.0,\"maximum\":180.0,\"description\":\"Stop theta angle in SOCS ["
    "deg]\"},\"theta_start\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":180.0,\"d"
    "escription\":\"Start theta angle in SOCS [deg]\"},\"program\":{\"$ref\":\"#/de"
    "finitions/program\"}},\"type\":\"object\",\"required\":[\"phi_start\",\"phi_stop"
    "\",\"phi_increment\",\"theta_start\",\"theta_stop\",\"theta_increment\"],\"descr"
    "iption\":\"Rectangular Field Of View Scan Pattern\"},\"line\":{\"properties\""
    ":{\"start\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":360.0,\"description\""
    ":\"Start angle in SOCS [deg]\"},\"stop\":{\"type\":\"number\",\"minimum\":0.0,\"m"
    "aximum\":720.0,\"description\":\"Stop angle in SOCS [deg]\"},\"increment\":{\""
    "type\":\"number\",\"minimum\":0.0,\"maximum\":90.0,\"description\":\"Increment "
    "of angle in SOCS [deg]\"},\"program\":{\"$ref\":\"#/definitions/program\"}},\""
    "type\":\"object\",\"required\":[\"start\",\"stop\",\"increment\"],\"description\":\""
    "Line Scan Pattern\"}},\"description\":\"Scan pattern description\",\"$schema"
    "\":\"http://json-schema.org/draft-04/schema#\",\"definitions\":{\"program\":{"
    "\"properties\":{\"name\":{\"type\":\"string\",\"description\":\"Name of "
    "measurement program\"},\"laser_prr\":{\"type\":\"number\",\"exclusiveMinimum\":"
    "false,\"minimum\":0,\"description\":\"Laser Pulse Repetition Rate "
    "[Hz]\"}},\"type\":\"object\",\"required\":[\"name\"],\"description\":\"Measurement"
    " program\"}},\"title\":\"Scan Pattern\"}"
    ,
    /* example: */
    "{\"rectangular\":{\"phi_start\":45.0,\"phi_stop\":270.0,\"theta_increment\":0."
    "04,\"phi_increment\":0.04,\"theta_stop\":130.0,\"theta_start\":30.0,\"program"
    "\":{\"name\":\"High Speed\",\"laser_prr\":100000.0}}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"shot_file\":{\"properties\":{\"file_uuid\":{\"type\":\"string\""
    ",\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Shot file "
    "extension, without the leading dot\"}},\"type\":\"object\",\"required\":[\"fil"
    "e_extension\"]}},\"description\":\"Details about laser shot files\",\"$schem"
    "a\":\"http://json-schema.org/draft-04/schema#\",\"type\":\"object\",\"title\":\""
    "Shot Information\"}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_uuid\":\"26a00815-67c0-4bff-8fe8-c577378fe663\",\"file"
    "_extension\":\"sodx\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"activated\":{\"type\":\"boolean\",\"description\":\"Apply "
    "('true') or ignore ('false') all filters\"},\"filters\":{\"type\":\"array\",\""
    "items\":{\"properties\":{\"activated\":{\"type\":\"boolean\",\"description\":\"App"
    "ly ('true') or ignore ('false') this "
    "filter\"},\"description\":{\"type\":\"string\",\"description\":\"A brief "
    "description of the filter (e.g. for display in a "
    "tooltip)\"},\"filter\":{\"type\":\"string\",\"description\":\"The RDB filter "
    "string to apply (e.g. when reading points or index), details see "
    "documentation of function "
    "select()\"},\"title\":{\"type\":\"string\",\"description\":\"A short filter "
    "title (e.g. for display in a selection list)\"}},\"type\":\"object\",\"requi"
    "red\":[\"activated\",\"title\",\"description\",\"filter\"],\"description\":\"Point"
    " filter definition\"},\"description\":\"List of point filters\"}},\"type\":\"o"
    "bject\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":"
    "[\"activated\",\"filters\"],\"title\":\"Stored filters\",\"description\":\"Point "
    "filters applied in addition to the application-defined filters\"}"
    ,
    /* example: */
    "{\"activated\":true,\"filters\":[{\"activated\":false,\"description\":\"Ignore "
    "points with uncertain MTA zone "
    "assignment\",\"filter\":\"riegl.mta_uncertain_point == 0\",\"title\":\"Ignore "
    "uncertain points from MTA "
    "resolution\"},{\"activated\":false,\"description\":\"Ignore points with an "
    "echo signal amplitude below the dynamic "
    "trigger\",\"filter\":\"riegl.dyntrig_uncertain_point == 0\",\"title\":\"Ignore"
    " points below dynamic trigger "
    "level\"},{\"activated\":false,\"description\":\"Ignore points that are "
    "outside the outer AABB in "
    "BOCS\",\"filter\":\"riegl.point_outside_aabb_bocs == 0\",\"title\":\"Ignore "
    "points outside outer AABB\"},{\"activated\":false,\"description\":\"Ignore "
    "points that are inside the inner AABB in "
    "BOCS\",\"filter\":\"riegl.point_inside_aabb_bocs == 0\",\"title\":\"Ignore "
    "points inside inner AABB\"}]}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"author\":{\"type\":\"string\",\"description\":\"Name of the "
    "author that created the "
    "document\"},\"version\":{\"type\":\"string\",\"description\":\"Document format v"
    "ersion\"},\"$class\":{\"type\":\"string\",\"enum\":[\"Document\"],\"description\":\""
    "Object class name\"},\"system\":{\"type\":\"object\",\"description\":\"The "
    "actual system description, details see documentation of RIEGL System "
    "Description\"},\"timestamp\":{\"type\":\"string\",\"description\":\"Date and "
    "time of creation (e.g. 2019-06-19T13:32:10+02:00)\"}},\"type\":\"object\",\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"$class"
    "\",\"version\",\"author\",\"timestamp\",\"system\"],\"title\":\"System "
    "Description\",\"description\":\"Details of major system components like "
    "lidar sensors, cameras and navigation devices\"}"
    ,
    /* example: */
    "{\"author\":\"RIEGL LMS GmbH, Austria\",\"version\":\"1.2.1\",\"$class\":\"Docume"
    "nt\",\"system\":{\"details\":\"see documentation of RIEGL System "
    "Description\"},\"timestamp\":\"2022-09-30T11:56:26+00:00\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"Si\":{\"$ref\":\"#/definitions/conversion_table\",\"descript"
    "ion\":\"Conversion table for Si channel\"},\"InGaAs_Si_Difference\":{\"$ref\""
    ":\"#/definitions/conversion_table\",\"description\":\"Conversion table for "
    "InGaAs - Si difference\"},\"InGaAs\":{\"$ref\":\"#/definitions/conversion_ta"
    "ble\",\"description\":\"Conversion table for InGaAs channel\"}},\"type\":\"obj"
    "ect\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Temp"
    "erature Calculation Table\",\"definitions\":{\"conversion_table\":{\"propert"
    "ies\":{\"value\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":"
    "\"LSB [1]\"},\"temperature\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"de"
    "scription\":\"Temperature [\\u00b0C]\"}},\"type\":\"object\",\"required\":[\"valu"
    "e\",\"temperature\"]}},\"description\":\"Conversion of background radiation "
    "raw values to temperatures in \\u00b0C\"}"
    ,
    /* example: */
    "{\"Si\":{\"value\":[0.0,64.00097659230323,128.0019531846065,192.0029297769"
    "097,256.0039063692129,320.00488296151616,384.0058595538194,448.0068361"
    "461226,512.0078127384259],\"temperature\":[546.300048828125,548.81640512"
    "12026,551.3143938500972,554.0144257850053,556.604252334815,559.2124464"
    "488079,561.8022729986177,564.4104671126105,567.0002936624203]},\"InGaAs"
    "_Si_Difference\":{\"value\":[1000.0,1100.090029602954,1200.04425183874,13"
    "00.1342814416948,1400.0885036774805,1500.0427259132668,1600.1327555162"
    "209,1700.0869777520065,1800.0411999877924],\"temperature\":[1749.9771111"
    "17893,1749.977111117893,1749.977111117893,1749.977111117893,1749.97711"
    "1117893,1749.977111117893,1744.7813348796044,1681.9971312601092,1622.3"
    "944822534868]},\"InGaAs\":{\"value\":[0.0,64.00097659230323,128.0019531846"
    "065,192.0029297769097,256.0039063692129,320.00488296151616,384.0058595"
    "538194,448.0068361461226,512.0078127384259],\"temperature\":[307.2219672"
    "2535614,309.1153478247277,311.1188086915047,313.10025350480055,315.213"
    "7946389828,317.2172555057597,319.2207163725366,321.2021611858325,323.3"
    "157023200148]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"source\":{\"type\":\"string\",\"enum\":[\"unknown\",\"RTC\",\"GNSS"
    "\"],\"description\":\"Timestamp "
    "source\"},\"epoch\":{\"type\":\"string\",\"description\":\"Date and time of "
    "timestamp '0' as proposed by RFC 3339 (e.g. 2015-10-27T00:00:00+01:00)"
    ".\"},\"system\":{\"type\":\"string\",\"enum\":[\"unknown\",\"UTC\",\"GPS\"],\"descript"
    "ion\":\"Time system (time standard)\"}},\"type\":\"object\",\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\",\"required\":[\"epoch\",\"source\"],\"titl"
    "e\":\"Time Base\",\"description\":\"Base of timestamps (epoch)\"}"
    ,
    /* example: */
    "{\"source\":\"GNSS\",\"epoch\":\"2015-10-27T00:00:00+00:00\",\"system\":\"UTC\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"time_interval\":{\"properties\":{\"minimum\":{\"type\":\"numbe"
    "r\",\"description\":\"Minimum time interval "
    "[s]\"},\"std_dev\":{\"type\":\"number\",\"description\":\"Standard deviation of "
    "intervals [s]\"},\"average\":{\"type\":\"number\",\"description\":\"Average time"
    " interval [s]\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum time"
    " interval [s]\"}},\"type\":\"object\",\"required\":[\"minimum\",\"average\",\"maxi"
    "mum\",\"std_dev\"],\"description\":\"Time interval statistics\"},\"field_of_ap"
    "plication\":{\"type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\","
    "\"ULS\",\"ALS\",\"BLS\",\"ILS\"],\"description\":\"Field of "
    "application\"},\"settings\":{\"type\":\"string\",\"description\":\"Settings used"
    " to calculate the trajectory (descriptive "
    "text)\"},\"software\":{\"type\":\"string\",\"description\":\"Software that "
    "calculated the trajectory (this may be the same or different software "
    "than the one that created the "
    "file)\"},\"device\":{\"type\":\"string\",\"description\":\"Navigation device, "
    "e.g. name/type/serial\"},\"navigation_frame\":{\"type\":\"string\",\"enum\":[\"u"
    "nknown\",\"NED\",\"ENU\"],\"description\":\"Navigation frame (NED: "
    "North-East-Down, ENU: "
    "East-North-Up)\"},\"company\":{\"type\":\"string\",\"description\":\"Company "
    "name\"},\"project\":{\"type\":\"string\",\"description\":\"Project "
    "name\"},\"location\":{\"type\":\"string\",\"description\":\"Project location, "
    "e.g. city/state/country\"}},\"type\":\"object\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"required\":[\"time_interval\",\"navigation_frame"
    "\"],\"title\":\"Trajectory Information\",\"description\":\"Details about "
    "position+orientation files\"}"
    ,
    /* example: */
    "{\"time_interval\":{\"minimum\":0.00500032,\"std_dev\":5.51e-07,\"average\":0."
    "00500053,\"maximum\":0.005004883},\"field_of_application\":\"MLS\",\"settings"
    "\":\"default\",\"software\":\"Navigation Software XYZ\",\"device\":\"IMU Model "
    "12/1, Serial# 12345\",\"navigation_frame\":\"NED\",\"company\":\"RIEGL "
    "LMS\",\"project\":\"Campaign 3\",\"location\":\"Horn\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"in_max_duration\":{\"type\":\"integer\",\"minimum\":0,\"descri"
    "ption\":\"stops measurement of pulse duration of input signal [0.1 msec]"
    "\"},\"ext_delay\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"external "
    "trigger, time delay [0.1 "
    "msec]\"},\"ext_signal\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 .."
    " use input signal, 1 .. use output signal of channel\"},\"in_max_delay\":"
    "{\"type\":\"integer\",\"minimum\":0,\"description\":\"maximum delay to output "
    "pulse before fake event is generated [0.1 msec], zero indicates that "
    "no fake events are "
    "generated\"},\"dmi_dist_per_tick\":{\"type\":\"number\",\"description\":\"dmi, "
    "distance per tick, just informative "
    "[m]\"},\"dmi_max_time\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"dmi,"
    " maximum time interval between trigger events [0.1 msec]\"},\"out_durati"
    "on\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"output pulse duration"
    " [0.1 msec]\"},\"num_channel\":{\"type\":\"integer\",\"minimum\":0,\"description"
    "\":\"number of input/output "
    "channels\"},\"input_descr\":{\"type\":\"string\",\"description\":\"descriptive s"
    "tring\"},\"out_interval\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"ou"
    "tput pulse interval [0.1 msec]\"},\"in_min_duration\":{\"type\":\"integer\",\""
    "minimum\":0,\"description\":\"input signals with smaller pulse durations "
    "are ignored [0.1 msec]\"},\"out_num_pulses\":{\"type\":\"integer\",\"minimum\":"
    "0,\"description\":\"number of output pulses to be generated, 0 .. "
    "infinite\"},\"output_descr\":{\"type\":\"string\",\"description\":\"descriptive "
    "string\"},\"ext_channel\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"ID"
    " of channel used as external trigger input, 32 indicates none\"},\"ttip_"
    "version\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"following "
    "main.sub.ss.sss\"},\"dmi_incr\":{\"type\":\"integer\",\"description\":\"dmi, "
    "increment in "
    "ticks\"},\"channel\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"ID of "
    "input/output channel\"},\"ext_bitmask\":{\"type\":\"integer\",\"minimum\":0,\"de"
    "scription\":\"defines which of the internal pulse generators are to be s"
    "tarted\"},\"out_delay\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"outp"
    "ut pulse initial delay after start [0.1 "
    "msec]\"},\"input_usage\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 "
    ".. deactivated, 1 .. just detecting and timestamping\"},\"out_polarity\":"
    "{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 .. positive edge, 1 .. "
    "negative "
    "edge\"},\"output_usage\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 "
    ".. no output, 1 .. output derived from internal clock, 2 .. output "
    "derived from dmi, 3 .. output derived from external signal, 4 .. "
    "output static low, 5 .. output static high\"},\"ext_subdivider\":{\"type\":"
    "\"integer\",\"minimum\":0,\"description\":\"reduces the frequency, default "
    "1\"},\"dmi_min_time\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"dmi, "
    "minimum time interval between trigger events [0.1 "
    "msec]\"},\"in_polarity\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 "
    ".. positive edge, 1 .. negative edge\"}},\"type\":\"object\",\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\",\"required\":[\"ttip_version\",\"num_"
    "channel\",\"ext_channel\",\"ext_signal\",\"ext_delay\",\"ext_subdivider\",\"ext_"
    "bitmask\",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\","
    "\"channel\",\"output_descr\",\"input_descr\",\"output_usage\",\"input_usage\",\"o"
    "ut_polarity\",\"out_duration\",\"out_interval\",\"out_delay\",\"out_num_pulses"
    "\",\"in_polarity\",\"in_min_duration\",\"in_max_duration\",\"in_max_delay\"],\"t"
    "itle\":\"TTIP Configuration\",\"description\":\"Trigger-Timestamping-Unit/IP"
    " configuration\"}"
    ,
    /* example: */
    "{\"in_max_duration\":10000,\"ext_delay\":0,\"ext_signal\":0,\"in_max_delay\":9"
    "990,\"dmi_dist_per_tick\":0.0,\"dmi_max_time\":0,\"out_duration\":300,\"num_c"
    "hannel\":9,\"input_descr\":\"Port 1 - Exposure\",\"out_interval\":10000,\"in_m"
    "in_duration\":0,\"out_num_pulses\":1,\"output_descr\":\"Port 1 - Trigger\",\"e"
    "xt_channel\":32,\"ttip_version\":1,\"dmi_incr\":0,\"channel\":0,\"ext_bitmask\""
    ":0,\"out_delay\":0,\"input_usage\":1,\"out_polarity\":0,\"output_usage\":1,\"ex"
    "t_subdivider\":1,\"dmi_min_time\":0,\"in_polarity\":0}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"vertex_file\":{\"properties\":{\"file_uuid\":{\"type\":\"strin"
    "g\",\"description\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Vertex file "
    "extension, without the leading dot\"}},\"type\":\"object\",\"required\":[\"fil"
    "e_extension\"]}},\"description\":\"Details about vertex file\",\"$schema\":\"h"
    "ttp://json-schema.org/draft-04/schema#\",\"type\":\"object\",\"title\":\"Verte"
    "x Information\"}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_uuid\":\"51534d95-d71f-4f36-ae1a-3e63a21fd1c7\",\"fi"
    "le_extension\":\"vtx\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\""
    "title\":\"Voxel Information\",\"description\":\"Details about the voxels "
    "contained in the file\",\"definitions\":{\"voxel_origin_enum\":{\"enum\":[\"ce"
    "nter\",\"corner\"],\"default\":\"corner\",\"description\":\"Defines whether the "
    "voxel's center or a corner is placed on CRS origin <tt>(0/0/0)</tt>.\"}"
    ",\"voxel_type\":{\"enum\":[\"center\",\"centroid\",\"index\"],\"default\":\"centroi"
    "d\",\"description\":\"Whether a point in a voxel represents its center or "
    "its centroid. If type is <tt>index</tt> there is no point but only an "
    "integer voxel index.\"},\"edge_length\":{\"type\":\"number\",\"exclusiveMinimu"
    "m\":true,\"minimum\":0,\"description\":\"Length of voxel edge [m].\"},\"voxel_"
    "size_cubic\":{\"$ref\":\"#/definitions/edge_length\",\"type\":\"number\"},\"shap"
    "e_thresholds\":{\"properties\":{\"plane\":{\"exclusiveMaximum\":true,\"minimum"
    "\":0,\"exclusiveMinimum\":true,\"maximum\":1,\"type\":\"number\",\"description\":"
    "\"If the smallest eigenvalue is smaller than the median eigenvalue * "
    "plane_threshold, the voxel is considered a plane.\"},\"line\":{\"type\":\"nu"
    "mber\",\"exclusiveMinimum\":true,\"minimum\":1,\"description\":\"If the "
    "biggest eigenvalue is bigger than the median eigenvalue * "
    "line_threshold, the voxel is considered a line.\"}},\"type\":\"object\",\"re"
    "quired\":[\"plane\",\"line\"],\"description\":\"Thresholds used to compute the"
    " voxel's shape_id value.\"},\"voxel_origin_point\":{\"type\":\"array\",\"items"
    "\":{\"type\":\"number\"},\"minItems\":3,\"maxItems\":3,\"description\":\"Origin "
    "point for all voxel indices in voxel "
    "CRS.\"},\"voxel_size\":{\"type\":\"array\",\"description\":\"Size of voxels.\",\"m"
    "inItems\":3,\"maxItems\":3,\"items\":{\"$ref\":\"#/definitions/edge_length\"}}}"
    ",\"oneOf\":[{\"additionalProperties\":false,\"properties\":{\"size\":{\"oneOf\":"
    "[{\"$ref\":\"#/definitions/voxel_size\"},{\"$ref\":\"#/definitions/voxel_size"
    "_cubic\"}],\"description\":\"Size of voxels in file coordinate system.\"},\""
    "voxel_origin\":{\"$ref\":\"#/definitions/voxel_origin_enum\"},\"voxel_type\":"
    "{\"$ref\":\"#/definitions/voxel_type\"},\"shape_thresholds\":{\"$ref\":\"#/defi"
    "nitions/shape_thresholds\"}},\"required\":[\"size\",\"voxel_origin\",\"voxel_t"
    "ype\"]},{\"additionalProperties\":false,\"properties\":{\"size\":{\"$ref\":\"#/d"
    "efinitions/voxel_size\",\"description\":\"Size of voxels in file "
    "coordinate system.\"},\"voxel_origin\":{\"oneOf\":[{\"$ref\":\"#/definitions/v"
    "oxel_origin_enum\"},{\"$ref\":\"#/definitions/voxel_origin_point\",\"descrip"
    "tion\":\"The base point of the voxel grid. Used together with "
    "<tt>voxel_size</tt> and <tt>voxel_index</tt> to compute actual point c"
    "oordinates.\"}]},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_type\"},\"size"
    "_llcs\":{\"$ref\":\"#/definitions/voxel_size\",\"description\":\"Size of "
    "voxels in a locally levelled cartesian coordinate system (xyz). This "
    "is only used for voxels based on a map "
    "projection.\"},\"reference_point\":{\"type\":\"array\",\"description\":\"Point "
    "in WGS84 geodetic decimal degree (EPSG:4326) that was used to compute "
    "the projection distortion parameters. The coefficient order is "
    "latitude, longitude. Only voxels with corresponding geo_tag, "
    "voxel_size and reference_point can be reliably processed together. "
    "This entry is available for voxel files in projected CRS only.\",\"minIt"
    "ems\":2,\"maxItems\":2,\"items\":{\"type\":\"number\",\"minimum\":-180,\"maximum\":"
    "180}},\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_thresholds\"}},\"r"
    "equired\":[\"reference_point\",\"size_llcs\",\"size\",\"voxel_origin\",\"voxel_t"
    "ype\"]}]}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395,0.5143705304787237],\"vo"
    "xel_origin\":\"corner\",\"voxel_type\":\"centroid\",\"size_llcs\":[0.5156575252"
    "891171,0.5130835356683303,0.5143705304787237],\"reference_point\":[48,16"
    "],\"shape_thresholds\":{\"plane\":0.16,\"line\":6}}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"mta_zone\":{\"type\":\"integer\",\"minimum\":1,\"description\":"
    "\"Fixed MTA zone for averaging.\"},\"trim\":{\"type\":\"number\",\"default\":0,\""
    "minimum\":0,\"maximum\":0.5,\"description\":\"Percentage for robust averagin"
    "g.\"},\"num_shots\":{\"type\":\"integer\",\"minimum\":1,\"description\":\"Number "
    "of consecutive shots to be used for averaging.\"}},\"type\":\"object\",\"$sc"
    "hema\":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"num_shots"
    "\",\"mta_zone\"],\"title\":\"Waveform Averaging "
    "Settings\",\"description\":\"Settings for waveform averaging\"}"
    ,
    /* example: */
    "{\"mta_zone\":1,\"trim\":0.05,\"num_shots\":7}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range_offset_m\":{\"type\":\"number\",\"description\":\"Calibr"
    "ated device specific range offset for waveform analysis by system "
    "response fitting in meters.\"},\"range_offset_waveform_samples_m\":{\"type"
    "\":\"number\",\"description\":\"Calibrated device specific range offset for "
    "waveform samples in meters.\"},\"sample_data_files\":{\"type\":\"array\",\"ite"
    "ms\":{\"properties\":{\"laser_wavelength\":{\"type\":\"number\",\"exclusiveMinim"
    "um\":false,\"minimum\":0,\"description\":\"Laser wavelength in meters (0 = u"
    "nknown)\"},\"channel\":{\"exclusiveMaximum\":false,\"type\":\"integer\",\"exclus"
    "iveMinimum\":false,\"maximum\":255,\"minimum\":0,\"description\":\"Sample "
    "block channel number (255 = invalid)\"},\"delta_st\":{\"type\":\"number\",\"de"
    "scription\":\"reserved\"},\"channel_name\":{\"type\":\"string\",\"description\":\""
    "Sample block channel "
    "name\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Sample data "
    "file extension, without the leading "
    "dot\"},\"file_uuid\":{\"type\":\"string\",\"description\":\"File's Universally "
    "Unique Identifier (RFC 4122)\"},\"sample_bits\":{\"exclusiveMaximum\":false"
    ",\"type\":\"integer\",\"exclusiveMinimum\":false,\"maximum\":32,\"minimum\":0,\"d"
    "escription\":\"Bitwidth of samples (e.g. 10 bit, 12 bit)\"},\"sample_inter"
    "val\":{\"type\":\"number\",\"exclusiveMinimum\":false,\"minimum\":0,\"descriptio"
    "n\":\"Sampling interval in seconds\"}},\"type\":\"object\",\"required\":[\"chann"
    "el\",\"channel_name\",\"sample_interval\",\"sample_bits\",\"laser_wavelength\","
    "\"delta_st\",\"file_extension\"]}},\"sample_block_file\":{\"properties\":{\"fil"
    "e_uuid\":{\"type\":\"string\",\"description\":\"File's Universally Unique "
    "Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Sample block "
    "file extension, without the leading dot\"}},\"type\":\"object\",\"required\":"
    "[\"file_extension\"]}},\"type\":\"object\",\"$schema\":\"http://json-schema.org"
    "/draft-04/schema#\",\"required\":[\"sample_block_file\",\"sample_data_files\""
    "],\"title\":\"Waveform Information\",\"description\":\"Details about waveform"
    " files\"}"
    ,
    /* example: */
    "{\"range_offset_m\":3.1415,\"sample_data_files\":[{\"laser_wavelength\":0,\"f"
    "ile_uuid\":\"da084413-e3e8-4655-a122-071de8490d8e\",\"channel\":0,\"delta_st"
    "\":0,\"channel_name\":\"high_power\",\"file_extension\":\"sp0\",\"sample_interva"
    "l\":1.00503e-09,\"sample_bits\":12},{\"laser_wavelength\":0,\"file_uuid\":\"93"
    "585b5e-5ea9-43a1-947b-e7ba3be642d2\",\"channel\":1,\"delta_st\":0,\"channel_"
    "name\":\"low_power\",\"file_extension\":\"sp1\",\"sample_interval\":1.00503e-09"
    ",\"sample_bits\":12},{\"laser_wavelength\":0,\"file_uuid\":\"9d2298c4-1036-46"
    "4f-b5cb-1cf8e517f3a0\",\"channel\":5,\"delta_st\":0,\"channel_name\":\"wwf\",\"f"
    "ile_extension\":\"sp5\",\"sample_interval\":1.00503e-09,\"sample_bits\":12}],"
    "\"range_offset_waveform_samples_m \":7.283,\"sample_block_file\":{\"file_uu"
    "id\":\"93a03615-66c0-4bea-8ff8-c577378fe660\",\"file_extension\":\"sbx\"}}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"description\":\"Scanner settings for waveform output\",\"$schema\":\"http:"
    "//json-schema.org/draft-04/schema#\",\"items\":{\"properties\":{\"pass_dev_g"
    "reater\":{\"type\":\"integer\",\"description\":\"Threshold for deviation "
    "greater than [1]\"},\"enabled\":{\"type\":\"boolean\",\"description\":\"Waveform"
    " output "
    "enabled\"},\"logic_expression\":{\"type\":\"string\",\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\"},\"smart_enabled\":{\"type\":\"boolean\",\"description\":\"Smart "
    "waveform output "
    "enabled\"},\"pass_ampl_less\":{\"type\":\"number\",\"description\":\"Threshold "
    "for amplitude less than "
    "[dB]\"},\"pass_rng_greater\":{\"type\":\"number\",\"description\":\"Threshold "
    "for range greater than "
    "[m]\"},\"channel_idx_mask\":{\"type\":\"integer\",\"description\":\"Bit mask for"
    " channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, ...\"},\"pass_dev_less\":{\"type\":\"integer\",\"description\":\"Threshold"
    " for deviation less than "
    "[1]\"},\"sbl_name\":{\"type\":\"string\",\"description\":\"Name of sample block,"
    " e.g.: wfm, "
    "wwf\"},\"pass_rng_less\":{\"type\":\"number\",\"description\":\"Threshold for "
    "range less than "
    "[m]\"},\"pass_ampl_greater\":{\"type\":\"number\",\"description\":\"Threshold "
    "for amplitude greater than [dB]\"}},\"type\":\"object\",\"required\":[\"sbl_na"
    "me\",\"enabled\",\"channel_idx_mask\"]},\"type\":\"array\",\"title\":\"Waveform "
    "Settings\"}"
    ,
    /* example: */
    "[{\"enabled\":true,\"smart_enabled\":true,\"pass_ampl_less\":5.0,\"pass_rng_g"
    "reater\":9.27,\"channel_idx_mask\":11,\"sbl_name\":\"wfm\",\"pass_rng_less\":13"
    ".11,\"pass_ampl_greater\":1.0},{\"channel_idx_mask\":32,\"sbl_name\":\"wwf\",\""
    "enabled\":false}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"filter\":{\"properties\":{\"range_max\":{\"type\":\"array\",\"de"
    "scription\":\"[m]\",\"items\":{\"type\":\"number\"}},\"amplitude_max\":{\"type\":\"a"
    "rray\",\"description\":\"[dB]\",\"items\":{\"type\":\"number\"}},\"range_min\":{\"ty"
    "pe\":\"array\",\"description\":\"[m]\",\"items\":{\"type\":\"number\"}},\"angle\":{\"t"
    "ype\":\"array\",\"description\":\"[deg]\",\"items\":{\"type\":\"number\"}}},\"type\":"
    "\"object\",\"required\":[\"angle\",\"range_min\",\"range_max\",\"amplitude_max\"]}"
    ",\"settings\":{\"properties\":{\"range\":{\"properties\":{\"additive_value\":{\"t"
    "ype\":\"number\"},\"sigma_factor\":{\"type\":\"number\"}},\"type\":\"object\",\"requ"
    "ired\":[\"sigma_factor\",\"additive_value\"]},\"amplitude\":{\"properties\":{\"a"
    "dditive_value\":{\"type\":\"number\"},\"sigma_factor\":{\"type\":\"number\"}},\"ty"
    "pe\":\"object\",\"required\":[\"sigma_factor\",\"additive_value\"]}},\"type\":\"ob"
    "ject\",\"required\":[\"range\",\"amplitude\"]},\"result\":{\"properties\":{\"ampli"
    "tude_mean\":{\"type\":\"array\",\"description\":\"[dB]\",\"items\":{\"type\":\"numbe"
    "r\"}},\"amplitude_offset\":{\"type\":\"array\",\"description\":\"[dB]\",\"items\":{"
    "\"type\":\"number\"}},\"amplitude_sigma\":{\"type\":\"array\",\"description\":\"[dB"
    "]\",\"items\":{\"type\":\"number\"}},\"range_sigma\":{\"type\":\"array\",\"descripti"
    "on\":\"[m]\",\"items\":{\"type\":\"number\"}},\"timestamp\":{\"type\":\"array\",\"desc"
    "ription\":\"[s]\",\"items\":{\"type\":\"number\"}},\"angle\":{\"type\":\"array\",\"des"
    "cription\":\"[deg]\",\"items\":{\"type\":\"number\"}},\"range_mean\":{\"type\":\"arr"
    "ay\",\"description\":\"[m]\",\"items\":{\"type\":\"number\"}}},\"type\":\"object\",\"r"
    "equired\":[\"angle\",\"range_mean\",\"range_sigma\",\"amplitude_mean\",\"amplitu"
    "de_sigma\",\"amplitude_offset\"]}},\"type\":\"object\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"required\":[\"result\",\"filter\",\"settings\""
    "],\"title\":\"Window Analysis\",\"description\":\"Window analysis data "
    "estimated from scandata and resulting filter parameters\"}"
    ,
    /* example: */
    "{\"filter\":{\"range_max\":[0.424,0.425,0.426,0.427,0.428,0.428,0.429,0.43"
    ",0.431,0.431,0.432],\"amplitude_max\":[8.04,8.01,7.99,7.96,7.93,7.9,7.88"
    ",7.85,7.83,7.8,7.78],\"range_min\":[-0.208,-0.21,-0.212,-0.214,-0.216,-0"
    ".218,-0.219,-0.221,-0.223,-0.225,-0.227],\"angle\":[14.9,15.0,15.1,15.2,"
    "15.3,15.4,15.5,15.6,15.7,15.8,15.9]},\"settings\":{\"range\":{\"additive_va"
    "lue\":0.1,\"sigma_factor\":8},\"amplitude\":{\"additive_value\":1.0,\"sigma_fa"
    "ctor\":4}},\"result\":{\"amplitude_mean\":[5.347396,5.263155,5.224655,5.179"
    "926,5.097782,5.116479,5.051756,4.983473,5.007885,5.002441,4.982],\"ampl"
    "itude_offset\":[1.9,1.9],\"amplitude_sigma\":[0.4272844,0.4298479,0.42368"
    "16,0.4283583,0.4362353,0.4315141,0.4373984,0.4472798,0.4346001,0.43454"
    "87,0.4540681],\"range_sigma\":[0.01869652,0.02151435,0.01747969,0.019187"
    "65,0.01945776,0.01934862,0.01955329,0.02225589,0.02229977,0.01899122,0"
    ".02009433],\"timestamp\":[408.4441,411.4443],\"angle\":[14.9,15.0,15.1,15."
    "2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],\"range_mean\":[0.1105621,0.107956"
    "4,0.1087088,0.1067261,0.1054582,0.1090412,0.102871,0.1019044,0.1051523"
    ",0.1058445,0.1031261]}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"properties\":{\"range\":{\"properties\":{\"entries\":{\"type\":\"integer\",\"min"
    "imum\":1,\"description\":\"Number of range entries\"},\"minimum\":{\"type\":\"nu"
    "mber\",\"minimum\":-2.0,\"maximum\":2.0,\"description\":\"Minimum range in m\"}"
    ",\"maximum\":{\"type\":\"number\",\"minimum\":-2.0,\"maximum\":2.0,\"description\""
    ":\"Maximum range in m\"}},\"type\":\"object\",\"required\":[\"minimum\",\"maximum"
    "\",\"entries\"],\"description\":\"Range axis of correction table\"},\"slices\":"
    "{\"type\":\"array\",\"items\":{\"properties\":{\"table\":{\"type\":\"array\",\"items\""
    ":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"des"
    "cription\":\"Table cell (item 0: amplitude in dB, 1: range in m, 2: "
    "flags)\"},\"minItems\":3,\"maxItems\":3,\"description\":\"Table column (= "
    "range axis)\"},\"minItems\":1,\"description\":\"Table row (= amplitude "
    "axis)\"},\"minItems\":1,\"description\":\"Correction table (dimension "
    "defined by the 'amplitude' and 'range' "
    "objects)\"},\"amplitude\":{\"type\":\"number\",\"description\":\"Window echo "
    "amplitude of slice in dB\"}},\"type\":\"object\",\"required\":[\"amplitude\",\"t"
    "able\"],\"description\":\"Window echo correction parameter slice\"}},\"ampli"
    "tude\":{\"properties\":{\"entries\":{\"type\":\"integer\",\"minimum\":1,\"descript"
    "ion\":\"Number of amplitude entries\"},\"minimum\":{\"type\":\"number\",\"minimu"
    "m\":0.0,\"description\":\"Minimum amplitude in "
    "dB\"},\"maximum\":{\"type\":\"number\",\"minimum\":0.0,\"description\":\"Maximum "
    "amplitude in dB\"}},\"type\":\"object\",\"required\":[\"minimum\",\"maximum\",\"en"
    "tries\"],\"description\":\"Amplitude axis of correction table\"}},\"type\":\"o"
    "bject\",\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"required\":"
    "[\"amplitude\",\"range\",\"slices\"],\"title\":\"Window Echo "
    "Correction\",\"description\":\"Correction parameters for window glass "
    "echoes\"}"
    ,
    /* example: */
    "{\"range\":{\"entries\":128,\"minimum\":-1.5060822940732335,\"maximum\":1.5060"
    "822940732335},\"slices\":[{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]"
    "],\"amplitude\":1.5},{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],\"am"
    "plitude\":2.0}],\"amplitude\":{\"entries\":128,\"minimum\":2,\"maximum\":20}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LAG_ANGLE_CORRECTION,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const std::size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);
const std::vector<const MetaDataItemInfo*> RDB_META_DATA_ITEMS_VECTOR(
    RDB_META_DATA_ITEMS, RDB_META_DATA_ITEMS + RDB_META_DATA_ITEM_COUNT
);

}}} // namespace riegl::rdb::pointcloud

#endif // RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
