/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    queryInvert.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point invert query (C++ wrapper code)
 * \version 2016-11-14/AW: Initial version
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< STRUCT QueryInvert::Private >--------------------------------------------

struct QueryInvert::Private
{
    RDBContext               *context;
    RDBPointcloud            *pointcloud;
    RDBPointcloudQueryInvert *query;

    Private(
        riegl::rdb::PointcloudData       *pcl,
        const std::vector<GraphNode::ID> *nodes,
        const std::string                &filter
    ):
        context(pcl->contextHandle()),
        pointcloud(pcl->pointcloud),
        query(nullptr)
    {
        if (nodes)
        {
            ErrorImpl::check(context, rdb_pointcloud_query_invert_nodes_new(
                context, pointcloud,
                nodes->data(), static_cast<std::uint32_t>(nodes->size()),
                filter.c_str(), &query
            ));
        }
        else
        {
            ErrorImpl::check(context, rdb_pointcloud_query_invert_new(
                context, pointcloud, 0, // = all nodes
                filter.c_str(), &query
            ));
        }
    }

    ~Private()
    {
        ErrorImpl::check(context, rdb_pointcloud_query_invert_delete(
            context, &query
        ));
    }
};

//---< QueryInvert::PUBLIC >----------------------------------------------------

QueryInvert::QueryInvert()
{
}

QueryInvert::QueryInvert(
    riegl::rdb::PointcloudData       *pointcloud,
    const std::vector<GraphNode::ID> *nodes,
    const std::string                &filter
):
    data(new Private(pointcloud, nodes, filter))
{
}

QueryInvert::operator bool() const
{
    return valid();
}

bool QueryInvert::valid() const
{
    return (data != nullptr);
}

void QueryInvert::close()
{
    data.reset();
}

void QueryInvert::attribute(const std::string &name)
{
    if (!data) return;
    ErrorImpl::check(data->context, rdb_pointcloud_query_invert_attribute(
        data->context, data->query, name.c_str()
    ));
}

std::uint32_t QueryInvert::next(std::uint32_t count)
{
    if (!data) return 0;
    std::uint32_t result(0);
    ErrorImpl::check(data->context, rdb_pointcloud_query_invert_next(
        data->context, data->query, count, &result
    ));
    return result;
}

std::uint32_t QueryInvert::progress() const
{
    if (!data) return 0;
    std::uint32_t result(0);
    ErrorImpl::check(data->context, rdb_pointcloud_query_invert_progress(
        data->context, data->query, &result
    ));
    return result;
}

}}} // namespace riegl::rdb::pointcloud
