/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    querySelect.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point select query
 * \version 2015-10-14/AW: Initial version
 * \version 2016-11-04/AW: Allow to read from multiple nodes at once (#2368)
 *
 * This class can be used to select (read) attributes of existing points.
 *
 * \see riegl::rdb::Pointcloud::select()
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_QUERYSELECT_H
#define RIEGL_RDB_POINTCLOUD_QUERYSELECT_H

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb.h"

//---< TYPE DEFINITIONS >-------------------------------------------------------

typedef struct RDBPointcloudQuerySelect RDBPointcloudQuerySelect; // forward declaration of implementation details

//---< FUNCTIONS >--------------------------------------------------------------

RDB_LIBRARY_API_BEGIN

//______________________________________________________________________________
/*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::select()
 */
RDB_FUNCTION(rdb_pointcloud_query_select_new,
    RDBContext                *context,    //!< [in] library context
    RDBPointcloud             *pointcloud, //!< [in] point cloud
    RDBPointcloudGraphNodeID   node,       //!< [in] index node ID
    RDBString                  filter,     //!< [in] filter string
    RDBPointcloudQuerySelect **query       //!< [out] query handle
);

//______________________________________________________________________________
/*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::select()
 */
RDB_FUNCTION(rdb_pointcloud_query_select_nodes_new,
    RDBContext                     *context,    //!< [in] library context
    RDBPointcloud                  *pointcloud, //!< [in] point cloud
    const RDBPointcloudGraphNodeID *nodes,      //!< [in] pointer to first element of node ID array
    uint32_t                        count,      //!< [in] number of elements in node ID array
    RDBString                       filter,     //!< [in] filter string
    RDBPointcloudQuerySelect      **query       //!< [out] query handle
);

//______________________________________________________________________________
/*!
 * \brief Destroy query instance
 */
RDB_FUNCTION(rdb_pointcloud_query_select_delete,
    RDBContext                *context, //!< [in] library context
    RDBPointcloudQuerySelect **query    //!< [in] query handle
);

//______________________________________________________________________________
/*!
 * \brief Bind attribute buffer
 *
 * Use this function to define a target buffer for a point attribute.
 * Exactly one buffer can be defined for an attribute (i.e. only the
 * most recently defined buffer will be used).
 *
 * The buffer is expected to be n*s*d bytes large, where
 * __n__ is the number of points defined in next(),
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *
 * \note This function just stores the buffer pointer. So make
 *       sure that the buffer remains valid until you call next().
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 */
RDB_FUNCTION(rdb_pointcloud_query_select_bind,
    RDBContext               *context,                    //!< [in] library context
    RDBPointcloudQuerySelect *query,                      //!< [in] query handle
    RDBString                 attribute,                  //!< [in] attribute name
    uint32_t                  dataType,                   //!< [in] buffer data type \see dataTypes.h
    void                     *buffer,                     //!< [in] buffer location
    int32_t                   stride RDB_DEFAULT_VALUE(0) //!< [in] bytes between beginnings of successive elements (0: auto)
);

//______________________________________________________________________________
/*!
 * \brief Select points
 *
 * Use this function to actually read the selected points from
 * database and copy the attributes to the defined buffers.
 *
 * Afterwards you may re-use the buffers or define new buffers
 * with bind() and call next() again until all points have been
 * read (i.e. next() returns zero).
 *
 * \returns the number of points read
 */
RDB_FUNCTION(rdb_pointcloud_query_select_next,
    RDBContext               *context,                       //!< [in] library context
    RDBPointcloudQuerySelect *query,                         //!< [in] query handle
    uint32_t                  count,                         //!< [in] size of source buffers in terms of points
    uint32_t                 *processed RDB_DEFAULT_VALUE(0) //!< [out] number of processed points (optional)
);

//______________________________________________________________________________
/*!
 * \brief Progress
 *
 * This function returns a coarse progress information in percent (0..100%).
 * Since the total number of returned points is not known in advance, this
 * value just reflects the progress of the (internal) index traversal.
 *
 * \note When the select query is used to count points (i.e. no buffers
 *       were bound), then this function always returns 0%.
 */
RDB_FUNCTION(rdb_pointcloud_query_select_progress,
    RDBContext               *context, //!< [in] library context
    RDBPointcloudQuerySelect *query,   //!< [in] query handle
    uint32_t                 *progress //!< [out] progress [0..100%]
);

RDB_LIBRARY_API_END

#endif // RIEGL_RDB_POINTCLOUD_QUERYSELECT_H
