/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    management.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Basic point cloud management interface (C++ wrapper code)
 * \version 2017-03-21/AW: Initial version
 * \version 2017-04-13/AW: Functions finalize() and vacuum() added
 * \version 2018-05-25/AW: Function validate() added (#3109)
 * \version 2019-01-18/AW: Parameter 'lodMode' added
 * \version 2020-03-30/AW: Function validate() is const (#3579)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< Management::PUBLIC >-----------------------------------------------------

Management::Management(riegl::rdb::PointcloudData* pointcloud):
    data(pointcloud)
{
}

std::uint32_t Management::getLodMode() const
{
    std::uint32_t value(0);
    ErrorImpl::check(data->contextHandle(),
        rdb_pointcloud_management_get_lod_mode(
            data->contextHandle(),
            data->pointcloud,
            &value
        )
    );
    return value;
}

void Management::setLodMode(const std::uint32_t value)
{
    ErrorImpl::check(data->contextHandle(),
        rdb_pointcloud_management_set_lod_mode(
            data->contextHandle(),
            data->pointcloud,
            value
        )
    );
}

std::uint32_t Management::getChunkSizeLOD() const
{
    std::uint32_t value(0);
    ErrorImpl::check(data->contextHandle(),
        rdb_pointcloud_management_get_chunk_size_lod(
            data->contextHandle(),
            data->pointcloud,
            &value
        )
    );
    return value;
}

void Management::setChunkSizeLOD(const std::uint32_t value)
{
    ErrorImpl::check(data->contextHandle(),
        rdb_pointcloud_management_set_chunk_size_lod(
            data->contextHandle(),
            data->pointcloud,
            value
        )
    );
}

void Management::finalize()
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_management_finalize(
        data->contextHandle(), data->pointcloud
    ));
}

void Management::vacuum(Progress progress, void *userdata)
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_management_vacuum(
        data->contextHandle(), data->pointcloud,
        reinterpret_cast<RDBProgress>(progress),
        userdata
    ));
}

void Management::validate(const std::string &schema, const bool strict) const
{
    ErrorImpl::check(data->contextHandle(), rdb_pointcloud_management_validate(
        data->contextHandle(), data->pointcloud, schema.c_str(), strict ? 1 : 0
    ));
}

}}} // namespace riegl::rdb::pointcloud
