/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-03-select-points.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 3: Select points
 * \version 2015-10-14/AW: Initial version
 * \version 2018-07-10/AW: Use new bindBuffer() instead of bind() function
 * \version 2023-06-12/AW: Use #if instead of ternary operator in select()
 *
 *  This example shows how to open an existing database and query some points.
 *  This example is based on the database of rdb-example-1-create-database.
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <array>
#include <vector>
#include <cstdint>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>
#include <riegl/rdb/default.hpp>

int main()
{
    try
    {
        // New RDB library context
        riegl::rdb::Context context;

        // Access existing database
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);

        // No transaction required here as we are only reading
        // points from database and are not going to modify them.

        // Prepare point attribute buffers
        const uint32_t BUFFER_SIZE = 10000;
        std::vector< uint64_t >               bufferPointNumber(BUFFER_SIZE);
        std::vector< std::array<double, 3> >  bufferCoordinates(BUFFER_SIZE);
        std::vector< float >                  bufferAmplitude  (BUFFER_SIZE);
        std::vector< float >                  bufferReflectance(BUFFER_SIZE);
        std::vector< std::array<uint8_t, 4> > bufferTrueColor  (BUFFER_SIZE);
        std::vector< uint16_t >               bufferPointClass (BUFFER_SIZE);

        // Start new select query to read...
        riegl::rdb::pointcloud::QuerySelect select = rdb.select(
#if 0
            // ...all points (empty filter string):
            ""
#else
            // ...or just some specific points:
            "(riegl.class == 3) && (riegl.xyz[2] > 5.0)"
#endif
        );

        // Tell select query where to store the data
        using namespace riegl::rdb::pointcloud;
        select.bindBuffer(RDB_RIEGL_ID,          bufferPointNumber);
        select.bindBuffer(RDB_RIEGL_XYZ,         bufferCoordinates);
        select.bindBuffer(RDB_RIEGL_AMPLITUDE,   bufferAmplitude);
        select.bindBuffer(RDB_RIEGL_REFLECTANCE, bufferReflectance);
        select.bindBuffer(RDB_RIEGL_RGBA,        bufferTrueColor);
        select.bindBuffer(RDB_RIEGL_CLASS,       bufferPointClass);

        // Read and process all points block-wise
        while (const uint32_t count = select.next(BUFFER_SIZE))
        {
            // Print points to output stream
            for (uint32_t i = 0; i < count; i++)
            {
                std::cout <<     bufferPointNumber[i]     << ","
                          <<     bufferCoordinates[i][0]  << ","
                          <<     bufferCoordinates[i][1]  << ","
                          <<     bufferCoordinates[i][2]  << ","
                          <<     bufferAmplitude  [i]     << ","
                          <<     bufferReflectance[i]     << ","
                          << int(bufferTrueColor  [i][0]) << ","
                          << int(bufferTrueColor  [i][1]) << ","
                          << int(bufferTrueColor  [i][2]) << ","
                          <<     bufferPointClass [i]     << std::endl;
            }
        }

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
