/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    context.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB library context (C++ wrapper code)
 * \version 2015-10-14/AW: Initial version
 * \version 2017-01-30/AW: Added function to check if a file is a RDB 2 file
 * \version 2020-09-17/AW: Added function libraryFilename()
 * \version 2025-01-29/AW: Added function rdb_context_new_with_api() (#5474)
 *
 *******************************************************************************
 */

//---< INCLUDES >---------------------------------------------------------------

#include <cstdlib>

#include "riegl/rdb.h"
#include "riegl/rdb.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {

//---< STRUCT Context::Private >------------------------------------------------

struct Context::Private
{
    RDBContext *context;

    Private(
        const std::string &logLevel,
        const std::string &logPath
    )
    {
        const std::string apiVersion(interfaceName() + " " + interfaceVersion());
        rdb_context_new_with_api(&context, apiVersion.c_str(), logLevel.c_str(), logPath.c_str());
    }

   ~Private()
    {
        rdb_context_delete(&context);
    }
};

//---< Context::PUBLIC >--------------------------------------------------------

Context::Context(
    const std::string &logLevel,
    const std::string &logPath
):
    data(new Context::Private(logLevel, logPath))
{
}

Context::~Context()
{
}

std::string Context::libraryName() const
{
    RDBString value;
    ErrorImpl::check(data->context, rdb_library_name(data->context, &value));
    return std::string(value);
}

std::string Context::libraryFilename() const
{
    RDBString value;
    ErrorImpl::check(data->context, rdb_library_filename(data->context, &value));
    return std::string(value);
}

std::string Context::libraryVersion() const
{
    RDBString value;
    ErrorImpl::check(data->context, rdb_library_version(data->context, &value));
    return std::string(value);
}

std::string Context::libraryLicense() const
{
    RDBString value;
    ErrorImpl::check(data->context, rdb_library_license(data->context, &value));
    return std::string(value);
}

std::string Context::databaseFileTypeTitle() const
{
    RDBString value;
    ErrorImpl::check(data->context, rdb_database_file_type_title(data->context, &value));
    return std::string(value);
}

std::string Context::databaseFileTypeSuffix() const
{
    RDBString value;
    ErrorImpl::check(data->context, rdb_database_file_type_suffix(data->context, &value));
    return std::string(value);
}

bool Context::databaseFileTypeCheck(const std::string &location) const
{
    std::uint32_t value(0);
    ErrorImpl::check(data->context, rdb_database_file_type_check(
        data->context, location.c_str(), &value
    ));
    return (value != 0);
}

}} // namespace riegl::rdb
