/*
 *******************************************************************************
 *
 *  Copyright 2026 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-08-invert-points.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 8: Invert points
 * \version 2016-11-14/AW: Initial version
 * \version 2018-07-10/AW: Use RIEGL default point attribute constants
 *
 *  This example shows how to open an existing database and to invert
 *  a point attribute by switching the "riegl.selected" and "riegl.visible"
 *  bits from 0 to 1 and vice-versa ("bit flip").
 *  Please note that this could also be accomplished by combining select and
 *  update queries as shown in "rdb-example-4-update-points.cpp", but the
 *  invert query might be faster (lower processing time) and easier to use.
 *  This example is based on the database of rdb-example-1-create-database.
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <cstdint>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>
#include <riegl/rdb/default.hpp>

int main()
{
    try
    {
        // New RDB library context
        riegl::rdb::Context context;

        // Access existing database
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);

        // Before we can modify the database, we must start a transaction
        riegl::rdb::pointcloud::TransactionScope transaction(
            rdb,                  // point cloud object
            "Invert",             // transaction title
            "Point Switcher v1.0" // software name
        );

        // Make sure that "selected" and "visible" attributes are defined
        {
            using namespace riegl::rdb::pointcloud;

            if (!rdb.pointAttribute().exists(RDB_RIEGL_SELECTED))
            {
                rdb.pointAttribute().add(RDB_RIEGL_SELECTED);
            }
            if (!rdb.pointAttribute().exists(RDB_RIEGL_VISIBLE))
            {
                rdb.pointAttribute().add(RDB_RIEGL_VISIBLE);
            }
        }

        // Flip the "riegl.selected" bits of all points
        {
            using namespace riegl::rdb::pointcloud;

            QueryInvert invert = rdb.invert();
            invert.attribute(RDB_RIEGL_SELECTED);

            while (const uint32_t count = invert.next(10000))
            {
                std::cout << invert.progress() << "% done" << std::endl;
            }
        }

        // Flip the "riegl.visible" bits of all selected points (i.e.
        // where the point attribute "riegl.selected" is set to 1).
        {
            using namespace riegl::rdb::pointcloud;

            QueryInvert invert = rdb.invert("riegl.selected == 1");
            invert.attribute(RDB_RIEGL_VISIBLE);

            while (const uint32_t count = invert.next(10000))
            {
                std::cout << invert.progress() << "% done" << std::endl;
            }
        }

        // Finally commit transaction
        transaction.commit();

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
