/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb.h
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Main RDB library include file
 * \version 2015-10-14/AW: Initial version
 * \version 2020-02-21/AW: Add tools for context-free functions (#3544)
 * \version 2020-04-17/AW: Add RDB_NO_EXCEPT macro
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_H
#define RIEGL_RDB_H

//---< MACROS >-----------------------------------------------------------------

#if defined(_MSC_VER)
    // MSVC
    #define RDB_LIBRARY_EXPORT __declspec(dllexport)
    #define RDB_LIBRARY_IMPORT __declspec(dllimport)
#elif (__GNUC__ >= 4)
    // GCC
    #define RDB_LIBRARY_EXPORT __attribute__((visibility("default")))
    #define RDB_LIBRARY_IMPORT
#else
    // Unknown
    #define RDB_LIBRARY_EXPORT
    #define RDB_LIBRARY_IMPORT
#endif

#ifdef RDB_LIBRARY
    #define RDB_LIBRARY_API RDB_LIBRARY_EXPORT
#else
    #define RDB_LIBRARY_API RDB_LIBRARY_IMPORT
#endif

#ifdef __cplusplus
    #define RDB_LIBRARY_API_BEGIN extern "C" {
    #define RDB_LIBRARY_API_END }

    #if __cplusplus >= 201103L || _MSC_VER >= 1900
        #define RDB_NO_EXCEPT noexcept
    #else
        #define RDB_NO_EXCEPT throw()
    #endif

    #include <cstdint>
#else
    #define RDB_LIBRARY_API_BEGIN
    #define RDB_LIBRARY_API_END
    #define RDB_NO_EXCEPT

    #include <stdint.h>
#endif

#if defined(__cplusplus) && !defined(RDB_RUNTIME_LINKING)
    // C++ compilers usually support default values for function parameters
    #define RDB_DEFAULT_VALUE(value) = value
#else
    // C compilers usually do NOT support default values for function parameters
    #define RDB_DEFAULT_VALUE(value)
#endif

#ifdef RDB_RUNTIME_LINKING
    // this case is intended for explicit run-time linking, see library.h/.c

    #define RDB_FUNCTION(NAME, ...) \
        typedef RDBResult (*NAME##_type)(__VA_ARGS__); \
        extern NAME##_type NAME

    #define RDB_FUNCTION_CF(NAME, ...) \
        typedef RDBResultCF (*NAME##_type)(__VA_ARGS__); \
        extern NAME##_type NAME
#else
    // this case is used to export/import functions for implicit dynamic linking

    #define RDB_FUNCTION(NAME, ...) \
        RDB_LIBRARY_API RDBResult (NAME)(__VA_ARGS__) RDB_NO_EXCEPT

    #define RDB_FUNCTION_CF(NAME, ...) \
        RDB_LIBRARY_API RDBResultCF (NAME)(__VA_ARGS__) RDB_NO_EXCEPT
#endif

#define RDB_SUCCESS 1 //!< function terminated successfully
#define RDB_FAILURE 0 //!< function execution failed

//---< TYPE DEFINITIONS >-------------------------------------------------------

typedef int32_t                     RDBResult;   //!< function return value type
typedef struct InternalRDBResultCF *RDBResultCF; //!< cf-function return value type
typedef const char*                 RDBString;   //!< simple string type, UTF-8 encoded

//---< INCLUDES >---------------------------------------------------------------

#include "riegl/rdb/error.h"
#include "riegl/rdb/result.h"
#include "riegl/rdb/library.h"
#include "riegl/rdb/version.h"
#include "riegl/rdb/context.h"
#include "riegl/rdb/progress.h"
#include "riegl/rdb/pointcloud.h"

#endif // RIEGL_RDB_HP
