/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 * \version 2025-11-11/AW: Item "riegl.lag_angle_correction" added (#6068)
 *
 *******************************************************************************
 */

#ifndef RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1
#define RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1

// Meta data item definition version
const char* const RDB_META_DATA_ITEMS_VERSION = "1.1.57";
const char* const RDB_META_DATA_ITEMS_DATE = "2025-11-11";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
typedef struct MetaDataItemInfo
{
    const char*   name;
    const char*   title;
    const char*   description;
    unsigned char status;
    const char*   schema;
    const char*   example;
} MetaDataItemInfo;

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Angular Notch Filter\",\"required\":[\"angle\",\"range_mean\",\"ampl"
    "itude_mean\"],\"type\":\"object\",\"properties\":{\"angle\":{\"items\":{\"type\":\"n"
    "umber\"},\"description\":\"Angle [deg]\",\"type\":\"array\"},\"amplitude_mean\":{"
    "\"items\":{\"type\":\"number\"},\"description\":\"Mean amplitude [dB]\",\"type\":\""
    "array\"},\"range_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"Mean "
    "range [m]\",\"type\":\"array\"}},\"$schema\":\"http://json-schema.org/draft-04"
    "/schema#\",\"description\":\"Angular notch filter parameters for window "
    "glass echoes\"}"
    ,
    /* example: */
    "{\"angle\":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],\"amp"
    "litude_mean\":[3.780913,3.780913,3.480913,3.120913,2.850913,2.720913,2."
    "680913,2.610913,2.530913,2.570913,2.570913],\"range_mean\":[0.094,0.094,"
    "0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Atmospheric Parameters\",\"required\":[\"temperature\",\"pressure\""
    ",\"rel_humidity\",\"pressure_sl\",\"amsl\",\"group_velocity\",\"attenuation\",\"w"
    "avelength\"],\"type\":\"object\",\"properties\":{\"temperature\":{\"description\""
    ":\"Temperature along measurement path "
    "[\\u00b0C]\",\"type\":\"number\"},\"wavelength\":{\"description\":\"Laser "
    "wavelength "
    "[nm]\",\"type\":\"number\"},\"attenuation\":{\"description\":\"Atmospheric "
    "attenuation [1/km]\",\"type\":\"number\"},\"amsl\":{\"description\":\"Height "
    "above mean sea level (AMSL) "
    "[m]\",\"type\":\"number\"},\"group_velocity\":{\"description\":\"Group velocity "
    "of laser beam "
    "[m/s]\",\"type\":\"number\"},\"rel_humidity\":{\"description\":\"Relative "
    "humidity along measurement path "
    "[%]\",\"type\":\"number\"},\"pressure_sl\":{\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\",\"type\":\"number\"},\"pressure\":{\"description\":\"Pressure along "
    "measurment path [mbar]\",\"type\":\"number\"}},\"$schema\":\"http://json-schem"
    "a.org/draft-04/schema#\",\"description\":\"Atmospheric parameters\"}"
    ,
    /* example: */
    "{\"temperature\":7,\"wavelength\":1550,\"attenuation\":0.028125,\"amsl\":0,\"gr"
    "oup_velocity\":299711000.0,\"rel_humidity\":63,\"pressure_sl\":970,\"pressur"
    "e\":970}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Beam Geometry\",\"required\":[\"beam_exit_diameter\",\"beam_diverg"
    "ence\"],\"type\":\"object\",\"properties\":{\"beam_divergence\":{\"minimum\":0,\"d"
    "escription\":\"Beam divergence in far field [rad]\",\"type\":\"number\",\"excl"
    "usiveMinimum\":false},\"beam_exit_diameter\":{\"minimum\":0,\"description\":\""
    "Beam width at exit aperture [m]\",\"type\":\"number\",\"exclusiveMinimum\":fa"
    "lse}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"description"
    "\":\"Laser beam geometry details\"}"
    ,
    /* example: */
    "{\"beam_divergence\":0.0003,\"beam_exit_diameter\":0.0072}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Control Object Catalog\",\"required\":[\"types\"],\"type\":\"object\""
    ",\"properties\":{\"types\":{\"items\":{\"oneOf\":[{\"$ref\":\"#/definitions/recta"
    "ngle\"},{\"$ref\":\"#/definitions/checkerboard2x2\"},{\"$ref\":\"#/definitions"
    "/chevron\"},{\"$ref\":\"#/definitions/circular_disk\"},{\"$ref\":\"#/definitio"
    "ns/cylinder\"},{\"$ref\":\"#/definitions/sphere\"},{\"$ref\":\"#/definitions/r"
    "ound_corner_cube_prism\"}],\"type\":\"object\"},\"type\":\"array\",\"uniqueItems"
    "\":true}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"descript"
    "ion\":\"List of control object type definitions\",\"definitions\":{\"checker"
    "board2x2\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"shap"
    "e\",\"square_length\"],\"description\":\"checkerboard specific properties\",\""
    "type\":\"object\",\"properties\":{\"shape\":{\"description\":\"shape identifier\""
    ",\"type\":\"string\",\"enum\":[\"checkerboard2x2\"]},\"square_length\":{\"minimum"
    "\":0,\"description\":\"length of a square of the checkerboard in meters\",\""
    "type\":\"number\",\"exclusiveMinimum\":true}}}],\"description\":\"checkerboard"
    " 2 by 2\",\"type\":\"object\"},\"sphere\":{\"allOf\":[{\"$ref\":\"#/definitions/co"
    "mmon\"},{\"required\":[\"shape\",\"diameter\"],\"description\":\"sphere specific"
    " properties\",\"type\":\"object\",\"properties\":{\"shape\":{\"description\":\"sha"
    "pe identifier\",\"type\":\"string\",\"enum\":[\"sphere\"]},\"diameter\":{\"minimum"
    "\":0,\"description\":\"diameter in meters\",\"type\":\"number\",\"exclusiveMinim"
    "um\":true}}}],\"description\":\"sphere\",\"type\":\"object\"},\"cylinder\":{\"allO"
    "f\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"shape\",\"diameter\",\"h"
    "eight\"],\"description\":\"cylinder specific properties\",\"type\":\"object\",\""
    "properties\":{\"height\":{\"minimum\":0,\"description\":\"height in meters\",\"t"
    "ype\":\"number\",\"exclusiveMinimum\":true},\"shape\":{\"description\":\"shape i"
    "dentifier\",\"type\":\"string\",\"enum\":[\"cylinder\"]},\"diameter\":{\"minimum\":"
    "0,\"description\":\"diameter in meters\",\"type\":\"number\",\"exclusiveMinimum"
    "\":true}}}],\"description\":\"cylinder\",\"type\":\"object\"},\"rectangle\":{\"all"
    "Of\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"shape\",\"length\",\"wi"
    "dth\"],\"description\":\"rectangle specific properties\",\"type\":\"object\",\"p"
    "roperties\":{\"width\":{\"minimum\":0,\"description\":\"width in meters\",\"type"
    "\":\"number\",\"exclusiveMinimum\":true},\"shape\":{\"type\":\"string\",\"enum\":[\""
    "rectangle\"]},\"length\":{\"minimum\":0,\"description\":\"length in meters\",\"t"
    "ype\":\"number\",\"exclusiveMinimum\":true}}}],\"description\":\"rectangle\",\"t"
    "ype\":\"object\"},\"round_corner_cube_prism\":{\"allOf\":[{\"$ref\":\"#/definiti"
    "ons/common\"},{\"required\":[\"shape\",\"diameter\"],\"description\":\"round "
    "corner cube prism specific properties\",\"type\":\"object\",\"properties\":{\""
    "shape\":{\"description\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"roun"
    "d_corner_cube_prism\"]},\"diameter\":{\"minimum\":0,\"description\":\"diameter"
    " in meters\",\"type\":\"number\",\"exclusiveMinimum\":true},\"offset\":{\"descri"
    "ption\":\"offset in meters, e.g. reflector constant "
    "(optional)\",\"type\":\"number\"}}}],\"description\":\"round corner cube prism"
    "\",\"type\":\"object\"},\"common\":{\"required\":[\"name\",\"shape\"],\"description\""
    ":\"common object properties\",\"type\":\"object\",\"properties\":{\"shape\":{\"de"
    "scription\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"rectangle\",\"che"
    "ckerboard2x2\",\"chevron\",\"circular_disk\",\"cylinder\",\"sphere\",\"round_cor"
    "ner_cube_prism\"]},\"description\":{\"description\":\"string describing the "
    "object\",\"type\":\"string\"},\"surface_type\":{\"description\":\"surface "
    "material type\",\"type\":\"string\",\"enum\":[\"retro_reflective_foil\",\"diffus"
    "e\"]},\"name\":{\"description\":\"unique type identifier\",\"type\":\"string\",\"m"
    "inLength\":3}}},\"chevron\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"r"
    "equired\":[\"shape\",\"outside_edge_length\",\"thickness\"],\"description\":\"ch"
    "evron specific properties\",\"type\":\"object\",\"properties\":{\"shape\":{\"des"
    "cription\":\"shape identifier\",\"type\":\"string\",\"enum\":[\"chevron\"]},\"thic"
    "kness\":{\"minimum\":0,\"description\":\"thickness in meters\",\"type\":\"number"
    "\",\"exclusiveMinimum\":true},\"outside_edge_length\":{\"minimum\":0,\"descrip"
    "tion\":\"length of the two outer edges in meters\",\"type\":\"number\",\"exclu"
    "siveMinimum\":true}}}],\"description\":\"chevron\",\"type\":\"object\"},\"circul"
    "ar_disk\":{\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"required\":[\"shape"
    "\",\"diameter\"],\"description\":\"circular disk specific properties\",\"type\""
    ":\"object\",\"properties\":{\"shape\":{\"description\":\"shape identifier\",\"typ"
    "e\":\"string\",\"enum\":[\"circular_disk\"]},\"diameter\":{\"minimum\":0,\"descrip"
    "tion\":\"diameter in meters\",\"type\":\"number\",\"exclusiveMinimum\":true},\"o"
    "ffset\":{\"description\":\"offset in meters, e.g. reflector constant "
    "(optional)\",\"type\":\"number\"}}}],\"description\":\"circular "
    "disk\",\"type\":\"object\"}}}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or 'diffuse')\"],\"types\":[{\"width\":0.3,"
    "\"shape\":\"rectangle\",\"description\":\"Rectangle (60cm x "
    "30cm)\",\"length\":0.6,\"name\":\"Rectangle "
    "60x30\"},{\"width\":0.4,\"shape\":\"rectangle\",\"description\":\"Rectangle "
    "(80cm x 40cm)\",\"length\":0.8,\"name\":\"Rectangle "
    "80x40\"},{\"shape\":\"checkerboard2x2\",\"description\":\"Checkerboard (square"
    " length: 30cm)\",\"square_length\":0.3,\"name\":\"Checkerboard2x2 "
    "30\"},{\"shape\":\"checkerboard2x2\",\"description\":\"Checkerboard (square "
    "length: 50cm)\",\"square_length\":0.5,\"name\":\"Checkerboard2x2 "
    "50\"},{\"shape\":\"chevron\",\"description\":\"Chevron (a=24''; b=4'')\",\"thick"
    "ness\":0.1016,\"outside_edge_length\":0.6096,\"name\":\"Chevron "
    "24''/4''\"},{\"shape\":\"circular_disk\",\"description\":\" Circular Disk "
    "(diameter: "
    "50cm)\",\"diameter\":0.5,\"surface_type\":\"diffuse\",\"name\":\"Circular Disk "
    "50\"},{\"diameter\":0.05,\"offset\":0.0,\"name\":\"RIEGL flat reflector 50 "
    "mm\",\"shape\":\"circular_disk\",\"description\":\"flat circular reflector "
    "from retro-reflective foil\",\"surface_type\":\"retro_reflective_foil\"},{\""
    "diameter\":0.1,\"offset\":0.0,\"name\":\"RIEGL flat reflector 100 "
    "mm\",\"shape\":\"circular_disk\",\"description\":\"flat circular reflector "
    "from retro-reflective foil\",\"surface_type\":\"retro_reflective_foil\"},{\""
    "diameter\":0.15,\"offset\":0.0,\"name\":\"RIEGL flat reflector 150 "
    "mm\",\"shape\":\"circular_disk\",\"description\":\"flat circular reflector "
    "from retro-reflective foil\",\"surface_type\":\"retro_reflective_foil\"},{\""
    "height\":0.05,\"diameter\":0.05,\"name\":\"RIEGL cylindrical reflector 50 "
    "mm\",\"shape\":\"cylinder\",\"description\":\"cylindrical reflector from "
    "retro-reflective foil\",\"surface_type\":\"retro_reflective_foil\"},{\"heigh"
    "t\":0.1,\"diameter\":0.1,\"name\":\"RIEGL cylindrical reflector 100 "
    "mm\",\"shape\":\"cylinder\",\"description\":\"cylindrical reflector from "
    "retro-reflective foil\",\"surface_type\":\"retro_reflective_foil\"},{\"shape"
    "\":\"sphere\",\"description\":\"Sphere (diameter: 200 "
    "mm)\",\"diameter\":0.2,\"name\":\"Sphere 200 "
    "mm\"},{\"shape\":\"round_corner_cube_prism\",\"description\":\"round corner "
    "cube prism\",\"diameter\":0.05,\"offset\":0.0,\"name\":\"Corner Cube Prism 50 "
    "mm\"}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Control "
    "Object Reference file\",\"description\":\"Details about the control object"
    " reference file\",\"type\":\"object\",\"properties\":{\"reference_file\":{\"requ"
    "ired\":[\"file_uuid\",\"file_path\"],\"description\":\"Reference to a control "
    "object file\",\"type\":\"object\",\"properties\":{\"file_uuid\":{\"description\":"
    "\"Control object file's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_path\":{\"description\":\"Path of the "
    "control object file relative to referring file\",\"type\":\"string\"}}}}}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\","
    "\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoints.cpx\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Detection Probability\",\"required\":[\"amplitude\",\"detection_pr"
    "obability\"],\"type\":\"object\",\"properties\":{\"detection_probability\":{\"it"
    "ems\":{\"type\":\"number\"},\"description\":\"Detection probability [0..1]\",\"t"
    "ype\":\"array\"},\"amplitude\":{\"items\":{\"type\":\"number\"},\"description\":\"Am"
    "plitude [dB]\",\"type\":\"array\"}},\"$schema\":\"http://json-schema.org/draft"
    "-04/schema#\",\"description\":\"Detection probability as a function of "
    "amplitude\"}"
    ,
    /* example: */
    "{\"detection_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]"
    ",\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Device Information\",\"required\":[\"device_type\",\"serial_number"
    "\"],\"type\":\"object\",\"properties\":{\"channel_number\":{\"minimum\":0,\"descri"
    "ption\":\"Laser channel number (not defined or 0: single channel device)"
    "\",\"type\":\"integer\",\"exclusiveMinimum\":false},\"device_build\":{\"descript"
    "ion\":\"Device build "
    "variant\",\"type\":\"string\"},\"channel_text\":{\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\",\"type\":\"string\"},\"device_name\":{\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"serial_number\":{\"description\":\"Device "
    "serial number (e.g. S2221234)\",\"type\":\"string\"}},\"$schema\":\"http://jso"
    "n-schema.org/draft-04/schema#\",\"description\":\"Details about the device"
    " used to acquire the point cloud\"}"
    ,
    /* example: */
    "{\"channel_number\":0,\"device_build\":\"\",\"channel_text\":\"\",\"device_name\":"
    "\"Scanner 1\",\"device_type\":\"VZ-400i\",\"serial_number\":\"S2221234\"}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Device Geometry\",\"required\":[\"primary\"],\"type\":\"object\",\"pro"
    "perties\":{\"secondary\":{\"required\":[\"ID\",\"content\"],\"description\":\"Addi"
    "tional device geometry structure (optional)\",\"type\":\"object\",\"properti"
    "es\":{\"ID\":{\"minItems\":2,\"items\":{\"type\":\"integer\"},\"description\":\"Stru"
    "cture identifier\",\"type\":\"array\",\"maxItems\":2},\"content\":{\"items\":{\"ty"
    "pe\":\"number\"},\"description\":\"Internal calibration "
    "values\",\"type\":\"array\"}}},\"amu\":{\"description\":\"Angle Measurement Unit"
    "\",\"type\":\"object\",\"properties\":{\"frameCC\":{\"minimum\":0,\"description\":\""
    "Frame Circle Count (number of LSBs per full rotation about frame axis)"
    "\",\"type\":\"number\",\"exclusiveMinimum\":false},\"lineCC\":{\"minimum\":0,\"des"
    "cription\":\"Line Circle Count (number of LSBs per full rotation about "
    "line axis)\",\"type\":\"number\",\"exclusiveMinimum\":false}}},\"primary\":{\"re"
    "quired\":[\"ID\",\"content\"],\"description\":\"Primary device geometry "
    "structure (mandatory)\",\"type\":\"object\",\"properties\":{\"ID\":{\"minItems\":"
    "2,\"items\":{\"type\":\"integer\"},\"description\":\"Structure identifier\",\"typ"
    "e\":\"array\",\"maxItems\":2},\"content\":{\"items\":{\"type\":\"number\"},\"descrip"
    "tion\":\"Internal calibration values\",\"type\":\"array\"}}}},\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"description\":\"Scanner device "
    "geometry details\"}"
    ,
    /* example: */
    "{\"secondary\":{\"ID\":[91,0],\"content\":[0]},\"amu\":{\"frameCC\":124000,\"line"
    "CC\":124000},\"primary\":{\"ID\":[4,0],\"content\":[0]}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Device Geometry Passive Channel\",\"required\":[\"primary\"],\"typ"
    "e\":\"object\",\"properties\":{\"primary\":{\"required\":[\"ID\",\"content\"],\"desc"
    "ription\":\"Primary device geometry structure (mandatory)\",\"type\":\"objec"
    "t\",\"properties\":{\"ID\":{\"minItems\":2,\"items\":{\"type\":\"integer\"},\"descri"
    "ption\":\"Structure identifier\",\"type\":\"array\",\"maxItems\":2},\"content\":{"
    "\"items\":{\"type\":\"number\"},\"description\":\"Internal calibration values\","
    "\"type\":\"array\"}}}},\"$schema\":\"http://json-schema.org/draft-04/schema#\""
    ",\"description\":\"Scanner passive channel device geometry details\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[143,0],\"content\":[0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Device "
    "Output Limits\",\"description\":\"Limits of the measured values output by "
    "the device. The limits depend on the device type, measurement program "
    "and/or scan pattern.\",\"type\":\"object\",\"properties\":{\"deviation_minimum"
    "\":{\"description\":\"Minimum possible pulse shape "
    "deviation.\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum "
    "possible range in meters.\",\"type\":\"number\"},\"background_radiation_mini"
    "mum\":{\"description\":\"Minimum possible background radiation.\",\"type\":\"n"
    "umber\"},\"echo_count_maximum\":{\"description\":\"Maximum number of echoes "
    "a laser shot can "
    "have.\",\"type\":\"number\"},\"reflectance_maximum\":{\"description\":\"Maximum "
    "possible reflectance in dB.\",\"type\":\"number\"},\"background_radiation_ma"
    "ximum\":{\"description\":\"Maximum possible background radiation.\",\"type\":"
    "\"number\"},\"mta_zone_count_maximum\":{\"description\":\"Maximum number of "
    "MTA "
    "zones.\",\"type\":\"number\"},\"amplitude_maximum\":{\"description\":\"Maximum "
    "possible amplitude in "
    "dB.\",\"type\":\"number\"},\"reflectance_minimum\":{\"description\":\"Minimum "
    "possible reflectance in "
    "dB.\",\"type\":\"number\"},\"range_minimum\":{\"description\":\"Minimum possible"
    " range in "
    "meters.\",\"type\":\"number\"},\"amplitude_minimum\":{\"description\":\"Minimum "
    "possible amplitude in "
    "dB.\",\"type\":\"number\"},\"deviation_maximum\":{\"description\":\"Maximum "
    "possible pulse shape deviation.\",\"type\":\"number\"}}}"
    ,
    /* example: */
    "{\"deviation_maximum\":32767,\"mta_zone_count_maximum\":7,\"deviation_minim"
    "um\":-1,\"amplitude_maximum\":100.0,\"range_maximum\":10000.0,\"background_r"
    "adiation_minimum\":0,\"range_minimum\":2.9,\"reflectance_maximum\":100.0,\"a"
    "mplitude_minimum\":0.0,\"reflectance_minimum\":-100.0,\"background_radiati"
    "on_maximum\":0}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Devices "
    "Information\",\"description\":\"Details about the devices used to acquire "
    "the point cloud (e.g. when merging point clouds of different devices)\""
    ",\"type\":\"array\",\"items\":{\"required\":[\"device_type\",\"serial_number\"],\"t"
    "ype\":\"object\",\"properties\":{\"channel_number\":{\"minimum\":0,\"description"
    "\":\"Laser channel number (not defined or 0: single channel device)\",\"ty"
    "pe\":\"integer\",\"exclusiveMinimum\":false},\"device_build\":{\"description\":"
    "\"Device build "
    "variant\",\"type\":\"string\"},\"channel_text\":{\"description\":\"Optional "
    "channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\",\"type\":\"string\"},\"signed\":{\"description\":\"Flag that is set "
    "when the original 'riegl.device' entry in the source file was "
    "correctly "
    "signed.\",\"type\":\"boolean\"},\"device_name\":{\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\",\"type\":\"string\"},\"device_type\":{\"description\":\"Device type "
    "identifier (e.g. "
    "VZ-400i)\",\"type\":\"string\"},\"serial_number\":{\"description\":\"Device "
    "serial number (e.g. S2221234)\",\"type\":\"string\"}}}}"
    ,
    /* example: */
    "[{\"channel_number\":0,\"device_build\":\"\",\"channel_text\":\"\",\"signed\":fals"
    "e,\"device_name\":\"Scanner 1\",\"device_type\":\"VZ-400i\",\"serial_number\":\"S"
    "2221234\"},{\"channel_number\":1,\"device_build\":\"\",\"channel_text\":\"\",\"sig"
    "ned\":true,\"device_name\":\"Scanner 2\",\"device_type\":\"VQ-1560i-DW\",\"seria"
    "l_number\":\"S2222680\"},{\"channel_number\":2,\"device_build\":\"\",\"channel_t"
    "ext\":\"\",\"signed\":true,\"device_name\":\"Scanner "
    "3\",\"device_type\":\"VQ-1560i-DW\",\"serial_number\":\"S2222680\"}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"DMI Configuration\",\"required\":[\"dmi_count_dir\",\"dmi_incr\",\"d"
    "mi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\"],\"type\":\"object\",\"prop"
    "erties\":{\"dmi_incr\":{\"minimum\":0,\"type\":\"integer\",\"exclusiveMinimum\":f"
    "alse,\"exclusiveMaximum\":false,\"description\":\"Increment in ticks, 0 "
    "means distance trigger is deactivated\",\"maximum\":2147483647},\"dmi_min_"
    "time\":{\"minimum\":0,\"type\":\"integer\",\"exclusiveMinimum\":false,\"exclusiv"
    "eMaximum\":false,\"description\":\"Minimum time interval between trigger "
    "events, 0 means rate limitation is inactive [0.1 msec]\",\"maximum\":6553"
    "5},\"dmi_count_dir\":{\"minimum\":0,\"type\":\"integer\",\"exclusiveMinimum\":fa"
    "lse,\"exclusiveMaximum\":false,\"description\":\"DMI count direction is "
    "inverted\",\"maximum\":1},\"dmi_dist_per_tick\":{\"description\":\"Distance "
    "per tick, just informative [m]\",\"type\":\"number\"},\"dmi_max_time\":{\"mini"
    "mum\":0,\"type\":\"integer\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":fa"
    "lse,\"description\":\"Maximum time interval between trigger events, 0 "
    "means timeout is inactive [0.1 msec]\",\"maximum\":65535}},\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\",\"description\":\"Distance "
    "measuring instrument configuration\"}"
    ,
    /* example: */
    "{\"dmi_incr\":0,\"dmi_min_time\":0,\"dmi_count_dir\":0,\"dmi_dist_per_tick\":0"
    ".00024399999529123306,\"dmi_max_time\":100}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Dyntrig\""
    ",\"description\":\"Dyntrig\",\"properties\":{\"post\":{\"required\":[\"nc\",\"a\"],\""
    "description\":\"Dyntrig post values\",\"type\":\"object\",\"properties\":{\"a\":{"
    "\"minItems\":1,\"items\":{\"type\":\"number\"},\"description\":\"...\",\"maxItems\":"
    "2080,\"type\":\"array\"},\"nc\":{\"minimum\":0,\"description\":\"Number of entrie"
    "s\",\"type\":\"integer\",\"maximum\":255}}},\"pre\":{\"required\":[\"nc\",\"a\"],\"des"
    "cription\":\"Dyntrig pre values\",\"type\":\"object\",\"properties\":{\"a\":{\"min"
    "Items\":1,\"items\":{\"type\":\"number\"},\"description\":\"...\",\"maxItems\":2080"
    ",\"type\":\"array\"},\"nc\":{\"minimum\":0,\"description\":\"Number of entries\",\""
    "type\":\"integer\",\"maximum\":255}}},\"tab\":{\"minItems\":1,\"items\":{\"require"
    "d\":[\"thra1\",\"del1\",\"dec1\",\"thra2\",\"del2\",\"dec2\"],\"description\":\"...\",\""
    "type\":\"object\",\"properties\":{\"del2\":{\"description\":\"Length of constant"
    " pre-trigger threshold elevation in units of "
    "m\",\"type\":\"number\"},\"thra2\":{\"description\":\"Pre-trigger threshold in "
    "units of dB\",\"type\":\"number\"},\"dec1\":{\"description\":\"Decay of "
    "post-trigger threshold after del1 in units of "
    "dB/m\",\"type\":\"number\"},\"thra1\":{\"description\":\"Post-trigger threshold "
    "in units of dB\",\"type\":\"number\"},\"dec2\":{\"description\":\"Rise of "
    "pre-trigger threshold after del2 in units of "
    "dB/m\",\"type\":\"number\"},\"del1\":{\"description\":\"Length of constant "
    "post-trigger threshold elevation in units of m\",\"type\":\"number\"}}},\"de"
    "scription\":\"...\",\"maxItems\":128,\"type\":\"array\"},\"offset\":{\"description"
    "\":\"Offset in units of dB, for calculation of "
    "line\",\"type\":\"number\"},\"scale\":{\"description\":\"Scale in units of dB, "
    "for calculation of line: line number "
    "N=(amplitude-offset)/scale\",\"type\":\"number\"}}}"
    ,
    /* example: */
    "{\"post\":{\"a\":[78,86,126,134,31],\"nc\":128},\"pre\":{\"a\":[78,86,126,134,31"
    "],\"nc\":128},\"tab\":[{\"del2\":0.0,\"thra2\":0.0,\"dec1\":0.0,\"thra1\":0.0,\"dec"
    "2\":0.0,\"del1\":0.0}],\"offset\":0.0,\"scale\":1.0}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Echo Information\",\"required\":[\"echo_file\"],\"type\":\"object\",\""
    "properties\":{\"echo_file\":{\"required\":[\"file_extension\"],\"type\":\"object"
    "\",\"properties\":{\"file_uuid\":{\"description\":\"File's Universally Unique "
    "Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Echo file "
    "extension, without the leading dot\",\"type\":\"string\"}}}},\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\",\"description\":\"Details about "
    "echo files\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_uuid\":\"26a03615-67c0-4bea-8fe8-c577378fe661\",\"file"
    "_extension\":\"owp\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Exponential Decomposition\",\"type\":\"object\",\"patternPropertie"
    "s\":{\"^[0-9]+$\":{\"$ref\":\"#/definitions/channel\",\"description\":\"one "
    "field per channel, field name is channel index\"}},\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"additionalProperties\":false,\"descript"
    "ion\":\"Details for exponential decomposition of full waveform data\",\"de"
    "finitions\":{\"channel\":{\"required\":[\"delay\",\"scale\",\"parameter\"],\"type\""
    ":\"object\",\"properties\":{\"parameter\":{\"required\":[\"A\",\"B\",\"gamma\",\"omeg"
    "a\"],\"description\":\"parameters of the syswave exponential sum\",\"type\":\""
    "object\",\"properties\":{\"B\":{\"items\":{\"type\":\"number\"},\"description\":\"im"
    "aginary part of amplitude factor in units of full-scale\",\"type\":\"array"
    "\"},\"omega\":{\"items\":{\"type\":\"number\"},\"description\":\"angular frequency"
    " in Hz\",\"type\":\"array\"},\"gamma\":{\"items\":{\"type\":\"number\"},\"descriptio"
    "n\":\"decay in 1/second\",\"type\":\"array\"},\"A\":{\"items\":{\"type\":\"number\"},"
    "\"description\":\"real part of amplitude factor in units of full-scale\",\""
    "type\":\"array\"}}},\"a_lin\":{\"minimum\":0,\"type\":\"number\",\"exclusiveMinimu"
    "m\":false,\"exclusiveMaximum\":false,\"description\":\"relative linear "
    "amplitude range [0..1]\",\"maximum\":1},\"delay\":{\"description\":\"delay "
    "calibration in "
    "seconds\",\"type\":\"number\"},\"scale\":{\"description\":\"amplitude "
    "calibration\",\"type\":\"number\"}}}}}"
    ,
    /* example: */
    "{\"0\":{\"parameter\":{\"B\":[-3.9813032150268555,0.08622030913829803,-0.315"
    "2860999107361],\"omega\":[352020896.0,3647927552.0,-1977987072.0],\"gamma"
    "\":[-1094726528.0,-769562752.0,-848000064.0],\"A\":[0.9772450923919678,0."
    "3354335129261017,-1.312678575515747]},\"a_lin\":0.27,\"delay\":3.783458418"
    "887631e-09,\"scale\":1.0},\"1\":{\"parameter\":{\"B\":[-3.9,0.0,-0.3],\"omega\":"
    "[352020896.0,3647927552.0,-1977987072.0],\"gamma\":[-1094726528.0,-76956"
    "2752.0,-848000064.0],\"A\":[0.9,0.3,-1.3]},\"a_lin\":0.9,\"delay\":3.5e-09,\""
    "scale\":1.0}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Gaussian Decomposition\",\"required\":[\"amplitude_lsb_low_power"
    "\",\"amplitude_lsb_high_power\",\"amplitude_db\",\"range_offset_sec_low_powe"
    "r\",\"range_offset_sec_high_power\"],\"type\":\"object\",\"properties\":{\"range"
    "_offset_sec_high_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"am"
    "plitude_lsb_low_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"amp"
    "litude_db\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"range_offset_se"
    "c_low_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"},\"amplitude_lsb"
    "_high_power\":{\"items\":{\"type\":\"number\"},\"type\":\"array\"}},\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"description\":\"riegl.gaussian_d"
    "ecomposition contains information relevant for extracting calibrated "
    "amplitudes and calibrated ranges from a Gaussian decomposition of full"
    " waveform data. This information is contained in a table with five "
    "columns. Two columns are to be used as input: amplitude_lsb_low_power "
    "and amplitude_lsb_high_power. The other three columns provide the "
    "outputs. Amplitude_db gives the calibrated amplitude in the optical "
    "regime in decibels. The range offset columns provide additive range "
    "offsets, given in units of seconds, for each channel.\"}"
    ,
    /* example: */
    "{\"range_offset_sec_high_power\":[],\"amplitude_lsb_low_power\":[],\"amplit"
    "ude_db\":[],\"range_offset_sec_low_power\":[],\"amplitude_lsb_high_power\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Geo "
    "Tag\",\"description\":\"Point cloud georeferencing information\",\"type\":\"ob"
    "ject\",\"properties\":{\"crs\":{\"description\":\"Global Coordinate Reference "
    "System. Please note that only 3D Cartesian Coordinate Systems are allo"
    "wed.\",\"type\":\"object\",\"properties\":{\"wkt\":{\"description\":\"\\\"Well-Known"
    " Text\\\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta"
    "ndards/wkt-crs)\",\"type\":\"string\"},\"epsg\":{\"minimum\":0,\"description\":\"E"
    "PSG code\",\"type\":\"integer\"},\"name\":{\"description\":\"Coordinate "
    "reference system name\",\"type\":\"string\"}}},\"pose\":{\"minItems\":4,\"items\""
    ":{\"minItems\":4,\"items\":{\"description\":\"columns\",\"type\":\"number\"},\"desc"
    "ription\":\"rows\",\"type\":\"array\",\"maxItems\":4},\"description\":\"Coordinate"
    " Transformation Matrix to transform from File Coordinate System to "
    "Global Coordinate Reference System. 4x4 matrix stored as two "
    "dimensional array, row major order.\",\"type\":\"array\",\"maxItems\":4}}}"
    ,
    /* example: */
    "{\"crs\":{\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\""
    "WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHOR"
    "ITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000000000000,AUTHOR"
    "ITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.00000000000000000000,AUTHORIT"
    "Y[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AXIS[\\\"Y\\\",EAST],AXIS[\\\"Z\\\",NO"
    "RTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\",\"epsg\":4978,\"name\":\"WGS84 Geocentr"
    "ic\"},\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,"
    "4063882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847"
    ",1138787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196"
    "],[0.0,0.0,0.0,1.0]]}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"minimum\":0,\"desc"
    "ription\":\"Geometric scale factor applied to point "
    "coordinates\",\"type\":\"number\",\"exclusiveMinimum\":true}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Georefer"
    "encing Parameters\",\"description\":\"Parameters used for georeferencing "
    "of the point cloud\",\"type\":\"object\",\"properties\":{\"socs_to_rocs_matrix"
    "\":{\"minItems\":4,\"items\":{\"minItems\":4,\"items\":{\"description\":\"columns\""
    ",\"type\":\"number\"},\"description\":\"rows\",\"type\":\"array\",\"maxItems\":4},\"d"
    "escription\":\"Coordinate Transformation Matrix to transform from "
    "Scanner's Own Coordinate System to Record Coordinate System. 4x4 "
    "matrix stored as two dimensional array, row major order.\",\"type\":\"arra"
    "y\",\"maxItems\":4},\"trajectory_file\":{\"required\":[\"file_extension\"],\"des"
    "cription\":\"Trajectory data used for georeferencing of the point cloud\""
    ",\"type\":\"object\",\"properties\":{\"file_uuid\":{\"description\":\"File's "
    "Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Trajectory "
    "file extension, without the leading dot\",\"type\":\"string\"}}},\"socs_to_b"
    "ody_matrix\":{\"minItems\":4,\"items\":{\"minItems\":4,\"items\":{\"description\""
    ":\"columns\",\"type\":\"number\"},\"description\":\"rows\",\"type\":\"array\",\"maxIt"
    "ems\":4},\"description\":\"Coordinate Transformation Matrix to transform "
    "from Scanner's Own Coordinate System to Body Coordinate System. 4x4 "
    "matrix stored as two dimensional array, row major order.\",\"type\":\"arra"
    "y\",\"maxItems\":4},\"body_coordinate_system_type\":{\"description\":\"BODY "
    "coordinate frame (NED: North-East-Down, ENU: East-North-Up), default: "
    "NED\",\"type\":\"string\",\"enum\":[\"NED\",\"ENU\"]},\"trajectory_offsets\":{\"desc"
    "ription\":\"Correction offsets applied to the trajectory data\",\"type\":\"o"
    "bject\",\"properties\":{\"offset_north\":{\"description\":\"[m]\",\"type\":\"numbe"
    "r\"},\"version\":{\"description\":\"Meaning of offset values and how to "
    "apply them; version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\",\"type\":\"integer\"},\"offset_height\":{\"description\":\"["
    "m]\",\"type\":\"number\"},\"offset_east\":{\"description\":\"[m]\",\"type\":\"number"
    "\"},\"offset_yaw\":{\"description\":\"[deg]\",\"type\":\"number\"},\"offset_time\":"
    "{\"description\":\"[s]\",\"type\":\"number\"},\"offset_pitch\":{\"description\":\"["
    "deg]\",\"type\":\"number\"},\"offset_roll\":{\"description\":\"[deg]\",\"type\":\"nu"
    "mber\"}}}}}"
    ,
    /* example: */
    "{\"trajectory_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe660\""
    ",\"file_extension\":\"pofx\"},\"socs_to_body_matrix\":[[-0.269827776749716,-"
    "0.723017716139738,0.635954678449952,0.0],[0.962908599449764,-0.2026051"
    "7250352,0.178208229833847,0.0],[0.0,0.660451759194338,0.7508684796801,"
    "0.0],[0.0,0.0,0.0,1.0]],\"body_coordinate_system_type\":\"NED\",\"trajector"
    "y_offsets\":{\"offset_north\":0.07,\"version\":0,\"offset_height\":-0.2,\"offs"
    "et_east\":0.15,\"offset_yaw\":-0.45,\"offset_time\":18.007,\"offset_pitch\":0"
    ".01,\"offset_roll\":0.03}}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"GNSS "
    "Processing Information\",\"description\":\"Details about GNSS processing\","
    "\"type\":\"object\",\"properties\":{\"base_stations\":{\"items\":{\"required\":[\"l"
    "atitude\",\"longitude\",\"altitude\"],\"description\":\"GNSS base station deta"
    "ils\",\"type\":\"object\",\"properties\":{\"longitude\":{\"description\":\"Longitu"
    "de [deg]\",\"type\":\"number\"},\"latitude\":{\"description\":\"Latitude "
    "[deg]\",\"type\":\"number\"},\"altitude\":{\"description\":\"Altitude "
    "[m]\",\"type\":\"number\"}}},\"description\":\"List of GNSS base "
    "stations\",\"type\":\"array\"}}}"
    ,
    /* example: */
    "{\"base_stations\":[{\"longitude\":15.663389822,\"latitude\":48.658391162,\"a"
    "ltitude\":379.8004150390625}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Item Names\",\"type\":\"object\",\"$schema\":\"http://json-schema.or"
    "g/draft-04/schema#\",\"additionalProperties\":false,\"description\":\"Map of"
    " item names\",\"patternProperties\":{\"^-?[0-9]+$\":{\"description\":\"One "
    "field per item, field name is item id, field value is item "
    "name\",\"type\":\"string\"}}}"
    ,
    /* example: */
    "{\"-1\":\"Name of item with id -1\",\"47\":\"Name of item with id "
    "47\",\"2\":\"Name of item with id 2\",\"1\":\"Name of item with id 1\"}"
};

// Lag angle depended range and reflectance correction
const MetaDataItemInfo RDB_RIEGL_LAG_ANGLE_CORRECTION = {
    /* name        */ "riegl.lag_angle_correction",
    /* title       */ "Lag Angle Correction",
    /* description */ "Lag angle depended range and reflectance correction",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Lag Angle Correction\",\"required\":[\"ranges_m\",\"range_correcti"
    "on_m\",\"reflectance_correction_db\"],\"type\":\"object\",\"properties\":{\"rang"
    "e_correction_m\":{\"items\":{\"type\":\"number\"},\"description\":\"Range "
    "correction [m] as a function of the range and for applied scanrate, "
    "which defines the lag angle\",\"type\":\"array\"},\"reflectance_correction_d"
    "b\":{\"items\":{\"type\":\"number\"},\"description\":\"Reflectance correction "
    "[db] as a function of the range and for applied scanrate\",\"type\":\"arra"
    "y\"},\"ranges_m\":{\"items\":{\"type\":\"number\"},\"description\":\"Range [m]\",\"t"
    "ype\":\"array\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"de"
    "scription\":\"Lag angle depended range and reflectance correction\"}"
    ,
    /* example: */
    "{\"range_correction_m\":[0.0,0.0,0.0004,0.0023,0.0039,0.0053,0.0053],\"re"
    "flectance_correction_db\":[0.0,0.0,0.16,0.46,1.18,2.97,2.97],\"ranges_m\""
    ":[0.0,4.795,31.964,63.928,95.892,127.856,10000.0]}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Software License Keys\",\"type\":\"object\",\"$schema\":\"http://jso"
    "n-schema.org/draft-04/schema#\",\"additionalProperties\":false,\"descripti"
    "on\":\"License keys for software features\",\"patternProperties\":{\"^.*$\":{"
    "\"minItems\":1,\"items\":{\"description\":\"License key (example: '46AE032A -"
    " 39882AC4 - 9EC0A184 - "
    "6F163D73')\",\"type\":\"string\"},\"description\":\"Each field of the object "
    "represents a feature and holds a list of license keys, where the field"
    " name is the feature name.\",\"type\":\"array\"}}}"
    ,
    /* example: */
    "{\"MTA resolution\":[\"468E020A - 39A922E4 - B681A184 - "
    "673E3D72\"],\"Georeferencing\":[\"46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73\"],\"Full Waveform Analysis Topography\":[\"0FD5FF07 - 011A1255 -"
    " 9F76CACA - 8D2ED557\"],\"Full Waveform Analysis Topography with GPU "
    "support\":[\"8AB44126 - 23B92250 - 16E2689F - 34EF7E7B\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"MTA "
    "Settings\",\"description\":\"Parameters for MTA processing\",\"type\":\"object"
    "\",\"properties\":{\"zone_count\":{\"minimum\":0,\"description\":\"Maximum "
    "number of MTA zones.\",\"type\":\"integer\",\"maximum\":255},\"zone_width\":{\"m"
    "inimum\":0,\"description\":\"Width of a MTA zone in meter.\",\"type\":\"number"
    "\"},\"modulation_depth\":{\"minimum\":0,\"description\":\"Depth of pulse "
    "position modulation in meter.\",\"type\":\"number\"}}}"
    ,
    /* example: */
    "{\"zone_count\":23,\"zone_width\":149.896225,\"modulation_depth\":9.368514}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Near Range Correction Table\",\"required\":[\"delta\",\"content\"],"
    "\"type\":\"object\",\"properties\":{\"delta\":{\"description\":\"Delta between "
    "table entries [m], first entry is at range = 0 m\",\"type\":\"number\"},\"co"
    "ntent\":{\"minItems\":1,\"items\":{\"type\":\"number\"},\"description\":\"Correcti"
    "on value [m] to be added to the raw range\",\"type\":\"array\",\"maxItems\":2"
    "000}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"description"
    "\":\"Lookup table for range correction based on raw range\"}"
    ,
    /* example: */
    "{\"delta\":0.512,\"content\":[0.0]}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Noise Estimates\",\"required\":[\"amplitude\",\"range_sigma\",\"ampl"
    "itude_sigma\"],\"type\":\"object\",\"properties\":{\"range_sigma\":{\"items\":{\"t"
    "ype\":\"number\"},\"description\":\"Sigma range [m]\",\"type\":\"array\"},\"amplit"
    "ude\":{\"items\":{\"type\":\"number\"},\"description\":\"Amplitude [dB]\",\"type\":"
    "\"array\"},\"amplitude_sigma\":{\"items\":{\"type\":\"number\"},\"description\":\"S"
    "igma amplitude [dB]\",\"type\":\"array\"}},\"$schema\":\"http://json-schema.or"
    "g/draft-04/schema#\",\"description\":\"Standard deviation for range and "
    "amplitude as a function of amplitude\"}"
    ,
    /* example: */
    "{\"range_sigma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.018,"
    "0.016,0.014],\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0"
    "],\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0."
    "432,0.39,0.354]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Notch Filter\",\"required\":[\"range_minimum\",\"range_maximum\",\"a"
    "mplitude_maximum\"],\"type\":\"object\",\"properties\":{\"range_minimum\":{\"des"
    "cription\":\"Minimum range [m]\",\"type\":\"number\"},\"amplitude_maximum\":{\"m"
    "inimum\":0,\"description\":\"Maximum amplitude "
    "[dB]\",\"type\":\"number\"},\"range_maximum\":{\"description\":\"Maximum range ["
    "m]\",\"type\":\"number\"}},\"$schema\":\"http://json-schema.org/draft-04/schem"
    "a#\",\"description\":\"Notch filter parameters for window glass echoes\"}"
    ,
    /* example: */
    "{\"range_minimum\":-0.5,\"amplitude_maximum\":18.0,\"range_maximum\":0.2}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Pixel Information\",\"required\":[\"size\"],\"type\":\"object\",\"prop"
    "erties\":{\"size_llcs\":{\"$ref\":\"#/definitions/pixel_size\",\"description\":"
    "\"Size of pixels in a locally levelled cartesian coordinate system "
    "(xy). This is only used for pixels based on a map projection.\"},\"size\""
    ":{\"$ref\":\"#/definitions/pixel_size\",\"description\":\"Size of pixels in "
    "file coordinate system.\"}},\"$schema\":\"http://json-schema.org/draft-04/"
    "schema#\",\"description\":\"Details about the pixels contained in the file"
    "\",\"definitions\":{\"pixel_size\":{\"minItems\":2,\"items\":{\"minimum\":0,\"desc"
    "ription\":\"Length of pixel edge "
    "[m].\",\"type\":\"number\"},\"description\":\"Size of "
    "pixels.\",\"type\":\"array\",\"maxItems\":2}}}"
    ,
    /* example: */
    "{\"size_llcs\":[0.5156575252891171,0.5130835356683303],\"size\":[0.5971642"
    "834779395,0.5971642834779395]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Plane Patch Matching\",\"required\":[\"plane_patch_file_one\",\"pl"
    "ane_patch_file_two\"],\"type\":\"object\",\"properties\":{\"plane_patch_file_o"
    "ne\":{\"$ref\":\"#/definitions/file_reference\",\"description\":\"Reference to"
    " the plane patch file one\"},\"plane_patch_file_two\":{\"$ref\":\"#/definiti"
    "ons/file_reference\",\"description\":\"Reference to the plane patch file t"
    "wo\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"description"
    "\":\"Details about the plane patch matching process\",\"definitions\":{\"fil"
    "e_reference\":{\"required\":[\"file_uuid\",\"file_path\"],\"description\":\"Refe"
    "rence to a plane patch "
    "file\",\"type\":\"object\",\"properties\":{\"file_uuid\":{\"description\":\"Plane "
    "patch file's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_path\":{\"description\":\"Path of the plane "
    "patch file relative to the match file\",\"type\":\"string\"}}}}}"
    ,
    /* example: */
    "{\"plane_patch_file_one\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b"
    "4213\",\"file_path\":\"Record009_Line001/191025_121410_Scanner_1.ptch\"},\"p"
    "lane_patch_file_two\":{\"file_uuid\":\"fa47d509-a64e-49ce-8b14-ff3130fbefa"
    "9\",\"file_path\":\"project.ptch\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Plane "
    "Patch Statistics\",\"description\":\"Statistics about plane patches found "
    "by plane patch extractor\",\"type\":\"object\",\"properties\":{\"total_horizon"
    "tal_area\":{\"description\":\"sum of all plane patch areas projected to "
    "horizontal plane "
    "[m\\u00b2]\",\"type\":\"number\"},\"total_area\":{\"description\":\"sum of all "
    "plane patch areas [m\\u00b2]\",\"type\":\"number\"}}}"
    ,
    /* example: */
    "{\"total_horizontal_area\":13954.601,\"total_area\":14007.965}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Plane Slope Class Info\",\"required\":[\"settings\",\"classes\"],\"t"
    "ype\":\"object\",\"properties\":{\"classes\":{\"additionalProperties\":false,\"d"
    "escription\":\"Class definition table\",\"type\":\"object\",\"patternPropertie"
    "s\":{\"^[0-9]+$\":{\"description\":\"one field per class, field name is "
    "class number, field value is class name\",\"type\":\"string\"}}},\"settings\""
    ":{\"oneOf\":[{\"$ref\":\"#/definitions/method-1\"},{\"$ref\":\"#/definitions/me"
    "thod-2\"}],\"description\":\"Classification settings, details see "
    "documentation of rdbplanes\",\"type\":\"object\"}},\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"description\":\"Settings and classes for "
    "plane slope classification\",\"definitions\":{\"method-2\":{\"required\":[\"pl"
    "ane_classification_method\",\"sloping_plane_classes_minimum_angle\",\"slop"
    "ing_plane_classes_maximum_angle\"],\"description\":\"Classification method"
    " 2\",\"type\":\"object\",\"properties\":{\"plane_classification_method\":{\"mini"
    "mum\":2,\"description\":\"method ID (=2)\",\"type\":\"integer\",\"maximum\":2},\"s"
    "loping_plane_classes_maximum_angle\":{\"minimum\":-360.0,\"description\":\"m"
    "aximum inclination angle of sloping plane patches [deg]\",\"type\":\"numbe"
    "r\",\"maximum\":360.0},\"sloping_plane_classes_minimum_angle\":{\"minimum\":-"
    "360.0,\"description\":\"minimum inclination angle of sloping plane "
    "patches [deg]\",\"type\":\"number\",\"maximum\":360.0}}},\"method-1\":{\"require"
    "d\":[\"plane_classification_method\",\"maximum_inclination_angle_horizonta"
    "l\"],\"description\":\"Classification method 1\",\"type\":\"object\",\"propertie"
    "s\":{\"plane_classification_method\":{\"minimum\":1,\"description\":\"method "
    "ID (=1)\",\"type\":\"integer\",\"maximum\":1},\"maximum_inclination_angle_hori"
    "zontal\":{\"minimum\":-360.0,\"description\":\"maximum inclination angle of "
    "horizontal plane patches [deg]\",\"type\":\"number\",\"maximum\":360.0}}}}}"
    ,
    /* example: */
    "{\"classes\":{\"14\":\"horizontal, pointing down\",\"6\":\"vertical, pointing "
    "east\",\"7\":\"vertical, pointing south\",\"8\":\"vertical, pointing "
    "north\",\"12\":\"sloping, pointing down and north\",\"10\":\"sloping, pointing"
    " down and east\",\"3\":\"sloping, pointing up and south\",\"2\":\"sloping, "
    "pointing up and east\",\"4\":\"sloping, pointing up and "
    "north\",\"1\":\"horizontal, pointing up\",\"9\":\"vertical, pointing "
    "west\",\"13\":\"sloping, pointing down and west\",\"5\":\"sloping, pointing up"
    " and west\",\"11\":\"sloping, pointing down and south\"},\"settings\":{\"plane"
    "_classification_method\":2,\"sloping_plane_classes_maximum_angle\":70.0,\""
    "sloping_plane_classes_minimum_angle\":10.0}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Point Attribute Groups\",\"type\":\"object\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"additionalProperties\":false,\"descript"
    "ion\":\"Grouping and sorting of point attributes for visualization purpo"
    "ses\",\"patternProperties\":{\"^.*$\":{\"minItems\":1,\"items\":{\"description\":"
    "\"Point attribute full name or name pattern (perl regular expression "
    "syntax)\",\"type\":\"string\"},\"description\":\"Each field of the object "
    "represents a point attribute group and holds a list of point "
    "attributes, where the field name is the group name.\",\"type\":\"array\"}}}"
    ,
    /* example: */
    "{\"Primary Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.de"
    "viation\"],\"Time\":[\"riegl.timestamp\"],\"Coordinates/Vectors\":[\"riegl.xyz"
    "\",\"riegl.range\",\"riegl.theta\",\"riegl.phi\"],\"Secondary "
    "Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available\"],\"Other "
    "Attributes\":[\"riegl.selected\",\"riegl.visible\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Point "
    "Cloud Information\",\"description\":\"Details about point cloud "
    "files\",\"type\":\"object\",\"properties\":{\"project\":{\"description\":\"Project"
    " name\",\"type\":\"string\"},\"comments\":{\"description\":\"Comments\",\"type\":\"s"
    "tring\"},\"field_of_application\":{\"description\":\"Field of application\",\""
    "type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\",\""
    "BLS\",\"ILS\"]}}}"
    ,
    /* example: */
    "{\"project\":\"Campaign 4\",\"comments\":\"Line "
    "3\",\"field_of_application\":\"ALS\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Pose Estimation\",\"required\":[\"orientation\"],\"type\":\"object\","
    "\"properties\":{\"orientation\":{\"required\":[\"roll\",\"pitch\",\"yaw\",\"roll_ac"
    "curacy\",\"pitch_accuracy\",\"yaw_accuracy\"],\"description\":\"Orientation "
    "values and orientation accuracies, measured with IMU or inclination se"
    "nsors.\",\"type\":\"object\",\"properties\":{\"pitch_accuracy\":{\"minimum\":0,\"d"
    "escription\":\"Pitch angle accuracy [deg]\",\"type\":\"number\",\"exclusiveMin"
    "imum\":true},\"pitch\":{\"minimum\":-360,\"description\":\"Pitch angle about "
    "scanner Y-axis [deg]\",\"type\":\"number\",\"maximum\":360},\"yaw\":{\"minimum\":"
    "-360,\"description\":\"Yaw angle about scanner Z-axis [deg]\",\"type\":\"numb"
    "er\",\"maximum\":360},\"roll\":{\"minimum\":-360,\"description\":\"Roll angle "
    "about scanner X-axis [deg]\",\"type\":\"number\",\"maximum\":360},\"yaw_accura"
    "cy\":{\"minimum\":0,\"description\":\"Yaw angle accuracy [deg]\",\"type\":\"numb"
    "er\",\"exclusiveMinimum\":true},\"roll_accuracy\":{\"minimum\":0,\"description"
    "\":\"Roll angle accuracy [deg]\",\"type\":\"number\",\"exclusiveMinimum\":true}"
    "}},\"barometric_height_amsl\":{\"description\":\"Altitude determined based "
    "on the atmospheric pressure according to the standard atmosphere laws "
    "[m].\",\"type\":\"number\"},\"position\":{\"required\":[\"coordinate_1\",\"coordin"
    "ate_2\",\"coordinate_3\",\"horizontal_accuracy\",\"vertical_accuracy\",\"crs\"]"
    ",\"description\":\"Position coordinates and position accuracy values as "
    "measured by GNSS in the specified Coordinate Reference System (CRS)\",\""
    "type\":\"object\",\"properties\":{\"coordinate_3\":{\"description\":\"Coordinate"
    " 3 as defined by axis 3 of the specified CRS (e.g., Z, "
    "Altitude)\",\"type\":\"number\"},\"coordinate_2\":{\"description\":\"Coordinate "
    "2 as defined by axis 2 of the specified CRS (e.g., Y, Longitude)\",\"typ"
    "e\":\"number\"},\"horizontal_accuracy\":{\"minimum\":0,\"description\":\"Horizon"
    "tal accuracy [m]\",\"type\":\"number\",\"exclusiveMinimum\":true},\"crs\":{\"req"
    "uired\":[\"epsg\"],\"description\":\"Global Coordinate Reference System\",\"ty"
    "pe\":\"object\",\"properties\":{\"wkt\":{\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\",\"type\":\"string\"},\"epsg\":{\"minimum\":0,\"description\":\"EPSG code"
    "\",\"type\":\"integer\"}}},\"vertical_accuracy\":{\"minimum\":0,\"description\":\""
    "Vertical accuracy [m]\",\"type\":\"number\",\"exclusiveMinimum\":true},\"coord"
    "inate_1\":{\"description\":\"Coordinate 1 as defined by axis 1 of the "
    "specified CRS (e.g., X, Latitude)\",\"type\":\"number\"}}}},\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"description\":\"Estimated position"
    " and orientation information as measured by GNSS, IMU or inclination "
    "sensors\"}"
    ,
    /* example: */
    "{\"orientation\":{\"pitch_accuracy\":0.009433783936875745,\"pitch\":1.509153"
    "024827064,\"yaw\":101.87293630292045,\"roll\":3.14743073066123,\"yaw_accura"
    "cy\":1.0094337839368757,\"roll_accuracy\":0.009433783936875745},\"barometr"
    "ic_height_amsl\":386.7457796227932,\"position\":{\"coordinate_3\":362.71249"
    "38964844,\"coordinate_2\":15.645033406,\"horizontal_accuracy\":0.810699999"
    "332428,\"crs\":{\"wkt\":\"GEOGCS[\\\"WGS84 / Geographic\\\",DATUM[\\\"WGS84\\\",SPH"
    "EROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]"
    "],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.000000000000000"
    "0,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Degree\\\",0.01745329251994329547"
    ",AUTHORITY[\\\"EPSG\\\",\\\"9102\\\"]],AXIS[\\\"Latitude\\\",NORTH],AXIS[\\\"Longitu"
    "de\\\",EAST],AUTHORITY[\\\"EPSG\\\",\\\"4979\\\"]]\",\"epsg\":4979},\"vertical_accur"
    "acy\":1.3314999341964722,\"coordinate_1\":48.655799473}}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Pose Sensors\",\"required\":[\"gyroscope\",\"accelerometer\",\"magne"
    "tic_field_sensor\"],\"type\":\"object\",\"properties\":{\"gyroscope\":{\"require"
    "d\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_nonl"
    "inearity\"],\"description\":\"Gyroscope details\",\"type\":\"object\",\"properti"
    "es\":{\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive "
    "X axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":"
    "\"#/definitions/vector\",\"description\":\"Relative nonlinearity, dimension"
    "less\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to"
    " be subtracted from raw measurement "
    "values\"},\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor"
    " origin in SOCS [m] at frame angle = "
    "0\"},\"unit\":{\"minimum\":0,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in rad/s\",\"type\":\"number\",\"exclusiveMinimum\""
    ":true},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitiv"
    "e Z axis of sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\"}}},\"magnetic_field_sensor\":{\"requi"
    "red\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"fixed\",\"relative_non"
    "linearity\"],\"description\":\"Magnetic Field Sensor details\",\"type\":\"obje"
    "ct\",\"properties\":{\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description"
    "\":\"Sensitive X axis of sensor at frame angle = 0\"},\"relative_nonlinear"
    "ity\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Relative "
    "nonlinearity, dimensionless\"},\"fixed\":{\"$ref\":\"#/definitions/vector\",\""
    "description\":\"Distortion of magnetic field caused by non-rotating "
    "scanner "
    "part\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to"
    " be subtracted from raw measurement "
    "values\"},\"unit\":{\"minimum\":0,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in nT\",\"type\":\"number\",\"exclusiveMinimum\":tr"
    "ue},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z"
    " axis of sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\"}}},\"accelerometer\":{\"required\":[\"u"
    "nit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_nonlineari"
    "ty\"],\"description\":\"Accelerometer details\",\"type\":\"object\",\"properties"
    "\":{\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive X "
    "axis of sensor at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\"#"
    "/definitions/vector\",\"description\":\"Relative nonlinearity, dimensionle"
    "ss\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to "
    "be subtracted from raw measurement "
    "values\"},\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor"
    " origin in SOCS [m] at frame angle = "
    "0\"},\"unit\":{\"minimum\":0,\"description\":\"Unit of raw data and "
    "calibration values, 1 LSB in 9.81 m/s\\u00b2\",\"type\":\"number\",\"exclusiv"
    "eMinimum\":true},\"z_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":"
    "\"Sensitive Z axis of sensor at frame angle = "
    "0\"},\"y_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Y"
    " axis of sensor at frame angle = 0\"}}}},\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\",\"description\":\"Details on position and "
    "orientation sensors\",\"definitions\":{\"vector\":{\"minItems\":3,\"items\":{\"d"
    "escription\":\"Index 0=X, 1=Y, 2=Z "
    "component\",\"type\":\"number\"},\"maxItems\":3,\"type\":\"array\"}}}"
    ,
    /* example: */
    "{\"gyroscope\":{\"x_axis\":[-121.195556640625,0.8219714164733887,0.2313031"
    "703233719],\"relative_nonlinearity\":[2.888176311444113e-07,1.0627416457"
    "9645e-07,-1.7186295080634935e-39],\"offset\":[-50.92609786987305,146.156"
    "43310546875,62.4327278137207],\"origin\":[0.026900000870227814,-0.039999"
    "99910593033,-0.08950000256299973],\"unit\":0.00014544410805683583,\"z_axi"
    "s\":[0.555869996547699,119.22135162353516,0.467585027217865],\"y_axis\":["
    "-0.440765917301178,-0.7897399663925171,119.5894775390625]},\"magnetic_f"
    "ield_sensor\":{\"x_axis\":[-0.011162743903696537,-2.315962774446234e-05,0"
    ".00016818844596855342],\"relative_nonlinearity\":[0.0,0.0,0.0],\"fixed\":["
    "-1576.010498046875,1596.081787109375,0.0],\"offset\":[-23812.052734375,5"
    "606.57666015625,2493.28125],\"unit\":91.74311828613281,\"z_axis\":[0.00041"
    "987866279669106,7.876977906562388e-05,0.011407104320824146],\"y_axis\":["
    "0.00027888521435670555,-0.011427424848079681,-5.204829722060822e-05]},"
    "\"accelerometer\":{\"x_axis\":[-15008.123046875,56.956390380859375,-60.517"
    "5666809082],\"relative_nonlinearity\":[0.0,0.0,0.0],\"offset\":[-733.36364"
    "74609375,58.969032287597656,1060.2550048828125],\"origin\":[0.0269000008"
    "70227814,-0.03999999910593033,-0.08950000256299973],\"unit\":6.666666740"
    "784422e-05,\"z_axis\":[1.639882206916809,15166.744140625,-116.9974288940"
    "4297],\"y_axis\":[-7.027288913726807,-44.12333679199219,14952.3701171875"
    "]}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Pulse Position Modulation\",\"required\":[\"length\",\"num_mod_amp"
    "l\",\"pulse_interval\"],\"type\":\"object\",\"properties\":{\"num_mod_ampl\":{\"mi"
    "nimum\":0,\"description\":\"Number of different modulation amplitudes (2: "
    "binary modulation)\",\"type\":\"integer\",\"maximum\":255},\"code_phase_mode\":"
    "{\"minimum\":0,\"description\":\"0: no synchronization, 1: toggle between 2"
    " phases, 2: increment with phase_increment\",\"type\":\"integer\",\"maximum\""
    ":255},\"phase_step\":{\"minimum\":0,\"description\":\"Step width in phase of "
    "modulation code from line to line\",\"type\":\"integer\",\"maximum\":255},\"le"
    "ngth\":{\"minimum\":0,\"description\":\"Length of code\",\"type\":\"integer\",\"ma"
    "ximum\":255},\"pulse_interval\":{\"items\":{\"minimum\":0,\"type\":\"number\"},\"d"
    "escription\":\"Explicit table of the pulse position modulation used for "
    "MTA resolution. Table gives times between successive laser pulses in s"
    "econds.\",\"type\":\"array\"}},\"$schema\":\"http://json-schema.org/draft-04/s"
    "chema#\",\"description\":\"Laser pulse position modulation used for MTA "
    "resolution\"}"
    ,
    /* example: */
    "{\"num_mod_ampl\":2,\"code_phase_mode\":2,\"phase_step\":5,\"length\":31,\"puls"
    "e_interval\":[2.759375e-06,2.759375e-06,2.759375e-06,2.759375e-06,2.821"
    "875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.821875e-"
    "06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-06,2."
    "821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.75937"
    "5e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.821875e-06"
    ",2.821875e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.821875e-06]}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Range Statistics\",\"required\":[\"minimum\",\"average\",\"maximum\","
    "\"std_dev\"],\"type\":\"object\",\"properties\":{\"std_dev\":{\"description\":\"Sta"
    "ndard deviation\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum "
    "value\",\"type\":\"number\"},\"average\":{\"description\":\"Average "
    "value\",\"type\":\"number\"},\"maximum\":{\"description\":\"Maximum value\",\"type"
    "\":\"number\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"desc"
    "ription\":\"Statistics about target distance wrt. SOCS origin\"}"
    ,
    /* example: */
    "{\"std_dev\":24.349,\"minimum\":0.919,\"average\":15.49738,\"maximum\":574.35}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Receiver Internals\",\"type\":\"object\",\"properties\":{\"nt\":{\"min"
    "imum\":0,\"description\":\"Number of traces\",\"type\":\"integer\",\"maximum\":25"
    "5},\"tbl\":{\"minItems\":1,\"items\":{\"$ref\":\"#/definitions/fp_table\"},\"desc"
    "ription\":\"various internal "
    "data\",\"type\":\"array\"},\"sr\":{\"minimum\":0,\"description\":\"Sample rate [Hz"
    "]\",\"type\":\"number\",\"exclusiveMinimum\":true},\"t\":{\"additionalProperties"
    "\":false,\"type\":\"object\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/def"
    "initions/fp\",\"description\":\"one field per channel, field name is "
    "channel index\"}}},\"ns\":{\"minimum\":0,\"description\":\"Number of samples\","
    "\"type\":\"integer\",\"maximum\":4095},\"ex\":{\"description\":\"DEPRECATED, use "
    "'riegl.exponential_decomposition' "
    "instead\",\"type\":\"object\"},\"mw\":{\"minimum\":0,\"description\":\"Maximum wei"
    "ght\",\"type\":\"number\",\"maximum\":4095,\"exclusiveMinimum\":true},\"a\":{\"min"
    "Items\":1,\"items\":{\"type\":\"number\"},\"description\":\"Amplitude [dB]\",\"max"
    "Items\":256,\"type\":\"array\"},\"si\":{\"minimum\":0,\"description\":\"Start "
    "index (hw_start)\",\"type\":\"number\",\"maximum\":255}},\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"description\":\"Receiver "
    "Internals\",\"definitions\":{\"fp_table\":{\"desription\":\"scanner internal d"
    "ata\",\"required\":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv\"],\"type\":\"object\",\"properties"
    "\":{\"ny\":{\"min\":1,\"max\":2048,\"description\":\"number of y entries\",\"type\""
    ":\"integer\"},\"tc\":{\"min\":0,\"max\":255,\"description\":\"table data type "
    "code\",\"type\":\"integer\"},\"nx\":{\"min\":1,\"max\":2048,\"description\":\"number"
    " of x entries\",\"type\":\"integer\"},\"ch\":{\"min\":0,\"max\":255,\"description\""
    ":\"channel number\",\"type\":\"integer\"},\"tv\":{\"minItems\":1,\"maxItems\":2048"
    ",\"type\":\"array\",\"items\":{\"oneOf\":[{\"$ref\":\"#/definitions/fp_table_row\""
    "},{\"type\":\"number\"}]}}}},\"fp_table_row\":{\"minItems\":1,\"maxItems\":2048,"
    "\"type\":\"array\",\"items\":{\"type\":\"number\"}},\"fp\":{\"required\":[\"s\",\"w\"],\""
    "description\":\"Fingerprint values\",\"type\":\"object\",\"properties\":{\"s\":{\""
    "minItems\":1,\"items\":{\"minItems\":1,\"items\":{\"type\":\"number\"},\"maxItems\""
    ":4096,\"type\":\"array\"},\"maxItems\":256,\"type\":\"array\"},\"w\":{\"minItems\":1"
    ",\"items\":{\"minItems\":5,\"items\":{\"type\":\"number\"},\"maxItems\":5,\"type\":\""
    "array\"},\"maxItems\":256,\"type\":\"array\"}}}}}"
    ,
    /* example: */
    "{\"nt\":128,\"tbl\":[{\"ny\":2,\"tc\":1,\"nx\":5,\"ch\":0,\"tv\":[[1,2,3,4,5],[1.1,2"
    ".2,3.3,4.4,5.5]]}],\"sr\":7959997000.0,\"t\":{\"0\":{\"s\":[[1.23,4.56],[7.89,"
    "0.12]],\"w\":[[78,86,126,134,31],[78,86,126,134,31]]},\"1\":{\"s\":[[1.23,4."
    "56],[7.89,0.12]],\"w\":[[78,86,126,134,31],[78,86,126,134,31]]}},\"ns\":40"
    "0,\"mw\":31,\"a\":[-1.55],\"si\":48}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Reflectance Calculation Table\",\"required\":[\"delta\",\"content\""
    "],\"type\":\"object\",\"properties\":{\"delta\":{\"description\":\"Delta between "
    "table entries [m], first entry is at range = 0 m\",\"type\":\"number\"},\"co"
    "ntent\":{\"minItems\":1,\"items\":{\"type\":\"number\"},\"description\":\"Correcti"
    "on value [dB] to be added to the amplitude\",\"type\":\"array\",\"maxItems\":"
    "2000}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"descriptio"
    "n\":\"Lookup table for reflectance calculation based on amplitude and "
    "range\"}"
    ,
    /* example: */
    "{\"delta\":0.150918,\"content\":[-33.01]}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Near-range reflectance correction\",\"required\":[\"ranges_m\",\"l"
    "ine_angles_deg\",\"reflectance_correction_db\"],\"type\":\"object\",\"properti"
    "es\":{\"reflectance_correction_db\":{\"items\":{\"items\":{\"description\":\"col"
    "umns (each value corresponds to an "
    "angle)\",\"type\":\"number\"},\"description\":\"rows (each array corresponds "
    "to a range)\",\"type\":\"array\"},\"description\":\"Near range reflectance "
    "correction in dB as a function of range over angle\",\"type\":\"array\"},\"l"
    "ine_angles_deg\":{\"items\":{\"type\":\"number\"},\"description\":\"Angle [deg]\""
    ",\"type\":\"array\"},\"ranges_m\":{\"items\":{\"type\":\"number\"},\"description\":\""
    "Range [m]\",\"type\":\"array\"}},\"$schema\":\"http://json-schema.org/draft-04"
    "/schema#\",\"description\":\"Range-dependent and scan-angle-dependent "
    "correction of reflectance reading\"}"
    ,
    /* example: */
    "{\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0."
    "01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0"
    ".3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01]],\"li"
    "ne_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2.0,2.5,3.0,3.5,4.0],\"ranges_m\":[0"
    ".0,1.0,2.0,3.0]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"RiLOC Processing Information\",\"required\":[\"trajectory\",\"opti"
    "mizer\",\"georeferencing\"],\"type\":\"object\",\"properties\":{\"optimizer\":{\"d"
    "escription\":\"Optimization result information and statistics.\",\"type\":\""
    "object\",\"properties\":{}},\"planes\":{\"description\":\"Information on plane"
    " correspondences and used extraction/matching templates.\",\"type\":\"obje"
    "ct\",\"properties\":{}},\"trajectory\":{\"description\":\"Trajectory "
    "information and statistics.\",\"type\":\"object\",\"properties\":{}},\"georefe"
    "rencing\":{\"description\":\"Internal georeferencing parameters.\",\"type\":\""
    "object\",\"properties\":{}}},\"$schema\":\"http://json-schema.org/draft-04/s"
    "chema#\",\"description\":\"Details about RiLOC processing\"}"
    ,
    /* example: */
    "{\"optimizer\":{},\"planes\":{},\"trajectory\":{},\"georeferencing\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Scan "
    "Pattern\",\"description\":\"Scan pattern description\",\"properties\":{\"recta"
    "ngular\":{\"required\":[\"phi_start\",\"phi_stop\",\"phi_increment\",\"theta_sta"
    "rt\",\"theta_stop\",\"theta_increment\"],\"description\":\"Rectangular Field "
    "Of View Scan Pattern\",\"type\":\"object\",\"properties\":{\"theta_increment\":"
    "{\"minimum\":0.0,\"description\":\"Increment of theta angle in SOCS [deg]\","
    "\"type\":\"number\",\"maximum\":90.0},\"phi_start\":{\"minimum\":0.0,\"descriptio"
    "n\":\"Start phi angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":360.0},\"p"
    "hi_increment\":{\"minimum\":0.0,\"description\":\"Increment of phi angle in "
    "SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0},\"program\":{\"$ref\":\"#/defin"
    "itions/program\"},\"theta_stop\":{\"minimum\":0.0,\"description\":\"Stop theta"
    " angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":180.0},\"phi_stop\":{\"mi"
    "nimum\":0.0,\"description\":\"Stop phi angle in SOCS [deg]\",\"type\":\"number"
    "\",\"maximum\":720.0},\"theta_start\":{\"minimum\":0.0,\"description\":\"Start "
    "theta angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":180.0}}},\"line\":{"
    "\"required\":[\"start\",\"stop\",\"increment\"],\"description\":\"Line Scan Patte"
    "rn\",\"type\":\"object\",\"properties\":{\"start\":{\"minimum\":0.0,\"description\""
    ":\"Start angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":360.0},\"program"
    "\":{\"$ref\":\"#/definitions/program\"},\"stop\":{\"minimum\":0.0,\"description\""
    ":\"Stop angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":720.0},\"incremen"
    "t\":{\"minimum\":0.0,\"description\":\"Increment of angle in SOCS [deg]\",\"ty"
    "pe\":\"number\",\"maximum\":90.0}}},\"segments\":{\"required\":[\"list\"],\"descri"
    "ption\":\"Segmented Line Scan Pattern\",\"type\":\"object\",\"properties\":{\"pr"
    "ogram\":{\"$ref\":\"#/definitions/program\"},\"list\":{\"items\":{\"required\":[\""
    "start\",\"stop\",\"increment\"],\"description\":\"Line Scan Segment\",\"type\":\"o"
    "bject\",\"properties\":{\"start\":{\"minimum\":0.0,\"description\":\"Start angle"
    " in SOCS [deg]\",\"type\":\"number\",\"maximum\":360.0},\"stop\":{\"minimum\":0.0"
    ",\"description\":\"Stop angle in SOCS [deg]\",\"type\":\"number\",\"maximum\":72"
    "0.0},\"increment\":{\"minimum\":0.0,\"description\":\"Increment of angle in "
    "SOCS [deg]\",\"type\":\"number\",\"maximum\":90.0}}},\"type\":\"array\"}}}},\"defi"
    "nitions\":{\"program\":{\"required\":[\"name\"],\"description\":\"Measurement pr"
    "ogram\",\"type\":\"object\",\"properties\":{\"laser_prr\":{\"minimum\":0,\"descrip"
    "tion\":\"Laser Pulse Repetition Rate [Hz]\",\"type\":\"number\",\"exclusiveMin"
    "imum\":false},\"name\":{\"description\":\"Name of measurement "
    "program\",\"type\":\"string\"}}}}}"
    ,
    /* example: */
    "{\"rectangular\":{\"theta_increment\":0.04,\"phi_start\":45.0,\"phi_increment"
    "\":0.04,\"program\":{\"laser_prr\":100000.0,\"name\":\"High "
    "Speed\"},\"theta_stop\":130.0,\"phi_stop\":270.0,\"theta_start\":30.0}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Shot "
    "Information\",\"description\":\"Details about laser shot files\",\"type\":\"ob"
    "ject\",\"properties\":{\"shot_file\":{\"required\":[\"file_extension\"],\"type\":"
    "\"object\",\"properties\":{\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Shot file "
    "extension, without the leading dot\",\"type\":\"string\"}}}}}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_uuid\":\"26a00815-67c0-4bff-8fe8-c577378fe663\",\"file"
    "_extension\":\"sodx\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Stored filters\",\"required\":[\"activated\",\"filters\"],\"type\":\"o"
    "bject\",\"properties\":{\"filters\":{\"items\":{\"required\":[\"activated\",\"titl"
    "e\",\"description\",\"filter\"],\"description\":\"Point filter definition\",\"ty"
    "pe\":\"object\",\"properties\":{\"description\":{\"description\":\"A brief "
    "description of the filter (e.g. for display in a "
    "tooltip)\",\"type\":\"string\"},\"title\":{\"description\":\"A short filter "
    "title (e.g. for display in a selection "
    "list)\",\"type\":\"string\"},\"activated\":{\"description\":\"Apply ('true') or "
    "ignore ('false') this "
    "filter\",\"type\":\"boolean\"},\"filter\":{\"description\":\"The RDB filter "
    "string to apply (e.g. when reading points or index), details see "
    "documentation of function "
    "select()\",\"type\":\"string\"}}},\"description\":\"List of point "
    "filters\",\"type\":\"array\"},\"activated\":{\"description\":\"Apply ('true') or"
    " ignore ('false') all filters\",\"type\":\"boolean\"}},\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"description\":\"Point filters applied "
    "in addition to the application-defined filters\"}"
    ,
    /* example: */
    "{\"filters\":[{\"description\":\"Ignore points with uncertain MTA zone "
    "assignment\",\"title\":\"Ignore uncertain points from MTA "
    "resolution\",\"activated\":false,\"filter\":\"riegl.mta_uncertain_point == "
    "0\"},{\"description\":\"Ignore points with an echo signal amplitude below "
    "the dynamic trigger\",\"title\":\"Ignore points below dynamic trigger "
    "level\",\"activated\":false,\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\"},{\"description\":\"Ignore points that are outside the outer AABB in "
    "BOCS\",\"title\":\"Ignore points outside outer "
    "AABB\",\"activated\":false,\"filter\":\"riegl.point_outside_aabb_bocs == "
    "0\"},{\"description\":\"Ignore points that are inside the inner AABB in "
    "BOCS\",\"title\":\"Ignore points inside inner "
    "AABB\",\"activated\":false,\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\"}],\"activated\":true}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"System Description\",\"required\":[\"$class\",\"version\",\"author\","
    "\"timestamp\",\"system\"],\"type\":\"object\",\"properties\":{\"timestamp\":{\"desc"
    "ription\":\"Date and time of creation (e.g. 2019-06-19T13:32:10+02:00)\","
    "\"type\":\"string\"},\"$class\":{\"description\":\"Object class name\",\"type\":\"s"
    "tring\",\"enum\":[\"Document\"]},\"version\":{\"description\":\"Document format "
    "version\",\"type\":\"string\"},\"author\":{\"description\":\"Name of the author "
    "that created the "
    "document\",\"type\":\"string\"},\"system\":{\"description\":\"The actual system "
    "description, details see documentation of RIEGL System Description\",\"t"
    "ype\":\"object\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"d"
    "escription\":\"Details of major system components like lidar sensors, "
    "cameras and navigation devices\"}"
    ,
    /* example: */
    "{\"timestamp\":\"2022-09-30T11:56:26+00:00\",\"$class\":\"Document\",\"version\""
    ":\"1.2.1\",\"author\":\"RIEGL LMS GmbH, Austria\",\"system\":{\"details\":\"see "
    "documentation of RIEGL System Description\"}}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Temperature Calculation Table\",\"type\":\"object\",\"properties\":"
    "{\"InGaAs\":{\"$ref\":\"#/definitions/conversion_table\",\"description\":\"Conv"
    "ersion table for InGaAs channel\"},\"Si\":{\"$ref\":\"#/definitions/conversi"
    "on_table\",\"description\":\"Conversion table for Si channel\"},\"InGaAs_Si_"
    "Difference\":{\"$ref\":\"#/definitions/conversion_table\",\"description\":\"Co"
    "nversion table for InGaAs - Si difference\"}},\"$schema\":\"http://json-sc"
    "hema.org/draft-04/schema#\",\"description\":\"Conversion of background "
    "radiation raw values to temperatures in \\u00b0C\",\"definitions\":{\"conve"
    "rsion_table\":{\"required\":[\"value\",\"temperature\"],\"type\":\"object\",\"prop"
    "erties\":{\"temperature\":{\"items\":{\"type\":\"number\"},\"description\":\"Tempe"
    "rature [\\u00b0C]\",\"type\":\"array\"},\"value\":{\"items\":{\"type\":\"number\"},\""
    "description\":\"LSB [1]\",\"type\":\"array\"}}}}}"
    ,
    /* example: */
    "{\"InGaAs\":{\"temperature\":[307.22196722535614,309.1153478247277,311.118"
    "8086915047,313.10025350480055,315.2137946389828,317.2172555057597,319."
    "2207163725366,321.2021611858325,323.3157023200148],\"value\":[0.0,64.000"
    "97659230323,128.0019531846065,192.0029297769097,256.0039063692129,320."
    "00488296151616,384.0058595538194,448.0068361461226,512.0078127384259]}"
    ",\"Si\":{\"temperature\":[546.300048828125,548.8164051212026,551.314393850"
    "0972,554.0144257850053,556.604252334815,559.2124464488079,561.80227299"
    "86177,564.4104671126105,567.0002936624203],\"value\":[0.0,64.00097659230"
    "323,128.0019531846065,192.0029297769097,256.0039063692129,320.00488296"
    "151616,384.0058595538194,448.0068361461226,512.0078127384259]},\"InGaAs"
    "_Si_Difference\":{\"temperature\":[1749.977111117893,1749.977111117893,17"
    "49.977111117893,1749.977111117893,1749.977111117893,1749.977111117893,"
    "1744.7813348796044,1681.9971312601092,1622.3944822534868],\"value\":[100"
    "0.0,1100.090029602954,1200.04425183874,1300.1342814416948,1400.0885036"
    "774805,1500.0427259132668,1600.1327555162209,1700.0869777520065,1800.0"
    "411999877924]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Time Base\",\"required\":[\"epoch\",\"source\"],\"type\":\"object\",\"pr"
    "operties\":{\"source\":{\"description\":\"Timestamp source\",\"type\":\"string\","
    "\"enum\":[\"unknown\",\"RTC\",\"GNSS\"]},\"epoch\":{\"description\":\"Date and time"
    " of timestamp '0' as proposed by RFC 3339 (e.g. 2015-10-27T00:00:00+01"
    ":00).\",\"type\":\"string\"},\"system\":{\"description\":\"Time system (time sta"
    "ndard)\",\"type\":\"string\",\"enum\":[\"unknown\",\"UTC\",\"GPS\"]}},\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"description\":\"Base of "
    "timestamps (epoch)\"}"
    ,
    /* example: */
    "{\"source\":\"GNSS\",\"epoch\":\"2015-10-27T00:00:00+00:00\",\"system\":\"UTC\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Trajectory Information\",\"required\":[\"time_interval\",\"navigat"
    "ion_frame\"],\"type\":\"object\",\"properties\":{\"project\":{\"description\":\"Pr"
    "oject name\",\"type\":\"string\"},\"time_interval\":{\"required\":[\"minimum\",\"a"
    "verage\",\"maximum\",\"std_dev\"],\"description\":\"Time interval statistics\","
    "\"type\":\"object\",\"properties\":{\"std_dev\":{\"description\":\"Standard "
    "deviation of intervals "
    "[s]\",\"type\":\"number\"},\"minimum\":{\"description\":\"Minimum time interval "
    "[s]\",\"type\":\"number\"},\"average\":{\"description\":\"Average time interval "
    "[s]\",\"type\":\"number\"},\"maximum\":{\"description\":\"Maximum time interval "
    "[s]\",\"type\":\"number\"}}},\"software\":{\"description\":\"Software that "
    "calculated the trajectory (this may be the same or different software "
    "than the one that created the "
    "file)\",\"type\":\"string\"},\"company\":{\"description\":\"Company "
    "name\",\"type\":\"string\"},\"location\":{\"description\":\"Project location, "
    "e.g. city/state/country\",\"type\":\"string\"},\"device\":{\"description\":\"Nav"
    "igation device, e.g. "
    "name/type/serial\",\"type\":\"string\"},\"settings\":{\"description\":\"Settings"
    " used to calculate the trajectory (descriptive "
    "text)\",\"type\":\"string\"},\"navigation_frame\":{\"description\":\"Navigation "
    "frame (NED: North-East-Down, ENU: East-North-Up)\",\"type\":\"string\",\"enu"
    "m\":[\"unknown\",\"NED\",\"ENU\"]},\"field_of_application\":{\"description\":\"Fie"
    "ld of application\",\"type\":\"string\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\""
    ",\"MLS\",\"ULS\",\"ALS\",\"BLS\",\"ILS\"]}},\"$schema\":\"http://json-schema.org/dr"
    "aft-04/schema#\",\"description\":\"Details about position+orientation "
    "files\"}"
    ,
    /* example: */
    "{\"project\":\"Campaign 3\",\"time_interval\":{\"std_dev\":5.51e-07,\"minimum\":"
    "0.00500032,\"average\":0.00500053,\"maximum\":0.005004883},\"software\":\"Nav"
    "igation Software XYZ\",\"company\":\"RIEGL "
    "LMS\",\"location\":\"Horn\",\"device\":\"IMU Model 12/1, Serial# 12345\",\"setti"
    "ngs\":\"default\",\"navigation_frame\":\"NED\",\"field_of_application\":\"MLS\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"TTIP Configuration\",\"required\":[\"ttip_version\",\"num_channel\""
    ",\"ext_channel\",\"ext_signal\",\"ext_delay\",\"ext_subdivider\",\"ext_bitmask\""
    ",\"dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\",\"channel"
    "\",\"output_descr\",\"input_descr\",\"output_usage\",\"input_usage\",\"out_polar"
    "ity\",\"out_duration\",\"out_interval\",\"out_delay\",\"out_num_pulses\",\"in_po"
    "larity\",\"in_min_duration\",\"in_max_duration\",\"in_max_delay\"],\"type\":\"ob"
    "ject\",\"properties\":{\"input_descr\":{\"description\":\"descriptive string\","
    "\"type\":\"string\"},\"out_num_pulses\":{\"minimum\":0,\"description\":\"number "
    "of output pulses to be generated, 0 .. infinite\",\"type\":\"integer\"},\"in"
    "_max_duration\":{\"minimum\":0,\"description\":\"stops measurement of pulse "
    "duration of input signal [0.1 "
    "msec]\",\"type\":\"integer\"},\"output_descr\":{\"description\":\"descriptive "
    "string\",\"type\":\"string\"},\"ext_signal\":{\"minimum\":0,\"description\":\"0 .."
    " use input signal, 1 .. use output signal of channel\",\"type\":\"integer\""
    "},\"ext_delay\":{\"minimum\":0,\"description\":\"external trigger, time delay"
    " [0.1 "
    "msec]\",\"type\":\"integer\"},\"out_polarity\":{\"minimum\":0,\"description\":\"0 "
    ".. positive edge, 1 .. negative "
    "edge\",\"type\":\"integer\"},\"in_polarity\":{\"minimum\":0,\"description\":\"0 .."
    " positive edge, 1 .. negative edge\",\"type\":\"integer\"},\"in_max_delay\":{"
    "\"minimum\":0,\"description\":\"maximum delay to output pulse before fake "
    "event is generated [0.1 msec], zero indicates that no fake events are "
    "generated\",\"type\":\"integer\"},\"dmi_incr\":{\"description\":\"dmi, increment"
    " in ticks\",\"type\":\"integer\"},\"ext_bitmask\":{\"minimum\":0,\"description\":"
    "\"defines which of the internal pulse generators are to be started\",\"ty"
    "pe\":\"integer\"},\"in_min_duration\":{\"minimum\":0,\"description\":\"input "
    "signals with smaller pulse durations are ignored [0.1 "
    "msec]\",\"type\":\"integer\"},\"ext_channel\":{\"minimum\":0,\"description\":\"ID "
    "of channel used as external trigger input, 32 indicates "
    "none\",\"type\":\"integer\"},\"out_delay\":{\"minimum\":0,\"description\":\"output"
    " pulse initial delay after start [0.1 "
    "msec]\",\"type\":\"integer\"},\"output_usage\":{\"minimum\":0,\"description\":\"0 "
    ".. no output, 1 .. output derived from internal clock, 2 .. output "
    "derived from dmi, 3 .. output derived from external signal, 4 .. "
    "output static low, 5 .. output static high\",\"type\":\"integer\"},\"dmi_min"
    "_time\":{\"minimum\":0,\"description\":\"dmi, minimum time interval between "
    "trigger events [0.1 msec]\",\"type\":\"integer\"},\"ext_subdivider\":{\"minimu"
    "m\":0,\"description\":\"reduces the frequency, default "
    "1\",\"type\":\"integer\"},\"out_duration\":{\"minimum\":0,\"description\":\"output"
    " pulse duration [0.1 "
    "msec]\",\"type\":\"integer\"},\"dmi_dist_per_tick\":{\"description\":\"dmi, "
    "distance per tick, just informative "
    "[m]\",\"type\":\"number\"},\"num_channel\":{\"minimum\":0,\"description\":\"number"
    " of input/output channels\",\"type\":\"integer\"},\"input_usage\":{\"minimum\":"
    "0,\"description\":\"0 .. deactivated, 1 .. just detecting and timestampin"
    "g\",\"type\":\"integer\"},\"ttip_version\":{\"minimum\":0,\"description\":\"follow"
    "ing main.sub.ss.sss\",\"type\":\"integer\"},\"channel\":{\"minimum\":0,\"descrip"
    "tion\":\"ID of input/output channel\",\"type\":\"integer\"},\"dmi_max_time\":{\""
    "minimum\":0,\"description\":\"dmi, maximum time interval between trigger "
    "events [0.1 msec]\",\"type\":\"integer\"},\"out_interval\":{\"minimum\":0,\"desc"
    "ription\":\"output pulse interval [0.1 msec]\",\"type\":\"integer\"}},\"$schem"
    "a\":\"http://json-schema.org/draft-04/schema#\",\"description\":\"Trigger-Ti"
    "mestamping-Unit/IP configuration\"}"
    ,
    /* example: */
    "{\"input_descr\":\"Port 1 - Exposure\",\"out_num_pulses\":1,\"in_max_duration"
    "\":10000,\"output_descr\":\"Port 1 - Trigger\",\"ext_signal\":0,\"ext_delay\":0"
    ",\"out_polarity\":0,\"in_polarity\":0,\"in_max_delay\":9990,\"dmi_incr\":0,\"ex"
    "t_bitmask\":0,\"in_min_duration\":0,\"ext_channel\":32,\"out_delay\":0,\"outpu"
    "t_usage\":1,\"dmi_min_time\":0,\"ext_subdivider\":1,\"out_duration\":300,\"dmi"
    "_dist_per_tick\":0.0,\"num_channel\":9,\"input_usage\":1,\"ttip_version\":1,\""
    "channel\":0,\"dmi_max_time\":0,\"out_interval\":10000}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Vertex "
    "Information\",\"description\":\"Details about vertex file\",\"type\":\"object\""
    ",\"properties\":{\"vertex_file\":{\"required\":[\"file_extension\"],\"type\":\"ob"
    "ject\",\"properties\":{\"file_uuid\":{\"description\":\"File's Universally "
    "Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Vertex file "
    "extension, without the leading dot\",\"type\":\"string\"}}}}}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_uuid\":\"51534d95-d71f-4f36-ae1a-3e63a21fd1c7\",\"fi"
    "le_extension\":\"vtx\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Voxel Information\",\"type\":\"object\",\"$schema\":\"http://json-sc"
    "hema.org/draft-04/schema#\",\"oneOf\":[{\"required\":[\"size\",\"voxel_origin\""
    ",\"voxel_type\"],\"additionalProperties\":false,\"properties\":{\"voxel_origi"
    "n\":{\"$ref\":\"#/definitions/voxel_origin_enum\"},\"shape_thresholds\":{\"$re"
    "f\":\"#/definitions/shape_thresholds\"},\"size\":{\"oneOf\":[{\"$ref\":\"#/defin"
    "itions/voxel_size\"},{\"$ref\":\"#/definitions/voxel_size_cubic\"}],\"descri"
    "ption\":\"Size of voxels in file coordinate system.\"},\"voxel_type\":{\"$re"
    "f\":\"#/definitions/voxel_type\"}}},{\"required\":[\"reference_point\",\"size_"
    "llcs\",\"size\",\"voxel_origin\",\"voxel_type\"],\"additionalProperties\":false"
    ",\"properties\":{\"size_llcs\":{\"$ref\":\"#/definitions/voxel_size\",\"descrip"
    "tion\":\"Size of voxels in a locally levelled cartesian coordinate "
    "system (xyz). This is only used for voxels based on a map projection.\""
    "},\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_thresholds\"},\"voxel_"
    "type\":{\"$ref\":\"#/definitions/voxel_type\"},\"voxel_origin\":{\"oneOf\":[{\"$"
    "ref\":\"#/definitions/voxel_origin_enum\"},{\"$ref\":\"#/definitions/voxel_o"
    "rigin_point\",\"description\":\"The base point of the voxel grid. Used "
    "together with <tt>voxel_size</tt> and <tt>voxel_index</tt> to compute "
    "actual point coordinates.\"}]},\"reference_point\":{\"minItems\":2,\"items\":"
    "{\"minimum\":-180,\"type\":\"number\",\"maximum\":180},\"description\":\"Point in"
    " WGS84 geodetic decimal degree (EPSG:4326) that was used to compute "
    "the projection distortion parameters. The coefficient order is "
    "latitude, longitude. Only voxels with corresponding geo_tag, "
    "voxel_size and reference_point can be reliably processed together. "
    "This entry is available for voxel files in projected CRS only.\",\"type\""
    ":\"array\",\"maxItems\":2},\"size\":{\"$ref\":\"#/definitions/voxel_size\",\"desc"
    "ription\":\"Size of voxels in file coordinate "
    "system.\"}}}],\"description\":\"Details about the voxels contained in the "
    "file\",\"definitions\":{\"voxel_origin_enum\":{\"description\":\"Defines "
    "whether the voxel's center or a corner is placed on CRS origin <tt>(0/"
    "0/0)</tt>.\",\"enum\":[\"center\",\"corner\"],\"default\":\"corner\"},\"shape_thre"
    "sholds\":{\"required\":[\"plane\",\"line\"],\"description\":\"Thresholds used to"
    " compute the voxel's shape_id value.\",\"type\":\"object\",\"properties\":{\"p"
    "lane\":{\"maximum\":1,\"type\":\"number\",\"exclusiveMinimum\":true,\"exclusiveM"
    "aximum\":true,\"description\":\"If the smallest eigenvalue is smaller than"
    " the median eigenvalue * plane_threshold, the voxel is considered a "
    "plane.\",\"minimum\":0},\"line\":{\"minimum\":1,\"description\":\"If the biggest"
    " eigenvalue is bigger than the median eigenvalue * line_threshold, the"
    " voxel is considered a line.\",\"type\":\"number\",\"exclusiveMinimum\":true}"
    "}},\"voxel_size_cubic\":{\"$ref\":\"#/definitions/edge_length\",\"type\":\"numb"
    "er\"},\"voxel_origin_point\":{\"minItems\":3,\"items\":{\"type\":\"number\"},\"des"
    "cription\":\"Origin point for all voxel indices in voxel CRS.\",\"type\":\"a"
    "rray\",\"maxItems\":3},\"voxel_type\":{\"description\":\"Whether a point in a "
    "voxel represents its center or its centroid. If type is <tt>index</tt>"
    " there is no point but only an integer voxel index.\",\"enum\":[\"center\","
    "\"centroid\",\"index\"],\"default\":\"centroid\"},\"edge_length\":{\"minimum\":0,\""
    "description\":\"Length of voxel edge [m].\",\"type\":\"number\",\"exclusiveMin"
    "imum\":true},\"voxel_size\":{\"minItems\":3,\"items\":{\"$ref\":\"#/definitions/"
    "edge_length\"},\"description\":\"Size of "
    "voxels.\",\"type\":\"array\",\"maxItems\":3}}}"
    ,
    /* example: */
    "{\"size_llcs\":[0.5156575252891171,0.5130835356683303,0.5143705304787237"
    "],\"shape_thresholds\":{\"plane\":0.16,\"line\":6},\"voxel_type\":\"centroid\",\""
    "voxel_origin\":\"corner\",\"reference_point\":[48,16],\"size\":[0.59716428347"
    "79395,0.5971642834779395,0.5143705304787237]}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Waveform Averaging Settings\",\"required\":[\"num_shots\",\"mta_zo"
    "ne\"],\"type\":\"object\",\"properties\":{\"num_shots\":{\"minimum\":1,\"descripti"
    "on\":\"Number of consecutive shots to be used for averaging.\",\"type\":\"in"
    "teger\"},\"mta_zone\":{\"minimum\":1,\"description\":\"Fixed MTA zone for aver"
    "aging.\",\"type\":\"integer\"},\"trim\":{\"minimum\":0,\"description\":\"Percentag"
    "e for robust averaging.\",\"type\":\"number\",\"maximum\":0.5,\"default\":0}},\""
    "$schema\":\"http://json-schema.org/draft-04/schema#\",\"description\":\"Sett"
    "ings for waveform averaging\"}"
    ,
    /* example: */
    "{\"num_shots\":7,\"mta_zone\":1,\"trim\":0.05}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Waveform Information\",\"required\":[\"sample_block_file\",\"sampl"
    "e_data_files\"],\"type\":\"object\",\"properties\":{\"sample_data_files\":{\"ite"
    "ms\":{\"required\":[\"channel\",\"channel_name\",\"sample_interval\",\"sample_bi"
    "ts\",\"laser_wavelength\",\"delta_st\",\"file_extension\"],\"type\":\"object\",\"p"
    "roperties\":{\"file_uuid\":{\"description\":\"File's Universally Unique "
    "Identifier (RFC 4122)\",\"type\":\"string\"},\"sample_bits\":{\"minimum\":0,\"ty"
    "pe\":\"integer\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":false,\"descr"
    "iption\":\"Bitwidth of samples (e.g. 10 bit, 12 "
    "bit)\",\"maximum\":32},\"file_extension\":{\"description\":\"Sample data file "
    "extension, without the leading dot\",\"type\":\"string\"},\"delta_st\":{\"desc"
    "ription\":\"reserved\",\"type\":\"number\"},\"channel\":{\"minimum\":0,\"type\":\"in"
    "teger\",\"exclusiveMinimum\":false,\"exclusiveMaximum\":false,\"description\""
    ":\"Sample block channel number (255 = "
    "invalid)\",\"maximum\":255},\"channel_name\":{\"description\":\"Sample block "
    "channel name\",\"type\":\"string\"},\"laser_wavelength\":{\"minimum\":0,\"descri"
    "ption\":\"Laser wavelength in meters (0 = unknown)\",\"type\":\"number\",\"exc"
    "lusiveMinimum\":false},\"sample_interval\":{\"minimum\":0,\"description\":\"Sa"
    "mpling interval in seconds\",\"type\":\"number\",\"exclusiveMinimum\":false}}"
    "},\"type\":\"array\"},\"range_offset_waveform_samples_m\":{\"description\":\"Ca"
    "librated device specific range offset for waveform samples in "
    "meters.\",\"type\":\"number\"},\"range_offset_m\":{\"description\":\"Calibrated "
    "device specific range offset for waveform analysis by system response "
    "fitting in meters.\",\"type\":\"number\"},\"sample_block_file\":{\"required\":["
    "\"file_extension\"],\"type\":\"object\",\"properties\":{\"file_uuid\":{\"descript"
    "ion\":\"File's Universally Unique Identifier (RFC "
    "4122)\",\"type\":\"string\"},\"file_extension\":{\"description\":\"Sample block "
    "file extension, without the leading dot\",\"type\":\"string\"}}}},\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\",\"description\":\"Details "
    "about waveform files\"}"
    ,
    /* example: */
    "{\"sample_data_files\":[{\"file_uuid\":\"da084413-e3e8-4655-a122-071de8490d"
    "8e\",\"sample_bits\":12,\"file_extension\":\"sp0\",\"delta_st\":0,\"channel\":0,\""
    "channel_name\":\"high_power\",\"laser_wavelength\":0,\"sample_interval\":1.00"
    "503e-09},{\"file_uuid\":\"93585b5e-5ea9-43a1-947b-e7ba3be642d2\",\"sample_b"
    "its\":12,\"file_extension\":\"sp1\",\"delta_st\":0,\"channel\":1,\"channel_name\""
    ":\"low_power\",\"laser_wavelength\":0,\"sample_interval\":1.00503e-09},{\"fil"
    "e_uuid\":\"9d2298c4-1036-464f-b5cb-1cf8e517f3a0\",\"sample_bits\":12,\"file_"
    "extension\":\"sp5\",\"delta_st\":0,\"channel\":5,\"channel_name\":\"wwf\",\"laser_"
    "wavelength\":0,\"sample_interval\":1.00503e-09}],\"range_offset_waveform_s"
    "amples_m \":7.283,\"range_offset_m\":3.1415,\"sample_block_file\":{\"file_uu"
    "id\":\"93a03615-66c0-4bea-8ff8-c577378fe660\",\"file_extension\":\"sbx\"}}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Waveform"
    " Settings\",\"description\":\"Scanner settings for waveform output\",\"type\""
    ":\"array\",\"items\":{\"required\":[\"sbl_name\",\"enabled\",\"channel_idx_mask\"]"
    ",\"type\":\"object\",\"properties\":{\"pass_dev_greater\":{\"description\":\"Thre"
    "shold for deviation greater than "
    "[1]\",\"type\":\"integer\"},\"enabled\":{\"description\":\"Waveform output "
    "enabled\",\"type\":\"boolean\"},\"pass_dev_less\":{\"description\":\"Threshold "
    "for deviation less than "
    "[1]\",\"type\":\"integer\"},\"smart_enabled\":{\"description\":\"Smart waveform "
    "output "
    "enabled\",\"type\":\"boolean\"},\"channel_idx_mask\":{\"description\":\"Bit mask"
    " for channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, "
    "...\",\"type\":\"integer\"},\"pass_ampl_greater\":{\"description\":\"Threshold "
    "for amplitude greater than "
    "[dB]\",\"type\":\"number\"},\"logic_expression\":{\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\",\"type\":\"string\"},\"sbl_name\":{\"description\":\"Name of sample "
    "block, e.g.: wfm, "
    "wwf\",\"type\":\"string\"},\"pass_rng_greater\":{\"description\":\"Threshold for"
    " range greater than "
    "[m]\",\"type\":\"number\"},\"pass_ampl_less\":{\"description\":\"Threshold for "
    "amplitude less than "
    "[dB]\",\"type\":\"number\"},\"pass_rng_less\":{\"description\":\"Threshold for "
    "range less than [m]\",\"type\":\"number\"}}}}"
    ,
    /* example: */
    "[{\"enabled\":true,\"smart_enabled\":true,\"channel_idx_mask\":11,\"pass_ampl"
    "_greater\":1.0,\"sbl_name\":\"wfm\",\"pass_rng_greater\":9.27,\"pass_ampl_less"
    "\":5.0,\"pass_rng_less\":13.11},{\"enabled\":false,\"channel_idx_mask\":32,\"s"
    "bl_name\":\"wwf\"}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Window Analysis\",\"required\":[\"result\",\"filter\",\"settings\"],\""
    "type\":\"object\",\"properties\":{\"result\":{\"required\":[\"angle\",\"range_mean"
    "\",\"range_sigma\",\"amplitude_mean\",\"amplitude_sigma\",\"amplitude_offset\"]"
    ",\"type\":\"object\",\"properties\":{\"angle\":{\"items\":{\"type\":\"number\"},\"des"
    "cription\":\"[deg]\",\"type\":\"array\"},\"amplitude_mean\":{\"items\":{\"type\":\"n"
    "umber\"},\"description\":\"[dB]\",\"type\":\"array\"},\"amplitude_sigma\":{\"items"
    "\":{\"type\":\"number\"},\"description\":\"[dB]\",\"type\":\"array\"},\"timestamp\":{"
    "\"items\":{\"type\":\"number\"},\"description\":\"[s]\",\"type\":\"array\"},\"range_s"
    "igma\":{\"items\":{\"type\":\"number\"},\"description\":\"[m]\",\"type\":\"array\"},\""
    "amplitude_offset\":{\"items\":{\"type\":\"number\"},\"description\":\"[dB]\",\"typ"
    "e\":\"array\"},\"range_mean\":{\"items\":{\"type\":\"number\"},\"description\":\"[m]"
    "\",\"type\":\"array\"}}},\"filter\":{\"required\":[\"angle\",\"range_min\",\"range_m"
    "ax\",\"amplitude_max\"],\"type\":\"object\",\"properties\":{\"angle\":{\"items\":{\""
    "type\":\"number\"},\"description\":\"[deg]\",\"type\":\"array\"},\"range_min\":{\"it"
    "ems\":{\"type\":\"number\"},\"description\":\"[m]\",\"type\":\"array\"},\"range_max\""
    ":{\"items\":{\"type\":\"number\"},\"description\":\"[m]\",\"type\":\"array\"},\"ampli"
    "tude_max\":{\"items\":{\"type\":\"number\"},\"description\":\"[dB]\",\"type\":\"arra"
    "y\"}}},\"settings\":{\"required\":[\"range\",\"amplitude\"],\"type\":\"object\",\"pr"
    "operties\":{\"range\":{\"required\":[\"sigma_factor\",\"additive_value\"],\"type"
    "\":\"object\",\"properties\":{\"sigma_factor\":{\"type\":\"number\"},\"additive_va"
    "lue\":{\"type\":\"number\"}}},\"amplitude\":{\"required\":[\"sigma_factor\",\"addi"
    "tive_value\"],\"type\":\"object\",\"properties\":{\"sigma_factor\":{\"type\":\"num"
    "ber\"},\"additive_value\":{\"type\":\"number\"}}}}}},\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"description\":\"Window analysis data "
    "estimated from scandata and resulting filter parameters\"}"
    ,
    /* example: */
    "{\"result\":{\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,"
    "15.9],\"amplitude_mean\":[5.347396,5.263155,5.224655,5.179926,5.097782,5"
    ".116479,5.051756,4.983473,5.007885,5.002441,4.982],\"amplitude_sigma\":["
    "0.4272844,0.4298479,0.4236816,0.4283583,0.4362353,0.4315141,0.4373984,"
    "0.4472798,0.4346001,0.4345487,0.4540681],\"timestamp\":[408.4441,411.444"
    "3],\"range_sigma\":[0.01869652,0.02151435,0.01747969,0.01918765,0.019457"
    "76,0.01934862,0.01955329,0.02225589,0.02229977,0.01899122,0.02009433],"
    "\"amplitude_offset\":[1.9,1.9],\"range_mean\":[0.1105621,0.1079564,0.10870"
    "88,0.1067261,0.1054582,0.1090412,0.102871,0.1019044,0.1051523,0.105844"
    "5,0.1031261]},\"filter\":{\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15"
    ".6,15.7,15.8,15.9],\"range_min\":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.2"
    "18,-0.219,-0.221,-0.223,-0.225,-0.227],\"range_max\":[0.424,0.425,0.426,"
    "0.427,0.428,0.428,0.429,0.43,0.431,0.431,0.432],\"amplitude_max\":[8.04,"
    "8.01,7.99,7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78]},\"settings\":{\"range\":"
    "{\"sigma_factor\":8,\"additive_value\":0.1},\"amplitude\":{\"sigma_factor\":4,"
    "\"additive_value\":1.0}}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"title\":\"Window Echo Correction\",\"required\":[\"amplitude\",\"range\",\"sli"
    "ces\"],\"type\":\"object\",\"properties\":{\"range\":{\"required\":[\"minimum\",\"ma"
    "ximum\",\"entries\"],\"description\":\"Range axis of correction table\",\"type"
    "\":\"object\",\"properties\":{\"minimum\":{\"minimum\":-2.0,\"description\":\"Mini"
    "mum range in m\",\"type\":\"number\",\"maximum\":2.0},\"entries\":{\"minimum\":1,"
    "\"description\":\"Number of range entries\",\"type\":\"integer\"},\"maximum\":{\""
    "minimum\":-2.0,\"description\":\"Maximum range in m\",\"type\":\"number\",\"maxi"
    "mum\":2.0}}},\"slices\":{\"items\":{\"required\":[\"amplitude\",\"table\"],\"descr"
    "iption\":\"Window echo correction parameter slice\",\"type\":\"object\",\"prop"
    "erties\":{\"table\":{\"minItems\":1,\"items\":{\"minItems\":1,\"items\":{\"minItem"
    "s\":3,\"items\":{\"description\":\"Table cell (item 0: amplitude in dB, 1: "
    "range in m, 2: flags)\",\"type\":\"number\"},\"description\":\"Table column (="
    " range axis)\",\"type\":\"array\",\"maxItems\":3},\"description\":\"Table row (="
    " amplitude axis)\",\"type\":\"array\"},\"description\":\"Correction table "
    "(dimension defined by the 'amplitude' and 'range' "
    "objects)\",\"type\":\"array\"},\"amplitude\":{\"description\":\"Window echo "
    "amplitude of slice in dB\",\"type\":\"number\"}}},\"type\":\"array\"},\"amplitud"
    "e\":{\"required\":[\"minimum\",\"maximum\",\"entries\"],\"description\":\"Amplitud"
    "e axis of correction table\",\"type\":\"object\",\"properties\":{\"minimum\":{\""
    "minimum\":0.0,\"description\":\"Minimum amplitude in "
    "dB\",\"type\":\"number\"},\"entries\":{\"minimum\":1,\"description\":\"Number of "
    "amplitude entries\",\"type\":\"integer\"},\"maximum\":{\"minimum\":0.0,\"descrip"
    "tion\":\"Maximum amplitude in dB\",\"type\":\"number\"}}}},\"$schema\":\"http://"
    "json-schema.org/draft-04/schema#\",\"description\":\"Correction parameters"
    " for window glass echoes\"}"
    ,
    /* example: */
    "{\"range\":{\"minimum\":-1.5060822940732335,\"entries\":128,\"maximum\":1.5060"
    "822940732335},\"slices\":[{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]"
    "],\"amplitude\":1.5},{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],\"am"
    "plitude\":2.0}],\"amplitude\":{\"minimum\":2,\"entries\":128,\"maximum\":20}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LAG_ANGLE_CORRECTION,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);

#endif // RDB_81EBCB65_FE94_471E_8D25_2B4D71CC91B1
