/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    changelog.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Manage point cloud changelog
 * \version 2020-06-29/AW: Initial version
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_CHANGELOG_HPP
#define RIEGL_RDB_POINTCLOUD_CHANGELOG_HPP

//---< INCLUDES >---------------------------------------------------------------

#include <string>
#include <cstdlib>
#include <cstdint>

#include "riegl/rdb/pointcloud/pointcloudData.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< CLASS Changelog >--------------------------------------------------------
/*!
 * \brief Manage point cloud changelog
 *
 * Each database maintains a log of changes made to a point cloud.
 *
 * A separate log entry is created for each database transaction or management
 * operation. A log entry contains log messages with high-level information
 * about the modifications as follows:
 *
 *   - current transaction number, title, agent, date and time
 *   - number of data points added, updated, removed
 *   - list of point attributes affected by point operations
 *   - list of point attributes added, updated, removed
 *   - list of metadata entries added, updated, removed
 *   - modification of level-of-detail parameters
 *   - management: restore transaction, finalize and vacuum database
 *   - optional: log messages (some text) provided by client software
 *
 * Details like actual point data, attribute definitions or metadata values are
 * not recorded.
 *
 * \see riegl::rdb::Pointcloud::changelog()
 *
 * \since 2.3.0
 */
class Changelog
{
public:
    /*!
     * \brief Constructor
     * \note  You cannot create new Changelog objects directly,
     *        use riegl::rdb::Pointcloud::changelog() instead.
     */
    explicit Changelog(riegl::rdb::PointcloudData* pointcloud);

    /*!
     * \brief Append text to current log entry
     *
     * This function appends text message(s) to the log entry of the current
     * database transaction. As with all strings in RDB, the characters are
     * expected to be UTF-8 encoded. Line endings are normalized (i.e. CR+LF
     * and CR are converted to LF).
     *
     * Calling this function while no transaction is active has no effect.
     *
     * \since 2.3.0
     */
    void appendMessage(const std::string &message);

    /*!
     * \brief Verify log entry signature
     *
     * Returns 'false' if:
     *
     * - there is no signature for the log entry
     * - the signature does not match the log messages
     * - a wrong signature encryption key was given
     *
     * Otherwise returns 'true'.
     *
     * \since 2.3.0
     */
    static bool verifyLogEntry(
        const std::string  &entry,    //!< [in] log entry messages to verify
        const std::uint32_t method,   //!< [in] signature method (1: default)
        const std::uint32_t key_size, //!< [in] signature encryption key size (at least 32 byte)
        const void* const   key_data  //!< [in] signature encryption key buffer
    );

    /*!
     * \brief Export changelog to text file
     *
     * This function exports the entire changelog to a single text file.
     * The file is UTF-8 encoded and text lines are separated by a single
     * line feed character (LF, ASCII 0x0A), regardless of the operating
     * system the file was created on.
     *
     * \since 2.3.0
     */
    void exportToTextfile(
        const std::string &filename //!< [in] output text filename
    );

    /*!
     * \brief Import changelog from database
     *
     * This function imports the entire changelog from an other database.
     * This is intended for applications where a database is to be replaced
     * by a new one (often derived from the old one) and the processing
     * history is to be retained.
     *
     * \since 2.3.0
     */
    void importFromDatabase(
        const std::string &filename //!< [in] input database filename
    );

private:
    riegl::rdb::PointcloudData *data;
};

}}} // namespace riegl::rdb::pointcloud

#endif // RIEGL_RDB_POINTCLOUD_CHANGELOG_HPP
