/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 * \version 2025-11-11/AW: Item "riegl.lag_angle_correction" added (#6068)
 *
 *******************************************************************************
 */

#ifndef RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
#define RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87

#include <vector>
#include <string>
#include <cstdlib>
#include <cstdint>

namespace riegl {
namespace rdb {
namespace pointcloud {

// Meta data item definition version
const std::string RDB_META_DATA_ITEMS_VERSION = "1.1.57";
const std::string RDB_META_DATA_ITEMS_DATE = "2025-11-11";

// Meta data item states
const unsigned char RDB_STATUS_REQUIRED = 1; // mandatory item
const unsigned char RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
struct MetaDataItemInfo
{
    std::string   name;
    std::string   title;
    std::string   description;
    unsigned char status;
    std::string   schema;
    std::string   example;

    operator const std::string&() const { return name; }
};

// Angular notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_ANGULAR_NOTCH_FILTER = {
    /* name        */ "riegl.angular_notch_filter",
    /* title       */ "Angular Notch Filter",
    /* description */ "Angular notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Angular Notch Filter\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"required\":[\"angle\",\"range_mean\",\"amplit"
    "ude_mean\"],\"properties\":{\"angle\":{\"type\":\"array\",\"items\":{\"type\":\"numb"
    "er\"},\"description\":\"Angle [deg]\"},\"range_mean\":{\"type\":\"array\",\"items\""
    ":{\"type\":\"number\"},\"description\":\"Mean range [m]\"},\"amplitude_mean\":{\""
    "type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Mean amplitude "
    "[dB]\"}},\"description\":\"Angular notch filter parameters for window "
    "glass echoes\"}"
    ,
    /* example: */
    "{\"angle\":[14.0,15.0,16.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],\"ran"
    "ge_mean\":[0.094,0.094,0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,"
    "0.0675,0.06475],\"amplitude_mean\":[3.780913,3.780913,3.480913,3.120913,"
    "2.850913,2.720913,2.680913,2.610913,2.530913,2.570913,2.570913]}"
};

// Atmospheric parameters
const MetaDataItemInfo RDB_RIEGL_ATMOSPHERE = {
    /* name        */ "riegl.atmosphere",
    /* title       */ "Atmosphere",
    /* description */ "Atmospheric parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Atmospheric Parameters\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"required\":[\"temperature\",\"pressure\",\""
    "rel_humidity\",\"pressure_sl\",\"amsl\",\"group_velocity\",\"attenuation\",\"wav"
    "elength\"],\"properties\":{\"wavelength\":{\"type\":\"number\",\"description\":\"L"
    "aser wavelength [nm]\"},\"amsl\":{\"type\":\"number\",\"description\":\"Height "
    "above mean sea level (AMSL) "
    "[m]\"},\"rel_humidity\":{\"type\":\"number\",\"description\":\"Relative humidity"
    " along measurement path "
    "[%]\"},\"attenuation\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "attenuation "
    "[1/km]\"},\"temperature\":{\"type\":\"number\",\"description\":\"Temperature "
    "along measurement path "
    "[\\u00b0C]\"},\"pressure_sl\":{\"type\":\"number\",\"description\":\"Atmospheric "
    "pressure at sea level "
    "[mbar]\"},\"pressure\":{\"type\":\"number\",\"description\":\"Pressure along "
    "measurment path "
    "[mbar]\"},\"group_velocity\":{\"type\":\"number\",\"description\":\"Group "
    "velocity of laser beam [m/s]\"}},\"description\":\"Atmospheric "
    "parameters\"}"
    ,
    /* example: */
    "{\"wavelength\":1550,\"amsl\":0,\"rel_humidity\":63,\"attenuation\":0.028125,\""
    "temperature\":7,\"pressure_sl\":970,\"pressure\":970,\"group_velocity\":29971"
    "1000.0}"
};

// Laser beam geometry details
const MetaDataItemInfo RDB_RIEGL_BEAM_GEOMETRY = {
    /* name        */ "riegl.beam_geometry",
    /* title       */ "Beam Geometry",
    /* description */ "Laser beam geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Beam Geometry\",\"$schema\":\"http://json-schema"
    ".org/draft-04/schema#\",\"required\":[\"beam_exit_diameter\",\"beam_divergen"
    "ce\"],\"properties\":{\"beam_divergence\":{\"type\":\"number\",\"minimum\":0,\"des"
    "cription\":\"Beam divergence in far field [rad]\",\"exclusiveMinimum\":fals"
    "e},\"beam_exit_diameter\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Be"
    "am width at exit aperture "
    "[m]\",\"exclusiveMinimum\":false}},\"description\":\"Laser beam geometry "
    "details\"}"
    ,
    /* example: */
    "{\"beam_divergence\":0.0003,\"beam_exit_diameter\":0.0072}"
};

// List of control object type definitions
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_CATALOG = {
    /* name        */ "riegl.control_object_catalog",
    /* title       */ "Control Object Catalog",
    /* description */ "List of control object type definitions",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Control Object Catalog\",\"definitions\":{\"chec"
    "kerboard2x2\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"}"
    ",{\"type\":\"object\",\"required\":[\"shape\",\"square_length\"],\"properties\":{\""
    "square_length\":{\"type\":\"number\",\"minimum\":0,\"description\":\"length of a"
    " square of the checkerboard in meters\",\"exclusiveMinimum\":true},\"shape"
    "\":{\"type\":\"string\",\"description\":\"shape "
    "identifier\",\"enum\":[\"checkerboard2x2\"]}},\"description\":\"checkerboard "
    "specific properties\"}],\"description\":\"checkerboard 2 by 2\"},\"sphere\":{"
    "\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"type\":\"obje"
    "ct\",\"required\":[\"shape\",\"diameter\"],\"properties\":{\"shape\":{\"type\":\"str"
    "ing\",\"description\":\"shape identifier\",\"enum\":[\"sphere\"]},\"diameter\":{\""
    "type\":\"number\",\"minimum\":0,\"description\":\"diameter in "
    "meters\",\"exclusiveMinimum\":true}},\"description\":\"sphere specific prope"
    "rties\"}],\"description\":\"sphere\"},\"cylinder\":{\"type\":\"object\",\"allOf\":["
    "{\"$ref\":\"#/definitions/common\"},{\"type\":\"object\",\"required\":[\"shape\",\""
    "diameter\",\"height\"],\"properties\":{\"height\":{\"type\":\"number\",\"minimum\":"
    "0,\"description\":\"height in meters\",\"exclusiveMinimum\":true},\"shape\":{\""
    "type\":\"string\",\"description\":\"shape identifier\",\"enum\":[\"cylinder\"]},\""
    "diameter\":{\"type\":\"number\",\"minimum\":0,\"description\":\"diameter in "
    "meters\",\"exclusiveMinimum\":true}},\"description\":\"cylinder specific pro"
    "perties\"}],\"description\":\"cylinder\"},\"rectangle\":{\"type\":\"object\",\"all"
    "Of\":[{\"$ref\":\"#/definitions/common\"},{\"type\":\"object\",\"required\":[\"sha"
    "pe\",\"length\",\"width\"],\"properties\":{\"width\":{\"type\":\"number\",\"minimum\""
    ":0,\"description\":\"width in meters\",\"exclusiveMinimum\":true},\"shape\":{\""
    "type\":\"string\",\"enum\":[\"rectangle\"]},\"length\":{\"type\":\"number\",\"minimu"
    "m\":0,\"description\":\"length in "
    "meters\",\"exclusiveMinimum\":true}},\"description\":\"rectangle specific pr"
    "operties\"}],\"description\":\"rectangle\"},\"round_corner_cube_prism\":{\"typ"
    "e\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\"type\":\"object\","
    "\"required\":[\"shape\",\"diameter\"],\"properties\":{\"offset\":{\"type\":\"number"
    "\",\"description\":\"offset in meters, e.g. reflector constant "
    "(optional)\"},\"shape\":{\"type\":\"string\",\"description\":\"shape identifier\""
    ",\"enum\":[\"round_corner_cube_prism\"]},\"diameter\":{\"type\":\"number\",\"mini"
    "mum\":0,\"description\":\"diameter in "
    "meters\",\"exclusiveMinimum\":true}},\"description\":\"round corner cube "
    "prism specific properties\"}],\"description\":\"round corner cube prism\"},"
    "\"chevron\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common\"},{\""
    "type\":\"object\",\"required\":[\"shape\",\"outside_edge_length\",\"thickness\"],"
    "\"properties\":{\"outside_edge_length\":{\"type\":\"number\",\"minimum\":0,\"desc"
    "ription\":\"length of the two outer edges in meters\",\"exclusiveMinimum\":"
    "true},\"shape\":{\"type\":\"string\",\"description\":\"shape identifier\",\"enum\""
    ":[\"chevron\"]},\"thickness\":{\"type\":\"number\",\"minimum\":0,\"description\":\""
    "thickness in meters\",\"exclusiveMinimum\":true}},\"description\":\"chevron "
    "specific properties\"}],\"description\":\"chevron\"},\"common\":{\"type\":\"obje"
    "ct\",\"required\":[\"name\",\"shape\"],\"properties\":{\"name\":{\"type\":\"string\","
    "\"minLength\":3,\"description\":\"unique type "
    "identifier\"},\"surface_type\":{\"type\":\"string\",\"description\":\"surface "
    "material type\",\"enum\":[\"retro_reflective_foil\",\"diffuse\"]},\"shape\":{\"t"
    "ype\":\"string\",\"description\":\"shape identifier\",\"enum\":[\"rectangle\",\"ch"
    "eckerboard2x2\",\"chevron\",\"circular_disk\",\"cylinder\",\"sphere\",\"round_co"
    "rner_cube_prism\"]},\"description\":{\"type\":\"string\",\"description\":\"strin"
    "g describing the object\"}},\"description\":\"common object properties\"},\""
    "circular_disk\":{\"type\":\"object\",\"allOf\":[{\"$ref\":\"#/definitions/common"
    "\"},{\"type\":\"object\",\"required\":[\"shape\",\"diameter\"],\"properties\":{\"off"
    "set\":{\"type\":\"number\",\"description\":\"offset in meters, e.g. reflector "
    "constant (optional)\"},\"shape\":{\"type\":\"string\",\"description\":\"shape id"
    "entifier\",\"enum\":[\"circular_disk\"]},\"diameter\":{\"type\":\"number\",\"minim"
    "um\":0,\"description\":\"diameter in "
    "meters\",\"exclusiveMinimum\":true}},\"description\":\"circular disk "
    "specific properties\"}],\"description\":\"circular disk\"}},\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"required\":[\"types\"],\"properties\""
    ":{\"types\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"oneOf\":[{\"$ref\":\"#"
    "/definitions/rectangle\"},{\"$ref\":\"#/definitions/checkerboard2x2\"},{\"$r"
    "ef\":\"#/definitions/chevron\"},{\"$ref\":\"#/definitions/circular_disk\"},{\""
    "$ref\":\"#/definitions/cylinder\"},{\"$ref\":\"#/definitions/sphere\"},{\"$ref"
    "\":\"#/definitions/round_corner_cube_prism\"}]},\"uniqueItems\":true}},\"des"
    "cription\":\"List of control object type definitions\"}"
    ,
    /* example: */
    "{\"comments\":[\"This file contains a list of control object types (aka. "
    "'catalog').\",\"Each type is described by an object,\",\"which must "
    "contain at least the following parameters:\",\"  - name: unique "
    "identifier of the type\",\"  - shape: one of the following supported "
    "shapes:\",\"      - rectangle\",\"      - checkerboard2x2\",\"      - "
    "chevron\",\"      - circular_disk\",\"      - cylinder\",\"      - sphere\",\""
    "      - round_corner_cube_prism\",\"Depending on 'shape', the following "
    "parameters must/may be specified:\",\"  - rectangle:\",\"      - length: "
    "length in meters\",\"      - width: width in meters\",\"  - "
    "checkerboard2x2:\",\"      - square_length: length of a square of the "
    "checkerboard in meters\",\"  - circular_disk:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"  - chevron:\",\"      - outside_edge_length: "
    "length of the two outer edges in meters\",\"      - thickness: thickness"
    " in meters\",\"  - cylinder:\",\"      - diameter: diameter in meters\",\""
    "      - height: height in meters\",\"  - sphere:\",\"      - diameter: "
    "diameter in meters\",\"  - round_corner_cube_prism:\",\"      - diameter: "
    "diameter in meters\",\"      - offset: offset in meters, e.g. reflector "
    "constant (optional)\",\"Optional parameters:\",\"    - description: string"
    " describing the object\",\"    - surface_type: surface material type "
    "(either 'retro_reflective_foil' or "
    "'diffuse')\"],\"types\":[{\"name\":\"Rectangle "
    "60x30\",\"width\":0.3,\"shape\":\"rectangle\",\"description\":\"Rectangle (60cm "
    "x 30cm)\",\"length\":0.6},{\"name\":\"Rectangle "
    "80x40\",\"width\":0.4,\"shape\":\"rectangle\",\"description\":\"Rectangle (80cm "
    "x 40cm)\",\"length\":0.8},{\"name\":\"Checkerboard2x2 30\",\"square_length\":0."
    "3,\"shape\":\"checkerboard2x2\",\"description\":\"Checkerboard (square "
    "length: 30cm)\"},{\"name\":\"Checkerboard2x2 50\",\"square_length\":0.5,\"shap"
    "e\":\"checkerboard2x2\",\"description\":\"Checkerboard (square length: "
    "50cm)\"},{\"name\":\"Chevron 24''/4''\",\"thickness\":0.1016,\"outside_edge_le"
    "ngth\":0.6096,\"shape\":\"chevron\",\"description\":\"Chevron (a=24''; "
    "b=4'')\"},{\"name\":\"Circular Disk "
    "50\",\"surface_type\":\"diffuse\",\"shape\":\"circular_disk\",\"description\":\" "
    "Circular Disk (diameter: 50cm)\",\"diameter\":0.5},{\"name\":\"RIEGL flat "
    "reflector 50 mm\",\"diameter\":0.05,\"offset\":0.0,\"surface_type\":\"retro_re"
    "flective_foil\",\"shape\":\"circular_disk\",\"description\":\"flat circular "
    "reflector from retro-reflective foil\"},{\"name\":\"RIEGL flat reflector "
    "100 mm\",\"diameter\":0.1,\"offset\":0.0,\"surface_type\":\"retro_reflective_f"
    "oil\",\"shape\":\"circular_disk\",\"description\":\"flat circular reflector "
    "from retro-reflective foil\"},{\"name\":\"RIEGL flat reflector 150 mm\",\"di"
    "ameter\":0.15,\"offset\":0.0,\"surface_type\":\"retro_reflective_foil\",\"shap"
    "e\":\"circular_disk\",\"description\":\"flat circular reflector from "
    "retro-reflective foil\"},{\"name\":\"RIEGL cylindrical reflector 50 mm\",\"d"
    "iameter\":0.05,\"surface_type\":\"retro_reflective_foil\",\"height\":0.05,\"de"
    "scription\":\"cylindrical reflector from retro-reflective "
    "foil\",\"shape\":\"cylinder\"},{\"name\":\"RIEGL cylindrical reflector 100 mm\""
    ",\"diameter\":0.1,\"surface_type\":\"retro_reflective_foil\",\"height\":0.1,\"d"
    "escription\":\"cylindrical reflector from retro-reflective "
    "foil\",\"shape\":\"cylinder\"},{\"name\":\"Sphere 200 "
    "mm\",\"shape\":\"sphere\",\"description\":\"Sphere (diameter: 200 "
    "mm)\",\"diameter\":0.2},{\"name\":\"Corner Cube Prism 50 mm\",\"offset\":0.0,\"s"
    "hape\":\"round_corner_cube_prism\",\"description\":\"round corner cube "
    "prism\",\"diameter\":0.05}]}"
};

// Details about the control object reference file
const MetaDataItemInfo RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE = {
    /* name        */ "riegl.control_object_reference_file",
    /* title       */ "Control Object Reference file",
    /* description */ "Details about the control object reference file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Control "
    "Object Reference file\",\"properties\":{\"reference_file\":{\"type\":\"object\""
    ",\"required\":[\"file_uuid\",\"file_path\"],\"properties\":{\"file_uuid\":{\"type"
    "\":\"string\",\"description\":\"Control object file's Universally Unique "
    "Identifier (RFC "
    "4122)\"},\"file_path\":{\"type\":\"string\",\"description\":\"Path of the "
    "control object file relative to referring "
    "file\"}},\"description\":\"Reference to a control object "
    "file\"}},\"description\":\"Details about the control object reference "
    "file\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"reference_file\":{\"file_uuid\":\"810f5d57-eccf-49ed-b07a-0cdd109b4213\","
    "\"file_path\":\"../../../10_CONTROL_OBJECTS/ControlPoints.cpx\"}}"
};

// Detection probability as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_DETECTION_PROBABILITY = {
    /* name        */ "riegl.detection_probability",
    /* title       */ "Detection Probability",
    /* description */ "Detection probability as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Detection Probability\",\"$schema\":\"http://jso"
    "n-schema.org/draft-04/schema#\",\"required\":[\"amplitude\",\"detection_prob"
    "ability\"],\"properties\":{\"detection_probability\":{\"type\":\"array\",\"items"
    "\":{\"type\":\"number\"},\"description\":\"Detection probability [0..1]\"},\"amp"
    "litude\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Ampli"
    "tude [dB]\"}},\"description\":\"Detection probability as a function of "
    "amplitude\"}"
    ,
    /* example: */
    "{\"detection_probability\":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]"
    ",\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0]}"
};

// Details about the device used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICE = {
    /* name        */ "riegl.device",
    /* title       */ "Device Information",
    /* description */ "Details about the device used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Information\",\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"required\":[\"device_type\",\"serial_number\"]"
    ",\"properties\":{\"channel_text\":{\"type\":\"string\",\"description\":\"Optional"
    " channel description (e.g. 'Green Channel' for multi-channel "
    "devices)\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\"},\"device_build\":{\"type\":\"string\",\"description\":\"Device build"
    " variant\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. VZ-400i)\"},\"channel_number\":{\"type\":\"integer\",\"minimu"
    "m\":0,\"description\":\"Laser channel number (not defined or 0: single "
    "channel device)\",\"exclusiveMinimum\":false}},\"description\":\"Details "
    "about the device used to acquire the point cloud\"}"
    ,
    /* example: */
    "{\"channel_text\":\"\",\"device_name\":\"Scanner 1\",\"device_build\":\"\",\"serial"
    "_number\":\"S2221234\",\"device_type\":\"VZ-400i\",\"channel_number\":0}"
};

// Scanner device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY = {
    /* name        */ "riegl.device_geometry",
    /* title       */ "Device Geometry",
    /* description */ "Scanner device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Geometry\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"required\":[\"primary\"],\"properties\":{\"seconda"
    "ry\":{\"type\":\"object\",\"required\":[\"ID\",\"content\"],\"properties\":{\"ID\":{\""
    "type\":\"array\",\"items\":{\"type\":\"integer\"},\"minItems\":2,\"description\":\"S"
    "tructure identifier\",\"maxItems\":2},\"content\":{\"type\":\"array\",\"items\":{"
    "\"type\":\"number\"},\"description\":\"Internal calibration "
    "values\"}},\"description\":\"Additional device geometry structure (optiona"
    "l)\"},\"primary\":{\"type\":\"object\",\"required\":[\"ID\",\"content\"],\"propertie"
    "s\":{\"ID\":{\"type\":\"array\",\"items\":{\"type\":\"integer\"},\"minItems\":2,\"desc"
    "ription\":\"Structure identifier\",\"maxItems\":2},\"content\":{\"type\":\"array"
    "\",\"items\":{\"type\":\"number\"},\"description\":\"Internal calibration "
    "values\"}},\"description\":\"Primary device geometry structure (mandatory)"
    "\"},\"amu\":{\"type\":\"object\",\"properties\":{\"lineCC\":{\"type\":\"number\",\"min"
    "imum\":0,\"description\":\"Line Circle Count (number of LSBs per full "
    "rotation about line axis)\",\"exclusiveMinimum\":false},\"frameCC\":{\"type\""
    ":\"number\",\"minimum\":0,\"description\":\"Frame Circle Count (number of "
    "LSBs per full rotation about frame "
    "axis)\",\"exclusiveMinimum\":false}},\"description\":\"Angle Measurement "
    "Unit\"}},\"description\":\"Scanner device geometry details\"}"
    ,
    /* example: */
    "{\"secondary\":{\"ID\":[91,0],\"content\":[0]},\"primary\":{\"ID\":[4,0],\"conten"
    "t\":[0]},\"amu\":{\"lineCC\":124000,\"frameCC\":124000}}"
};

// Scanner passive channel device geometry details
const MetaDataItemInfo RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL = {
    /* name        */ "riegl.device_geometry_passive_channel",
    /* title       */ "Device Geometry Passive Channel",
    /* description */ "Scanner passive channel device geometry details",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Device Geometry Passive Channel\",\"$schema\":\""
    "http://json-schema.org/draft-04/schema#\",\"required\":[\"primary\"],\"prope"
    "rties\":{\"primary\":{\"type\":\"object\",\"required\":[\"ID\",\"content\"],\"proper"
    "ties\":{\"ID\":{\"type\":\"array\",\"items\":{\"type\":\"integer\"},\"minItems\":2,\"d"
    "escription\":\"Structure identifier\",\"maxItems\":2},\"content\":{\"type\":\"ar"
    "ray\",\"items\":{\"type\":\"number\"},\"description\":\"Internal calibration "
    "values\"}},\"description\":\"Primary device geometry structure "
    "(mandatory)\"}},\"description\":\"Scanner passive channel device geometry "
    "details\"}"
    ,
    /* example: */
    "{\"primary\":{\"ID\":[143,0],\"content\":[0]}}"
};

// Limits of the measured values output by the device
const MetaDataItemInfo RDB_RIEGL_DEVICE_OUTPUT_LIMITS = {
    /* name        */ "riegl.device_output_limits",
    /* title       */ "Device Output Limits",
    /* description */ "Limits of the measured values output by the device",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Device "
    "Output Limits\",\"properties\":{\"deviation_minimum\":{\"type\":\"number\",\"des"
    "cription\":\"Minimum possible pulse shape "
    "deviation.\"},\"range_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible range in "
    "meters.\"},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible range in meters.\"},\"background_radiation_minimum\":{\"type\":\"nu"
    "mber\",\"description\":\"Minimum possible background radiation.\"},\"mta_zon"
    "e_count_maximum\":{\"type\":\"number\",\"description\":\"Maximum number of MTA"
    " zones.\"},\"amplitude_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible amplitude in dB.\"},\"background_radiation_maximum\":{\"type\":\"nu"
    "mber\",\"description\":\"Maximum possible background radiation.\"},\"echo_co"
    "unt_maximum\":{\"type\":\"number\",\"description\":\"Maximum number of echoes "
    "a laser shot can "
    "have.\"},\"reflectance_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible reflectance in "
    "dB.\"},\"deviation_maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "possible pulse shape deviation.\"},\"amplitude_minimum\":{\"type\":\"number\""
    ",\"description\":\"Minimum possible amplitude in "
    "dB.\"},\"reflectance_minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "possible reflectance in dB.\"}},\"description\":\"Limits of the measured "
    "values output by the device. The limits depend on the device type, "
    "measurement program and/or scan pattern.\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"deviation_minimum\":-1,\"background_radiation_maximum\":0,\"range_maximu"
    "m\":10000.0,\"range_minimum\":2.9,\"mta_zone_count_maximum\":7,\"background_"
    "radiation_minimum\":0,\"reflectance_maximum\":100.0,\"deviation_maximum\":3"
    "2767,\"amplitude_minimum\":0.0,\"amplitude_maximum\":100.0,\"reflectance_mi"
    "nimum\":-100.0}"
};

// Details about the devices used to acquire the point cloud
const MetaDataItemInfo RDB_RIEGL_DEVICES = {
    /* name        */ "riegl.devices",
    /* title       */ "Devices Information",
    /* description */ "Details about the devices used to acquire the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Devices "
    "Information\",\"items\":{\"type\":\"object\",\"required\":[\"device_type\",\"seria"
    "l_number\"],\"properties\":{\"channel_text\":{\"type\":\"string\",\"description\""
    ":\"Optional channel description (e.g. 'Green Channel' for multi-channel"
    " devices)\"},\"signed\":{\"type\":\"boolean\",\"description\":\"Flag that is set"
    " when the original 'riegl.device' entry in the source file was "
    "correctly "
    "signed.\"},\"device_name\":{\"type\":\"string\",\"description\":\"Optional "
    "device name (e.g. 'Scanner 1' for multi-scanner "
    "systems)\"},\"device_build\":{\"type\":\"string\",\"description\":\"Device build"
    " variant\"},\"serial_number\":{\"type\":\"string\",\"description\":\"Device "
    "serial number (e.g. "
    "S2221234)\"},\"device_type\":{\"type\":\"string\",\"description\":\"Device type "
    "identifier (e.g. VZ-400i)\"},\"channel_number\":{\"type\":\"integer\",\"minimu"
    "m\":0,\"description\":\"Laser channel number (not defined or 0: single "
    "channel device)\",\"exclusiveMinimum\":false}}},\"description\":\"Details "
    "about the devices used to acquire the point cloud (e.g. when merging "
    "point clouds of different devices)\",\"type\":\"array\"}"
    ,
    /* example: */
    "[{\"channel_text\":\"\",\"signed\":false,\"device_name\":\"Scanner 1\",\"device_b"
    "uild\":\"\",\"serial_number\":\"S2221234\",\"device_type\":\"VZ-400i\",\"channel_n"
    "umber\":0},{\"channel_text\":\"\",\"signed\":true,\"device_name\":\"Scanner 2\",\""
    "device_build\":\"\",\"serial_number\":\"S2222680\",\"device_type\":\"VQ-1560i-DW"
    "\",\"channel_number\":1},{\"channel_text\":\"\",\"signed\":true,\"device_name\":\""
    "Scanner 3\",\"device_build\":\"\",\"serial_number\":\"S2222680\",\"device_type\":"
    "\"VQ-1560i-DW\",\"channel_number\":2}]"
};

// Distance measuring instrument configuration
const MetaDataItemInfo RDB_RIEGL_DMI_CONFIG = {
    /* name        */ "riegl.dmi_config",
    /* title       */ "DMI Configuration",
    /* description */ "Distance measuring instrument configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"DMI Configuration\",\"$schema\":\"http://json-sc"
    "hema.org/draft-04/schema#\",\"required\":[\"dmi_count_dir\",\"dmi_incr\",\"dmi"
    "_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\"],\"properties\":{\"dmi_max_"
    "time\":{\"type\":\"integer\",\"maximum\":65535,\"minimum\":0,\"exclusiveMaximum\""
    ":false,\"description\":\"Maximum time interval between trigger events, 0 "
    "means timeout is inactive [0.1 msec]\",\"exclusiveMinimum\":false},\"dmi_d"
    "ist_per_tick\":{\"type\":\"number\",\"description\":\"Distance per tick, just "
    "informative [m]\"},\"dmi_incr\":{\"type\":\"integer\",\"maximum\":2147483647,\"m"
    "inimum\":0,\"exclusiveMaximum\":false,\"description\":\"Increment in ticks, "
    "0 means distance trigger is deactivated\",\"exclusiveMinimum\":false},\"dm"
    "i_min_time\":{\"type\":\"integer\",\"maximum\":65535,\"minimum\":0,\"exclusiveMa"
    "ximum\":false,\"description\":\"Minimum time interval between trigger "
    "events, 0 means rate limitation is inactive [0.1 msec]\",\"exclusiveMini"
    "mum\":false},\"dmi_count_dir\":{\"type\":\"integer\",\"maximum\":1,\"minimum\":0,"
    "\"exclusiveMaximum\":false,\"description\":\"DMI count direction is "
    "inverted\",\"exclusiveMinimum\":false}},\"description\":\"Distance measuring"
    " instrument configuration\"}"
    ,
    /* example: */
    "{\"dmi_max_time\":100,\"dmi_dist_per_tick\":0.00024399999529123306,\"dmi_in"
    "cr\":0,\"dmi_min_time\":0,\"dmi_count_dir\":0}"
};

// Dyntrig
const MetaDataItemInfo RDB_RIEGL_DYNTRIG = {
    /* name        */ "riegl.dyntrig",
    /* title       */ "Dyntrig",
    /* description */ "Dyntrig",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Dyntrig\""
    ",\"properties\":{\"offset\":{\"type\":\"number\",\"description\":\"Offset in "
    "units of dB, for calculation of line\"},\"tab\":{\"type\":\"array\",\"items\":{"
    "\"type\":\"object\",\"required\":[\"thra1\",\"del1\",\"dec1\",\"thra2\",\"del2\",\"dec2"
    "\"],\"properties\":{\"thra2\":{\"type\":\"number\",\"description\":\"Pre-trigger "
    "threshold in units of "
    "dB\"},\"thra1\":{\"type\":\"number\",\"description\":\"Post-trigger threshold in"
    " units of dB\"},\"del1\":{\"type\":\"number\",\"description\":\"Length of "
    "constant post-trigger threshold elevation in units of "
    "m\"},\"del2\":{\"type\":\"number\",\"description\":\"Length of constant "
    "pre-trigger threshold elevation in units of "
    "m\"},\"dec1\":{\"type\":\"number\",\"description\":\"Decay of post-trigger "
    "threshold after del1 in units of "
    "dB/m\"},\"dec2\":{\"type\":\"number\",\"description\":\"Rise of pre-trigger "
    "threshold after del2 in units of dB/m\"}},\"description\":\"...\"},\"minItem"
    "s\":1,\"description\":\"...\",\"maxItems\":128},\"pre\":{\"type\":\"object\",\"requi"
    "red\":[\"nc\",\"a\"],\"properties\":{\"nc\":{\"type\":\"integer\",\"minimum\":0,\"maxi"
    "mum\":255,\"description\":\"Number of entries\"},\"a\":{\"type\":\"array\",\"items"
    "\":{\"type\":\"number\"},\"minItems\":1,\"description\":\"...\",\"maxItems\":2080}}"
    ",\"description\":\"Dyntrig pre values\"},\"post\":{\"type\":\"object\",\"required"
    "\":[\"nc\",\"a\"],\"properties\":{\"nc\":{\"type\":\"integer\",\"minimum\":0,\"maximum"
    "\":255,\"description\":\"Number of entries\"},\"a\":{\"type\":\"array\",\"items\":{"
    "\"type\":\"number\"},\"minItems\":1,\"description\":\"...\",\"maxItems\":2080}},\"d"
    "escription\":\"Dyntrig post "
    "values\"},\"scale\":{\"type\":\"number\",\"description\":\"Scale in units of dB,"
    " for calculation of line: line number "
    "N=(amplitude-offset)/scale\"}},\"description\":\"Dyntrig\"}"
    ,
    /* example: */
    "{\"offset\":0.0,\"tab\":[{\"thra2\":0.0,\"thra1\":0.0,\"del1\":0.0,\"del2\":0.0,\"d"
    "ec1\":0.0,\"dec2\":0.0}],\"pre\":{\"nc\":128,\"a\":[78,86,126,134,31]},\"post\":{"
    "\"nc\":128,\"a\":[78,86,126,134,31]},\"scale\":1.0}"
};

// Details about echo files
const MetaDataItemInfo RDB_RIEGL_ECHO_INFO = {
    /* name        */ "riegl.echo_info",
    /* title       */ "Echo Information",
    /* description */ "Details about echo files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Echo Information\",\"$schema\":\"http://json-sch"
    "ema.org/draft-04/schema#\",\"required\":[\"echo_file\"],\"properties\":{\"echo"
    "_file\":{\"type\":\"object\",\"required\":[\"file_extension\"],\"properties\":{\"f"
    "ile_uuid\":{\"type\":\"string\",\"description\":\"File's Universally Unique "
    "Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Echo file "
    "extension, without the leading dot\"}}}},\"description\":\"Details about "
    "echo files\"}"
    ,
    /* example: */
    "{\"echo_file\":{\"file_uuid\":\"26a03615-67c0-4bea-8fe8-c577378fe661\",\"file"
    "_extension\":\"owp\"}}"
};

// Details for exponential decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_EXPONENTIAL_DECOMPOSITION = {
    /* name        */ "riegl.exponential_decomposition",
    /* title       */ "Exponential Decomposition",
    /* description */ "Details for exponential decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Exponential Decomposition\",\"definitions\":{\"c"
    "hannel\":{\"type\":\"object\",\"required\":[\"delay\",\"scale\",\"parameter\"],\"pro"
    "perties\":{\"scale\":{\"type\":\"number\",\"description\":\"amplitude "
    "calibration\"},\"delay\":{\"type\":\"number\",\"description\":\"delay "
    "calibration in seconds\"},\"parameter\":{\"type\":\"object\",\"required\":[\"A\","
    "\"B\",\"gamma\",\"omega\"],\"properties\":{\"B\":{\"type\":\"array\",\"items\":{\"type\""
    ":\"number\"},\"description\":\"imaginary part of amplitude factor in units "
    "of full-scale\"},\"A\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"descrip"
    "tion\":\"real part of amplitude factor in units of full-scale\"},\"gamma\":"
    "{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"decay in 1/se"
    "cond\"},\"omega\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\""
    ":\"angular frequency in Hz\"}},\"description\":\"parameters of the syswave "
    "exponential sum\"},\"a_lin\":{\"type\":\"number\",\"maximum\":1,\"minimum\":0,\"ex"
    "clusiveMaximum\":false,\"description\":\"relative linear amplitude range ["
    "0..1]\",\"exclusiveMinimum\":false}}}},\"$schema\":\"http://json-schema.org/"
    "draft-04/schema#\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/definitio"
    "ns/channel\",\"description\":\"one field per channel, field name is "
    "channel index\"}},\"additionalProperties\":false,\"description\":\"Details "
    "for exponential decomposition of full waveform data\"}"
    ,
    /* example: */
    "{\"1\":{\"scale\":1.0,\"delay\":3.5e-09,\"parameter\":{\"B\":[-3.9,0.0,-0.3],\"A\""
    ":[0.9,0.3,-1.3],\"gamma\":[-1094726528.0,-769562752.0,-848000064.0],\"ome"
    "ga\":[352020896.0,3647927552.0,-1977987072.0]},\"a_lin\":0.9},\"0\":{\"scale"
    "\":1.0,\"delay\":3.783458418887631e-09,\"parameter\":{\"B\":[-3.9813032150268"
    "555,0.08622030913829803,-0.3152860999107361],\"A\":[0.9772450923919678,0"
    ".3354335129261017,-1.312678575515747],\"gamma\":[-1094726528.0,-76956275"
    "2.0,-848000064.0],\"omega\":[352020896.0,3647927552.0,-1977987072.0]},\"a"
    "_lin\":0.27}}"
};

// Details for Gaussian decomposition of full waveform data
const MetaDataItemInfo RDB_RIEGL_GAUSSIAN_DECOMPOSITION = {
    /* name        */ "riegl.gaussian_decomposition",
    /* title       */ "Gaussian Decomposition",
    /* description */ "Details for Gaussian decomposition of full waveform data",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Gaussian Decomposition\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"required\":[\"amplitude_lsb_low_power\","
    "\"amplitude_lsb_high_power\",\"amplitude_db\",\"range_offset_sec_low_power\""
    ",\"range_offset_sec_high_power\"],\"properties\":{\"amplitude_db\":{\"type\":\""
    "array\",\"items\":{\"type\":\"number\"}},\"range_offset_sec_low_power\":{\"type\""
    ":\"array\",\"items\":{\"type\":\"number\"}},\"amplitude_lsb_low_power\":{\"type\":"
    "\"array\",\"items\":{\"type\":\"number\"}},\"range_offset_sec_high_power\":{\"typ"
    "e\":\"array\",\"items\":{\"type\":\"number\"}},\"amplitude_lsb_high_power\":{\"typ"
    "e\":\"array\",\"items\":{\"type\":\"number\"}}},\"description\":\"riegl.gaussian_d"
    "ecomposition contains information relevant for extracting calibrated "
    "amplitudes and calibrated ranges from a Gaussian decomposition of full"
    " waveform data. This information is contained in a table with five "
    "columns. Two columns are to be used as input: amplitude_lsb_low_power "
    "and amplitude_lsb_high_power. The other three columns provide the "
    "outputs. Amplitude_db gives the calibrated amplitude in the optical "
    "regime in decibels. The range offset columns provide additive range "
    "offsets, given in units of seconds, for each channel.\"}"
    ,
    /* example: */
    "{\"amplitude_db\":[],\"range_offset_sec_low_power\":[],\"amplitude_lsb_low_"
    "power\":[],\"range_offset_sec_high_power\":[],\"amplitude_lsb_high_power\":"
    "[]}"
};

// Point cloud georeferencing information
const MetaDataItemInfo RDB_RIEGL_GEO_TAG = {
    /* name        */ "riegl.geo_tag",
    /* title       */ "Geo Tag",
    /* description */ "Point cloud georeferencing information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Geo Tag\""
    ",\"properties\":{\"pose\":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":"
    "{\"type\":\"number\",\"description\":\"columns\"},\"maxItems\":4,\"description\":\""
    "rows\",\"minItems\":4},\"maxItems\":4,\"description\":\"Coordinate "
    "Transformation Matrix to transform from File Coordinate System to "
    "Global Coordinate Reference System. 4x4 matrix stored as two "
    "dimensional array, row major order.\",\"minItems\":4},\"crs\":{\"type\":\"obje"
    "ct\",\"properties\":{\"name\":{\"type\":\"string\",\"description\":\"Coordinate "
    "reference system "
    "name\"},\"epsg\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"EPSG "
    "code\"},\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known Text\\\" "
    "string, OGC WKT dialect (see http://www.opengeospatial.org/standards/w"
    "kt-crs)\"}},\"description\":\"Global Coordinate Reference System. Please "
    "note that only 3D Cartesian Coordinate Systems are "
    "allowed.\"}},\"description\":\"Point cloud georeferencing "
    "information\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"pose\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,4063"
    "882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847,113"
    "8787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196],[0"
    ".0,0.0,0.0,1.0]],\"crs\":{\"name\":\"WGS84 "
    "Geocentric\",\"epsg\":4978,\"wkt\":\"GEOCCS[\\\"WGS84 Geocentric\\\",DATUM[\\\"WGS"
    "84\\\",SPHEROID[\\\"WGS84\\\",6378137.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\"
    "\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000"
    "000000000,AUTHORITY[\\\"EPSG\\\",\\\"8901\\\"]],UNIT[\\\"Meter\\\",1.0000000000000"
    "0000000,AUTHORITY[\\\"EPSG\\\",\\\"9001\\\"]],AXIS[\\\"X\\\",OTHER],AXIS[\\\"Y\\\",EAS"
    "T],AXIS[\\\"Z\\\",NORTH],AUTHORITY[\\\"EPSG\\\",\\\"4978\\\"]]\"}}"
};

// Geometric scale factor applied to point coordinates
const MetaDataItemInfo RDB_RIEGL_GEOMETRIC_SCALE_FACTOR = {
    /* name        */ "riegl.geometric_scale_factor",
    /* title       */ "Geometric Scale Factor",
    /* description */ "Geometric scale factor applied to point coordinates",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"type\":\"number\",\""
    "exclusiveMinimum\":true,\"description\":\"Geometric scale factor applied "
    "to point coordinates\",\"minimum\":0}"
    ,
    /* example: */
    "1.0"
};

// Parameters used for georeferencing of the point cloud
const MetaDataItemInfo RDB_RIEGL_GEOREFERENCING_PARAMETERS = {
    /* name        */ "riegl.georeferencing_parameters",
    /* title       */ "Georeferencing Parameters",
    /* description */ "Parameters used for georeferencing of the point cloud",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Georefer"
    "encing Parameters\",\"properties\":{\"trajectory_offsets\":{\"type\":\"object\""
    ",\"properties\":{\"offset_height\":{\"type\":\"number\",\"description\":\"[m]\"},\""
    "offset_pitch\":{\"type\":\"number\",\"description\":\"[deg]\"},\"version\":{\"type"
    "\":\"integer\",\"description\":\"Meaning of offset values and how to apply "
    "them; version 0: "
    "Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), "
    "version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch"
    ")*Rx(offset_roll)\"},\"offset_north\":{\"type\":\"number\",\"description\":\"[m]"
    "\"},\"offset_yaw\":{\"type\":\"number\",\"description\":\"[deg]\"},\"offset_east\":"
    "{\"type\":\"number\",\"description\":\"[m]\"},\"offset_time\":{\"type\":\"number\",\""
    "description\":\"[s]\"},\"offset_roll\":{\"type\":\"number\",\"description\":\"[deg"
    "]\"}},\"description\":\"Correction offsets applied to the trajectory data\""
    "},\"trajectory_file\":{\"type\":\"object\",\"required\":[\"file_extension\"],\"pr"
    "operties\":{\"file_uuid\":{\"type\":\"string\",\"description\":\"File's "
    "Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Trajectory "
    "file extension, without the leading dot\"}},\"description\":\"Trajectory "
    "data used for georeferencing of the point cloud\"},\"socs_to_body_matrix"
    "\":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"de"
    "scription\":\"columns\"},\"maxItems\":4,\"description\":\"rows\",\"minItems\":4},"
    "\"maxItems\":4,\"description\":\"Coordinate Transformation Matrix to "
    "transform from Scanner's Own Coordinate System to Body Coordinate "
    "System. 4x4 matrix stored as two dimensional array, row major order.\","
    "\"minItems\":4},\"body_coordinate_system_type\":{\"type\":\"string\",\"descript"
    "ion\":\"BODY coordinate frame (NED: North-East-Down, ENU: "
    "East-North-Up), default: NED\",\"enum\":[\"NED\",\"ENU\"]},\"socs_to_rocs_matr"
    "ix\":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\""
    "description\":\"columns\"},\"maxItems\":4,\"description\":\"rows\",\"minItems\":4"
    "},\"maxItems\":4,\"description\":\"Coordinate Transformation Matrix to "
    "transform from Scanner's Own Coordinate System to Record Coordinate "
    "System. 4x4 matrix stored as two dimensional array, row major "
    "order.\",\"minItems\":4}},\"description\":\"Parameters used for "
    "georeferencing of the point cloud\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"trajectory_offsets\":{\"offset_height\":-0.2,\"offset_pitch\":0.01,\"versi"
    "on\":0,\"offset_north\":0.07,\"offset_yaw\":-0.45,\"offset_east\":0.15,\"offse"
    "t_time\":18.007,\"offset_roll\":0.03},\"trajectory_file\":{\"file_uuid\":\"93a"
    "03615-66c0-4bea-8ff8-c577378fe660\",\"file_extension\":\"pofx\"},\"socs_to_b"
    "ody_matrix\":[[-0.269827776749716,-0.723017716139738,0.635954678449952,"
    "0.0],[0.962908599449764,-0.20260517250352,0.178208229833847,0.0],[0.0,"
    "0.660451759194338,0.7508684796801,0.0],[0.0,0.0,0.0,1.0]],\"body_coordi"
    "nate_system_type\":\"NED\"}"
};

// Details about GNSS processing
const MetaDataItemInfo RDB_RIEGL_GNSS_PROCESSING_INFO = {
    /* name        */ "riegl.gnss_processing_info",
    /* title       */ "GNSS Processing Information",
    /* description */ "Details about GNSS processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"GNSS "
    "Processing Information\",\"properties\":{\"base_stations\":{\"type\":\"array\","
    "\"items\":{\"type\":\"object\",\"required\":[\"latitude\",\"longitude\",\"altitude\""
    "],\"properties\":{\"longitude\":{\"type\":\"number\",\"description\":\"Longitude "
    "[deg]\"},\"altitude\":{\"type\":\"number\",\"description\":\"Altitude "
    "[m]\"},\"latitude\":{\"type\":\"number\",\"description\":\"Latitude "
    "[deg]\"}},\"description\":\"GNSS base station "
    "details\"},\"description\":\"List of GNSS base "
    "stations\"}},\"description\":\"Details about GNSS "
    "processing\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"base_stations\":[{\"longitude\":15.663389822,\"altitude\":379.80041503906"
    "25,\"latitude\":48.658391162}]}"
};

// Map of item names
const MetaDataItemInfo RDB_RIEGL_ITEM_NAMES = {
    /* name        */ "riegl.item_names",
    /* title       */ "Item Names",
    /* description */ "Map of item names",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Item Names\",\"$schema\":\"http://json-schema.or"
    "g/draft-04/schema#\",\"patternProperties\":{\"^-?[0-9]+$\":{\"type\":\"string\""
    ",\"description\":\"One field per item, field name is item id, field value"
    " is item name\"}},\"additionalProperties\":false,\"description\":\"Map of "
    "item names\"}"
    ,
    /* example: */
    "{\"1\":\"Name of item with id 1\",\"2\":\"Name of item with id 2\",\"47\":\"Name "
    "of item with id 47\",\"-1\":\"Name of item with id -1\"}"
};

// Lag angle depended range and reflectance correction
const MetaDataItemInfo RDB_RIEGL_LAG_ANGLE_CORRECTION = {
    /* name        */ "riegl.lag_angle_correction",
    /* title       */ "Lag Angle Correction",
    /* description */ "Lag angle depended range and reflectance correction",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Lag Angle Correction\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"required\":[\"ranges_m\",\"range_correction"
    "_m\",\"reflectance_correction_db\"],\"properties\":{\"reflectance_correction"
    "_db\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Reflecta"
    "nce correction [db] as a function of the range and for applied scanrat"
    "e\"},\"ranges_m\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\""
    ":\"Range [m]\"},\"range_correction_m\":{\"type\":\"array\",\"items\":{\"type\":\"nu"
    "mber\"},\"description\":\"Range correction [m] as a function of the range "
    "and for applied scanrate, which defines the lag "
    "angle\"}},\"description\":\"Lag angle depended range and reflectance "
    "correction\"}"
    ,
    /* example: */
    "{\"reflectance_correction_db\":[0.0,0.0,0.16,0.46,1.18,2.97,2.97],\"range"
    "s_m\":[0.0,4.795,31.964,63.928,95.892,127.856,10000.0],\"range_correctio"
    "n_m\":[0.0,0.0,0.0004,0.0023,0.0039,0.0053,0.0053]}"
};

// License keys for software features
const MetaDataItemInfo RDB_RIEGL_LICENSES = {
    /* name        */ "riegl.licenses",
    /* title       */ "Software License Keys",
    /* description */ "License keys for software features",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Software License Keys\",\"$schema\":\"http://jso"
    "n-schema.org/draft-04/schema#\",\"patternProperties\":{\"^.*$\":{\"type\":\"ar"
    "ray\",\"items\":{\"type\":\"string\",\"description\":\"License key (example: "
    "'46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73')\"},\"minItems\":1,\"description\":\"Each field of the object "
    "represents a feature and holds a list of license keys, where the field"
    " name is the feature "
    "name.\"}},\"additionalProperties\":false,\"description\":\"License keys for "
    "software features\"}"
    ,
    /* example: */
    "{\"Full Waveform Analysis Topography\":[\"0FD5FF07 - 011A1255 - 9F76CACA "
    "- 8D2ED557\"],\"Full Waveform Analysis Topography with GPU "
    "support\":[\"8AB44126 - 23B92250 - 16E2689F - "
    "34EF7E7B\"],\"Georeferencing\":[\"46AE032A - 39882AC4 - 9EC0A184 - "
    "6F163D73\"],\"MTA resolution\":[\"468E020A - 39A922E4 - B681A184 - "
    "673E3D72\"]}"
};

// Parameters for MTA processing
const MetaDataItemInfo RDB_RIEGL_MTA_SETTINGS = {
    /* name        */ "riegl.mta_settings",
    /* title       */ "MTA Settings",
    /* description */ "Parameters for MTA processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"MTA Sett"
    "ings\",\"properties\":{\"zone_width\":{\"type\":\"number\",\"minimum\":0,\"descrip"
    "tion\":\"Width of a MTA zone in meter.\"},\"zone_count\":{\"type\":\"integer\","
    "\"minimum\":0,\"maximum\":255,\"description\":\"Maximum number of MTA zones.\""
    "},\"modulation_depth\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Depth"
    " of pulse position modulation in meter.\"}},\"description\":\"Parameters "
    "for MTA processing\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"zone_width\":149.896225,\"zone_count\":23,\"modulation_depth\":9.368514}"
};

// Lookup table for range correction based on raw range
const MetaDataItemInfo RDB_RIEGL_NEAR_RANGE_CORRECTION = {
    /* name        */ "riegl.near_range_correction",
    /* title       */ "Near Range Correction Table",
    /* description */ "Lookup table for range correction based on raw range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Near Range Correction Table\",\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"required\":[\"delta\",\"content\"],\"p"
    "roperties\":{\"delta\":{\"type\":\"number\",\"description\":\"Delta between "
    "table entries [m], first entry is at range = 0 m\"},\"content\":{\"type\":\""
    "array\",\"items\":{\"type\":\"number\"},\"minItems\":1,\"description\":\"Correctio"
    "n value [m] to be added to the raw "
    "range\",\"maxItems\":2000}},\"description\":\"Lookup table for range "
    "correction based on raw range\"}"
    ,
    /* example: */
    "{\"delta\":0.512,\"content\":[0.0]}"
};

// Standard deviation for range and amplitude as a function of amplitude
const MetaDataItemInfo RDB_RIEGL_NOISE_ESTIMATES = {
    /* name        */ "riegl.noise_estimates",
    /* title       */ "Noise Estimates",
    /* description */ "Standard deviation for range and amplitude as a function of amplitude",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Noise Estimates\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"required\":[\"amplitude\",\"range_sigma\",\"amplit"
    "ude_sigma\"],\"properties\":{\"amplitude_sigma\":{\"type\":\"array\",\"items\":{\""
    "type\":\"number\"},\"description\":\"Sigma amplitude [dB]\"},\"amplitude\":{\"ty"
    "pe\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Amplitude [dB]\"},"
    "\"range_sigma\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":"
    "\"Sigma range [m]\"}},\"description\":\"Standard deviation for range and "
    "amplitude as a function of amplitude\"}"
    ,
    /* example: */
    "{\"amplitude_sigma\":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0.4"
    "32,0.39,0.354],\"amplitude\":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10"
    ".0],\"range_sigma\":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.0"
    "18,0.016,0.014]}"
};

// Notch filter parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_NOTCH_FILTER = {
    /* name        */ "riegl.notch_filter",
    /* title       */ "Notch Filter",
    /* description */ "Notch filter parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Notch Filter\",\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\",\"required\":[\"range_minimum\",\"range_maximum\",\"amp"
    "litude_maximum\"],\"properties\":{\"range_maximum\":{\"type\":\"number\",\"descr"
    "iption\":\"Maximum range "
    "[m]\"},\"range_minimum\":{\"type\":\"number\",\"description\":\"Minimum range [m"
    "]\"},\"amplitude_maximum\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Ma"
    "ximum amplitude [dB]\"}},\"description\":\"Notch filter parameters for "
    "window glass echoes\"}"
    ,
    /* example: */
    "{\"range_maximum\":0.2,\"range_minimum\":-0.5,\"amplitude_maximum\":18.0}"
};

// Details about the pixels contained in the file
const MetaDataItemInfo RDB_RIEGL_PIXEL_INFO = {
    /* name        */ "riegl.pixel_info",
    /* title       */ "Pixel Information",
    /* description */ "Details about the pixels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pixel Information\",\"definitions\":{\"pixel_siz"
    "e\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"minimum\":0,\"description\":"
    "\"Length of pixel edge [m].\"},\"maxItems\":2,\"description\":\"Size of pixel"
    "s.\",\"minItems\":2}},\"$schema\":\"http://json-schema.org/draft-04/schema#\""
    ",\"required\":[\"size\"],\"properties\":{\"size\":{\"$ref\":\"#/definitions/pixel"
    "_size\",\"description\":\"Size of pixels in file coordinate system.\"},\"siz"
    "e_llcs\":{\"$ref\":\"#/definitions/pixel_size\",\"description\":\"Size of "
    "pixels in a locally levelled cartesian coordinate system (xy). This is"
    " only used for pixels based on a map "
    "projection.\"}},\"description\":\"Details about the pixels contained in "
    "the file\"}"
    ,
    /* example: */
    "{\"size\":[0.5971642834779395,0.5971642834779395],\"size_llcs\":[0.5156575"
    "252891171,0.5130835356683303]}"
};

// Details about the plane patch matching process
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_MATCHING = {
    /* name        */ "riegl.plane_patch_matching",
    /* title       */ "Plane Patch Matching",
    /* description */ "Details about the plane patch matching process",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Plane Patch Matching\",\"definitions\":{\"file_r"
    "eference\":{\"type\":\"object\",\"required\":[\"file_uuid\",\"file_path\"],\"prope"
    "rties\":{\"file_uuid\":{\"type\":\"string\",\"description\":\"Plane patch file's"
    " Universally Unique Identifier (RFC "
    "4122)\"},\"file_path\":{\"type\":\"string\",\"description\":\"Path of the plane "
    "patch file relative to the match file\"}},\"description\":\"Reference to a"
    " plane patch file\"}},\"$schema\":\"http://json-schema.org/draft-04/schema"
    "#\",\"required\":[\"plane_patch_file_one\",\"plane_patch_file_two\"],\"propert"
    "ies\":{\"plane_patch_file_two\":{\"$ref\":\"#/definitions/file_reference\",\"d"
    "escription\":\"Reference to the plane patch file two\"},\"plane_patch_file"
    "_one\":{\"$ref\":\"#/definitions/file_reference\",\"description\":\"Reference "
    "to the plane patch file one\"}},\"description\":\"Details about the plane "
    "patch matching process\"}"
    ,
    /* example: */
    "{\"plane_patch_file_two\":{\"file_uuid\":\"fa47d509-a64e-49ce-8b14-ff3130fb"
    "efa9\",\"file_path\":\"project.ptch\"},\"plane_patch_file_one\":{\"file_uuid\":"
    "\"810f5d57-eccf-49ed-b07a-0cdd109b4213\",\"file_path\":\"Record009_Line001/"
    "191025_121410_Scanner_1.ptch\"}}"
};

// Statistics about plane patches found by plane patch extractor
const MetaDataItemInfo RDB_RIEGL_PLANE_PATCH_STATISTICS = {
    /* name        */ "riegl.plane_patch_statistics",
    /* title       */ "Plane Patch Statistics",
    /* description */ "Statistics about plane patches found by plane patch extractor",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Plane "
    "Patch Statistics\",\"properties\":{\"total_horizontal_area\":{\"type\":\"numbe"
    "r\",\"description\":\"sum of all plane patch areas projected to horizontal"
    " plane [m\\u00b2]\"},\"total_area\":{\"type\":\"number\",\"description\":\"sum of"
    " all plane patch areas [m\\u00b2]\"}},\"description\":\"Statistics about "
    "plane patches found by plane patch extractor\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"total_horizontal_area\":13954.601,\"total_area\":14007.965}"
};

// Settings and classes for plane slope classification
const MetaDataItemInfo RDB_RIEGL_PLANE_SLOPE_CLASS_INFO = {
    /* name        */ "riegl.plane_slope_class_info",
    /* title       */ "Plane Slope Class Info",
    /* description */ "Settings and classes for plane slope classification",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Plane Slope Class Info\",\"definitions\":{\"meth"
    "od-1\":{\"type\":\"object\",\"required\":[\"plane_classification_method\",\"maxi"
    "mum_inclination_angle_horizontal\"],\"properties\":{\"maximum_inclination_"
    "angle_horizontal\":{\"type\":\"number\",\"minimum\":-360.0,\"maximum\":360.0,\"d"
    "escription\":\"maximum inclination angle of horizontal plane patches [de"
    "g]\"},\"plane_classification_method\":{\"type\":\"integer\",\"minimum\":1,\"maxi"
    "mum\":1,\"description\":\"method ID (=1)\"}},\"description\":\"Classification "
    "method 1\"},\"method-2\":{\"type\":\"object\",\"required\":[\"plane_classificati"
    "on_method\",\"sloping_plane_classes_minimum_angle\",\"sloping_plane_classe"
    "s_maximum_angle\"],\"properties\":{\"sloping_plane_classes_minimum_angle\":"
    "{\"type\":\"number\",\"minimum\":-360.0,\"maximum\":360.0,\"description\":\"minim"
    "um inclination angle of sloping plane patches [deg]\"},\"sloping_plane_c"
    "lasses_maximum_angle\":{\"type\":\"number\",\"minimum\":-360.0,\"maximum\":360."
    "0,\"description\":\"maximum inclination angle of sloping plane patches [d"
    "eg]\"},\"plane_classification_method\":{\"type\":\"integer\",\"minimum\":2,\"max"
    "imum\":2,\"description\":\"method ID (=2)\"}},\"description\":\"Classification"
    " method 2\"}},\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"requ"
    "ired\":[\"settings\",\"classes\"],\"properties\":{\"classes\":{\"type\":\"object\","
    "\"patternProperties\":{\"^[0-9]+$\":{\"type\":\"string\",\"description\":\"one "
    "field per class, field name is class number, field value is class "
    "name\"}},\"additionalProperties\":false,\"description\":\"Class definition t"
    "able\"},\"settings\":{\"type\":\"object\",\"oneOf\":[{\"$ref\":\"#/definitions/met"
    "hod-1\"},{\"$ref\":\"#/definitions/method-2\"}],\"description\":\"Classificati"
    "on settings, details see documentation of "
    "rdbplanes\"}},\"description\":\"Settings and classes for plane slope "
    "classification\"}"
    ,
    /* example: */
    "{\"classes\":{\"2\":\"sloping, pointing up and east\",\"11\":\"sloping, "
    "pointing down and south\",\"5\":\"sloping, pointing up and "
    "west\",\"3\":\"sloping, pointing up and south\",\"8\":\"vertical, pointing "
    "north\",\"4\":\"sloping, pointing up and north\",\"1\":\"horizontal, pointing "
    "up\",\"12\":\"sloping, pointing down and north\",\"9\":\"vertical, pointing "
    "west\",\"10\":\"sloping, pointing down and east\",\"6\":\"vertical, pointing "
    "east\",\"14\":\"horizontal, pointing down\",\"7\":\"vertical, pointing "
    "south\",\"13\":\"sloping, pointing down and west\"},\"settings\":{\"sloping_pl"
    "ane_classes_minimum_angle\":10.0,\"sloping_plane_classes_maximum_angle\":"
    "70.0,\"plane_classification_method\":2}}"
};

// Grouping and sorting of point attributes for visualization purposes
const MetaDataItemInfo RDB_RIEGL_POINT_ATTRIBUTE_GROUPS = {
    /* name        */ "riegl.point_attribute_groups",
    /* title       */ "Point Attribute Groups",
    /* description */ "Grouping and sorting of point attributes for visualization purposes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Point Attribute Groups\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"patternProperties\":{\"^.*$\":{\"type\":\"a"
    "rray\",\"items\":{\"type\":\"string\",\"description\":\"Point attribute full "
    "name or name pattern (perl regular expression "
    "syntax)\"},\"minItems\":1,\"description\":\"Each field of the object "
    "represents a point attribute group and holds a list of point "
    "attributes, where the field name is the group "
    "name.\"}},\"additionalProperties\":false,\"description\":\"Grouping and "
    "sorting of point attributes for visualization purposes\"}"
    ,
    /* example: */
    "{\"Primary Attributes\":[\"riegl.reflectance\",\"riegl.amplitude\",\"riegl.de"
    "viation\"],\"Coordinates/Vectors\":[\"riegl.xyz\",\"riegl.range\",\"riegl.thet"
    "a\",\"riegl.phi\"],\"Secondary "
    "Attributes\":[\"riegl.mirror_facet\",\"riegl.waveform_available\"],\"Other A"
    "ttributes\":[\"riegl.selected\",\"riegl.visible\"],\"Time\":[\"riegl.timestamp"
    "\"]}"
};

// Details about point cloud files
const MetaDataItemInfo RDB_RIEGL_POINTCLOUD_INFO = {
    /* name        */ "riegl.pointcloud_info",
    /* title       */ "Point Cloud Information",
    /* description */ "Details about point cloud files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Point "
    "Cloud Information\",\"properties\":{\"comments\":{\"type\":\"string\",\"descript"
    "ion\":\"Comments\"},\"field_of_application\":{\"type\":\"string\",\"description\""
    ":\"Field of application\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS"
    "\",\"ALS\",\"BLS\",\"ILS\"]},\"project\":{\"type\":\"string\",\"description\":\"Projec"
    "t name\"}},\"description\":\"Details about point cloud "
    "files\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"comments\":\"Line 3\",\"field_of_application\":\"ALS\",\"project\":\"Campaign "
    "4\"}"
};

// Estimated position and orientation information
const MetaDataItemInfo RDB_RIEGL_POSE_ESTIMATION = {
    /* name        */ "riegl.pose_estimation",
    /* title       */ "Pose Estimation",
    /* description */ "Estimated position and orientation information",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pose Estimation\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"required\":[\"orientation\"],\"properties\":{\"bar"
    "ometric_height_amsl\":{\"type\":\"number\",\"description\":\"Altitude "
    "determined based on the atmospheric pressure according to the standard"
    " atmosphere laws [m].\"},\"orientation\":{\"type\":\"object\",\"required\":[\"ro"
    "ll\",\"pitch\",\"yaw\",\"roll_accuracy\",\"pitch_accuracy\",\"yaw_accuracy\"],\"pr"
    "operties\":{\"roll_accuracy\":{\"type\":\"number\",\"minimum\":0,\"description\":"
    "\"Roll angle accuracy [deg]\",\"exclusiveMinimum\":true},\"yaw\":{\"type\":\"nu"
    "mber\",\"minimum\":-360,\"maximum\":360,\"description\":\"Yaw angle about "
    "scanner Z-axis [deg]\"},\"pitch\":{\"type\":\"number\",\"minimum\":-360,\"maximu"
    "m\":360,\"description\":\"Pitch angle about scanner Y-axis [deg]\"},\"pitch_"
    "accuracy\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Pitch angle "
    "accuracy [deg]\",\"exclusiveMinimum\":true},\"roll\":{\"type\":\"number\",\"mini"
    "mum\":-360,\"maximum\":360,\"description\":\"Roll angle about scanner X-axis"
    " "
    "[deg]\"},\"yaw_accuracy\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Yaw"
    " angle accuracy "
    "[deg]\",\"exclusiveMinimum\":true}},\"description\":\"Orientation values and"
    " orientation accuracies, measured with IMU or inclination sensors.\"},\""
    "position\":{\"type\":\"object\",\"required\":[\"coordinate_1\",\"coordinate_2\",\""
    "coordinate_3\",\"horizontal_accuracy\",\"vertical_accuracy\",\"crs\"],\"proper"
    "ties\":{\"coordinate_2\":{\"type\":\"number\",\"description\":\"Coordinate 2 as "
    "defined by axis 2 of the specified CRS (e.g., Y, Longitude)\"},\"vertica"
    "l_accuracy\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Vertical "
    "accuracy [m]\",\"exclusiveMinimum\":true},\"crs\":{\"type\":\"object\",\"require"
    "d\":[\"epsg\"],\"properties\":{\"epsg\":{\"type\":\"integer\",\"minimum\":0,\"descri"
    "ption\":\"EPSG code\"},\"wkt\":{\"type\":\"string\",\"description\":\"\\\"Well-Known"
    " Text\\\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta"
    "ndards/wkt-crs)\"}},\"description\":\"Global Coordinate Reference System\"}"
    ",\"horizontal_accuracy\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Hor"
    "izontal accuracy [m]\",\"exclusiveMinimum\":true},\"coordinate_1\":{\"type\":"
    "\"number\",\"description\":\"Coordinate 1 as defined by axis 1 of the "
    "specified CRS (e.g., X, "
    "Latitude)\"},\"coordinate_3\":{\"type\":\"number\",\"description\":\"Coordinate "
    "3 as defined by axis 3 of the specified CRS (e.g., Z, "
    "Altitude)\"}},\"description\":\"Position coordinates and position accuracy"
    " values as measured by GNSS in the specified Coordinate Reference "
    "System (CRS)\"}},\"description\":\"Estimated position and orientation "
    "information as measured by GNSS, IMU or inclination sensors\"}"
    ,
    /* example: */
    "{\"barometric_height_amsl\":386.7457796227932,\"orientation\":{\"roll_accur"
    "acy\":0.009433783936875745,\"yaw\":101.87293630292045,\"pitch\":1.509153024"
    "827064,\"pitch_accuracy\":0.009433783936875745,\"roll\":3.14743073066123,\""
    "yaw_accuracy\":1.0094337839368757},\"position\":{\"coordinate_2\":15.645033"
    "406,\"vertical_accuracy\":1.3314999341964722,\"crs\":{\"epsg\":4979,\"wkt\":\"G"
    "EOGCS[\\\"WGS84 / Geographic\\\",DATUM[\\\"WGS84\\\",SPHEROID[\\\"WGS84\\\",637813"
    "7.000,298.257223563,AUTHORITY[\\\"EPSG\\\",\\\"7030\\\"]],AUTHORITY[\\\"EPSG\\\",\\"
    "\"6326\\\"]],PRIMEM[\\\"Greenwich\\\",0.0000000000000000,AUTHORITY[\\\"EPSG\\\",\\"
    "\"8901\\\"]],UNIT[\\\"Degree\\\",0.01745329251994329547,AUTHORITY[\\\"EPSG\\\",\\\""
    "9102\\\"]],AXIS[\\\"Latitude\\\",NORTH],AXIS[\\\"Longitude\\\",EAST],AUTHORITY[\\"
    "\"EPSG\\\",\\\"4979\\\"]]\"},\"horizontal_accuracy\":0.810699999332428,\"coordina"
    "te_1\":48.655799473,\"coordinate_3\":362.7124938964844}}"
};

// Details on position and orientation sensors
const MetaDataItemInfo RDB_RIEGL_POSE_SENSORS = {
    /* name        */ "riegl.pose_sensors",
    /* title       */ "Pose Sensors",
    /* description */ "Details on position and orientation sensors",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pose Sensors\",\"definitions\":{\"vector\":{\"type"
    "\":\"array\",\"items\":{\"type\":\"number\",\"description\":\"Index 0=X, 1=Y, 2=Z "
    "component\"},\"minItems\":3,\"maxItems\":3}},\"$schema\":\"http://json-schema."
    "org/draft-04/schema#\",\"required\":[\"gyroscope\",\"accelerometer\",\"magneti"
    "c_field_sensor\"],\"properties\":{\"accelerometer\":{\"type\":\"object\",\"requi"
    "red\":[\"unit\",\"x_axis\",\"y_axis\",\"z_axis\",\"offset\",\"origin\",\"relative_no"
    "nlinearity\"],\"properties\":{\"origin\":{\"$ref\":\"#/definitions/vector\",\"de"
    "scription\":\"Sensor origin in SOCS [m] at frame angle = 0\"},\"relative_n"
    "onlinearity\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Relative "
    "nonlinearity, dimensionless\"},\"unit\":{\"type\":\"number\",\"minimum\":0,\"des"
    "cription\":\"Unit of raw data and calibration values, 1 LSB in 9.81 m/s\\"
    "u00b2\",\"exclusiveMinimum\":true},\"z_axis\":{\"$ref\":\"#/definitions/vector"
    "\",\"description\":\"Sensitive Z axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement values\"},\"y_axis\":{\"$ref\":\"#/definiti"
    "ons/vector\",\"description\":\"Sensitive Y axis of sensor at frame angle ="
    " 0\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive "
    "X axis of sensor at frame angle = 0\"}},\"description\":\"Accelerometer de"
    "tails\"},\"gyroscope\":{\"type\":\"object\",\"required\":[\"unit\",\"x_axis\",\"y_ax"
    "is\",\"z_axis\",\"offset\",\"origin\",\"relative_nonlinearity\"],\"properties\":{"
    "\"origin\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensor origin "
    "in SOCS [m] at frame angle = 0\"},\"relative_nonlinearity\":{\"$ref\":\"#/de"
    "finitions/vector\",\"description\":\"Relative nonlinearity, dimensionless\""
    "},\"unit\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Unit of raw data "
    "and calibration values, 1 LSB in rad/s\",\"exclusiveMinimum\":true},\"z_ax"
    "is\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive Z axis of "
    "sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement values\"},\"y_axis\":{\"$ref\":\"#/definiti"
    "ons/vector\",\"description\":\"Sensitive Y axis of sensor at frame angle ="
    " 0\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive "
    "X axis of sensor at frame angle = 0\"}},\"description\":\"Gyroscope detail"
    "s\"},\"magnetic_field_sensor\":{\"type\":\"object\",\"required\":[\"unit\",\"x_axi"
    "s\",\"y_axis\",\"z_axis\",\"offset\",\"fixed\",\"relative_nonlinearity\"],\"proper"
    "ties\":{\"relative_nonlinearity\":{\"$ref\":\"#/definitions/vector\",\"descrip"
    "tion\":\"Relative nonlinearity, dimensionless\"},\"unit\":{\"type\":\"number\","
    "\"minimum\":0,\"description\":\"Unit of raw data and calibration values, 1 "
    "LSB in nT\",\"exclusiveMinimum\":true},\"z_axis\":{\"$ref\":\"#/definitions/ve"
    "ctor\",\"description\":\"Sensitive Z axis of sensor at frame angle = "
    "0\"},\"offset\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Value to be"
    " subtracted from raw measurement values\"},\"y_axis\":{\"$ref\":\"#/definiti"
    "ons/vector\",\"description\":\"Sensitive Y axis of sensor at frame angle ="
    " 0\"},\"x_axis\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Sensitive "
    "X axis of sensor at frame angle = "
    "0\"},\"fixed\":{\"$ref\":\"#/definitions/vector\",\"description\":\"Distortion "
    "of magnetic field caused by non-rotating scanner "
    "part\"}},\"description\":\"Magnetic Field Sensor "
    "details\"}},\"description\":\"Details on position and orientation "
    "sensors\"}"
    ,
    /* example: */
    "{\"accelerometer\":{\"origin\":[0.026900000870227814,-0.03999999910593033,"
    "-0.08950000256299973],\"relative_nonlinearity\":[0.0,0.0,0.0],\"unit\":6.6"
    "66666740784422e-05,\"z_axis\":[1.639882206916809,15166.744140625,-116.99"
    "742889404297],\"offset\":[-733.3636474609375,58.969032287597656,1060.255"
    "0048828125],\"y_axis\":[-7.027288913726807,-44.12333679199219,14952.3701"
    "171875],\"x_axis\":[-15008.123046875,56.956390380859375,-60.517566680908"
    "2]},\"gyroscope\":{\"origin\":[0.026900000870227814,-0.03999999910593033,-"
    "0.08950000256299973],\"relative_nonlinearity\":[2.888176311444113e-07,1."
    "06274164579645e-07,-1.7186295080634935e-39],\"unit\":0.00014544410805683"
    "583,\"z_axis\":[0.555869996547699,119.22135162353516,0.467585027217865],"
    "\"offset\":[-50.92609786987305,146.15643310546875,62.4327278137207],\"y_a"
    "xis\":[-0.440765917301178,-0.7897399663925171,119.5894775390625],\"x_axi"
    "s\":[-121.195556640625,0.8219714164733887,0.2313031703233719]},\"magneti"
    "c_field_sensor\":{\"relative_nonlinearity\":[0.0,0.0,0.0],\"unit\":91.74311"
    "828613281,\"z_axis\":[0.00041987866279669106,7.876977906562388e-05,0.011"
    "407104320824146],\"offset\":[-23812.052734375,5606.57666015625,2493.2812"
    "5],\"y_axis\":[0.00027888521435670555,-0.011427424848079681,-5.204829722"
    "060822e-05],\"x_axis\":[-0.011162743903696537,-2.315962774446234e-05,0.0"
    "0016818844596855342],\"fixed\":[-1576.010498046875,1596.081787109375,0.0"
    "]}}"
};

// Laser pulse position modulation used for MTA resolution
const MetaDataItemInfo RDB_RIEGL_PULSE_POSITION_MODULATION = {
    /* name        */ "riegl.pulse_position_modulation",
    /* title       */ "Pulse Position Modulation",
    /* description */ "Laser pulse position modulation used for MTA resolution",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Pulse Position Modulation\",\"$schema\":\"http:/"
    "/json-schema.org/draft-04/schema#\",\"required\":[\"length\",\"num_mod_ampl\""
    ",\"pulse_interval\"],\"properties\":{\"phase_step\":{\"type\":\"integer\",\"minim"
    "um\":0,\"maximum\":255,\"description\":\"Step width in phase of modulation "
    "code from line to line\"},\"code_phase_mode\":{\"type\":\"integer\",\"minimum\""
    ":0,\"maximum\":255,\"description\":\"0: no synchronization, 1: toggle "
    "between 2 phases, 2: increment with phase_increment\"},\"pulse_interval\""
    ":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"minimum\":0},\"description\":\""
    "Explicit table of the pulse position modulation used for MTA "
    "resolution. Table gives times between successive laser pulses in secon"
    "ds.\"},\"num_mod_ampl\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":255,\"desc"
    "ription\":\"Number of different modulation amplitudes (2: binary modulat"
    "ion)\"},\"length\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":255,\"descripti"
    "on\":\"Length of code\"}},\"description\":\"Laser pulse position modulation "
    "used for MTA resolution\"}"
    ,
    /* example: */
    "{\"phase_step\":5,\"code_phase_mode\":2,\"pulse_interval\":[2.759375e-06,2.7"
    "59375e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.759375"
    "e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,"
    "2.759375e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821"
    "875e-06,2.759375e-06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-"
    "06,2.759375e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2."
    "821875e-06,2.759375e-06,2.821875e-06],\"num_mod_ampl\":2,\"length\":31}"
};

// Statistics about target distance wrt. SOCS origin
const MetaDataItemInfo RDB_RIEGL_RANGE_STATISTICS = {
    /* name        */ "riegl.range_statistics",
    /* title       */ "Range Statistics",
    /* description */ "Statistics about target distance wrt. SOCS origin",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Range Statistics\",\"$schema\":\"http://json-sch"
    "ema.org/draft-04/schema#\",\"required\":[\"minimum\",\"average\",\"maximum\",\"s"
    "td_dev\"],\"properties\":{\"average\":{\"type\":\"number\",\"description\":\"Avera"
    "ge value\"},\"minimum\":{\"type\":\"number\",\"description\":\"Minimum "
    "value\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum "
    "value\"},\"std_dev\":{\"type\":\"number\",\"description\":\"Standard "
    "deviation\"}},\"description\":\"Statistics about target distance wrt. SOCS"
    " origin\"}"
    ,
    /* example: */
    "{\"average\":15.49738,\"minimum\":0.919,\"maximum\":574.35,\"std_dev\":24.349}"
};

// Receiver Internals
const MetaDataItemInfo RDB_RIEGL_RECEIVER_INTERNALS = {
    /* name        */ "riegl.receiver_internals",
    /* title       */ "Receiver Internals",
    /* description */ "Receiver Internals",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Receiver Internals\",\"definitions\":{\"fp\":{\"ty"
    "pe\":\"object\",\"required\":[\"s\",\"w\"],\"properties\":{\"w\":{\"type\":\"array\",\"i"
    "tems\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"minItems\":5,\"maxItems"
    "\":5},\"minItems\":1,\"maxItems\":256},\"s\":{\"type\":\"array\",\"items\":{\"type\":"
    "\"array\",\"items\":{\"type\":\"number\"},\"minItems\":1,\"maxItems\":4096},\"minIt"
    "ems\":1,\"maxItems\":256}},\"description\":\"Fingerprint values\"},\"fp_table\""
    ":{\"type\":\"object\",\"required\":[\"ch\",\"tc\",\"nx\",\"ny\",\"tv\"],\"properties\":{"
    "\"tv\":{\"type\":\"array\",\"items\":{\"oneOf\":[{\"$ref\":\"#/definitions/fp_table"
    "_row\"},{\"type\":\"number\"}]},\"minItems\":1,\"maxItems\":2048},\"nx\":{\"type\":"
    "\"integer\",\"description\":\"number of x entries\",\"min\":1,\"max\":2048},\"ch\""
    ":{\"type\":\"integer\",\"description\":\"channel "
    "number\",\"min\":0,\"max\":255},\"tc\":{\"type\":\"integer\",\"description\":\"table"
    " data type "
    "code\",\"min\":0,\"max\":255},\"ny\":{\"type\":\"integer\",\"description\":\"number "
    "of y entries\",\"min\":1,\"max\":2048}},\"desription\":\"scanner internal data"
    "\"},\"fp_table_row\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"minItems\""
    ":1,\"maxItems\":2048}},\"$schema\":\"http://json-schema.org/draft-04/schema"
    "#\",\"properties\":{\"ns\":{\"type\":\"integer\",\"minimum\":0,\"maximum\":4095,\"de"
    "scription\":\"Number of "
    "samples\"},\"ex\":{\"type\":\"object\",\"description\":\"DEPRECATED, use "
    "'riegl.exponential_decomposition' instead\"},\"nt\":{\"type\":\"integer\",\"mi"
    "nimum\":0,\"maximum\":255,\"description\":\"Number of traces\"},\"mw\":{\"type\":"
    "\"number\",\"minimum\":0,\"maximum\":4095,\"description\":\"Maximum weight\",\"ex"
    "clusiveMinimum\":true},\"tbl\":{\"type\":\"array\",\"items\":{\"$ref\":\"#/definit"
    "ions/fp_table\"},\"minItems\":1,\"description\":\"various internal data\"},\"a"
    "\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"minItems\":1,\"description\""
    ":\"Amplitude [dB]\",\"maxItems\":256},\"sr\":{\"type\":\"number\",\"minimum\":0,\"d"
    "escription\":\"Sample rate [Hz]\",\"exclusiveMinimum\":true},\"t\":{\"type\":\"o"
    "bject\",\"patternProperties\":{\"^[0-9]+$\":{\"$ref\":\"#/definitions/fp\",\"des"
    "cription\":\"one field per channel, field name is channel index\"}},\"addi"
    "tionalProperties\":false},\"si\":{\"type\":\"number\",\"minimum\":0,\"maximum\":2"
    "55,\"description\":\"Start index (hw_start)\"}},\"description\":\"Receiver "
    "Internals\"}"
    ,
    /* example: */
    "{\"ns\":400,\"nt\":128,\"mw\":31,\"tbl\":[{\"tv\":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,"
    "5.5]],\"nx\":5,\"ch\":0,\"tc\":1,\"ny\":2}],\"a\":[-1.55],\"sr\":7959997000.0,\"t\":"
    "{\"1\":{\"w\":[[78,86,126,134,31],[78,86,126,134,31]],\"s\":[[1.23,4.56],[7."
    "89,0.12]]},\"0\":{\"w\":[[78,86,126,134,31],[78,86,126,134,31]],\"s\":[[1.23"
    ",4.56],[7.89,0.12]]}},\"si\":48}"
};

// Lookup table for reflectance calculation based on amplitude and range
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CALCULATION = {
    /* name        */ "riegl.reflectance_calculation",
    /* title       */ "Reflectance Calculation Table",
    /* description */ "Lookup table for reflectance calculation based on amplitude and range",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Reflectance Calculation Table\",\"$schema\":\"ht"
    "tp://json-schema.org/draft-04/schema#\",\"required\":[\"delta\",\"content\"],"
    "\"properties\":{\"delta\":{\"type\":\"number\",\"description\":\"Delta between "
    "table entries [m], first entry is at range = 0 m\"},\"content\":{\"type\":\""
    "array\",\"items\":{\"type\":\"number\"},\"minItems\":1,\"description\":\"Correctio"
    "n value [dB] to be added to the "
    "amplitude\",\"maxItems\":2000}},\"description\":\"Lookup table for "
    "reflectance calculation based on amplitude and range\"}"
    ,
    /* example: */
    "{\"delta\":0.150918,\"content\":[-33.01]}"
};

// Range-dependent and scan-angle-dependent correction of reflectance reading
const MetaDataItemInfo RDB_RIEGL_REFLECTANCE_CORRECTION = {
    /* name        */ "riegl.reflectance_correction",
    /* title       */ "Near-Range Reflectance Correction",
    /* description */ "Range-dependent and scan-angle-dependent correction of reflectance reading",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Near-range reflectance correction\",\"$schema\""
    ":\"http://json-schema.org/draft-04/schema#\",\"required\":[\"ranges_m\",\"lin"
    "e_angles_deg\",\"reflectance_correction_db\"],\"properties\":{\"line_angles_"
    "deg\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"Angle [d"
    "eg]\"},\"ranges_m\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"descriptio"
    "n\":\"Range [m]\"},\"reflectance_correction_db\":{\"type\":\"array\",\"items\":{\""
    "type\":\"array\",\"items\":{\"type\":\"number\",\"description\":\"columns (each "
    "value corresponds to an angle)\"},\"description\":\"rows (each array "
    "corresponds to a range)\"},\"description\":\"Near range reflectance "
    "correction in dB as a function of range over "
    "angle\"}},\"description\":\"Range-dependent and scan-angle-dependent "
    "correction of reflectance reading\"}"
    ,
    /* example: */
    "{\"line_angles_deg\":[0.0,0.5,1.0,1.5,1.0,2.0,2.5,3.0,3.5,4.0],\"ranges_m"
    "\":[0.0,1.0,2.0,3.0],\"reflectance_correction_db\":[[0.8,0.7,0.6,0.5,0.4,"
    "0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0."
    "8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,"
    "0.1,0.05,0.01]]}"
};

// Details about RiLOC processing
const MetaDataItemInfo RDB_RIEGL_RILOC_PROCESSING_INFO = {
    /* name        */ "riegl.riloc_processing_info",
    /* title       */ "RiLOC Processing Information",
    /* description */ "Details about RiLOC processing",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"RiLOC Processing Information\",\"$schema\":\"htt"
    "p://json-schema.org/draft-04/schema#\",\"required\":[\"trajectory\",\"optimi"
    "zer\",\"georeferencing\"],\"properties\":{\"planes\":{\"type\":\"object\",\"proper"
    "ties\":{},\"description\":\"Information on plane correspondences and used "
    "extraction/matching templates.\"},\"optimizer\":{\"type\":\"object\",\"propert"
    "ies\":{},\"description\":\"Optimization result information and statistics."
    "\"},\"georeferencing\":{\"type\":\"object\",\"properties\":{},\"description\":\"In"
    "ternal georeferencing parameters.\"},\"trajectory\":{\"type\":\"object\",\"pro"
    "perties\":{},\"description\":\"Trajectory information and "
    "statistics.\"}},\"description\":\"Details about RiLOC processing\"}"
    ,
    /* example: */
    "{\"planes\":{},\"optimizer\":{},\"georeferencing\":{},\"trajectory\":{}}"
};

// Scan pattern description
const MetaDataItemInfo RDB_RIEGL_SCAN_PATTERN = {
    /* name        */ "riegl.scan_pattern",
    /* title       */ "Scan Pattern",
    /* description */ "Scan pattern description",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Scan Pat"
    "tern\",\"definitions\":{\"program\":{\"type\":\"object\",\"required\":[\"name\"],\"p"
    "roperties\":{\"name\":{\"type\":\"string\",\"description\":\"Name of measurement"
    " program\"},\"laser_prr\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Las"
    "er Pulse Repetition Rate "
    "[Hz]\",\"exclusiveMinimum\":false}},\"description\":\"Measurement program\"}}"
    ",\"properties\":{\"rectangular\":{\"type\":\"object\",\"required\":[\"phi_start\","
    "\"phi_stop\",\"phi_increment\",\"theta_start\",\"theta_stop\",\"theta_increment"
    "\"],\"properties\":{\"phi_increment\":{\"type\":\"number\",\"minimum\":0.0,\"maxim"
    "um\":90.0,\"description\":\"Increment of phi angle in SOCS [deg]\"},\"theta_"
    "start\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":180.0,\"description\":\"S"
    "tart theta angle in SOCS [deg]\"},\"theta_stop\":{\"type\":\"number\",\"minimu"
    "m\":0.0,\"maximum\":180.0,\"description\":\"Stop theta angle in SOCS [deg]\"}"
    ",\"program\":{\"$ref\":\"#/definitions/program\"},\"phi_start\":{\"type\":\"numbe"
    "r\",\"minimum\":0.0,\"maximum\":360.0,\"description\":\"Start phi angle in "
    "SOCS [deg]\"},\"phi_stop\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":720.0"
    ",\"description\":\"Stop phi angle in SOCS [deg]\"},\"theta_increment\":{\"typ"
    "e\":\"number\",\"minimum\":0.0,\"maximum\":90.0,\"description\":\"Increment of "
    "theta angle in SOCS [deg]\"}},\"description\":\"Rectangular Field Of View "
    "Scan Pattern\"},\"segments\":{\"type\":\"object\",\"required\":[\"list\"],\"proper"
    "ties\":{\"list\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"required\":[\"st"
    "art\",\"stop\",\"increment\"],\"properties\":{\"start\":{\"type\":\"number\",\"minim"
    "um\":0.0,\"maximum\":360.0,\"description\":\"Start angle in SOCS [deg]\"},\"in"
    "crement\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":90.0,\"description\":\""
    "Increment of angle in SOCS [deg]\"},\"stop\":{\"type\":\"number\",\"minimum\":0"
    ".0,\"maximum\":720.0,\"description\":\"Stop angle in SOCS "
    "[deg]\"}},\"description\":\"Line Scan Segment\"}},\"program\":{\"$ref\":\"#/defi"
    "nitions/program\"}},\"description\":\"Segmented Line Scan Pattern\"},\"line\""
    ":{\"type\":\"object\",\"required\":[\"start\",\"stop\",\"increment\"],\"properties\""
    ":{\"start\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":360.0,\"description\""
    ":\"Start angle in SOCS [deg]\"},\"program\":{\"$ref\":\"#/definitions/program"
    "\"},\"increment\":{\"type\":\"number\",\"minimum\":0.0,\"maximum\":90.0,\"descript"
    "ion\":\"Increment of angle in SOCS [deg]\"},\"stop\":{\"type\":\"number\",\"mini"
    "mum\":0.0,\"maximum\":720.0,\"description\":\"Stop angle in SOCS "
    "[deg]\"}},\"description\":\"Line Scan Pattern\"}},\"description\":\"Scan "
    "pattern description\"}"
    ,
    /* example: */
    "{\"rectangular\":{\"phi_increment\":0.04,\"theta_start\":30.0,\"theta_stop\":1"
    "30.0,\"program\":{\"name\":\"High Speed\",\"laser_prr\":100000.0},\"phi_start\":"
    "45.0,\"phi_stop\":270.0,\"theta_increment\":0.04}}"
};

// Details about laser shot files
const MetaDataItemInfo RDB_RIEGL_SHOT_INFO = {
    /* name        */ "riegl.shot_info",
    /* title       */ "Shot Information",
    /* description */ "Details about laser shot files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Shot Inf"
    "ormation\",\"properties\":{\"shot_file\":{\"type\":\"object\",\"required\":[\"file"
    "_extension\"],\"properties\":{\"file_uuid\":{\"type\":\"string\",\"description\":"
    "\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Shot file "
    "extension, without the leading dot\"}}}},\"description\":\"Details about "
    "laser shot files\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"shot_file\":{\"file_uuid\":\"26a00815-67c0-4bff-8fe8-c577378fe663\",\"file"
    "_extension\":\"sodx\"}}"
};

// Point filters applied in addition to the application-defined filters
const MetaDataItemInfo RDB_RIEGL_STORED_FILTERS = {
    /* name        */ "riegl.stored_filters",
    /* title       */ "Stored Filters",
    /* description */ "Point filters applied in addition to the application-defined filters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Stored filters\",\"$schema\":\"http://json-schem"
    "a.org/draft-04/schema#\",\"required\":[\"activated\",\"filters\"],\"properties"
    "\":{\"filters\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"required\":[\"act"
    "ivated\",\"title\",\"description\",\"filter\"],\"properties\":{\"title\":{\"type\":"
    "\"string\",\"description\":\"A short filter title (e.g. for display in a "
    "selection list)\"},\"filter\":{\"type\":\"string\",\"description\":\"The RDB "
    "filter string to apply (e.g. when reading points or index), details "
    "see documentation of function "
    "select()\"},\"activated\":{\"type\":\"boolean\",\"description\":\"Apply ('true')"
    " or ignore ('false') this "
    "filter\"},\"description\":{\"type\":\"string\",\"description\":\"A brief "
    "description of the filter (e.g. for display in a "
    "tooltip)\"}},\"description\":\"Point filter "
    "definition\"},\"description\":\"List of point "
    "filters\"},\"activated\":{\"type\":\"boolean\",\"description\":\"Apply ('true') "
    "or ignore ('false') all filters\"}},\"description\":\"Point filters "
    "applied in addition to the application-defined filters\"}"
    ,
    /* example: */
    "{\"filters\":[{\"title\":\"Ignore uncertain points from MTA "
    "resolution\",\"filter\":\"riegl.mta_uncertain_point == "
    "0\",\"activated\":false,\"description\":\"Ignore points with uncertain MTA "
    "zone assignment\"},{\"title\":\"Ignore points below dynamic trigger "
    "level\",\"filter\":\"riegl.dyntrig_uncertain_point == "
    "0\",\"activated\":false,\"description\":\"Ignore points with an echo signal "
    "amplitude below the dynamic trigger\"},{\"title\":\"Ignore points outside "
    "outer AABB\",\"filter\":\"riegl.point_outside_aabb_bocs == "
    "0\",\"activated\":false,\"description\":\"Ignore points that are outside the"
    " outer AABB in BOCS\"},{\"title\":\"Ignore points inside inner "
    "AABB\",\"filter\":\"riegl.point_inside_aabb_bocs == "
    "0\",\"activated\":false,\"description\":\"Ignore points that are inside the "
    "inner AABB in BOCS\"}],\"activated\":true}"
};

// Details of major system components like lidar sensors, cameras and navigation devices
const MetaDataItemInfo RDB_RIEGL_SYSTEM_DESCRIPTION = {
    /* name        */ "riegl.system_description",
    /* title       */ "System Description",
    /* description */ "Details of major system components like lidar sensors, cameras and navigation devices",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"System Description\",\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"required\":[\"$class\",\"version\",\"author\",\"t"
    "imestamp\",\"system\"],\"properties\":{\"timestamp\":{\"type\":\"string\",\"descri"
    "ption\":\"Date and time of creation (e.g. 2019-06-19T13:32:10+02:00)\"},\""
    "version\":{\"type\":\"string\",\"description\":\"Document format "
    "version\"},\"system\":{\"type\":\"object\",\"description\":\"The actual system "
    "description, details see documentation of RIEGL System "
    "Description\"},\"$class\":{\"type\":\"string\",\"description\":\"Object class na"
    "me\",\"enum\":[\"Document\"]},\"author\":{\"type\":\"string\",\"description\":\"Name"
    " of the author that created the document\"}},\"description\":\"Details of "
    "major system components like lidar sensors, cameras and navigation "
    "devices\"}"
    ,
    /* example: */
    "{\"timestamp\":\"2022-09-30T11:56:26+00:00\",\"version\":\"1.2.1\",\"system\":{\""
    "details\":\"see documentation of RIEGL System "
    "Description\"},\"$class\":\"Document\",\"author\":\"RIEGL LMS GmbH, Austria\"}"
};

// Conversion of background radiation raw values to temperatures in °C
const MetaDataItemInfo RDB_RIEGL_TEMPERATURE_CALCULATION = {
    /* name        */ "riegl.temperature_calculation",
    /* title       */ "Temperature Calculation Table",
    /* description */ "Conversion of background radiation raw values to temperatures in °C",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Temperature Calculation Table\",\"definitions\""
    ":{\"conversion_table\":{\"type\":\"object\",\"required\":[\"value\",\"temperature"
    "\"],\"properties\":{\"value\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"de"
    "scription\":\"LSB [1]\"},\"temperature\":{\"type\":\"array\",\"items\":{\"type\":\"n"
    "umber\"},\"description\":\"Temperature [\\u00b0C]\"}}}},\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"properties\":{\"InGaAs\":{\"$ref\":\"#/defi"
    "nitions/conversion_table\",\"description\":\"Conversion table for InGaAs c"
    "hannel\"},\"Si\":{\"$ref\":\"#/definitions/conversion_table\",\"description\":\""
    "Conversion table for Si channel\"},\"InGaAs_Si_Difference\":{\"$ref\":\"#/de"
    "finitions/conversion_table\",\"description\":\"Conversion table for InGaAs"
    " - Si difference\"}},\"description\":\"Conversion of background radiation "
    "raw values to temperatures in \\u00b0C\"}"
    ,
    /* example: */
    "{\"InGaAs\":{\"value\":[0.0,64.00097659230323,128.0019531846065,192.002929"
    "7769097,256.0039063692129,320.00488296151616,384.0058595538194,448.006"
    "8361461226,512.0078127384259],\"temperature\":[307.22196722535614,309.11"
    "53478247277,311.1188086915047,313.10025350480055,315.2137946389828,317"
    ".2172555057597,319.2207163725366,321.2021611858325,323.3157023200148]}"
    ",\"Si\":{\"value\":[0.0,64.00097659230323,128.0019531846065,192.0029297769"
    "097,256.0039063692129,320.00488296151616,384.0058595538194,448.0068361"
    "461226,512.0078127384259],\"temperature\":[546.300048828125,548.81640512"
    "12026,551.3143938500972,554.0144257850053,556.604252334815,559.2124464"
    "488079,561.8022729986177,564.4104671126105,567.0002936624203]},\"InGaAs"
    "_Si_Difference\":{\"value\":[1000.0,1100.090029602954,1200.04425183874,13"
    "00.1342814416948,1400.0885036774805,1500.0427259132668,1600.1327555162"
    "209,1700.0869777520065,1800.0411999877924],\"temperature\":[1749.9771111"
    "17893,1749.977111117893,1749.977111117893,1749.977111117893,1749.97711"
    "1117893,1749.977111117893,1744.7813348796044,1681.9971312601092,1622.3"
    "944822534868]}}"
};

// Base of timestamps (epoch)
const MetaDataItemInfo RDB_RIEGL_TIME_BASE = {
    /* name        */ "riegl.time_base",
    /* title       */ "Time Base",
    /* description */ "Base of timestamps (epoch)",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Time Base\",\"$schema\":\"http://json-schema.org"
    "/draft-04/schema#\",\"required\":[\"epoch\",\"source\"],\"properties\":{\"epoch\""
    ":{\"type\":\"string\",\"description\":\"Date and time of timestamp '0' as "
    "proposed by RFC 3339 (e.g. 2015-10-27T00:00:00+01:00).\"},\"source\":{\"ty"
    "pe\":\"string\",\"description\":\"Timestamp source\",\"enum\":[\"unknown\",\"RTC\","
    "\"GNSS\"]},\"system\":{\"type\":\"string\",\"description\":\"Time system (time "
    "standard)\",\"enum\":[\"unknown\",\"UTC\",\"GPS\"]}},\"description\":\"Base of "
    "timestamps (epoch)\"}"
    ,
    /* example: */
    "{\"epoch\":\"2015-10-27T00:00:00+00:00\",\"source\":\"GNSS\",\"system\":\"UTC\"}"
};

// Details about position+orientation files
const MetaDataItemInfo RDB_RIEGL_TRAJECTORY_INFO = {
    /* name        */ "riegl.trajectory_info",
    /* title       */ "Trajectory Information",
    /* description */ "Details about position+orientation files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Trajectory Information\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"required\":[\"time_interval\",\"navigatio"
    "n_frame\"],\"properties\":{\"navigation_frame\":{\"type\":\"string\",\"descripti"
    "on\":\"Navigation frame (NED: North-East-Down, ENU: East-North-Up)\",\"enu"
    "m\":[\"unknown\",\"NED\",\"ENU\"]},\"location\":{\"type\":\"string\",\"description\":"
    "\"Project location, e.g. "
    "city/state/country\"},\"project\":{\"type\":\"string\",\"description\":\"Project"
    " name\"},\"software\":{\"type\":\"string\",\"description\":\"Software that "
    "calculated the trajectory (this may be the same or different software "
    "than the one that created the file)\"},\"time_interval\":{\"type\":\"object\""
    ",\"required\":[\"minimum\",\"average\",\"maximum\",\"std_dev\"],\"properties\":{\"a"
    "verage\":{\"type\":\"number\",\"description\":\"Average time interval "
    "[s]\"},\"minimum\":{\"type\":\"number\",\"description\":\"Minimum time interval "
    "[s]\"},\"maximum\":{\"type\":\"number\",\"description\":\"Maximum time interval "
    "[s]\"},\"std_dev\":{\"type\":\"number\",\"description\":\"Standard deviation of "
    "intervals [s]\"}},\"description\":\"Time interval "
    "statistics\"},\"device\":{\"type\":\"string\",\"description\":\"Navigation "
    "device, e.g. "
    "name/type/serial\"},\"settings\":{\"type\":\"string\",\"description\":\"Settings"
    " used to calculate the trajectory (descriptive "
    "text)\"},\"field_of_application\":{\"type\":\"string\",\"description\":\"Field "
    "of application\",\"enum\":[\"unknown\",\"SLS\",\"TLS\",\"KLS\",\"MLS\",\"ULS\",\"ALS\","
    "\"BLS\",\"ILS\"]},\"company\":{\"type\":\"string\",\"description\":\"Company "
    "name\"}},\"description\":\"Details about position+orientation files\"}"
    ,
    /* example: */
    "{\"navigation_frame\":\"NED\",\"location\":\"Horn\",\"project\":\"Campaign "
    "3\",\"software\":\"Navigation Software XYZ\",\"time_interval\":{\"average\":0.0"
    "0500053,\"minimum\":0.00500032,\"maximum\":0.005004883,\"std_dev\":5.51e-07}"
    ",\"device\":\"IMU Model 12/1, Serial# 12345\",\"settings\":\"default\",\"field_"
    "of_application\":\"MLS\",\"company\":\"RIEGL LMS\"}"
};

// Trigger-Timestamping-Unit/IP configuration
const MetaDataItemInfo RDB_RIEGL_TTIP_CONFIGURATION = {
    /* name        */ "riegl.ttip_configuration",
    /* title       */ "TTIP Configuration",
    /* description */ "Trigger-Timestamping-Unit/IP configuration",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"TTIP Configuration\",\"$schema\":\"http://json-s"
    "chema.org/draft-04/schema#\",\"required\":[\"ttip_version\",\"num_channel\",\""
    "ext_channel\",\"ext_signal\",\"ext_delay\",\"ext_subdivider\",\"ext_bitmask\",\""
    "dmi_incr\",\"dmi_min_time\",\"dmi_max_time\",\"dmi_dist_per_tick\",\"channel\","
    "\"output_descr\",\"input_descr\",\"output_usage\",\"input_usage\",\"out_polarit"
    "y\",\"out_duration\",\"out_interval\",\"out_delay\",\"out_num_pulses\",\"in_pola"
    "rity\",\"in_min_duration\",\"in_max_duration\",\"in_max_delay\"],\"properties\""
    ":{\"out_polarity\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 .. "
    "positive edge, 1 .. negative edge\"},\"in_max_duration\":{\"type\":\"integer"
    "\",\"minimum\":0,\"description\":\"stops measurement of pulse duration of "
    "input signal [0.1 msec]\"},\"dmi_max_time\":{\"type\":\"integer\",\"minimum\":0"
    ",\"description\":\"dmi, maximum time interval between trigger events [0.1"
    " msec]\"},\"in_min_duration\":{\"type\":\"integer\",\"minimum\":0,\"description\""
    ":\"input signals with smaller pulse durations are ignored [0.1 msec]\"},"
    "\"in_max_delay\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"maximum "
    "delay to output pulse before fake event is generated [0.1 msec], zero "
    "indicates that no fake events are generated\"},\"ttip_version\":{\"type\":\""
    "integer\",\"minimum\":0,\"description\":\"following main.sub.ss.sss\"},\"ext_s"
    "ubdivider\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"reduces the "
    "frequency, default "
    "1\"},\"input_descr\":{\"type\":\"string\",\"description\":\"descriptive "
    "string\"},\"dmi_incr\":{\"type\":\"integer\",\"description\":\"dmi, increment in"
    " ticks\"},\"dmi_dist_per_tick\":{\"type\":\"number\",\"description\":\"dmi, "
    "distance per tick, just informative "
    "[m]\"},\"dmi_min_time\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"dmi,"
    " minimum time interval between trigger events [0.1 msec]\"},\"out_durati"
    "on\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"output pulse duration"
    " [0.1 "
    "msec]\"},\"output_descr\":{\"type\":\"string\",\"description\":\"descriptive "
    "string\"},\"input_usage\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 "
    ".. deactivated, 1 .. just detecting and timestamping\"},\"ext_delay\":{\"t"
    "ype\":\"integer\",\"minimum\":0,\"description\":\"external trigger, time delay"
    " [0.1 msec]\"},\"out_delay\":{\"type\":\"integer\",\"minimum\":0,\"description\":"
    "\"output pulse initial delay after start [0.1 "
    "msec]\"},\"output_usage\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 "
    ".. no output, 1 .. output derived from internal clock, 2 .. output "
    "derived from dmi, 3 .. output derived from external signal, 4 .. "
    "output static low, 5 .. output static "
    "high\"},\"ext_channel\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"ID "
    "of channel used as external trigger input, 32 indicates none\"},\"num_ch"
    "annel\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"number of "
    "input/output "
    "channels\"},\"ext_signal\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0"
    " .. use input signal, 1 .. use output signal of "
    "channel\"},\"channel\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"ID of"
    " input/output channel\"},\"out_interval\":{\"type\":\"integer\",\"minimum\":0,\""
    "description\":\"output pulse interval [0.1 "
    "msec]\"},\"in_polarity\":{\"type\":\"integer\",\"minimum\":0,\"description\":\"0 "
    ".. positive edge, 1 .. negative edge\"},\"ext_bitmask\":{\"type\":\"integer\""
    ",\"minimum\":0,\"description\":\"defines which of the internal pulse "
    "generators are to be started\"},\"out_num_pulses\":{\"type\":\"integer\",\"min"
    "imum\":0,\"description\":\"number of output pulses to be generated, 0 .. "
    "infinite\"}},\"description\":\"Trigger-Timestamping-Unit/IP "
    "configuration\"}"
    ,
    /* example: */
    "{\"out_polarity\":0,\"in_max_duration\":10000,\"dmi_max_time\":0,\"in_min_dur"
    "ation\":0,\"in_max_delay\":9990,\"ttip_version\":1,\"ext_subdivider\":1,\"inpu"
    "t_descr\":\"Port 1 - Exposure\",\"dmi_incr\":0,\"dmi_dist_per_tick\":0.0,\"dmi"
    "_min_time\":0,\"out_duration\":300,\"output_descr\":\"Port 1 - Trigger\",\"inp"
    "ut_usage\":1,\"ext_delay\":0,\"out_delay\":0,\"output_usage\":1,\"ext_channel\""
    ":32,\"num_channel\":9,\"ext_signal\":0,\"channel\":0,\"out_interval\":10000,\"i"
    "n_polarity\":0,\"ext_bitmask\":0,\"out_num_pulses\":1}"
};

// Details about vertex file
const MetaDataItemInfo RDB_RIEGL_VERTEX_INFO = {
    /* name        */ "riegl.vertex_info",
    /* title       */ "Vertex Information",
    /* description */ "Details about vertex file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Vertex I"
    "nformation\",\"properties\":{\"vertex_file\":{\"type\":\"object\",\"required\":[\""
    "file_extension\"],\"properties\":{\"file_uuid\":{\"type\":\"string\",\"descripti"
    "on\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Vertex file "
    "extension, without the leading dot\"}}}},\"description\":\"Details about "
    "vertex file\",\"type\":\"object\"}"
    ,
    /* example: */
    "{\"vertex_file\":{\"file_uuid\":\"51534d95-d71f-4f36-ae1a-3e63a21fd1c7\",\"fi"
    "le_extension\":\"vtx\"}}"
};

// Details about the voxels contained in the file
const MetaDataItemInfo RDB_RIEGL_VOXEL_INFO = {
    /* name        */ "riegl.voxel_info",
    /* title       */ "Voxel Information",
    /* description */ "Details about the voxels contained in the file",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Voxel Information\",\"definitions\":{\"edge_leng"
    "th\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Length of voxel edge ["
    "m].\",\"exclusiveMinimum\":true},\"voxel_origin_point\":{\"type\":\"array\",\"it"
    "ems\":{\"type\":\"number\"},\"minItems\":3,\"description\":\"Origin point for "
    "all voxel indices in voxel CRS.\",\"maxItems\":3},\"voxel_origin_enum\":{\"d"
    "efault\":\"corner\",\"description\":\"Defines whether the voxel's center or "
    "a corner is placed on CRS origin <tt>(0/0/0)</tt>.\",\"enum\":[\"center\",\""
    "corner\"]},\"shape_thresholds\":{\"type\":\"object\",\"required\":[\"plane\",\"lin"
    "e\"],\"properties\":{\"plane\":{\"type\":\"number\",\"maximum\":1,\"minimum\":0,\"ex"
    "clusiveMaximum\":true,\"description\":\"If the smallest eigenvalue is "
    "smaller than the median eigenvalue * plane_threshold, the voxel is "
    "considered a plane.\",\"exclusiveMinimum\":true},\"line\":{\"type\":\"number\","
    "\"minimum\":1,\"description\":\"If the biggest eigenvalue is bigger than "
    "the median eigenvalue * line_threshold, the voxel is considered a "
    "line.\",\"exclusiveMinimum\":true}},\"description\":\"Thresholds used to "
    "compute the voxel's shape_id "
    "value.\"},\"voxel_type\":{\"default\":\"centroid\",\"description\":\"Whether a "
    "point in a voxel represents its center or its centroid. If type is "
    "<tt>index</tt> there is no point but only an integer voxel index.\",\"en"
    "um\":[\"center\",\"centroid\",\"index\"]},\"voxel_size_cubic\":{\"$ref\":\"#/defin"
    "itions/edge_length\",\"type\":\"number\"},\"voxel_size\":{\"type\":\"array\",\"ite"
    "ms\":{\"$ref\":\"#/definitions/edge_length\"},\"minItems\":3,\"description\":\"S"
    "ize of voxels.\",\"maxItems\":3}},\"$schema\":\"http://json-schema.org/draft"
    "-04/schema#\",\"oneOf\":[{\"required\":[\"size\",\"voxel_origin\",\"voxel_type\"]"
    ",\"properties\":{\"shape_thresholds\":{\"$ref\":\"#/definitions/shape_thresho"
    "lds\"},\"size\":{\"oneOf\":[{\"$ref\":\"#/definitions/voxel_size\"},{\"$ref\":\"#/"
    "definitions/voxel_size_cubic\"}],\"description\":\"Size of voxels in file "
    "coordinate system.\"},\"voxel_origin\":{\"$ref\":\"#/definitions/voxel_origi"
    "n_enum\"},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_type\"}},\"additional"
    "Properties\":false},{\"required\":[\"reference_point\",\"size_llcs\",\"size\",\""
    "voxel_origin\",\"voxel_type\"],\"properties\":{\"size\":{\"$ref\":\"#/definition"
    "s/voxel_size\",\"description\":\"Size of voxels in file coordinate system."
    "\"},\"voxel_origin\":{\"oneOf\":[{\"$ref\":\"#/definitions/voxel_origin_enum\"}"
    ",{\"$ref\":\"#/definitions/voxel_origin_point\",\"description\":\"The base "
    "point of the voxel grid. Used together with <tt>voxel_size</tt> and "
    "<tt>voxel_index</tt> to compute actual point coordinates.\"}]},\"referen"
    "ce_point\":{\"type\":\"array\",\"items\":{\"type\":\"number\",\"minimum\":-180,\"max"
    "imum\":180},\"minItems\":2,\"description\":\"Point in WGS84 geodetic decimal"
    " degree (EPSG:4326) that was used to compute the projection distortion"
    " parameters. The coefficient order is latitude, longitude. Only voxels"
    " with corresponding geo_tag, voxel_size and reference_point can be "
    "reliably processed together. This entry is available for voxel files "
    "in projected CRS only.\",\"maxItems\":2},\"size_llcs\":{\"$ref\":\"#/definitio"
    "ns/voxel_size\",\"description\":\"Size of voxels in a locally levelled "
    "cartesian coordinate system (xyz). This is only used for voxels based "
    "on a map projection.\"},\"shape_thresholds\":{\"$ref\":\"#/definitions/shape"
    "_thresholds\"},\"voxel_type\":{\"$ref\":\"#/definitions/voxel_type\"}},\"addit"
    "ionalProperties\":false}],\"description\":\"Details about the voxels "
    "contained in the file\"}"
    ,
    /* example: */
    "{\"size_llcs\":[0.5156575252891171,0.5130835356683303,0.5143705304787237"
    "],\"voxel_origin\":\"corner\",\"reference_point\":[48,16],\"size\":[0.59716428"
    "34779395,0.5971642834779395,0.5143705304787237],\"voxel_type\":\"centroid"
    "\",\"shape_thresholds\":{\"line\":6,\"plane\":0.16}}"
};

// Settings for waveform averaging
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS = {
    /* name        */ "riegl.waveform_averaging_settings",
    /* title       */ "Waveform Averaging Settings",
    /* description */ "Settings for waveform averaging",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Waveform Averaging Settings\",\"$schema\":\"http"
    "://json-schema.org/draft-04/schema#\",\"required\":[\"num_shots\",\"mta_zone"
    "\"],\"properties\":{\"num_shots\":{\"type\":\"integer\",\"minimum\":1,\"descriptio"
    "n\":\"Number of consecutive shots to be used for averaging.\"},\"mta_zone\""
    ":{\"type\":\"integer\",\"minimum\":1,\"description\":\"Fixed MTA zone for avera"
    "ging.\"},\"trim\":{\"type\":\"number\",\"minimum\":0,\"default\":0,\"maximum\":0.5,"
    "\"description\":\"Percentage for robust "
    "averaging.\"}},\"description\":\"Settings for waveform averaging\"}"
    ,
    /* example: */
    "{\"num_shots\":7,\"mta_zone\":1,\"trim\":0.05}"
};

// Details about waveform files
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_INFO = {
    /* name        */ "riegl.waveform_info",
    /* title       */ "Waveform Information",
    /* description */ "Details about waveform files",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Waveform Information\",\"$schema\":\"http://json"
    "-schema.org/draft-04/schema#\",\"required\":[\"sample_block_file\",\"sample_"
    "data_files\"],\"properties\":{\"sample_block_file\":{\"type\":\"object\",\"requi"
    "red\":[\"file_extension\"],\"properties\":{\"file_uuid\":{\"type\":\"string\",\"de"
    "scription\":\"File's Universally Unique Identifier (RFC "
    "4122)\"},\"file_extension\":{\"type\":\"string\",\"description\":\"Sample block "
    "file extension, without the leading dot\"}}},\"sample_data_files\":{\"type"
    "\":\"array\",\"items\":{\"type\":\"object\",\"required\":[\"channel\",\"channel_name"
    "\",\"sample_interval\",\"sample_bits\",\"laser_wavelength\",\"delta_st\",\"file_"
    "extension\"],\"properties\":{\"sample_interval\":{\"type\":\"number\",\"minimum\""
    ":0,\"description\":\"Sampling interval in seconds\",\"exclusiveMinimum\":fal"
    "se},\"channel\":{\"type\":\"integer\",\"maximum\":255,\"minimum\":0,\"exclusiveMa"
    "ximum\":false,\"description\":\"Sample block channel number (255 = invalid"
    ")\",\"exclusiveMinimum\":false},\"channel_name\":{\"type\":\"string\",\"descript"
    "ion\":\"Sample block channel name\"},\"sample_bits\":{\"type\":\"integer\",\"max"
    "imum\":32,\"minimum\":0,\"exclusiveMaximum\":false,\"description\":\"Bitwidth "
    "of samples (e.g. 10 bit, 12 bit)\",\"exclusiveMinimum\":false},\"delta_st\""
    ":{\"type\":\"number\",\"description\":\"reserved\"},\"file_uuid\":{\"type\":\"strin"
    "g\",\"description\":\"File's Universally Unique Identifier (RFC 4122)\"},\"l"
    "aser_wavelength\":{\"type\":\"number\",\"minimum\":0,\"description\":\"Laser "
    "wavelength in meters (0 = unknown)\",\"exclusiveMinimum\":false},\"file_ex"
    "tension\":{\"type\":\"string\",\"description\":\"Sample data file extension, "
    "without the leading dot\"}}}},\"range_offset_waveform_samples_m\":{\"type\""
    ":\"number\",\"description\":\"Calibrated device specific range offset for "
    "waveform samples in "
    "meters.\"},\"range_offset_m\":{\"type\":\"number\",\"description\":\"Calibrated "
    "device specific range offset for waveform analysis by system response "
    "fitting in meters.\"}},\"description\":\"Details about waveform files\"}"
    ,
    /* example: */
    "{\"sample_block_file\":{\"file_uuid\":\"93a03615-66c0-4bea-8ff8-c577378fe66"
    "0\",\"file_extension\":\"sbx\"},\"sample_data_files\":[{\"sample_interval\":1.0"
    "0503e-09,\"channel\":0,\"channel_name\":\"high_power\",\"laser_wavelength\":0,"
    "\"delta_st\":0,\"file_uuid\":\"da084413-e3e8-4655-a122-071de8490d8e\",\"sampl"
    "e_bits\":12,\"file_extension\":\"sp0\"},{\"sample_interval\":1.00503e-09,\"cha"
    "nnel\":1,\"channel_name\":\"low_power\",\"laser_wavelength\":0,\"delta_st\":0,\""
    "file_uuid\":\"93585b5e-5ea9-43a1-947b-e7ba3be642d2\",\"sample_bits\":12,\"fi"
    "le_extension\":\"sp1\"},{\"sample_interval\":1.00503e-09,\"channel\":5,\"chann"
    "el_name\":\"wwf\",\"laser_wavelength\":0,\"delta_st\":0,\"file_uuid\":\"9d2298c4"
    "-1036-464f-b5cb-1cf8e517f3a0\",\"sample_bits\":12,\"file_extension\":\"sp5\"}"
    "],\"range_offset_waveform_samples_m \":7.283,\"range_offset_m\":3.1415}"
};

// Scanner settings for waveform output
const MetaDataItemInfo RDB_RIEGL_WAVEFORM_SETTINGS = {
    /* name        */ "riegl.waveform_settings",
    /* title       */ "Waveform Settings",
    /* description */ "Scanner settings for waveform output",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Waveform"
    " Settings\",\"items\":{\"type\":\"object\",\"required\":[\"sbl_name\",\"enabled\",\""
    "channel_idx_mask\"],\"properties\":{\"pass_rng_greater\":{\"type\":\"number\",\""
    "description\":\"Threshold for range greater than "
    "[m]\"},\"pass_dev_less\":{\"type\":\"integer\",\"description\":\"Threshold for "
    "deviation less than "
    "[1]\"},\"enabled\":{\"type\":\"boolean\",\"description\":\"Waveform output "
    "enabled\"},\"pass_rng_less\":{\"type\":\"number\",\"description\":\"Threshold "
    "for range less than "
    "[m]\"},\"sbl_name\":{\"type\":\"string\",\"description\":\"Name of sample block,"
    " e.g.: wfm, "
    "wwf\"},\"pass_ampl_less\":{\"type\":\"number\",\"description\":\"Threshold for "
    "amplitude less than "
    "[dB]\"},\"channel_idx_mask\":{\"type\":\"integer\",\"description\":\"Bit mask "
    "for channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = "
    "Bit1, ...\"},\"logic_expression\":{\"type\":\"string\",\"description\":\"Logic "
    "expression of smart waveforms "
    "filter\"},\"smart_enabled\":{\"type\":\"boolean\",\"description\":\"Smart "
    "waveform output enabled\"},\"pass_ampl_greater\":{\"type\":\"number\",\"descri"
    "ption\":\"Threshold for amplitude greater than "
    "[dB]\"},\"pass_dev_greater\":{\"type\":\"integer\",\"description\":\"Threshold "
    "for deviation greater than [1]\"}}},\"description\":\"Scanner settings for"
    " waveform output\",\"type\":\"array\"}"
    ,
    /* example: */
    "[{\"pass_rng_greater\":9.27,\"enabled\":true,\"pass_rng_less\":13.11,\"sbl_na"
    "me\":\"wfm\",\"pass_ampl_less\":5.0,\"channel_idx_mask\":11,\"smart_enabled\":t"
    "rue,\"pass_ampl_greater\":1.0},{\"channel_idx_mask\":32,\"enabled\":false,\"s"
    "bl_name\":\"wwf\"}]"
};

// Window analysis data estimated from scandata and resulting filter parameters
const MetaDataItemInfo RDB_RIEGL_WINDOW_ANALYSIS = {
    /* name        */ "riegl.window_analysis",
    /* title       */ "Window Analysis",
    /* description */ "Window analysis data estimated from scandata and resulting filter parameters",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Window Analysis\",\"$schema\":\"http://json-sche"
    "ma.org/draft-04/schema#\",\"required\":[\"result\",\"filter\",\"settings\"],\"pr"
    "operties\":{\"result\":{\"type\":\"object\",\"required\":[\"angle\",\"range_mean\","
    "\"range_sigma\",\"amplitude_mean\",\"amplitude_sigma\",\"amplitude_offset\"],\""
    "properties\":{\"angle\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"descri"
    "ption\":\"[deg]\"},\"range_mean\":{\"type\":\"array\",\"items\":{\"type\":\"number\"}"
    ",\"description\":\"[m]\"},\"range_sigma\":{\"type\":\"array\",\"items\":{\"type\":\"n"
    "umber\"},\"description\":\"[m]\"},\"timestamp\":{\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\"},\"description\":\"[s]\"},\"amplitude_sigma\":{\"type\":\"array\",\"i"
    "tems\":{\"type\":\"number\"},\"description\":\"[dB]\"},\"amplitude_offset\":{\"typ"
    "e\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"[dB]\"},\"amplitude_"
    "mean\":{\"type\":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"[dB]\"}}"
    "},\"filter\":{\"type\":\"object\",\"required\":[\"angle\",\"range_min\",\"range_max"
    "\",\"amplitude_max\"],\"properties\":{\"angle\":{\"type\":\"array\",\"items\":{\"typ"
    "e\":\"number\"},\"description\":\"[deg]\"},\"range_min\":{\"type\":\"array\",\"items"
    "\":{\"type\":\"number\"},\"description\":\"[m]\"},\"amplitude_max\":{\"type\":\"arra"
    "y\",\"items\":{\"type\":\"number\"},\"description\":\"[dB]\"},\"range_max\":{\"type\""
    ":\"array\",\"items\":{\"type\":\"number\"},\"description\":\"[m]\"}}},\"settings\":{"
    "\"type\":\"object\",\"required\":[\"range\",\"amplitude\"],\"properties\":{\"range\""
    ":{\"type\":\"object\",\"required\":[\"sigma_factor\",\"additive_value\"],\"proper"
    "ties\":{\"sigma_factor\":{\"type\":\"number\"},\"additive_value\":{\"type\":\"numb"
    "er\"}}},\"amplitude\":{\"type\":\"object\",\"required\":[\"sigma_factor\",\"additi"
    "ve_value\"],\"properties\":{\"sigma_factor\":{\"type\":\"number\"},\"additive_va"
    "lue\":{\"type\":\"number\"}}}}}},\"description\":\"Window analysis data "
    "estimated from scandata and resulting filter parameters\"}"
    ,
    /* example: */
    "{\"result\":{\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,"
    "15.9],\"range_mean\":[0.1105621,0.1079564,0.1087088,0.1067261,0.1054582,"
    "0.1090412,0.102871,0.1019044,0.1051523,0.1058445,0.1031261],\"range_sig"
    "ma\":[0.01869652,0.02151435,0.01747969,0.01918765,0.01945776,0.01934862"
    ",0.01955329,0.02225589,0.02229977,0.01899122,0.02009433],\"timestamp\":["
    "408.4441,411.4443],\"amplitude_sigma\":[0.4272844,0.4298479,0.4236816,0."
    "4283583,0.4362353,0.4315141,0.4373984,0.4472798,0.4346001,0.4345487,0."
    "4540681],\"amplitude_offset\":[1.9,1.9],\"amplitude_mean\":[5.347396,5.263"
    "155,5.224655,5.179926,5.097782,5.116479,5.051756,4.983473,5.007885,5.0"
    "02441,4.982]},\"filter\":{\"angle\":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15"
    ".6,15.7,15.8,15.9],\"range_min\":[-0.208,-0.21,-0.212,-0.214,-0.216,-0.2"
    "18,-0.219,-0.221,-0.223,-0.225,-0.227],\"amplitude_max\":[8.04,8.01,7.99"
    ",7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78],\"range_max\":[0.424,0.425,0.426"
    ",0.427,0.428,0.428,0.429,0.43,0.431,0.431,0.432]},\"settings\":{\"range\":"
    "{\"sigma_factor\":8,\"additive_value\":0.1},\"amplitude\":{\"sigma_factor\":4,"
    "\"additive_value\":1.0}}}"
};

// Correction parameters for window glass echoes
const MetaDataItemInfo RDB_RIEGL_WINDOW_ECHO_CORRECTION = {
    /* name        */ "riegl.window_echo_correction",
    /* title       */ "Window Echo Correction",
    /* description */ "Correction parameters for window glass echoes",
    /* status      */ RDB_STATUS_OPTIONAL,

    /* schema: */
    "{\"type\":\"object\",\"title\":\"Window Echo Correction\",\"$schema\":\"http://js"
    "on-schema.org/draft-04/schema#\",\"required\":[\"amplitude\",\"range\",\"slice"
    "s\"],\"properties\":{\"slices\":{\"type\":\"array\",\"items\":{\"type\":\"object\",\"r"
    "equired\":[\"amplitude\",\"table\"],\"properties\":{\"table\":{\"type\":\"array\",\""
    "items\":{\"type\":\"array\",\"items\":{\"type\":\"array\",\"items\":{\"type\":\"number"
    "\",\"description\":\"Table cell (item 0: amplitude in dB, 1: range in m, "
    "2: flags)\"},\"minItems\":3,\"description\":\"Table column (= range "
    "axis)\",\"maxItems\":3},\"minItems\":1,\"description\":\"Table row (= "
    "amplitude axis)\"},\"minItems\":1,\"description\":\"Correction table "
    "(dimension defined by the 'amplitude' and 'range' "
    "objects)\"},\"amplitude\":{\"type\":\"number\",\"description\":\"Window echo "
    "amplitude of slice in dB\"}},\"description\":\"Window echo correction "
    "parameter slice\"}},\"range\":{\"type\":\"object\",\"required\":[\"minimum\",\"max"
    "imum\",\"entries\"],\"properties\":{\"minimum\":{\"type\":\"number\",\"minimum\":-2"
    ".0,\"maximum\":2.0,\"description\":\"Minimum range in m\"},\"maximum\":{\"type\""
    ":\"number\",\"minimum\":-2.0,\"maximum\":2.0,\"description\":\"Maximum range in"
    " m\"},\"entries\":{\"type\":\"integer\",\"minimum\":1,\"description\":\"Number of "
    "range entries\"}},\"description\":\"Range axis of correction table\"},\"ampl"
    "itude\":{\"type\":\"object\",\"required\":[\"minimum\",\"maximum\",\"entries\"],\"pr"
    "operties\":{\"minimum\":{\"type\":\"number\",\"minimum\":0.0,\"description\":\"Min"
    "imum amplitude in "
    "dB\"},\"maximum\":{\"type\":\"number\",\"minimum\":0.0,\"description\":\"Maximum "
    "amplitude in "
    "dB\"},\"entries\":{\"type\":\"integer\",\"minimum\":1,\"description\":\"Number of "
    "amplitude entries\"}},\"description\":\"Amplitude axis of correction "
    "table\"}},\"description\":\"Correction parameters for window glass "
    "echoes\"}"
    ,
    /* example: */
    "{\"slices\":[{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],\"amplitude\""
    ":1.5},{\"table\":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],\"amplitude\":2.0}"
    "],\"range\":{\"minimum\":-1.5060822940732335,\"maximum\":1.5060822940732335,"
    "\"entries\":128},\"amplitude\":{\"minimum\":2,\"maximum\":20,\"entries\":128}}"
};

// Table of all meta data item details records
const MetaDataItemInfo* const RDB_META_DATA_ITEMS[] = {
    &RDB_RIEGL_ANGULAR_NOTCH_FILTER,
    &RDB_RIEGL_ATMOSPHERE,
    &RDB_RIEGL_BEAM_GEOMETRY,
    &RDB_RIEGL_CONTROL_OBJECT_CATALOG,
    &RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE,
    &RDB_RIEGL_DETECTION_PROBABILITY,
    &RDB_RIEGL_DEVICE,
    &RDB_RIEGL_DEVICE_GEOMETRY,
    &RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL,
    &RDB_RIEGL_DEVICE_OUTPUT_LIMITS,
    &RDB_RIEGL_DEVICES,
    &RDB_RIEGL_DMI_CONFIG,
    &RDB_RIEGL_DYNTRIG,
    &RDB_RIEGL_ECHO_INFO,
    &RDB_RIEGL_EXPONENTIAL_DECOMPOSITION,
    &RDB_RIEGL_GAUSSIAN_DECOMPOSITION,
    &RDB_RIEGL_GEO_TAG,
    &RDB_RIEGL_GEOMETRIC_SCALE_FACTOR,
    &RDB_RIEGL_GEOREFERENCING_PARAMETERS,
    &RDB_RIEGL_GNSS_PROCESSING_INFO,
    &RDB_RIEGL_ITEM_NAMES,
    &RDB_RIEGL_LAG_ANGLE_CORRECTION,
    &RDB_RIEGL_LICENSES,
    &RDB_RIEGL_MTA_SETTINGS,
    &RDB_RIEGL_NEAR_RANGE_CORRECTION,
    &RDB_RIEGL_NOISE_ESTIMATES,
    &RDB_RIEGL_NOTCH_FILTER,
    &RDB_RIEGL_PIXEL_INFO,
    &RDB_RIEGL_PLANE_PATCH_MATCHING,
    &RDB_RIEGL_PLANE_PATCH_STATISTICS,
    &RDB_RIEGL_PLANE_SLOPE_CLASS_INFO,
    &RDB_RIEGL_POINT_ATTRIBUTE_GROUPS,
    &RDB_RIEGL_POINTCLOUD_INFO,
    &RDB_RIEGL_POSE_ESTIMATION,
    &RDB_RIEGL_POSE_SENSORS,
    &RDB_RIEGL_PULSE_POSITION_MODULATION,
    &RDB_RIEGL_RANGE_STATISTICS,
    &RDB_RIEGL_RECEIVER_INTERNALS,
    &RDB_RIEGL_REFLECTANCE_CALCULATION,
    &RDB_RIEGL_REFLECTANCE_CORRECTION,
    &RDB_RIEGL_RILOC_PROCESSING_INFO,
    &RDB_RIEGL_SCAN_PATTERN,
    &RDB_RIEGL_SHOT_INFO,
    &RDB_RIEGL_STORED_FILTERS,
    &RDB_RIEGL_SYSTEM_DESCRIPTION,
    &RDB_RIEGL_TEMPERATURE_CALCULATION,
    &RDB_RIEGL_TIME_BASE,
    &RDB_RIEGL_TRAJECTORY_INFO,
    &RDB_RIEGL_TTIP_CONFIGURATION,
    &RDB_RIEGL_VERTEX_INFO,
    &RDB_RIEGL_VOXEL_INFO,
    &RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS,
    &RDB_RIEGL_WAVEFORM_INFO,
    &RDB_RIEGL_WAVEFORM_SETTINGS,
    &RDB_RIEGL_WINDOW_ANALYSIS,
    &RDB_RIEGL_WINDOW_ECHO_CORRECTION
};
const std::size_t RDB_META_DATA_ITEM_COUNT(
    sizeof(RDB_META_DATA_ITEMS) / sizeof(RDB_META_DATA_ITEMS[0])
);
const std::vector<const MetaDataItemInfo*> RDB_META_DATA_ITEMS_VECTOR(
    RDB_META_DATA_ITEMS, RDB_META_DATA_ITEMS + RDB_META_DATA_ITEM_COUNT
);

}}} // namespace riegl::rdb::pointcloud

#endif // RDB_354CFFB3_BDAA_4E38_98C8_9C09D0C0AD87
