/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    rdb-example-10-attribute-groups.cpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   RDB example 10: Attribute groups
 * \version 2019-04-17/AW: Initial version
 *
 *  This example shows how to open an existing database and output the
 *  point attribute group names and the point attributes in each group.
 *
 *  Example output:
 *
 *    Coordinates/Vectors:
 *      - XYZ (riegl.xyz)
 *    Time:
 *      - Timestamp (riegl.timestamp)
 *    Primary Attributes:
 *      - Reflectance (riegl.reflectance)
 *      - Amplitude (riegl.amplitude)
 *      - Deviation (riegl.deviation)
 *      - True Color (riegl.rgba)
 *      - Target Index (riegl.target_index)
 *      - Target Count (riegl.target_count)
 *    Secondary Attributes:
 *      - Mirror Facet (riegl.mirror_facet)
 *      - PID (riegl.id)
 *    Other Attributes:
 *      - Selected (riegl.selected)
 *      - Visible (riegl.visible)
 *
 *  Build instructions see "interface/cpp/riegl/README.TXT".
 *
 *******************************************************************************
 */

#include <vector>
#include <iostream>
#include <exception>

#include <riegl/rdb.hpp>

int main()
{
    try
    {
        // Access existing database
        riegl::rdb::Context context;
        riegl::rdb::Pointcloud rdb(context);
        riegl::rdb::pointcloud::OpenSettings settings;
        rdb.open("pointcloud.rdbx", settings);

        // Get list of point attributes
        const std::vector<std::string> attributes(rdb.pointAttribute().list());
        for (auto it = attributes.cbegin(); it != attributes.cend(); it++)
        {
            // Query point attribute group
            static std::string current_group;
            std::string group; uint32_t index(0);
            rdb.pointAttribute().group(*it, group, index);
            if (current_group != group) // print name of new group:
            {
                std::cout << group << ":" << std::endl;
                current_group = group;
            }

            // Query and print point attribute details
            const riegl::rdb::pointcloud::PointAttribute details(
                rdb.pointAttribute().get(*it)
            );
            std::cout << "  -"
                      << " " << details.title << " "
                      << "(" << details.name  << ")"
                      << std::endl;
        }

        // Success
        return 0;
    }
    catch(const riegl::rdb::Error &error)
    {
        std::cerr << error.what() << " (" << error.details() << ")" << std::endl;
        return 1; // error
    }
    catch(const std::exception &error)
    {
        std::cerr << error.what() << std::endl;
        return 1; // error
    }
}
