{*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL RDB 2 database meta data items.
 * \version 2015-10-27/AW: Initial version
 * \version 2015-11-25/AW: Item "Geo Tag" added
 * \version 2016-10-27/AW: Item "Voxel Information" added
 * \version 2016-11-17/AW: Item "Voxel Information" updated
 * \version 2016-12-12/AW: Item "Range Statistics" added
 * \version 2017-03-08/AW: Item "Plane Patch Statistics" added
 * \version 2017-04-05/AW: Items "Atmosphere" and "Geometric Scale Factor" added
 * \version 2017-08-22/AW: Items for waveform sample block and value files added
 * \version 2017-10-24/AW: Item "Gaussian Decomposition" added
 * \version 2017-11-13/AW: Item "riegl.scan_pattern" updated
 * \version 2017-11-21/AW: Item "riegl.trajectory_info" added
 * \version 2018-01-11/AW: Item "riegl.beam_geometry" added
 * \version 2018-01-15/AW: Item "riegl.reflectance_calculation" added
 * \version 2018-01-15/AW: Item "riegl.near_range_correction" added
 * \version 2018-01-15/AW: Item "riegl.device_geometry" added
 * \version 2018-02-13/AW: Item "riegl.notch_filter" added
 * \version 2018-03-08/AW: Item "riegl.window_echo_correction" added
 * \version 2018-03-15/AW: Item "riegl.pulse_position_modulation" added
 * \version 2018-05-24/AW: Item "riegl.pixel_info" added
 * \version 2018-06-08/AW: Item "riegl.shot_info" added
 * \version 2018-06-08/AW: Item "riegl.echo_info" added
 * \version 2018-06-14/AW: Item "riegl.mta_settings" added
 * \version 2018-06-14/AW: Item "riegl.receiver_internals" added
 * \version 2018-06-14/AW: Item "riegl.device_output_limits" added
 * \version 2018-06-26/AW: Schema: replace "number" with "integer" if applicable
 * \version 2018-07-09/AW: Item "riegl.pose_estimation" added
 * \version 2018-07-09/AW: Item "riegl.pose_sensors" added
 * \version 2018-09-20/AW: Item "riegl.pointcloud_info" added
 * \version 2018-11-08/AW: Item "riegl.scan_pattern" updated
 * \version 2018-11-13/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-06/AW: Item "riegl.receiver_internals" updated
 * \version 2019-03-21/AW: Item "riegl.device_geometry" updated
 * \version 2019-04-15/AW: Item "riegl.point_attribute_groups" added
 * \version 2019-04-30/AW: Item "riegl.waveform_settings" added
 * \version 2019-10-03/AW: Item "riegl.angular_notch_filter" added
 * \version 2019-10-03/AW: Item "riegl.noise_estimates" added
 * \version 2019-10-25/AW: Item "riegl.window_analysis" added
 * \version 2019-11-06/AW: Item "riegl.georeferencing_parameters" added
 * \version 2019-11-27/AW: Item "riegl.plane_patch_matching" added
 * \version 2019-12-13/AW: Items for tie-/control objects added
 * \version 2019-12-19/AW: Items for tie-/control objects added
 * \version 2020-02-04/AW: Item "riegl.detection_probability" added
 * \version 2020-02-04/AW: Item "riegl.licenses" added
 * \version 2020-04-27/AW: Item "riegl.waveform_info" updated (schema+example)
 * \version 2020-09-03/AW: Item "riegl.reflectance_correction" added
 * \version 2020-09-10/AW: Item "riegl.device_geometry_passive_channel" added
 * \version 2020-09-25/AW: Item "riegl.georeferencing_parameters" updated
 * \version 2020-09-25/AW: Item "riegl.trajectory_info" updated
 * \version 2020-09-29/AW: Item "riegl.temperature_calculation" added
 * \version 2020-10-23/AW: Item "riegl.exponential_decomposition" added (#3709)
 * \version 2020-11-30/AW: Item "riegl.notch_filter" updated (schema)
 * \version 2020-12-02/AW: Item "riegl.georeferencing_parameters" updated (schema)
 * \version 2021-02-02/AW: Item "riegl.plane_slope_class_info" added (rdbplanes/#7)
 * \version 2021-02-16/AW: Item "riegl.device_output_limits" updated (schema, #3811)
 * \version 2021-03-03/AW: Item "riegl.exponential_decomposition" updated (schema, #3822)
 * \version 2021-03-03/AW: Item "riegl.waveform_averaging_settings" added (#3821)
 * \version 2021-04-01/AW: Item "riegl.voxel_info" updated (schema, #3854)
 * \version 2021-04-16/AW: Item "riegl.voxel_info" updated (schema, #3866)
 * \version 2021-09-30/AW: Item "riegl.waveform_info" updated (schema+example, #4016)
 * \version 2021-10-04/AW: Improved spelling of the descriptions of some items
 * \version 2021-11-04/AW: Rename "riegl.record_names" to "riegl.item_names" (#4034)
 * \version 2022-03-11/AW: Item "riegl.devices" added (#4162)
 * \version 2022-03-14/AW: Item "riegl.stored_filters" added (#4164)
 * \version 2022-09-20/AW: Fix typos in schema of "riegl.georeferencing_parameters"
 * \version 2022-09-30/AW: Item "riegl.system_description" added (#4368)
 * \version 2022-10-10/AW: Correction of typos in the titles of some entries
 * \version 2022-10-20/AW: Item "riegl.stored_filters" updated (example, #4164)
 * \version 2023-01-19/AW: Item "riegl.ttip_configuration" added (#4449)
 * \version 2024-09-26/AW: Item "riegl.dyntrig" added (#5310)
 * \version 2025-03-21/AW: Item "riegl.dmi_config" added (#5571)
 * \version 2025-07-01/AW: Item "riegl.gnss_processing_info" added (#5777)
 * \version 2025-07-01/AW: Item "riegl.riloc_processing_info" added (#5777)
 * \version 2025-11-11/AW: Item "riegl.lag_angle_correction" added (#6068)
 *
 *******************************************************************************
 *}

unit riegl.rdb.default.metadata;

{$WRITEABLECONST ON} // see section "initialization"

interface

// Meta data items definition version
const
  RDB_META_DATA_ITEMS_VERSION = '1.1.57';
  RDB_META_DATA_ITEMS_DATE = '2025-11-11';

// Meta data item states
const
  RDB_STATUS_REQUIRED = 1; // mandatory item
  RDB_STATUS_OPTIONAL = 2; // optional item

// Container record for meta data item details
type
  TRDBPointcloudMetaDataItemInfo = record
    Name        : System.String;
    Title       : System.String;
    Description : System.String;
    Status      : System.Byte;
    Schema      : System.String;
    Example     : System.String;
  end;
  PRDBPointcloudMetaDataItemInfo = ^TRDBPointcloudMetaDataItemInfo;

// Angular notch filter parameters for window glass echoes
const
  RDB_RIEGL_ANGULAR_NOTCH_FILTER : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.angular_notch_filter';
    Title       : 'Angular Notch Filter';
    Description : 'Angular notch filter parameters for window glass echoes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Angular Notch Filter","required":["angle","range_mean","ampl'
      + 'itude_mean"],"$schema":"http://json-schema.org/draft-04/schema#","prop'
      + 'erties":{"amplitude_mean":{"items":{"type":"number"},"type":"array","d'
      + 'escription":"Mean amplitude [dB]"},"angle":{"items":{"type":"number"},'
      + '"type":"array","description":"Angle [deg]"},"range_mean":{"items":{"ty'
      + 'pe":"number"},"type":"array","description":"Mean range '
      + '[m]"}},"description":"Angular notch filter parameters for window glass'
      + ' echoes","type":"object"}'
    );

    Example : (''
      + '{"amplitude_mean":[3.780913,3.780913,3.480913,3.120913,2.850913,2.7209'
      + '13,2.680913,2.610913,2.530913,2.570913,2.570913],"angle":[14.0,15.0,16'
      + '.0,17.0,18.0,19.0,20.0,21.0,22.0,23.0,24.0],"range_mean":[0.094,0.094,'
      + '0.09075,0.08675,0.0815,0.0775,0.074,0.071,0.068,0.0675,0.06475]}'
    );
  );

// Atmospheric parameters
const
  RDB_RIEGL_ATMOSPHERE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.atmosphere';
    Title       : 'Atmosphere';
    Description : 'Atmospheric parameters';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Atmospheric Parameters","required":["temperature","pressure"'
      + ',"rel_humidity","pressure_sl","amsl","group_velocity","attenuation","w'
      + 'avelength"],"$schema":"http://json-schema.org/draft-04/schema#","prope'
      + 'rties":{"group_velocity":{"type":"number","description":"Group '
      + 'velocity of laser beam '
      + '[m/s]"},"rel_humidity":{"type":"number","description":"Relative '
      + 'humidity along measurement path '
      + '[%]"},"temperature":{"type":"number","description":"Temperature along '
      + 'measurement path '
      + '[\u00b0C]"},"pressure_sl":{"type":"number","description":"Atmospheric '
      + 'pressure at sea level '
      + '[mbar]"},"amsl":{"type":"number","description":"Height above mean sea '
      + 'level (AMSL) [m]"},"wavelength":{"type":"number","description":"Laser '
      + 'wavelength [nm]"},"pressure":{"type":"number","description":"Pressure '
      + 'along measurment path '
      + '[mbar]"},"attenuation":{"type":"number","description":"Atmospheric '
      + 'attenuation [1/km]"}},"description":"Atmospheric '
      + 'parameters","type":"object"}'
    );

    Example : (''
      + '{"group_velocity":299711000.0,"rel_humidity":63,"temperature":7,"press'
      + 'ure_sl":970,"amsl":0,"wavelength":1550,"pressure":970,"attenuation":0.'
      + '028125}'
    );
  );

// Laser beam geometry details
const
  RDB_RIEGL_BEAM_GEOMETRY : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.beam_geometry';
    Title       : 'Beam Geometry';
    Description : 'Laser beam geometry details';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Beam Geometry","required":["beam_exit_diameter","beam_diverg'
      + 'ence"],"$schema":"http://json-schema.org/draft-04/schema#","properties'
      + '":{"beam_divergence":{"minimum":0,"exclusiveMinimum":false,"descriptio'
      + 'n":"Beam divergence in far field [rad]","type":"number"},"beam_exit_di'
      + 'ameter":{"minimum":0,"exclusiveMinimum":false,"description":"Beam '
      + 'width at exit aperture [m]","type":"number"}},"description":"Laser '
      + 'beam geometry details","type":"object"}'
    );

    Example : (''
      + '{"beam_divergence":0.0003,"beam_exit_diameter":0.0072}'
    );
  );

// List of control object type definitions
const
  RDB_RIEGL_CONTROL_OBJECT_CATALOG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.control_object_catalog';
    Title       : 'Control Object Catalog';
    Description : 'List of control object type definitions';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Control Object Catalog","definitions":{"cylinder":{"allOf":['
      + '{"$ref":"#/definitions/common"},{"required":["shape","diameter","heigh'
      + 't"],"properties":{"shape":{"description":"shape identifier","enum":["c'
      + 'ylinder"],"type":"string"},"diameter":{"minimum":0,"exclusiveMinimum":'
      + 'true,"description":"diameter in meters","type":"number"},"height":{"mi'
      + 'nimum":0,"exclusiveMinimum":true,"description":"height in '
      + 'meters","type":"number"}},"description":"cylinder specific properties"'
      + ',"type":"object"}],"description":"cylinder","type":"object"},"round_co'
      + 'rner_cube_prism":{"allOf":[{"$ref":"#/definitions/common"},{"required"'
      + ':["shape","diameter"],"properties":{"shape":{"description":"shape iden'
      + 'tifier","enum":["round_corner_cube_prism"],"type":"string"},"diameter"'
      + ':{"minimum":0,"exclusiveMinimum":true,"description":"diameter in '
      + 'meters","type":"number"},"offset":{"description":"offset in meters, '
      + 'e.g. reflector constant '
      + '(optional)","type":"number"}},"description":"round corner cube prism '
      + 'specific properties","type":"object"}],"description":"round corner '
      + 'cube prism","type":"object"},"rectangle":{"allOf":[{"$ref":"#/definiti'
      + 'ons/common"},{"required":["shape","length","width"],"properties":{"wid'
      + 'th":{"minimum":0,"exclusiveMinimum":true,"description":"width in meter'
      + 's","type":"number"},"shape":{"type":"string","enum":["rectangle"]},"le'
      + 'ngth":{"minimum":0,"exclusiveMinimum":true,"description":"length in '
      + 'meters","type":"number"}},"description":"rectangle specific properties'
      + '","type":"object"}],"description":"rectangle","type":"object"},"circul'
      + 'ar_disk":{"allOf":[{"$ref":"#/definitions/common"},{"required":["shape'
      + '","diameter"],"properties":{"shape":{"description":"shape identifier",'
      + '"enum":["circular_disk"],"type":"string"},"diameter":{"minimum":0,"exc'
      + 'lusiveMinimum":true,"description":"diameter in '
      + 'meters","type":"number"},"offset":{"description":"offset in meters, '
      + 'e.g. reflector constant '
      + '(optional)","type":"number"}},"description":"circular disk specific '
      + 'properties","type":"object"}],"description":"circular disk","type":"ob'
      + 'ject"},"checkerboard2x2":{"allOf":[{"$ref":"#/definitions/common"},{"r'
      + 'equired":["shape","square_length"],"properties":{"shape":{"description'
      + '":"shape identifier","enum":["checkerboard2x2"],"type":"string"},"squa'
      + 're_length":{"minimum":0,"exclusiveMinimum":true,"description":"length '
      + 'of a square of the checkerboard in '
      + 'meters","type":"number"}},"description":"checkerboard specific '
      + 'properties","type":"object"}],"description":"checkerboard 2 by 2","typ'
      + 'e":"object"},"common":{"required":["name","shape"],"properties":{"surf'
      + 'ace_type":{"description":"surface material type","enum":["retro_reflec'
      + 'tive_foil","diffuse"],"type":"string"},"shape":{"description":"shape i'
      + 'dentifier","enum":["rectangle","checkerboard2x2","chevron","circular_d'
      + 'isk","cylinder","sphere","round_corner_cube_prism"],"type":"string"},"'
      + 'description":{"description":"string describing the '
      + 'object","type":"string"},"name":{"minLength":3,"description":"unique '
      + 'type identifier","type":"string"}},"description":"common object proper'
      + 'ties","type":"object"},"sphere":{"allOf":[{"$ref":"#/definitions/commo'
      + 'n"},{"required":["shape","diameter"],"properties":{"shape":{"descripti'
      + 'on":"shape identifier","enum":["sphere"],"type":"string"},"diameter":{'
      + '"minimum":0,"exclusiveMinimum":true,"description":"diameter in '
      + 'meters","type":"number"}},"description":"sphere specific properties","'
      + 'type":"object"}],"description":"sphere","type":"object"},"chevron":{"a'
      + 'llOf":[{"$ref":"#/definitions/common"},{"required":["shape","outside_e'
      + 'dge_length","thickness"],"properties":{"thickness":{"minimum":0,"exclu'
      + 'siveMinimum":true,"description":"thickness in meters","type":"number"}'
      + ',"outside_edge_length":{"minimum":0,"exclusiveMinimum":true,"descripti'
      + 'on":"length of the two outer edges in '
      + 'meters","type":"number"},"shape":{"description":"shape identifier","en'
      + 'um":["chevron"],"type":"string"}},"description":"chevron specific prop'
      + 'erties","type":"object"}],"description":"chevron","type":"object"}},"r'
      + 'equired":["types"],"properties":{"types":{"items":{"oneOf":[{"$ref":"#'
      + '/definitions/rectangle"},{"$ref":"#/definitions/checkerboard2x2"},{"$r'
      + 'ef":"#/definitions/chevron"},{"$ref":"#/definitions/circular_disk"},{"'
      + '$ref":"#/definitions/cylinder"},{"$ref":"#/definitions/sphere"},{"$ref'
      + '":"#/definitions/round_corner_cube_prism"}],"type":"object"},"type":"a'
      + 'rray","uniqueItems":true}},"$schema":"http://json-schema.org/draft-04/'
      + 'schema#","description":"List of control object type '
      + 'definitions","type":"object"}'
    );

    Example : (''
      + '{"comments":["This file contains a list of control object types (aka. '
      + '''catalog'').","Each type is described by an object,","which must '
      + 'contain at least the following parameters:","  - name: unique '
      + 'identifier of the type","  - shape: one of the following supported '
      + 'shapes:","      - rectangle","      - checkerboard2x2","      - '
      + 'chevron","      - circular_disk","      - cylinder","      - sphere","'
      + '      - round_corner_cube_prism","Depending on ''shape'', the following '
      + 'parameters must/may be specified:","  - rectangle:","      - length: '
      + 'length in meters","      - width: width in meters","  - '
      + 'checkerboard2x2:","      - square_length: length of a square of the '
      + 'checkerboard in meters","  - circular_disk:","      - diameter: '
      + 'diameter in meters","      - offset: offset in meters, e.g. reflector '
      + 'constant (optional)","  - chevron:","      - outside_edge_length: '
      + 'length of the two outer edges in meters","      - thickness: thickness'
      + ' in meters","  - cylinder:","      - diameter: diameter in meters","'
      + '      - height: height in meters","  - sphere:","      - diameter: '
      + 'diameter in meters","  - round_corner_cube_prism:","      - diameter: '
      + 'diameter in meters","      - offset: offset in meters, e.g. reflector '
      + 'constant (optional)","Optional parameters:","    - description: string'
      + ' describing the object","    - surface_type: surface material type '
      + '(either ''retro_reflective_foil'' or ''diffuse'')"],"types":[{"width":0.3,'
      + '"shape":"rectangle","description":"Rectangle (60cm x '
      + '30cm)","length":0.6,"name":"Rectangle '
      + '60x30"},{"width":0.4,"shape":"rectangle","description":"Rectangle '
      + '(80cm x 40cm)","length":0.8,"name":"Rectangle 80x40"},{"square_length"'
      + ':0.3,"shape":"checkerboard2x2","description":"Checkerboard (square '
      + 'length: 30cm)","name":"Checkerboard2x2 30"},{"square_length":0.5,"shap'
      + 'e":"checkerboard2x2","description":"Checkerboard (square length: '
      + '50cm)","name":"Checkerboard2x2 50"},{"thickness":0.1016,"outside_edge_'
      + 'length":0.6096,"shape":"chevron","description":"Chevron (a=24''''; '
      + 'b=4'''')","name":"Chevron 24''''/4''''"},{"diameter":0.5,"surface_type":"dif'
      + 'fuse","shape":"circular_disk","description":" Circular Disk (diameter:'
      + ' 50cm)","name":"Circular Disk 50"},{"name":"RIEGL flat reflector 50 mm'
      + '","surface_type":"retro_reflective_foil","shape":"circular_disk","diam'
      + 'eter":0.05,"offset":0.0,"description":"flat circular reflector from '
      + 'retro-reflective foil"},{"name":"RIEGL flat reflector 100 mm","surface'
      + '_type":"retro_reflective_foil","shape":"circular_disk","diameter":0.1,'
      + '"offset":0.0,"description":"flat circular reflector from '
      + 'retro-reflective foil"},{"name":"RIEGL flat reflector 150 mm","surface'
      + '_type":"retro_reflective_foil","shape":"circular_disk","diameter":0.15'
      + ',"offset":0.0,"description":"flat circular reflector from '
      + 'retro-reflective foil"},{"name":"RIEGL cylindrical reflector 50 mm","s'
      + 'urface_type":"retro_reflective_foil","shape":"cylinder","diameter":0.0'
      + '5,"height":0.05,"description":"cylindrical reflector from '
      + 'retro-reflective foil"},{"name":"RIEGL cylindrical reflector 100 mm","'
      + 'surface_type":"retro_reflective_foil","shape":"cylinder","diameter":0.'
      + '1,"height":0.1,"description":"cylindrical reflector from '
      + 'retro-reflective '
      + 'foil"},{"diameter":0.2,"shape":"sphere","description":"Sphere '
      + '(diameter: 200 mm)","name":"Sphere 200 mm"},{"diameter":0.05,"shape":"'
      + 'round_corner_cube_prism","description":"round corner cube '
      + 'prism","offset":0.0,"name":"Corner Cube Prism 50 mm"}]}'
    );
  );

// Details about the control object reference file
const
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.control_object_reference_file';
    Title       : 'Control Object Reference file';
    Description : 'Details about the control object reference file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"reference_file":{"required":["file_uuid","file_path"],'
      + '"properties":{"file_path":{"description":"Path of the control object '
      + 'file relative to referring '
      + 'file","type":"string"},"file_uuid":{"description":"Control object '
      + 'file''s Universally Unique Identifier (RFC '
      + '4122)","type":"string"}},"type":"object","description":"Reference to a'
      + ' control object file"}},"title":"Control Object Reference file","$sche'
      + 'ma":"http://json-schema.org/draft-04/schema#","description":"Details '
      + 'about the control object reference file","type":"object"}'
    );

    Example : (''
      + '{"reference_file":{"file_path":"../../../10_CONTROL_OBJECTS/ControlPoi'
      + 'nts.cpx","file_uuid":"810f5d57-eccf-49ed-b07a-0cdd109b4213"}}'
    );
  );

// Detection probability as a function of amplitude
const
  RDB_RIEGL_DETECTION_PROBABILITY : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.detection_probability';
    Title       : 'Detection Probability';
    Description : 'Detection probability as a function of amplitude';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Detection Probability","required":["amplitude","detection_pr'
      + 'obability"],"$schema":"http://json-schema.org/draft-04/schema#","prope'
      + 'rties":{"amplitude":{"items":{"type":"number"},"type":"array","descrip'
      + 'tion":"Amplitude [dB]"},"detection_probability":{"items":{"type":"numb'
      + 'er"},"type":"array","description":"Detection probability '
      + '[0..1]"}},"description":"Detection probability as a function of '
      + 'amplitude","type":"object"}'
    );

    Example : (''
      + '{"amplitude":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10.0],"detection'
      + '_probability":[0.0,0.5,0.8,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]}'
    );
  );

// Details about the device used to acquire the point cloud
const
  RDB_RIEGL_DEVICE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device';
    Title       : 'Device Information';
    Description : 'Details about the device used to acquire the point cloud';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Device Information","required":["device_type","serial_number'
      + '"],"$schema":"http://json-schema.org/draft-04/schema#","properties":{"'
      + 'device_name":{"description":"Optional device name (e.g. ''Scanner 1'' '
      + 'for multi-scanner '
      + 'systems)","type":"string"},"device_build":{"description":"Device build'
      + ' variant","type":"string"},"device_type":{"description":"Device type '
      + 'identifier (e.g. VZ-400i)","type":"string"},"channel_number":{"minimum'
      + '":0,"exclusiveMinimum":false,"description":"Laser channel number (not '
      + 'defined or 0: single channel '
      + 'device)","type":"integer"},"serial_number":{"description":"Device '
      + 'serial number (e.g. '
      + 'S2221234)","type":"string"},"channel_text":{"description":"Optional '
      + 'channel description (e.g. ''Green Channel'' for multi-channel '
      + 'devices)","type":"string"}},"description":"Details about the device '
      + 'used to acquire the point cloud","type":"object"}'
    );

    Example : (''
      + '{"device_name":"Scanner 1","device_build":"","device_type":"VZ-400i","'
      + 'channel_number":0,"serial_number":"S2221234","channel_text":""}'
    );
  );

// Scanner device geometry details
const
  RDB_RIEGL_DEVICE_GEOMETRY : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device_geometry';
    Title       : 'Device Geometry';
    Description : 'Scanner device geometry details';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Device Geometry","required":["primary"],"$schema":"http://js'
      + 'on-schema.org/draft-04/schema#","properties":{"secondary":{"required":'
      + '["ID","content"],"properties":{"ID":{"items":{"type":"integer"},"maxIt'
      + 'ems":2,"description":"Structure identifier","minItems":2,"type":"array'
      + '"},"content":{"items":{"type":"number"},"description":"Internal '
      + 'calibration values","type":"array"}},"description":"Additional device '
      + 'geometry structure (optional)","type":"object"},"amu":{"properties":{"'
      + 'lineCC":{"minimum":0,"exclusiveMinimum":false,"description":"Line '
      + 'Circle Count (number of LSBs per full rotation about line axis)","type'
      + '":"number"},"frameCC":{"minimum":0,"exclusiveMinimum":false,"descripti'
      + 'on":"Frame Circle Count (number of LSBs per full rotation about frame '
      + 'axis)","type":"number"}},"description":"Angle Measurement Unit","type"'
      + ':"object"},"primary":{"required":["ID","content"],"properties":{"ID":{'
      + '"items":{"type":"integer"},"maxItems":2,"description":"Structure ident'
      + 'ifier","minItems":2,"type":"array"},"content":{"items":{"type":"number'
      + '"},"description":"Internal calibration '
      + 'values","type":"array"}},"description":"Primary device geometry '
      + 'structure (mandatory)","type":"object"}},"description":"Scanner device'
      + ' geometry details","type":"object"}'
    );

    Example : (''
      + '{"secondary":{"ID":[91,0],"content":[0]},"amu":{"lineCC":124000,"frame'
      + 'CC":124000},"primary":{"ID":[4,0],"content":[0]}}'
    );
  );

// Scanner passive channel device geometry details
const
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device_geometry_passive_channel';
    Title       : 'Device Geometry Passive Channel';
    Description : 'Scanner passive channel device geometry details';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Device Geometry Passive Channel","required":["primary"],"$sc'
      + 'hema":"http://json-schema.org/draft-04/schema#","properties":{"primary'
      + '":{"required":["ID","content"],"properties":{"ID":{"items":{"type":"in'
      + 'teger"},"maxItems":2,"description":"Structure identifier","minItems":2'
      + ',"type":"array"},"content":{"items":{"type":"number"},"description":"I'
      + 'nternal calibration values","type":"array"}},"description":"Primary '
      + 'device geometry structure '
      + '(mandatory)","type":"object"}},"description":"Scanner passive channel '
      + 'device geometry details","type":"object"}'
    );

    Example : (''
      + '{"primary":{"ID":[143,0],"content":[0]}}'
    );
  );

// Limits of the measured values output by the device
const
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.device_output_limits';
    Title       : 'Device Output Limits';
    Description : 'Limits of the measured values output by the device';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"range_maximum":{"type":"number","description":"Maximum'
      + ' possible range in meters."},"reflectance_minimum":{"type":"number","d'
      + 'escription":"Minimum possible reflectance in '
      + 'dB."},"echo_count_maximum":{"type":"number","description":"Maximum '
      + 'number of echoes a laser shot can have."},"background_radiation_minimu'
      + 'm":{"type":"number","description":"Minimum possible background radiati'
      + 'on."},"deviation_minimum":{"type":"number","description":"Minimum '
      + 'possible pulse shape deviation."},"amplitude_minimum":{"type":"number"'
      + ',"description":"Minimum possible amplitude in '
      + 'dB."},"amplitude_maximum":{"type":"number","description":"Maximum '
      + 'possible amplitude in '
      + 'dB."},"deviation_maximum":{"type":"number","description":"Maximum '
      + 'possible pulse shape deviation."},"mta_zone_count_maximum":{"type":"nu'
      + 'mber","description":"Maximum number of MTA zones."},"background_radiat'
      + 'ion_maximum":{"type":"number","description":"Maximum possible '
      + 'background '
      + 'radiation."},"range_minimum":{"type":"number","description":"Minimum '
      + 'possible range in meters."},"reflectance_maximum":{"type":"number","de'
      + 'scription":"Maximum possible reflectance in dB."}},"title":"Device '
      + 'Output Limits","$schema":"http://json-schema.org/draft-04/schema#","de'
      + 'scription":"Limits of the measured values output by the device. The '
      + 'limits depend on the device type, measurement program and/or scan '
      + 'pattern.","type":"object"}'
    );

    Example : (''
      + '{"range_maximum":10000.0,"deviation_minimum":-1,"range_minimum":2.9,"a'
      + 'mplitude_maximum":100.0,"deviation_maximum":32767,"background_radiatio'
      + 'n_maximum":0,"mta_zone_count_maximum":7,"reflectance_minimum":-100.0,"'
      + 'background_radiation_minimum":0,"amplitude_minimum":0.0,"reflectance_m'
      + 'aximum":100.0}'
    );
  );

// Details about the devices used to acquire the point cloud
const
  RDB_RIEGL_DEVICES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.devices';
    Title       : 'Devices Information';
    Description : 'Details about the devices used to acquire the point cloud';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"items":{"properties":{"device_name":{"description":"Optional device '
      + 'name (e.g. ''Scanner 1'' for multi-scanner '
      + 'systems)","type":"string"},"device_build":{"description":"Device build'
      + ' variant","type":"string"},"device_type":{"description":"Device type '
      + 'identifier (e.g. VZ-400i)","type":"string"},"channel_number":{"minimum'
      + '":0,"exclusiveMinimum":false,"description":"Laser channel number (not '
      + 'defined or 0: single channel '
      + 'device)","type":"integer"},"signed":{"description":"Flag that is set '
      + 'when the original ''riegl.device'' entry in the source file was '
      + 'correctly '
      + 'signed.","type":"boolean"},"serial_number":{"description":"Device '
      + 'serial number (e.g. '
      + 'S2221234)","type":"string"},"channel_text":{"description":"Optional '
      + 'channel description (e.g. ''Green Channel'' for multi-channel devices)",'
      + '"type":"string"}},"type":"object","required":["device_type","serial_nu'
      + 'mber"]},"title":"Devices Information","$schema":"http://json-schema.or'
      + 'g/draft-04/schema#","description":"Details about the devices used to '
      + 'acquire the point cloud (e.g. when merging point clouds of different '
      + 'devices)","type":"array"}'
    );

    Example : (''
      + '[{"device_name":"Scanner 1","device_build":"","device_type":"VZ-400i",'
      + '"channel_number":0,"signed":false,"serial_number":"S2221234","channel_'
      + 'text":""},{"device_name":"Scanner 2","device_build":"","device_type":"'
      + 'VQ-1560i-DW","channel_number":1,"signed":true,"serial_number":"S222268'
      + '0","channel_text":""},{"device_name":"Scanner 3","device_build":"","de'
      + 'vice_type":"VQ-1560i-DW","channel_number":2,"signed":true,"serial_numb'
      + 'er":"S2222680","channel_text":""}]'
    );
  );

// Distance measuring instrument configuration
const
  RDB_RIEGL_DMI_CONFIG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.dmi_config';
    Title       : 'DMI Configuration';
    Description : 'Distance measuring instrument configuration';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"DMI Configuration","required":["dmi_count_dir","dmi_incr","d'
      + 'mi_min_time","dmi_max_time","dmi_dist_per_tick"],"$schema":"http://jso'
      + 'n-schema.org/draft-04/schema#","properties":{"dmi_max_time":{"minimum"'
      + ':0,"type":"integer","exclusiveMaximum":false,"description":"Maximum '
      + 'time interval between trigger events, 0 means timeout is inactive [0.1'
      + ' msec]","maximum":65535,"exclusiveMinimum":false},"dmi_dist_per_tick":'
      + '{"description":"Distance per tick, just informative [m]","type":"numbe'
      + 'r"},"dmi_min_time":{"minimum":0,"type":"integer","exclusiveMaximum":fa'
      + 'lse,"description":"Minimum time interval between trigger events, 0 '
      + 'means rate limitation is inactive [0.1 msec]","maximum":65535,"exclusi'
      + 'veMinimum":false},"dmi_incr":{"minimum":0,"type":"integer","exclusiveM'
      + 'aximum":false,"description":"Increment in ticks, 0 means distance '
      + 'trigger is deactivated","maximum":2147483647,"exclusiveMinimum":false}'
      + ',"dmi_count_dir":{"minimum":0,"type":"integer","exclusiveMaximum":fals'
      + 'e,"description":"DMI count direction is inverted","maximum":1,"exclusi'
      + 'veMinimum":false}},"description":"Distance measuring instrument '
      + 'configuration","type":"object"}'
    );

    Example : (''
      + '{"dmi_max_time":100,"dmi_dist_per_tick":0.00024399999529123306,"dmi_mi'
      + 'n_time":0,"dmi_incr":0,"dmi_count_dir":0}'
    );
  );

// Dyntrig
const
  RDB_RIEGL_DYNTRIG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.dyntrig';
    Title       : 'Dyntrig';
    Description : 'Dyntrig';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"post":{"required":["nc","a"],"properties":{"nc":{"mini'
      + 'mum":0,"type":"integer","maximum":255,"description":"Number of entries'
      + '"},"a":{"items":{"type":"number"},"description":"...","type":"array","'
      + 'minItems":1,"maxItems":2080}},"type":"object","description":"Dyntrig '
      + 'post values"},"tab":{"items":{"required":["thra1","del1","dec1","thra2'
      + '","del2","dec2"],"properties":{"dec1":{"description":"Decay of '
      + 'post-trigger threshold after del1 in units of '
      + 'dB/m","type":"number"},"del1":{"description":"Length of constant '
      + 'post-trigger threshold elevation in units of '
      + 'm","type":"number"},"thra1":{"description":"Post-trigger threshold in '
      + 'units of dB","type":"number"},"dec2":{"description":"Rise of '
      + 'pre-trigger threshold after del2 in units of '
      + 'dB/m","type":"number"},"thra2":{"description":"Pre-trigger threshold '
      + 'in units of dB","type":"number"},"del2":{"description":"Length of '
      + 'constant pre-trigger threshold elevation in units of m","type":"number'
      + '"}},"description":"...","type":"object"},"description":"...","type":"a'
      + 'rray","minItems":1,"maxItems":128},"pre":{"required":["nc","a"],"prope'
      + 'rties":{"nc":{"minimum":0,"type":"integer","maximum":255,"description"'
      + ':"Number of entries"},"a":{"items":{"type":"number"},"description":"..'
      + '.","type":"array","minItems":1,"maxItems":2080}},"type":"object","desc'
      + 'ription":"Dyntrig pre '
      + 'values"},"offset":{"type":"number","description":"Offset in units of '
      + 'dB, for calculation of '
      + 'line"},"scale":{"type":"number","description":"Scale in units of dB, '
      + 'for calculation of line: line number N=(amplitude-offset)/scale"}},"ti'
      + 'tle":"Dyntrig","$schema":"http://json-schema.org/draft-04/schema#","de'
      + 'scription":"Dyntrig"}'
    );

    Example : (''
      + '{"post":{"nc":128,"a":[78,86,126,134,31]},"tab":[{"dec1":0.0,"del1":0.'
      + '0,"thra1":0.0,"dec2":0.0,"thra2":0.0,"del2":0.0}],"pre":{"nc":128,"a":'
      + '[78,86,126,134,31]},"offset":0.0,"scale":1.0}'
    );
  );

// Details about echo files
const
  RDB_RIEGL_ECHO_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.echo_info';
    Title       : 'Echo Information';
    Description : 'Details about echo files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Echo Information","required":["echo_file"],"$schema":"http:/'
      + '/json-schema.org/draft-04/schema#","properties":{"echo_file":{"propert'
      + 'ies":{"file_extension":{"description":"Echo file extension, without '
      + 'the leading dot","type":"string"},"file_uuid":{"description":"File''s '
      + 'Universally Unique Identifier (RFC 4122)","type":"string"}},"type":"ob'
      + 'ject","required":["file_extension"]}},"description":"Details about '
      + 'echo files","type":"object"}'
    );

    Example : (''
      + '{"echo_file":{"file_extension":"owp","file_uuid":"26a03615-67c0-4bea-8'
      + 'fe8-c577378fe661"}}'
    );
  );

// Details for exponential decomposition of full waveform data
const
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.exponential_decomposition';
    Title       : 'Exponential Decomposition';
    Description : 'Details for exponential decomposition of full waveform data';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"patternProperties":{"^[0-9]+$":{"$ref":"#/definitions/channel","desc'
      + 'ription":"one field per channel, field name is channel '
      + 'index"}},"title":"Exponential Decomposition","definitions":{"channel":'
      + '{"properties":{"parameter":{"required":["A","B","gamma","omega"],"prop'
      + 'erties":{"gamma":{"items":{"type":"number"},"description":"decay in 1/'
      + 'second","type":"array"},"B":{"items":{"type":"number"},"description":"'
      + 'imaginary part of amplitude factor in units of full-scale","type":"arr'
      + 'ay"},"omega":{"items":{"type":"number"},"description":"angular '
      + 'frequency in '
      + 'Hz","type":"array"},"A":{"items":{"type":"number"},"description":"real'
      + ' part of amplitude factor in units of '
      + 'full-scale","type":"array"}},"description":"parameters of the syswave '
      + 'exponential sum","type":"object"},"a_lin":{"minimum":0,"type":"number"'
      + ',"exclusiveMaximum":false,"description":"relative linear amplitude '
      + 'range [0..1]","maximum":1,"exclusiveMinimum":false},"delay":{"descript'
      + 'ion":"delay calibration in '
      + 'seconds","type":"number"},"scale":{"description":"amplitude calibratio'
      + 'n","type":"number"}},"type":"object","required":["delay","scale","para'
      + 'meter"]}},"additionalProperties":false,"$schema":"http://json-schema.o'
      + 'rg/draft-04/schema#","description":"Details for exponential '
      + 'decomposition of full waveform data","type":"object"}'
    );

    Example : (''
      + '{"0":{"parameter":{"gamma":[-1094726528.0,-769562752.0,-848000064.0],"'
      + 'B":[-3.9813032150268555,0.08622030913829803,-0.3152860999107361],"omeg'
      + 'a":[352020896.0,3647927552.0,-1977987072.0],"A":[0.9772450923919678,0.'
      + '3354335129261017,-1.312678575515747]},"a_lin":0.27,"delay":3.783458418'
      + '887631e-09,"scale":1.0},"1":{"parameter":{"gamma":[-1094726528.0,-7695'
      + '62752.0,-848000064.0],"B":[-3.9,0.0,-0.3],"omega":[352020896.0,3647927'
      + '552.0,-1977987072.0],"A":[0.9,0.3,-1.3]},"a_lin":0.9,"delay":3.5e-09,"'
      + 'scale":1.0}}'
    );
  );

// Details for Gaussian decomposition of full waveform data
const
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.gaussian_decomposition';
    Title       : 'Gaussian Decomposition';
    Description : 'Details for Gaussian decomposition of full waveform data';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Gaussian Decomposition","required":["amplitude_lsb_low_power'
      + '","amplitude_lsb_high_power","amplitude_db","range_offset_sec_low_powe'
      + 'r","range_offset_sec_high_power"],"$schema":"http://json-schema.org/dr'
      + 'aft-04/schema#","properties":{"range_offset_sec_high_power":{"items":{'
      + '"type":"number"},"type":"array"},"amplitude_db":{"items":{"type":"numb'
      + 'er"},"type":"array"},"range_offset_sec_low_power":{"items":{"type":"nu'
      + 'mber"},"type":"array"},"amplitude_lsb_low_power":{"items":{"type":"num'
      + 'ber"},"type":"array"},"amplitude_lsb_high_power":{"items":{"type":"num'
      + 'ber"},"type":"array"}},"description":"riegl.gaussian_decomposition '
      + 'contains information relevant for extracting calibrated amplitudes and'
      + ' calibrated ranges from a Gaussian decomposition of full waveform '
      + 'data. This information is contained in a table with five columns. Two '
      + 'columns are to be used as input: amplitude_lsb_low_power and '
      + 'amplitude_lsb_high_power. The other three columns provide the outputs.'
      + ' Amplitude_db gives the calibrated amplitude in the optical regime in '
      + 'decibels. The range offset columns provide additive range offsets, '
      + 'given in units of seconds, for each channel.","type":"object"}'
    );

    Example : (''
      + '{"range_offset_sec_high_power":[],"amplitude_db":[],"range_offset_sec_'
      + 'low_power":[],"amplitude_lsb_low_power":[],"amplitude_lsb_high_power":'
      + '[]}'
    );
  );

// Point cloud georeferencing information
const
  RDB_RIEGL_GEO_TAG : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.geo_tag';
    Title       : 'Geo Tag';
    Description : 'Point cloud georeferencing information';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"crs":{"properties":{"wkt":{"description":"\"Well-Known'
      + ' Text\" string, OGC WKT dialect (see http://www.opengeospatial.org/sta'
      + 'ndards/wkt-crs)","type":"string"},"epsg":{"minimum":0,"description":"E'
      + 'PSG code","type":"integer"},"name":{"description":"Coordinate '
      + 'reference system '
      + 'name","type":"string"}},"type":"object","description":"Global '
      + 'Coordinate Reference System. Please note that only 3D Cartesian '
      + 'Coordinate Systems are allowed."},"pose":{"items":{"items":{"descripti'
      + 'on":"columns","type":"number"},"maxItems":4,"description":"rows","minI'
      + 'tems":4,"type":"array"},"maxItems":4,"type":"array","minItems":4,"desc'
      + 'ription":"Coordinate Transformation Matrix to transform from File '
      + 'Coordinate System to Global Coordinate Reference System. 4x4 matrix '
      + 'stored as two dimensional array, row major order."}},"title":"Geo Tag"'
      + ',"$schema":"http://json-schema.org/draft-04/schema#","description":"Po'
      + 'int cloud georeferencing information","type":"object"}'
    );

    Example : (''
      + '{"crs":{"wkt":"GEOCCS[\"WGS84 Geocentric\",DATUM[\"WGS84\",SPHEROID[\"'
      + 'WGS84\",6378137.000,298.257223563,AUTHORITY[\"EPSG\",\"7030\"]],AUTHOR'
      + 'ITY[\"EPSG\",\"6326\"]],PRIMEM[\"Greenwich\",0.0000000000000000,AUTHOR'
      + 'ITY[\"EPSG\",\"8901\"]],UNIT[\"Meter\",1.00000000000000000000,AUTHORIT'
      + 'Y[\"EPSG\",\"9001\"]],AXIS[\"X\",OTHER],AXIS[\"Y\",EAST],AXIS[\"Z\",NO'
      + 'RTH],AUTHORITY[\"EPSG\",\"4978\"]]","epsg":4978,"name":"WGS84 Geocentr'
      + 'ic"},"pose":[[-0.269827776749716,-0.723017716139738,0.635954678449952,'
      + '4063882.500831],[0.962908599449764,-0.20260517250352,0.178208229833847'
      + ',1138787.607461],[0.0,0.660451759194338,0.7508684796801,4766084.550196'
      + '],[0.0,0.0,0.0,1.0]]}'
    );
  );

// Geometric scale factor applied to point coordinates
const
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.geometric_scale_factor';
    Title       : 'Geometric Scale Factor';
    Description : 'Geometric scale factor applied to point coordinates';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"minimum":0,"exclusiveMinimum":true,"$schema":"http://json-schema.org'
      + '/draft-04/schema#","description":"Geometric scale factor applied to '
      + 'point coordinates","type":"number"}'
    );

    Example : (''
      + '1.0'
    );
  );

// Parameters used for georeferencing of the point cloud
const
  RDB_RIEGL_GEOREFERENCING_PARAMETERS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.georeferencing_parameters';
    Title       : 'Georeferencing Parameters';
    Description : 'Parameters used for georeferencing of the point cloud';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"trajectory_file":{"required":["file_extension"],"prope'
      + 'rties":{"file_extension":{"description":"Trajectory file extension, '
      + 'without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC '
      + '4122)","type":"string"}},"type":"object","description":"Trajectory '
      + 'data used for georeferencing of the point cloud"},"socs_to_body_matrix'
      + '":{"items":{"items":{"description":"columns","type":"number"},"maxItem'
      + 's":4,"description":"rows","minItems":4,"type":"array"},"maxItems":4,"t'
      + 'ype":"array","minItems":4,"description":"Coordinate Transformation '
      + 'Matrix to transform from Scanner''s Own Coordinate System to Body '
      + 'Coordinate System. 4x4 matrix stored as two dimensional array, row '
      + 'major order."},"trajectory_offsets":{"properties":{"offset_yaw":{"type'
      + '":"number","description":"[deg]"},"offset_height":{"type":"number","de'
      + 'scription":"[m]"},"offset_pitch":{"type":"number","description":"[deg]'
      + '"},"offset_north":{"type":"number","description":"[m]"},"offset_time":'
      + '{"type":"number","description":"[s]"},"version":{"type":"integer","des'
      + 'cription":"Meaning of offset values and how to apply them; version 0: '
      + 'Rz(yaw+offset_yaw)*Ry(pitch+offset_pitch)*Rx(roll+offset_roll), '
      + 'version 1: Rz(yaw)*Ry(pitch)*Rx(roll) * Rz(offset_yaw)*Ry(offset_pitch'
      + ')*Rx(offset_roll)"},"offset_east":{"type":"number","description":"[m]"'
      + '},"offset_roll":{"type":"number","description":"[deg]"}},"type":"objec'
      + 't","description":"Correction offsets applied to the trajectory data"},'
      + '"socs_to_rocs_matrix":{"items":{"items":{"description":"columns","type'
      + '":"number"},"maxItems":4,"description":"rows","minItems":4,"type":"arr'
      + 'ay"},"maxItems":4,"type":"array","minItems":4,"description":"Coordinat'
      + 'e Transformation Matrix to transform from Scanner''s Own Coordinate '
      + 'System to Record Coordinate System. 4x4 matrix stored as two '
      + 'dimensional array, row major order."},"body_coordinate_system_type":{"'
      + 'type":"string","enum":["NED","ENU"],"description":"BODY coordinate '
      + 'frame (NED: North-East-Down, ENU: East-North-Up), default: '
      + 'NED"}},"title":"Georeferencing Parameters","$schema":"http://json-sche'
      + 'ma.org/draft-04/schema#","description":"Parameters used for '
      + 'georeferencing of the point cloud","type":"object"}'
    );

    Example : (''
      + '{"trajectory_file":{"file_extension":"pofx","file_uuid":"93a03615-66c0'
      + '-4bea-8ff8-c577378fe660"},"socs_to_body_matrix":[[-0.269827776749716,-'
      + '0.723017716139738,0.635954678449952,0.0],[0.962908599449764,-0.2026051'
      + '7250352,0.178208229833847,0.0],[0.0,0.660451759194338,0.7508684796801,'
      + '0.0],[0.0,0.0,0.0,1.0]],"trajectory_offsets":{"offset_yaw":-0.45,"offs'
      + 'et_height":-0.2,"offset_pitch":0.01,"offset_north":0.07,"offset_time":'
      + '18.007,"version":0,"offset_east":0.15,"offset_roll":0.03},"body_coordi'
      + 'nate_system_type":"NED"}'
    );
  );

// Details about GNSS processing
const
  RDB_RIEGL_GNSS_PROCESSING_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.gnss_processing_info';
    Title       : 'GNSS Processing Information';
    Description : 'Details about GNSS processing';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"base_stations":{"items":{"required":["latitude","longi'
      + 'tude","altitude"],"properties":{"altitude":{"description":"Altitude '
      + '[m]","type":"number"},"latitude":{"description":"Latitude '
      + '[deg]","type":"number"},"longitude":{"description":"Longitude '
      + '[deg]","type":"number"}},"description":"GNSS base station '
      + 'details","type":"object"},"description":"List of GNSS base '
      + 'stations","type":"array"}},"title":"GNSS Processing Information","$sch'
      + 'ema":"http://json-schema.org/draft-04/schema#","description":"Details '
      + 'about GNSS processing","type":"object"}'
    );

    Example : (''
      + '{"base_stations":[{"altitude":379.8004150390625,"latitude":48.65839116'
      + '2,"longitude":15.663389822}]}'
    );
  );

// Map of item names
const
  RDB_RIEGL_ITEM_NAMES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.item_names';
    Title       : 'Item Names';
    Description : 'Map of item names';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"patternProperties":{"^-?[0-9]+$":{"description":"One field per item,'
      + ' field name is item id, field value is item '
      + 'name","type":"string"}},"title":"Item Names","additionalProperties":fa'
      + 'lse,"$schema":"http://json-schema.org/draft-04/schema#","description":'
      + '"Map of item names","type":"object"}'
    );

    Example : (''
      + '{"-1":"Name of item with id -1","2":"Name of item with id 2","1":"Name'
      + ' of item with id 1","47":"Name of item with id 47"}'
    );
  );

// Lag angle depended range and reflectance correction
const
  RDB_RIEGL_LAG_ANGLE_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.lag_angle_correction';
    Title       : 'Lag Angle Correction';
    Description : 'Lag angle depended range and reflectance correction';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Lag Angle Correction","required":["ranges_m","range_correcti'
      + 'on_m","reflectance_correction_db"],"$schema":"http://json-schema.org/d'
      + 'raft-04/schema#","properties":{"reflectance_correction_db":{"items":{"'
      + 'type":"number"},"type":"array","description":"Reflectance correction '
      + '[db] as a function of the range and for applied scanrate"},"range_corr'
      + 'ection_m":{"items":{"type":"number"},"type":"array","description":"Ran'
      + 'ge correction [m] as a function of the range and for applied scanrate,'
      + ' which defines the lag angle"},"ranges_m":{"items":{"type":"number"},"'
      + 'type":"array","description":"Range [m]"}},"description":"Lag angle '
      + 'depended range and reflectance correction","type":"object"}'
    );

    Example : (''
      + '{"reflectance_correction_db":[0.0,0.0,0.16,0.46,1.18,2.97,2.97],"range'
      + '_correction_m":[0.0,0.0,0.0004,0.0023,0.0039,0.0053,0.0053],"ranges_m"'
      + ':[0.0,4.795,31.964,63.928,95.892,127.856,10000.0]}'
    );
  );

// License keys for software features
const
  RDB_RIEGL_LICENSES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.licenses';
    Title       : 'Software License Keys';
    Description : 'License keys for software features';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"patternProperties":{"^.*$":{"items":{"description":"License key '
      + '(example: ''46AE032A - 39882AC4 - 9EC0A184 - '
      + '6F163D73'')","type":"string"},"description":"Each field of the object '
      + 'represents a feature and holds a list of license keys, where the field'
      + ' name is the feature '
      + 'name.","minItems":1,"type":"array"}},"title":"Software License Keys","'
      + 'additionalProperties":false,"$schema":"http://json-schema.org/draft-04'
      + '/schema#","description":"License keys for software '
      + 'features","type":"object"}'
    );

    Example : (''
      + '{"Full Waveform Analysis Topography with GPU support":["8AB44126 - '
      + '23B92250 - 16E2689F - 34EF7E7B"],"Georeferencing":["46AE032A - '
      + '39882AC4 - 9EC0A184 - 6F163D73"],"Full Waveform Analysis '
      + 'Topography":["0FD5FF07 - 011A1255 - 9F76CACA - 8D2ED557"],"MTA '
      + 'resolution":["468E020A - 39A922E4 - B681A184 - 673E3D72"]}'
    );
  );

// Parameters for MTA processing
const
  RDB_RIEGL_MTA_SETTINGS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.mta_settings';
    Title       : 'MTA Settings';
    Description : 'Parameters for MTA processing';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"zone_width":{"minimum":0,"type":"number","description"'
      + ':"Width of a MTA zone in meter."},"zone_count":{"minimum":0,"type":"in'
      + 'teger","maximum":255,"description":"Maximum number of MTA zones."},"mo'
      + 'dulation_depth":{"minimum":0,"type":"number","description":"Depth of '
      + 'pulse position modulation in meter."}},"title":"MTA Settings","$schema'
      + '":"http://json-schema.org/draft-04/schema#","description":"Parameters '
      + 'for MTA processing","type":"object"}'
    );

    Example : (''
      + '{"zone_width":149.896225,"zone_count":23,"modulation_depth":9.368514}'
    );
  );

// Lookup table for range correction based on raw range
const
  RDB_RIEGL_NEAR_RANGE_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.near_range_correction';
    Title       : 'Near Range Correction Table';
    Description : 'Lookup table for range correction based on raw range';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Near Range Correction Table","required":["delta","content"],'
      + '"$schema":"http://json-schema.org/draft-04/schema#","properties":{"del'
      + 'ta":{"description":"Delta between table entries [m], first entry is at'
      + ' range = 0 m","type":"number"},"content":{"items":{"type":"number"},"m'
      + 'axItems":2000,"description":"Correction value [m] to be added to the '
      + 'raw range","minItems":1,"type":"array"}},"description":"Lookup table '
      + 'for range correction based on raw range","type":"object"}'
    );

    Example : (''
      + '{"delta":0.512,"content":[0.0]}'
    );
  );

// Standard deviation for range and amplitude as a function of amplitude
const
  RDB_RIEGL_NOISE_ESTIMATES : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.noise_estimates';
    Title       : 'Noise Estimates';
    Description : 'Standard deviation for range and amplitude as a function of amplitude';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Noise Estimates","required":["amplitude","range_sigma","ampl'
      + 'itude_sigma"],"$schema":"http://json-schema.org/draft-04/schema#","pro'
      + 'perties":{"amplitude_sigma":{"items":{"type":"number"},"type":"array",'
      + '"description":"Sigma amplitude [dB]"},"amplitude":{"items":{"type":"nu'
      + 'mber"},"type":"array","description":"Amplitude [dB]"},"range_sigma":{"'
      + 'items":{"type":"number"},"type":"array","description":"Sigma range '
      + '[m]"}},"description":"Standard deviation for range and amplitude as a '
      + 'function of amplitude","type":"object"}'
    );

    Example : (''
      + '{"amplitude_sigma":[0.988,0.988,0.875,0.774,0.686,0.608,0.54,0.482,0.4'
      + '32,0.39,0.354],"amplitude":[0.0,1.0,2.0,3.0,4.0,5.0,6.0,7.0,8.0,9.0,10'
      + '.0],"range_sigma":[0.065,0.056,0.046,0.038,0.032,0.027,0.024,0.021,0.0'
      + '18,0.016,0.014]}'
    );
  );

// Notch filter parameters for window glass echoes
const
  RDB_RIEGL_NOTCH_FILTER : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.notch_filter';
    Title       : 'Notch Filter';
    Description : 'Notch filter parameters for window glass echoes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Notch Filter","required":["range_minimum","range_maximum","a'
      + 'mplitude_maximum"],"$schema":"http://json-schema.org/draft-04/schema#"'
      + ',"properties":{"range_maximum":{"type":"number","description":"Maximum'
      + ' range [m]"},"amplitude_maximum":{"minimum":0,"type":"number","descrip'
      + 'tion":"Maximum amplitude '
      + '[dB]"},"range_minimum":{"type":"number","description":"Minimum range '
      + '[m]"}},"description":"Notch filter parameters for window glass '
      + 'echoes","type":"object"}'
    );

    Example : (''
      + '{"range_maximum":0.2,"amplitude_maximum":18.0,"range_minimum":-0.5}'
    );
  );

// Details about the pixels contained in the file
const
  RDB_RIEGL_PIXEL_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pixel_info';
    Title       : 'Pixel Information';
    Description : 'Details about the pixels contained in the file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Pixel Information","definitions":{"pixel_size":{"items":{"mi'
      + 'nimum":0,"type":"number","description":"Length of pixel edge '
      + '[m]."},"maxItems":2,"type":"array","minItems":2,"description":"Size of'
      + ' pixels."}},"required":["size"],"properties":{"size":{"$ref":"#/defini'
      + 'tions/pixel_size","description":"Size of pixels in file coordinate sys'
      + 'tem."},"size_llcs":{"$ref":"#/definitions/pixel_size","description":"S'
      + 'ize of pixels in a locally levelled cartesian coordinate system (xy). '
      + 'This is only used for pixels based on a map projection."}},"$schema":"'
      + 'http://json-schema.org/draft-04/schema#","description":"Details about '
      + 'the pixels contained in the file","type":"object"}'
    );

    Example : (''
      + '{"size":[0.5971642834779395,0.5971642834779395],"size_llcs":[0.5156575'
      + '252891171,0.5130835356683303]}'
    );
  );

// Details about the plane patch matching process
const
  RDB_RIEGL_PLANE_PATCH_MATCHING : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.plane_patch_matching';
    Title       : 'Plane Patch Matching';
    Description : 'Details about the plane patch matching process';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Plane Patch Matching","definitions":{"file_reference":{"requ'
      + 'ired":["file_uuid","file_path"],"properties":{"file_path":{"descriptio'
      + 'n":"Path of the plane patch file relative to the match '
      + 'file","type":"string"},"file_uuid":{"description":"Plane patch file''s '
      + 'Universally Unique Identifier (RFC '
      + '4122)","type":"string"}},"type":"object","description":"Reference to a'
      + ' plane patch file"}},"required":["plane_patch_file_one","plane_patch_f'
      + 'ile_two"],"properties":{"plane_patch_file_one":{"$ref":"#/definitions/'
      + 'file_reference","description":"Reference to the plane patch file one"}'
      + ',"plane_patch_file_two":{"$ref":"#/definitions/file_reference","descri'
      + 'ption":"Reference to the plane patch file two"}},"$schema":"http://jso'
      + 'n-schema.org/draft-04/schema#","description":"Details about the plane '
      + 'patch matching process","type":"object"}'
    );

    Example : (''
      + '{"plane_patch_file_one":{"file_path":"Record009_Line001/191025_121410_'
      + 'Scanner_1.ptch","file_uuid":"810f5d57-eccf-49ed-b07a-0cdd109b4213"},"p'
      + 'lane_patch_file_two":{"file_path":"project.ptch","file_uuid":"fa47d509'
      + '-a64e-49ce-8b14-ff3130fbefa9"}}'
    );
  );

// Statistics about plane patches found by plane patch extractor
const
  RDB_RIEGL_PLANE_PATCH_STATISTICS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.plane_patch_statistics';
    Title       : 'Plane Patch Statistics';
    Description : 'Statistics about plane patches found by plane patch extractor';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"total_horizontal_area":{"type":"number","description":'
      + '"sum of all plane patch areas projected to horizontal plane '
      + '[m\u00b2]"},"total_area":{"type":"number","description":"sum of all '
      + 'plane patch areas [m\u00b2]"}},"title":"Plane Patch Statistics","$sche'
      + 'ma":"http://json-schema.org/draft-04/schema#","description":"Statistic'
      + 's about plane patches found by plane patch extractor","type":"object"}'
    );

    Example : (''
      + '{"total_horizontal_area":13954.601,"total_area":14007.965}'
    );
  );

// Settings and classes for plane slope classification
const
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.plane_slope_class_info';
    Title       : 'Plane Slope Class Info';
    Description : 'Settings and classes for plane slope classification';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Plane Slope Class Info","definitions":{"method-1":{"required'
      + '":["plane_classification_method","maximum_inclination_angle_horizontal'
      + '"],"properties":{"maximum_inclination_angle_horizontal":{"minimum":-36'
      + '0.0,"type":"number","maximum":360.0,"description":"maximum inclination'
      + ' angle of horizontal plane patches [deg]"},"plane_classification_metho'
      + 'd":{"minimum":1,"type":"integer","maximum":1,"description":"method ID '
      + '(=1)"}},"type":"object","description":"Classification method 1"},"meth'
      + 'od-2":{"required":["plane_classification_method","sloping_plane_classe'
      + 's_minimum_angle","sloping_plane_classes_maximum_angle"],"properties":{'
      + '"sloping_plane_classes_maximum_angle":{"minimum":-360.0,"type":"number'
      + '","maximum":360.0,"description":"maximum inclination angle of sloping '
      + 'plane patches [deg]"},"plane_classification_method":{"minimum":2,"type'
      + '":"integer","maximum":2,"description":"method ID (=2)"},"sloping_plane'
      + '_classes_minimum_angle":{"minimum":-360.0,"type":"number","maximum":36'
      + '0.0,"description":"minimum inclination angle of sloping plane patches '
      + '[deg]"}},"type":"object","description":"Classification method 2"}},"re'
      + 'quired":["settings","classes"],"properties":{"settings":{"oneOf":[{"$r'
      + 'ef":"#/definitions/method-1"},{"$ref":"#/definitions/method-2"}],"type'
      + '":"object","description":"Classification settings, details see '
      + 'documentation of rdbplanes"},"classes":{"patternProperties":{"^[0-9]+$'
      + '":{"description":"one field per class, field name is class number, '
      + 'field value is class name","type":"string"}},"additionalProperties":fa'
      + 'lse,"type":"object","description":"Class definition table"}},"$schema"'
      + ':"http://json-schema.org/draft-04/schema#","description":"Settings and'
      + ' classes for plane slope classification","type":"object"}'
    );

    Example : (''
      + '{"settings":{"sloping_plane_classes_maximum_angle":70.0,"plane_classif'
      + 'ication_method":2,"sloping_plane_classes_minimum_angle":10.0},"classes'
      + '":{"6":"vertical, pointing east","8":"vertical, pointing '
      + 'north","10":"sloping, pointing down and east","4":"sloping, pointing '
      + 'up and north","14":"horizontal, pointing down","9":"vertical, pointing'
      + ' west","12":"sloping, pointing down and north","1":"horizontal, '
      + 'pointing up","13":"sloping, pointing down and west","11":"sloping, '
      + 'pointing down and south","2":"sloping, pointing up and '
      + 'east","5":"sloping, pointing up and west","3":"sloping, pointing up '
      + 'and south","7":"vertical, pointing south"}}'
    );
  );

// Grouping and sorting of point attributes for visualization purposes
const
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.point_attribute_groups';
    Title       : 'Point Attribute Groups';
    Description : 'Grouping and sorting of point attributes for visualization purposes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"patternProperties":{"^.*$":{"items":{"description":"Point attribute '
      + 'full name or name pattern (perl regular expression '
      + 'syntax)","type":"string"},"description":"Each field of the object '
      + 'represents a point attribute group and holds a list of point '
      + 'attributes, where the field name is the group '
      + 'name.","minItems":1,"type":"array"}},"title":"Point Attribute Groups",'
      + '"additionalProperties":false,"$schema":"http://json-schema.org/draft-0'
      + '4/schema#","description":"Grouping and sorting of point attributes for'
      + ' visualization purposes","type":"object"}'
    );

    Example : (''
      + '{"Primary Attributes":["riegl.reflectance","riegl.amplitude","riegl.de'
      + 'viation"],"Secondary Attributes":["riegl.mirror_facet","riegl.waveform'
      + '_available"],"Time":["riegl.timestamp"],"Other Attributes":["riegl.sel'
      + 'ected","riegl.visible"],"Coordinates/Vectors":["riegl.xyz","riegl.rang'
      + 'e","riegl.theta","riegl.phi"]}'
    );
  );

// Details about point cloud files
const
  RDB_RIEGL_POINTCLOUD_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pointcloud_info';
    Title       : 'Point Cloud Information';
    Description : 'Details about point cloud files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"comments":{"description":"Comments","type":"string"},"'
      + 'project":{"description":"Project '
      + 'name","type":"string"},"field_of_application":{"description":"Field of'
      + ' application","enum":["unknown","SLS","TLS","KLS","MLS","ULS","ALS","B'
      + 'LS","ILS"],"type":"string"}},"title":"Point Cloud Information","$schem'
      + 'a":"http://json-schema.org/draft-04/schema#","description":"Details '
      + 'about point cloud files","type":"object"}'
    );

    Example : (''
      + '{"comments":"Line 3","project":"Campaign '
      + '4","field_of_application":"ALS"}'
    );
  );

// Estimated position and orientation information
const
  RDB_RIEGL_POSE_ESTIMATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pose_estimation';
    Title       : 'Pose Estimation';
    Description : 'Estimated position and orientation information';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Pose Estimation","required":["orientation"],"$schema":"http:'
      + '//json-schema.org/draft-04/schema#","properties":{"orientation":{"requ'
      + 'ired":["roll","pitch","yaw","roll_accuracy","pitch_accuracy","yaw_accu'
      + 'racy"],"properties":{"yaw_accuracy":{"minimum":0,"exclusiveMinimum":tr'
      + 'ue,"description":"Yaw angle accuracy '
      + '[deg]","type":"number"},"pitch":{"minimum":-360,"description":"Pitch '
      + 'angle about scanner Y-axis [deg]","maximum":360,"type":"number"},"yaw"'
      + ':{"minimum":-360,"description":"Yaw angle about scanner Z-axis [deg]",'
      + '"maximum":360,"type":"number"},"roll":{"minimum":-360,"description":"R'
      + 'oll angle about scanner X-axis [deg]","maximum":360,"type":"number"},"'
      + 'pitch_accuracy":{"minimum":0,"exclusiveMinimum":true,"description":"Pi'
      + 'tch angle accuracy [deg]","type":"number"},"roll_accuracy":{"minimum":'
      + '0,"exclusiveMinimum":true,"description":"Roll angle accuracy '
      + '[deg]","type":"number"}},"description":"Orientation values and '
      + 'orientation accuracies, measured with IMU or inclination sensors.","ty'
      + 'pe":"object"},"barometric_height_amsl":{"description":"Altitude '
      + 'determined based on the atmospheric pressure according to the standard'
      + ' atmosphere laws [m].","type":"number"},"position":{"required":["coord'
      + 'inate_1","coordinate_2","coordinate_3","horizontal_accuracy","vertical'
      + '_accuracy","crs"],"properties":{"crs":{"required":["epsg"],"properties'
      + '":{"wkt":{"description":"\"Well-Known Text\" string, OGC WKT dialect '
      + '(see http://www.opengeospatial.org/standards/wkt-crs)","type":"string"'
      + '},"epsg":{"minimum":0,"description":"EPSG '
      + 'code","type":"integer"}},"type":"object","description":"Global '
      + 'Coordinate Reference System"},"vertical_accuracy":{"minimum":0,"exclus'
      + 'iveMinimum":true,"description":"Vertical accuracy '
      + '[m]","type":"number"},"coordinate_2":{"description":"Coordinate 2 as '
      + 'defined by axis 2 of the specified CRS (e.g., Y, '
      + 'Longitude)","type":"number"},"coordinate_1":{"description":"Coordinate'
      + ' 1 as defined by axis 1 of the specified CRS (e.g., X, '
      + 'Latitude)","type":"number"},"coordinate_3":{"description":"Coordinate '
      + '3 as defined by axis 3 of the specified CRS (e.g., Z, Altitude)","type'
      + '":"number"},"horizontal_accuracy":{"minimum":0,"exclusiveMinimum":true'
      + ',"description":"Horizontal accuracy '
      + '[m]","type":"number"}},"description":"Position coordinates and '
      + 'position accuracy values as measured by GNSS in the specified '
      + 'Coordinate Reference System '
      + '(CRS)","type":"object"}},"description":"Estimated position and '
      + 'orientation information as measured by GNSS, IMU or inclination '
      + 'sensors","type":"object"}'
    );

    Example : (''
      + '{"orientation":{"yaw_accuracy":1.0094337839368757,"pitch":1.5091530248'
      + '27064,"yaw":101.87293630292045,"roll":3.14743073066123,"pitch_accuracy'
      + '":0.009433783936875745,"roll_accuracy":0.009433783936875745},"barometr'
      + 'ic_height_amsl":386.7457796227932,"position":{"crs":{"wkt":"GEOGCS[\"W'
      + 'GS84 / Geographic\",DATUM[\"WGS84\",SPHEROID[\"WGS84\",6378137.000,298'
      + '.257223563,AUTHORITY[\"EPSG\",\"7030\"]],AUTHORITY[\"EPSG\",\"6326\"]]'
      + ',PRIMEM[\"Greenwich\",0.0000000000000000,AUTHORITY[\"EPSG\",\"8901\"]]'
      + ',UNIT[\"Degree\",0.01745329251994329547,AUTHORITY[\"EPSG\",\"9102\"]],'
      + 'AXIS[\"Latitude\",NORTH],AXIS[\"Longitude\",EAST],AUTHORITY[\"EPSG\",\'
      + '"4979\"]]","epsg":4979},"vertical_accuracy":1.3314999341964722,"coordi'
      + 'nate_2":15.645033406,"coordinate_1":48.655799473,"coordinate_3":362.71'
      + '24938964844,"horizontal_accuracy":0.810699999332428}}'
    );
  );

// Details on position and orientation sensors
const
  RDB_RIEGL_POSE_SENSORS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pose_sensors';
    Title       : 'Pose Sensors';
    Description : 'Details on position and orientation sensors';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Pose Sensors","definitions":{"vector":{"items":{"type":"numb'
      + 'er","description":"Index 0=X, 1=Y, 2=Z component"},"type":"array","min'
      + 'Items":3,"maxItems":3}},"required":["gyroscope","accelerometer","magne'
      + 'tic_field_sensor"],"properties":{"accelerometer":{"required":["unit","'
      + 'x_axis","y_axis","z_axis","offset","origin","relative_nonlinearity"],"'
      + 'properties":{"z_axis":{"$ref":"#/definitions/vector","description":"Se'
      + 'nsitive Z axis of sensor at frame angle = '
      + '0"},"x_axis":{"$ref":"#/definitions/vector","description":"Sensitive X'
      + ' axis of sensor at frame angle = 0"},"unit":{"minimum":0,"exclusiveMin'
      + 'imum":true,"type":"number","description":"Unit of raw data and '
      + 'calibration values, 1 LSB in 9.81 m/s\u00b2"},"relative_nonlinearity":'
      + '{"$ref":"#/definitions/vector","description":"Relative nonlinearity, d'
      + 'imensionless"},"y_axis":{"$ref":"#/definitions/vector","description":"'
      + 'Sensitive Y axis of sensor at frame angle = '
      + '0"},"origin":{"$ref":"#/definitions/vector","description":"Sensor '
      + 'origin in SOCS [m] at frame angle = '
      + '0"},"offset":{"$ref":"#/definitions/vector","description":"Value to be'
      + ' subtracted from raw measurement '
      + 'values"}},"type":"object","description":"Accelerometer details"},"magn'
      + 'etic_field_sensor":{"required":["unit","x_axis","y_axis","z_axis","off'
      + 'set","fixed","relative_nonlinearity"],"properties":{"z_axis":{"$ref":"'
      + '#/definitions/vector","description":"Sensitive Z axis of sensor at '
      + 'frame angle = '
      + '0"},"x_axis":{"$ref":"#/definitions/vector","description":"Sensitive X'
      + ' axis of sensor at frame angle = 0"},"unit":{"minimum":0,"exclusiveMin'
      + 'imum":true,"type":"number","description":"Unit of raw data and '
      + 'calibration values, 1 LSB in '
      + 'nT"},"fixed":{"$ref":"#/definitions/vector","description":"Distortion '
      + 'of magnetic field caused by non-rotating scanner part"},"relative_nonl'
      + 'inearity":{"$ref":"#/definitions/vector","description":"Relative '
      + 'nonlinearity, dimensionless"},"y_axis":{"$ref":"#/definitions/vector",'
      + '"description":"Sensitive Y axis of sensor at frame angle = '
      + '0"},"offset":{"$ref":"#/definitions/vector","description":"Value to be'
      + ' subtracted from raw measurement '
      + 'values"}},"type":"object","description":"Magnetic Field Sensor details'
      + '"},"gyroscope":{"required":["unit","x_axis","y_axis","z_axis","offset"'
      + ',"origin","relative_nonlinearity"],"properties":{"z_axis":{"$ref":"#/d'
      + 'efinitions/vector","description":"Sensitive Z axis of sensor at frame '
      + 'angle = '
      + '0"},"x_axis":{"$ref":"#/definitions/vector","description":"Sensitive X'
      + ' axis of sensor at frame angle = 0"},"unit":{"minimum":0,"exclusiveMin'
      + 'imum":true,"type":"number","description":"Unit of raw data and '
      + 'calibration values, 1 LSB in rad/s"},"relative_nonlinearity":{"$ref":"'
      + '#/definitions/vector","description":"Relative nonlinearity, dimensionl'
      + 'ess"},"y_axis":{"$ref":"#/definitions/vector","description":"Sensitive'
      + ' Y axis of sensor at frame angle = '
      + '0"},"origin":{"$ref":"#/definitions/vector","description":"Sensor '
      + 'origin in SOCS [m] at frame angle = '
      + '0"},"offset":{"$ref":"#/definitions/vector","description":"Value to be'
      + ' subtracted from raw measurement '
      + 'values"}},"type":"object","description":"Gyroscope details"}},"$schema'
      + '":"http://json-schema.org/draft-04/schema#","description":"Details on '
      + 'position and orientation sensors","type":"object"}'
    );

    Example : (''
      + '{"accelerometer":{"z_axis":[1.639882206916809,15166.744140625,-116.997'
      + '42889404297],"x_axis":[-15008.123046875,56.956390380859375,-60.5175666'
      + '809082],"unit":6.666666740784422e-05,"relative_nonlinearity":[0.0,0.0,'
      + '0.0],"y_axis":[-7.027288913726807,-44.12333679199219,14952.3701171875]'
      + ',"origin":[0.026900000870227814,-0.03999999910593033,-0.08950000256299'
      + '973],"offset":[-733.3636474609375,58.969032287597656,1060.255004882812'
      + '5]},"magnetic_field_sensor":{"z_axis":[0.00041987866279669106,7.876977'
      + '906562388e-05,0.011407104320824146],"x_axis":[-0.011162743903696537,-2'
      + '.315962774446234e-05,0.00016818844596855342],"unit":91.74311828613281,'
      + '"fixed":[-1576.010498046875,1596.081787109375,0.0],"relative_nonlinear'
      + 'ity":[0.0,0.0,0.0],"y_axis":[0.00027888521435670555,-0.011427424848079'
      + '681,-5.204829722060822e-05],"offset":[-23812.052734375,5606.5766601562'
      + '5,2493.28125]},"gyroscope":{"z_axis":[0.555869996547699,119.2213516235'
      + '3516,0.467585027217865],"x_axis":[-121.195556640625,0.8219714164733887'
      + ',0.2313031703233719],"unit":0.00014544410805683583,"relative_nonlinear'
      + 'ity":[2.888176311444113e-07,1.06274164579645e-07,-1.7186295080634935e-'
      + '39],"y_axis":[-0.440765917301178,-0.7897399663925171,119.5894775390625'
      + '],"origin":[0.026900000870227814,-0.03999999910593033,-0.0895000025629'
      + '9973],"offset":[-50.92609786987305,146.15643310546875,62.4327278137207'
      + ']}}'
    );
  );

// Laser pulse position modulation used for MTA resolution
const
  RDB_RIEGL_PULSE_POSITION_MODULATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.pulse_position_modulation';
    Title       : 'Pulse Position Modulation';
    Description : 'Laser pulse position modulation used for MTA resolution';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Pulse Position Modulation","required":["length","num_mod_amp'
      + 'l","pulse_interval"],"$schema":"http://json-schema.org/draft-04/schema'
      + '#","properties":{"num_mod_ampl":{"minimum":0,"type":"integer","maximum'
      + '":255,"description":"Number of different modulation amplitudes (2: '
      + 'binary modulation)"},"pulse_interval":{"items":{"minimum":0,"type":"nu'
      + 'mber"},"type":"array","description":"Explicit table of the pulse '
      + 'position modulation used for MTA resolution. Table gives times between'
      + ' successive laser pulses in seconds."},"phase_step":{"minimum":0,"type'
      + '":"integer","maximum":255,"description":"Step width in phase of '
      + 'modulation code from line to line"},"length":{"minimum":0,"type":"inte'
      + 'ger","maximum":255,"description":"Length of code"},"code_phase_mode":{'
      + '"minimum":0,"type":"integer","maximum":255,"description":"0: no '
      + 'synchronization, 1: toggle between 2 phases, 2: increment with '
      + 'phase_increment"}},"description":"Laser pulse position modulation used'
      + ' for MTA resolution","type":"object"}'
    );

    Example : (''
      + '{"num_mod_ampl":2,"pulse_interval":[2.759375e-06,2.759375e-06,2.759375'
      + 'e-06,2.759375e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.821875e-06,'
      + '2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.759375e-06,2.821'
      + '875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.821875e-06,2.759375e-'
      + '06,2.759375e-06,2.759375e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.'
      + '821875e-06,2.821875e-06,2.821875e-06,2.759375e-06,2.821875e-06,2.75937'
      + '5e-06,2.821875e-06],"phase_step":5,"length":31,"code_phase_mode":2}'
    );
  );

// Statistics about target distance wrt. SOCS origin
const
  RDB_RIEGL_RANGE_STATISTICS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.range_statistics';
    Title       : 'Range Statistics';
    Description : 'Statistics about target distance wrt. SOCS origin';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Range Statistics","required":["minimum","average","maximum",'
      + '"std_dev"],"$schema":"http://json-schema.org/draft-04/schema#","proper'
      + 'ties":{"average":{"type":"number","description":"Average '
      + 'value"},"minimum":{"type":"number","description":"Minimum '
      + 'value"},"maximum":{"type":"number","description":"Maximum '
      + 'value"},"std_dev":{"type":"number","description":"Standard '
      + 'deviation"}},"description":"Statistics about target distance wrt. SOCS'
      + ' origin","type":"object"}'
    );

    Example : (''
      + '{"average":15.49738,"minimum":0.919,"maximum":574.35,"std_dev":24.349}'
    );
  );

// Receiver Internals
const
  RDB_RIEGL_RECEIVER_INTERNALS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.receiver_internals';
    Title       : 'Receiver Internals';
    Description : 'Receiver Internals';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Receiver Internals","definitions":{"fp_table":{"properties":'
      + '{"nx":{"min":1,"max":2048,"type":"integer","description":"number of x '
      + 'entries"},"ny":{"min":1,"max":2048,"type":"integer","description":"num'
      + 'ber of y entries"},"ch":{"min":0,"max":255,"type":"integer","descripti'
      + 'on":"channel number"},"tv":{"maxItems":2048,"type":"array","minItems":'
      + '1,"items":{"oneOf":[{"$ref":"#/definitions/fp_table_row"},{"type":"num'
      + 'ber"}]}},"tc":{"min":0,"max":255,"type":"integer","description":"table'
      + ' data type code"}},"type":"object","desription":"scanner internal data'
      + '","required":["ch","tc","nx","ny","tv"]},"fp":{"required":["s","w"],"p'
      + 'roperties":{"s":{"items":{"items":{"type":"number"},"type":"array","mi'
      + 'nItems":1,"maxItems":4096},"type":"array","minItems":1,"maxItems":256}'
      + ',"w":{"items":{"items":{"type":"number"},"type":"array","minItems":5,"'
      + 'maxItems":5},"type":"array","minItems":1,"maxItems":256}},"type":"obje'
      + 'ct","description":"Fingerprint values"},"fp_table_row":{"maxItems":204'
      + '8,"type":"array","minItems":1,"items":{"type":"number"}}},"properties"'
      + ':{"mw":{"minimum":0,"description":"Maximum weight","type":"number","ma'
      + 'ximum":4095,"exclusiveMinimum":true},"nt":{"minimum":0,"type":"integer'
      + '","maximum":255,"description":"Number of '
      + 'traces"},"a":{"items":{"type":"number"},"description":"Amplitude [dB]"'
      + ',"type":"array","minItems":1,"maxItems":256},"sr":{"minimum":0,"descri'
      + 'ption":"Sample rate [Hz]","type":"number","exclusiveMinimum":true},"t"'
      + ':{"patternProperties":{"^[0-9]+$":{"$ref":"#/definitions/fp","descript'
      + 'ion":"one field per channel, field name is channel index"}},"additiona'
      + 'lProperties":false,"type":"object"},"tbl":{"items":{"$ref":"#/definiti'
      + 'ons/fp_table"},"type":"array","minItems":1,"description":"various '
      + 'internal data"},"ns":{"minimum":0,"type":"integer","maximum":4095,"des'
      + 'cription":"Number of '
      + 'samples"},"ex":{"type":"object","description":"DEPRECATED, use '
      + '''riegl.exponential_decomposition'' instead"},"si":{"minimum":0,"type":"'
      + 'number","maximum":255,"description":"Start index (hw_start)"}},"$schem'
      + 'a":"http://json-schema.org/draft-04/schema#","description":"Receiver '
      + 'Internals","type":"object"}'
    );

    Example : (''
      + '{"mw":31,"nt":128,"a":[-1.55],"sr":7959997000.0,"t":{"1":{"s":[[1.23,4'
      + '.56],[7.89,0.12]],"w":[[78,86,126,134,31],[78,86,126,134,31]]},"0":{"s'
      + '":[[1.23,4.56],[7.89,0.12]],"w":[[78,86,126,134,31],[78,86,126,134,31]'
      + ']}},"tbl":[{"nx":5,"ny":2,"ch":0,"tv":[[1,2,3,4,5],[1.1,2.2,3.3,4.4,5.'
      + '5]],"tc":1}],"ns":400,"si":48}'
    );
  );

// Lookup table for reflectance calculation based on amplitude and range
const
  RDB_RIEGL_REFLECTANCE_CALCULATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.reflectance_calculation';
    Title       : 'Reflectance Calculation Table';
    Description : 'Lookup table for reflectance calculation based on amplitude and range';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Reflectance Calculation Table","required":["delta","content"'
      + '],"$schema":"http://json-schema.org/draft-04/schema#","properties":{"d'
      + 'elta":{"description":"Delta between table entries [m], first entry is '
      + 'at range = 0 m","type":"number"},"content":{"items":{"type":"number"},'
      + '"maxItems":2000,"description":"Correction value [dB] to be added to '
      + 'the amplitude","minItems":1,"type":"array"}},"description":"Lookup '
      + 'table for reflectance calculation based on amplitude and '
      + 'range","type":"object"}'
    );

    Example : (''
      + '{"delta":0.150918,"content":[-33.01]}'
    );
  );

// Range-dependent and scan-angle-dependent correction of reflectance reading
const
  RDB_RIEGL_REFLECTANCE_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.reflectance_correction';
    Title       : 'Near-Range Reflectance Correction';
    Description : 'Range-dependent and scan-angle-dependent correction of reflectance reading';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Near-range reflectance correction","required":["ranges_m","l'
      + 'ine_angles_deg","reflectance_correction_db"],"$schema":"http://json-sc'
      + 'hema.org/draft-04/schema#","properties":{"line_angles_deg":{"items":{"'
      + 'type":"number"},"type":"array","description":"Angle [deg]"},"reflectan'
      + 'ce_correction_db":{"items":{"items":{"description":"columns (each '
      + 'value corresponds to an angle)","type":"number"},"description":"rows '
      + '(each array corresponds to a '
      + 'range)","type":"array"},"type":"array","description":"Near range '
      + 'reflectance correction in dB as a function of range over angle"},"rang'
      + 'es_m":{"items":{"type":"number"},"type":"array","description":"Range '
      + '[m]"}},"description":"Range-dependent and scan-angle-dependent '
      + 'correction of reflectance reading","type":"object"}'
    );

    Example : (''
      + '{"line_angles_deg":[0.0,0.5,1.0,1.5,1.0,2.0,2.5,3.0,3.5,4.0],"reflecta'
      + 'nce_correction_db":[[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0'
      + '.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1'
      + ',0.05,0.01],[0.8,0.7,0.6,0.5,0.4,0.3,0.2,0.1,0.05,0.01]],"ranges_m":[0'
      + '.0,1.0,2.0,3.0]}'
    );
  );

// Details about RiLOC processing
const
  RDB_RIEGL_RILOC_PROCESSING_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.riloc_processing_info';
    Title       : 'RiLOC Processing Information';
    Description : 'Details about RiLOC processing';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"RiLOC Processing Information","required":["trajectory","opti'
      + 'mizer","georeferencing"],"$schema":"http://json-schema.org/draft-04/sc'
      + 'hema#","properties":{"optimizer":{"properties":{},"description":"Optim'
      + 'ization result information and statistics.","type":"object"},"planes":'
      + '{"properties":{},"description":"Information on plane correspondences '
      + 'and used extraction/matching templates.","type":"object"},"georeferenc'
      + 'ing":{"properties":{},"description":"Internal georeferencing parameter'
      + 's.","type":"object"},"trajectory":{"properties":{},"description":"Traj'
      + 'ectory information and '
      + 'statistics.","type":"object"}},"description":"Details about RiLOC '
      + 'processing","type":"object"}'
    );

    Example : (''
      + '{"optimizer":{},"planes":{},"georeferencing":{},"trajectory":{}}'
    );
  );

// Scan pattern description
const
  RDB_RIEGL_SCAN_PATTERN : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.scan_pattern';
    Title       : 'Scan Pattern';
    Description : 'Scan pattern description';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"definitions":{"program":{"required":["name"],"properties":{"laser_pr'
      + 'r":{"minimum":0,"exclusiveMinimum":false,"description":"Laser Pulse '
      + 'Repetition Rate [Hz]","type":"number"},"name":{"description":"Name of '
      + 'measurement program","type":"string"}},"description":"Measurement '
      + 'program","type":"object"}},"title":"Scan Pattern","properties":{"line"'
      + ':{"required":["start","stop","increment"],"properties":{"program":{"$r'
      + 'ef":"#/definitions/program"},"increment":{"minimum":0.0,"description":'
      + '"Increment of angle in SOCS [deg]","maximum":90.0,"type":"number"},"st'
      + 'art":{"minimum":0.0,"description":"Start angle in SOCS [deg]","maximum'
      + '":360.0,"type":"number"},"stop":{"minimum":0.0,"description":"Stop '
      + 'angle in SOCS '
      + '[deg]","maximum":720.0,"type":"number"}},"description":"Line Scan Patt'
      + 'ern","type":"object"},"rectangular":{"required":["phi_start","phi_stop'
      + '","phi_increment","theta_start","theta_stop","theta_increment"],"prope'
      + 'rties":{"program":{"$ref":"#/definitions/program"},"theta_increment":{'
      + '"minimum":0.0,"description":"Increment of theta angle in SOCS [deg]","'
      + 'maximum":90.0,"type":"number"},"theta_start":{"minimum":0.0,"descripti'
      + 'on":"Start theta angle in SOCS [deg]","maximum":180.0,"type":"number"}'
      + ',"phi_stop":{"minimum":0.0,"description":"Stop phi angle in SOCS [deg]'
      + '","maximum":720.0,"type":"number"},"phi_increment":{"minimum":0.0,"des'
      + 'cription":"Increment of phi angle in SOCS [deg]","maximum":90.0,"type"'
      + ':"number"},"phi_start":{"minimum":0.0,"description":"Start phi angle '
      + 'in SOCS [deg]","maximum":360.0,"type":"number"},"theta_stop":{"minimum'
      + '":0.0,"description":"Stop theta angle in SOCS '
      + '[deg]","maximum":180.0,"type":"number"}},"description":"Rectangular '
      + 'Field Of View Scan Pattern","type":"object"},"segments":{"required":["'
      + 'list"],"properties":{"program":{"$ref":"#/definitions/program"},"list"'
      + ':{"items":{"required":["start","stop","increment"],"properties":{"incr'
      + 'ement":{"minimum":0.0,"description":"Increment of angle in SOCS [deg]"'
      + ',"maximum":90.0,"type":"number"},"start":{"minimum":0.0,"description":'
      + '"Start angle in SOCS [deg]","maximum":360.0,"type":"number"},"stop":{"'
      + 'minimum":0.0,"description":"Stop angle in SOCS '
      + '[deg]","maximum":720.0,"type":"number"}},"description":"Line Scan '
      + 'Segment","type":"object"},"type":"array"}},"description":"Segmented '
      + 'Line Scan Pattern","type":"object"}},"$schema":"http://json-schema.org'
      + '/draft-04/schema#","description":"Scan pattern description"}'
    );

    Example : (''
      + '{"rectangular":{"program":{"laser_prr":100000.0,"name":"High Speed"},"'
      + 'theta_increment":0.04,"theta_start":30.0,"phi_stop":270.0,"phi_increme'
      + 'nt":0.04,"phi_start":45.0,"theta_stop":130.0}}'
    );
  );

// Details about laser shot files
const
  RDB_RIEGL_SHOT_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.shot_info';
    Title       : 'Shot Information';
    Description : 'Details about laser shot files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"shot_file":{"properties":{"file_extension":{"descripti'
      + 'on":"Shot file extension, without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC 4122)","type":"string"}},"type":"object","requi'
      + 'red":["file_extension"]}},"title":"Shot Information","$schema":"http:/'
      + '/json-schema.org/draft-04/schema#","description":"Details about laser '
      + 'shot files","type":"object"}'
    );

    Example : (''
      + '{"shot_file":{"file_extension":"sodx","file_uuid":"26a00815-67c0-4bff-'
      + '8fe8-c577378fe663"}}'
    );
  );

// Point filters applied in addition to the application-defined filters
const
  RDB_RIEGL_STORED_FILTERS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.stored_filters';
    Title       : 'Stored Filters';
    Description : 'Point filters applied in addition to the application-defined filters';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Stored filters","required":["activated","filters"],"$schema"'
      + ':"http://json-schema.org/draft-04/schema#","properties":{"filters":{"i'
      + 'tems":{"required":["activated","title","description","filter"],"proper'
      + 'ties":{"activated":{"description":"Apply (''true'') or ignore (''false'') '
      + 'this filter","type":"boolean"},"title":{"description":"A short filter '
      + 'title (e.g. for display in a selection '
      + 'list)","type":"string"},"filter":{"description":"The RDB filter string'
      + ' to apply (e.g. when reading points or index), details see '
      + 'documentation of function '
      + 'select()","type":"string"},"description":{"description":"A brief '
      + 'description of the filter (e.g. for display in a '
      + 'tooltip)","type":"string"}},"description":"Point filter '
      + 'definition","type":"object"},"description":"List of point '
      + 'filters","type":"array"},"activated":{"description":"Apply (''true'') or'
      + ' ignore (''false'') all filters","type":"boolean"}},"description":"Point'
      + ' filters applied in addition to the application-defined '
      + 'filters","type":"object"}'
    );

    Example : (''
      + '{"filters":[{"activated":false,"title":"Ignore uncertain points from '
      + 'MTA resolution","filter":"riegl.mta_uncertain_point == '
      + '0","description":"Ignore points with uncertain MTA zone '
      + 'assignment"},{"activated":false,"title":"Ignore points below dynamic '
      + 'trigger level","filter":"riegl.dyntrig_uncertain_point == '
      + '0","description":"Ignore points with an echo signal amplitude below '
      + 'the dynamic trigger"},{"activated":false,"title":"Ignore points '
      + 'outside outer AABB","filter":"riegl.point_outside_aabb_bocs == '
      + '0","description":"Ignore points that are outside the outer AABB in '
      + 'BOCS"},{"activated":false,"title":"Ignore points inside inner '
      + 'AABB","filter":"riegl.point_inside_aabb_bocs == '
      + '0","description":"Ignore points that are inside the inner AABB in '
      + 'BOCS"}],"activated":true}'
    );
  );

// Details of major system components like lidar sensors, cameras and navigation devices
const
  RDB_RIEGL_SYSTEM_DESCRIPTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.system_description';
    Title       : 'System Description';
    Description : 'Details of major system components like lidar sensors, cameras and navigation devices';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"System Description","required":["$class","version","author",'
      + '"timestamp","system"],"$schema":"http://json-schema.org/draft-04/schem'
      + 'a#","properties":{"$class":{"description":"Object class name","enum":['
      + '"Document"],"type":"string"},"version":{"description":"Document format'
      + ' version","type":"string"},"timestamp":{"description":"Date and time '
      + 'of creation (e.g. 2019-06-19T13:32:10+02:00)","type":"string"},"system'
      + '":{"description":"The actual system description, details see '
      + 'documentation of RIEGL System '
      + 'Description","type":"object"},"author":{"description":"Name of the '
      + 'author that created the '
      + 'document","type":"string"}},"description":"Details of major system '
      + 'components like lidar sensors, cameras and navigation '
      + 'devices","type":"object"}'
    );

    Example : (''
      + '{"$class":"Document","version":"1.2.1","timestamp":"2022-09-30T11:56:2'
      + '6+00:00","system":{"details":"see documentation of RIEGL System '
      + 'Description"},"author":"RIEGL LMS GmbH, Austria"}'
    );
  );

// Conversion of background radiation raw values to temperatures in °C
const
  RDB_RIEGL_TEMPERATURE_CALCULATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.temperature_calculation';
    Title       : 'Temperature Calculation Table';
    Description : 'Conversion of background radiation raw values to temperatures in °C';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Temperature Calculation Table","definitions":{"conversion_ta'
      + 'ble":{"properties":{"value":{"items":{"type":"number"},"type":"array",'
      + '"description":"LSB [1]"},"temperature":{"items":{"type":"number"},"typ'
      + 'e":"array","description":"Temperature [\u00b0C]"}},"type":"object","re'
      + 'quired":["value","temperature"]}},"properties":{"InGaAs_Si_Difference"'
      + ':{"$ref":"#/definitions/conversion_table","description":"Conversion '
      + 'table for InGaAs - Si difference"},"Si":{"$ref":"#/definitions/convers'
      + 'ion_table","description":"Conversion table for Si channel"},"InGaAs":{'
      + '"$ref":"#/definitions/conversion_table","description":"Conversion '
      + 'table for InGaAs channel"}},"$schema":"http://json-schema.org/draft-04'
      + '/schema#","description":"Conversion of background radiation raw values'
      + ' to temperatures in \u00b0C","type":"object"}'
    );

    Example : (''
      + '{"InGaAs_Si_Difference":{"value":[1000.0,1100.090029602954,1200.044251'
      + '83874,1300.1342814416948,1400.0885036774805,1500.0427259132668,1600.13'
      + '27555162209,1700.0869777520065,1800.0411999877924],"temperature":[1749'
      + '.977111117893,1749.977111117893,1749.977111117893,1749.977111117893,17'
      + '49.977111117893,1749.977111117893,1744.7813348796044,1681.997131260109'
      + '2,1622.3944822534868]},"Si":{"value":[0.0,64.00097659230323,128.001953'
      + '1846065,192.0029297769097,256.0039063692129,320.00488296151616,384.005'
      + '8595538194,448.0068361461226,512.0078127384259],"temperature":[546.300'
      + '048828125,548.8164051212026,551.3143938500972,554.0144257850053,556.60'
      + '4252334815,559.2124464488079,561.8022729986177,564.4104671126105,567.0'
      + '002936624203]},"InGaAs":{"value":[0.0,64.00097659230323,128.0019531846'
      + '065,192.0029297769097,256.0039063692129,320.00488296151616,384.0058595'
      + '538194,448.0068361461226,512.0078127384259],"temperature":[307.2219672'
      + '2535614,309.1153478247277,311.1188086915047,313.10025350480055,315.213'
      + '7946389828,317.2172555057597,319.2207163725366,321.2021611858325,323.3'
      + '157023200148]}}'
    );
  );

// Base of timestamps (epoch)
const
  RDB_RIEGL_TIME_BASE : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.time_base';
    Title       : 'Time Base';
    Description : 'Base of timestamps (epoch)';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Time Base","required":["epoch","source"],"$schema":"http://j'
      + 'son-schema.org/draft-04/schema#","properties":{"epoch":{"description":'
      + '"Date and time of timestamp ''0'' as proposed by RFC 3339 (e.g. 2015-10-'
      + '27T00:00:00+01:00).","type":"string"},"system":{"description":"Time '
      + 'system (time standard)","enum":["unknown","UTC","GPS"],"type":"string"'
      + '},"source":{"description":"Timestamp source","enum":["unknown","RTC","'
      + 'GNSS"],"type":"string"}},"description":"Base of timestamps '
      + '(epoch)","type":"object"}'
    );

    Example : (''
      + '{"epoch":"2015-10-27T00:00:00+00:00","system":"UTC","source":"GNSS"}'
    );
  );

// Details about position+orientation files
const
  RDB_RIEGL_TRAJECTORY_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.trajectory_info';
    Title       : 'Trajectory Information';
    Description : 'Details about position+orientation files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Trajectory Information","required":["time_interval","navigat'
      + 'ion_frame"],"$schema":"http://json-schema.org/draft-04/schema#","prope'
      + 'rties":{"time_interval":{"required":["minimum","average","maximum","st'
      + 'd_dev"],"properties":{"average":{"description":"Average time interval '
      + '[s]","type":"number"},"minimum":{"description":"Minimum time interval '
      + '[s]","type":"number"},"maximum":{"description":"Maximum time interval '
      + '[s]","type":"number"},"std_dev":{"description":"Standard deviation of '
      + 'intervals [s]","type":"number"}},"description":"Time interval statisti'
      + 'cs","type":"object"},"navigation_frame":{"description":"Navigation '
      + 'frame (NED: North-East-Down, ENU: East-North-Up)","enum":["unknown","N'
      + 'ED","ENU"],"type":"string"},"software":{"description":"Software that '
      + 'calculated the trajectory (this may be the same or different software '
      + 'than the one that created the '
      + 'file)","type":"string"},"company":{"description":"Company '
      + 'name","type":"string"},"device":{"description":"Navigation device, '
      + 'e.g. '
      + 'name/type/serial","type":"string"},"settings":{"description":"Settings'
      + ' used to calculate the trajectory (descriptive '
      + 'text)","type":"string"},"location":{"description":"Project location, '
      + 'e.g. '
      + 'city/state/country","type":"string"},"project":{"description":"Project'
      + ' name","type":"string"},"field_of_application":{"description":"Field '
      + 'of application","enum":["unknown","SLS","TLS","KLS","MLS","ULS","ALS",'
      + '"BLS","ILS"],"type":"string"}},"description":"Details about '
      + 'position+orientation files","type":"object"}'
    );

    Example : (''
      + '{"time_interval":{"average":0.00500053,"minimum":0.00500032,"maximum":'
      + '0.005004883,"std_dev":5.51e-07},"navigation_frame":"NED","software":"N'
      + 'avigation Software XYZ","company":"RIEGL LMS","device":"IMU Model '
      + '12/1, Serial# '
      + '12345","settings":"default","location":"Horn","project":"Campaign '
      + '3","field_of_application":"MLS"}'
    );
  );

// Trigger-Timestamping-Unit/IP configuration
const
  RDB_RIEGL_TTIP_CONFIGURATION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.ttip_configuration';
    Title       : 'TTIP Configuration';
    Description : 'Trigger-Timestamping-Unit/IP configuration';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"TTIP Configuration","required":["ttip_version","num_channel"'
      + ',"ext_channel","ext_signal","ext_delay","ext_subdivider","ext_bitmask"'
      + ',"dmi_incr","dmi_min_time","dmi_max_time","dmi_dist_per_tick","channel'
      + '","output_descr","input_descr","output_usage","input_usage","out_polar'
      + 'ity","out_duration","out_interval","out_delay","out_num_pulses","in_po'
      + 'larity","in_min_duration","in_max_duration","in_max_delay"],"$schema":'
      + '"http://json-schema.org/draft-04/schema#","properties":{"in_polarity":'
      + '{"minimum":0,"description":"0 .. positive edge, 1 .. negative edge","t'
      + 'ype":"integer"},"in_min_duration":{"minimum":0,"description":"input '
      + 'signals with smaller pulse durations are ignored [0.1 msec]","type":"i'
      + 'nteger"},"ext_subdivider":{"minimum":0,"description":"reduces the '
      + 'frequency, default 1","type":"integer"},"ttip_version":{"minimum":0,"d'
      + 'escription":"following main.sub.ss.sss","type":"integer"},"in_max_dela'
      + 'y":{"minimum":0,"description":"maximum delay to output pulse before '
      + 'fake event is generated [0.1 msec], zero indicates that no fake events'
      + ' are generated","type":"integer"},"in_max_duration":{"minimum":0,"desc'
      + 'ription":"stops measurement of pulse duration of input signal [0.1 '
      + 'msec]","type":"integer"},"ext_channel":{"minimum":0,"description":"ID '
      + 'of channel used as external trigger input, 32 indicates '
      + 'none","type":"integer"},"output_usage":{"minimum":0,"description":"0 '
      + '.. no output, 1 .. output derived from internal clock, 2 .. output '
      + 'derived from dmi, 3 .. output derived from external signal, 4 .. '
      + 'output static low, 5 .. output static high","type":"integer"},"dmi_max'
      + '_time":{"minimum":0,"description":"dmi, maximum time interval between '
      + 'trigger events [0.1 msec]","type":"integer"},"ext_bitmask":{"minimum":'
      + '0,"description":"defines which of the internal pulse generators are to'
      + ' be started","type":"integer"},"out_interval":{"minimum":0,"descriptio'
      + 'n":"output pulse interval [0.1 '
      + 'msec]","type":"integer"},"dmi_dist_per_tick":{"description":"dmi, '
      + 'distance per tick, just informative '
      + '[m]","type":"number"},"out_delay":{"minimum":0,"description":"output '
      + 'pulse initial delay after start [0.1 '
      + 'msec]","type":"integer"},"input_descr":{"description":"descriptive '
      + 'string","type":"string"},"out_polarity":{"minimum":0,"description":"0 '
      + '.. positive edge, 1 .. negative edge","type":"integer"},"out_num_pulse'
      + 's":{"minimum":0,"description":"number of output pulses to be '
      + 'generated, 0 .. '
      + 'infinite","type":"integer"},"output_descr":{"description":"descriptive'
      + ' string","type":"string"},"channel":{"minimum":0,"description":"ID of '
      + 'input/output channel","type":"integer"},"ext_delay":{"minimum":0,"desc'
      + 'ription":"external trigger, time delay [0.1 msec]","type":"integer"},"'
      + 'num_channel":{"minimum":0,"description":"number of input/output channe'
      + 'ls","type":"integer"},"dmi_min_time":{"minimum":0,"description":"dmi, '
      + 'minimum time interval between trigger events [0.1 '
      + 'msec]","type":"integer"},"dmi_incr":{"description":"dmi, increment in '
      + 'ticks","type":"integer"},"input_usage":{"minimum":0,"description":"0 '
      + '.. deactivated, 1 .. just detecting and timestamping","type":"integer"'
      + '},"ext_signal":{"minimum":0,"description":"0 .. use input signal, 1 ..'
      + ' use output signal of channel","type":"integer"},"out_duration":{"mini'
      + 'mum":0,"description":"output pulse duration [0.1 '
      + 'msec]","type":"integer"}},"description":"Trigger-Timestamping-Unit/IP '
      + 'configuration","type":"object"}'
    );

    Example : (''
      + '{"in_polarity":0,"in_min_duration":0,"ext_subdivider":1,"ttip_version"'
      + ':1,"in_max_delay":9990,"in_max_duration":10000,"ext_channel":32,"outpu'
      + 't_usage":1,"dmi_max_time":0,"ext_bitmask":0,"out_interval":10000,"dmi_'
      + 'dist_per_tick":0.0,"out_delay":0,"input_descr":"Port 1 - '
      + 'Exposure","out_polarity":0,"out_num_pulses":1,"output_descr":"Port 1 -'
      + ' Trigger","channel":0,"ext_delay":0,"num_channel":9,"dmi_min_time":0,"'
      + 'dmi_incr":0,"input_usage":1,"ext_signal":0,"out_duration":300}'
    );
  );

// Details about vertex file
const
  RDB_RIEGL_VERTEX_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.vertex_info';
    Title       : 'Vertex Information';
    Description : 'Details about vertex file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"properties":{"vertex_file":{"properties":{"file_extension":{"descrip'
      + 'tion":"Vertex file extension, without the leading '
      + 'dot","type":"string"},"file_uuid":{"description":"File''s Universally '
      + 'Unique Identifier (RFC 4122)","type":"string"}},"type":"object","requi'
      + 'red":["file_extension"]}},"title":"Vertex Information","$schema":"http'
      + '://json-schema.org/draft-04/schema#","description":"Details about '
      + 'vertex file","type":"object"}'
    );

    Example : (''
      + '{"vertex_file":{"file_extension":"vtx","file_uuid":"51534d95-d71f-4f36'
      + '-ae1a-3e63a21fd1c7"}}'
    );
  );

// Details about the voxels contained in the file
const
  RDB_RIEGL_VOXEL_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.voxel_info';
    Title       : 'Voxel Information';
    Description : 'Details about the voxels contained in the file';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Voxel Information","oneOf":[{"additionalProperties":false,"p'
      + 'roperties":{"voxel_origin":{"$ref":"#/definitions/voxel_origin_enum"},'
      + '"shape_thresholds":{"$ref":"#/definitions/shape_thresholds"},"size":{"'
      + 'oneOf":[{"$ref":"#/definitions/voxel_size"},{"$ref":"#/definitions/vox'
      + 'el_size_cubic"}],"description":"Size of voxels in file coordinate syst'
      + 'em."},"voxel_type":{"$ref":"#/definitions/voxel_type"}},"required":["s'
      + 'ize","voxel_origin","voxel_type"]},{"additionalProperties":false,"prop'
      + 'erties":{"voxel_type":{"$ref":"#/definitions/voxel_type"},"voxel_origi'
      + 'n":{"oneOf":[{"$ref":"#/definitions/voxel_origin_enum"},{"$ref":"#/def'
      + 'initions/voxel_origin_point","description":"The base point of the '
      + 'voxel grid. Used together with <tt>voxel_size</tt> and '
      + '<tt>voxel_index</tt> to compute actual point coordinates."}]},"shape_t'
      + 'hresholds":{"$ref":"#/definitions/shape_thresholds"},"size":{"$ref":"#'
      + '/definitions/voxel_size","description":"Size of voxels in file '
      + 'coordinate system."},"size_llcs":{"$ref":"#/definitions/voxel_size","d'
      + 'escription":"Size of voxels in a locally levelled cartesian coordinate'
      + ' system (xyz). This is only used for voxels based on a map projection.'
      + '"},"reference_point":{"items":{"minimum":-180,"type":"number","maximum'
      + '":180},"maxItems":2,"description":"Point in WGS84 geodetic decimal '
      + 'degree (EPSG:4326) that was used to compute the projection distortion '
      + 'parameters. The coefficient order is latitude, longitude. Only voxels '
      + 'with corresponding geo_tag, voxel_size and reference_point can be '
      + 'reliably processed together. This entry is available for voxel files '
      + 'in projected CRS only.","minItems":2,"type":"array"}},"required":["ref'
      + 'erence_point","size_llcs","size","voxel_origin","voxel_type"]}],"defin'
      + 'itions":{"voxel_size":{"items":{"$ref":"#/definitions/edge_length"},"m'
      + 'axItems":3,"description":"Size of voxels.","minItems":3,"type":"array"'
      + '},"voxel_size_cubic":{"$ref":"#/definitions/edge_length","type":"numbe'
      + 'r"},"voxel_origin_point":{"items":{"type":"number"},"maxItems":3,"desc'
      + 'ription":"Origin point for all voxel indices in voxel CRS.","minItems"'
      + ':3,"type":"array"},"voxel_origin_enum":{"default":"corner","descriptio'
      + 'n":"Defines whether the voxel''s center or a corner is placed on CRS '
      + 'origin <tt>(0/0/0)</tt>.","enum":["center","corner"]},"shape_threshold'
      + 's":{"type":"object","properties":{"line":{"type":"number","minimum":1,'
      + '"description":"If the biggest eigenvalue is bigger than the median '
      + 'eigenvalue * line_threshold, the voxel is considered a line.","exclusi'
      + 'veMinimum":true},"plane":{"minimum":0,"type":"number","exclusiveMaximu'
      + 'm":true,"description":"If the smallest eigenvalue is smaller than the '
      + 'median eigenvalue * plane_threshold, the voxel is considered a plane."'
      + ',"maximum":1,"exclusiveMinimum":true}},"description":"Thresholds used '
      + 'to compute the voxel''s shape_id value.","required":["plane","line"]},"'
      + 'edge_length":{"type":"number","minimum":0,"description":"Length of '
      + 'voxel edge [m].","exclusiveMinimum":true},"voxel_type":{"default":"cen'
      + 'troid","description":"Whether a point in a voxel represents its center'
      + ' or its centroid. If type is <tt>index</tt> there is no point but only'
      + ' an integer voxel index.","enum":["center","centroid","index"]}},"$sch'
      + 'ema":"http://json-schema.org/draft-04/schema#","description":"Details '
      + 'about the voxels contained in the file","type":"object"}'
    );

    Example : (''
      + '{"voxel_type":"centroid","voxel_origin":"corner","shape_thresholds":{"'
      + 'line":6,"plane":0.16},"size":[0.5971642834779395,0.5971642834779395,0.'
      + '5143705304787237],"size_llcs":[0.5156575252891171,0.5130835356683303,0'
      + '.5143705304787237],"reference_point":[48,16]}'
    );
  );

// Settings for waveform averaging
const
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.waveform_averaging_settings';
    Title       : 'Waveform Averaging Settings';
    Description : 'Settings for waveform averaging';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Waveform Averaging Settings","required":["num_shots","mta_zo'
      + 'ne"],"$schema":"http://json-schema.org/draft-04/schema#","properties":'
      + '{"mta_zone":{"minimum":1,"type":"integer","description":"Fixed MTA '
      + 'zone for averaging."},"num_shots":{"minimum":1,"type":"integer","descr'
      + 'iption":"Number of consecutive shots to be used for averaging."},"trim'
      + '":{"minimum":0,"default":0,"type":"number","maximum":0.5,"description"'
      + ':"Percentage for robust averaging."}},"description":"Settings for '
      + 'waveform averaging","type":"object"}'
    );

    Example : (''
      + '{"mta_zone":1,"num_shots":7,"trim":0.05}'
    );
  );

// Details about waveform files
const
  RDB_RIEGL_WAVEFORM_INFO : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.waveform_info';
    Title       : 'Waveform Information';
    Description : 'Details about waveform files';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Waveform Information","required":["sample_block_file","sampl'
      + 'e_data_files"],"properties":{"range_offset_waveform_samples_m":{"descr'
      + 'iption":"Calibrated device specific range offset for waveform samples '
      + 'in meters.","type":"number"},"sample_block_file":{"properties":{"file_'
      + 'extension":{"description":"Sample block file extension, without the '
      + 'leading dot","type":"string"},"file_uuid":{"description":"File''s '
      + 'Universally Unique Identifier (RFC 4122)","type":"string"}},"type":"ob'
      + 'ject","required":["file_extension"]},"sample_data_files":{"items":{"pr'
      + 'operties":{"sample_bits":{"minimum":0,"exclusiveMaximum":false,"exclus'
      + 'iveMinimum":false,"description":"Bitwidth of samples (e.g. 10 bit, 12 '
      + 'bit)","maximum":32,"type":"integer"},"file_extension":{"description":"'
      + 'Sample data file extension, without the leading dot","type":"string"},'
      + '"channel":{"minimum":0,"exclusiveMaximum":false,"exclusiveMinimum":fal'
      + 'se,"description":"Sample block channel number (255 = invalid)","maximu'
      + 'm":255,"type":"integer"},"file_uuid":{"description":"File''s '
      + 'Universally Unique Identifier (RFC 4122)","type":"string"},"sample_int'
      + 'erval":{"minimum":0,"exclusiveMinimum":false,"description":"Sampling '
      + 'interval in seconds","type":"number"},"delta_st":{"description":"reser'
      + 'ved","type":"number"},"laser_wavelength":{"minimum":0,"exclusiveMinimu'
      + 'm":false,"description":"Laser wavelength in meters (0 = '
      + 'unknown)","type":"number"},"channel_name":{"description":"Sample block'
      + ' channel name","type":"string"}},"type":"object","required":["channel"'
      + ',"channel_name","sample_interval","sample_bits","laser_wavelength","de'
      + 'lta_st","file_extension"]},"type":"array"},"range_offset_m":{"descript'
      + 'ion":"Calibrated device specific range offset for waveform analysis by'
      + ' system response fitting in meters.","type":"number"}},"$schema":"http'
      + '://json-schema.org/draft-04/schema#","type":"object","description":"De'
      + 'tails about waveform files"}'
    );

    Example : (''
      + '{"range_offset_waveform_samples_m ":7.283,"sample_block_file":{"file_e'
      + 'xtension":"sbx","file_uuid":"93a03615-66c0-4bea-8ff8-c577378fe660"},"s'
      + 'ample_data_files":[{"delta_st":0,"channel":0,"channel_name":"high_powe'
      + 'r","sample_interval":1.00503e-09,"laser_wavelength":0,"file_extension"'
      + ':"sp0","sample_bits":12,"file_uuid":"da084413-e3e8-4655-a122-071de8490'
      + 'd8e"},{"delta_st":0,"channel":1,"channel_name":"low_power","sample_int'
      + 'erval":1.00503e-09,"laser_wavelength":0,"file_extension":"sp1","sample'
      + '_bits":12,"file_uuid":"93585b5e-5ea9-43a1-947b-e7ba3be642d2"},{"delta_'
      + 'st":0,"channel":5,"channel_name":"wwf","sample_interval":1.00503e-09,"'
      + 'laser_wavelength":0,"file_extension":"sp5","sample_bits":12,"file_uuid'
      + '":"9d2298c4-1036-464f-b5cb-1cf8e517f3a0"}],"range_offset_m":3.1415}'
    );
  );

// Scanner settings for waveform output
const
  RDB_RIEGL_WAVEFORM_SETTINGS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.waveform_settings';
    Title       : 'Waveform Settings';
    Description : 'Scanner settings for waveform output';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"items":{"properties":{"smart_enabled":{"description":"Smart waveform'
      + ' output '
      + 'enabled","type":"boolean"},"pass_rng_less":{"description":"Threshold '
      + 'for range less than '
      + '[m]","type":"number"},"pass_ampl_greater":{"description":"Threshold '
      + 'for amplitude greater than '
      + '[dB]","type":"number"},"pass_dev_greater":{"description":"Threshold '
      + 'for deviation greater than '
      + '[1]","type":"integer"},"pass_dev_less":{"description":"Threshold for '
      + 'deviation less than '
      + '[1]","type":"integer"},"pass_ampl_less":{"description":"Threshold for '
      + 'amplitude less than '
      + '[dB]","type":"number"},"sbl_name":{"description":"Name of sample '
      + 'block, e.g.: wfm, '
      + 'wwf","type":"string"},"enabled":{"description":"Waveform output enable'
      + 'd","type":"boolean"},"pass_rng_greater":{"description":"Threshold for '
      + 'range greater than '
      + '[m]","type":"number"},"channel_idx_mask":{"description":"Bit mask for '
      + 'channels which belong to sbl_name: Channel 0 = Bit0, Channel 1 = Bit1,'
      + ' ...","type":"integer"},"logic_expression":{"description":"Logic '
      + 'expression of smart waveforms filter","type":"string"}},"type":"object'
      + '","required":["sbl_name","enabled","channel_idx_mask"]},"title":"Wavef'
      + 'orm Settings","$schema":"http://json-schema.org/draft-04/schema#","des'
      + 'cription":"Scanner settings for waveform output","type":"array"}'
    );

    Example : (''
      + '[{"smart_enabled":true,"pass_rng_less":13.11,"pass_ampl_greater":1.0,"'
      + 'pass_ampl_less":5.0,"sbl_name":"wfm","enabled":true,"pass_rng_greater"'
      + ':9.27,"channel_idx_mask":11},{"enabled":false,"channel_idx_mask":32,"s'
      + 'bl_name":"wwf"}]'
    );
  );

// Window analysis data estimated from scandata and resulting filter parameters
const
  RDB_RIEGL_WINDOW_ANALYSIS : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.window_analysis';
    Title       : 'Window Analysis';
    Description : 'Window analysis data estimated from scandata and resulting filter parameters';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Window Analysis","required":["result","filter","settings"],"'
      + '$schema":"http://json-schema.org/draft-04/schema#","properties":{"sett'
      + 'ings":{"properties":{"range":{"properties":{"sigma_factor":{"type":"nu'
      + 'mber"},"additive_value":{"type":"number"}},"type":"object","required":'
      + '["sigma_factor","additive_value"]},"amplitude":{"properties":{"sigma_f'
      + 'actor":{"type":"number"},"additive_value":{"type":"number"}},"type":"o'
      + 'bject","required":["sigma_factor","additive_value"]}},"type":"object",'
      + '"required":["range","amplitude"]},"filter":{"properties":{"amplitude_m'
      + 'ax":{"items":{"type":"number"},"type":"array","description":"[dB]"},"a'
      + 'ngle":{"items":{"type":"number"},"type":"array","description":"[deg]"}'
      + ',"range_min":{"items":{"type":"number"},"type":"array","description":"'
      + '[m]"},"range_max":{"items":{"type":"number"},"type":"array","descripti'
      + 'on":"[m]"}},"type":"object","required":["angle","range_min","range_max'
      + '","amplitude_max"]},"result":{"properties":{"range_mean":{"items":{"ty'
      + 'pe":"number"},"type":"array","description":"[m]"},"amplitude_offset":{'
      + '"items":{"type":"number"},"type":"array","description":"[dB]"},"amplit'
      + 'ude_mean":{"items":{"type":"number"},"type":"array","description":"[dB'
      + ']"},"angle":{"items":{"type":"number"},"type":"array","description":"['
      + 'deg]"},"range_sigma":{"items":{"type":"number"},"type":"array","descri'
      + 'ption":"[m]"},"amplitude_sigma":{"items":{"type":"number"},"type":"arr'
      + 'ay","description":"[dB]"},"timestamp":{"items":{"type":"number"},"type'
      + '":"array","description":"[s]"}},"type":"object","required":["angle","r'
      + 'ange_mean","range_sigma","amplitude_mean","amplitude_sigma","amplitude'
      + '_offset"]}},"description":"Window analysis data estimated from '
      + 'scandata and resulting filter parameters","type":"object"}'
    );

    Example : (''
      + '{"settings":{"range":{"sigma_factor":8,"additive_value":0.1},"amplitud'
      + 'e":{"sigma_factor":4,"additive_value":1.0}},"filter":{"amplitude_max":'
      + '[8.04,8.01,7.99,7.96,7.93,7.9,7.88,7.85,7.83,7.8,7.78],"angle":[14.9,1'
      + '5.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.9],"range_min":[-0.208,'
      + '-0.21,-0.212,-0.214,-0.216,-0.218,-0.219,-0.221,-0.223,-0.225,-0.227],'
      + '"range_max":[0.424,0.425,0.426,0.427,0.428,0.428,0.429,0.43,0.431,0.43'
      + '1,0.432]},"result":{"range_mean":[0.1105621,0.1079564,0.1087088,0.1067'
      + '261,0.1054582,0.1090412,0.102871,0.1019044,0.1051523,0.1058445,0.10312'
      + '61],"amplitude_offset":[1.9,1.9],"amplitude_mean":[5.347396,5.263155,5'
      + '.224655,5.179926,5.097782,5.116479,5.051756,4.983473,5.007885,5.002441'
      + ',4.982],"angle":[14.9,15.0,15.1,15.2,15.3,15.4,15.5,15.6,15.7,15.8,15.'
      + '9],"range_sigma":[0.01869652,0.02151435,0.01747969,0.01918765,0.019457'
      + '76,0.01934862,0.01955329,0.02225589,0.02229977,0.01899122,0.02009433],'
      + '"amplitude_sigma":[0.4272844,0.4298479,0.4236816,0.4283583,0.4362353,0'
      + '.4315141,0.4373984,0.4472798,0.4346001,0.4345487,0.4540681],"timestamp'
      + '":[408.4441,411.4443]}}'
    );
  );

// Correction parameters for window glass echoes
const
  RDB_RIEGL_WINDOW_ECHO_CORRECTION : TRDBPointcloudMetaDataItemInfo = (
    Name        : 'riegl.window_echo_correction';
    Title       : 'Window Echo Correction';
    Description : 'Correction parameters for window glass echoes';
    Status      : RDB_STATUS_OPTIONAL;

    Schema : (''
      + '{"title":"Window Echo Correction","required":["amplitude","range","sli'
      + 'ces"],"$schema":"http://json-schema.org/draft-04/schema#","properties"'
      + ':{"range":{"required":["minimum","maximum","entries"],"properties":{"m'
      + 'inimum":{"minimum":-2.0,"type":"number","maximum":2.0,"description":"M'
      + 'inimum range in m"},"maximum":{"minimum":-2.0,"type":"number","maximum'
      + '":2.0,"description":"Maximum range in '
      + 'm"},"entries":{"minimum":1,"type":"integer","description":"Number of '
      + 'range entries"}},"type":"object","description":"Range axis of '
      + 'correction table"},"amplitude":{"required":["minimum","maximum","entri'
      + 'es"],"properties":{"minimum":{"minimum":0.0,"type":"number","descripti'
      + 'on":"Minimum amplitude in '
      + 'dB"},"maximum":{"minimum":0.0,"type":"number","description":"Maximum '
      + 'amplitude in '
      + 'dB"},"entries":{"minimum":1,"type":"integer","description":"Number of '
      + 'amplitude entries"}},"type":"object","description":"Amplitude axis of '
      + 'correction table"},"slices":{"items":{"required":["amplitude","table"]'
      + ',"properties":{"table":{"items":{"items":{"items":{"description":"Tabl'
      + 'e cell (item 0: amplitude in dB, 1: range in m, 2: flags)","type":"num'
      + 'ber"},"maxItems":3,"type":"array","minItems":3,"description":"Table '
      + 'column (= range '
      + 'axis)"},"type":"array","minItems":1,"description":"Table row (= '
      + 'amplitude '
      + 'axis)"},"type":"array","minItems":1,"description":"Correction table '
      + '(dimension defined by the ''amplitude'' and ''range'' '
      + 'objects)"},"amplitude":{"type":"number","description":"Window echo '
      + 'amplitude of slice in dB"}},"type":"object","description":"Window echo'
      + ' correction parameter '
      + 'slice"},"type":"array"}},"description":"Correction parameters for '
      + 'window glass echoes","type":"object"}'
    );

    Example : (''
      + '{"range":{"minimum":-1.5060822940732335,"maximum":1.5060822940732335,"'
      + 'entries":128},"amplitude":{"minimum":2,"maximum":20,"entries":128},"sl'
      + 'ices":[{"table":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],"amplitude":1.5'
      + '},{"table":[[[6.23,0.3535,1]],[[5.54,0.72375,1]]],"amplitude":2.0}]}'
    );
  );

// Table of all meta data item details records
const
  RDB_META_DATA_ITEMS : array of PRDBPointcloudMetaDataItemInfo = nil; // filled during initialization

implementation

procedure FixStringEncoding;
begin
  // As the unit (.pas file) doesn't contain a BOM, we need to convert all strings
  // from UTF-8 to "native" encoding at runtime (tested on Delphi 7 and Delphi XE5)

  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Name)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Title)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Description)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Schema)));
  RDB_RIEGL_ANGULAR_NOTCH_FILTER.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGULAR_NOTCH_FILTER.Example)));

  RDB_RIEGL_ATMOSPHERE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Name)));
  RDB_RIEGL_ATMOSPHERE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Title)));
  RDB_RIEGL_ATMOSPHERE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Description)));
  RDB_RIEGL_ATMOSPHERE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Schema)));
  RDB_RIEGL_ATMOSPHERE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ATMOSPHERE.Example)));

  RDB_RIEGL_BEAM_GEOMETRY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Name)));
  RDB_RIEGL_BEAM_GEOMETRY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Title)));
  RDB_RIEGL_BEAM_GEOMETRY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Description)));
  RDB_RIEGL_BEAM_GEOMETRY.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Schema)));
  RDB_RIEGL_BEAM_GEOMETRY.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BEAM_GEOMETRY.Example)));

  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Name)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Title)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Description)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Schema)));
  RDB_RIEGL_CONTROL_OBJECT_CATALOG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_CATALOG.Example)));

  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Name)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Title)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Description)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Schema)));
  RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE.Example)));

  RDB_RIEGL_DETECTION_PROBABILITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Name)));
  RDB_RIEGL_DETECTION_PROBABILITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Title)));
  RDB_RIEGL_DETECTION_PROBABILITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Description)));
  RDB_RIEGL_DETECTION_PROBABILITY.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Schema)));
  RDB_RIEGL_DETECTION_PROBABILITY.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DETECTION_PROBABILITY.Example)));

  RDB_RIEGL_DEVICE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Name)));
  RDB_RIEGL_DEVICE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Title)));
  RDB_RIEGL_DEVICE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Description)));
  RDB_RIEGL_DEVICE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Schema)));
  RDB_RIEGL_DEVICE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE.Example)));

  RDB_RIEGL_DEVICE_GEOMETRY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Name)));
  RDB_RIEGL_DEVICE_GEOMETRY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Title)));
  RDB_RIEGL_DEVICE_GEOMETRY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Description)));
  RDB_RIEGL_DEVICE_GEOMETRY.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Schema)));
  RDB_RIEGL_DEVICE_GEOMETRY.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY.Example)));

  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Name)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Title)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Description)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Schema)));
  RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL.Example)));

  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Name)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Title)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Description)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Schema)));
  RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICE_OUTPUT_LIMITS.Example)));

  RDB_RIEGL_DEVICES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Name)));
  RDB_RIEGL_DEVICES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Title)));
  RDB_RIEGL_DEVICES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Description)));
  RDB_RIEGL_DEVICES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Schema)));
  RDB_RIEGL_DEVICES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVICES.Example)));

  RDB_RIEGL_DMI_CONFIG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Name)));
  RDB_RIEGL_DMI_CONFIG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Title)));
  RDB_RIEGL_DMI_CONFIG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Description)));
  RDB_RIEGL_DMI_CONFIG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Schema)));
  RDB_RIEGL_DMI_CONFIG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_CONFIG.Example)));

  RDB_RIEGL_DYNTRIG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Name)));
  RDB_RIEGL_DYNTRIG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Title)));
  RDB_RIEGL_DYNTRIG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Description)));
  RDB_RIEGL_DYNTRIG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Schema)));
  RDB_RIEGL_DYNTRIG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG.Example)));

  RDB_RIEGL_ECHO_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Name)));
  RDB_RIEGL_ECHO_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Title)));
  RDB_RIEGL_ECHO_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Description)));
  RDB_RIEGL_ECHO_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Schema)));
  RDB_RIEGL_ECHO_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_INFO.Example)));

  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Name)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Title)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Description)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Schema)));
  RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXPONENTIAL_DECOMPOSITION.Example)));

  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Name)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Title)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Description)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Schema)));
  RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAUSSIAN_DECOMPOSITION.Example)));

  RDB_RIEGL_GEO_TAG.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Name)));
  RDB_RIEGL_GEO_TAG.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Title)));
  RDB_RIEGL_GEO_TAG.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Description)));
  RDB_RIEGL_GEO_TAG.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Schema)));
  RDB_RIEGL_GEO_TAG.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEO_TAG.Example)));

  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Name)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Title)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Description)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Schema)));
  RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOMETRIC_SCALE_FACTOR.Example)));

  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Name)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Title)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Description)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Schema)));
  RDB_RIEGL_GEOREFERENCING_PARAMETERS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GEOREFERENCING_PARAMETERS.Example)));

  RDB_RIEGL_GNSS_PROCESSING_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Name)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Title)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Description)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Schema)));
  RDB_RIEGL_GNSS_PROCESSING_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GNSS_PROCESSING_INFO.Example)));

  RDB_RIEGL_ITEM_NAMES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Name)));
  RDB_RIEGL_ITEM_NAMES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Title)));
  RDB_RIEGL_ITEM_NAMES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Description)));
  RDB_RIEGL_ITEM_NAMES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Schema)));
  RDB_RIEGL_ITEM_NAMES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ITEM_NAMES.Example)));

  RDB_RIEGL_LAG_ANGLE_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LAG_ANGLE_CORRECTION.Name)));
  RDB_RIEGL_LAG_ANGLE_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LAG_ANGLE_CORRECTION.Title)));
  RDB_RIEGL_LAG_ANGLE_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LAG_ANGLE_CORRECTION.Description)));
  RDB_RIEGL_LAG_ANGLE_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LAG_ANGLE_CORRECTION.Schema)));
  RDB_RIEGL_LAG_ANGLE_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LAG_ANGLE_CORRECTION.Example)));

  RDB_RIEGL_LICENSES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Name)));
  RDB_RIEGL_LICENSES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Title)));
  RDB_RIEGL_LICENSES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Description)));
  RDB_RIEGL_LICENSES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Schema)));
  RDB_RIEGL_LICENSES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LICENSES.Example)));

  RDB_RIEGL_MTA_SETTINGS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Name)));
  RDB_RIEGL_MTA_SETTINGS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Title)));
  RDB_RIEGL_MTA_SETTINGS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Description)));
  RDB_RIEGL_MTA_SETTINGS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Schema)));
  RDB_RIEGL_MTA_SETTINGS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_SETTINGS.Example)));

  RDB_RIEGL_NEAR_RANGE_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Name)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Title)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Description)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Schema)));
  RDB_RIEGL_NEAR_RANGE_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NEAR_RANGE_CORRECTION.Example)));

  RDB_RIEGL_NOISE_ESTIMATES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Name)));
  RDB_RIEGL_NOISE_ESTIMATES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Title)));
  RDB_RIEGL_NOISE_ESTIMATES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Description)));
  RDB_RIEGL_NOISE_ESTIMATES.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Schema)));
  RDB_RIEGL_NOISE_ESTIMATES.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOISE_ESTIMATES.Example)));

  RDB_RIEGL_NOTCH_FILTER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Name)));
  RDB_RIEGL_NOTCH_FILTER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Title)));
  RDB_RIEGL_NOTCH_FILTER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Description)));
  RDB_RIEGL_NOTCH_FILTER.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Schema)));
  RDB_RIEGL_NOTCH_FILTER.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NOTCH_FILTER.Example)));

  RDB_RIEGL_PIXEL_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Name)));
  RDB_RIEGL_PIXEL_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Title)));
  RDB_RIEGL_PIXEL_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Description)));
  RDB_RIEGL_PIXEL_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Schema)));
  RDB_RIEGL_PIXEL_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_INFO.Example)));

  RDB_RIEGL_PLANE_PATCH_MATCHING.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Name)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Title)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Description)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Schema)));
  RDB_RIEGL_PLANE_PATCH_MATCHING.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_MATCHING.Example)));

  RDB_RIEGL_PLANE_PATCH_STATISTICS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Name)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Title)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Description)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Schema)));
  RDB_RIEGL_PLANE_PATCH_STATISTICS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_STATISTICS.Example)));

  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Name)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Title)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Description)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Schema)));
  RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS_INFO.Example)));

  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Name)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Title)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Description)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Schema)));
  RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_ATTRIBUTE_GROUPS.Example)));

  RDB_RIEGL_POINTCLOUD_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Name)));
  RDB_RIEGL_POINTCLOUD_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Title)));
  RDB_RIEGL_POINTCLOUD_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Description)));
  RDB_RIEGL_POINTCLOUD_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Schema)));
  RDB_RIEGL_POINTCLOUD_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINTCLOUD_INFO.Example)));

  RDB_RIEGL_POSE_ESTIMATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Name)));
  RDB_RIEGL_POSE_ESTIMATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Title)));
  RDB_RIEGL_POSE_ESTIMATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Description)));
  RDB_RIEGL_POSE_ESTIMATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Schema)));
  RDB_RIEGL_POSE_ESTIMATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_ESTIMATION.Example)));

  RDB_RIEGL_POSE_SENSORS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Name)));
  RDB_RIEGL_POSE_SENSORS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Title)));
  RDB_RIEGL_POSE_SENSORS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Description)));
  RDB_RIEGL_POSE_SENSORS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Schema)));
  RDB_RIEGL_POSE_SENSORS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POSE_SENSORS.Example)));

  RDB_RIEGL_PULSE_POSITION_MODULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Name)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Title)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Description)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Schema)));
  RDB_RIEGL_PULSE_POSITION_MODULATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_POSITION_MODULATION.Example)));

  RDB_RIEGL_RANGE_STATISTICS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Name)));
  RDB_RIEGL_RANGE_STATISTICS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Title)));
  RDB_RIEGL_RANGE_STATISTICS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Description)));
  RDB_RIEGL_RANGE_STATISTICS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Schema)));
  RDB_RIEGL_RANGE_STATISTICS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE_STATISTICS.Example)));

  RDB_RIEGL_RECEIVER_INTERNALS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Name)));
  RDB_RIEGL_RECEIVER_INTERNALS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Title)));
  RDB_RIEGL_RECEIVER_INTERNALS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Description)));
  RDB_RIEGL_RECEIVER_INTERNALS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Schema)));
  RDB_RIEGL_RECEIVER_INTERNALS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RECEIVER_INTERNALS.Example)));

  RDB_RIEGL_REFLECTANCE_CALCULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Name)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Title)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Description)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Schema)));
  RDB_RIEGL_REFLECTANCE_CALCULATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CALCULATION.Example)));

  RDB_RIEGL_REFLECTANCE_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Name)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Title)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Description)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Schema)));
  RDB_RIEGL_REFLECTANCE_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE_CORRECTION.Example)));

  RDB_RIEGL_RILOC_PROCESSING_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Name)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Title)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Description)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Schema)));
  RDB_RIEGL_RILOC_PROCESSING_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RILOC_PROCESSING_INFO.Example)));

  RDB_RIEGL_SCAN_PATTERN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Name)));
  RDB_RIEGL_SCAN_PATTERN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Title)));
  RDB_RIEGL_SCAN_PATTERN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Description)));
  RDB_RIEGL_SCAN_PATTERN.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Schema)));
  RDB_RIEGL_SCAN_PATTERN.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_PATTERN.Example)));

  RDB_RIEGL_SHOT_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Name)));
  RDB_RIEGL_SHOT_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Title)));
  RDB_RIEGL_SHOT_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Description)));
  RDB_RIEGL_SHOT_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Schema)));
  RDB_RIEGL_SHOT_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INFO.Example)));

  RDB_RIEGL_STORED_FILTERS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Name)));
  RDB_RIEGL_STORED_FILTERS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Title)));
  RDB_RIEGL_STORED_FILTERS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Description)));
  RDB_RIEGL_STORED_FILTERS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Schema)));
  RDB_RIEGL_STORED_FILTERS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STORED_FILTERS.Example)));

  RDB_RIEGL_SYSTEM_DESCRIPTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Name)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Title)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Description)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Schema)));
  RDB_RIEGL_SYSTEM_DESCRIPTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SYSTEM_DESCRIPTION.Example)));

  RDB_RIEGL_TEMPERATURE_CALCULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Name)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Title)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Description)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Schema)));
  RDB_RIEGL_TEMPERATURE_CALCULATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_CALCULATION.Example)));

  RDB_RIEGL_TIME_BASE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Name)));
  RDB_RIEGL_TIME_BASE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Title)));
  RDB_RIEGL_TIME_BASE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Description)));
  RDB_RIEGL_TIME_BASE.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Schema)));
  RDB_RIEGL_TIME_BASE.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIME_BASE.Example)));

  RDB_RIEGL_TRAJECTORY_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Name)));
  RDB_RIEGL_TRAJECTORY_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Title)));
  RDB_RIEGL_TRAJECTORY_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Description)));
  RDB_RIEGL_TRAJECTORY_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Schema)));
  RDB_RIEGL_TRAJECTORY_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TRAJECTORY_INFO.Example)));

  RDB_RIEGL_TTIP_CONFIGURATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Name)));
  RDB_RIEGL_TTIP_CONFIGURATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Title)));
  RDB_RIEGL_TTIP_CONFIGURATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Description)));
  RDB_RIEGL_TTIP_CONFIGURATION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Schema)));
  RDB_RIEGL_TTIP_CONFIGURATION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TTIP_CONFIGURATION.Example)));

  RDB_RIEGL_VERTEX_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Name)));
  RDB_RIEGL_VERTEX_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Title)));
  RDB_RIEGL_VERTEX_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Description)));
  RDB_RIEGL_VERTEX_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Schema)));
  RDB_RIEGL_VERTEX_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_INFO.Example)));

  RDB_RIEGL_VOXEL_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Name)));
  RDB_RIEGL_VOXEL_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Title)));
  RDB_RIEGL_VOXEL_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Description)));
  RDB_RIEGL_VOXEL_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Schema)));
  RDB_RIEGL_VOXEL_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INFO.Example)));

  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Name)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Title)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Description)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Schema)));
  RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS.Example)));

  RDB_RIEGL_WAVEFORM_INFO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Name)));
  RDB_RIEGL_WAVEFORM_INFO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Title)));
  RDB_RIEGL_WAVEFORM_INFO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Description)));
  RDB_RIEGL_WAVEFORM_INFO.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Schema)));
  RDB_RIEGL_WAVEFORM_INFO.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_INFO.Example)));

  RDB_RIEGL_WAVEFORM_SETTINGS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Name)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Title)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Description)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Schema)));
  RDB_RIEGL_WAVEFORM_SETTINGS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_SETTINGS.Example)));

  RDB_RIEGL_WINDOW_ANALYSIS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Name)));
  RDB_RIEGL_WINDOW_ANALYSIS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Title)));
  RDB_RIEGL_WINDOW_ANALYSIS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Description)));
  RDB_RIEGL_WINDOW_ANALYSIS.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Schema)));
  RDB_RIEGL_WINDOW_ANALYSIS.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ANALYSIS.Example)));

  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Name)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Title)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Description)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Schema      := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Schema)));
  RDB_RIEGL_WINDOW_ECHO_CORRECTION.Example     := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_CORRECTION.Example)));
end;

procedure FillMetadataArray;
var
  i : Integer;
begin
  SetLength(RDB_META_DATA_ITEMS, 56); i := -1;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ANGULAR_NOTCH_FILTER;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ATMOSPHERE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_BEAM_GEOMETRY;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_CONTROL_OBJECT_CATALOG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_CONTROL_OBJECT_REFERENCE_FILE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DETECTION_PROBABILITY;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE_GEOMETRY;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE_GEOMETRY_PASSIVE_CHANNEL;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICE_OUTPUT_LIMITS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DEVICES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DMI_CONFIG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_DYNTRIG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ECHO_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_EXPONENTIAL_DECOMPOSITION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GAUSSIAN_DECOMPOSITION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GEO_TAG;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GEOMETRIC_SCALE_FACTOR;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GEOREFERENCING_PARAMETERS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_GNSS_PROCESSING_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_ITEM_NAMES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_LAG_ANGLE_CORRECTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_LICENSES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_MTA_SETTINGS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_NEAR_RANGE_CORRECTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_NOISE_ESTIMATES;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_NOTCH_FILTER;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PIXEL_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PLANE_PATCH_MATCHING;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PLANE_PATCH_STATISTICS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PLANE_SLOPE_CLASS_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POINT_ATTRIBUTE_GROUPS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POINTCLOUD_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POSE_ESTIMATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_POSE_SENSORS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_PULSE_POSITION_MODULATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_RANGE_STATISTICS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_RECEIVER_INTERNALS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_REFLECTANCE_CALCULATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_REFLECTANCE_CORRECTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_RILOC_PROCESSING_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_SCAN_PATTERN;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_SHOT_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_STORED_FILTERS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_SYSTEM_DESCRIPTION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TEMPERATURE_CALCULATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TIME_BASE;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TRAJECTORY_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_TTIP_CONFIGURATION;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_VERTEX_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_VOXEL_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WAVEFORM_AVERAGING_SETTINGS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WAVEFORM_INFO;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WAVEFORM_SETTINGS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WINDOW_ANALYSIS;
  Inc(i); RDB_META_DATA_ITEMS[i] := @RDB_RIEGL_WINDOW_ECHO_CORRECTION;
end;

initialization

FixStringEncoding;
FillMetadataArray;

end.
