{*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Description of RIEGL point attributes in RDB 2 database files
 * \version 2015-11-05/AW: Initial version
 * \version 2015-11-20/AW: Minimum value of "riegl.pulse_width" fixed
 * \version 2016-09-30/AW: New attributes for voxel and plane patch datasets
 * \version 2016-10-25/AW: Point coordinates correction vector attribute added
 * \version 2016-11-17/AW: Attributes for voxel datasets updated
 * \version 2016-11-23/AW: Added constants for named point attribute values
 * \version 2016-12-05/AW: Compression options added
 * \version 2016-12-20/AW: JSON: switched names from camel case to underscores
 * \version 2017-01-20/AW: Maximum value of "riegl.pulse_width" fixed
 * \version 2017-01-23/AW: New attributes for laser shots added: riegl.shot_...
 * \version 2017-01-31/AW: New attributes for FWA and MTA tools added
 * \version 2017-02-24/AW: Definition of riegl.covariances fixed
 * \version 2017-03-02/AW: New point attribute riegl.plane_count added
 * \version 2017-03-22/AW: Optional attribute scale factor added
 * \version 2017-05-10/AW: Definition of riegl.plane_count fixed
 * \version 2017-06-27/AW: Description of target_index and target_count fixed
 * \version 2017-08-22/AW: New attributes for waveform sample blocks and values
 * \version 2017-10-03/AW: Description of wfm_sbl_first and wfm_sda_first fixed
 * \version 2017-10-09/AW: Definition of riegl.amplitude modified
 * \version 2017-10-24/AW: New attribute riegl.line_angle_coarse added
 * \version 2017-11-10/AW: Definition of riegl.timestamp modified (#2588)
 * \version 2017-11-20/AW: Resolution of riegl.pca_extents increased
 * \version 2017-11-21/AW: Attributes for trajectory (position+orientation) files
 * \version 2017-11-22/AW: New attribute riegl.voxel_collapsed added
 * \version 2017-11-22/AW: Resolution of riegl.std_dev increased
 * \version 2018-01-10/AW: New attribute riegl.direction_coarse added
 * \version 2018-01-15/AW: Comments of riegl.fwa updated
 * \version 2018-02-14/AW: Definition of riegl.timestamp fixed (#2588)
 * \version 2018-02-22/AW: Description of target_index and target_count fixed
 * \version 2018-03-09/AW: New attribute property "invalid value" added (#3047)
 * \version 2018-03-26/AW: Definition of riegl.temperature modified
 * \version 2018-04-20/AW: Definition of riegl.[pof_]timestamp fixed (#2588)
 * \version 2018-05-24/AW: Attributes for voxel and pixel datasets added
 * \version 2018-06-08/AW: Attributes for waveform and echo datasets added
 * \version 2018-06-25/AW: Definition of riegl.wfm_sbl_std_dev fixed
 * \version 2018-06-28/AW: New attributes riegl.pps_timestamp_[ex|in]tern added
 * \version 2018-07-04/AW: Invalid value for "riegl.pulse_width" added
 * \version 2018-11-19/AW: New attribute riegl.direction_medium added
 * \version 2018-11-28/AW: Definition of riegl.scan_angle modified
 * \version 2018-12-04/AW: Definition of riegl.plane_spread removed
 * \version 2019-02-25/AW: New attribute riegl.xyz_socs added
 * \version 2019-03-14/AW: New attribute riegl.voxel_index added
 * \version 2019-03-14/AW: New attribute riegl.voxel_linear_sums added
 * \version 2019-03-14/AW: New attribute riegl.voxel_square_sums added
 * \version 2019-03-14/AW: New attribute riegl.pixel_linear_sums added
 * \version 2019-03-14/AW: New attribute riegl.pixel_square_sums added
 * \version 2019-03-14/AW: New attribute riegl.hydro_refraction_corrected added
 * \version 2019-03-14/AW: New attribute riegl.hydro_intersection_point added
 * \version 2019-03-14/AW: New attribute riegl.hydro_intersection_normal added
 * \version 2019-04-10/AW: Definition of riegl.xy_map and riegl.xyz_map fixed
 * \version 2019-04-15/AW: Attributes were separated into groups
 * \version 2019-05-16/AW: New attribute riegl.hydro_wsm_uncertainty added
 * \version 2019-05-22/AW: New attribute riegl.plane_patch_distance added
 * \version 2019-05-22/AW: New attribute riegl.plane_patch_angular_distance added
 * \version 2019-05-22/AW: Titles of riegl.pof_satellites_* attributes fixed
 * \version 2019-06-25/AW: New attribute riegl.pof_satellites_qzss added
 * \version 2019-11-07/AW: Attributes riegl.pof_xyz and riegl.pof_*_ned added
 * \version 2019-11-11/AW: Attribute riegl.plane_patch_lateral_distance added
 * \version 2019-11-15/AW: Attribute riegl.plane_patch_link_vector added
 * \version 2019-11-22/AW: Attribute riegl.plane_occupancy added
 * \version 2019-11-22/AW: Attributes riegl.raw_(line|frame)_angle added
 * \version 2019-11-25/AW: Attribute riegl.plane_cog_link added
 * \version 2019-11-25/AW: Attribute riegl.plane_confidence_normal added
 * \version 2019-12-02/AW: Attribute riegl.match_count added
 * \version 2019-12-13/AW: Attributes for tie-/control objects added
 * \version 2019-12-19/AW: Attributes for tie-/control objects added
 * \version 2019-12-19/AW: Title and description of riegl.mta_unresolved updated
 * \version 2020-01-08/AW: Move riegl.height_center/mean/min/max before riegl.point_count
 * \version 2020-04-15/AW: Attribute riegl.reference_object_id updated
 * \version 2020-06-12/AW: Attribute riegl.scanner_position added
 * \version 2020-09-10/AW: Attribute riegl.line_angle_reduced added
 * \version 2020-09-17/AW: Attribute riegl.background_radiation updated
 * \version 2020-09-24/AW: Attribute riegl.background_radiation_si added
 * \version 2020-09-24/AW: Attribute riegl.background_radiation_ingaas added
 * \version 2020-09-24/AW: Attribute riegl.temperature_estimated_si added
 * \version 2020-09-24/AW: Attribute riegl.temperature_estimated_ingaas added
 * \version 2020-09-25/AW: Attribute riegl.temperature_estimated_ingaas_si added
 * \version 2020-10-08/AW: Attribute riegl.window_echo_impact_corrected added
 * \version 2020-11-16/AW: Attribute riegl.point_count_grid_cell added (#3720)
 * \version 2020-11-26/AW: Resolution of riegl.pof_accuracy_* attributes updated (#3761)
 * \version 2020-12-02/AW: Move riegl.pof_xyz after riegl.pof_yaw (#3760)
 * \version 2020-12-03/AW: Add riegl.scan_line_index and riegl.shot_index_line (#3759)
 * \version 2021-02-02/AW: Attribute riegl.mirror_facet updated (storage "variable")
 * \version 2021-02-02/AW: Attribute riegl.plane_slope_class added (rdbplanes/#7)
 * \version 2021-02-16/AW: Attribute riegl.source_cloud_count added (#3810)
 * \version 2021-03-03/AW: Attribute riegl.shot_direction_levelled added (#3820)
 * \version 2021-04-14/AW: Attribute riegl.obs_confidence_xy added (#3861)
 * \version 2021-04-14/AW: Attribute riegl.obs_confidence_z added (#3861)
 * \version 2021-04-14/AW: Attribute riegl.obs_signal_confidence_rot added (#3861)
 * \version 2021-06-30/AW: Attributes riegl.pca_axis_min and _max updated (#3930)
 * \version 2021-07-09/AW: Attribute riegl.extinction added (#3935)
 * \version 2021-07-14/AW: Attributes riegl.svb_* added (#3945)
 * \version 2021-09-08/AW: Attribute riegl.dynamic_object_point added (#3979)
 * \version 2021-09-08/AW: Attribute riegl.single_source_point added (#3975)
 * \version 2021-09-08/AW: Attribute riegl.mirror_object_point added (#3978)
 * \version 2021-09-20/AW: Attribute riegl.plane_cluster_id added (#3997)
 * \version 2021-11-04/AW: Attribute riegl.nir added (#4042)
 * \version 2021-12-07/AW: Add riegl.obs_confidence_range/_theta/_phi (#4075)
 * \version 2022-01-26/AW: Add optional point attribute tags (#4128)
 * \version 2022-03-11/AW: Add "boolean" and "enumeration" tags (#4128)
 * \version 2022-03-18/AW: Add named values for enumeration attributes (#4128)
 * \version 2022-03-31/AW: Attribute riegl.source_indicator updated (#4128/17)
 * \version 2022-04-13/AW: Attribute riegl.target_type added (#4188)
 * \version 2022-04-20/AW: Attribute riegl.shot_biaxial_shift added (#4195)
 * \version 2022-05-20/AW: Attribute riegl.cp_search_radius added (#4236)
 * \version 2022-05-20/AW: Attribute riegl.cp_maximum_distance added (#4236)
 * \version 2022-05-30/AW: Attribute riegl.dyntrig_uncertain_point added (#4217)
 * \version 2022-08-16/AW: Attribute riegl.segment_id added (#4307)
 * \version 2022-09-06/AW: Attribute renamed to riegl.mta_uncertain_point (#4335)
 * \version 2022-10-06/AW: Attribute riegl.rgba updated (description)
 * \version 2022-10-10/AW: Add "NED" suffix to the titles of "riegl.pof_*_ned"
 * \version 2022-11-09/AW: Attribute riegl.pof_age_of_corrections added (#4412)
 * \version 2022-11-09/AW: Attribute riegl.pof_baseline_length added (#4412)
 * \version 2022-11-09/AW: Attribute riegl.pof_solution_gnss added (#4412)
 * \version 2023-01-19/AW: Attribute riegl.event_timestamp added (#4449)
 * \version 2023-01-19/AW: Attribute riegl.event_index added (#4449)
 * \version 2023-01-19/AW: Attribute riegl.event_pulse_width added (#4449)
 * \version 2023-01-19/AW: Attribute riegl.event_fake_pulse added (#4449)
 * \version 2023-01-30/AW: Attribute riegl.accelerometer updated (#4523)
 * \version 2023-01-30/AW: Attribute riegl.gyroscope updated (#4523)
 * \version 2023-04-21/AW: Attribute riegl.magnetic_field_sensor updated (#4603)
 * \version 2023-04-21/AW: Attribute riegl.distance_to_surface updated (#4612)
 * \version 2023-05-24/AW: Attribute riegl.point_vector added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.pca_diameter added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.angle_of_incidence added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.eigenvalue_norm_min added (#4642)
 * \version 2023-05-24/AW: Attribute riegl.eigenvalue_norm_max added (#4642)
 * \version 2023-05-30/AW: Add "timestamp" tag for timestamp attributes (#4433)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_axis_direction added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_radius added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_height added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_origin_confidence added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_radius_confidence added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_direction_confidence added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_inlier_count added (#4648)
 * \version 2023-05-31/AW: Attribute riegl.cylinder_outlier_count added (#4648)
 * \version 2023-06-02/AW: Attribute riegl.point_inside_aabb_bocs added (#4653)
 * \version 2023-06-02/AW: Attribute riegl.point_outside_aabb_bocs added (#4653)
 * \version 2023-06-07/AW: Update resolutions of direction vector attributes (#4669)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_min added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_max added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_mean added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_median added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_std added (#4673)
 * \version 2023-06-07/AW: Attribute riegl.nbhd_distance_mad added (#4673)
 * \version 2023-06-15/AW: Attribute riegl.accelerometer updated (#4689)
 * \version 2023-06-15/AW: Attribute riegl.gyroscope updated (#4689)
 * \version 2023-11-13/AW: Attribute riegl.svb_reflectance_volumetric added (#4819)
 * \version 2023-11-13/AW: Attribute riegl.svb_reflectance_bottom_uncorrected added (#4819)
 * \version 2023-11-13/AW: Attribute riegl.svb_attenuation_volumetric added (#4819)
 * \version 2023-11-13/AW: Attribute riegl.svb_target_attenuation_coefficient_volumetric added (#4819)
 * \version 2023-12-06/AW: Update invalid value of riegl.amplitude (#4870)
 * \version 2023-12-06/AW: Update invalid value of riegl.reflectance (#4870)
 * \version 2023-12-11/AW: Update invalid value of riegl.background_radiation_si (#4873)
 * \version 2023-12-11/AW: Update invalid value of riegl.background_radiation_ingaas (#4873)
 * \version 2024-01-30/AW: Attribute riegl.xyz_lvcs added (#4929)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_north_east added (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_east_down added (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_down_north added (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_north updated (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_east updated (#5076)
 * \version 2024-04-16/AW: Attribute riegl.pof_accuracy_down updated (#5076)
 * \version 2024-05-16/AW: Attribute riegl.tpu_wsm_z_uncertainty added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_wsm_normal_uncertainty added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_xyz_uncertainties added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_thu added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.tpu_tvu added (#5114)
 * \version 2024-05-16/AW: Attribute riegl.xyz_map_min added (#5117)
 * \version 2024-05-16/AW: Attribute riegl.xyz_map_max added (#5117)
 * \version 2024-10-09/AW: Attribute riegl.cp_surface_northing_angle added (#5327)
 * \version 2024-10-09/AW: Attribute riegl.cp_surface_northing_tolerance_angle added (#5327)
 * \version 2024-10-15/AW: Attribute riegl.pof_xyz_velocity added (#5333)
 * \version 2024-10-15/AW: Attribute riegl.pof_xyz_lvcs_velocity added (#5333)
 * \version 2024-10-15/AW: Attribute riegl.pof_yaw_velocity added (#5333)
 * \version 2024-10-16/AW: Update titles to use "Title Case" style (#5335)
 * \version 2024-10-21/AW: Attribute riegl.pof_roll_velocity added (#5342)
 * \version 2024-10-21/AW: Attribute riegl.pof_pitch_velocity added (#5342)
 * \version 2024-11-15/AW: Attribute riegl.accelerometer_bias added (#5385)
 * \version 2024-11-15/AW: Attribute riegl.accelerometer_scale added (#5385)
 * \version 2024-11-15/AW: Attribute riegl.gyroscope_bias added (#5385)
 * \version 2024-11-15/AW: Attribute riegl.gyroscope_scale added (#5385)
 * \version 2024-12-11/AW: Attribute riegl.pof_xyz_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.pof_roll_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.pof_pitch_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.pof_yaw_velocity updated (#5342)
 * \version 2024-12-11/AW: Attribute riegl.mta_uncertain_point_vol added (#5357)
 * \version 2024-12-11/AW: Attribute riegl.mta_uncertain_point_vol_diff added (#5357)
 * \version 2025-03-21/AW: Attribute riegl.dmi_count added (#5571)
 * \version 2025-03-21/AW: Attribute riegl.dmi_path_length_raw added (#5571)
 * \version 2025-03-21/AW: Attribute riegl.pof_course added (#5595)
 * \version 2025-03-21/AW: Attribute riegl.pof_xyz_ned_velocity added (#5595)
 * \version 2025-03-24/AW: Attribute riegl.dmi_scale added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_ned added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_north added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_east added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_down added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_north_east added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_east_down added (#5595)
 * \version 2025-03-31/AW: Attribute riegl.pof_velocity_accuracy_down_north added (#5595)
 * \version 2025-03-31/AW: Update description of riegl.pof_xyz_velocity (#5595)
 * \version 2025-03-31/AW: Update description of riegl.pof_xyz_ned_velocity (#5595)
 * \version 2025-03-31/AW: Update description of riegl.pof_xyz_lvcs_velocity (#5595)
 * \version 2025-04-03/AW: Attribute riegl.pof_undulation added (#5595)
 * \version 2025-08-29/AW: Attribute riegl.obs_confidence_xyz added (#5864)
 *
 *******************************************************************************
 *}

unit riegl.rdb.default.attributes;

{$WRITEABLECONST ON} // see section "initialization"

interface

uses
  Math,
  riegl.rdb,
  riegl.rdb.pointcloud.pointAttribute;

// Point attribute definition version
const
  RDB_POINT_ATTRIBUTES_VERSION = '1.4.8';
  RDB_POINT_ATTRIBUTES_DATE = '2025-08-29';

// Invalid value
const
  RDB_NO_INVALID_VALUE = NaN; // attribute has no invalid value

// Storage classes
{$IF not Declared(RDB_STORAGE_CONSTANT)}
const RDB_STORAGE_CONSTANT = 1; // value cannot be changed
{$IFEND}
{$IF not Declared(RDB_STORAGE_VARIABLE)}
const RDB_STORAGE_VARIABLE = 2; // value can change from time to time
{$IFEND}
{$IF not Declared(RDB_STORAGE_DYNAMIC)}
const RDB_STORAGE_DYNAMIC = 3; // value is likely to be changed often
{$IFEND}

// Compression options
{$IF not Declared(RDB_COMPRESSION_DEFAULT)}
const RDB_COMPRESSION_DEFAULT = 0; // nothing special, just use default compression algorithm
{$IFEND}
{$IF not Declared(RDB_COMPRESSION_DELTA)}
const RDB_COMPRESSION_DELTA = 1; // calculate differences between two consecutive values
{$IFEND}
{$IF not Declared(RDB_COMPRESSION_SHUFFLE)}
const RDB_COMPRESSION_SHUFFLE = 2; // shuffle bytes of point attribute values
{$IFEND}
{$IF not Declared(RDB_COMPRESSION_DELTA_SHUFFLE)}
const RDB_COMPRESSION_DELTA_SHUFFLE = 3; // calculate differences and shuffle bytes
{$IFEND}

// Container record for point attribute details
type
  TRDBPointcloudPointAttributeInfo = record
    Name               : System.String;
    Title              : System.String;
    Tags               : System.String;
    Description        : System.String;
    UnitSymbol         : System.String;
    Length             : System.Cardinal;
    Resolution         : System.Double;
    MinimumValue       : System.Double;
    MaximumValue       : System.Double;
    DefaultValue       : System.Double;
    InvalidValue       : System.Double;
    NamedValues        : System.String;
    StorageClass       : System.Byte;
    CompressionOptions : System.Byte;
    LodSettings        : System.String;
    ScaleFactor        : System.Double;
  end;
  PRDBPointcloudPointAttributeInfo = ^TRDBPointcloudPointAttributeInfo;

// Copy point attribute definition
procedure RDBAssignPointAttribute(
  const Source : TRDBPointcloudPointAttributeInfo;
  var   Target : TRDBPointcloudPointAttribute
); overload;
//
procedure RDBAssignPointAttribute(
  const Source : TRDBPointcloudPointAttribute;
  var   Target : TRDBPointcloudPointAttribute
); overload;

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Coordinates/Vectors"
//______________________________________________________________________________
//

// Cartesian point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_XYZ : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz';
    Title              : 'XYZ';
    Tags               : 'position, transform';
    Description        : (''
      + 'Cartesian point coordinates wrt. application coordinate system (0: X, '
      + '1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -535000.0;
    MaximumValue       : 535000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_XYZ_X : Cardinal = 0;
  RDB_RIEGL_XYZ_Y : Cardinal = 1;
  RDB_RIEGL_XYZ_Z : Cardinal = 2;

// Cartesian point coordinates wrt. scanner's coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_XYZ_SOCS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_socs';
    Title              : 'XYZ SOCS';
    Tags               : 'position';
    Description        : (''
      + 'Cartesian point coordinates wrt. scanner''s coordinate system (0: X, 1:'
      + ' Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -535000.0;
    MaximumValue       : 535000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_XYZ_SOCS_X : Cardinal = 0;
  RDB_RIEGL_XYZ_SOCS_Y : Cardinal = 1;
  RDB_RIEGL_XYZ_SOCS_Z : Cardinal = 2;

// Cartesian point coordinates wrt. levelled vehicle coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_XYZ_LVCS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_lvcs';
    Title              : 'XYZ LVCS';
    Tags               : 'position';
    Description        : (''
      + 'Cartesian point coordinates wrt. levelled vehicle coordinate system '
      + '(0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.1;
    MinimumValue       : -10000.0;
    MaximumValue       : 10000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_XYZ_LVCS_X : Cardinal = 0;
  RDB_RIEGL_XYZ_LVCS_Y : Cardinal = 1;
  RDB_RIEGL_XYZ_LVCS_Z : Cardinal = 2;

// Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, 0: Easting, 1: Northing, 2: Height)
const
  RDB_RIEGL_XYZ_MAP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_map';
    Title              : 'XYZ Map';
    Tags               : 'position, transform';
    Description        : (''
      + 'Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, '
      + '0: Easting, 1: Northing, 2: Height)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -20037508.343;
    MaximumValue       : 20037508.343;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_XYZ_MAP_EASTING : Cardinal = 0;
  RDB_RIEGL_XYZ_MAP_NORTHING : Cardinal = 1;
  RDB_RIEGL_XYZ_MAP_HEIGHT : Cardinal = 2;

// Minimum distances to the voxel corner (0: dx, 1: dy, 2: dz)
const
  RDB_RIEGL_XYZ_MAP_MIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_map_min';
    Title              : 'XYZ Map Minimum';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Minimum distances to the voxel corner (0: dx, 1: dy, 2: dz)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Maximum distances to the voxel corner (0: dx, 1: dy, 2: dz)
const
  RDB_RIEGL_XYZ_MAP_MAX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_map_max';
    Title              : 'XYZ Map Maximum';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Maximum distances to the voxel corner (0: dx, 1: dy, 2: dz)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, 0: Easting, 1: Northing)
const
  RDB_RIEGL_XY_MAP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xy_map';
    Title              : 'XY Map';
    Tags               : 'position, transform';
    Description        : (''
      + 'Point coordinates wrt. a projected CRS (e.g. Web Mercator EPSG:3857, '
      + '0: Easting, 1: Northing)'
    );
    UnitSymbol         : 'm';
    Length             : 2;
    Resolution         : 0.00933069192934280443318950659659094526432454586029052734375;
    MinimumValue       : -20037508.3427892439067363739013671875;
    MaximumValue       : 20037508.333458550274372100830078125;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_XY_MAP_EASTING : Cardinal = 0;
  RDB_RIEGL_XY_MAP_NORTHING : Cardinal = 1;

// Corrections that were applied (added) to the Cartesian point coordinates (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_XYZ_CORRECTIONS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_corrections';
    Title              : 'XYZ Corrections';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Corrections that were applied (added) to the Cartesian point '
      + 'coordinates (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -5000.0;
    MaximumValue       : 5000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_XYZ_CORRECTIONS_X : Cardinal = 0;
  RDB_RIEGL_XYZ_CORRECTIONS_Y : Cardinal = 1;
  RDB_RIEGL_XYZ_CORRECTIONS_Z : Cardinal = 2;

// Target distance wrt. SOCS origin
const
  RDB_RIEGL_RANGE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.range';
    Title              : 'Range';
    Tags               : '';
    Description        : (''
      + 'Target distance wrt. SOCS origin'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Polar angle (inclination) wrt. SOCS (0..180°)
const
  RDB_RIEGL_THETA : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.theta';
    Title              : 'Theta';
    Tags               : '';
    Description        : (''
      + 'Polar angle (inclination) wrt. SOCS (0..180°)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Azimuthal angle wrt. SOCS (0..360°)
const
  RDB_RIEGL_PHI : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.phi';
    Title              : 'Phi';
    Tags               : '';
    Description        : (''
      + 'Azimuthal angle wrt. SOCS (0..360°)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Cartesian coordinates of the scanner position wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_SCANNER_POSITION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.scanner_position';
    Title              : 'Scanner Position';
    Tags               : 'position, transform';
    Description        : (''
      + 'Cartesian coordinates of the scanner position wrt. application '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -535000.0;
    MaximumValue       : 535000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_DIRECTION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.direction';
    Title              : 'Direction';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Laser beam direction vector wrt. application coordinate system (0: X, '
      + '1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DIRECTION_X : Cardinal = 0;
  RDB_RIEGL_DIRECTION_Y : Cardinal = 1;
  RDB_RIEGL_DIRECTION_Z : Cardinal = 2;

// Laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_DIRECTION_MEDIUM : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.direction_medium';
    Title              : 'Direction';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Laser beam direction vector wrt. application coordinate system (0: X, '
      + '1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 0.0007;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DIRECTION_MEDIUM_X : Cardinal = 0;
  RDB_RIEGL_DIRECTION_MEDIUM_Y : Cardinal = 1;
  RDB_RIEGL_DIRECTION_MEDIUM_Z : Cardinal = 2;

// Coarse laser beam direction vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_DIRECTION_COARSE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.direction_coarse';
    Title              : 'Direction';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Coarse laser beam direction vector wrt. application coordinate system '
      + '(0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 0.015;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DIRECTION_COARSE_X : Cardinal = 0;
  RDB_RIEGL_DIRECTION_COARSE_Y : Cardinal = 1;
  RDB_RIEGL_DIRECTION_COARSE_Z : Cardinal = 2;

// Laser beam origin wrt. SOCS (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_SHOT_ORIGIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_origin';
    Title              : 'Laser Shot Origin';
    Tags               : 'position';
    Description        : (''
      + 'Laser beam origin wrt. SOCS (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 250.0e-6;
    MinimumValue       : -8.0;
    MaximumValue       : 8.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHOT_ORIGIN_X : Cardinal = 0;
  RDB_RIEGL_SHOT_ORIGIN_Y : Cardinal = 1;
  RDB_RIEGL_SHOT_ORIGIN_Z : Cardinal = 2;

// Laser beam biaxial shift vector wrt. SOCS (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_biaxial_shift';
    Title              : 'Laser Shot Biaxial Shift';
    Tags               : 'direction';
    Description        : (''
      + 'Laser beam biaxial shift vector wrt. SOCS (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 250.0e-6;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT_X : Cardinal = 0;
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT_Y : Cardinal = 1;
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT_Z : Cardinal = 2;

// Laser beam direction vector wrt. SOCS (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_SHOT_DIRECTION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_direction';
    Title              : 'Laser Shot Direction';
    Tags               : 'direction';
    Description        : (''
      + 'Laser beam direction vector wrt. SOCS (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 250.0e-9;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHOT_DIRECTION_X : Cardinal = 0;
  RDB_RIEGL_SHOT_DIRECTION_Y : Cardinal = 1;
  RDB_RIEGL_SHOT_DIRECTION_Z : Cardinal = 2;

// Laser beam direction vector wrt. ROCS (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_direction_levelled';
    Title              : 'Laser Shot Direction Levelled';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Laser beam direction vector wrt. ROCS (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 250.0e-9;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED_X : Cardinal = 0;
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED_Y : Cardinal = 1;
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED_Z : Cardinal = 2;

// Target surface normal vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_SURFACE_NORMAL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.surface_normal';
    Title              : 'Surface Normal';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Target surface normal vector wrt. application coordinate system (0: X,'
      + ' 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SURFACE_NORMAL_X : Cardinal = 0;
  RDB_RIEGL_SURFACE_NORMAL_Y : Cardinal = 1;
  RDB_RIEGL_SURFACE_NORMAL_Z : Cardinal = 2;

// Vector derived from PCA on neighborhood: normal vector for planar and volumetric neighborhoods, direction vector for linear neighborhoods
const
  RDB_RIEGL_POINT_VECTOR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.point_vector';
    Title              : 'Point Vector';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Vector derived from PCA on neighborhood: normal vector for planar and '
      + 'volumetric neighborhoods, direction vector for linear neighborhoods'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 0.0005;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_POINT_VECTOR_X : Cardinal = 0;
  RDB_RIEGL_POINT_VECTOR_Y : Cardinal = 1;
  RDB_RIEGL_POINT_VECTOR_Z : Cardinal = 2;

// Direction vector of shorter edge of plane patch wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_PLANE_UP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_up';
    Title              : 'Plane Patch Up Vector';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Direction vector of shorter edge of plane patch wrt. application '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_UP_X : Cardinal = 0;
  RDB_RIEGL_PLANE_UP_Y : Cardinal = 1;
  RDB_RIEGL_PLANE_UP_Z : Cardinal = 2;

// Vector connecting the center point of a plane patch with its center of gravity (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_PLANE_COG_LINK : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_cog_link';
    Title              : 'Plane COG Link Vector';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Vector connecting the center point of a plane patch with its center of'
      + ' gravity (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -5000.0;
    MaximumValue       : 5000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_COG_LINK_X : Cardinal = 0;
  RDB_RIEGL_PLANE_COG_LINK_Y : Cardinal = 1;
  RDB_RIEGL_PLANE_COG_LINK_Z : Cardinal = 2;

// Vector interconnecting the centers (riegl.xyz) of two matched plane patches (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_patch_link_vector';
    Title              : 'Plane Patch Link Vector';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Vector interconnecting the centers (riegl.xyz) of two matched plane '
      + 'patches (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -5000.0;
    MaximumValue       : 5000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR_X : Cardinal = 0;
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR_Y : Cardinal = 1;
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR_Z : Cardinal = 2;

// The eigenvector that belongs to the smallest eigenvalue (result of PCA, 0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_PCA_AXIS_MIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pca_axis_min';
    Title              : 'PCA Axis Minimum';
    Tags               : 'direction, transform';
    Description        : (''
      + 'The eigenvector that belongs to the smallest eigenvalue (result of '
      + 'PCA, 0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PCA_AXIS_MIN_X : Cardinal = 0;
  RDB_RIEGL_PCA_AXIS_MIN_Y : Cardinal = 1;
  RDB_RIEGL_PCA_AXIS_MIN_Z : Cardinal = 2;

// The eigenvector that belongs to the greatest eigenvalue (result of PCA, 0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_PCA_AXIS_MAX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pca_axis_max';
    Title              : 'PCA Axis Maximum';
    Tags               : 'direction, transform';
    Description        : (''
      + 'The eigenvector that belongs to the greatest eigenvalue (result of '
      + 'PCA, 0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PCA_AXIS_MAX_X : Cardinal = 0;
  RDB_RIEGL_PCA_AXIS_MAX_Y : Cardinal = 1;
  RDB_RIEGL_PCA_AXIS_MAX_Z : Cardinal = 2;

// Direction of x-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_MODEL_CS_AXIS_X : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.model_cs_axis_x';
    Title              : 'Model CS X-Axis';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Direction of x-axis of model coordinate system wrt. application '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_MODEL_CS_AXIS_X_X : Cardinal = 0;
  RDB_RIEGL_MODEL_CS_AXIS_X_Y : Cardinal = 1;
  RDB_RIEGL_MODEL_CS_AXIS_X_Z : Cardinal = 2;

// Direction of y-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_MODEL_CS_AXIS_Y : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.model_cs_axis_y';
    Title              : 'Model CS Y-Axis';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Direction of y-axis of model coordinate system wrt. application '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_MODEL_CS_AXIS_Y_X : Cardinal = 0;
  RDB_RIEGL_MODEL_CS_AXIS_Y_Y : Cardinal = 1;
  RDB_RIEGL_MODEL_CS_AXIS_Y_Z : Cardinal = 2;

// Direction of z-axis of model coordinate system wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_MODEL_CS_AXIS_Z : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.model_cs_axis_z';
    Title              : 'Model CS Z-Axis';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Direction of z-axis of model coordinate system wrt. application '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_MODEL_CS_AXIS_Z_X : Cardinal = 0;
  RDB_RIEGL_MODEL_CS_AXIS_Z_Y : Cardinal = 1;
  RDB_RIEGL_MODEL_CS_AXIS_Z_Z : Cardinal = 2;

// Normalized cylinder axis direction vector
const
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_axis_direction';
    Title              : 'Cylinder Axis Direction';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Normalized cylinder axis direction vector'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Processed values of an accelerometer
const
  RDB_RIEGL_ACCELEROMETER : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.accelerometer';
    Title              : 'Accelerometer Values';
    Tags               : 'direction';
    Description        : (''
      + 'Processed values of an accelerometer'
    );
    UnitSymbol         : 'm/s²';
    Length             : 3;
    Resolution         : 5.0e-6;
    MinimumValue       : -300.0;
    MaximumValue       : 300.0;
    DefaultValue       : -300.0;
    InvalidValue       : -300.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Raw measurement values of an accelerometer (unit see metadata riegl.pose_sensors)
const
  RDB_RIEGL_ACCELEROMETER_RAW : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.accelerometer_raw';
    Title              : 'Accelerometer Raw Values';
    Tags               : 'direction';
    Description        : (''
      + 'Raw measurement values of an accelerometer (unit see metadata '
      + 'riegl.pose_sensors)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 1.0;
    MinimumValue       : -32768.0;
    MaximumValue       : 32767.0;
    DefaultValue       : -32768.0;
    InvalidValue       : -32768.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated bias of accelerometer
const
  RDB_RIEGL_ACCELEROMETER_BIAS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.accelerometer_bias';
    Title              : 'Accelerometer Bias';
    Tags               : '';
    Description        : (''
      + 'Estimated bias of accelerometer'
    );
    UnitSymbol         : 'm/s²';
    Length             : 3;
    Resolution         : 1e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated deviation of scale of accelerometer from 1
const
  RDB_RIEGL_ACCELEROMETER_SCALE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.accelerometer_scale';
    Title              : 'Accelerometer Scale';
    Tags               : '';
    Description        : (''
      + 'Estimated deviation of scale of accelerometer from 1'
    );
    UnitSymbol         : '1';
    Length             : 3;
    Resolution         : 1e-06;
    MinimumValue       : -0.1;
    MaximumValue       : 0.1;
    DefaultValue       : 0.0;
    InvalidValue       : -0.1;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Processed values of a gyroscope
const
  RDB_RIEGL_GYROSCOPE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.gyroscope';
    Title              : 'Gyroscope Values';
    Tags               : 'direction';
    Description        : (''
      + 'Processed values of a gyroscope'
    );
    UnitSymbol         : 'deg/s';
    Length             : 3;
    Resolution         : 1.0e-7;
    MinimumValue       : -2000.0;
    MaximumValue       : 2000.0;
    DefaultValue       : -2000.0;
    InvalidValue       : -2000.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Raw measurement values of a gyroscope (unit see metadata riegl.pose_sensors)
const
  RDB_RIEGL_GYROSCOPE_RAW : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.gyroscope_raw';
    Title              : 'Gyroscope Raw Values';
    Tags               : 'direction';
    Description        : (''
      + 'Raw measurement values of a gyroscope (unit see metadata '
      + 'riegl.pose_sensors)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 1.0;
    MinimumValue       : -32768.0;
    MaximumValue       : 32767.0;
    DefaultValue       : -32768.0;
    InvalidValue       : -32768.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated bias of gyroscope
const
  RDB_RIEGL_GYROSCOPE_BIAS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.gyroscope_bias';
    Title              : 'Gyroscope Bias';
    Tags               : '';
    Description        : (''
      + 'Estimated bias of gyroscope'
    );
    UnitSymbol         : 'deg/s';
    Length             : 3;
    Resolution         : 1e-06;
    MinimumValue       : -10.0;
    MaximumValue       : 10.0;
    DefaultValue       : 0.0;
    InvalidValue       : -10.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated deviation of scale of gyroscope from 1
const
  RDB_RIEGL_GYROSCOPE_SCALE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.gyroscope_scale';
    Title              : 'Gyroscope Scale';
    Tags               : '';
    Description        : (''
      + 'Estimated deviation of scale of gyroscope from 1'
    );
    UnitSymbol         : '1';
    Length             : 3;
    Resolution         : 1e-06;
    MinimumValue       : -0.1;
    MaximumValue       : 0.1;
    DefaultValue       : 0.0;
    InvalidValue       : -0.1;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Processed values of a magnetic field sensor
const
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.magnetic_field_sensor';
    Title              : 'Magnetic Field Sensor Values';
    Tags               : 'direction';
    Description        : (''
      + 'Processed values of a magnetic field sensor'
    );
    UnitSymbol         : 'µT';
    Length             : 3;
    Resolution         : 0.01;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : -1000.0;
    InvalidValue       : -1000.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Raw measurement values of a magnetic field sensor (unit see metadata riegl.pose_sensors)
const
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.magnetic_field_sensor_raw';
    Title              : 'Magnetic Field Sensor Raw Values';
    Tags               : 'direction';
    Description        : (''
      + 'Raw measurement values of a magnetic field sensor (unit see metadata '
      + 'riegl.pose_sensors)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 1.0;
    MinimumValue       : -32768.0;
    MaximumValue       : 32767.0;
    DefaultValue       : -32768.0;
    InvalidValue       : -32768.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Geodetic (Geographic) Latitude
const
  RDB_RIEGL_POF_LATITUDE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_latitude';
    Title              : 'Latitude';
    Tags               : '';
    Description        : (''
      + 'Geodetic (Geographic) Latitude'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-9;
    MinimumValue       : -90.0;
    MaximumValue       : 90.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_POF_LATITUDE_NORTH : TRDBFloat64 = 90.0;
  RDB_RIEGL_POF_LATITUDE_SOUTH : TRDBFloat64 = -90.0;

// Geodetic (Geographic) Longitude
const
  RDB_RIEGL_POF_LONGITUDE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_longitude';
    Title              : 'Longitude';
    Tags               : '';
    Description        : (''
      + 'Geodetic (Geographic) Longitude'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-9;
    MinimumValue       : -180.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_POF_LONGITUDE_EAST : TRDBFloat64 = 180.0;
  RDB_RIEGL_POF_LONGITUDE_WEST : TRDBFloat64 = -180.0;

// Height (vertical distance) wrt. to the ellipsoid of the defined geodetic datum
const
  RDB_RIEGL_POF_HEIGHT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_height';
    Title              : 'Ellipsoidal Height';
    Tags               : '';
    Description        : (''
      + 'Height (vertical distance) wrt. to the ellipsoid of the defined '
      + 'geodetic datum'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 250.0e-6;
    MinimumValue       : -10000.0;
    MaximumValue       : 40000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Height of the geoid relative to the ellipsoid
const
  RDB_RIEGL_POF_UNDULATION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_undulation';
    Title              : 'Geoid Undulation';
    Tags               : '';
    Description        : (''
      + 'Height of the geoid relative to the ellipsoid'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -300.0;
    MaximumValue       : 300.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation about the body axis pointing in forward direction (x-axis for NED systems)
const
  RDB_RIEGL_POF_ROLL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_roll';
    Title              : 'Roll Angle';
    Tags               : '';
    Description        : (''
      + 'Rotation about the body axis pointing in forward direction (x-axis for'
      + ' NED systems)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -180.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation about the body axis pointing in transverse direction (y-axis for NED systems)
const
  RDB_RIEGL_POF_PITCH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_pitch';
    Title              : 'Pitch Angle';
    Tags               : '';
    Description        : (''
      + 'Rotation about the body axis pointing in transverse direction (y-axis '
      + 'for NED systems)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -180.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation about the body axis pointing in vertical direction (z-axis for NED systems)
const
  RDB_RIEGL_POF_YAW : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_yaw';
    Title              : 'Yaw Angle';
    Tags               : '';
    Description        : (''
      + 'Rotation about the body axis pointing in vertical direction (z-axis '
      + 'for NED systems)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 2D angle of the course over ground (0 deg: north, 90 deg: east)
const
  RDB_RIEGL_POF_COURSE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_course';
    Title              : 'Course Angle';
    Tags               : '';
    Description        : (''
      + '2D angle of the course over ground (0 deg: north, 90 deg: east)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -1.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Position of the platform in the coordinate system defined by the geo_tag (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_POF_XYZ : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_xyz';
    Title              : 'XYZ';
    Tags               : 'position, transform';
    Description        : (''
      + 'Position of the platform in the coordinate system defined by the '
      + 'geo_tag (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -535000.0;
    MaximumValue       : 535000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_POF_XYZ_X : Cardinal = 0;
  RDB_RIEGL_POF_XYZ_Y : Cardinal = 1;
  RDB_RIEGL_POF_XYZ_Z : Cardinal = 2;

// Rotation about y-axis of the coordinate system defined by the geo_tag
const
  RDB_RIEGL_POF_ROLL_NED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_roll_ned';
    Title              : 'Roll Angle NED';
    Tags               : '';
    Description        : (''
      + 'Rotation about y-axis of the coordinate system defined by the geo_tag'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -180.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation about x-axis of the coordinate system defined by the geo_tag
const
  RDB_RIEGL_POF_PITCH_NED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_pitch_ned';
    Title              : 'Pitch Angle NED';
    Tags               : '';
    Description        : (''
      + 'Rotation about x-axis of the coordinate system defined by the geo_tag'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -180.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation about inverted z-axis of the coordinate system defined by the geo_tag
const
  RDB_RIEGL_POF_YAW_NED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_yaw_ned';
    Title              : 'Yaw Angle NED';
    Tags               : '';
    Description        : (''
      + 'Rotation about inverted z-axis of the coordinate system defined by the'
      + ' geo_tag'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Time-smoothed velocity vector of the platform in the body coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_POF_XYZ_VELOCITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_xyz_velocity';
    Title              : 'XYZ Velocity';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Time-smoothed velocity vector of the platform in the body coordinate '
      + 'system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm/s';
    Length             : 3;
    Resolution         : 0.01;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Time-smoothed velocity vector of the platform in the local NED system (0: N, 1: E, 2: D)
const
  RDB_RIEGL_POF_XYZ_NED_VELOCITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_xyz_ned_velocity';
    Title              : 'XYZ NED Velocity';
    Tags               : 'direction';
    Description        : (''
      + 'Time-smoothed velocity vector of the platform in the local NED system '
      + '(0: N, 1: E, 2: D)'
    );
    UnitSymbol         : 'm/s';
    Length             : 3;
    Resolution         : 0.01;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Time-smoothed velocity vector of the platform wrt. levelled vehicle coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_xyz_lvcs_velocity';
    Title              : 'XYZ LVCS Velocity';
    Tags               : 'direction';
    Description        : (''
      + 'Time-smoothed velocity vector of the platform wrt. levelled vehicle '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm/s';
    Length             : 3;
    Resolution         : 0.01;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation velocity about x-axis of the body coordinate system
const
  RDB_RIEGL_POF_ROLL_VELOCITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_roll_velocity';
    Title              : 'Roll Angle Velocity';
    Tags               : '';
    Description        : (''
      + 'Rotation velocity about x-axis of the body coordinate system'
    );
    UnitSymbol         : 'deg/s';
    Length             : 1;
    Resolution         : 0.05;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation velocity about y-axis of the body coordinate system
const
  RDB_RIEGL_POF_PITCH_VELOCITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_pitch_velocity';
    Title              : 'Pitch Angle Velocity';
    Tags               : '';
    Description        : (''
      + 'Rotation velocity about y-axis of the body coordinate system'
    );
    UnitSymbol         : 'deg/s';
    Length             : 1;
    Resolution         : 0.05;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Rotation velocity about z-axis of the body coordinate system
const
  RDB_RIEGL_POF_YAW_VELOCITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_yaw_velocity';
    Title              : 'Yaw Angle Velocity';
    Tags               : '';
    Description        : (''
      + 'Rotation velocity about z-axis of the body coordinate system'
    );
    UnitSymbol         : 'deg/s';
    Length             : 1;
    Resolution         : 0.05;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Velocity vector in the local NED system (0: N, 1: E, 2: D)
const
  RDB_RIEGL_POF_VELOCITY_NED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_ned';
    Title              : 'Velocity NED';
    Tags               : 'direction';
    Description        : (''
      + 'Velocity vector in the local NED system (0: N, 1: E, 2: D)'
    );
    UnitSymbol         : 'm/s';
    Length             : 3;
    Resolution         : 0.001;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Velocity in North direction (standard deviation)
const
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_accuracy_north';
    Title              : 'Velocity Accuracy North';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Velocity in North direction (standard deviation)'
    );
    UnitSymbol         : 'm/s';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Velocity in East direction (standard deviation)
const
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_accuracy_east';
    Title              : 'Velocity Accuracy East';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Velocity in East direction (standard deviation)'
    );
    UnitSymbol         : 'm/s';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Velocity in Down direction (standard deviation)
const
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_accuracy_down';
    Title              : 'Velocity Accuracy Down';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Velocity in Down direction (standard deviation)'
    );
    UnitSymbol         : 'm/s';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Velocity in North/East direction (signed square root of covariance)
const
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_accuracy_north_east';
    Title              : 'Velocity Accuracy North/East';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Velocity in North/East direction (signed square root of '
      + 'covariance)'
    );
    UnitSymbol         : 'm/s';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Velocity in East/Down direction (signed square root of covariance)
const
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_accuracy_east_down';
    Title              : 'Velocity Accuracy East/Down';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Velocity in East/Down direction (signed square root of '
      + 'covariance)'
    );
    UnitSymbol         : 'm/s';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Velocity in Down/North direction (signed square root of covariance)
const
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_velocity_accuracy_down_north';
    Title              : 'Velocity Accuracy Down/North';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Velocity in Down/North direction (signed square root of '
      + 'covariance)'
    );
    UnitSymbol         : 'm/s';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -200.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Water surface intersection point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_HYDRO_INTERSECTION_POINT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.hydro_intersection_point';
    Title              : 'Water Surface Intersection Point';
    Tags               : 'position, transform';
    Description        : (''
      + 'Water surface intersection point coordinates wrt. application '
      + 'coordinate system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : -535000.0;
    MaximumValue       : 535000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_HYDRO_INTERSECTION_POINT_X : Cardinal = 0;
  RDB_RIEGL_HYDRO_INTERSECTION_POINT_Y : Cardinal = 1;
  RDB_RIEGL_HYDRO_INTERSECTION_POINT_Z : Cardinal = 2;

// Water surface intersection normal vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.hydro_intersection_normal';
    Title              : 'Water Surface Intersection Normal';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Water surface intersection normal vector wrt. application coordinate '
      + 'system (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL_X : Cardinal = 0;
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL_Y : Cardinal = 1;
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL_Z : Cardinal = 2;

// Water surface model uncertainty (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.hydro_wsm_uncertainty';
    Title              : 'Water Surface Model Uncertainty';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Water surface model uncertainty (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.0001;
    MinimumValue       : -3.2768;
    MaximumValue       : 3.2767;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY_X : Cardinal = 0;
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY_Y : Cardinal = 1;
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY_Z : Cardinal = 2;

// Accuracy of Cartesian coordinates (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_XYZ_ACCURACIES : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.xyz_accuracies';
    Title              : 'XYZ Accuracies';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Accuracy of Cartesian coordinates (0: X, 1: Y, 2: Z)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Direction of zenith vector wrt. application coordinate system (0: X, 1: Y, 2: Z)
const
  RDB_RIEGL_ZENITH_VECTOR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.zenith_vector';
    Title              : 'Zenith Vector';
    Tags               : 'direction, transform';
    Description        : (''
      + 'Direction of zenith vector wrt. application coordinate system (0: X, '
      + '1: Y, 2: Z)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 3.125e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Time"
//______________________________________________________________________________
//

// Laser shot timestamp (in units of 4 pico seconds = 4e-12 s)
const
  RDB_RIEGL_SHOT_TIMESTAMP_HR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_timestamp_hr';
    Title              : 'Laser Shot Timestamp';
    Tags               : 'timestamp';
    Description        : (''
      + 'Laser shot timestamp (in units of 4 pico seconds = 4e-12 s)'
    );
    UnitSymbol         : '4ps';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e18;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHOT_TIMESTAMP_HR_RESOLUTION : TRDBFloat64 = 4.0e-12;

// Laser shot timestamp (100 nano seconds resolution)
const
  RDB_RIEGL_TIMESTAMP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.timestamp';
    Title              : 'Timestamp';
    Tags               : 'timestamp';
    Description        : (''
      + 'Laser shot timestamp (100 nano seconds resolution)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 1.0e-7;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e8;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Start of waveform sample block relative to laser shot timestamp (in units of 4 pico seconds = 4e-12 s)
const
  RDB_RIEGL_WFM_SBL_TIME_OFFSET : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_time_offset';
    Title              : 'Waveform Sample Block Time Offset';
    Tags               : '';
    Description        : (''
      + 'Start of waveform sample block relative to laser shot timestamp (in '
      + 'units of 4 pico seconds = 4e-12 s)'
    );
    UnitSymbol         : '4ps';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -2.0e9;
    MaximumValue       : 2.0e9;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WFM_SBL_TIME_OFFSET_RESOLUTION : TRDBFloat64 = 4.0e-12;

// Position of echo relative to start of waveform sample block (in units of 4 pico seconds = 4e-12 s)
const
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_echo_time_offset';
    Title              : 'Echo Time Offset';
    Tags               : '';
    Description        : (''
      + 'Position of echo relative to start of waveform sample block (in units '
      + 'of 4 pico seconds = 4e-12 s)'
    );
    UnitSymbol         : '4ps';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -2.0e9;
    MaximumValue       : 2.0e9;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET_RESOLUTION : TRDBFloat64 = 4.0e-12;

// External timestamp (100 nano seconds resolution)
const
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pps_timestamp_extern';
    Title              : 'External Timestamp';
    Tags               : 'timestamp';
    Description        : (''
      + 'External timestamp (100 nano seconds resolution)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 1.0e-7;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e8;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Internal timestamp (in units of 4 pico seconds = 4e-12 s)
const
  RDB_RIEGL_PPS_TIMESTAMP_INTERN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pps_timestamp_intern';
    Title              : 'Internal Timestamp';
    Tags               : '';
    Description        : (''
      + 'Internal timestamp (in units of 4 pico seconds = 4e-12 s)'
    );
    UnitSymbol         : '4ps';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e18;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PPS_TIMESTAMP_INTERN_RESOLUTION : TRDBFloat64 = 4.0e-12;

// Timestamp of event (seconds since epoch defined in riegl.time_base)
const
  RDB_RIEGL_EVENT_TIMESTAMP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.event_timestamp';
    Title              : 'Event Timestamp';
    Tags               : 'timestamp';
    Description        : (''
      + 'Timestamp of event (seconds since epoch defined in riegl.time_base)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 1e-07;
    MinimumValue       : 0.0;
    MaximumValue       : 900000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Minimum laser shot timestamp within voxel (100 nano seconds resolution)
const
  RDB_RIEGL_TIMESTAMP_MIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.timestamp_min';
    Title              : 'Timestamp Minimum';
    Tags               : 'timestamp';
    Description        : (''
      + 'Minimum laser shot timestamp within voxel (100 nano seconds '
      + 'resolution)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 1.0e-7;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e8;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Maximum laser shot timestamp within voxel (100 nano seconds resolution)
const
  RDB_RIEGL_TIMESTAMP_MAX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.timestamp_max';
    Title              : 'Timestamp Maximum';
    Tags               : 'timestamp';
    Description        : (''
      + 'Maximum laser shot timestamp within voxel (100 nano seconds '
      + 'resolution)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 1.0e-7;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e8;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Timestamp (100 nano seconds resolution)
const
  RDB_RIEGL_POF_TIMESTAMP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_timestamp';
    Title              : 'Timestamp';
    Tags               : 'timestamp';
    Description        : (''
      + 'Timestamp (100 nano seconds resolution)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 1.0e-7;
    MinimumValue       : 0.0;
    MaximumValue       : 9.0e8;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Date of acquisition (0: year, 1: month [1-12], 2: day [1-31], not specified if any is 0)
const
  RDB_RIEGL_ACQUISITION_DATE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.acquisition_date';
    Title              : 'Acquisition Date';
    Tags               : '';
    Description        : (''
      + 'Date of acquisition (0: year, 1: month [1-12], 2: day [1-31], not '
      + 'specified if any is 0)'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4095.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_ACQUISITION_DATE_YEAR : Cardinal = 0;
  RDB_RIEGL_ACQUISITION_DATE_MONTH : Cardinal = 1;
  RDB_RIEGL_ACQUISITION_DATE_DAY : Cardinal = 2;

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Primary Attributes"
//______________________________________________________________________________
//

// Target surface reflectance
const
  RDB_RIEGL_REFLECTANCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.reflectance';
    Title              : 'Reflectance';
    Tags               : '';
    Description        : (''
      + 'Target surface reflectance'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -327.68;
    MaximumValue       : 327.67;
    DefaultValue       : 0.0;
    InvalidValue       : -327.68;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Echo signal amplitude
const
  RDB_RIEGL_AMPLITUDE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.amplitude';
    Title              : 'Amplitude';
    Tags               : '';
    Description        : (''
      + 'Echo signal amplitude'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -327.68;
    MaximumValue       : 327.67;
    DefaultValue       : 0.0;
    InvalidValue       : -327.68;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Uncalibrated echo signal amplitude (for legacy RIEGL LMS instruments)
const
  RDB_RIEGL_INTENSITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.intensity';
    Title              : 'Intensity';
    Tags               : '';
    Description        : (''
      + 'Uncalibrated echo signal amplitude (for legacy RIEGL LMS instruments)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Gain of photodiode
const
  RDB_RIEGL_GAIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.gain';
    Title              : 'Gain';
    Tags               : '';
    Description        : (''
      + 'Gain of photodiode'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -327.68;
    MaximumValue       : 327.67;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Pulse shape deviation (negative means unavailable)
const
  RDB_RIEGL_DEVIATION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.deviation';
    Title              : 'Deviation';
    Tags               : '';
    Description        : (''
      + 'Pulse shape deviation (negative means unavailable)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 32767.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DEVIATION_UNAVAILABLE : TRDBInt16 = -1;

// Pulse width (full width at half maximum, for Q-Series instruments, 0 = invalid)
const
  RDB_RIEGL_PULSE_WIDTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pulse_width';
    Title              : 'Pulse Width';
    Tags               : '';
    Description        : (''
      + 'Pulse width (full width at half maximum, for Q-Series instruments, 0 ='
      + ' invalid)'
    );
    UnitSymbol         : 'ns';
    Length             : 1;
    Resolution         : 0.1;
    MinimumValue       : 0.0;
    MaximumValue       : 6553.5;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PULSE_WIDTH_INVALID : TRDBFloat64 = 0.0;

// Point class number
const
  RDB_RIEGL_CLASS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.class';
    Title              : 'Point Class';
    Tags               : 'enumeration';
    Description        : (''
      + 'Point class number'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=Created, never classified'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Point color derived from digital camera, 0: Red, 1: Green, 2: Blue, 3: Alpha (0 = no color); additional colors (if any) are stored in riegl.rgba_2, riegl.rgba_3, ...
const
  RDB_RIEGL_RGBA : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.rgba';
    Title              : 'True Color';
    Tags               : 'color';
    Description        : (''
      + 'Point color derived from digital camera, 0: Red, 1: Green, 2: Blue, 3:'
      + ' Alpha (0 = no color); additional colors (if any) are stored in '
      + 'riegl.rgba_2, riegl.rgba_3, ...'
    );
    UnitSymbol         : '';
    Length             : 4;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_RGBA_RED : Cardinal = 0;
  RDB_RIEGL_RGBA_GREEN : Cardinal = 1;
  RDB_RIEGL_RGBA_BLUE : Cardinal = 2;
  RDB_RIEGL_RGBA_ALPHA : Cardinal = 3;

// Point near infrared (NIR) value derived from NIR camera
const
  RDB_RIEGL_NIR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nir';
    Title              : 'Near Infrared Brightness';
    Tags               : '';
    Description        : (''
      + 'Point near infrared (NIR) value derived from NIR camera'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Target temperature measured by thermal camera
const
  RDB_RIEGL_TEMPERATURE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.temperature';
    Title              : 'Temperature';
    Tags               : '';
    Description        : (''
      + 'Target temperature measured by thermal camera'
    );
    UnitSymbol         : '°C';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -273.15;
    MaximumValue       : 4.0e6;
    DefaultValue       : 4.0e6;
    InvalidValue       : 4.0e6;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// DMI counter value (unwrapped)
const
  RDB_RIEGL_DMI_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.dmi_count';
    Title              : 'DMI Counter Value';
    Tags               : '';
    Description        : (''
      + 'DMI counter value (unwrapped)'
    );
    UnitSymbol         : '1';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -100000000000.0;
    MaximumValue       : 100000000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// DMI path length (uncalibrated)
const
  RDB_RIEGL_DMI_PATH_LENGTH_RAW : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.dmi_path_length_raw';
    Title              : 'DMI Path Length';
    Tags               : '';
    Description        : (''
      + 'DMI path length (uncalibrated)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -10000000.0;
    MaximumValue       : 10000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated deviation of scale of DMI from 1 with respect to the riegl.dmi_path_length_raw attribute in the dmix file
const
  RDB_RIEGL_DMI_SCALE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.dmi_scale';
    Title              : 'DMI Scale';
    Tags               : '';
    Description        : (''
      + 'Estimated deviation of scale of DMI from 1 with respect to the '
      + 'riegl.dmi_path_length_raw attribute in the dmix file'
    );
    UnitSymbol         : '1';
    Length             : 1;
    Resolution         : 1e-05;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Index of assigned MTA zone. Nearest MTA zone has index 1
const
  RDB_RIEGL_MTA_ZONE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.mta_zone';
    Title              : 'MTA Zone Assigned';
    Tags               : 'enumeration';
    Description        : (''
      + 'Index of assigned MTA zone. Nearest MTA zone has index 1'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 1.0;
    MaximumValue       : 255.0;
    DefaultValue       : 1.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 1 for points with uncertain MTA zone assignment, 0 for certainly assigned MTA zone
const
  RDB_RIEGL_MTA_UNCERTAIN_POINT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.mta_uncertain_point';
    Title              : 'MTA Uncertain Point';
    Tags               : 'boolean';
    Description        : (''
      + '1 for points with uncertain MTA zone assignment, 0 for certainly '
      + 'assigned MTA zone'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_MTA_UNCERTAIN_POINT_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_MTA_UNCERTAIN_POINT_TRUE : TRDBUInt8 = 1;

// 1 for all points originating from a full waveform analysis (FWA), 0 for all points originating from online waveform processing (OWP)
const
  RDB_RIEGL_FWA : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.fwa';
    Title              : 'Full Waveform Analysis';
    Tags               : 'boolean';
    Description        : (''
      + '1 for all points originating from a full waveform analysis (FWA), 0 '
      + 'for all points originating from online waveform processing (OWP)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=Origin OWP'#10'1=Origin FWA'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_FWA_ONLINE_WAVEFORM_PROCESSING : TRDBUInt8 = 0;
  RDB_RIEGL_FWA_FULL_WAVEFORM_ANALYSIS : TRDBUInt8 = 1;

// Background Radiation (for VZ-400-HT/HAT only)
const
  RDB_RIEGL_BACKGROUND_RADIATION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.background_radiation';
    Title              : 'Background Radiation';
    Tags               : '';
    Description        : (''
      + 'Background Radiation (for VZ-400-HT/HAT only)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Background Radiation (for VZ-400-HTo only, Si-PD)
const
  RDB_RIEGL_BACKGROUND_RADIATION_SI : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.background_radiation_si';
    Title              : 'Background Radiation Si-PD';
    Tags               : '';
    Description        : (''
      + 'Background Radiation (for VZ-400-HTo only, Si-PD)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : 65535.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Background Radiation (for VZ-400-HTo only, InGaAs-PD)
const
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.background_radiation_ingaas';
    Title              : 'Background Radiation InGaAs-PD';
    Tags               : '';
    Description        : (''
      + 'Background Radiation (for VZ-400-HTo only, InGaAs-PD)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : 65535.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated temperature derived from Background Radiation Si-PD
const
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.temperature_estimated_si';
    Title              : 'Temperature Estimated Si-PD';
    Tags               : '';
    Description        : (''
      + 'Estimated temperature derived from Background Radiation Si-PD'
    );
    UnitSymbol         : '°C';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 2000.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated temperature derived from Background Radiation InGaAs-PD
const
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.temperature_estimated_ingaas';
    Title              : 'Temperature Estimated InGaAs-PD';
    Tags               : '';
    Description        : (''
      + 'Estimated temperature derived from Background Radiation InGaAs-PD'
    );
    UnitSymbol         : '°C';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 2000.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Estimated temperature derived from difference in Background Radiation InGaAs and Si
const
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.temperature_estimated_ingaas_si';
    Title              : 'Temperature Estimated InGaAs/Si-PD';
    Tags               : '';
    Description        : (''
      + 'Estimated temperature derived from difference in Background Radiation '
      + 'InGaAs and Si'
    );
    UnitSymbol         : '°C';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 2000.0;
    DefaultValue       : 0.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// nth target of a laser-shot (0 = unknown, 1 = first target, ...)
const
  RDB_RIEGL_TARGET_INDEX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.target_index';
    Title              : 'Target Index';
    Tags               : '';
    Description        : (''
      + 'nth target of a laser-shot (0 = unknown, 1 = first target, ...)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 1.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of targets of a laser-shot (0 = unknown, 1 = one target, ...)
const
  RDB_RIEGL_TARGET_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.target_count';
    Title              : 'Target Count';
    Tags               : '';
    Description        : (''
      + 'Number of targets of a laser-shot (0 = unknown, 1 = one target, ...)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 1.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Target rank among all targets of a laser-shot (0 = unknown, 1 = single, 2 = first, 3 = intermediate, 4 = last target)
const
  RDB_RIEGL_TARGET_TYPE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.target_type';
    Title              : 'Target Type';
    Tags               : 'enumeration';
    Description        : (''
      + 'Target rank among all targets of a laser-shot (0 = unknown, 1 = '
      + 'single, 2 = first, 3 = intermediate, 4 = last target)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '1=Single target'#10'2=First target'#10'3=Intermediate target'#10'4=Last target'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_TARGET_TYPE_UNKNOWN : TRDBUInt8 = 0;
  RDB_RIEGL_TARGET_TYPE_SINGLE : TRDBUInt8 = 1;
  RDB_RIEGL_TARGET_TYPE_FIRST : TRDBUInt8 = 2;
  RDB_RIEGL_TARGET_TYPE_INTERMEDIATE : TRDBUInt8 = 3;
  RDB_RIEGL_TARGET_TYPE_LAST : TRDBUInt8 = 4;

// Identifier of first echo that belongs to the laser shot (0 = invalid). This is not an array index but the value of riegl.id of the echo.
const
  RDB_RIEGL_ECHO_FIRST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.echo_first';
    Title              : 'Echo First';
    Tags               : '';
    Description        : (''
      + 'Identifier of first echo that belongs to the laser shot (0 = invalid).'
      + ' This is not an array index but the value of riegl.id of the echo.'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_ECHO_FIRST_INVALID : TRDBUInt64 = 0;

// Number of echoes that belong to the laser shot
const
  RDB_RIEGL_ECHO_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.echo_count';
    Title              : 'Echo Count';
    Tags               : '';
    Description        : (''
      + 'Number of echoes that belong to the laser shot'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Height at center of grid cell
const
  RDB_RIEGL_HEIGHT_CENTER : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.height_center';
    Title              : 'Height Center';
    Tags               : '';
    Description        : (''
      + 'Height at center of grid cell'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -100000.0;
    MaximumValue       : 100000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Mean height within grid cell
const
  RDB_RIEGL_HEIGHT_MEAN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.height_mean';
    Title              : 'Height Mean';
    Tags               : '';
    Description        : (''
      + 'Mean height within grid cell'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -100000.0;
    MaximumValue       : 100000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Minimum height within grid cell
const
  RDB_RIEGL_HEIGHT_MIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.height_min';
    Title              : 'Height Minimum';
    Tags               : '';
    Description        : (''
      + 'Minimum height within grid cell'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -100000.0;
    MaximumValue       : 100000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Maximum height within grid cell
const
  RDB_RIEGL_HEIGHT_MAX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.height_max';
    Title              : 'Height Maximum';
    Tags               : '';
    Description        : (''
      + 'Maximum height within grid cell'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -100000.0;
    MaximumValue       : 100000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of points this point represents (e.g. points combined to voxels or plane patches, 0 = unknown)
const
  RDB_RIEGL_POINT_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.point_count';
    Title              : 'Point Count';
    Tags               : '';
    Description        : (''
      + 'Number of points this point represents (e.g. points combined to voxels'
      + ' or plane patches, 0 = unknown)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4294967295.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_POINT_COUNT_UNKNOWN : TRDBUInt32 = 0;

// Number of total points in a grid cell this point represents.
const
  RDB_RIEGL_POINT_COUNT_GRID_CELL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.point_count_grid_cell';
    Title              : 'Point Count of Grid Cell';
    Tags               : '';
    Description        : (''
      + 'Number of total points in a grid cell this point represents.'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4294967295.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_POINT_COUNT_GRID_CELL_UNKNOWN : TRDBUInt32 = 0;

// Volume extents along 0: riegl.pca_axis_max, 1: riegl.pca_axis_min x riegl.pca_axis_max, 2: riegl.pca_axis_min
const
  RDB_RIEGL_PCA_EXTENTS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pca_extents';
    Title              : 'PCA Extents';
    Tags               : '';
    Description        : (''
      + 'Volume extents along 0: riegl.pca_axis_max, 1: riegl.pca_axis_min x '
      + 'riegl.pca_axis_max, 2: riegl.pca_axis_min'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Representative volume of the neighborhood, calculated as product of the components of riegl.pca_extents
const
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.mta_uncertain_point_vol';
    Title              : 'MTA Uncertain Point Volume';
    Tags               : '';
    Description        : (''
      + 'Representative volume of the neighborhood, calculated as product of '
      + 'the components of riegl.pca_extents'
    );
    UnitSymbol         : 'dBm³';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -120.0;
    MaximumValue       : 120.0;
    DefaultValue       : 0.0;
    InvalidValue       : -120.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Difference in representative volume of the neighborhood to next smallest volume of point from the same echo but different MTA-zone
const
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.mta_uncertain_point_vol_diff';
    Title              : 'Difference in MTA Uncertain Point Volume';
    Tags               : '';
    Description        : (''
      + 'Difference in representative volume of the neighborhood to next '
      + 'smallest volume of point from the same echo but different MTA-zone'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 240.0;
    DefaultValue       : 240.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Volume extents along riegl.pca_axis_min or riegl.surface_normal (result of PCA)
const
  RDB_RIEGL_PCA_THICKNESS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pca_thickness';
    Title              : 'PCA Thickness';
    Tags               : '';
    Description        : (''
      + 'Volume extents along riegl.pca_axis_min or riegl.surface_normal '
      + '(result of PCA)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Representative diameter of the neighborhood, calculated as sqrt(12*sum_of_eigenvalues/number_of_points)
const
  RDB_RIEGL_PCA_DIAMETER : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pca_diameter';
    Title              : 'PCA Diameter of Neighborhood';
    Tags               : '';
    Description        : (''
      + 'Representative diameter of the neighborhood, calculated as '
      + 'sqrt(12*sum_of_eigenvalues/number_of_points)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// The smallest eigenvalue from PCA normalized by the sum of all eigenvalues
const
  RDB_RIEGL_EIGENVALUE_NORM_MIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.eigenvalue_norm_min';
    Title              : 'PCA Minimum Eigen Value Normalized';
    Tags               : '';
    Description        : (''
      + 'The smallest eigenvalue from PCA normalized by the sum of all '
      + 'eigenvalues'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.52591e-05;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// The largest eigenvalue from PCA normalized by the sum of all eigenvalues
const
  RDB_RIEGL_EIGENVALUE_NORM_MAX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.eigenvalue_norm_max';
    Title              : 'PCA Maximum Eigen Value Normalized';
    Tags               : '';
    Description        : (''
      + 'The largest eigenvalue from PCA normalized by the sum of all '
      + 'eigenvalues'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.52591e-05;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Standard deviation, e.g. of residual point-to-plane distances (0 = unknown)
const
  RDB_RIEGL_STD_DEV : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.std_dev';
    Title              : 'Standard Deviation';
    Tags               : '';
    Description        : (''
      + 'Standard deviation, e.g. of residual point-to-plane distances (0 = '
      + 'unknown)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.0001;
    MinimumValue       : 0.0;
    MaximumValue       : 65.535;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_STD_DEV_UNKNOWN : TRDBFloat64 = 0.0;

// Confidence ellipse for the normal vector of a plane patch (0: up-axis, 1: width-axis, tilt angle wrt. up-axis)
const
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_confidence_normal';
    Title              : 'Confidence Normal Direction';
    Tags               : '';
    Description        : (''
      + 'Confidence ellipse for the normal vector of a plane patch (0: up-axis,'
      + ' 1: width-axis, tilt angle wrt. up-axis)'
    );
    UnitSymbol         : 'deg';
    Length             : 3;
    Resolution         : 0.006;
    MinimumValue       : -180.006;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : -180.006;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Plane patch slope class number. Meaning see metadata riegl.plane_slope_class_info (0 = invalid)
const
  RDB_RIEGL_PLANE_SLOPE_CLASS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_slope_class';
    Title              : 'Plane Patch Slope Class';
    Tags               : 'enumeration';
    Description        : (''
      + 'Plane patch slope class number. Meaning see metadata '
      + 'riegl.plane_slope_class_info (0 = invalid)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 8x8 occupancy matrix indicating point support of plane patch
const
  RDB_RIEGL_PLANE_OCCUPANCY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_occupancy';
    Title              : 'Plane Patch Occupancy';
    Tags               : '';
    Description        : (''
      + '8x8 occupancy matrix indicating point support of plane patch'
    );
    UnitSymbol         : '';
    Length             : 8;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Size of plane patch along the longer edge (0 = unknown)
const
  RDB_RIEGL_PLANE_WIDTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_width';
    Title              : 'Plane Patch Width';
    Tags               : '';
    Description        : (''
      + 'Size of plane patch along the longer edge (0 = unknown)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_WIDTH_UNKNOWN : TRDBFloat64 = 0.0;

// Size of plane patch along the shorter edge (0 = unknown)
const
  RDB_RIEGL_PLANE_HEIGHT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_height';
    Title              : 'Plane Patch Height';
    Tags               : '';
    Description        : (''
      + 'Size of plane patch along the shorter edge (0 = unknown)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_HEIGHT_UNKNOWN : TRDBFloat64 = 0.0;

// For merged plane patches, the number of plane patches the merged plane patch is based on
const
  RDB_RIEGL_PLANE_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_count';
    Title              : 'Plane Patch Count';
    Tags               : '';
    Description        : (''
      + 'For merged plane patches, the number of plane patches the merged plane'
      + ' patch is based on'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 1.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// The number of source plane patch files the plane patch has matches to
const
  RDB_RIEGL_MATCH_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.match_count';
    Title              : 'Plane Patch Match Count';
    Tags               : '';
    Description        : (''
      + 'The number of source plane patch files the plane patch has matches to'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Distance between the origins of two plane patches, projected onto the average of their normal vectors
const
  RDB_RIEGL_PLANE_PATCH_DISTANCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_patch_distance';
    Title              : 'Plane Patch Distance';
    Tags               : '';
    Description        : (''
      + 'Distance between the origins of two plane patches, projected onto the '
      + 'average of their normal vectors'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -2100.0;
    MaximumValue       : 2000.0;
    DefaultValue       : -2100.0;
    InvalidValue       : -2100.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_PATCH_DISTANCE_LOWEST : TRDBFloat64 = -2000.0;

// Distance between the origins of two plane patches, projected onto the plane defined by the average of their normal vectors
const
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_patch_lateral_distance';
    Title              : 'Plane Patch Lateral Distance';
    Tags               : '';
    Description        : (''
      + 'Distance between the origins of two plane patches, projected onto the '
      + 'plane defined by the average of their normal vectors'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -2100.0;
    MaximumValue       : 2000.0;
    DefaultValue       : -2100.0;
    InvalidValue       : -2100.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE_LOWEST : TRDBFloat64 = -2000.0;

// Angle between the normal vectors of two plane patches
const
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_patch_angular_distance';
    Title              : 'Plane Patch Angular Distance';
    Tags               : '';
    Description        : (''
      + 'Angle between the normal vectors of two plane patches'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -1.0;
    MaximumValue       : 180.0;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Radius of cylinder
const
  RDB_RIEGL_CYLINDER_RADIUS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_radius';
    Title              : 'Cylinder Radius';
    Tags               : '';
    Description        : (''
      + 'Radius of cylinder'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 100.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Height of cylinder (cylinder origin is at half height)
const
  RDB_RIEGL_CYLINDER_HEIGHT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_height';
    Title              : 'Cylinder Height';
    Tags               : '';
    Description        : (''
      + 'Height of cylinder (cylinder origin is at half height)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence of estimated cylinder origin (perpendicular to axis)
const
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_origin_confidence';
    Title              : 'Cylinder Origin Confidence';
    Tags               : '';
    Description        : (''
      + 'Confidence of estimated cylinder origin (perpendicular to axis)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.0001;
    MinimumValue       : 0.0;
    MaximumValue       : 10.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence of estimated cylinder radius
const
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_radius_confidence';
    Title              : 'Cylinder Radius Confidence';
    Tags               : '';
    Description        : (''
      + 'Confidence of estimated cylinder radius'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.0001;
    MinimumValue       : 0.0;
    MaximumValue       : 10.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence of estimated cylinder axis direction
const
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_direction_confidence';
    Title              : 'Cylinder Direction Confidence';
    Tags               : '';
    Description        : (''
      + 'Confidence of estimated cylinder axis direction'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.003;
    MinimumValue       : 0.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of points used for cylinder fitting
const
  RDB_RIEGL_CYLINDER_INLIER_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_inlier_count';
    Title              : 'Cylinder Inlier Count';
    Tags               : '';
    Description        : (''
      + 'Number of points used for cylinder fitting'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1;
    MinimumValue       : 0.0;
    MaximumValue       : 16000000;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of points not used for cylinder fitting
const
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cylinder_outlier_count';
    Title              : 'Cylinder Outlier Count';
    Tags               : '';
    Description        : (''
      + 'Number of points not used for cylinder fitting'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1;
    MinimumValue       : 0.0;
    MaximumValue       : 16000000;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy in North direction (standard deviation)
const
  RDB_RIEGL_POF_ACCURACY_NORTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_north';
    Title              : 'Accuracy North';
    Tags               : '';
    Description        : (''
      + 'Accuracy in North direction (standard deviation)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy in East direction (standard deviation)
const
  RDB_RIEGL_POF_ACCURACY_EAST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_east';
    Title              : 'Accuracy East';
    Tags               : '';
    Description        : (''
      + 'Accuracy in East direction (standard deviation)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy in Down direction (standard deviation)
const
  RDB_RIEGL_POF_ACCURACY_DOWN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_down';
    Title              : 'Accuracy Down';
    Tags               : '';
    Description        : (''
      + 'Accuracy in Down direction (standard deviation)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy in North/East direction (signed square root of covariance)
const
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_north_east';
    Title              : 'Accuracy North/East';
    Tags               : '';
    Description        : (''
      + 'Accuracy in North/East direction (signed square root of covariance)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy in East/Down direction (signed square root of covariance)
const
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_east_down';
    Title              : 'Accuracy East/Down';
    Tags               : '';
    Description        : (''
      + 'Accuracy in East/Down direction (signed square root of covariance)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy in Down/North direction (signed square root of covariance)
const
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_down_north';
    Title              : 'Accuracy Down/North';
    Tags               : '';
    Description        : (''
      + 'Accuracy in Down/North direction (signed square root of covariance)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : -1000.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Roll rotation
const
  RDB_RIEGL_POF_ACCURACY_ROLL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_roll';
    Title              : 'Accuracy Roll';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Roll rotation'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Pitch rotation
const
  RDB_RIEGL_POF_ACCURACY_PITCH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_pitch';
    Title              : 'Accuracy Pitch';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Pitch rotation'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Accuracy of Yaw rotation
const
  RDB_RIEGL_POF_ACCURACY_YAW : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_accuracy_yaw';
    Title              : 'Accuracy Yaw';
    Tags               : '';
    Description        : (''
      + 'Accuracy of Yaw rotation'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1.0e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Waveform sample block channel number (255 = invalid)
const
  RDB_RIEGL_WFM_SBL_CHANNEL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_channel';
    Title              : 'Waveform Sample Block Channel';
    Tags               : 'enumeration';
    Description        : (''
      + 'Waveform sample block channel number (255 = invalid)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 255.0;
    InvalidValue       : 255.0;
    NamedValues        : '0=High power'#10'1=Low power'#10'3=Reference pulse'#10'';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WFM_SBL_CHANNEL_INVALID : TRDBUInt8 = 255;

// Waveform sample value mean
const
  RDB_RIEGL_WFM_SBL_MEAN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_mean';
    Title              : 'Waveform Sample Value Mean';
    Tags               : '';
    Description        : (''
      + 'Waveform sample value mean'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 0.0625;
    MinimumValue       : 0.0;
    MaximumValue       : 4095.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Waveform sample value standard deviation
const
  RDB_RIEGL_WFM_SBL_STD_DEV : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_std_dev';
    Title              : 'Waveform Sample Value Standard Deviation';
    Tags               : '';
    Description        : (''
      + 'Waveform sample value standard deviation'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 0.25;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Identifier of first waveform sample block that belongs to the laser shot (0 = invalid). This is not an array index but the value of riegl.id of the block.
const
  RDB_RIEGL_WFM_SBL_FIRST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_first';
    Title              : 'Waveform Sample Block First';
    Tags               : '';
    Description        : (''
      + 'Identifier of first waveform sample block that belongs to the laser '
      + 'shot (0 = invalid). This is not an array index but the value of '
      + 'riegl.id of the block.'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WFM_SBL_FIRST_INVALID : TRDBUInt64 = 0;

// Number of waveform sample blocks that belong to the laser shot
const
  RDB_RIEGL_WFM_SBL_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_count';
    Title              : 'Waveform Sample Block Count';
    Tags               : '';
    Description        : (''
      + 'Number of waveform sample blocks that belong to the laser shot'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Identifier of first waveform sample value that belongs to the sample block (0 = invalid). This is not an array index but the value of riegl.id of the sample.
const
  RDB_RIEGL_WFM_SDA_FIRST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sda_first';
    Title              : 'Waveform Sample Data First';
    Tags               : '';
    Description        : (''
      + 'Identifier of first waveform sample value that belongs to the sample '
      + 'block (0 = invalid). This is not an array index but the value of '
      + 'riegl.id of the sample.'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WFM_SDA_FIRST_INVALID : TRDBUInt64 = 0;

// Number of waveform sample values that belong to the sample block
const
  RDB_RIEGL_WFM_SDA_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sda_count';
    Title              : 'Waveform Sample Data Count';
    Tags               : '';
    Description        : (''
      + 'Number of waveform sample values that belong to the sample block'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Raw waveform sample value
const
  RDB_RIEGL_WFM_SAMPLE_VALUE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sample_value';
    Title              : 'Waveform Sample Value';
    Tags               : '';
    Description        : (''
      + 'Raw waveform sample value'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Zero-based index of control object type listed in riegl.control_object_catalog (meta data entry)
const
  RDB_RIEGL_CONTROL_OBJECT_TYPE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.control_object_type';
    Title              : 'Control Object Type';
    Tags               : 'enumeration';
    Description        : (''
      + 'Zero-based index of control object type listed in '
      + 'riegl.control_object_catalog (meta data entry)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 32767.0;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Model fit quality value between 0 and 1 (0 = unspecified)
const
  RDB_RIEGL_MODEL_FIT_QUALITY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.model_fit_quality';
    Title              : 'Model Fit Quality';
    Tags               : '';
    Description        : (''
      + 'Model fit quality value between 0 and 1 (0 = unspecified)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Observed inclination angle of the surface in the vicinity of a control point. It is defined as the arccosine of the vertical component of the surface normal vector. Hence: angle = 0 deg: horizontal (floor); angle = 90 deg: vertical (wall); angle = 180 deg: horizontal (ceiling)
const
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cp_surface_inclination_angle';
    Title              : 'Surface Inclination Angle';
    Tags               : '';
    Description        : (''
      + 'Observed inclination angle of the surface in the vicinity of a control'
      + ' point. It is defined as the arccosine of the vertical component of '
      + 'the surface normal vector. Hence: angle = 0 deg: horizontal (floor); '
      + 'angle = 90 deg: vertical (wall); angle = 180 deg: horizontal (ceiling)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE_FLOOR : TRDBFloat64 = 0.0;
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE_WALL : TRDBFloat64 = 90.0;
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE_CEILING : TRDBFloat64 = 180.0;

// Tolerance angle for the inclination of the surface in the vicinity of a control point
const
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cp_surface_inclination_tolerance_angle';
    Title              : 'Surface Inclination Tolerance Angle';
    Tags               : '';
    Description        : (''
      + 'Tolerance angle for the inclination of the surface in the vicinity of '
      + 'a control point'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 180.0;
    DefaultValue       : 180.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Observed northing angle (azimuth) of the surface in the vicinity of a control point. It is defined as the angle of the horizontal component of the surface normal vector wrt. north direction (measured clockwise). Hence: 0 deg: North; 90 deg: East; 180 deg: South; 270 deg: West
const
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cp_surface_northing_angle';
    Title              : 'Surface Northing Angle';
    Tags               : '';
    Description        : (''
      + 'Observed northing angle (azimuth) of the surface in the vicinity of a '
      + 'control point. It is defined as the angle of the horizontal component '
      + 'of the surface normal vector wrt. north direction (measured '
      + 'clockwise). Hence: 0 deg: North; 90 deg: East; 180 deg: South; 270 '
      + 'deg: West'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Tolerance angle for the northing angle of the surface in the vicinity of a control point
const
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cp_surface_northing_tolerance_angle';
    Title              : 'Surface Northing Tolerance Angle';
    Tags               : '';
    Description        : (''
      + 'Tolerance angle for the northing angle of the surface in the vicinity '
      + 'of a control point'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 180.0;
    DefaultValue       : 180.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence values of observed XYZ point coordinates wrt. local observation coordinate system
const
  RDB_RIEGL_OBS_CONFIDENCE_XYZ : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_confidence_xyz';
    Title              : 'XYZ Confidences of Observed Point';
    Tags               : '';
    Description        : (''
      + 'Confidence values of observed XYZ point coordinates wrt. local '
      + 'observation coordinate system'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence value for xy position of observed point wrt. local observation coordinate system
const
  RDB_RIEGL_OBS_CONFIDENCE_XY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_confidence_xy';
    Title              : 'XY Confidence of Observed Point';
    Tags               : '';
    Description        : (''
      + 'Confidence value for xy position of observed point wrt. local '
      + 'observation coordinate system'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence value for z position of observed point wrt. local observation coordinate system
const
  RDB_RIEGL_OBS_CONFIDENCE_Z : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_confidence_z';
    Title              : 'Z Confidence of Observed Point';
    Tags               : '';
    Description        : (''
      + 'Confidence value for z position of observed point wrt. local '
      + 'observation coordinate system'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence of observation coordinates in local radial direction (range)
const
  RDB_RIEGL_OBS_CONFIDENCE_RANGE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_confidence_range';
    Title              : 'Positional Confidence of Observation in Radial Direction';
    Tags               : '';
    Description        : (''
      + 'Confidence of observation coordinates in local radial direction '
      + '(range)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence of observation coordinates in local elevation direction (theta)
const
  RDB_RIEGL_OBS_CONFIDENCE_THETA : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_confidence_theta';
    Title              : 'Positional Confidence of Observation in Theta Direction';
    Tags               : '';
    Description        : (''
      + 'Confidence of observation coordinates in local elevation direction '
      + '(theta)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence of observation coordinates in local azimuth direction (phi)
const
  RDB_RIEGL_OBS_CONFIDENCE_PHI : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_confidence_phi';
    Title              : 'Positional Confidence of Observation in Phi Direction';
    Tags               : '';
    Description        : (''
      + 'Confidence of observation coordinates in local azimuth direction (phi)'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Confidence value for rotation angle within plane of observed signal
const
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.obs_signal_confidence_rot';
    Title              : 'Rotation Angle Confidence of Observed Signal';
    Tags               : '';
    Description        : (''
      + 'Confidence value for rotation angle within plane of observed signal'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1e-06;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Flag indicating if observation shall be used for adjustment (0 = used as verification point; 1 = used for adjustment)
const
  RDB_RIEGL_USED_FOR_ADJUSTMENT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.used_for_adjustment';
    Title              : 'Used for Adjustment';
    Tags               : 'enumeration';
    Description        : (''
      + 'Flag indicating if observation shall be used for adjustment (0 = used '
      + 'as verification point; 1 = used for adjustment)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 1.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=Used as verification point'#10'1=Used for adjustment'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_USED_FOR_ADJUSTMENT_VERIFICATION : TRDBUInt8 = 0;
  RDB_RIEGL_USED_FOR_ADJUSTMENT_ADJUSTMENT : TRDBUInt8 = 1;

// ID (riegl.id) of a referenced object (0 = invalid)
const
  RDB_RIEGL_REFERENCE_OBJECT_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.reference_object_id';
    Title              : 'Reference Object ID';
    Tags               : '';
    Description        : (''
      + 'ID (riegl.id) of a referenced object (0 = invalid)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of events in specific channel, separate counters for input and output
const
  RDB_RIEGL_EVENT_INDEX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.event_index';
    Title              : 'Event Index';
    Tags               : '';
    Description        : (''
      + 'Number of events in specific channel, separate counters for input and '
      + 'output'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4294967295.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Width of input pulse (0 = output pulse)
const
  RDB_RIEGL_EVENT_PULSE_WIDTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.event_pulse_width';
    Title              : 'Event Pulse Width';
    Tags               : '';
    Description        : (''
      + 'Width of input pulse (0 = output pulse)'
    );
    UnitSymbol         : 'ms';
    Length             : 1;
    Resolution         : 0.1;
    MinimumValue       : 0.0;
    MaximumValue       : 6553.5;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Secondary Attributes"
//______________________________________________________________________________
//

// Raw range of echo
const
  RDB_RIEGL_RAW_RANGE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.raw_range';
    Title              : 'Raw Range';
    Tags               : '';
    Description        : (''
      + 'Raw range of echo'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : -50000.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Raw line angle
const
  RDB_RIEGL_RAW_LINE_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.raw_line_angle';
    Title              : 'Raw Line Angle';
    Tags               : '';
    Description        : (''
      + 'Raw line angle'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Raw frame angle
const
  RDB_RIEGL_RAW_FRAME_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.raw_frame_angle';
    Title              : 'Raw Frame Angle';
    Tags               : '';
    Description        : (''
      + 'Raw frame angle'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 1e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Coarse line scan angle
const
  RDB_RIEGL_LINE_ANGLE_COARSE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.line_angle_coarse';
    Title              : 'Line Angle Coarse';
    Tags               : '';
    Description        : (''
      + 'Coarse line scan angle'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.025;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Reduced line scan angle
const
  RDB_RIEGL_LINE_ANGLE_REDUCED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.line_angle_reduced';
    Title              : 'Line Angle Reduced';
    Tags               : '';
    Description        : (''
      + 'Reduced line scan angle'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.025;
    MinimumValue       : -360.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Coarse frame scan angle
const
  RDB_RIEGL_FRAME_ANGLE_COARSE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.frame_angle_coarse';
    Title              : 'Frame Angle Coarse';
    Tags               : '';
    Description        : (''
      + 'Coarse frame scan angle'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : 0.0;
    MaximumValue       : 360.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Angle of incidence of the laser beam with respect to a planar neighborhood, normal incidence gives 0 deg
const
  RDB_RIEGL_ANGLE_OF_INCIDENCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.angle_of_incidence';
    Title              : 'Angle of Incidence';
    Tags               : '';
    Description        : (''
      + 'Angle of incidence of the laser beam with respect to a planar '
      + 'neighborhood, normal incidence gives 0 deg'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 90.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Scan Line Index in Scan
const
  RDB_RIEGL_SCAN_LINE_INDEX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.scan_line_index';
    Title              : 'Scan Line Index';
    Tags               : '';
    Description        : (''
      + 'Scan Line Index in Scan'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -2000000000.0;
    MaximumValue       : 2000000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Laser Shot Index in Scan Line
const
  RDB_RIEGL_SHOT_INDEX_LINE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_index_line';
    Title              : 'Laser Shot Index in Line';
    Tags               : '';
    Description        : (''
      + 'Laser Shot Index in Scan Line'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -2000000000.0;
    MaximumValue       : 2000000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Mirror facet number (0 = invalid, 1 = first facet, ...)
const
  RDB_RIEGL_MIRROR_FACET : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.mirror_facet';
    Title              : 'Mirror Facet';
    Tags               : 'enumeration';
    Description        : (''
      + 'Mirror facet number (0 = invalid, 1 = first facet, ...)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 15.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_MIRROR_FACET_INVALID : TRDBUInt8 = 0;

// Scan segment number (0 = invalid, 1 = first segment, ...)
const
  RDB_RIEGL_SCAN_SEGMENT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.scan_segment';
    Title              : 'Scan Segment';
    Tags               : 'enumeration';
    Description        : (''
      + 'Scan segment number (0 = invalid, 1 = first segment, ...)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 15.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SCAN_SEGMENT_INVALID : TRDBUInt8 = 0;

// Waveform data available for laser-shot (0 = no, 1 = yes)
const
  RDB_RIEGL_WAVEFORM_AVAILABLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.waveform_available';
    Title              : 'Waveform Available';
    Tags               : 'boolean';
    Description        : (''
      + 'Waveform data available for laser-shot (0 = no, 1 = yes)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=No waveform available'#10'1=Waveform available'#10'';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WAVEFORM_AVAILABLE_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_WAVEFORM_AVAILABLE_TRUE : TRDBUInt8 = 1;

// 1 if the point was refraction corrected, 0 otherwise
const
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.hydro_refraction_corrected';
    Title              : 'Refraction Corrected';
    Tags               : 'boolean';
    Description        : (''
      + '1 if the point was refraction corrected, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=No refraction correction applied'#10'1=Refraction correction applied'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED_TRUE : TRDBUInt8 = 1;

// Extinction coefficient, i.e. exponential damping coefficient usually present in water bodies. N.B.: The factor is subject to refraction correction.
const
  RDB_RIEGL_EXTINCTION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.extinction';
    Title              : 'Extinction Coefficient';
    Tags               : '';
    Description        : (''
      + 'Extinction coefficient, i.e. exponential damping coefficient usually '
      + 'present in water bodies. N.B.: The factor is subject to refraction '
      + 'correction.'
    );
    UnitSymbol         : 'dB/m';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -30.0;
    MaximumValue       : 30.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Amplitude from volumetric backscatter in SVB FWA
const
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_amplitude_volumetric';
    Title              : 'SVB Volumetric Backscatter Amplitude';
    Tags               : '';
    Description        : (''
      + 'Amplitude from volumetric backscatter in SVB FWA'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -327.68;
    MaximumValue       : 327.67;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Reflectance from volumetric backscatter in SVB FWA
const
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_reflectance_volumetric';
    Title              : 'SVB Volumetric Backscatter Reflectance';
    Tags               : '';
    Description        : (''
      + 'Reflectance from volumetric backscatter in SVB FWA'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -327.68;
    MaximumValue       : 327.67;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Reflectance in SVB FWA of bottom return without correcting for the attenuation of the water column
const
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_reflectance_bottom_uncorrected';
    Title              : 'SVB Bottom Reflectance Uncorrected';
    Tags               : '';
    Description        : (''
      + 'Reflectance in SVB FWA of bottom return without correcting for the '
      + 'attenuation of the water column'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -327.68;
    MaximumValue       : 327.67;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Attenuation of the reflectance of bottom target due to backscatter and absorption in the water column in SVB FWA
const
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_attenuation_volumetric';
    Title              : 'SVB Attenuation of Bottom Return';
    Tags               : '';
    Description        : (''
      + 'Attenuation of the reflectance of bottom target due to backscatter and'
      + ' absorption in the water column in SVB FWA'
    );
    UnitSymbol         : 'dB';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 100.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Bottom target amplitude is attenuated by svb_target_attenuation_coefficient_volumetric for every additional meter of target depth in the water column.
const
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_target_attenuation_coefficient_volumetric';
    Title              : 'SVB Target Attenuation Coefficient';
    Tags               : '';
    Description        : (''
      + 'Bottom target amplitude is attenuated by '
      + 'svb_target_attenuation_coefficient_volumetric for every additional '
      + 'meter of target depth in the water column.'
    );
    UnitSymbol         : 'dB/m';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 100.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Indicates surface point from SVB FWA (1 = surface)
const
  RDB_RIEGL_SVB_SURFACE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_surface';
    Title              : 'SVB Surface Point';
    Tags               : 'boolean';
    Description        : (''
      + 'Indicates surface point from SVB FWA (1 = surface)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SVB_SURFACE_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_SVB_SURFACE_TRUE : TRDBUInt8 = 1;

// Indicates bottom point from SVB FWA (1 = bottom)
const
  RDB_RIEGL_SVB_BOTTOM : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_bottom';
    Title              : 'SVB Bottom Point';
    Tags               : 'boolean';
    Description        : (''
      + 'Indicates bottom point from SVB FWA (1 = bottom)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SVB_BOTTOM_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_SVB_BOTTOM_TRUE : TRDBUInt8 = 1;

// Path length between surface and bottom from SVB FWA
const
  RDB_RIEGL_SVB_PATH_LENGTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.svb_path_length';
    Title              : 'SVB Path Length';
    Tags               : '';
    Description        : (''
      + 'Path length between surface and bottom from SVB FWA'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.004;
    MinimumValue       : 0.0;
    MaximumValue       : 200.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 1 for all points of the first laser shot of a scan line, 0 otherwise
const
  RDB_RIEGL_START_OF_SCAN_LINE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.start_of_scan_line';
    Title              : 'Start of Scan Line';
    Tags               : 'boolean';
    Description        : (''
      + '1 for all points of the first laser shot of a scan line, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_START_OF_SCAN_LINE_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_START_OF_SCAN_LINE_TRUE : TRDBUInt8 = 1;

// 1 for all points of the last laser shot of a scan line, 0 otherwise
const
  RDB_RIEGL_END_OF_SCAN_LINE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.end_of_scan_line';
    Title              : 'End of Scan Line';
    Tags               : 'boolean';
    Description        : (''
      + '1 for all points of the last laser shot of a scan line, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_END_OF_SCAN_LINE_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_END_OF_SCAN_LINE_TRUE : TRDBUInt8 = 1;

// see LAS format specification 1.4-R13
const
  RDB_RIEGL_SCAN_ANGLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.scan_angle';
    Title              : 'Scan Angle';
    Tags               : '';
    Description        : (''
      + 'see LAS format specification 1.4-R13'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.096;
    MinimumValue       : -180.0;
    MaximumValue       : 180.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// see LAS format specification 1.4-R13
const
  RDB_RIEGL_SCAN_DIRECTION : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.scan_direction';
    Title              : 'Scan Direction';
    Tags               : 'enumeration';
    Description        : (''
      + 'see LAS format specification 1.4-R13'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=Negative'#10'1=Positive'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Voxel has been collapsed with neighbor (0 = not collapsed, 1 = collapsed)
const
  RDB_RIEGL_VOXEL_COLLAPSED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.voxel_collapsed';
    Title              : 'Voxel Collapsed with Neighbor';
    Tags               : 'boolean';
    Description        : (''
      + 'Voxel has been collapsed with neighbor (0 = not collapsed, 1 = '
      + 'collapsed)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_VOXEL_COLLAPSED_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_VOXEL_COLLAPSED_TRUE : TRDBUInt8 = 1;

// 1 if the mirror wheel rotates, 0 otherwise
const
  RDB_RIEGL_LINE_SCAN_ACTIVE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.line_scan_active';
    Title              : 'Line Scan Active';
    Tags               : '';
    Description        : (''
      + '1 if the mirror wheel rotates, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_LINE_SCAN_ACTIVE_FALSE : TRDBInt8 = 0;
  RDB_RIEGL_LINE_SCAN_ACTIVE_TRUE : TRDBInt8 = 1;

// 1 if the scanner head rotates, 0 otherwise
const
  RDB_RIEGL_FRAME_SCAN_ACTIVE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.frame_scan_active';
    Title              : 'Frame Scan Active';
    Tags               : '';
    Description        : (''
      + '1 if the scanner head rotates, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_FRAME_SCAN_ACTIVE_FALSE : TRDBInt8 = 0;
  RDB_RIEGL_FRAME_SCAN_ACTIVE_TRUE : TRDBInt8 = 1;

// 1 if the data acquisition is in progress, 0 otherwise
const
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.data_acquisition_active';
    Title              : 'Data Acquisition Active';
    Tags               : '';
    Description        : (''
      + '1 if the data acquisition is in progress, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : -1.0;
    MaximumValue       : 1.0;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE_FALSE : TRDBInt8 = 0;
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE_TRUE : TRDBInt8 = 1;

// IDs (riegl.id) of plane patches this observation refers to (0 = invalid)
const
  RDB_RIEGL_PLANE_REFERENCES : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_references';
    Title              : 'Plane Patch References';
    Tags               : '';
    Description        : (''
      + 'IDs (riegl.id) of plane patches this observation refers to (0 = '
      + 'invalid)'
    );
    UnitSymbol         : '';
    Length             : 2;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PLANE_REFERENCES_INVALID : TRDBUInt64 = 0;

// Cumulative distance travelled
const
  RDB_RIEGL_POF_PATH_LENGTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_path_length';
    Title              : 'Path Length';
    Tags               : '';
    Description        : (''
      + 'Cumulative distance travelled'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 1.0e-3;
    MinimumValue       : 0.0;
    MaximumValue       : 4.0e6;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Position (3D) dilution of precision
const
  RDB_RIEGL_POF_PDOP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_pdop';
    Title              : 'PDOP';
    Tags               : '';
    Description        : (''
      + 'Position (3D) dilution of precision'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 100.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Horizontal dilution of precision
const
  RDB_RIEGL_POF_HDOP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_hdop';
    Title              : 'HDOP';
    Tags               : '';
    Description        : (''
      + 'Horizontal dilution of precision'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 100.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Vertical dilution of precision
const
  RDB_RIEGL_POF_VDOP : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_vdop';
    Title              : 'VDOP';
    Tags               : '';
    Description        : (''
      + 'Vertical dilution of precision'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 100.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Age of GNSS RTK corrections (-1 = unknown)
const
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_age_of_corrections';
    Title              : 'Age Of Corrections';
    Tags               : '';
    Description        : (''
      + 'Age of GNSS RTK corrections (-1 = unknown)'
    );
    UnitSymbol         : 's';
    Length             : 1;
    Resolution         : 0.1;
    MinimumValue       : -1.0;
    MaximumValue       : 1000.0;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Length of GNSS RTK baseline, i.e. the distance between the antennas of the base station and the rover
const
  RDB_RIEGL_POF_BASELINE_LENGTH : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_baseline_length';
    Title              : 'Baseline Length';
    Tags               : '';
    Description        : (''
      + 'Length of GNSS RTK baseline, i.e. the distance between the antennas of'
      + ' the base station and the rover'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.1;
    MinimumValue       : 0.0;
    MaximumValue       : 1000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Solution type of GNSS
const
  RDB_RIEGL_POF_SOLUTION_GNSS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_solution_gnss';
    Title              : 'GNSS Solution';
    Tags               : 'enumeration';
    Description        : (''
      + 'Solution type of GNSS'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 8.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '0=GNSS fix invalid'#10'1=GNSS fix single'#10'2=GNSS fix DGPS'#10'3=GNSS fix time only'#10'4=GNSS fix RTK fixed'#10'5=GNSS fix RTK TRDBFloat32'#10'6=GNSS fix estimated'#10'7=GNSS fix manual'#10'8=GNSS fix simulated'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Total number of GNSS satellites that were used to calculate the position
const
  RDB_RIEGL_POF_SATELLITES_GNSS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_satellites_gnss';
    Title              : 'Number of GNSS Satellites';
    Tags               : '';
    Description        : (''
      + 'Total number of GNSS satellites that were used to calculate the '
      + 'position'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of GPS satellites that were used to calculate the position
const
  RDB_RIEGL_POF_SATELLITES_GPS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_satellites_gps';
    Title              : 'Number of GPS Satellites';
    Tags               : '';
    Description        : (''
      + 'Number of GPS satellites that were used to calculate the position'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of GLONASS satellites that were used to calculate the position
const
  RDB_RIEGL_POF_SATELLITES_GLONASS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_satellites_glonass';
    Title              : 'Number of GLONASS Satellites';
    Tags               : '';
    Description        : (''
      + 'Number of GLONASS satellites that were used to calculate the position'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of Beidou satellites that were used to calculate the position
const
  RDB_RIEGL_POF_SATELLITES_BEIDOU : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_satellites_beidou';
    Title              : 'Number of Beidou Satellites';
    Tags               : '';
    Description        : (''
      + 'Number of Beidou satellites that were used to calculate the position'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of Galileo satellites that were used to calculate the position
const
  RDB_RIEGL_POF_SATELLITES_GALILEO : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_satellites_galileo';
    Title              : 'Number of Galileo Satellites';
    Tags               : '';
    Description        : (''
      + 'Number of Galileo satellites that were used to calculate the position'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of Quasi-Zenith Satellite System (QZSS) satellites that were used to calculate the position
const
  RDB_RIEGL_POF_SATELLITES_QZSS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pof_satellites_qzss';
    Title              : 'Number of QZSS Satellites';
    Tags               : '';
    Description        : (''
      + 'Number of Quasi-Zenith Satellite System (QZSS) satellites that were '
      + 'used to calculate the position'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Sums of the linear point distances to the pixel corner (0: dx, 1: dy, 2: dz)
const
  RDB_RIEGL_PIXEL_LINEAR_SUMS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pixel_linear_sums';
    Title              : 'Linear Sums';
    Tags               : '';
    Description        : (''
      + 'Sums of the linear point distances to the pixel corner (0: dx, 1: dy, '
      + '2: dz)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 1e-3;
    MinimumValue       : 0.0;
    MaximumValue       : 64000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PIXEL_LINEAR_SUMS_DX : Cardinal = 0;
  RDB_RIEGL_PIXEL_LINEAR_SUMS_DY : Cardinal = 1;
  RDB_RIEGL_PIXEL_LINEAR_SUMS_DZ : Cardinal = 2;

// Sums of the square point distances to the pixel corner (0: dx*dx, 1: dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)
const
  RDB_RIEGL_PIXEL_SQUARE_SUMS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.pixel_square_sums';
    Title              : 'Square Sums';
    Tags               : '';
    Description        : (''
      + 'Sums of the square point distances to the pixel corner (0: dx*dx, 1: '
      + 'dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)'
    );
    UnitSymbol         : 'm²';
    Length             : 6;
    Resolution         : 1e-6;
    MinimumValue       : 0.0;
    MaximumValue       : 5120000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_PIXEL_SQUARE_SUMS_DX_DX : Cardinal = 0;
  RDB_RIEGL_PIXEL_SQUARE_SUMS_DY_DY : Cardinal = 1;
  RDB_RIEGL_PIXEL_SQUARE_SUMS_DZ_DZ : Cardinal = 2;
  RDB_RIEGL_PIXEL_SQUARE_SUMS_DX_DY : Cardinal = 3;
  RDB_RIEGL_PIXEL_SQUARE_SUMS_DY_DZ : Cardinal = 4;
  RDB_RIEGL_PIXEL_SQUARE_SUMS_DX_DZ : Cardinal = 5;

// Estimated shape of point cloud (0 = undefined, 1 = plane, 2 = line, 3 = volume)
const
  RDB_RIEGL_SHAPE_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shape_id';
    Title              : 'Point Cloud Shape';
    Tags               : 'enumeration';
    Description        : (''
      + 'Estimated shape of point cloud (0 = undefined, 1 = plane, 2 = line, 3 '
      + '= volume)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 3.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '0=Undefined'#10'1=Plane'#10'2=Line'#10'3=Volume'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHAPE_ID_UNDEFINED : TRDBUInt8 = 0;
  RDB_RIEGL_SHAPE_ID_PLANE : TRDBUInt8 = 1;
  RDB_RIEGL_SHAPE_ID_LINE : TRDBUInt8 = 2;
  RDB_RIEGL_SHAPE_ID_VOLUME : TRDBUInt8 = 3;

// The plane cluster ID this point belongs to (0 = no cluster)
const
  RDB_RIEGL_PLANE_CLUSTER_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.plane_cluster_id';
    Title              : 'Plane Cluster ID';
    Tags               : '';
    Description        : (''
      + 'The plane cluster ID this point belongs to (0 = no cluster)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4000000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// The segment ID this point belongs to (0 = no segment)
const
  RDB_RIEGL_SEGMENT_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.segment_id';
    Title              : 'Segment ID';
    Tags               : '';
    Description        : (''
      + 'The segment ID this point belongs to (0 = no segment)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4000000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Sums of the linear point distances to the voxel corner (0: dx, 1: dy, 2: dz)
const
  RDB_RIEGL_VOXEL_LINEAR_SUMS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.voxel_linear_sums';
    Title              : 'Linear Sums';
    Tags               : '';
    Description        : (''
      + 'Sums of the linear point distances to the voxel corner (0: dx, 1: dy, '
      + '2: dz)'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 2.5e-4;
    MinimumValue       : 0.0;
    MaximumValue       : 16000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_VOXEL_LINEAR_SUMS_DX : Cardinal = 0;
  RDB_RIEGL_VOXEL_LINEAR_SUMS_DY : Cardinal = 1;
  RDB_RIEGL_VOXEL_LINEAR_SUMS_DZ : Cardinal = 2;

// Sums of the square point distances to the voxel corner (0: dx*dx, 1: dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)
const
  RDB_RIEGL_VOXEL_SQUARE_SUMS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.voxel_square_sums';
    Title              : 'Square Sums';
    Tags               : '';
    Description        : (''
      + 'Sums of the square point distances to the voxel corner (0: dx*dx, 1: '
      + 'dy*dy, 2: dz*dz, 3: dx*dy, 4: dy*dz, 5: dx*dz)'
    );
    UnitSymbol         : 'm²';
    Length             : 6;
    Resolution         : 6.25e-8;
    MinimumValue       : 0.0;
    MaximumValue       : 320000000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_VOXEL_SQUARE_SUMS_DX_DX : Cardinal = 0;
  RDB_RIEGL_VOXEL_SQUARE_SUMS_DY_DY : Cardinal = 1;
  RDB_RIEGL_VOXEL_SQUARE_SUMS_DZ_DZ : Cardinal = 2;
  RDB_RIEGL_VOXEL_SQUARE_SUMS_DX_DY : Cardinal = 3;
  RDB_RIEGL_VOXEL_SQUARE_SUMS_DY_DZ : Cardinal = 4;
  RDB_RIEGL_VOXEL_SQUARE_SUMS_DX_DZ : Cardinal = 5;

// Integer coordinates of voxel corner
const
  RDB_RIEGL_VOXEL_INDEX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.voxel_index';
    Title              : 'Voxel Index';
    Tags               : '';
    Description        : (''
      + 'Integer coordinates of voxel corner'
    );
    UnitSymbol         : '';
    Length             : 3;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4294967295.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Elements 00, 11, 22, 10, 21 and 20 (in that order) of point cloud covariance matrix
const
  RDB_RIEGL_COVARIANCES : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.covariances';
    Title              : 'Point Cloud Covariances';
    Tags               : '';
    Description        : (''
      + 'Elements 00, 11, 22, 10, 21 and 20 (in that order) of point cloud '
      + 'covariance matrix'
    );
    UnitSymbol         : '';
    Length             : 6;
    Resolution         : 1.0e-6;
    MinimumValue       : -4.5e9;
    MaximumValue       : 4.5e9;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of voxels this point represents (e.g. voxels combined to pixels, 0 = unknown)
const
  RDB_RIEGL_VOXEL_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.voxel_count';
    Title              : 'Voxel Count';
    Tags               : '';
    Description        : (''
      + 'Number of voxels this point represents (e.g. voxels combined to '
      + 'pixels, 0 = unknown)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4294967295.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_VOXEL_COUNT_UNKNOWN : TRDBUInt32 = 0;

// Point identifier, unique within database (0 = invalid)
const
  RDB_RIEGL_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.id';
    Title              : 'PID';
    Tags               : '';
    Description        : (''
      + 'Point identifier, unique within database (0 = invalid)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_ID_INVALID : TRDBUInt64 = 0;

// Identifier of first vertex that belongs to a geometry object (e.g. polyline) (0 = invalid)
const
  RDB_RIEGL_VERTEX_FIRST : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.vertex_first';
    Title              : 'Vertex First';
    Tags               : '';
    Description        : (''
      + 'Identifier of first vertex that belongs to a geometry object (e.g. '
      + 'polyline) (0 = invalid)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 4294967295.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Number of vertices that belong to a geometry object (e.g. polyline)
const
  RDB_RIEGL_VERTEX_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.vertex_count';
    Title              : 'Vertex Count';
    Tags               : '';
    Description        : (''
      + 'Number of vertices that belong to a geometry object (e.g. polyline)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Search radius for finding the control point in the scan data. This value (if valid) shall be preferred over that of the common settings when finding observation points for the respective control point.
const
  RDB_RIEGL_CP_SEARCH_RADIUS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cp_search_radius';
    Title              : 'Search Radius';
    Tags               : '';
    Description        : (''
      + 'Search radius for finding the control point in the scan data. This '
      + 'value (if valid) shall be preferred over that of the common settings '
      + 'when finding observation points for the respective control point.'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : 0.0;
    MaximumValue       : 655.35;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_CP_SEARCH_RADIUS_INVALID : TRDBFloat64 = 0.0;

// Maximum accepted distance of an observed control point from SOCS center. This value (if valid) shall be preferred over that of the common settings when finding observation points for the respective control point.
const
  RDB_RIEGL_CP_MAXIMUM_DISTANCE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.cp_maximum_distance';
    Title              : 'Maximum Distance';
    Tags               : '';
    Description        : (''
      + 'Maximum accepted distance of an observed control point from SOCS '
      + 'center. This value (if valid) shall be preferred over that of the '
      + 'common settings when finding observation points for the respective '
      + 'control point.'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.1;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_CP_MAXIMUM_DISTANCE_INVALID : TRDBFloat64 = 0.0;

//______________________________________________________________________________
//
// POINT ATTRIBUTE GROUP "Other Attributes"
//______________________________________________________________________________
//

// Point selected by user (0 = not selected, 1 = selected)
const
  RDB_RIEGL_SELECTED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.selected';
    Title              : 'Selected';
    Tags               : 'boolean';
    Description        : (''
      + 'Point selected by user (0 = not selected, 1 = selected)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_DYNAMIC);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SELECTED_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_SELECTED_TRUE : TRDBUInt8 = 1;

// Point visible (i.e. not hidden) in view (0 = hidden, 1 = visible)
const
  RDB_RIEGL_VISIBLE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.visible';
    Title              : 'Visible';
    Tags               : 'boolean';
    Description        : (''
      + 'Point visible (i.e. not hidden) in view (0 = hidden, 1 = visible)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 1.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_DYNAMIC);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_VISIBLE_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_VISIBLE_TRUE : TRDBUInt8 = 1;

// ID of original point cloud (0 = unknown)
const
  RDB_RIEGL_SOURCE_CLOUD_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.source_cloud_id';
    Title              : 'Point Cloud ID';
    Tags               : '';
    Description        : (''
      + 'ID of original point cloud (0 = unknown)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 10000.0;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SOURCE_CLOUD_ID_UNKNOWN : TRDBUInt16 = 0;

// For points merged from multiple source files, the number of source files contributing to the point (0 = unknown)
const
  RDB_RIEGL_SOURCE_CLOUD_COUNT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.source_cloud_count';
    Title              : 'Point Cloud Count';
    Tags               : '';
    Description        : (''
      + 'For points merged from multiple source files, the number of source '
      + 'files contributing to the point (0 = unknown)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 65535.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SOURCE_CLOUD_COUNT_UNKNOWN : TRDBUInt16 = 0;

// Index of point in original point cloud (0 = unknown)
const
  RDB_RIEGL_SOURCE_INDEX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.source_index';
    Title              : 'Point Index';
    Tags               : '';
    Description        : (''
      + 'Index of point in original point cloud (0 = unknown)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SOURCE_INDEX_UNKNOWN : TRDBUInt64 = 0;

// 0 for all points derived by standard waveform processing, 1 for additional points derived from interpolation, other enum values to be defined separately
const
  RDB_RIEGL_SOURCE_INDICATOR : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.source_indicator';
    Title              : 'Source Indicator';
    Tags               : 'enumeration';
    Description        : (''
      + '0 for all points derived by standard waveform processing, 1 for '
      + 'additional points derived from interpolation, other enum values to be '
      + 'defined separately'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 255.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=Standard processing'#10'1=Target interpolation'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SOURCE_INDICATOR_STANDARD_PROCESSING : TRDBUInt8 = 0;
  RDB_RIEGL_SOURCE_INDICATOR_TARGET_INTERPOLATION : TRDBUInt8 = 1;

// Marks points that belong to dynamic objects (0 = no dynamic object, 1 = dynamic object)
const
  RDB_RIEGL_DYNAMIC_OBJECT_POINT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.dynamic_object_point';
    Title              : 'Dynamic Object Point';
    Tags               : 'boolean';
    Description        : (''
      + 'Marks points that belong to dynamic objects (0 = no dynamic object, 1 '
      + '= dynamic object)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Marks points that originate from one source file (0 = multiple source files, 1 = single source file)
const
  RDB_RIEGL_SINGLE_SOURCE_POINT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.single_source_point';
    Title              : 'Single Source Point';
    Tags               : 'boolean';
    Description        : (''
      + 'Marks points that originate from one source file (0 = multiple source '
      + 'files, 1 = single source file)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 1.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Marks points that belong to mirror objects (0 = no mirror object, 1 = mirror object)
const
  RDB_RIEGL_MIRROR_OBJECT_POINT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.mirror_object_point';
    Title              : 'Mirror Object Point';
    Tags               : 'boolean';
    Description        : (''
      + 'Marks points that belong to mirror objects (0 = no mirror object, 1 = '
      + 'mirror object)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 1 for all points in temporal vicinity of echoes from the exit aperture and corrected for the impact of the exit pane on amplitude and range, 0 otherwise
const
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.window_echo_impact_corrected';
    Title              : 'Window Echo Impact Corrected';
    Tags               : 'boolean';
    Description        : (''
      + '1 for all points in temporal vicinity of echoes from the exit aperture'
      + ' and corrected for the impact of the exit pane on amplitude and range,'
      + ' 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=Not impacted by exit aperture'#10'1=Impacted by exit aperture'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED_TRUE : TRDBUInt8 = 1;

// 1 for all points with an echo signal amplitude below the dynamic trigger level, 0 otherwise
const
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.dyntrig_uncertain_point';
    Title              : 'Dyntrig Uncertain Point';
    Tags               : 'boolean';
    Description        : (''
      + '1 for all points with an echo signal amplitude below the dynamic '
      + 'trigger level, 0 otherwise'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT_FALSE : TRDBUInt8 = 0;
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT_TRUE : TRDBUInt8 = 1;

// Altitude determined based on the atmospheric pressure according to the standard atmosphere laws
const
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.barometric_height_amsl';
    Title              : 'Barometric Altitude';
    Tags               : '';
    Description        : (''
      + 'Altitude determined based on the atmospheric pressure according to the'
      + ' standard atmosphere laws'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.01;
    MinimumValue       : -100.0;
    MaximumValue       : 10000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Distance between point and a reference surface (values less than -2000.0 mean no distance available)
const
  RDB_RIEGL_DISTANCE_TO_SURFACE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.distance_to_surface';
    Title              : 'Distance to Surface';
    Tags               : '';
    Description        : (''
      + 'Distance between point and a reference surface (values less than '
      + '-2000.0 mean no distance available)'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.0001;
    MinimumValue       : -2100.0;
    MaximumValue       : 2000.0;
    DefaultValue       : -2100.0;
    InvalidValue       : -2100.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_DISTANCE_TO_SURFACE_LOWEST : TRDBFloat64 = -2000.0;

// Minimum distance to points in neighborhood
const
  RDB_RIEGL_NBHD_DISTANCE_MIN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nbhd_distance_min';
    Title              : 'Neighborhood Minimum Distance';
    Tags               : '';
    Description        : (''
      + 'Minimum distance to points in neighborhood'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Maximum distance to points in neighborhood
const
  RDB_RIEGL_NBHD_DISTANCE_MAX : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nbhd_distance_max';
    Title              : 'Neighborhood Maximum Distance';
    Tags               : '';
    Description        : (''
      + 'Maximum distance to points in neighborhood'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Mean distance to points in neighborhood
const
  RDB_RIEGL_NBHD_DISTANCE_MEAN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nbhd_distance_mean';
    Title              : 'Neighborhood Mean Distance';
    Tags               : '';
    Description        : (''
      + 'Mean distance to points in neighborhood'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Median distance to points in neighborhood
const
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nbhd_distance_median';
    Title              : 'Neighborhood Median Distance';
    Tags               : '';
    Description        : (''
      + 'Median distance to points in neighborhood'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Standard deviation of distances to points in neighborhood
const
  RDB_RIEGL_NBHD_DISTANCE_STD : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nbhd_distance_std';
    Title              : 'Neighborhood Distance, Standard Deviation';
    Tags               : '';
    Description        : (''
      + 'Standard deviation of distances to points in neighborhood'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// MAD standard deviation of distances to points in neighborhood
const
  RDB_RIEGL_NBHD_DISTANCE_MAD : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.nbhd_distance_mad';
    Title              : 'Neighborhood Distance, Median Absolute Deviation';
    Tags               : '';
    Description        : (''
      + 'MAD standard deviation of distances to points in neighborhood'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.00025;
    MinimumValue       : 0.0;
    MaximumValue       : 50000.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_SHUFFLE);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Identifier of laser shot (0 = invalid). This is not an array index but the value of riegl.id of the laser shot.
const
  RDB_RIEGL_SHOT_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.shot_id';
    Title              : 'Shot ID';
    Tags               : '';
    Description        : (''
      + 'Identifier of laser shot (0 = invalid). This is not an array index but'
      + ' the value of riegl.id of the laser shot.'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_SHOT_ID_INVALID : TRDBUInt64 = 0;

// Identifier of waveform sample block (0 = invalid). This is not an array index but the value of riegl.id of the block.
const
  RDB_RIEGL_WFM_SBL_ID : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.wfm_sbl_id';
    Title              : 'Waveform Sample Block ID';
    Tags               : '';
    Description        : (''
      + 'Identifier of waveform sample block (0 = invalid). This is not an '
      + 'array index but the value of riegl.id of the block.'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0e12;
    DefaultValue       : 0.0;
    InvalidValue       : 0.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_CONSTANT);
    CompressionOptions : Ord(RDB_COMPRESSION_DELTA);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );
  RDB_RIEGL_WFM_SBL_ID_INVALID : TRDBUInt64 = 0;

// Event fake pulse (0 = real input event, 1 = fake input event to indicate that no event was detected within set time window)
const
  RDB_RIEGL_EVENT_FAKE_PULSE : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.event_fake_pulse';
    Title              : 'Event Fake Pulse';
    Tags               : 'boolean';
    Description        : (''
      + 'Event fake pulse (0 = real input event, 1 = fake input event to '
      + 'indicate that no event was detected within set time window)'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '0=real input event'#10'1=fake input event'#10'';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 1 if point is inside inner box, 0 if outside
const
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.point_inside_aabb_bocs';
    Title              : 'Inside Inner AABB in BOCS';
    Tags               : 'boolean';
    Description        : (''
      + '1 if point is inside inner box, 0 if outside'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// 1 if point is outside outer box, 0 if inside
const
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.point_outside_aabb_bocs';
    Title              : 'Outside Outer AABB in BOCS';
    Tags               : 'boolean';
    Description        : (''
      + '1 if point is outside outer box, 0 if inside'
    );
    UnitSymbol         : '';
    Length             : 1;
    Resolution         : 1.0;
    MinimumValue       : 0.0;
    MaximumValue       : 1.0;
    DefaultValue       : 0.0;
    InvalidValue       : RDB_NO_INVALID_VALUE;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Water surface model z uncertainty at intersection point
const
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.tpu_wsm_z_uncertainty';
    Title              : 'Water Surface Model Z Uncertainty';
    Tags               : '';
    Description        : (''
      + 'Water surface model z uncertainty at intersection point'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -1.0;
    MaximumValue       : 64.535;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Water surface model normal uncertainty at intersection point
const
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.tpu_wsm_normal_uncertainty';
    Title              : 'Water Surface Model Normal Uncertainty';
    Tags               : '';
    Description        : (''
      + 'Water surface model normal uncertainty at intersection point'
    );
    UnitSymbol         : 'deg';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -1.0;
    MaximumValue       : 64.535;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Uncertainties of Cartesian coordinates (0: X, 1: Y, 2: Z) resulting from uncertainty propagation based on contributing systematic and random measurement uncertainties from several sources
const
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.tpu_xyz_uncertainties';
    Title              : 'XYZ Uncertainties';
    Tags               : '';
    Description        : (''
      + 'Uncertainties of Cartesian coordinates (0: X, 1: Y, 2: Z) resulting '
      + 'from uncertainty propagation based on contributing systematic and '
      + 'random measurement uncertainties from several sources'
    );
    UnitSymbol         : 'm';
    Length             : 3;
    Resolution         : 0.001;
    MinimumValue       : -1.0;
    MaximumValue       : 64.535;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Total Horizontal Uncertainty (THU) at a confidence level of 95% derived from x- and y-component of xyz uncertainties
const
  RDB_RIEGL_TPU_THU : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.tpu_thu';
    Title              : 'Total Horizontal Uncertainty';
    Tags               : '';
    Description        : (''
      + 'Total Horizontal Uncertainty (THU) at a confidence level of 95% '
      + 'derived from x- and y-component of xyz uncertainties'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -1.0;
    MaximumValue       : 64.535;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Total Vertical Uncertainty (TVU) at a confidence level of 95% derived from z-component of xyz uncertainties
const
  RDB_RIEGL_TPU_TVU : TRDBPointcloudPointAttributeInfo = (
    Name               : 'riegl.tpu_tvu';
    Title              : 'Total Vertical Uncertainty';
    Tags               : '';
    Description        : (''
      + 'Total Vertical Uncertainty (TVU) at a confidence level of 95% derived '
      + 'from z-component of xyz uncertainties'
    );
    UnitSymbol         : 'm';
    Length             : 1;
    Resolution         : 0.001;
    MinimumValue       : -1.0;
    MaximumValue       : 64.535;
    DefaultValue       : -1.0;
    InvalidValue       : -1.0;
    NamedValues        : '';
    StorageClass       : Ord(RDB_STORAGE_VARIABLE);
    CompressionOptions : Ord(RDB_COMPRESSION_DEFAULT);
    LodSettings        : 'default';
    ScaleFactor        : 1.0;
  );

// Table of all point attribute details records
const RDB_POINT_ATTRIBUTES : array of PRDBPointcloudPointAttributeInfo = nil; // filled during initialization

// Table of all point attribute group names
// (same length+order as RDB_POINT_ATTRIBUTES)
const RDB_POINT_ATTRIBUTE_GROUPS : array of System.String = nil; // filled during initialization

// Point attribute tags
const
  RDB_TAG_BOOLEAN : String = 'boolean'; // state/flag
  RDB_TAG_COLOR : String = 'color'; // color values
  RDB_TAG_DIRECTION : String = 'direction'; // direction vector
  RDB_TAG_ENUMERATION : String = 'enumeration'; // finite set of unique values, optionally with a name
  RDB_TAG_POSITION : String = 'position'; // position vector
  RDB_TAG_TIMESTAMP : String = 'timestamp'; // timestamp of some event, relative to epoch defined in riegl.time_base
  RDB_TAG_TRANSFORM : String = 'transform'; // transform position or direction vector during import/export

// Table of all point attribute tags (alphabetical order)
const RDB_POINT_ATTRIBUTE_TAGS : array of System.String = nil; // filled during initialization

implementation

procedure RDBAssignPointAttribute(
  const Source : TRDBPointcloudPointAttributeInfo;
  var   Target : TRDBPointcloudPointAttribute
);
begin
  Target.Name               := Source.Name;
  Target.Title              := Source.Title;
  Target.Tags               := Source.Tags;
  Target.Description        := Source.Description;
  Target.UnitSymbol         := Source.UnitSymbol;
  Target.Length             := Source.Length;
  Target.Resolution         := Source.Resolution;
  Target.MinimumValue       := Source.MinimumValue;
  Target.MaximumValue       := Source.MaximumValue;
  Target.DefaultValue       := Source.DefaultValue;
  Target.InvalidValue       := Source.InvalidValue;
  Target.NamedValues        := Source.NamedValues;
  Target.StorageClass       := TRDBPointcloudStorageClass(Source.StorageClass);
  Target.CompressionOptions := TRDBPointcloudCompressionOptions(Source.CompressionOptions);
  Target.LodSettings        := Source.LodSettings;
  Target.ScaleFactor        := Source.ScaleFactor;
end;

procedure RDBAssignPointAttribute(
  const Source : TRDBPointcloudPointAttribute;
  var   Target : TRDBPointcloudPointAttribute
);
begin
  Target.Load(Source.Save()); // let the library do the job...
end;

procedure FixStringEncoding;
begin
  // As the unit (.pas file) doesn't contain a BOM, we need to convert all strings
  // from UTF-8 to "native" encoding at runtime (tested on Delphi 7 and Delphi XE5)

  RDB_RIEGL_XYZ.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.Name)));
  RDB_RIEGL_XYZ.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.Title)));
  RDB_RIEGL_XYZ.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.Tags)));
  RDB_RIEGL_XYZ.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.Description)));
  RDB_RIEGL_XYZ.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.UnitSymbol)));
  RDB_RIEGL_XYZ.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.NamedValues)));
  RDB_RIEGL_XYZ.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ.LodSettings)));

  RDB_RIEGL_XYZ_SOCS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.Name)));
  RDB_RIEGL_XYZ_SOCS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.Title)));
  RDB_RIEGL_XYZ_SOCS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.Tags)));
  RDB_RIEGL_XYZ_SOCS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.Description)));
  RDB_RIEGL_XYZ_SOCS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.UnitSymbol)));
  RDB_RIEGL_XYZ_SOCS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.NamedValues)));
  RDB_RIEGL_XYZ_SOCS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_SOCS.LodSettings)));

  RDB_RIEGL_XYZ_LVCS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.Name)));
  RDB_RIEGL_XYZ_LVCS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.Title)));
  RDB_RIEGL_XYZ_LVCS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.Tags)));
  RDB_RIEGL_XYZ_LVCS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.Description)));
  RDB_RIEGL_XYZ_LVCS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.UnitSymbol)));
  RDB_RIEGL_XYZ_LVCS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.NamedValues)));
  RDB_RIEGL_XYZ_LVCS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_LVCS.LodSettings)));

  RDB_RIEGL_XYZ_MAP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.Name)));
  RDB_RIEGL_XYZ_MAP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.Title)));
  RDB_RIEGL_XYZ_MAP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.Tags)));
  RDB_RIEGL_XYZ_MAP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.Description)));
  RDB_RIEGL_XYZ_MAP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.UnitSymbol)));
  RDB_RIEGL_XYZ_MAP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.NamedValues)));
  RDB_RIEGL_XYZ_MAP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP.LodSettings)));

  RDB_RIEGL_XYZ_MAP_MIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.Name)));
  RDB_RIEGL_XYZ_MAP_MIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.Title)));
  RDB_RIEGL_XYZ_MAP_MIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.Tags)));
  RDB_RIEGL_XYZ_MAP_MIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.Description)));
  RDB_RIEGL_XYZ_MAP_MIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.UnitSymbol)));
  RDB_RIEGL_XYZ_MAP_MIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.NamedValues)));
  RDB_RIEGL_XYZ_MAP_MIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MIN.LodSettings)));

  RDB_RIEGL_XYZ_MAP_MAX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.Name)));
  RDB_RIEGL_XYZ_MAP_MAX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.Title)));
  RDB_RIEGL_XYZ_MAP_MAX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.Tags)));
  RDB_RIEGL_XYZ_MAP_MAX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.Description)));
  RDB_RIEGL_XYZ_MAP_MAX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.UnitSymbol)));
  RDB_RIEGL_XYZ_MAP_MAX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.NamedValues)));
  RDB_RIEGL_XYZ_MAP_MAX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_MAP_MAX.LodSettings)));

  RDB_RIEGL_XY_MAP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.Name)));
  RDB_RIEGL_XY_MAP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.Title)));
  RDB_RIEGL_XY_MAP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.Tags)));
  RDB_RIEGL_XY_MAP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.Description)));
  RDB_RIEGL_XY_MAP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.UnitSymbol)));
  RDB_RIEGL_XY_MAP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.NamedValues)));
  RDB_RIEGL_XY_MAP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XY_MAP.LodSettings)));

  RDB_RIEGL_XYZ_CORRECTIONS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.Name)));
  RDB_RIEGL_XYZ_CORRECTIONS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.Title)));
  RDB_RIEGL_XYZ_CORRECTIONS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.Tags)));
  RDB_RIEGL_XYZ_CORRECTIONS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.Description)));
  RDB_RIEGL_XYZ_CORRECTIONS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.UnitSymbol)));
  RDB_RIEGL_XYZ_CORRECTIONS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.NamedValues)));
  RDB_RIEGL_XYZ_CORRECTIONS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_CORRECTIONS.LodSettings)));

  RDB_RIEGL_RANGE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.Name)));
  RDB_RIEGL_RANGE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.Title)));
  RDB_RIEGL_RANGE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.Tags)));
  RDB_RIEGL_RANGE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.Description)));
  RDB_RIEGL_RANGE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.UnitSymbol)));
  RDB_RIEGL_RANGE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.NamedValues)));
  RDB_RIEGL_RANGE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RANGE.LodSettings)));

  RDB_RIEGL_THETA.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.Name)));
  RDB_RIEGL_THETA.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.Title)));
  RDB_RIEGL_THETA.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.Tags)));
  RDB_RIEGL_THETA.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.Description)));
  RDB_RIEGL_THETA.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.UnitSymbol)));
  RDB_RIEGL_THETA.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.NamedValues)));
  RDB_RIEGL_THETA.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_THETA.LodSettings)));

  RDB_RIEGL_PHI.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.Name)));
  RDB_RIEGL_PHI.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.Title)));
  RDB_RIEGL_PHI.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.Tags)));
  RDB_RIEGL_PHI.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.Description)));
  RDB_RIEGL_PHI.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.UnitSymbol)));
  RDB_RIEGL_PHI.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.NamedValues)));
  RDB_RIEGL_PHI.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PHI.LodSettings)));

  RDB_RIEGL_SCANNER_POSITION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.Name)));
  RDB_RIEGL_SCANNER_POSITION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.Title)));
  RDB_RIEGL_SCANNER_POSITION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.Tags)));
  RDB_RIEGL_SCANNER_POSITION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.Description)));
  RDB_RIEGL_SCANNER_POSITION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.UnitSymbol)));
  RDB_RIEGL_SCANNER_POSITION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.NamedValues)));
  RDB_RIEGL_SCANNER_POSITION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCANNER_POSITION.LodSettings)));

  RDB_RIEGL_DIRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.Name)));
  RDB_RIEGL_DIRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.Title)));
  RDB_RIEGL_DIRECTION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.Tags)));
  RDB_RIEGL_DIRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.Description)));
  RDB_RIEGL_DIRECTION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.UnitSymbol)));
  RDB_RIEGL_DIRECTION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.NamedValues)));
  RDB_RIEGL_DIRECTION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION.LodSettings)));

  RDB_RIEGL_DIRECTION_MEDIUM.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.Name)));
  RDB_RIEGL_DIRECTION_MEDIUM.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.Title)));
  RDB_RIEGL_DIRECTION_MEDIUM.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.Tags)));
  RDB_RIEGL_DIRECTION_MEDIUM.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.Description)));
  RDB_RIEGL_DIRECTION_MEDIUM.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.UnitSymbol)));
  RDB_RIEGL_DIRECTION_MEDIUM.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.NamedValues)));
  RDB_RIEGL_DIRECTION_MEDIUM.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_MEDIUM.LodSettings)));

  RDB_RIEGL_DIRECTION_COARSE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.Name)));
  RDB_RIEGL_DIRECTION_COARSE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.Title)));
  RDB_RIEGL_DIRECTION_COARSE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.Tags)));
  RDB_RIEGL_DIRECTION_COARSE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.Description)));
  RDB_RIEGL_DIRECTION_COARSE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.UnitSymbol)));
  RDB_RIEGL_DIRECTION_COARSE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.NamedValues)));
  RDB_RIEGL_DIRECTION_COARSE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DIRECTION_COARSE.LodSettings)));

  RDB_RIEGL_SHOT_ORIGIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.Name)));
  RDB_RIEGL_SHOT_ORIGIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.Title)));
  RDB_RIEGL_SHOT_ORIGIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.Tags)));
  RDB_RIEGL_SHOT_ORIGIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.Description)));
  RDB_RIEGL_SHOT_ORIGIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.UnitSymbol)));
  RDB_RIEGL_SHOT_ORIGIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.NamedValues)));
  RDB_RIEGL_SHOT_ORIGIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ORIGIN.LodSettings)));

  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Name)));
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Title)));
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Tags)));
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.Description)));
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.UnitSymbol)));
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.NamedValues)));
  RDB_RIEGL_SHOT_BIAXIAL_SHIFT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_BIAXIAL_SHIFT.LodSettings)));

  RDB_RIEGL_SHOT_DIRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.Name)));
  RDB_RIEGL_SHOT_DIRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.Title)));
  RDB_RIEGL_SHOT_DIRECTION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.Tags)));
  RDB_RIEGL_SHOT_DIRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.Description)));
  RDB_RIEGL_SHOT_DIRECTION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.UnitSymbol)));
  RDB_RIEGL_SHOT_DIRECTION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.NamedValues)));
  RDB_RIEGL_SHOT_DIRECTION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION.LodSettings)));

  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Name)));
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Title)));
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Tags)));
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.Description)));
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.UnitSymbol)));
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.NamedValues)));
  RDB_RIEGL_SHOT_DIRECTION_LEVELLED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_DIRECTION_LEVELLED.LodSettings)));

  RDB_RIEGL_SURFACE_NORMAL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.Name)));
  RDB_RIEGL_SURFACE_NORMAL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.Title)));
  RDB_RIEGL_SURFACE_NORMAL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.Tags)));
  RDB_RIEGL_SURFACE_NORMAL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.Description)));
  RDB_RIEGL_SURFACE_NORMAL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.UnitSymbol)));
  RDB_RIEGL_SURFACE_NORMAL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.NamedValues)));
  RDB_RIEGL_SURFACE_NORMAL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SURFACE_NORMAL.LodSettings)));

  RDB_RIEGL_POINT_VECTOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.Name)));
  RDB_RIEGL_POINT_VECTOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.Title)));
  RDB_RIEGL_POINT_VECTOR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.Tags)));
  RDB_RIEGL_POINT_VECTOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.Description)));
  RDB_RIEGL_POINT_VECTOR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.UnitSymbol)));
  RDB_RIEGL_POINT_VECTOR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.NamedValues)));
  RDB_RIEGL_POINT_VECTOR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_VECTOR.LodSettings)));

  RDB_RIEGL_PLANE_UP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.Name)));
  RDB_RIEGL_PLANE_UP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.Title)));
  RDB_RIEGL_PLANE_UP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.Tags)));
  RDB_RIEGL_PLANE_UP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.Description)));
  RDB_RIEGL_PLANE_UP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.UnitSymbol)));
  RDB_RIEGL_PLANE_UP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.NamedValues)));
  RDB_RIEGL_PLANE_UP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_UP.LodSettings)));

  RDB_RIEGL_PLANE_COG_LINK.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.Name)));
  RDB_RIEGL_PLANE_COG_LINK.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.Title)));
  RDB_RIEGL_PLANE_COG_LINK.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.Tags)));
  RDB_RIEGL_PLANE_COG_LINK.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.Description)));
  RDB_RIEGL_PLANE_COG_LINK.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.UnitSymbol)));
  RDB_RIEGL_PLANE_COG_LINK.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.NamedValues)));
  RDB_RIEGL_PLANE_COG_LINK.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COG_LINK.LodSettings)));

  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Name)));
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Title)));
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Tags)));
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.Description)));
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.UnitSymbol)));
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.NamedValues)));
  RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LINK_VECTOR.LodSettings)));

  RDB_RIEGL_PCA_AXIS_MIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.Name)));
  RDB_RIEGL_PCA_AXIS_MIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.Title)));
  RDB_RIEGL_PCA_AXIS_MIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.Tags)));
  RDB_RIEGL_PCA_AXIS_MIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.Description)));
  RDB_RIEGL_PCA_AXIS_MIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.UnitSymbol)));
  RDB_RIEGL_PCA_AXIS_MIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.NamedValues)));
  RDB_RIEGL_PCA_AXIS_MIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MIN.LodSettings)));

  RDB_RIEGL_PCA_AXIS_MAX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.Name)));
  RDB_RIEGL_PCA_AXIS_MAX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.Title)));
  RDB_RIEGL_PCA_AXIS_MAX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.Tags)));
  RDB_RIEGL_PCA_AXIS_MAX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.Description)));
  RDB_RIEGL_PCA_AXIS_MAX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.UnitSymbol)));
  RDB_RIEGL_PCA_AXIS_MAX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.NamedValues)));
  RDB_RIEGL_PCA_AXIS_MAX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_AXIS_MAX.LodSettings)));

  RDB_RIEGL_MODEL_CS_AXIS_X.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.Name)));
  RDB_RIEGL_MODEL_CS_AXIS_X.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.Title)));
  RDB_RIEGL_MODEL_CS_AXIS_X.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.Tags)));
  RDB_RIEGL_MODEL_CS_AXIS_X.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.Description)));
  RDB_RIEGL_MODEL_CS_AXIS_X.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.UnitSymbol)));
  RDB_RIEGL_MODEL_CS_AXIS_X.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.NamedValues)));
  RDB_RIEGL_MODEL_CS_AXIS_X.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_X.LodSettings)));

  RDB_RIEGL_MODEL_CS_AXIS_Y.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.Name)));
  RDB_RIEGL_MODEL_CS_AXIS_Y.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.Title)));
  RDB_RIEGL_MODEL_CS_AXIS_Y.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.Tags)));
  RDB_RIEGL_MODEL_CS_AXIS_Y.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.Description)));
  RDB_RIEGL_MODEL_CS_AXIS_Y.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.UnitSymbol)));
  RDB_RIEGL_MODEL_CS_AXIS_Y.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.NamedValues)));
  RDB_RIEGL_MODEL_CS_AXIS_Y.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Y.LodSettings)));

  RDB_RIEGL_MODEL_CS_AXIS_Z.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.Name)));
  RDB_RIEGL_MODEL_CS_AXIS_Z.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.Title)));
  RDB_RIEGL_MODEL_CS_AXIS_Z.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.Tags)));
  RDB_RIEGL_MODEL_CS_AXIS_Z.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.Description)));
  RDB_RIEGL_MODEL_CS_AXIS_Z.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.UnitSymbol)));
  RDB_RIEGL_MODEL_CS_AXIS_Z.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.NamedValues)));
  RDB_RIEGL_MODEL_CS_AXIS_Z.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_CS_AXIS_Z.LodSettings)));

  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Name)));
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Title)));
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Tags)));
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.Description)));
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.UnitSymbol)));
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.NamedValues)));
  RDB_RIEGL_CYLINDER_AXIS_DIRECTION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_AXIS_DIRECTION.LodSettings)));

  RDB_RIEGL_ACCELEROMETER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.Name)));
  RDB_RIEGL_ACCELEROMETER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.Title)));
  RDB_RIEGL_ACCELEROMETER.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.Tags)));
  RDB_RIEGL_ACCELEROMETER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.Description)));
  RDB_RIEGL_ACCELEROMETER.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.UnitSymbol)));
  RDB_RIEGL_ACCELEROMETER.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.NamedValues)));
  RDB_RIEGL_ACCELEROMETER.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER.LodSettings)));

  RDB_RIEGL_ACCELEROMETER_RAW.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.Name)));
  RDB_RIEGL_ACCELEROMETER_RAW.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.Title)));
  RDB_RIEGL_ACCELEROMETER_RAW.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.Tags)));
  RDB_RIEGL_ACCELEROMETER_RAW.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.Description)));
  RDB_RIEGL_ACCELEROMETER_RAW.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.UnitSymbol)));
  RDB_RIEGL_ACCELEROMETER_RAW.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.NamedValues)));
  RDB_RIEGL_ACCELEROMETER_RAW.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_RAW.LodSettings)));

  RDB_RIEGL_ACCELEROMETER_BIAS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.Name)));
  RDB_RIEGL_ACCELEROMETER_BIAS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.Title)));
  RDB_RIEGL_ACCELEROMETER_BIAS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.Tags)));
  RDB_RIEGL_ACCELEROMETER_BIAS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.Description)));
  RDB_RIEGL_ACCELEROMETER_BIAS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.UnitSymbol)));
  RDB_RIEGL_ACCELEROMETER_BIAS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.NamedValues)));
  RDB_RIEGL_ACCELEROMETER_BIAS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_BIAS.LodSettings)));

  RDB_RIEGL_ACCELEROMETER_SCALE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.Name)));
  RDB_RIEGL_ACCELEROMETER_SCALE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.Title)));
  RDB_RIEGL_ACCELEROMETER_SCALE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.Tags)));
  RDB_RIEGL_ACCELEROMETER_SCALE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.Description)));
  RDB_RIEGL_ACCELEROMETER_SCALE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.UnitSymbol)));
  RDB_RIEGL_ACCELEROMETER_SCALE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.NamedValues)));
  RDB_RIEGL_ACCELEROMETER_SCALE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACCELEROMETER_SCALE.LodSettings)));

  RDB_RIEGL_GYROSCOPE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.Name)));
  RDB_RIEGL_GYROSCOPE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.Title)));
  RDB_RIEGL_GYROSCOPE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.Tags)));
  RDB_RIEGL_GYROSCOPE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.Description)));
  RDB_RIEGL_GYROSCOPE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.UnitSymbol)));
  RDB_RIEGL_GYROSCOPE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.NamedValues)));
  RDB_RIEGL_GYROSCOPE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE.LodSettings)));

  RDB_RIEGL_GYROSCOPE_RAW.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.Name)));
  RDB_RIEGL_GYROSCOPE_RAW.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.Title)));
  RDB_RIEGL_GYROSCOPE_RAW.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.Tags)));
  RDB_RIEGL_GYROSCOPE_RAW.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.Description)));
  RDB_RIEGL_GYROSCOPE_RAW.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.UnitSymbol)));
  RDB_RIEGL_GYROSCOPE_RAW.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.NamedValues)));
  RDB_RIEGL_GYROSCOPE_RAW.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_RAW.LodSettings)));

  RDB_RIEGL_GYROSCOPE_BIAS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.Name)));
  RDB_RIEGL_GYROSCOPE_BIAS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.Title)));
  RDB_RIEGL_GYROSCOPE_BIAS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.Tags)));
  RDB_RIEGL_GYROSCOPE_BIAS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.Description)));
  RDB_RIEGL_GYROSCOPE_BIAS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.UnitSymbol)));
  RDB_RIEGL_GYROSCOPE_BIAS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.NamedValues)));
  RDB_RIEGL_GYROSCOPE_BIAS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_BIAS.LodSettings)));

  RDB_RIEGL_GYROSCOPE_SCALE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.Name)));
  RDB_RIEGL_GYROSCOPE_SCALE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.Title)));
  RDB_RIEGL_GYROSCOPE_SCALE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.Tags)));
  RDB_RIEGL_GYROSCOPE_SCALE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.Description)));
  RDB_RIEGL_GYROSCOPE_SCALE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.UnitSymbol)));
  RDB_RIEGL_GYROSCOPE_SCALE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.NamedValues)));
  RDB_RIEGL_GYROSCOPE_SCALE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GYROSCOPE_SCALE.LodSettings)));

  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Name)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Title)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Tags)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.Description)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.UnitSymbol)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.NamedValues)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR.LodSettings)));

  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Name)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Title)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Tags)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.Description)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.UnitSymbol)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.NamedValues)));
  RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW.LodSettings)));

  RDB_RIEGL_POF_LATITUDE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.Name)));
  RDB_RIEGL_POF_LATITUDE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.Title)));
  RDB_RIEGL_POF_LATITUDE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.Tags)));
  RDB_RIEGL_POF_LATITUDE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.Description)));
  RDB_RIEGL_POF_LATITUDE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.UnitSymbol)));
  RDB_RIEGL_POF_LATITUDE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.NamedValues)));
  RDB_RIEGL_POF_LATITUDE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LATITUDE.LodSettings)));

  RDB_RIEGL_POF_LONGITUDE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.Name)));
  RDB_RIEGL_POF_LONGITUDE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.Title)));
  RDB_RIEGL_POF_LONGITUDE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.Tags)));
  RDB_RIEGL_POF_LONGITUDE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.Description)));
  RDB_RIEGL_POF_LONGITUDE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.UnitSymbol)));
  RDB_RIEGL_POF_LONGITUDE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.NamedValues)));
  RDB_RIEGL_POF_LONGITUDE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_LONGITUDE.LodSettings)));

  RDB_RIEGL_POF_HEIGHT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.Name)));
  RDB_RIEGL_POF_HEIGHT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.Title)));
  RDB_RIEGL_POF_HEIGHT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.Tags)));
  RDB_RIEGL_POF_HEIGHT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.Description)));
  RDB_RIEGL_POF_HEIGHT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.UnitSymbol)));
  RDB_RIEGL_POF_HEIGHT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.NamedValues)));
  RDB_RIEGL_POF_HEIGHT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HEIGHT.LodSettings)));

  RDB_RIEGL_POF_UNDULATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.Name)));
  RDB_RIEGL_POF_UNDULATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.Title)));
  RDB_RIEGL_POF_UNDULATION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.Tags)));
  RDB_RIEGL_POF_UNDULATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.Description)));
  RDB_RIEGL_POF_UNDULATION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.UnitSymbol)));
  RDB_RIEGL_POF_UNDULATION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.NamedValues)));
  RDB_RIEGL_POF_UNDULATION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_UNDULATION.LodSettings)));

  RDB_RIEGL_POF_ROLL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.Name)));
  RDB_RIEGL_POF_ROLL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.Title)));
  RDB_RIEGL_POF_ROLL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.Tags)));
  RDB_RIEGL_POF_ROLL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.Description)));
  RDB_RIEGL_POF_ROLL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.UnitSymbol)));
  RDB_RIEGL_POF_ROLL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.NamedValues)));
  RDB_RIEGL_POF_ROLL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL.LodSettings)));

  RDB_RIEGL_POF_PITCH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.Name)));
  RDB_RIEGL_POF_PITCH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.Title)));
  RDB_RIEGL_POF_PITCH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.Tags)));
  RDB_RIEGL_POF_PITCH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.Description)));
  RDB_RIEGL_POF_PITCH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.UnitSymbol)));
  RDB_RIEGL_POF_PITCH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.NamedValues)));
  RDB_RIEGL_POF_PITCH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH.LodSettings)));

  RDB_RIEGL_POF_YAW.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.Name)));
  RDB_RIEGL_POF_YAW.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.Title)));
  RDB_RIEGL_POF_YAW.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.Tags)));
  RDB_RIEGL_POF_YAW.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.Description)));
  RDB_RIEGL_POF_YAW.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.UnitSymbol)));
  RDB_RIEGL_POF_YAW.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.NamedValues)));
  RDB_RIEGL_POF_YAW.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW.LodSettings)));

  RDB_RIEGL_POF_COURSE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.Name)));
  RDB_RIEGL_POF_COURSE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.Title)));
  RDB_RIEGL_POF_COURSE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.Tags)));
  RDB_RIEGL_POF_COURSE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.Description)));
  RDB_RIEGL_POF_COURSE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.UnitSymbol)));
  RDB_RIEGL_POF_COURSE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.NamedValues)));
  RDB_RIEGL_POF_COURSE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_COURSE.LodSettings)));

  RDB_RIEGL_POF_XYZ.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.Name)));
  RDB_RIEGL_POF_XYZ.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.Title)));
  RDB_RIEGL_POF_XYZ.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.Tags)));
  RDB_RIEGL_POF_XYZ.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.Description)));
  RDB_RIEGL_POF_XYZ.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.UnitSymbol)));
  RDB_RIEGL_POF_XYZ.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.NamedValues)));
  RDB_RIEGL_POF_XYZ.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ.LodSettings)));

  RDB_RIEGL_POF_ROLL_NED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.Name)));
  RDB_RIEGL_POF_ROLL_NED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.Title)));
  RDB_RIEGL_POF_ROLL_NED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.Tags)));
  RDB_RIEGL_POF_ROLL_NED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.Description)));
  RDB_RIEGL_POF_ROLL_NED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.UnitSymbol)));
  RDB_RIEGL_POF_ROLL_NED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.NamedValues)));
  RDB_RIEGL_POF_ROLL_NED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_NED.LodSettings)));

  RDB_RIEGL_POF_PITCH_NED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.Name)));
  RDB_RIEGL_POF_PITCH_NED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.Title)));
  RDB_RIEGL_POF_PITCH_NED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.Tags)));
  RDB_RIEGL_POF_PITCH_NED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.Description)));
  RDB_RIEGL_POF_PITCH_NED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.UnitSymbol)));
  RDB_RIEGL_POF_PITCH_NED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.NamedValues)));
  RDB_RIEGL_POF_PITCH_NED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_NED.LodSettings)));

  RDB_RIEGL_POF_YAW_NED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.Name)));
  RDB_RIEGL_POF_YAW_NED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.Title)));
  RDB_RIEGL_POF_YAW_NED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.Tags)));
  RDB_RIEGL_POF_YAW_NED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.Description)));
  RDB_RIEGL_POF_YAW_NED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.UnitSymbol)));
  RDB_RIEGL_POF_YAW_NED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.NamedValues)));
  RDB_RIEGL_POF_YAW_NED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_NED.LodSettings)));

  RDB_RIEGL_POF_XYZ_VELOCITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.Name)));
  RDB_RIEGL_POF_XYZ_VELOCITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.Title)));
  RDB_RIEGL_POF_XYZ_VELOCITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.Tags)));
  RDB_RIEGL_POF_XYZ_VELOCITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.Description)));
  RDB_RIEGL_POF_XYZ_VELOCITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.UnitSymbol)));
  RDB_RIEGL_POF_XYZ_VELOCITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.NamedValues)));
  RDB_RIEGL_POF_XYZ_VELOCITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_VELOCITY.LodSettings)));

  RDB_RIEGL_POF_XYZ_NED_VELOCITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.Name)));
  RDB_RIEGL_POF_XYZ_NED_VELOCITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.Title)));
  RDB_RIEGL_POF_XYZ_NED_VELOCITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.Tags)));
  RDB_RIEGL_POF_XYZ_NED_VELOCITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.Description)));
  RDB_RIEGL_POF_XYZ_NED_VELOCITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.UnitSymbol)));
  RDB_RIEGL_POF_XYZ_NED_VELOCITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.NamedValues)));
  RDB_RIEGL_POF_XYZ_NED_VELOCITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_NED_VELOCITY.LodSettings)));

  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Name)));
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Title)));
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Tags)));
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.Description)));
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.UnitSymbol)));
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.NamedValues)));
  RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_XYZ_LVCS_VELOCITY.LodSettings)));

  RDB_RIEGL_POF_ROLL_VELOCITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.Name)));
  RDB_RIEGL_POF_ROLL_VELOCITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.Title)));
  RDB_RIEGL_POF_ROLL_VELOCITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.Tags)));
  RDB_RIEGL_POF_ROLL_VELOCITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.Description)));
  RDB_RIEGL_POF_ROLL_VELOCITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.UnitSymbol)));
  RDB_RIEGL_POF_ROLL_VELOCITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.NamedValues)));
  RDB_RIEGL_POF_ROLL_VELOCITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ROLL_VELOCITY.LodSettings)));

  RDB_RIEGL_POF_PITCH_VELOCITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.Name)));
  RDB_RIEGL_POF_PITCH_VELOCITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.Title)));
  RDB_RIEGL_POF_PITCH_VELOCITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.Tags)));
  RDB_RIEGL_POF_PITCH_VELOCITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.Description)));
  RDB_RIEGL_POF_PITCH_VELOCITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.UnitSymbol)));
  RDB_RIEGL_POF_PITCH_VELOCITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.NamedValues)));
  RDB_RIEGL_POF_PITCH_VELOCITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PITCH_VELOCITY.LodSettings)));

  RDB_RIEGL_POF_YAW_VELOCITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.Name)));
  RDB_RIEGL_POF_YAW_VELOCITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.Title)));
  RDB_RIEGL_POF_YAW_VELOCITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.Tags)));
  RDB_RIEGL_POF_YAW_VELOCITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.Description)));
  RDB_RIEGL_POF_YAW_VELOCITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.UnitSymbol)));
  RDB_RIEGL_POF_YAW_VELOCITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.NamedValues)));
  RDB_RIEGL_POF_YAW_VELOCITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_YAW_VELOCITY.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_NED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.Name)));
  RDB_RIEGL_POF_VELOCITY_NED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.Title)));
  RDB_RIEGL_POF_VELOCITY_NED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.Tags)));
  RDB_RIEGL_POF_VELOCITY_NED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.Description)));
  RDB_RIEGL_POF_VELOCITY_NED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_NED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_NED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_NED.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Name)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Title)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Tags)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.Description)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Name)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Title)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Tags)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.Description)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Name)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Title)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Tags)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.Description)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Name)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Title)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Tags)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.Description)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Name)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Title)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Tags)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.Description)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN.LodSettings)));

  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Name)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Title)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Tags)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.Description)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.UnitSymbol)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.NamedValues)));
  RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH.LodSettings)));

  RDB_RIEGL_HYDRO_INTERSECTION_POINT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.Name)));
  RDB_RIEGL_HYDRO_INTERSECTION_POINT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.Title)));
  RDB_RIEGL_HYDRO_INTERSECTION_POINT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.Tags)));
  RDB_RIEGL_HYDRO_INTERSECTION_POINT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.Description)));
  RDB_RIEGL_HYDRO_INTERSECTION_POINT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.UnitSymbol)));
  RDB_RIEGL_HYDRO_INTERSECTION_POINT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.NamedValues)));
  RDB_RIEGL_HYDRO_INTERSECTION_POINT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_POINT.LodSettings)));

  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Name)));
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Title)));
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Tags)));
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.Description)));
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.UnitSymbol)));
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.NamedValues)));
  RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_INTERSECTION_NORMAL.LodSettings)));

  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Name)));
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Title)));
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Tags)));
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.Description)));
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.UnitSymbol)));
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.NamedValues)));
  RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_WSM_UNCERTAINTY.LodSettings)));

  RDB_RIEGL_XYZ_ACCURACIES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.Name)));
  RDB_RIEGL_XYZ_ACCURACIES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.Title)));
  RDB_RIEGL_XYZ_ACCURACIES.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.Tags)));
  RDB_RIEGL_XYZ_ACCURACIES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.Description)));
  RDB_RIEGL_XYZ_ACCURACIES.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.UnitSymbol)));
  RDB_RIEGL_XYZ_ACCURACIES.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.NamedValues)));
  RDB_RIEGL_XYZ_ACCURACIES.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_XYZ_ACCURACIES.LodSettings)));

  RDB_RIEGL_ZENITH_VECTOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.Name)));
  RDB_RIEGL_ZENITH_VECTOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.Title)));
  RDB_RIEGL_ZENITH_VECTOR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.Tags)));
  RDB_RIEGL_ZENITH_VECTOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.Description)));
  RDB_RIEGL_ZENITH_VECTOR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.UnitSymbol)));
  RDB_RIEGL_ZENITH_VECTOR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.NamedValues)));
  RDB_RIEGL_ZENITH_VECTOR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ZENITH_VECTOR.LodSettings)));

  RDB_RIEGL_SHOT_TIMESTAMP_HR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.Name)));
  RDB_RIEGL_SHOT_TIMESTAMP_HR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.Title)));
  RDB_RIEGL_SHOT_TIMESTAMP_HR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.Tags)));
  RDB_RIEGL_SHOT_TIMESTAMP_HR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.Description)));
  RDB_RIEGL_SHOT_TIMESTAMP_HR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.UnitSymbol)));
  RDB_RIEGL_SHOT_TIMESTAMP_HR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.NamedValues)));
  RDB_RIEGL_SHOT_TIMESTAMP_HR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_TIMESTAMP_HR.LodSettings)));

  RDB_RIEGL_TIMESTAMP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.Name)));
  RDB_RIEGL_TIMESTAMP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.Title)));
  RDB_RIEGL_TIMESTAMP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.Tags)));
  RDB_RIEGL_TIMESTAMP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.Description)));
  RDB_RIEGL_TIMESTAMP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.UnitSymbol)));
  RDB_RIEGL_TIMESTAMP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.NamedValues)));
  RDB_RIEGL_TIMESTAMP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP.LodSettings)));

  RDB_RIEGL_WFM_SBL_TIME_OFFSET.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.Name)));
  RDB_RIEGL_WFM_SBL_TIME_OFFSET.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.Title)));
  RDB_RIEGL_WFM_SBL_TIME_OFFSET.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.Tags)));
  RDB_RIEGL_WFM_SBL_TIME_OFFSET.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.Description)));
  RDB_RIEGL_WFM_SBL_TIME_OFFSET.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_TIME_OFFSET.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.NamedValues)));
  RDB_RIEGL_WFM_SBL_TIME_OFFSET.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_TIME_OFFSET.LodSettings)));

  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Name)));
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Title)));
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Tags)));
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.Description)));
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.UnitSymbol)));
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.NamedValues)));
  RDB_RIEGL_WFM_ECHO_TIME_OFFSET.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_ECHO_TIME_OFFSET.LodSettings)));

  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Name)));
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Title)));
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Tags)));
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.Description)));
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.UnitSymbol)));
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.NamedValues)));
  RDB_RIEGL_PPS_TIMESTAMP_EXTERN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_EXTERN.LodSettings)));

  RDB_RIEGL_PPS_TIMESTAMP_INTERN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.Name)));
  RDB_RIEGL_PPS_TIMESTAMP_INTERN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.Title)));
  RDB_RIEGL_PPS_TIMESTAMP_INTERN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.Tags)));
  RDB_RIEGL_PPS_TIMESTAMP_INTERN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.Description)));
  RDB_RIEGL_PPS_TIMESTAMP_INTERN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.UnitSymbol)));
  RDB_RIEGL_PPS_TIMESTAMP_INTERN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.NamedValues)));
  RDB_RIEGL_PPS_TIMESTAMP_INTERN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PPS_TIMESTAMP_INTERN.LodSettings)));

  RDB_RIEGL_EVENT_TIMESTAMP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.Name)));
  RDB_RIEGL_EVENT_TIMESTAMP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.Title)));
  RDB_RIEGL_EVENT_TIMESTAMP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.Tags)));
  RDB_RIEGL_EVENT_TIMESTAMP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.Description)));
  RDB_RIEGL_EVENT_TIMESTAMP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.UnitSymbol)));
  RDB_RIEGL_EVENT_TIMESTAMP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.NamedValues)));
  RDB_RIEGL_EVENT_TIMESTAMP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_TIMESTAMP.LodSettings)));

  RDB_RIEGL_TIMESTAMP_MIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.Name)));
  RDB_RIEGL_TIMESTAMP_MIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.Title)));
  RDB_RIEGL_TIMESTAMP_MIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.Tags)));
  RDB_RIEGL_TIMESTAMP_MIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.Description)));
  RDB_RIEGL_TIMESTAMP_MIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.UnitSymbol)));
  RDB_RIEGL_TIMESTAMP_MIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.NamedValues)));
  RDB_RIEGL_TIMESTAMP_MIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MIN.LodSettings)));

  RDB_RIEGL_TIMESTAMP_MAX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.Name)));
  RDB_RIEGL_TIMESTAMP_MAX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.Title)));
  RDB_RIEGL_TIMESTAMP_MAX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.Tags)));
  RDB_RIEGL_TIMESTAMP_MAX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.Description)));
  RDB_RIEGL_TIMESTAMP_MAX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.UnitSymbol)));
  RDB_RIEGL_TIMESTAMP_MAX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.NamedValues)));
  RDB_RIEGL_TIMESTAMP_MAX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TIMESTAMP_MAX.LodSettings)));

  RDB_RIEGL_POF_TIMESTAMP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.Name)));
  RDB_RIEGL_POF_TIMESTAMP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.Title)));
  RDB_RIEGL_POF_TIMESTAMP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.Tags)));
  RDB_RIEGL_POF_TIMESTAMP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.Description)));
  RDB_RIEGL_POF_TIMESTAMP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.UnitSymbol)));
  RDB_RIEGL_POF_TIMESTAMP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.NamedValues)));
  RDB_RIEGL_POF_TIMESTAMP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_TIMESTAMP.LodSettings)));

  RDB_RIEGL_ACQUISITION_DATE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.Name)));
  RDB_RIEGL_ACQUISITION_DATE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.Title)));
  RDB_RIEGL_ACQUISITION_DATE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.Tags)));
  RDB_RIEGL_ACQUISITION_DATE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.Description)));
  RDB_RIEGL_ACQUISITION_DATE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.UnitSymbol)));
  RDB_RIEGL_ACQUISITION_DATE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.NamedValues)));
  RDB_RIEGL_ACQUISITION_DATE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ACQUISITION_DATE.LodSettings)));

  RDB_RIEGL_REFLECTANCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.Name)));
  RDB_RIEGL_REFLECTANCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.Title)));
  RDB_RIEGL_REFLECTANCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.Tags)));
  RDB_RIEGL_REFLECTANCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.Description)));
  RDB_RIEGL_REFLECTANCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.UnitSymbol)));
  RDB_RIEGL_REFLECTANCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.NamedValues)));
  RDB_RIEGL_REFLECTANCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFLECTANCE.LodSettings)));

  RDB_RIEGL_AMPLITUDE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.Name)));
  RDB_RIEGL_AMPLITUDE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.Title)));
  RDB_RIEGL_AMPLITUDE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.Tags)));
  RDB_RIEGL_AMPLITUDE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.Description)));
  RDB_RIEGL_AMPLITUDE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.UnitSymbol)));
  RDB_RIEGL_AMPLITUDE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.NamedValues)));
  RDB_RIEGL_AMPLITUDE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_AMPLITUDE.LodSettings)));

  RDB_RIEGL_INTENSITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.Name)));
  RDB_RIEGL_INTENSITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.Title)));
  RDB_RIEGL_INTENSITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.Tags)));
  RDB_RIEGL_INTENSITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.Description)));
  RDB_RIEGL_INTENSITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.UnitSymbol)));
  RDB_RIEGL_INTENSITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.NamedValues)));
  RDB_RIEGL_INTENSITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_INTENSITY.LodSettings)));

  RDB_RIEGL_GAIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.Name)));
  RDB_RIEGL_GAIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.Title)));
  RDB_RIEGL_GAIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.Tags)));
  RDB_RIEGL_GAIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.Description)));
  RDB_RIEGL_GAIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.UnitSymbol)));
  RDB_RIEGL_GAIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.NamedValues)));
  RDB_RIEGL_GAIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_GAIN.LodSettings)));

  RDB_RIEGL_DEVIATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.Name)));
  RDB_RIEGL_DEVIATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.Title)));
  RDB_RIEGL_DEVIATION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.Tags)));
  RDB_RIEGL_DEVIATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.Description)));
  RDB_RIEGL_DEVIATION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.UnitSymbol)));
  RDB_RIEGL_DEVIATION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.NamedValues)));
  RDB_RIEGL_DEVIATION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DEVIATION.LodSettings)));

  RDB_RIEGL_PULSE_WIDTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.Name)));
  RDB_RIEGL_PULSE_WIDTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.Title)));
  RDB_RIEGL_PULSE_WIDTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.Tags)));
  RDB_RIEGL_PULSE_WIDTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.Description)));
  RDB_RIEGL_PULSE_WIDTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.UnitSymbol)));
  RDB_RIEGL_PULSE_WIDTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.NamedValues)));
  RDB_RIEGL_PULSE_WIDTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PULSE_WIDTH.LodSettings)));

  RDB_RIEGL_CLASS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.Name)));
  RDB_RIEGL_CLASS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.Title)));
  RDB_RIEGL_CLASS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.Tags)));
  RDB_RIEGL_CLASS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.Description)));
  RDB_RIEGL_CLASS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.UnitSymbol)));
  RDB_RIEGL_CLASS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.NamedValues)));
  RDB_RIEGL_CLASS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CLASS.LodSettings)));

  RDB_RIEGL_RGBA.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.Name)));
  RDB_RIEGL_RGBA.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.Title)));
  RDB_RIEGL_RGBA.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.Tags)));
  RDB_RIEGL_RGBA.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.Description)));
  RDB_RIEGL_RGBA.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.UnitSymbol)));
  RDB_RIEGL_RGBA.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.NamedValues)));
  RDB_RIEGL_RGBA.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RGBA.LodSettings)));

  RDB_RIEGL_NIR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.Name)));
  RDB_RIEGL_NIR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.Title)));
  RDB_RIEGL_NIR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.Tags)));
  RDB_RIEGL_NIR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.Description)));
  RDB_RIEGL_NIR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.UnitSymbol)));
  RDB_RIEGL_NIR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.NamedValues)));
  RDB_RIEGL_NIR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NIR.LodSettings)));

  RDB_RIEGL_TEMPERATURE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.Name)));
  RDB_RIEGL_TEMPERATURE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.Title)));
  RDB_RIEGL_TEMPERATURE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.Tags)));
  RDB_RIEGL_TEMPERATURE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.Description)));
  RDB_RIEGL_TEMPERATURE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.UnitSymbol)));
  RDB_RIEGL_TEMPERATURE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.NamedValues)));
  RDB_RIEGL_TEMPERATURE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE.LodSettings)));

  RDB_RIEGL_DMI_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.Name)));
  RDB_RIEGL_DMI_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.Title)));
  RDB_RIEGL_DMI_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.Tags)));
  RDB_RIEGL_DMI_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.Description)));
  RDB_RIEGL_DMI_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.UnitSymbol)));
  RDB_RIEGL_DMI_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.NamedValues)));
  RDB_RIEGL_DMI_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_COUNT.LodSettings)));

  RDB_RIEGL_DMI_PATH_LENGTH_RAW.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.Name)));
  RDB_RIEGL_DMI_PATH_LENGTH_RAW.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.Title)));
  RDB_RIEGL_DMI_PATH_LENGTH_RAW.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.Tags)));
  RDB_RIEGL_DMI_PATH_LENGTH_RAW.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.Description)));
  RDB_RIEGL_DMI_PATH_LENGTH_RAW.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.UnitSymbol)));
  RDB_RIEGL_DMI_PATH_LENGTH_RAW.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.NamedValues)));
  RDB_RIEGL_DMI_PATH_LENGTH_RAW.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_PATH_LENGTH_RAW.LodSettings)));

  RDB_RIEGL_DMI_SCALE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.Name)));
  RDB_RIEGL_DMI_SCALE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.Title)));
  RDB_RIEGL_DMI_SCALE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.Tags)));
  RDB_RIEGL_DMI_SCALE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.Description)));
  RDB_RIEGL_DMI_SCALE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.UnitSymbol)));
  RDB_RIEGL_DMI_SCALE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.NamedValues)));
  RDB_RIEGL_DMI_SCALE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DMI_SCALE.LodSettings)));

  RDB_RIEGL_MTA_ZONE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.Name)));
  RDB_RIEGL_MTA_ZONE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.Title)));
  RDB_RIEGL_MTA_ZONE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.Tags)));
  RDB_RIEGL_MTA_ZONE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.Description)));
  RDB_RIEGL_MTA_ZONE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.UnitSymbol)));
  RDB_RIEGL_MTA_ZONE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.NamedValues)));
  RDB_RIEGL_MTA_ZONE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_ZONE.LodSettings)));

  RDB_RIEGL_MTA_UNCERTAIN_POINT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.Name)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.Title)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.Tags)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.Description)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.UnitSymbol)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.NamedValues)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT.LodSettings)));

  RDB_RIEGL_FWA.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.Name)));
  RDB_RIEGL_FWA.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.Title)));
  RDB_RIEGL_FWA.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.Tags)));
  RDB_RIEGL_FWA.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.Description)));
  RDB_RIEGL_FWA.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.UnitSymbol)));
  RDB_RIEGL_FWA.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.NamedValues)));
  RDB_RIEGL_FWA.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FWA.LodSettings)));

  RDB_RIEGL_BACKGROUND_RADIATION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.Name)));
  RDB_RIEGL_BACKGROUND_RADIATION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.Title)));
  RDB_RIEGL_BACKGROUND_RADIATION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.Tags)));
  RDB_RIEGL_BACKGROUND_RADIATION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.Description)));
  RDB_RIEGL_BACKGROUND_RADIATION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.UnitSymbol)));
  RDB_RIEGL_BACKGROUND_RADIATION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.NamedValues)));
  RDB_RIEGL_BACKGROUND_RADIATION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION.LodSettings)));

  RDB_RIEGL_BACKGROUND_RADIATION_SI.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.Name)));
  RDB_RIEGL_BACKGROUND_RADIATION_SI.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.Title)));
  RDB_RIEGL_BACKGROUND_RADIATION_SI.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.Tags)));
  RDB_RIEGL_BACKGROUND_RADIATION_SI.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.Description)));
  RDB_RIEGL_BACKGROUND_RADIATION_SI.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.UnitSymbol)));
  RDB_RIEGL_BACKGROUND_RADIATION_SI.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.NamedValues)));
  RDB_RIEGL_BACKGROUND_RADIATION_SI.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_SI.LodSettings)));

  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Name)));
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Title)));
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Tags)));
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.Description)));
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.UnitSymbol)));
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.NamedValues)));
  RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BACKGROUND_RADIATION_INGAAS.LodSettings)));

  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Name)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Title)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Tags)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.Description)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.UnitSymbol)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.NamedValues)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_SI.LodSettings)));

  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Name)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Title)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Tags)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.Description)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.UnitSymbol)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.NamedValues)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS.LodSettings)));

  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Name)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Title)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Tags)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.Description)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.UnitSymbol)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.NamedValues)));
  RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI.LodSettings)));

  RDB_RIEGL_TARGET_INDEX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.Name)));
  RDB_RIEGL_TARGET_INDEX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.Title)));
  RDB_RIEGL_TARGET_INDEX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.Tags)));
  RDB_RIEGL_TARGET_INDEX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.Description)));
  RDB_RIEGL_TARGET_INDEX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.UnitSymbol)));
  RDB_RIEGL_TARGET_INDEX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.NamedValues)));
  RDB_RIEGL_TARGET_INDEX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_INDEX.LodSettings)));

  RDB_RIEGL_TARGET_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.Name)));
  RDB_RIEGL_TARGET_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.Title)));
  RDB_RIEGL_TARGET_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.Tags)));
  RDB_RIEGL_TARGET_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.Description)));
  RDB_RIEGL_TARGET_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.UnitSymbol)));
  RDB_RIEGL_TARGET_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.NamedValues)));
  RDB_RIEGL_TARGET_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_COUNT.LodSettings)));

  RDB_RIEGL_TARGET_TYPE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.Name)));
  RDB_RIEGL_TARGET_TYPE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.Title)));
  RDB_RIEGL_TARGET_TYPE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.Tags)));
  RDB_RIEGL_TARGET_TYPE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.Description)));
  RDB_RIEGL_TARGET_TYPE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.UnitSymbol)));
  RDB_RIEGL_TARGET_TYPE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.NamedValues)));
  RDB_RIEGL_TARGET_TYPE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TARGET_TYPE.LodSettings)));

  RDB_RIEGL_ECHO_FIRST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.Name)));
  RDB_RIEGL_ECHO_FIRST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.Title)));
  RDB_RIEGL_ECHO_FIRST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.Tags)));
  RDB_RIEGL_ECHO_FIRST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.Description)));
  RDB_RIEGL_ECHO_FIRST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.UnitSymbol)));
  RDB_RIEGL_ECHO_FIRST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.NamedValues)));
  RDB_RIEGL_ECHO_FIRST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_FIRST.LodSettings)));

  RDB_RIEGL_ECHO_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.Name)));
  RDB_RIEGL_ECHO_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.Title)));
  RDB_RIEGL_ECHO_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.Tags)));
  RDB_RIEGL_ECHO_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.Description)));
  RDB_RIEGL_ECHO_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.UnitSymbol)));
  RDB_RIEGL_ECHO_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.NamedValues)));
  RDB_RIEGL_ECHO_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ECHO_COUNT.LodSettings)));

  RDB_RIEGL_HEIGHT_CENTER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.Name)));
  RDB_RIEGL_HEIGHT_CENTER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.Title)));
  RDB_RIEGL_HEIGHT_CENTER.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.Tags)));
  RDB_RIEGL_HEIGHT_CENTER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.Description)));
  RDB_RIEGL_HEIGHT_CENTER.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.UnitSymbol)));
  RDB_RIEGL_HEIGHT_CENTER.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.NamedValues)));
  RDB_RIEGL_HEIGHT_CENTER.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_CENTER.LodSettings)));

  RDB_RIEGL_HEIGHT_MEAN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.Name)));
  RDB_RIEGL_HEIGHT_MEAN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.Title)));
  RDB_RIEGL_HEIGHT_MEAN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.Tags)));
  RDB_RIEGL_HEIGHT_MEAN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.Description)));
  RDB_RIEGL_HEIGHT_MEAN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.UnitSymbol)));
  RDB_RIEGL_HEIGHT_MEAN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.NamedValues)));
  RDB_RIEGL_HEIGHT_MEAN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MEAN.LodSettings)));

  RDB_RIEGL_HEIGHT_MIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.Name)));
  RDB_RIEGL_HEIGHT_MIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.Title)));
  RDB_RIEGL_HEIGHT_MIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.Tags)));
  RDB_RIEGL_HEIGHT_MIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.Description)));
  RDB_RIEGL_HEIGHT_MIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.UnitSymbol)));
  RDB_RIEGL_HEIGHT_MIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.NamedValues)));
  RDB_RIEGL_HEIGHT_MIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MIN.LodSettings)));

  RDB_RIEGL_HEIGHT_MAX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.Name)));
  RDB_RIEGL_HEIGHT_MAX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.Title)));
  RDB_RIEGL_HEIGHT_MAX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.Tags)));
  RDB_RIEGL_HEIGHT_MAX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.Description)));
  RDB_RIEGL_HEIGHT_MAX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.UnitSymbol)));
  RDB_RIEGL_HEIGHT_MAX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.NamedValues)));
  RDB_RIEGL_HEIGHT_MAX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HEIGHT_MAX.LodSettings)));

  RDB_RIEGL_POINT_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.Name)));
  RDB_RIEGL_POINT_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.Title)));
  RDB_RIEGL_POINT_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.Tags)));
  RDB_RIEGL_POINT_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.Description)));
  RDB_RIEGL_POINT_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.UnitSymbol)));
  RDB_RIEGL_POINT_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.NamedValues)));
  RDB_RIEGL_POINT_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT.LodSettings)));

  RDB_RIEGL_POINT_COUNT_GRID_CELL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.Name)));
  RDB_RIEGL_POINT_COUNT_GRID_CELL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.Title)));
  RDB_RIEGL_POINT_COUNT_GRID_CELL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.Tags)));
  RDB_RIEGL_POINT_COUNT_GRID_CELL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.Description)));
  RDB_RIEGL_POINT_COUNT_GRID_CELL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.UnitSymbol)));
  RDB_RIEGL_POINT_COUNT_GRID_CELL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.NamedValues)));
  RDB_RIEGL_POINT_COUNT_GRID_CELL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_COUNT_GRID_CELL.LodSettings)));

  RDB_RIEGL_PCA_EXTENTS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.Name)));
  RDB_RIEGL_PCA_EXTENTS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.Title)));
  RDB_RIEGL_PCA_EXTENTS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.Tags)));
  RDB_RIEGL_PCA_EXTENTS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.Description)));
  RDB_RIEGL_PCA_EXTENTS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.UnitSymbol)));
  RDB_RIEGL_PCA_EXTENTS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.NamedValues)));
  RDB_RIEGL_PCA_EXTENTS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_EXTENTS.LodSettings)));

  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Name)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Title)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Tags)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.Description)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.UnitSymbol)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.NamedValues)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL.LodSettings)));

  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Name)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Title)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Tags)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.Description)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.UnitSymbol)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.NamedValues)));
  RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF.LodSettings)));

  RDB_RIEGL_PCA_THICKNESS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.Name)));
  RDB_RIEGL_PCA_THICKNESS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.Title)));
  RDB_RIEGL_PCA_THICKNESS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.Tags)));
  RDB_RIEGL_PCA_THICKNESS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.Description)));
  RDB_RIEGL_PCA_THICKNESS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.UnitSymbol)));
  RDB_RIEGL_PCA_THICKNESS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.NamedValues)));
  RDB_RIEGL_PCA_THICKNESS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_THICKNESS.LodSettings)));

  RDB_RIEGL_PCA_DIAMETER.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.Name)));
  RDB_RIEGL_PCA_DIAMETER.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.Title)));
  RDB_RIEGL_PCA_DIAMETER.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.Tags)));
  RDB_RIEGL_PCA_DIAMETER.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.Description)));
  RDB_RIEGL_PCA_DIAMETER.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.UnitSymbol)));
  RDB_RIEGL_PCA_DIAMETER.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.NamedValues)));
  RDB_RIEGL_PCA_DIAMETER.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PCA_DIAMETER.LodSettings)));

  RDB_RIEGL_EIGENVALUE_NORM_MIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.Name)));
  RDB_RIEGL_EIGENVALUE_NORM_MIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.Title)));
  RDB_RIEGL_EIGENVALUE_NORM_MIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.Tags)));
  RDB_RIEGL_EIGENVALUE_NORM_MIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.Description)));
  RDB_RIEGL_EIGENVALUE_NORM_MIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.UnitSymbol)));
  RDB_RIEGL_EIGENVALUE_NORM_MIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.NamedValues)));
  RDB_RIEGL_EIGENVALUE_NORM_MIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MIN.LodSettings)));

  RDB_RIEGL_EIGENVALUE_NORM_MAX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.Name)));
  RDB_RIEGL_EIGENVALUE_NORM_MAX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.Title)));
  RDB_RIEGL_EIGENVALUE_NORM_MAX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.Tags)));
  RDB_RIEGL_EIGENVALUE_NORM_MAX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.Description)));
  RDB_RIEGL_EIGENVALUE_NORM_MAX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.UnitSymbol)));
  RDB_RIEGL_EIGENVALUE_NORM_MAX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.NamedValues)));
  RDB_RIEGL_EIGENVALUE_NORM_MAX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EIGENVALUE_NORM_MAX.LodSettings)));

  RDB_RIEGL_STD_DEV.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.Name)));
  RDB_RIEGL_STD_DEV.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.Title)));
  RDB_RIEGL_STD_DEV.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.Tags)));
  RDB_RIEGL_STD_DEV.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.Description)));
  RDB_RIEGL_STD_DEV.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.UnitSymbol)));
  RDB_RIEGL_STD_DEV.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.NamedValues)));
  RDB_RIEGL_STD_DEV.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_STD_DEV.LodSettings)));

  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Name)));
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Title)));
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Tags)));
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.Description)));
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.UnitSymbol)));
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.NamedValues)));
  RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CONFIDENCE_NORMAL.LodSettings)));

  RDB_RIEGL_PLANE_SLOPE_CLASS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.Name)));
  RDB_RIEGL_PLANE_SLOPE_CLASS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.Title)));
  RDB_RIEGL_PLANE_SLOPE_CLASS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.Tags)));
  RDB_RIEGL_PLANE_SLOPE_CLASS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.Description)));
  RDB_RIEGL_PLANE_SLOPE_CLASS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.UnitSymbol)));
  RDB_RIEGL_PLANE_SLOPE_CLASS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.NamedValues)));
  RDB_RIEGL_PLANE_SLOPE_CLASS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_SLOPE_CLASS.LodSettings)));

  RDB_RIEGL_PLANE_OCCUPANCY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.Name)));
  RDB_RIEGL_PLANE_OCCUPANCY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.Title)));
  RDB_RIEGL_PLANE_OCCUPANCY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.Tags)));
  RDB_RIEGL_PLANE_OCCUPANCY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.Description)));
  RDB_RIEGL_PLANE_OCCUPANCY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.UnitSymbol)));
  RDB_RIEGL_PLANE_OCCUPANCY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.NamedValues)));
  RDB_RIEGL_PLANE_OCCUPANCY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_OCCUPANCY.LodSettings)));

  RDB_RIEGL_PLANE_WIDTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.Name)));
  RDB_RIEGL_PLANE_WIDTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.Title)));
  RDB_RIEGL_PLANE_WIDTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.Tags)));
  RDB_RIEGL_PLANE_WIDTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.Description)));
  RDB_RIEGL_PLANE_WIDTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.UnitSymbol)));
  RDB_RIEGL_PLANE_WIDTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.NamedValues)));
  RDB_RIEGL_PLANE_WIDTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_WIDTH.LodSettings)));

  RDB_RIEGL_PLANE_HEIGHT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.Name)));
  RDB_RIEGL_PLANE_HEIGHT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.Title)));
  RDB_RIEGL_PLANE_HEIGHT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.Tags)));
  RDB_RIEGL_PLANE_HEIGHT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.Description)));
  RDB_RIEGL_PLANE_HEIGHT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.UnitSymbol)));
  RDB_RIEGL_PLANE_HEIGHT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.NamedValues)));
  RDB_RIEGL_PLANE_HEIGHT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_HEIGHT.LodSettings)));

  RDB_RIEGL_PLANE_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.Name)));
  RDB_RIEGL_PLANE_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.Title)));
  RDB_RIEGL_PLANE_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.Tags)));
  RDB_RIEGL_PLANE_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.Description)));
  RDB_RIEGL_PLANE_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.UnitSymbol)));
  RDB_RIEGL_PLANE_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.NamedValues)));
  RDB_RIEGL_PLANE_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_COUNT.LodSettings)));

  RDB_RIEGL_MATCH_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.Name)));
  RDB_RIEGL_MATCH_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.Title)));
  RDB_RIEGL_MATCH_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.Tags)));
  RDB_RIEGL_MATCH_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.Description)));
  RDB_RIEGL_MATCH_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.UnitSymbol)));
  RDB_RIEGL_MATCH_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.NamedValues)));
  RDB_RIEGL_MATCH_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MATCH_COUNT.LodSettings)));

  RDB_RIEGL_PLANE_PATCH_DISTANCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.Name)));
  RDB_RIEGL_PLANE_PATCH_DISTANCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.Title)));
  RDB_RIEGL_PLANE_PATCH_DISTANCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.Tags)));
  RDB_RIEGL_PLANE_PATCH_DISTANCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.Description)));
  RDB_RIEGL_PLANE_PATCH_DISTANCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.UnitSymbol)));
  RDB_RIEGL_PLANE_PATCH_DISTANCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.NamedValues)));
  RDB_RIEGL_PLANE_PATCH_DISTANCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_DISTANCE.LodSettings)));

  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Name)));
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Title)));
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Tags)));
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.Description)));
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.UnitSymbol)));
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.NamedValues)));
  RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE.LodSettings)));

  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Name)));
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Title)));
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Tags)));
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.Description)));
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.UnitSymbol)));
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.NamedValues)));
  RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE.LodSettings)));

  RDB_RIEGL_CYLINDER_RADIUS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.Name)));
  RDB_RIEGL_CYLINDER_RADIUS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.Title)));
  RDB_RIEGL_CYLINDER_RADIUS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.Tags)));
  RDB_RIEGL_CYLINDER_RADIUS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.Description)));
  RDB_RIEGL_CYLINDER_RADIUS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.UnitSymbol)));
  RDB_RIEGL_CYLINDER_RADIUS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.NamedValues)));
  RDB_RIEGL_CYLINDER_RADIUS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS.LodSettings)));

  RDB_RIEGL_CYLINDER_HEIGHT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.Name)));
  RDB_RIEGL_CYLINDER_HEIGHT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.Title)));
  RDB_RIEGL_CYLINDER_HEIGHT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.Tags)));
  RDB_RIEGL_CYLINDER_HEIGHT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.Description)));
  RDB_RIEGL_CYLINDER_HEIGHT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.UnitSymbol)));
  RDB_RIEGL_CYLINDER_HEIGHT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.NamedValues)));
  RDB_RIEGL_CYLINDER_HEIGHT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_HEIGHT.LodSettings)));

  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Name)));
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Title)));
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Tags)));
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.Description)));
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.UnitSymbol)));
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.NamedValues)));
  RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE.LodSettings)));

  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Name)));
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Title)));
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Tags)));
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.Description)));
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.UnitSymbol)));
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.NamedValues)));
  RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE.LodSettings)));

  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Name)));
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Title)));
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Tags)));
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.Description)));
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.UnitSymbol)));
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.NamedValues)));
  RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE.LodSettings)));

  RDB_RIEGL_CYLINDER_INLIER_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.Name)));
  RDB_RIEGL_CYLINDER_INLIER_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.Title)));
  RDB_RIEGL_CYLINDER_INLIER_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.Tags)));
  RDB_RIEGL_CYLINDER_INLIER_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.Description)));
  RDB_RIEGL_CYLINDER_INLIER_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.UnitSymbol)));
  RDB_RIEGL_CYLINDER_INLIER_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.NamedValues)));
  RDB_RIEGL_CYLINDER_INLIER_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_INLIER_COUNT.LodSettings)));

  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Name)));
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Title)));
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Tags)));
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.Description)));
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.UnitSymbol)));
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.NamedValues)));
  RDB_RIEGL_CYLINDER_OUTLIER_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CYLINDER_OUTLIER_COUNT.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_NORTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.Name)));
  RDB_RIEGL_POF_ACCURACY_NORTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.Title)));
  RDB_RIEGL_POF_ACCURACY_NORTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.Tags)));
  RDB_RIEGL_POF_ACCURACY_NORTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.Description)));
  RDB_RIEGL_POF_ACCURACY_NORTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_NORTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_NORTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_EAST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.Name)));
  RDB_RIEGL_POF_ACCURACY_EAST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.Title)));
  RDB_RIEGL_POF_ACCURACY_EAST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.Tags)));
  RDB_RIEGL_POF_ACCURACY_EAST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.Description)));
  RDB_RIEGL_POF_ACCURACY_EAST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_EAST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_EAST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_DOWN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.Name)));
  RDB_RIEGL_POF_ACCURACY_DOWN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.Title)));
  RDB_RIEGL_POF_ACCURACY_DOWN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.Tags)));
  RDB_RIEGL_POF_ACCURACY_DOWN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.Description)));
  RDB_RIEGL_POF_ACCURACY_DOWN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_DOWN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_DOWN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Name)));
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Title)));
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Tags)));
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.Description)));
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_NORTH_EAST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_NORTH_EAST.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Name)));
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Title)));
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Tags)));
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.Description)));
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_EAST_DOWN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_EAST_DOWN.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Name)));
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Title)));
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Tags)));
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.Description)));
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_DOWN_NORTH.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_ROLL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.Name)));
  RDB_RIEGL_POF_ACCURACY_ROLL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.Title)));
  RDB_RIEGL_POF_ACCURACY_ROLL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.Tags)));
  RDB_RIEGL_POF_ACCURACY_ROLL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.Description)));
  RDB_RIEGL_POF_ACCURACY_ROLL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_ROLL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_ROLL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_ROLL.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_PITCH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.Name)));
  RDB_RIEGL_POF_ACCURACY_PITCH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.Title)));
  RDB_RIEGL_POF_ACCURACY_PITCH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.Tags)));
  RDB_RIEGL_POF_ACCURACY_PITCH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.Description)));
  RDB_RIEGL_POF_ACCURACY_PITCH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_PITCH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_PITCH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_PITCH.LodSettings)));

  RDB_RIEGL_POF_ACCURACY_YAW.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.Name)));
  RDB_RIEGL_POF_ACCURACY_YAW.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.Title)));
  RDB_RIEGL_POF_ACCURACY_YAW.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.Tags)));
  RDB_RIEGL_POF_ACCURACY_YAW.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.Description)));
  RDB_RIEGL_POF_ACCURACY_YAW.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.UnitSymbol)));
  RDB_RIEGL_POF_ACCURACY_YAW.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.NamedValues)));
  RDB_RIEGL_POF_ACCURACY_YAW.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_ACCURACY_YAW.LodSettings)));

  RDB_RIEGL_WFM_SBL_CHANNEL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.Name)));
  RDB_RIEGL_WFM_SBL_CHANNEL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.Title)));
  RDB_RIEGL_WFM_SBL_CHANNEL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.Tags)));
  RDB_RIEGL_WFM_SBL_CHANNEL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.Description)));
  RDB_RIEGL_WFM_SBL_CHANNEL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_CHANNEL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.NamedValues)));
  RDB_RIEGL_WFM_SBL_CHANNEL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_CHANNEL.LodSettings)));

  RDB_RIEGL_WFM_SBL_MEAN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.Name)));
  RDB_RIEGL_WFM_SBL_MEAN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.Title)));
  RDB_RIEGL_WFM_SBL_MEAN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.Tags)));
  RDB_RIEGL_WFM_SBL_MEAN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.Description)));
  RDB_RIEGL_WFM_SBL_MEAN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_MEAN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.NamedValues)));
  RDB_RIEGL_WFM_SBL_MEAN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_MEAN.LodSettings)));

  RDB_RIEGL_WFM_SBL_STD_DEV.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.Name)));
  RDB_RIEGL_WFM_SBL_STD_DEV.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.Title)));
  RDB_RIEGL_WFM_SBL_STD_DEV.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.Tags)));
  RDB_RIEGL_WFM_SBL_STD_DEV.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.Description)));
  RDB_RIEGL_WFM_SBL_STD_DEV.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_STD_DEV.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.NamedValues)));
  RDB_RIEGL_WFM_SBL_STD_DEV.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_STD_DEV.LodSettings)));

  RDB_RIEGL_WFM_SBL_FIRST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.Name)));
  RDB_RIEGL_WFM_SBL_FIRST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.Title)));
  RDB_RIEGL_WFM_SBL_FIRST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.Tags)));
  RDB_RIEGL_WFM_SBL_FIRST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.Description)));
  RDB_RIEGL_WFM_SBL_FIRST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_FIRST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.NamedValues)));
  RDB_RIEGL_WFM_SBL_FIRST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_FIRST.LodSettings)));

  RDB_RIEGL_WFM_SBL_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.Name)));
  RDB_RIEGL_WFM_SBL_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.Title)));
  RDB_RIEGL_WFM_SBL_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.Tags)));
  RDB_RIEGL_WFM_SBL_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.Description)));
  RDB_RIEGL_WFM_SBL_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.NamedValues)));
  RDB_RIEGL_WFM_SBL_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_COUNT.LodSettings)));

  RDB_RIEGL_WFM_SDA_FIRST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.Name)));
  RDB_RIEGL_WFM_SDA_FIRST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.Title)));
  RDB_RIEGL_WFM_SDA_FIRST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.Tags)));
  RDB_RIEGL_WFM_SDA_FIRST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.Description)));
  RDB_RIEGL_WFM_SDA_FIRST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.UnitSymbol)));
  RDB_RIEGL_WFM_SDA_FIRST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.NamedValues)));
  RDB_RIEGL_WFM_SDA_FIRST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_FIRST.LodSettings)));

  RDB_RIEGL_WFM_SDA_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.Name)));
  RDB_RIEGL_WFM_SDA_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.Title)));
  RDB_RIEGL_WFM_SDA_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.Tags)));
  RDB_RIEGL_WFM_SDA_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.Description)));
  RDB_RIEGL_WFM_SDA_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.UnitSymbol)));
  RDB_RIEGL_WFM_SDA_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.NamedValues)));
  RDB_RIEGL_WFM_SDA_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SDA_COUNT.LodSettings)));

  RDB_RIEGL_WFM_SAMPLE_VALUE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.Name)));
  RDB_RIEGL_WFM_SAMPLE_VALUE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.Title)));
  RDB_RIEGL_WFM_SAMPLE_VALUE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.Tags)));
  RDB_RIEGL_WFM_SAMPLE_VALUE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.Description)));
  RDB_RIEGL_WFM_SAMPLE_VALUE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.UnitSymbol)));
  RDB_RIEGL_WFM_SAMPLE_VALUE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.NamedValues)));
  RDB_RIEGL_WFM_SAMPLE_VALUE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SAMPLE_VALUE.LodSettings)));

  RDB_RIEGL_CONTROL_OBJECT_TYPE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.Name)));
  RDB_RIEGL_CONTROL_OBJECT_TYPE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.Title)));
  RDB_RIEGL_CONTROL_OBJECT_TYPE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.Tags)));
  RDB_RIEGL_CONTROL_OBJECT_TYPE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.Description)));
  RDB_RIEGL_CONTROL_OBJECT_TYPE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.UnitSymbol)));
  RDB_RIEGL_CONTROL_OBJECT_TYPE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.NamedValues)));
  RDB_RIEGL_CONTROL_OBJECT_TYPE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CONTROL_OBJECT_TYPE.LodSettings)));

  RDB_RIEGL_MODEL_FIT_QUALITY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.Name)));
  RDB_RIEGL_MODEL_FIT_QUALITY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.Title)));
  RDB_RIEGL_MODEL_FIT_QUALITY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.Tags)));
  RDB_RIEGL_MODEL_FIT_QUALITY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.Description)));
  RDB_RIEGL_MODEL_FIT_QUALITY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.UnitSymbol)));
  RDB_RIEGL_MODEL_FIT_QUALITY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.NamedValues)));
  RDB_RIEGL_MODEL_FIT_QUALITY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MODEL_FIT_QUALITY.LodSettings)));

  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Name)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Title)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Tags)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.Description)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.UnitSymbol)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.NamedValues)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE.LodSettings)));

  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Name)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Title)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Tags)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.Description)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.UnitSymbol)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.NamedValues)));
  RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE.LodSettings)));

  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Name)));
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Title)));
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Tags)));
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.Description)));
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.UnitSymbol)));
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.NamedValues)));
  RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE.LodSettings)));

  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Name)));
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Title)));
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Tags)));
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.Description)));
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.UnitSymbol)));
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.NamedValues)));
  RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE.LodSettings)));

  RDB_RIEGL_OBS_CONFIDENCE_XYZ.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.Name)));
  RDB_RIEGL_OBS_CONFIDENCE_XYZ.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.Title)));
  RDB_RIEGL_OBS_CONFIDENCE_XYZ.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.Tags)));
  RDB_RIEGL_OBS_CONFIDENCE_XYZ.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.Description)));
  RDB_RIEGL_OBS_CONFIDENCE_XYZ.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.UnitSymbol)));
  RDB_RIEGL_OBS_CONFIDENCE_XYZ.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.NamedValues)));
  RDB_RIEGL_OBS_CONFIDENCE_XYZ.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XYZ.LodSettings)));

  RDB_RIEGL_OBS_CONFIDENCE_XY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.Name)));
  RDB_RIEGL_OBS_CONFIDENCE_XY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.Title)));
  RDB_RIEGL_OBS_CONFIDENCE_XY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.Tags)));
  RDB_RIEGL_OBS_CONFIDENCE_XY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.Description)));
  RDB_RIEGL_OBS_CONFIDENCE_XY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.UnitSymbol)));
  RDB_RIEGL_OBS_CONFIDENCE_XY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.NamedValues)));
  RDB_RIEGL_OBS_CONFIDENCE_XY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_XY.LodSettings)));

  RDB_RIEGL_OBS_CONFIDENCE_Z.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.Name)));
  RDB_RIEGL_OBS_CONFIDENCE_Z.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.Title)));
  RDB_RIEGL_OBS_CONFIDENCE_Z.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.Tags)));
  RDB_RIEGL_OBS_CONFIDENCE_Z.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.Description)));
  RDB_RIEGL_OBS_CONFIDENCE_Z.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.UnitSymbol)));
  RDB_RIEGL_OBS_CONFIDENCE_Z.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.NamedValues)));
  RDB_RIEGL_OBS_CONFIDENCE_Z.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_Z.LodSettings)));

  RDB_RIEGL_OBS_CONFIDENCE_RANGE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.Name)));
  RDB_RIEGL_OBS_CONFIDENCE_RANGE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.Title)));
  RDB_RIEGL_OBS_CONFIDENCE_RANGE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.Tags)));
  RDB_RIEGL_OBS_CONFIDENCE_RANGE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.Description)));
  RDB_RIEGL_OBS_CONFIDENCE_RANGE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.UnitSymbol)));
  RDB_RIEGL_OBS_CONFIDENCE_RANGE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.NamedValues)));
  RDB_RIEGL_OBS_CONFIDENCE_RANGE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_RANGE.LodSettings)));

  RDB_RIEGL_OBS_CONFIDENCE_THETA.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.Name)));
  RDB_RIEGL_OBS_CONFIDENCE_THETA.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.Title)));
  RDB_RIEGL_OBS_CONFIDENCE_THETA.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.Tags)));
  RDB_RIEGL_OBS_CONFIDENCE_THETA.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.Description)));
  RDB_RIEGL_OBS_CONFIDENCE_THETA.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.UnitSymbol)));
  RDB_RIEGL_OBS_CONFIDENCE_THETA.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.NamedValues)));
  RDB_RIEGL_OBS_CONFIDENCE_THETA.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_THETA.LodSettings)));

  RDB_RIEGL_OBS_CONFIDENCE_PHI.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.Name)));
  RDB_RIEGL_OBS_CONFIDENCE_PHI.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.Title)));
  RDB_RIEGL_OBS_CONFIDENCE_PHI.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.Tags)));
  RDB_RIEGL_OBS_CONFIDENCE_PHI.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.Description)));
  RDB_RIEGL_OBS_CONFIDENCE_PHI.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.UnitSymbol)));
  RDB_RIEGL_OBS_CONFIDENCE_PHI.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.NamedValues)));
  RDB_RIEGL_OBS_CONFIDENCE_PHI.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_CONFIDENCE_PHI.LodSettings)));

  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Name)));
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Title)));
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Tags)));
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.Description)));
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.UnitSymbol)));
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.NamedValues)));
  RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT.LodSettings)));

  RDB_RIEGL_USED_FOR_ADJUSTMENT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.Name)));
  RDB_RIEGL_USED_FOR_ADJUSTMENT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.Title)));
  RDB_RIEGL_USED_FOR_ADJUSTMENT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.Tags)));
  RDB_RIEGL_USED_FOR_ADJUSTMENT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.Description)));
  RDB_RIEGL_USED_FOR_ADJUSTMENT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.UnitSymbol)));
  RDB_RIEGL_USED_FOR_ADJUSTMENT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.NamedValues)));
  RDB_RIEGL_USED_FOR_ADJUSTMENT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_USED_FOR_ADJUSTMENT.LodSettings)));

  RDB_RIEGL_REFERENCE_OBJECT_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.Name)));
  RDB_RIEGL_REFERENCE_OBJECT_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.Title)));
  RDB_RIEGL_REFERENCE_OBJECT_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.Tags)));
  RDB_RIEGL_REFERENCE_OBJECT_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.Description)));
  RDB_RIEGL_REFERENCE_OBJECT_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.UnitSymbol)));
  RDB_RIEGL_REFERENCE_OBJECT_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.NamedValues)));
  RDB_RIEGL_REFERENCE_OBJECT_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_REFERENCE_OBJECT_ID.LodSettings)));

  RDB_RIEGL_EVENT_INDEX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.Name)));
  RDB_RIEGL_EVENT_INDEX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.Title)));
  RDB_RIEGL_EVENT_INDEX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.Tags)));
  RDB_RIEGL_EVENT_INDEX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.Description)));
  RDB_RIEGL_EVENT_INDEX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.UnitSymbol)));
  RDB_RIEGL_EVENT_INDEX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.NamedValues)));
  RDB_RIEGL_EVENT_INDEX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_INDEX.LodSettings)));

  RDB_RIEGL_EVENT_PULSE_WIDTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.Name)));
  RDB_RIEGL_EVENT_PULSE_WIDTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.Title)));
  RDB_RIEGL_EVENT_PULSE_WIDTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.Tags)));
  RDB_RIEGL_EVENT_PULSE_WIDTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.Description)));
  RDB_RIEGL_EVENT_PULSE_WIDTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.UnitSymbol)));
  RDB_RIEGL_EVENT_PULSE_WIDTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.NamedValues)));
  RDB_RIEGL_EVENT_PULSE_WIDTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_PULSE_WIDTH.LodSettings)));

  RDB_RIEGL_RAW_RANGE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.Name)));
  RDB_RIEGL_RAW_RANGE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.Title)));
  RDB_RIEGL_RAW_RANGE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.Tags)));
  RDB_RIEGL_RAW_RANGE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.Description)));
  RDB_RIEGL_RAW_RANGE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.UnitSymbol)));
  RDB_RIEGL_RAW_RANGE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.NamedValues)));
  RDB_RIEGL_RAW_RANGE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_RANGE.LodSettings)));

  RDB_RIEGL_RAW_LINE_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.Name)));
  RDB_RIEGL_RAW_LINE_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.Title)));
  RDB_RIEGL_RAW_LINE_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.Tags)));
  RDB_RIEGL_RAW_LINE_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.Description)));
  RDB_RIEGL_RAW_LINE_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.UnitSymbol)));
  RDB_RIEGL_RAW_LINE_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.NamedValues)));
  RDB_RIEGL_RAW_LINE_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_LINE_ANGLE.LodSettings)));

  RDB_RIEGL_RAW_FRAME_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.Name)));
  RDB_RIEGL_RAW_FRAME_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.Title)));
  RDB_RIEGL_RAW_FRAME_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.Tags)));
  RDB_RIEGL_RAW_FRAME_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.Description)));
  RDB_RIEGL_RAW_FRAME_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.UnitSymbol)));
  RDB_RIEGL_RAW_FRAME_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.NamedValues)));
  RDB_RIEGL_RAW_FRAME_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_RAW_FRAME_ANGLE.LodSettings)));

  RDB_RIEGL_LINE_ANGLE_COARSE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.Name)));
  RDB_RIEGL_LINE_ANGLE_COARSE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.Title)));
  RDB_RIEGL_LINE_ANGLE_COARSE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.Tags)));
  RDB_RIEGL_LINE_ANGLE_COARSE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.Description)));
  RDB_RIEGL_LINE_ANGLE_COARSE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.UnitSymbol)));
  RDB_RIEGL_LINE_ANGLE_COARSE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.NamedValues)));
  RDB_RIEGL_LINE_ANGLE_COARSE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_COARSE.LodSettings)));

  RDB_RIEGL_LINE_ANGLE_REDUCED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.Name)));
  RDB_RIEGL_LINE_ANGLE_REDUCED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.Title)));
  RDB_RIEGL_LINE_ANGLE_REDUCED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.Tags)));
  RDB_RIEGL_LINE_ANGLE_REDUCED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.Description)));
  RDB_RIEGL_LINE_ANGLE_REDUCED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.UnitSymbol)));
  RDB_RIEGL_LINE_ANGLE_REDUCED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.NamedValues)));
  RDB_RIEGL_LINE_ANGLE_REDUCED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_ANGLE_REDUCED.LodSettings)));

  RDB_RIEGL_FRAME_ANGLE_COARSE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.Name)));
  RDB_RIEGL_FRAME_ANGLE_COARSE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.Title)));
  RDB_RIEGL_FRAME_ANGLE_COARSE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.Tags)));
  RDB_RIEGL_FRAME_ANGLE_COARSE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.Description)));
  RDB_RIEGL_FRAME_ANGLE_COARSE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.UnitSymbol)));
  RDB_RIEGL_FRAME_ANGLE_COARSE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.NamedValues)));
  RDB_RIEGL_FRAME_ANGLE_COARSE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_ANGLE_COARSE.LodSettings)));

  RDB_RIEGL_ANGLE_OF_INCIDENCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.Name)));
  RDB_RIEGL_ANGLE_OF_INCIDENCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.Title)));
  RDB_RIEGL_ANGLE_OF_INCIDENCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.Tags)));
  RDB_RIEGL_ANGLE_OF_INCIDENCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.Description)));
  RDB_RIEGL_ANGLE_OF_INCIDENCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.UnitSymbol)));
  RDB_RIEGL_ANGLE_OF_INCIDENCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.NamedValues)));
  RDB_RIEGL_ANGLE_OF_INCIDENCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ANGLE_OF_INCIDENCE.LodSettings)));

  RDB_RIEGL_SCAN_LINE_INDEX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.Name)));
  RDB_RIEGL_SCAN_LINE_INDEX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.Title)));
  RDB_RIEGL_SCAN_LINE_INDEX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.Tags)));
  RDB_RIEGL_SCAN_LINE_INDEX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.Description)));
  RDB_RIEGL_SCAN_LINE_INDEX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.UnitSymbol)));
  RDB_RIEGL_SCAN_LINE_INDEX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.NamedValues)));
  RDB_RIEGL_SCAN_LINE_INDEX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_LINE_INDEX.LodSettings)));

  RDB_RIEGL_SHOT_INDEX_LINE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.Name)));
  RDB_RIEGL_SHOT_INDEX_LINE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.Title)));
  RDB_RIEGL_SHOT_INDEX_LINE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.Tags)));
  RDB_RIEGL_SHOT_INDEX_LINE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.Description)));
  RDB_RIEGL_SHOT_INDEX_LINE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.UnitSymbol)));
  RDB_RIEGL_SHOT_INDEX_LINE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.NamedValues)));
  RDB_RIEGL_SHOT_INDEX_LINE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_INDEX_LINE.LodSettings)));

  RDB_RIEGL_MIRROR_FACET.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.Name)));
  RDB_RIEGL_MIRROR_FACET.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.Title)));
  RDB_RIEGL_MIRROR_FACET.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.Tags)));
  RDB_RIEGL_MIRROR_FACET.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.Description)));
  RDB_RIEGL_MIRROR_FACET.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.UnitSymbol)));
  RDB_RIEGL_MIRROR_FACET.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.NamedValues)));
  RDB_RIEGL_MIRROR_FACET.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_FACET.LodSettings)));

  RDB_RIEGL_SCAN_SEGMENT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.Name)));
  RDB_RIEGL_SCAN_SEGMENT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.Title)));
  RDB_RIEGL_SCAN_SEGMENT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.Tags)));
  RDB_RIEGL_SCAN_SEGMENT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.Description)));
  RDB_RIEGL_SCAN_SEGMENT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.UnitSymbol)));
  RDB_RIEGL_SCAN_SEGMENT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.NamedValues)));
  RDB_RIEGL_SCAN_SEGMENT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_SEGMENT.LodSettings)));

  RDB_RIEGL_WAVEFORM_AVAILABLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.Name)));
  RDB_RIEGL_WAVEFORM_AVAILABLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.Title)));
  RDB_RIEGL_WAVEFORM_AVAILABLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.Tags)));
  RDB_RIEGL_WAVEFORM_AVAILABLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.Description)));
  RDB_RIEGL_WAVEFORM_AVAILABLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.UnitSymbol)));
  RDB_RIEGL_WAVEFORM_AVAILABLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.NamedValues)));
  RDB_RIEGL_WAVEFORM_AVAILABLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WAVEFORM_AVAILABLE.LodSettings)));

  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Name)));
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Title)));
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Tags)));
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.Description)));
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.UnitSymbol)));
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.NamedValues)));
  RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_HYDRO_REFRACTION_CORRECTED.LodSettings)));

  RDB_RIEGL_EXTINCTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.Name)));
  RDB_RIEGL_EXTINCTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.Title)));
  RDB_RIEGL_EXTINCTION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.Tags)));
  RDB_RIEGL_EXTINCTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.Description)));
  RDB_RIEGL_EXTINCTION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.UnitSymbol)));
  RDB_RIEGL_EXTINCTION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.NamedValues)));
  RDB_RIEGL_EXTINCTION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EXTINCTION.LodSettings)));

  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Name)));
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Title)));
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Tags)));
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.Description)));
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.UnitSymbol)));
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.NamedValues)));
  RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC.LodSettings)));

  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Name)));
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Title)));
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Tags)));
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.Description)));
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.UnitSymbol)));
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.NamedValues)));
  RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC.LodSettings)));

  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Name)));
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Title)));
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Tags)));
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.Description)));
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.UnitSymbol)));
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.NamedValues)));
  RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED.LodSettings)));

  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Name)));
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Title)));
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Tags)));
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.Description)));
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.UnitSymbol)));
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.NamedValues)));
  RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC.LodSettings)));

  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Name)));
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Title)));
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Tags)));
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.Description)));
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.UnitSymbol)));
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.NamedValues)));
  RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC.LodSettings)));

  RDB_RIEGL_SVB_SURFACE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.Name)));
  RDB_RIEGL_SVB_SURFACE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.Title)));
  RDB_RIEGL_SVB_SURFACE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.Tags)));
  RDB_RIEGL_SVB_SURFACE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.Description)));
  RDB_RIEGL_SVB_SURFACE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.UnitSymbol)));
  RDB_RIEGL_SVB_SURFACE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.NamedValues)));
  RDB_RIEGL_SVB_SURFACE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_SURFACE.LodSettings)));

  RDB_RIEGL_SVB_BOTTOM.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.Name)));
  RDB_RIEGL_SVB_BOTTOM.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.Title)));
  RDB_RIEGL_SVB_BOTTOM.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.Tags)));
  RDB_RIEGL_SVB_BOTTOM.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.Description)));
  RDB_RIEGL_SVB_BOTTOM.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.UnitSymbol)));
  RDB_RIEGL_SVB_BOTTOM.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.NamedValues)));
  RDB_RIEGL_SVB_BOTTOM.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_BOTTOM.LodSettings)));

  RDB_RIEGL_SVB_PATH_LENGTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.Name)));
  RDB_RIEGL_SVB_PATH_LENGTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.Title)));
  RDB_RIEGL_SVB_PATH_LENGTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.Tags)));
  RDB_RIEGL_SVB_PATH_LENGTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.Description)));
  RDB_RIEGL_SVB_PATH_LENGTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.UnitSymbol)));
  RDB_RIEGL_SVB_PATH_LENGTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.NamedValues)));
  RDB_RIEGL_SVB_PATH_LENGTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SVB_PATH_LENGTH.LodSettings)));

  RDB_RIEGL_START_OF_SCAN_LINE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.Name)));
  RDB_RIEGL_START_OF_SCAN_LINE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.Title)));
  RDB_RIEGL_START_OF_SCAN_LINE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.Tags)));
  RDB_RIEGL_START_OF_SCAN_LINE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.Description)));
  RDB_RIEGL_START_OF_SCAN_LINE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.UnitSymbol)));
  RDB_RIEGL_START_OF_SCAN_LINE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.NamedValues)));
  RDB_RIEGL_START_OF_SCAN_LINE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_START_OF_SCAN_LINE.LodSettings)));

  RDB_RIEGL_END_OF_SCAN_LINE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.Name)));
  RDB_RIEGL_END_OF_SCAN_LINE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.Title)));
  RDB_RIEGL_END_OF_SCAN_LINE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.Tags)));
  RDB_RIEGL_END_OF_SCAN_LINE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.Description)));
  RDB_RIEGL_END_OF_SCAN_LINE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.UnitSymbol)));
  RDB_RIEGL_END_OF_SCAN_LINE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.NamedValues)));
  RDB_RIEGL_END_OF_SCAN_LINE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_END_OF_SCAN_LINE.LodSettings)));

  RDB_RIEGL_SCAN_ANGLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.Name)));
  RDB_RIEGL_SCAN_ANGLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.Title)));
  RDB_RIEGL_SCAN_ANGLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.Tags)));
  RDB_RIEGL_SCAN_ANGLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.Description)));
  RDB_RIEGL_SCAN_ANGLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.UnitSymbol)));
  RDB_RIEGL_SCAN_ANGLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.NamedValues)));
  RDB_RIEGL_SCAN_ANGLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_ANGLE.LodSettings)));

  RDB_RIEGL_SCAN_DIRECTION.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.Name)));
  RDB_RIEGL_SCAN_DIRECTION.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.Title)));
  RDB_RIEGL_SCAN_DIRECTION.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.Tags)));
  RDB_RIEGL_SCAN_DIRECTION.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.Description)));
  RDB_RIEGL_SCAN_DIRECTION.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.UnitSymbol)));
  RDB_RIEGL_SCAN_DIRECTION.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.NamedValues)));
  RDB_RIEGL_SCAN_DIRECTION.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SCAN_DIRECTION.LodSettings)));

  RDB_RIEGL_VOXEL_COLLAPSED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.Name)));
  RDB_RIEGL_VOXEL_COLLAPSED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.Title)));
  RDB_RIEGL_VOXEL_COLLAPSED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.Tags)));
  RDB_RIEGL_VOXEL_COLLAPSED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.Description)));
  RDB_RIEGL_VOXEL_COLLAPSED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.UnitSymbol)));
  RDB_RIEGL_VOXEL_COLLAPSED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.NamedValues)));
  RDB_RIEGL_VOXEL_COLLAPSED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COLLAPSED.LodSettings)));

  RDB_RIEGL_LINE_SCAN_ACTIVE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.Name)));
  RDB_RIEGL_LINE_SCAN_ACTIVE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.Title)));
  RDB_RIEGL_LINE_SCAN_ACTIVE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.Tags)));
  RDB_RIEGL_LINE_SCAN_ACTIVE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.Description)));
  RDB_RIEGL_LINE_SCAN_ACTIVE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.UnitSymbol)));
  RDB_RIEGL_LINE_SCAN_ACTIVE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.NamedValues)));
  RDB_RIEGL_LINE_SCAN_ACTIVE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_LINE_SCAN_ACTIVE.LodSettings)));

  RDB_RIEGL_FRAME_SCAN_ACTIVE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.Name)));
  RDB_RIEGL_FRAME_SCAN_ACTIVE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.Title)));
  RDB_RIEGL_FRAME_SCAN_ACTIVE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.Tags)));
  RDB_RIEGL_FRAME_SCAN_ACTIVE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.Description)));
  RDB_RIEGL_FRAME_SCAN_ACTIVE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.UnitSymbol)));
  RDB_RIEGL_FRAME_SCAN_ACTIVE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.NamedValues)));
  RDB_RIEGL_FRAME_SCAN_ACTIVE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_FRAME_SCAN_ACTIVE.LodSettings)));

  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Name)));
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Title)));
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Tags)));
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.Description)));
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.UnitSymbol)));
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.NamedValues)));
  RDB_RIEGL_DATA_ACQUISITION_ACTIVE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DATA_ACQUISITION_ACTIVE.LodSettings)));

  RDB_RIEGL_PLANE_REFERENCES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.Name)));
  RDB_RIEGL_PLANE_REFERENCES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.Title)));
  RDB_RIEGL_PLANE_REFERENCES.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.Tags)));
  RDB_RIEGL_PLANE_REFERENCES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.Description)));
  RDB_RIEGL_PLANE_REFERENCES.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.UnitSymbol)));
  RDB_RIEGL_PLANE_REFERENCES.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.NamedValues)));
  RDB_RIEGL_PLANE_REFERENCES.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_REFERENCES.LodSettings)));

  RDB_RIEGL_POF_PATH_LENGTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.Name)));
  RDB_RIEGL_POF_PATH_LENGTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.Title)));
  RDB_RIEGL_POF_PATH_LENGTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.Tags)));
  RDB_RIEGL_POF_PATH_LENGTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.Description)));
  RDB_RIEGL_POF_PATH_LENGTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.UnitSymbol)));
  RDB_RIEGL_POF_PATH_LENGTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.NamedValues)));
  RDB_RIEGL_POF_PATH_LENGTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PATH_LENGTH.LodSettings)));

  RDB_RIEGL_POF_PDOP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.Name)));
  RDB_RIEGL_POF_PDOP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.Title)));
  RDB_RIEGL_POF_PDOP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.Tags)));
  RDB_RIEGL_POF_PDOP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.Description)));
  RDB_RIEGL_POF_PDOP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.UnitSymbol)));
  RDB_RIEGL_POF_PDOP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.NamedValues)));
  RDB_RIEGL_POF_PDOP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_PDOP.LodSettings)));

  RDB_RIEGL_POF_HDOP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.Name)));
  RDB_RIEGL_POF_HDOP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.Title)));
  RDB_RIEGL_POF_HDOP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.Tags)));
  RDB_RIEGL_POF_HDOP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.Description)));
  RDB_RIEGL_POF_HDOP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.UnitSymbol)));
  RDB_RIEGL_POF_HDOP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.NamedValues)));
  RDB_RIEGL_POF_HDOP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_HDOP.LodSettings)));

  RDB_RIEGL_POF_VDOP.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.Name)));
  RDB_RIEGL_POF_VDOP.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.Title)));
  RDB_RIEGL_POF_VDOP.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.Tags)));
  RDB_RIEGL_POF_VDOP.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.Description)));
  RDB_RIEGL_POF_VDOP.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.UnitSymbol)));
  RDB_RIEGL_POF_VDOP.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.NamedValues)));
  RDB_RIEGL_POF_VDOP.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_VDOP.LodSettings)));

  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Name)));
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Title)));
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Tags)));
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.Description)));
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.UnitSymbol)));
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.NamedValues)));
  RDB_RIEGL_POF_AGE_OF_CORRECTIONS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_AGE_OF_CORRECTIONS.LodSettings)));

  RDB_RIEGL_POF_BASELINE_LENGTH.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.Name)));
  RDB_RIEGL_POF_BASELINE_LENGTH.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.Title)));
  RDB_RIEGL_POF_BASELINE_LENGTH.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.Tags)));
  RDB_RIEGL_POF_BASELINE_LENGTH.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.Description)));
  RDB_RIEGL_POF_BASELINE_LENGTH.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.UnitSymbol)));
  RDB_RIEGL_POF_BASELINE_LENGTH.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.NamedValues)));
  RDB_RIEGL_POF_BASELINE_LENGTH.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_BASELINE_LENGTH.LodSettings)));

  RDB_RIEGL_POF_SOLUTION_GNSS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.Name)));
  RDB_RIEGL_POF_SOLUTION_GNSS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.Title)));
  RDB_RIEGL_POF_SOLUTION_GNSS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.Tags)));
  RDB_RIEGL_POF_SOLUTION_GNSS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.Description)));
  RDB_RIEGL_POF_SOLUTION_GNSS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.UnitSymbol)));
  RDB_RIEGL_POF_SOLUTION_GNSS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.NamedValues)));
  RDB_RIEGL_POF_SOLUTION_GNSS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SOLUTION_GNSS.LodSettings)));

  RDB_RIEGL_POF_SATELLITES_GNSS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.Name)));
  RDB_RIEGL_POF_SATELLITES_GNSS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.Title)));
  RDB_RIEGL_POF_SATELLITES_GNSS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.Tags)));
  RDB_RIEGL_POF_SATELLITES_GNSS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.Description)));
  RDB_RIEGL_POF_SATELLITES_GNSS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.UnitSymbol)));
  RDB_RIEGL_POF_SATELLITES_GNSS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.NamedValues)));
  RDB_RIEGL_POF_SATELLITES_GNSS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GNSS.LodSettings)));

  RDB_RIEGL_POF_SATELLITES_GPS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.Name)));
  RDB_RIEGL_POF_SATELLITES_GPS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.Title)));
  RDB_RIEGL_POF_SATELLITES_GPS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.Tags)));
  RDB_RIEGL_POF_SATELLITES_GPS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.Description)));
  RDB_RIEGL_POF_SATELLITES_GPS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.UnitSymbol)));
  RDB_RIEGL_POF_SATELLITES_GPS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.NamedValues)));
  RDB_RIEGL_POF_SATELLITES_GPS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GPS.LodSettings)));

  RDB_RIEGL_POF_SATELLITES_GLONASS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.Name)));
  RDB_RIEGL_POF_SATELLITES_GLONASS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.Title)));
  RDB_RIEGL_POF_SATELLITES_GLONASS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.Tags)));
  RDB_RIEGL_POF_SATELLITES_GLONASS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.Description)));
  RDB_RIEGL_POF_SATELLITES_GLONASS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.UnitSymbol)));
  RDB_RIEGL_POF_SATELLITES_GLONASS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.NamedValues)));
  RDB_RIEGL_POF_SATELLITES_GLONASS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GLONASS.LodSettings)));

  RDB_RIEGL_POF_SATELLITES_BEIDOU.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.Name)));
  RDB_RIEGL_POF_SATELLITES_BEIDOU.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.Title)));
  RDB_RIEGL_POF_SATELLITES_BEIDOU.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.Tags)));
  RDB_RIEGL_POF_SATELLITES_BEIDOU.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.Description)));
  RDB_RIEGL_POF_SATELLITES_BEIDOU.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.UnitSymbol)));
  RDB_RIEGL_POF_SATELLITES_BEIDOU.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.NamedValues)));
  RDB_RIEGL_POF_SATELLITES_BEIDOU.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_BEIDOU.LodSettings)));

  RDB_RIEGL_POF_SATELLITES_GALILEO.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.Name)));
  RDB_RIEGL_POF_SATELLITES_GALILEO.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.Title)));
  RDB_RIEGL_POF_SATELLITES_GALILEO.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.Tags)));
  RDB_RIEGL_POF_SATELLITES_GALILEO.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.Description)));
  RDB_RIEGL_POF_SATELLITES_GALILEO.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.UnitSymbol)));
  RDB_RIEGL_POF_SATELLITES_GALILEO.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.NamedValues)));
  RDB_RIEGL_POF_SATELLITES_GALILEO.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_GALILEO.LodSettings)));

  RDB_RIEGL_POF_SATELLITES_QZSS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.Name)));
  RDB_RIEGL_POF_SATELLITES_QZSS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.Title)));
  RDB_RIEGL_POF_SATELLITES_QZSS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.Tags)));
  RDB_RIEGL_POF_SATELLITES_QZSS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.Description)));
  RDB_RIEGL_POF_SATELLITES_QZSS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.UnitSymbol)));
  RDB_RIEGL_POF_SATELLITES_QZSS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.NamedValues)));
  RDB_RIEGL_POF_SATELLITES_QZSS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POF_SATELLITES_QZSS.LodSettings)));

  RDB_RIEGL_PIXEL_LINEAR_SUMS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.Name)));
  RDB_RIEGL_PIXEL_LINEAR_SUMS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.Title)));
  RDB_RIEGL_PIXEL_LINEAR_SUMS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.Tags)));
  RDB_RIEGL_PIXEL_LINEAR_SUMS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.Description)));
  RDB_RIEGL_PIXEL_LINEAR_SUMS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.UnitSymbol)));
  RDB_RIEGL_PIXEL_LINEAR_SUMS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.NamedValues)));
  RDB_RIEGL_PIXEL_LINEAR_SUMS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_LINEAR_SUMS.LodSettings)));

  RDB_RIEGL_PIXEL_SQUARE_SUMS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.Name)));
  RDB_RIEGL_PIXEL_SQUARE_SUMS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.Title)));
  RDB_RIEGL_PIXEL_SQUARE_SUMS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.Tags)));
  RDB_RIEGL_PIXEL_SQUARE_SUMS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.Description)));
  RDB_RIEGL_PIXEL_SQUARE_SUMS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.UnitSymbol)));
  RDB_RIEGL_PIXEL_SQUARE_SUMS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.NamedValues)));
  RDB_RIEGL_PIXEL_SQUARE_SUMS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PIXEL_SQUARE_SUMS.LodSettings)));

  RDB_RIEGL_SHAPE_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.Name)));
  RDB_RIEGL_SHAPE_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.Title)));
  RDB_RIEGL_SHAPE_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.Tags)));
  RDB_RIEGL_SHAPE_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.Description)));
  RDB_RIEGL_SHAPE_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.UnitSymbol)));
  RDB_RIEGL_SHAPE_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.NamedValues)));
  RDB_RIEGL_SHAPE_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHAPE_ID.LodSettings)));

  RDB_RIEGL_PLANE_CLUSTER_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.Name)));
  RDB_RIEGL_PLANE_CLUSTER_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.Title)));
  RDB_RIEGL_PLANE_CLUSTER_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.Tags)));
  RDB_RIEGL_PLANE_CLUSTER_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.Description)));
  RDB_RIEGL_PLANE_CLUSTER_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.UnitSymbol)));
  RDB_RIEGL_PLANE_CLUSTER_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.NamedValues)));
  RDB_RIEGL_PLANE_CLUSTER_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_PLANE_CLUSTER_ID.LodSettings)));

  RDB_RIEGL_SEGMENT_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.Name)));
  RDB_RIEGL_SEGMENT_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.Title)));
  RDB_RIEGL_SEGMENT_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.Tags)));
  RDB_RIEGL_SEGMENT_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.Description)));
  RDB_RIEGL_SEGMENT_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.UnitSymbol)));
  RDB_RIEGL_SEGMENT_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.NamedValues)));
  RDB_RIEGL_SEGMENT_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SEGMENT_ID.LodSettings)));

  RDB_RIEGL_VOXEL_LINEAR_SUMS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.Name)));
  RDB_RIEGL_VOXEL_LINEAR_SUMS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.Title)));
  RDB_RIEGL_VOXEL_LINEAR_SUMS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.Tags)));
  RDB_RIEGL_VOXEL_LINEAR_SUMS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.Description)));
  RDB_RIEGL_VOXEL_LINEAR_SUMS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.UnitSymbol)));
  RDB_RIEGL_VOXEL_LINEAR_SUMS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.NamedValues)));
  RDB_RIEGL_VOXEL_LINEAR_SUMS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_LINEAR_SUMS.LodSettings)));

  RDB_RIEGL_VOXEL_SQUARE_SUMS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.Name)));
  RDB_RIEGL_VOXEL_SQUARE_SUMS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.Title)));
  RDB_RIEGL_VOXEL_SQUARE_SUMS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.Tags)));
  RDB_RIEGL_VOXEL_SQUARE_SUMS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.Description)));
  RDB_RIEGL_VOXEL_SQUARE_SUMS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.UnitSymbol)));
  RDB_RIEGL_VOXEL_SQUARE_SUMS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.NamedValues)));
  RDB_RIEGL_VOXEL_SQUARE_SUMS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_SQUARE_SUMS.LodSettings)));

  RDB_RIEGL_VOXEL_INDEX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.Name)));
  RDB_RIEGL_VOXEL_INDEX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.Title)));
  RDB_RIEGL_VOXEL_INDEX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.Tags)));
  RDB_RIEGL_VOXEL_INDEX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.Description)));
  RDB_RIEGL_VOXEL_INDEX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.UnitSymbol)));
  RDB_RIEGL_VOXEL_INDEX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.NamedValues)));
  RDB_RIEGL_VOXEL_INDEX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_INDEX.LodSettings)));

  RDB_RIEGL_COVARIANCES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.Name)));
  RDB_RIEGL_COVARIANCES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.Title)));
  RDB_RIEGL_COVARIANCES.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.Tags)));
  RDB_RIEGL_COVARIANCES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.Description)));
  RDB_RIEGL_COVARIANCES.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.UnitSymbol)));
  RDB_RIEGL_COVARIANCES.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.NamedValues)));
  RDB_RIEGL_COVARIANCES.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_COVARIANCES.LodSettings)));

  RDB_RIEGL_VOXEL_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.Name)));
  RDB_RIEGL_VOXEL_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.Title)));
  RDB_RIEGL_VOXEL_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.Tags)));
  RDB_RIEGL_VOXEL_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.Description)));
  RDB_RIEGL_VOXEL_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.UnitSymbol)));
  RDB_RIEGL_VOXEL_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.NamedValues)));
  RDB_RIEGL_VOXEL_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VOXEL_COUNT.LodSettings)));

  RDB_RIEGL_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.Name)));
  RDB_RIEGL_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.Title)));
  RDB_RIEGL_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.Tags)));
  RDB_RIEGL_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.Description)));
  RDB_RIEGL_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.UnitSymbol)));
  RDB_RIEGL_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.NamedValues)));
  RDB_RIEGL_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_ID.LodSettings)));

  RDB_RIEGL_VERTEX_FIRST.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.Name)));
  RDB_RIEGL_VERTEX_FIRST.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.Title)));
  RDB_RIEGL_VERTEX_FIRST.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.Tags)));
  RDB_RIEGL_VERTEX_FIRST.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.Description)));
  RDB_RIEGL_VERTEX_FIRST.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.UnitSymbol)));
  RDB_RIEGL_VERTEX_FIRST.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.NamedValues)));
  RDB_RIEGL_VERTEX_FIRST.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_FIRST.LodSettings)));

  RDB_RIEGL_VERTEX_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.Name)));
  RDB_RIEGL_VERTEX_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.Title)));
  RDB_RIEGL_VERTEX_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.Tags)));
  RDB_RIEGL_VERTEX_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.Description)));
  RDB_RIEGL_VERTEX_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.UnitSymbol)));
  RDB_RIEGL_VERTEX_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.NamedValues)));
  RDB_RIEGL_VERTEX_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VERTEX_COUNT.LodSettings)));

  RDB_RIEGL_CP_SEARCH_RADIUS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.Name)));
  RDB_RIEGL_CP_SEARCH_RADIUS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.Title)));
  RDB_RIEGL_CP_SEARCH_RADIUS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.Tags)));
  RDB_RIEGL_CP_SEARCH_RADIUS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.Description)));
  RDB_RIEGL_CP_SEARCH_RADIUS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.UnitSymbol)));
  RDB_RIEGL_CP_SEARCH_RADIUS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.NamedValues)));
  RDB_RIEGL_CP_SEARCH_RADIUS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_SEARCH_RADIUS.LodSettings)));

  RDB_RIEGL_CP_MAXIMUM_DISTANCE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.Name)));
  RDB_RIEGL_CP_MAXIMUM_DISTANCE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.Title)));
  RDB_RIEGL_CP_MAXIMUM_DISTANCE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.Tags)));
  RDB_RIEGL_CP_MAXIMUM_DISTANCE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.Description)));
  RDB_RIEGL_CP_MAXIMUM_DISTANCE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.UnitSymbol)));
  RDB_RIEGL_CP_MAXIMUM_DISTANCE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.NamedValues)));
  RDB_RIEGL_CP_MAXIMUM_DISTANCE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_CP_MAXIMUM_DISTANCE.LodSettings)));

  RDB_RIEGL_SELECTED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.Name)));
  RDB_RIEGL_SELECTED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.Title)));
  RDB_RIEGL_SELECTED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.Tags)));
  RDB_RIEGL_SELECTED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.Description)));
  RDB_RIEGL_SELECTED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.UnitSymbol)));
  RDB_RIEGL_SELECTED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.NamedValues)));
  RDB_RIEGL_SELECTED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SELECTED.LodSettings)));

  RDB_RIEGL_VISIBLE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.Name)));
  RDB_RIEGL_VISIBLE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.Title)));
  RDB_RIEGL_VISIBLE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.Tags)));
  RDB_RIEGL_VISIBLE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.Description)));
  RDB_RIEGL_VISIBLE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.UnitSymbol)));
  RDB_RIEGL_VISIBLE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.NamedValues)));
  RDB_RIEGL_VISIBLE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_VISIBLE.LodSettings)));

  RDB_RIEGL_SOURCE_CLOUD_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.Name)));
  RDB_RIEGL_SOURCE_CLOUD_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.Title)));
  RDB_RIEGL_SOURCE_CLOUD_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.Tags)));
  RDB_RIEGL_SOURCE_CLOUD_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.Description)));
  RDB_RIEGL_SOURCE_CLOUD_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.UnitSymbol)));
  RDB_RIEGL_SOURCE_CLOUD_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.NamedValues)));
  RDB_RIEGL_SOURCE_CLOUD_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_ID.LodSettings)));

  RDB_RIEGL_SOURCE_CLOUD_COUNT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.Name)));
  RDB_RIEGL_SOURCE_CLOUD_COUNT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.Title)));
  RDB_RIEGL_SOURCE_CLOUD_COUNT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.Tags)));
  RDB_RIEGL_SOURCE_CLOUD_COUNT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.Description)));
  RDB_RIEGL_SOURCE_CLOUD_COUNT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.UnitSymbol)));
  RDB_RIEGL_SOURCE_CLOUD_COUNT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.NamedValues)));
  RDB_RIEGL_SOURCE_CLOUD_COUNT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_CLOUD_COUNT.LodSettings)));

  RDB_RIEGL_SOURCE_INDEX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.Name)));
  RDB_RIEGL_SOURCE_INDEX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.Title)));
  RDB_RIEGL_SOURCE_INDEX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.Tags)));
  RDB_RIEGL_SOURCE_INDEX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.Description)));
  RDB_RIEGL_SOURCE_INDEX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.UnitSymbol)));
  RDB_RIEGL_SOURCE_INDEX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.NamedValues)));
  RDB_RIEGL_SOURCE_INDEX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDEX.LodSettings)));

  RDB_RIEGL_SOURCE_INDICATOR.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.Name)));
  RDB_RIEGL_SOURCE_INDICATOR.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.Title)));
  RDB_RIEGL_SOURCE_INDICATOR.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.Tags)));
  RDB_RIEGL_SOURCE_INDICATOR.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.Description)));
  RDB_RIEGL_SOURCE_INDICATOR.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.UnitSymbol)));
  RDB_RIEGL_SOURCE_INDICATOR.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.NamedValues)));
  RDB_RIEGL_SOURCE_INDICATOR.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SOURCE_INDICATOR.LodSettings)));

  RDB_RIEGL_DYNAMIC_OBJECT_POINT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.Name)));
  RDB_RIEGL_DYNAMIC_OBJECT_POINT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.Title)));
  RDB_RIEGL_DYNAMIC_OBJECT_POINT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.Tags)));
  RDB_RIEGL_DYNAMIC_OBJECT_POINT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.Description)));
  RDB_RIEGL_DYNAMIC_OBJECT_POINT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.UnitSymbol)));
  RDB_RIEGL_DYNAMIC_OBJECT_POINT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.NamedValues)));
  RDB_RIEGL_DYNAMIC_OBJECT_POINT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNAMIC_OBJECT_POINT.LodSettings)));

  RDB_RIEGL_SINGLE_SOURCE_POINT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.Name)));
  RDB_RIEGL_SINGLE_SOURCE_POINT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.Title)));
  RDB_RIEGL_SINGLE_SOURCE_POINT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.Tags)));
  RDB_RIEGL_SINGLE_SOURCE_POINT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.Description)));
  RDB_RIEGL_SINGLE_SOURCE_POINT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.UnitSymbol)));
  RDB_RIEGL_SINGLE_SOURCE_POINT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.NamedValues)));
  RDB_RIEGL_SINGLE_SOURCE_POINT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SINGLE_SOURCE_POINT.LodSettings)));

  RDB_RIEGL_MIRROR_OBJECT_POINT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.Name)));
  RDB_RIEGL_MIRROR_OBJECT_POINT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.Title)));
  RDB_RIEGL_MIRROR_OBJECT_POINT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.Tags)));
  RDB_RIEGL_MIRROR_OBJECT_POINT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.Description)));
  RDB_RIEGL_MIRROR_OBJECT_POINT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.UnitSymbol)));
  RDB_RIEGL_MIRROR_OBJECT_POINT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.NamedValues)));
  RDB_RIEGL_MIRROR_OBJECT_POINT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_MIRROR_OBJECT_POINT.LodSettings)));

  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Name)));
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Title)));
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Tags)));
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.Description)));
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.UnitSymbol)));
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.NamedValues)));
  RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED.LodSettings)));

  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Name)));
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Title)));
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Tags)));
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.Description)));
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.UnitSymbol)));
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.NamedValues)));
  RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT.LodSettings)));

  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Name)));
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Title)));
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Tags)));
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.Description)));
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.UnitSymbol)));
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.NamedValues)));
  RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL.LodSettings)));

  RDB_RIEGL_DISTANCE_TO_SURFACE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.Name)));
  RDB_RIEGL_DISTANCE_TO_SURFACE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.Title)));
  RDB_RIEGL_DISTANCE_TO_SURFACE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.Tags)));
  RDB_RIEGL_DISTANCE_TO_SURFACE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.Description)));
  RDB_RIEGL_DISTANCE_TO_SURFACE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.UnitSymbol)));
  RDB_RIEGL_DISTANCE_TO_SURFACE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.NamedValues)));
  RDB_RIEGL_DISTANCE_TO_SURFACE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_DISTANCE_TO_SURFACE.LodSettings)));

  RDB_RIEGL_NBHD_DISTANCE_MIN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.Name)));
  RDB_RIEGL_NBHD_DISTANCE_MIN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.Title)));
  RDB_RIEGL_NBHD_DISTANCE_MIN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.Tags)));
  RDB_RIEGL_NBHD_DISTANCE_MIN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.Description)));
  RDB_RIEGL_NBHD_DISTANCE_MIN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.UnitSymbol)));
  RDB_RIEGL_NBHD_DISTANCE_MIN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.NamedValues)));
  RDB_RIEGL_NBHD_DISTANCE_MIN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MIN.LodSettings)));

  RDB_RIEGL_NBHD_DISTANCE_MAX.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.Name)));
  RDB_RIEGL_NBHD_DISTANCE_MAX.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.Title)));
  RDB_RIEGL_NBHD_DISTANCE_MAX.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.Tags)));
  RDB_RIEGL_NBHD_DISTANCE_MAX.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.Description)));
  RDB_RIEGL_NBHD_DISTANCE_MAX.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.UnitSymbol)));
  RDB_RIEGL_NBHD_DISTANCE_MAX.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.NamedValues)));
  RDB_RIEGL_NBHD_DISTANCE_MAX.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAX.LodSettings)));

  RDB_RIEGL_NBHD_DISTANCE_MEAN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.Name)));
  RDB_RIEGL_NBHD_DISTANCE_MEAN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.Title)));
  RDB_RIEGL_NBHD_DISTANCE_MEAN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.Tags)));
  RDB_RIEGL_NBHD_DISTANCE_MEAN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.Description)));
  RDB_RIEGL_NBHD_DISTANCE_MEAN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.UnitSymbol)));
  RDB_RIEGL_NBHD_DISTANCE_MEAN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.NamedValues)));
  RDB_RIEGL_NBHD_DISTANCE_MEAN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEAN.LodSettings)));

  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Name)));
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Title)));
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Tags)));
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.Description)));
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.UnitSymbol)));
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.NamedValues)));
  RDB_RIEGL_NBHD_DISTANCE_MEDIAN.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MEDIAN.LodSettings)));

  RDB_RIEGL_NBHD_DISTANCE_STD.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.Name)));
  RDB_RIEGL_NBHD_DISTANCE_STD.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.Title)));
  RDB_RIEGL_NBHD_DISTANCE_STD.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.Tags)));
  RDB_RIEGL_NBHD_DISTANCE_STD.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.Description)));
  RDB_RIEGL_NBHD_DISTANCE_STD.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.UnitSymbol)));
  RDB_RIEGL_NBHD_DISTANCE_STD.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.NamedValues)));
  RDB_RIEGL_NBHD_DISTANCE_STD.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_STD.LodSettings)));

  RDB_RIEGL_NBHD_DISTANCE_MAD.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.Name)));
  RDB_RIEGL_NBHD_DISTANCE_MAD.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.Title)));
  RDB_RIEGL_NBHD_DISTANCE_MAD.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.Tags)));
  RDB_RIEGL_NBHD_DISTANCE_MAD.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.Description)));
  RDB_RIEGL_NBHD_DISTANCE_MAD.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.UnitSymbol)));
  RDB_RIEGL_NBHD_DISTANCE_MAD.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.NamedValues)));
  RDB_RIEGL_NBHD_DISTANCE_MAD.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_NBHD_DISTANCE_MAD.LodSettings)));

  RDB_RIEGL_SHOT_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.Name)));
  RDB_RIEGL_SHOT_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.Title)));
  RDB_RIEGL_SHOT_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.Tags)));
  RDB_RIEGL_SHOT_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.Description)));
  RDB_RIEGL_SHOT_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.UnitSymbol)));
  RDB_RIEGL_SHOT_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.NamedValues)));
  RDB_RIEGL_SHOT_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_SHOT_ID.LodSettings)));

  RDB_RIEGL_WFM_SBL_ID.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.Name)));
  RDB_RIEGL_WFM_SBL_ID.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.Title)));
  RDB_RIEGL_WFM_SBL_ID.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.Tags)));
  RDB_RIEGL_WFM_SBL_ID.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.Description)));
  RDB_RIEGL_WFM_SBL_ID.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.UnitSymbol)));
  RDB_RIEGL_WFM_SBL_ID.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.NamedValues)));
  RDB_RIEGL_WFM_SBL_ID.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_WFM_SBL_ID.LodSettings)));

  RDB_RIEGL_EVENT_FAKE_PULSE.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.Name)));
  RDB_RIEGL_EVENT_FAKE_PULSE.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.Title)));
  RDB_RIEGL_EVENT_FAKE_PULSE.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.Tags)));
  RDB_RIEGL_EVENT_FAKE_PULSE.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.Description)));
  RDB_RIEGL_EVENT_FAKE_PULSE.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.UnitSymbol)));
  RDB_RIEGL_EVENT_FAKE_PULSE.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.NamedValues)));
  RDB_RIEGL_EVENT_FAKE_PULSE.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_EVENT_FAKE_PULSE.LodSettings)));

  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Name)));
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Title)));
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Tags)));
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.Description)));
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.UnitSymbol)));
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.NamedValues)));
  RDB_RIEGL_POINT_INSIDE_AABB_BOCS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_INSIDE_AABB_BOCS.LodSettings)));

  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Name)));
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Title)));
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Tags)));
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.Description)));
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.UnitSymbol)));
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.NamedValues)));
  RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS.LodSettings)));

  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Name)));
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Title)));
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Tags)));
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.Description)));
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.UnitSymbol)));
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.NamedValues)));
  RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY.LodSettings)));

  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Name)));
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Title)));
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Tags)));
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.Description)));
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.UnitSymbol)));
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.NamedValues)));
  RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY.LodSettings)));

  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Name)));
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Title)));
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Tags)));
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.Description)));
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.UnitSymbol)));
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.NamedValues)));
  RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_XYZ_UNCERTAINTIES.LodSettings)));

  RDB_RIEGL_TPU_THU.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.Name)));
  RDB_RIEGL_TPU_THU.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.Title)));
  RDB_RIEGL_TPU_THU.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.Tags)));
  RDB_RIEGL_TPU_THU.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.Description)));
  RDB_RIEGL_TPU_THU.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.UnitSymbol)));
  RDB_RIEGL_TPU_THU.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.NamedValues)));
  RDB_RIEGL_TPU_THU.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_THU.LodSettings)));

  RDB_RIEGL_TPU_TVU.Name        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.Name)));
  RDB_RIEGL_TPU_TVU.Title       := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.Title)));
  RDB_RIEGL_TPU_TVU.Tags        := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.Tags)));
  RDB_RIEGL_TPU_TVU.Description := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.Description)));
  RDB_RIEGL_TPU_TVU.UnitSymbol  := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.UnitSymbol)));
  RDB_RIEGL_TPU_TVU.NamedValues := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.NamedValues)));
  RDB_RIEGL_TPU_TVU.LodSettings := String(UTF8ToAnsi(AnsiString(RDB_RIEGL_TPU_TVU.LodSettings)));
end;

procedure FillAttributeArray;
var
  i : Integer;
begin
  SetLength(RDB_POINT_ATTRIBUTES, 261); i := -1;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_SOCS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_LVCS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_MAP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_MAP_MIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_MAP_MAX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XY_MAP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_CORRECTIONS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_RANGE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_THETA;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PHI;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SCANNER_POSITION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DIRECTION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DIRECTION_MEDIUM;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DIRECTION_COARSE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_ORIGIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_BIAXIAL_SHIFT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_DIRECTION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_DIRECTION_LEVELLED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SURFACE_NORMAL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POINT_VECTOR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_UP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_COG_LINK;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_PATCH_LINK_VECTOR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PCA_AXIS_MIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PCA_AXIS_MAX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MODEL_CS_AXIS_X;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MODEL_CS_AXIS_Y;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MODEL_CS_AXIS_Z;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_AXIS_DIRECTION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ACCELEROMETER;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ACCELEROMETER_RAW;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ACCELEROMETER_BIAS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ACCELEROMETER_SCALE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_GYROSCOPE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_GYROSCOPE_RAW;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_GYROSCOPE_BIAS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_GYROSCOPE_SCALE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MAGNETIC_FIELD_SENSOR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MAGNETIC_FIELD_SENSOR_RAW;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_LATITUDE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_LONGITUDE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_HEIGHT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_UNDULATION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ROLL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_PITCH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_YAW;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_COURSE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_XYZ;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ROLL_NED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_PITCH_NED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_YAW_NED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_XYZ_VELOCITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_XYZ_NED_VELOCITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_XYZ_LVCS_VELOCITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ROLL_VELOCITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_PITCH_VELOCITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_YAW_VELOCITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_NED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_ACCURACY_NORTH_EAST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_ACCURACY_EAST_DOWN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VELOCITY_ACCURACY_DOWN_NORTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HYDRO_INTERSECTION_POINT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HYDRO_INTERSECTION_NORMAL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HYDRO_WSM_UNCERTAINTY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_XYZ_ACCURACIES;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ZENITH_VECTOR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_TIMESTAMP_HR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TIMESTAMP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_TIME_OFFSET;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_ECHO_TIME_OFFSET;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PPS_TIMESTAMP_EXTERN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PPS_TIMESTAMP_INTERN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EVENT_TIMESTAMP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TIMESTAMP_MIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TIMESTAMP_MAX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_TIMESTAMP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ACQUISITION_DATE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_REFLECTANCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_AMPLITUDE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_INTENSITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_GAIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DEVIATION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PULSE_WIDTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CLASS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_RGBA;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NIR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TEMPERATURE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DMI_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DMI_PATH_LENGTH_RAW;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DMI_SCALE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MTA_ZONE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MTA_UNCERTAIN_POINT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_FWA;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_BACKGROUND_RADIATION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_BACKGROUND_RADIATION_SI;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_BACKGROUND_RADIATION_INGAAS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TEMPERATURE_ESTIMATED_SI;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TEMPERATURE_ESTIMATED_INGAAS_SI;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TARGET_INDEX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TARGET_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TARGET_TYPE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ECHO_FIRST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ECHO_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HEIGHT_CENTER;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HEIGHT_MEAN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HEIGHT_MIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HEIGHT_MAX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POINT_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POINT_COUNT_GRID_CELL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PCA_EXTENTS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MTA_UNCERTAIN_POINT_VOL_DIFF;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PCA_THICKNESS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PCA_DIAMETER;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EIGENVALUE_NORM_MIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EIGENVALUE_NORM_MAX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_STD_DEV;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_CONFIDENCE_NORMAL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_SLOPE_CLASS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_OCCUPANCY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_WIDTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_HEIGHT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MATCH_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_PATCH_DISTANCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_PATCH_LATERAL_DISTANCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_PATCH_ANGULAR_DISTANCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_RADIUS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_HEIGHT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_ORIGIN_CONFIDENCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_RADIUS_CONFIDENCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_DIRECTION_CONFIDENCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_INLIER_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CYLINDER_OUTLIER_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_NORTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_EAST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_DOWN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_NORTH_EAST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_EAST_DOWN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_DOWN_NORTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_ROLL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_PITCH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_ACCURACY_YAW;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_CHANNEL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_MEAN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_STD_DEV;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_FIRST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SDA_FIRST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SDA_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SAMPLE_VALUE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CONTROL_OBJECT_TYPE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MODEL_FIT_QUALITY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CP_SURFACE_INCLINATION_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CP_SURFACE_INCLINATION_TOLERANCE_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CP_SURFACE_NORTHING_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CP_SURFACE_NORTHING_TOLERANCE_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_CONFIDENCE_XYZ;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_CONFIDENCE_XY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_CONFIDENCE_Z;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_CONFIDENCE_RANGE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_CONFIDENCE_THETA;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_CONFIDENCE_PHI;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_OBS_SIGNAL_CONFIDENCE_ROT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_USED_FOR_ADJUSTMENT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_REFERENCE_OBJECT_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EVENT_INDEX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EVENT_PULSE_WIDTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_RAW_RANGE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_RAW_LINE_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_RAW_FRAME_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_LINE_ANGLE_COARSE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_LINE_ANGLE_REDUCED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_FRAME_ANGLE_COARSE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ANGLE_OF_INCIDENCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SCAN_LINE_INDEX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_INDEX_LINE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MIRROR_FACET;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SCAN_SEGMENT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WAVEFORM_AVAILABLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_HYDRO_REFRACTION_CORRECTED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EXTINCTION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_AMPLITUDE_VOLUMETRIC;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_REFLECTANCE_VOLUMETRIC;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_REFLECTANCE_BOTTOM_UNCORRECTED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_ATTENUATION_VOLUMETRIC;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_TARGET_ATTENUATION_COEFFICIENT_VOLUMETRIC;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_SURFACE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_BOTTOM;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SVB_PATH_LENGTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_START_OF_SCAN_LINE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_END_OF_SCAN_LINE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SCAN_ANGLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SCAN_DIRECTION;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VOXEL_COLLAPSED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_LINE_SCAN_ACTIVE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_FRAME_SCAN_ACTIVE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DATA_ACQUISITION_ACTIVE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_REFERENCES;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_PATH_LENGTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_PDOP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_HDOP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_VDOP;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_AGE_OF_CORRECTIONS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_BASELINE_LENGTH;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SOLUTION_GNSS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SATELLITES_GNSS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SATELLITES_GPS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SATELLITES_GLONASS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SATELLITES_BEIDOU;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SATELLITES_GALILEO;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POF_SATELLITES_QZSS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PIXEL_LINEAR_SUMS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PIXEL_SQUARE_SUMS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHAPE_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_PLANE_CLUSTER_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SEGMENT_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VOXEL_LINEAR_SUMS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VOXEL_SQUARE_SUMS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VOXEL_INDEX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_COVARIANCES;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VOXEL_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VERTEX_FIRST;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VERTEX_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CP_SEARCH_RADIUS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_CP_MAXIMUM_DISTANCE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SELECTED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_VISIBLE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SOURCE_CLOUD_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SOURCE_CLOUD_COUNT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SOURCE_INDEX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SOURCE_INDICATOR;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DYNAMIC_OBJECT_POINT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SINGLE_SOURCE_POINT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_MIRROR_OBJECT_POINT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WINDOW_ECHO_IMPACT_CORRECTED;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DYNTRIG_UNCERTAIN_POINT;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_BAROMETRIC_HEIGHT_AMSL;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_DISTANCE_TO_SURFACE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NBHD_DISTANCE_MIN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NBHD_DISTANCE_MAX;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NBHD_DISTANCE_MEAN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NBHD_DISTANCE_MEDIAN;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NBHD_DISTANCE_STD;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_NBHD_DISTANCE_MAD;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_SHOT_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_WFM_SBL_ID;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_EVENT_FAKE_PULSE;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POINT_INSIDE_AABB_BOCS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_POINT_OUTSIDE_AABB_BOCS;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TPU_WSM_Z_UNCERTAINTY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TPU_WSM_NORMAL_UNCERTAINTY;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TPU_XYZ_UNCERTAINTIES;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TPU_THU;
  Inc(i); RDB_POINT_ATTRIBUTES[i] := @RDB_RIEGL_TPU_TVU;
end;

procedure FillAttributeGroupArray;
var
  i : Integer;
begin
  // Comments about string encoding see function FixStringEncoding()
  SetLength(RDB_POINT_ATTRIBUTE_GROUPS, 261); i := -1;
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Coordinates/Vectors')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Time')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Primary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Secondary Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
  Inc(i); RDB_POINT_ATTRIBUTE_GROUPS[i] := String(UTF8ToAnsi(AnsiString('Other Attributes')));
end;

procedure FillAttributeTagsArray;
var
  i : Integer;
begin
  // Comments about string encoding see function FixStringEncoding()

  i := 0;
  Inc(i); RDB_TAG_BOOLEAN := String(UTF8ToAnsi(AnsiString(RDB_TAG_BOOLEAN)));
  Inc(i); RDB_TAG_COLOR := String(UTF8ToAnsi(AnsiString(RDB_TAG_COLOR)));
  Inc(i); RDB_TAG_DIRECTION := String(UTF8ToAnsi(AnsiString(RDB_TAG_DIRECTION)));
  Inc(i); RDB_TAG_ENUMERATION := String(UTF8ToAnsi(AnsiString(RDB_TAG_ENUMERATION)));
  Inc(i); RDB_TAG_POSITION := String(UTF8ToAnsi(AnsiString(RDB_TAG_POSITION)));
  Inc(i); RDB_TAG_TIMESTAMP := String(UTF8ToAnsi(AnsiString(RDB_TAG_TIMESTAMP)));
  Inc(i); RDB_TAG_TRANSFORM := String(UTF8ToAnsi(AnsiString(RDB_TAG_TRANSFORM)));

  SetLength(RDB_POINT_ATTRIBUTE_TAGS, i); i := -1;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_BOOLEAN;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_COLOR;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_DIRECTION;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_ENUMERATION;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_POSITION;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_TIMESTAMP;
  Inc(i); RDB_POINT_ATTRIBUTE_TAGS[i] := RDB_TAG_TRANSFORM;
end;

initialization

FixStringEncoding;
FillAttributeArray;
FillAttributeGroupArray;
FillAttributeTagsArray;

end.
