/*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 */
/*!
 *******************************************************************************
 *
 * \file    createSettings.hpp
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Database create settings
 * \version 2015-10-14/AW: Initial version
 * \version 2016-07-13/AW: Documentation of 'chunkSizeLOD' updated
 * \version 2016-09-22/AW: Parameter 'cacheSize' changed from 32 to 64 bit type
 * \version 2016-12-20/AW: New functions to load/save settings from/to JSON
 * \version 2017-03-28/AW: Documentation of JSON load/save functions updated
 * \version 2017-11-24/AW: Constructors declared as "explicit" (#2825)
 * \version 2019-01-15/AW: Parameter 'chunkMode' added
 * \version 2019-01-18/AW: Parameter 'lodMode' added
 * \version 2019-02-15/AW: Fix C++ API wrapper of CreateSettings class
 * \version 2019-10-30/AW: Parameter 'optimizePointID' added (#3458)
 * \version 2020-02-21/AW: Class 'CreateSettings' is now context-free (#3544)
 * \version 2024-11-29/AW: Parameter 'pointInsertMode' added (#5339)
 *
 *******************************************************************************
 */

#ifndef RIEGL_RDB_POINTCLOUD_CREATESETTINGS_HPP
#define RIEGL_RDB_POINTCLOUD_CREATESETTINGS_HPP

//---< INCLUDES >---------------------------------------------------------------

#include <string>
#include <cstdlib>

#include "riegl/rdb/context.hpp"
#include "riegl/rdb/pointcloud/pointAttribute.hpp"

//---< NAMESPACE >--------------------------------------------------------------

namespace riegl {
namespace rdb {
namespace pointcloud {

//---< CLASS CreateSettings >---------------------------------------------------
/*!
 * \brief Database create settings
 *
 * This class defines settings for creating a new point cloud database.
 */
class CreateSettings
{
public:
    /*!
     * \brief Primary point attribute
     *
     * The primary point attribute defines the attribute that is used
     * to sort and index the points. Usually the 3D point coordinates are
     * used for that. The primary attribute is automatically added to the
     * point cloud (using Pointcloud::attributeAdd()) and cannot be deleted.
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     */
    riegl::rdb::pointcloud::PointAttribute primaryAttribute;

public:
    /*!
     * \brief Point chunk mode
     *
     * Points are internally organized in chunks (primary point attribute index
     * tree leaves). The size of a chunk (in the dimension of the primary point
     * attribute) may either be fixed (predefined) or adapted automatically so
     * that the number of points in a chunk does not exceed a certain limit. In
     * both cases, the "size" is defined by parameter CreateSettings::chunkSize
     * and parameter CreateSettings::chunkMode defines the meaning of the value.
     */
    enum ChunkMode
    {
        /*!
         * the chunk size defines the maximum number of points per chunk
         * (the default mode)
         */
        POINT_COUNT = 1,

        /*!
         * the chunk size defines the edge length of a chunk as 2^N times
         * resolution of the primary point attribute
         */
        EDGE_LENGTH = 2
    };

    /*!
     * \brief Point chunk mode
     *
     * Details see: CreateSettings::ChunkMode
     *
     * Default: CreateSettings::POINT_COUNT
     */
    std::uint32_t chunkMode;

    /*!
     * \brief Point chunk size
     *
     * Details see: CreateSettings::ChunkMode
     *
     * Default: 65536 points
     */
    std::uint32_t chunkSize;

public:
    /*!
     * \brief Level of detail mode
     *
     * A tree structure is used to sort and organize the point cloud. To create
     * a coarse representation of the point cloud (level of detail = "LOD"), a
     * number of equally distributed points is extracted from the tree leaf
     * nodes and copied to the parent nodes.
     *
     * The parameter CreateSettings::chunkSizeLOD defines how many points to
     * extract for LOD whereas the meaning of the value and the LOD creation
     * algorithm are defined by the parameter CreateSettings::lodMode.
     */
    enum LodMode
    {
        /*!
         * the LOD size defines the number of points to copy as a fraction of
         * the total (original) number of points. So if the original point count
         * is for example 19820526 and the size is set to 20%, then the number
         * of LOD points to add is 3964106 (rounded) and the final total number
         * of points is 23784632 (actual value may differ a little bit).
         */
        THINOUT = 1,

        /*!
         * the LOD size defines the number of binary subdivisions of the LOD
         * node's volume in each dimension. So if the primary point attribute
         * for example has a length of 2 (2D data) and the LOD size is set to 8,
         * then each LOD node is divided into 2^8 * 2^8 = 2^(8*2) = 2^16 = 65536
         * sub-volumes. All points of the node's immediate sub-nodes that fall
         * into one of the sub-volumes are merged to a single point and stored
         * in the LOD node. The method to merge the attribute values of a group
         * of points can be defined for each point attribute separately (details
         * see class PointAttribute).
         */
        COMBINE = 2
    };

    /*!
     * \brief Level of detail mode
     *
     * Details see: CreateSettings::LodMode
     *
     * Default: CreateSettings::THINOUT
     */
    std::uint32_t lodMode;

    /*!
     * \brief Level of detail size
     *
     * Details see: CreateSettings::LodMode
     *
     * To disable LOD generation, set this parameter to zero (no matter which
     * LOD mode is used).
     *
     * Default: 20 (i.e. 20% of the original point count).
     *
     * \note In RDB library versions before 2.0.850 this parameter had a
     *       different meaning. To retain compatibility, the parameter was
     *       not renamed to e.g. 'overheadLOD' - hence the strange name.
     */
    std::uint32_t chunkSizeLOD;

public:
    /*!
     * \brief Point cache size
     *
     * The database engine may buffer read and write operations in an
     * internal cache. This value defines the cache size in bytes (octets).
     *
     * Default: 500 MB
     */
    std::uint64_t cacheSize;

    /*!
     * \brief Data compression level
     *
     * The database automatically compresses point data before it is
     * stored. The compression level defines the compression quality,
     * i.e. higher values produce smaller files.
     *
     * Range: 0..100 (i.e. percent)
     * Default: 10%
     */
    std::uint8_t compressionLevel;

    /*!
     * \brief Point ID optimization
     *
     * Enable this option to apply optimizations to the point ID
     * attribute (riegl.id) that can result in smaller files.
     *
     * Default: false
     *
     * \note No optimizations are applied when:
     *       - points were inserted in a previous transaction
     *       - buffers for the point ID (riegl.id) or dynamic point attributes
     *         (e.g. "riegl.selected", "riegl.visible") are passed to the insert
     *         query (QueryInsert class)
     *
     * \warning When optimizations are enabled, the point ID no longer
     *          reflects the order in which the points were inserted.
     */
    bool optimizePointID;

    /*!
     * \brief Point insert mode
     *
     * This setting specifies how points are to be inserted into the database.
     *
     * Supported values:
     *   - "single-pass" (default)
     *          Points are sorted and inserted into the internal index structure
     *          each time QueryInsert::next() is called. This is suitable for
     *          scenarios where the same objects are only scanned once or just
     *          a few times (e.g in TLS or MLS).
     *
     *   - "multi-pass"
     *          Points are sorted and inserted into the internal index structure
     *          after the last call to QueryInsert::next() when the transaction is
     *          committed. This is suitable for scenarios where the same objects are
     *          scanned multiple times (e.g. in ALS or ULS when the system passes
     *          over the same area several times without toggling data acquisition).
     *          This mode is only supported for 3D point clouds (e.g. when the
     *          primary point attribute is "riegl.xyz") and when no points have
     *          been inserted before. Otherwise "single-pass" mode is used instead.
     *          This mode implies "optimizePointID" and point IDs are not returned
     *          by the insert query.
     *
     * \note It is recommended to only check whether the string starts with one of
     *       the above values and to ignore all subsequent characters (mode-specific
     *       parameters may be added after the mode name in the future).
     */
    std::string pointInsertMode;

public:
    ~CreateSettings();

    /*!
     * \brief Default constructor
     *
     * All properties are set to default values.
     */
    CreateSettings();

    /// \deprecated since 2.2.3 - use the context-free constructors instead
    explicit CreateSettings(riegl::rdb::Context &context);

    /*!
     * \brief Copy constructor
     *
     * All properties are copied from the given settings object.
     */
    CreateSettings(const CreateSettings &settings);

    /*!
     * \brief Assignment operator
     *
     * All properties are copied from the given settings object.
     */
    CreateSettings& operator=(const CreateSettings &settings);

    /*!
     * \brief Load settings from JSON string
     *
     * This function parses the given JSON string and applies all available
     * properties - missing properties are silently ignored (i.e. the value
     * remains unchanged). When parsing the JSON string fails, an exception
     * is thrown.
     *
     * Example JSON string:
     *
     *     {
     *         "primary_attribute": {
     *             "name": "riegl.xyz",
     *             "title": "XYZ",
     *             "description": "Cartesian point coordinates wrt. application coordinate system (0: X, 1: Y, 2: Z)",
     *             "unit_symbol": "m",
     *             "length": 3,
     *             "resolution": 0.00025,
     *             "minimum_value": -535000.0,
     *             "maximum_value": 535000.0,
     *             "default_value": 0.0,
     *             "storage_class": "variable",
     *             "compression_options": "shuffle",
     *             "scale_factor": 1.0
     *         },
     *         "chunk_mode": "point_count",
     *         "chunk_size": 65536,
     *         "lod_mode": "thinout",
     *         "chunk_size_lod": 20,
     *         "cache_size": 524288000,
     *         "compression_level": 10,
     *         "optimize_point_id": false,
     *         "point_insert_mode": "single-pass"
     *     }
     */
    void load(const std::string &json);

    /*!
     * \brief Save settings to JSON string
     * \see load()
     */
    std::string save() const;

private:
    friend class CreateSettingsWrapper;
    void *data;
};

}}} // namespace riegl::rdb::pointcloud

#endif // RIEGL_RDB_POINTCLOUD_CREATESETTINGS_HPP
