{*
 *******************************************************************************
 *
 *  Copyright 2025 RIEGL Laser Measurement Systems
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  SPDX-License-Identifier: Apache-2.0
 *
 *******************************************************************************
 *}
{*!
 *******************************************************************************
 *
 * \file    riegl.rdb.pointcloud.queryFill.pas
 * \author  RIEGL LMS GmbH, Austria
 * \brief   Point fill query (Pascal wrapper code)
 * \version 2016-11-08/AW: Initial version
 * \version 2018-07-06/AW: Wrapper function BindBuffer() added
 *
 * This class can be used to set attributes of existing points to a (one)
 * specific value (e.g. set "riegl.selected" to "1" for all points). This
 * query is similar to using select() and update() except that it only accepts
 * one value per point attribute and this value is copied to all points.
 *
 * \see riegl::rdb::Pointcloud::fill()
 *
 *******************************************************************************
 *}

unit riegl.rdb.pointcloud.queryFill;

{******************************************************************************}
{***} INTERFACE {**************************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb,
  riegl.rdb.context,
  riegl.rdb.pointcloud.graphNode,
  riegl.rdb.pointcloud.dataTypes,
  riegl.rdb.pointcloud.pointAttribute,
  riegl.rdb.default.attributes;

//---< RDB LIBRARY IMPORTS >----------------------------------------------------

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::fill()
 *}
function rdb_pointcloud_query_fill_new(
  Context    : TRDBContextHandle;            //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;         //!< [in] point cloud
  Node       : TRDBPointcloudGraphNodeID;    //!< [in] index node ID
  Filter     : TRDBString;                   //!< [in] filter string
  var Query  : TRDBPointcloudQueryFillHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Constructor
 * \see   riegl::rdb::Pointcloud::fill()
 *}
function rdb_pointcloud_query_fill_nodes_new(
  Context    : TRDBContextHandle;            //!< [in] library context
  Pointcloud : TRDBPointcloudHandle;         //!< [in] point cloud
  Nodes      : PRDBPointcloudGraphNodeID;    //!< [in] pointer to first element of node ID array
  Count      : TRDBUInt32;                   //!< [in] number of elements in node ID array
  Filter     : TRDBString;                   //!< [in] filter string
  var Query  : TRDBPointcloudQueryFillHandle //!< [out] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Destroy query instance
 *}
function rdb_pointcloud_query_fill_delete(
  Context   : TRDBContextHandle;            //!< [in] library context
  var Query : TRDBPointcloudQueryFillHandle //!< [in] query handle
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Bind attribute buffer
 *
 * Use this function to define a target buffer for a point attribute.
 * Exactly one buffer can be defined for an attribute (i.e. only the
 * most recently defined buffer will be used).
 *
 * The buffer is expected to be 1*s*d bytes large, where
 * __s__ is the size of one element as defined by 'dataType' and
 * __d__ is the number of attribute dimensions (elements).
 *
 * \note This function just stores the buffer pointer. So make
 *       sure that the buffer remains valid until you call next().
 *
 * \note If the value contained in the buffer is equal to the value
 *       of a point in the database, then the point is not updated.
 *       So if the values of all points are equal to the given value,
 *       then no points are updated and the point attribute revision
 *       stored at the primary index node will not change.
 *
 * \see riegl::rdb::pointcloud::PointAttributes
 *}
function rdb_pointcloud_query_fill_bind(
  Context   : TRDBContextHandle;             //!< [in] library context
  Query     : TRDBPointcloudQueryFillHandle; //!< [in] query handle
  Attribute : TRDBString;                    //!< [in] attribute name
  DataType  : TRDBUInt32;                    //!< [in] buffer data type \see dataTypes.h
  Buffer    : Pointer                        //!< [in] buffer location
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Fill points
 *
 * Use this function to actually read the point attributes from
 * all defined buffers and update the points in the database.
 *
 * To process all points, you need to repeatedly call next() until it
 * returns 0 (zero, see example 7). The number of points to process in
 * one step is defined by 'count'. Please note that the actual number
 * of processed points may be slightly different. To cancel processing
 * just stop calling next() and cancel (rollback) the transaction.
 *
 * \note IEEE-754 "NaN" values contained in floating point source
 *       buffers are ignored and the attribute's default value is
 *       used instead. Furthermore IEEE-754 "Infinity" values will
 *       always cause next() to fail with error code 10414, i.e.
 *       riegl::rdb::Error::QueryAttributeValueOutOfRange.
 *
 * \note Since version 2.4.2, unit vector point attributes (e.g.
 *       "riegl.direction") are automatically normalized (divided
 *       by the vector length) on insertion. This is done for all
 *       point attributes that meet the following requirements:
 *       `length` is 3, `minimumValue` is -1, `maximumValue` is 1,
 *       `resolution` is not 1, `scaleFactor` is 1, `unitSymbol` is
 *       empty, `invalidValue` is undefined and `tags` contains the
 *       "direction" tag but not the "do-not-normalize" tag.
 *
 * \returns the number of points processed
 *}
function rdb_pointcloud_query_fill_next(
  Context   : TRDBContextHandle;             //!< [in] library context
  Query     : TRDBPointcloudQueryFillHandle; //!< [in] query handle
  Count     : TRDBUInt32;                    //!< [in] size of source buffers in terms of points
  Processed : PRDBUInt32 = nil               //!< [out] number of processed points (optional)
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//______________________________________________________________________________
{*!
 * \brief Progress
 *
 * This function returns a coarse progress information in percent (0..100%).
 * Since the total number of returned points is not known in advance, this
 * value just reflects the progress of the (internal) index traversal.
 *}
function rdb_pointcloud_query_fill_progress(
  Context  : TRDBContextHandle;             //!< [in] library context
  Query    : TRDBPointcloudQueryFillHandle; //!< [in] query handle
  Progress : PRDBUInt32                     //!< [out] progress [0..100%]
) : TRDBResult; cdecl; external riegl.rdb.RDB_LIBRARY_FILENAME;

//---< CLASS TRDBPointcloudQueryFill >------------------------------------------
{*!
 * \brief Point fill query
 *
 * This class can be used to set attributes of existing points to a (one)
 * specific value (e.g. set "riegl.selected" to "1" for all points). This
 * query is similar to using select() and update() except that it only accepts
 * one value per point attribute and this value is copied to all points.
 *
 * \see riegl::rdb::Pointcloud::fill()
 *}
type
  TRDBPointcloudQueryFill = class(System.TObject)
  public
    {*!
     * \brief Default constructor
     *
     * Creates a null query - i.e. the query cannot be used to modify points.
     *
     * \see riegl::rdb::Pointcloud::fill()
     *}
    constructor Create; reintroduce; overload;

    {*!
     * \brief Constructor
     * \note  You cannot create new TRDBPointcloudQueryFill objects directly,
     *        use riegl::rdb::Pointcloud::fill() instead.
     *}
    constructor Create(
      Parent : System.TObject;
      Nodes  : PRDBPointcloudGraphNodeIDArray;
      Filter : String
    ); overload;

    destructor Destroy; override;

    {*!
     * \brief Bind attribute buffer
     *
     * Use this function to define a source buffer for a point attribute.
     * Exactly one buffer can be defined for an attribute (i.e. only the
     * most recently defined buffer will be used).
     *
     * The buffer is expected to be 1*s*d bytes large, where
     * __s__ is the size of one element as defined by 'dataType' and
     * __d__ is the number of attribute dimensions (elements).
     *
     * \note This function just stores the buffer pointer. So make
     *       sure that the buffer remains valid until you call next().
     *
     * \note If the value contained in the buffer is equal to the value
     *       of a point in the database, then the point is not updated.
     *       So if the values of all points are equal to the given value,
     *       then no points are updated and the point attribute revision
     *       stored at the primary index node will not change.
     *
     * \see riegl::rdb::pointcloud::PointAttributes
     *}
    procedure Bind(
      const Attribute : System.String;          //!< [in] attribute name
      const DataType  : TRDBPointcloudDataType; //!< [in] buffer data type
      const Buffer    : Pointer                 //!< [in] buffer location
    );

    // Helper for fundamental types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt8);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt8);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt8);    overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt8);    overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);    overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt16);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt16);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);  overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt16);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt16);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt32);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt32);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);  overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt32);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt32);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBUInt64);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBUInt64);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);  overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBInt64);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBInt64);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat32); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat32); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBFloat64); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBFloat64); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64); overload;

    // Helper for pointer types
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt8);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt8);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);   overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt8);    overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt8);    overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);    overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt16);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt16);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);  overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt16);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt16);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);   overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt32);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt32);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);  overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt32);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt32);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);   overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBUInt64);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBUInt64);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);  overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBInt64);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBInt64);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);   overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat32); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat32); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32); overload;
    procedure BindBuffer(const Attribute : System.String;                    const Buffer : PRDBFloat64); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     const Buffer : PRDBFloat64); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64); overload;

    // Helper for array types
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt8);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt8);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt8);    overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt8);    overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);    overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt16);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt16);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);  overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt16);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt16);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt32);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt32);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);  overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt32);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt32);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfUInt64);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfUInt64);  overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);  overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfInt64);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfInt64);   overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);   overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat32); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat32); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32); overload;
    procedure BindBuffer(const Attribute : System.String;                    var Buffer : TRDBArrayOfFloat64); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttribute;     var Buffer : TRDBArrayOfFloat64); overload;
    procedure BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64); overload;

    {*!
     * \brief Fill points
     *
     * Use this function to actually read the point attributes from
     * all defined buffers and update the points in the database.
     *
     * Afterwards you may re-fill the buffers or define new buffers
     * with bind() and call next() again until all points have been
     * updated.
     *
     * \note IEEE-754 "NaN" values contained in floating point source
     *       buffers are ignored and the attribute's default value is
     *       used instead. Furthermore IEEE-754 "Infinity" values will
     *       always cause next() to fail with error code 10414, i.e.
     *       riegl::rdb::Error::QueryAttributeValueOutOfRange.
     *
     * \note Since version 2.4.2, unit vector point attributes (e.g.
     *       "riegl.direction") are automatically normalized (divided
     *       by the vector length) on insertion. This is done for all
     *       point attributes that meet the following requirements:
     *       `length` is 3, `minimumValue` is -1, `maximumValue` is 1,
     *       `resolution` is not 1, `scaleFactor` is 1, `unitSymbol` is
     *       empty, `invalidValue` is undefined and `tags` contains the
     *       "direction" tag but not the "do-not-normalize" tag.
     *
     * \returns the number of points filled
     *}
    function Next(
      const Count : TRDBUInt32 //!< [in] number of points to process
    ) : TRDBUInt32;

    {*!
     * \brief Progress
     *
     * This function returns a coarse progress information in percent (0..100%).
     * Since the total number of returned points is not known in advance, this
     * value just reflects the progress of the (internal) index traversal.
     *}
    function Progress() : TRDBUInt32;

  private
    FContext    : TRDBContext;
    FPointcloud : TRDBPointcloudHandle;
    FQuery      : TRDBPointcloudQueryFillHandle;
  end;

{******************************************************************************}
{***} IMPLEMENTATION {*********************************************************}
{******************************************************************************}

//---< INCLUDES >---------------------------------------------------------------

uses
  riegl.rdb.pointcloud;

//---< TRDBPointcloudQueryFill::PUBLIC >----------------------------------------

constructor TRDBPointcloudQueryFill.Create;
begin
  FContext    := nil;
  FPointcloud := nil;
  FQuery      := nil;
end; // Create()

constructor TRDBPointcloudQueryFill.Create(
  Parent : System.TObject;
  Nodes  : PRDBPointcloudGraphNodeIDArray;
  Filter : String
);
begin
  inherited Create;
  FQuery      := nil;
  FContext    := (Parent as TRDBPointcloud).Context;
  FPointcloud := (Parent as TRDBPointcloud).Pointcloud;
  //
  if (Nodes <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_fill_nodes_new(
      FContext.Handle, FPointcloud,
      @Nodes^[0], Length(Nodes^),
      TRDBString(AnsiToUtf8(Filter)),
      FQuery
    ));
  end
  else
  begin
    FContext.Check(rdb_pointcloud_query_fill_new(
      FContext.Handle, FPointcloud, 0, // = all nodes
      TRDBString(AnsiToUtf8(Filter)),
      FQuery
    ));
  end;
end; // Create()

destructor TRDBPointcloudQueryFill.Destroy;
begin
  FContext.Check(rdb_pointcloud_query_fill_delete(FContext.Handle, FQuery));
  inherited;
end; // Destroy()

procedure TRDBPointcloudQueryFill.Bind(
  const Attribute : System.String;
  const DataType  : TRDBPointcloudDataType;
  const Buffer    : Pointer
);
begin
  if (FQuery <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_fill_bind(
      FContext.Handle, FQuery,
      TRDBString(AnsiToUtf8(Attribute)),
      Ord(DataType), Buffer
    ));
  end;
end; // Bind()

function TRDBPointcloudQueryFill.Next(const Count : TRDBUInt32) : TRDBUInt32;
begin
  if (FQuery <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_fill_next(
      FContext.Handle, FQuery, Count, @Result
    ));
  end
  else Result := 0;
end; // Next()

function TRDBPointcloudQueryFill.Progress() : TRDBUInt32;
begin
  if (FQuery <> nil) then
  begin
    FContext.Check(rdb_pointcloud_query_fill_progress(
      FContext.Handle, FQuery, @Result
    ));
  end
  else Result := 0;
end; // Progress()

//______________________________________________________________________________
//
// Helper for fundamental types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt8);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt8);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt16);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt16);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBUInt64);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBUInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBUInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBInt64);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBFloat64);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBFloat64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBFloat64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer);
end;

//______________________________________________________________________________
//
// Helper for pointer types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt8);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt8);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt16);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt16);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBUInt64);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBUInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBUInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBInt64);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; const Buffer : PRDBFloat64);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; const Buffer : PRDBFloat64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; const Buffer : PRDBFloat64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, Buffer);
end;

//______________________________________________________________________________
//
// Helper for array types
//______________________________________________________________________________
//

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt8);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt8);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt8);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT8, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt16);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt16);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt16);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT16, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt32);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfUInt64);
begin
  Bind(Attribute, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfUInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfUInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeUINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfInt64);
begin
  Bind(Attribute, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfInt64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeINT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat32);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat32);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT32, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : System.String; var Buffer : TRDBArrayOfFloat64);
begin
  Bind(Attribute, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttribute; var Buffer : TRDBArrayOfFloat64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

procedure TRDBPointcloudQueryFill.BindBuffer(const Attribute : TRDBPointcloudPointAttributeInfo; var Buffer : TRDBArrayOfFloat64);
begin
  Bind(Attribute.Name, RDBPointcloudDataTypeFLOAT64, @Buffer[0]);
end;

end.
